/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file is used for interal testint, not for public use.


*****************************************************************************/

#include <assert.h>
#include <typeinfo>
#include "cdldumper.h"

// Remove non printable characters,
string remove_non_printable_chars(const string& oldname)
{
    string newname = oldname;
    string::iterator it;
    for(it = newname.begin(); it != newname.end(); it++)
    {
        char c = (*it);
        if(!(c>31 && c<127)) //Printable ASCII characters only.
            (*it) = ' ';
    }
    return newname;
}

string get_cdl_type_str(value_type_t type)
{
    switch(type)
    {
#define HANDLE(type, cdltype) \
        case DFNT_##type: \
            return #cdltype; 

        HANDLE(CHAR8, char);
        HANDLE(INT8, byte);
        HANDLE(INT16, short);
        HANDLE(INT32, int);

        HANDLE(UCHAR8, char);
        HANDLE(UINT8, ubyte);

        HANDLE(UINT16, ushort);
        HANDLE(UINT32, uint);
        HANDLE(FLOAT32, float);
        HANDLE(FLOAT64, double);

#undef HANDLE
        default:
            throw range_error("Unknown type" "(" __FILE__ ":" TOSTRING(__LINE__)")" );
    };
}

// Dump CDL file. For internal testing only.
void dump_cdl(bool bDumpData)
{

    // Getting file name without path and ext
    string title;
    {
        vector<string> tokens;
        util_split_str(file->filename, "/", &tokens);
        assert(!tokens.empty());
        title = *tokens.rbegin();
    }
    std::cout << "netCDF " << title << " {" << std::endl;
    std::cout << "dimensions: " << std::endl;
    std::cout << "    ";
    int size = (int)file->sd->fulldimnamelist.size(), nr=0;

    for(std::set<std::string>::const_iterator i=file->sd->fulldimnamelist.begin(); i!=file->sd->fulldimnamelist.end(); ++i)
    {
        ++nr;
        int32 dimsize = file->sd->n1dimnamelist[*i];
        std::cout << *i << " = " << dimsize;
        if(nr!=size)
            std::cout << ", ";
        else
            std::cout << ";" << std::endl;
    }	

    std::cout << "variables:" << std::endl;
    for(std::vector<hdf4_var_sdfield *>::const_iterator i=file->sd->sdfields.begin(); i!=file->sd->sdfields.end(); i++)
    {
        std::cout << "    " << get_cdl_type_str((*i)->type);
        std::cout << "    " << (*i)->newname << "(";
        int size = (*i)->correcteddims.size(), nr=0;
        for(std::vector<hdf4_dim *>::const_iterator j=(*i)->get_corrected_dimensions().begin(); j!=(*i)->get_corrected_dimensions().end(); j++)
        {
            ++nr;
            std::cout << (*j)->get_name();
            if(nr!=size)
                std::cout << ",";
            else
                std::cout << ");" << std::endl;
        }
        for(std::vector<hdf4_attr *>::const_iterator j=(*i)->attrs.begin(); j!=(*i)->attrs.end(); j++)
        {
            std::cout << "     " << (*i)->newname << ":" << (*j)->get_name() << "=" << remove_non_printable_chars((*j)->get_str_value()) << ";" << std::endl;
        }
    }

    for(std::vector<VDATA *>::const_iterator i=file->vds.begin(); i!=file->vds.end(); i++)
    {
        for(std::vector<hdf4_var_vdfield *>::const_iterator j=(*i)->get_fields().begin(); j!=(*i)->get_fields().end(); j++) 
        {
            std::cout << "    " << get_cdl_type_str((*j)->type);
            std::cout << "    " << (*j)->newname << "()" << std::endl;
            for(std::vector<hdf4_attr *>::const_iterator k=(*j)->get_attributes().begin(); k!=(*j)->get_attributes().end(); k++)
            {
         	std::cout << "     " << (*j)->newname << ":" << (*k)->get_name() << "=" << remove_non_printable_chars((*k)->get_str_value()) << ";" << std::endl; 
            }
        }
        for(std::vector<hdf4_attr *>::const_iterator j=(*i)->get_attributes().begin(); j!=(*i)->get_attributes().end(); j++)
        {
                std::cout << "     " << (*i)->get_new_name() << ":" << (*j)->get_name() << "=" << remove_non_printable_chars((*j)->get_str_value()) << ";" << std::endl;
        }
    }		

    // File attributes.
    for(std::vector<hdf4_attr *>::const_iterator i=file->sd->get_attrs().begin(); i!=file->sd->get_attrs().end(); i++)
    {
        std::cout << ":" << (*i)->get_name() << "=";
        std::cout << (*i)->get_str_value();
      	std::cout << ";" << std::endl;
    }

    // Data.
    if(bDumpData)
    {
        std::cout << "data:" << std::endl;
        for(std::vector<hdf4_var_sdfield *>::const_iterator i=file->sd->sdfields.begin(); i!=file->sd->sdfields.end(); i++)
        {
            int32 *start = (int32*)malloc((*i)->get_rank()*sizeof(int32));
            int32 *stride = (int32*)malloc((*i)->get_rank()*sizeof(int32));
            int32 *edge = (int32*)malloc((*i)->get_rank()*sizeof(int32));	
            int k = 0;
            for(std::vector<hdf4_dim *>::const_iterator j=(*i)->get_corrected_dimensions().begin(); j!=(*i)->get_corrected_dimensions().end(); j++)
    	    {
                start[k] = 0;
                stride[k] = 1;
                edge[k++] = (*j)->get_size();
            }

            std::cout << "    " << (*i)->newname << "=";
            (*i)->dump_str(file->get_sd_id());

            // Testing subsetting. For debugging only.
            //(*i)->dump_str(file->get_sd_id(), start, stride, edge);
            std::cout << ";" << std::endl; 
        }
	
        for(std::vector<VDATA *>::const_iterator i=file->vds.begin(); i!=file->vds.end(); i++)
        {
	    for(std::vector<hdf4_var_vdfield *>::const_iterator j=(*i)->get_fields().begin(); j!=(*i)->get_fields().end(); j++)
                std::cout << "    " << (*j)->get_new_name() << "=" << (*j)->get_str_value() << ";" << std::endl;
	}
    }
    std::cout << "}" << std::endl;
}

// Dump CDL file. For internal testing only.
void dump_cdl(group *g, ostream &os, bool bDumpData)
{

        if (!g) {
            os <<"The root group of the converted CF file is NULL." ;
            os <<endl;
            os <<"This is perhaps due to the failure of internal filter operation. ";
            os <<endl;
            os <<"CDL output cannot be generated. ";
            os << endl;
            os.flush();
            return;
        }
	const list<group*>& children = g->get_child_groups_c();
	const list<var*>& vars= g->get_vars_c();
	const list<dim*>& dims= g->get_dims_c();
	const list<attr*>& attrs= g->get_attrs_c();

        // We find another way handle this issue. leave the comments and code for the time being.
        // KY 2013-01-11
#if 0
        // WARNING: This is just a temporary fix to check the "NULL" dimension names.
        // Dimension name to be NULL should not be allowed.
        // But the HDF-EOS2 doesn't check if a dimension name is NULL.  
        // I don't think real NASA files are created with a NULL dimension name.
        // Here I just check dimension names of the dims list. If I find a NULL
        // dimension name, the dumper will stop and issue an error. This needs to be fixed
        // in the future release(Jira HFRCFLIB-23 and Jira HFRCFLIB-24). KY 2013-01-11
//cerr<<"number of dimensions= "<<dims.size() <<endl;

	for(list<dim*>::const_iterator it = dims.begin(); it != dims.end(); it++)
	{
            if (""==(*it)->get_name()) {
                os <<"One dimension name of a variable is NULL." ;
                os <<endl;
                os <<"This is perhaps because the dimension name is set to NULL in the original file. ";
                os <<endl;
                os <<"CDL output will not be generated. ";
                os << endl;
            }
	}
#endif
	
	// Getting file name without path and ext
	string title;
	{
		file_info *f = g->get_file_info();
		assert(f);
		vector<string> tokens;
		util_split_str(f->m_filename, "/", &tokens);

		assert(!tokens.empty());
		title = *tokens.rbegin();
	}
	os << "netCDF " << title << " {" << endl;
	os << "dimensions: " << endl;
	os << "    ";

	size_t i=0;
	for(list<dim*>::const_iterator it = dims.begin(); it != dims.end(); it++)
	{
		os << (*it)->get_name() << " = " << (*it)->get_size();
		i++;
		if(i != dims.size())
			os << ", ";
		else
			os << ";" << endl;
	}
	os << endl;
	os << "variables: " << endl;
	for(list<var*>::const_iterator it = vars.begin(); it != vars.end(); it++)
	{
		os << "    " << get_cdl_type_str((*it)->get_type());
		os << "    " << (*it)->get_name() << "(";
		const list<dim*>& dims = (*it)->get_dims();
		
		i=0;
		for(list<dim*>::const_iterator di = dims.begin(); di != dims.end(); di++)
		{
			os << (*di)->get_name();
			i++;
			if(i != dims.size())
				os << ", ";
		}
		os << ");" << endl;


		const list<attr*>& subattrs = (*it)->get_attrs();
		for(list<attr*>::const_iterator ai = subattrs.begin(); ai != subattrs.end(); ai++)
		{
			os << "        ";
			os << (*it)->get_name() << ":" << (*ai)->get_name();
			os << " = ";

			cdl_output(*ai, os);
			os << ";" << endl;
		}
	}
		
	os << "    // Global attributes" << endl;
	for(list<attr*>::const_iterator ai = attrs.begin(); ai != attrs.end(); ai++)
	{
		os << "        ";
		os << ":" << (*ai)->get_name();
		os << " = ";

		cdl_output(*ai, os);
		os << ";" << endl;
	}
	
	if(bDumpData)
	{
		os << "data: " << endl;
		for(list<var*>::const_iterator vi = vars.begin(); vi != vars.end(); vi++)
		{
			// For debugging only.
			/*
			if((*vi)->get_name().find("EV_250_Aggr1km_RefSB")==string::npos)
				continue;
			*/
			os << "    ";
			os << (*vi)->get_name() << " = ";
			cdl_output(*vi, os);
			os << ";" << endl;
		}
	}
	os << "}" << endl;
	os.flush();
}

void cdl_output(attr* attr, ostream &os)
		throw(range_error)
{
	value_type_t type = attr->get_type();
	unsigned int tot_elems = attr->get_num_elements();

	switch(type)
	{
	case DFNT_UCHAR8:
	case DFNT_CHAR8:
		{
			string buf;
			attr->get_value_str(&buf);
			os << "\"" << buf << "\"";
		}
		break;

#define HANDLE3(TYPE, type, cast) \
		case DFNT_##TYPE:	\
			{	\
				vector<type> buf(tot_elems); \
				attr->get_value(&buf[0]); \
				for(size_t i = 0; i<buf.size(); i++) \
				{ \
					if((i != 0) && (i % 6==0) && (i+1 != buf.size())) \
						os << endl << "\t\t\t";	\
					os << (cast)(buf[i]); \
					if(i < buf.size() - 1) \
						os << ",";	\
				} \
			} \
			break;
#define HANDLE(TYPE, type)  \
		HANDLE3(TYPE, type, type)

	HANDLE3(INT8, int8, int);
	HANDLE(INT16, int16);
	HANDLE(INT32, int32);
	HANDLE3(UINT8, uint8, unsigned int);
	HANDLE(UINT16, uint16);
	HANDLE(UINT32, uint32);
	HANDLE(FLOAT32, float32);
	HANDLE(FLOAT64, float64);
	default:
			throw range_error(string("") + "Attribute " + attr->get_name() + " has unknown type (" __FILE__ ":" TOSTRING(__LINE__)")" );
#undef HANDLE
#undef HANDLE3
	}
}

void cdl_output(var *v, ostream &os)
{
	const list<dim*>& dims = v->get_dims_c();
	size_t tot_elems= 1, tot_bytes;

	vector<int32> start;
	vector<int32> stride;
	vector<int32> edge;

	for(list<dim*>::const_iterator it = dims.begin();
			it != dims.end(); it++)
	{
		start.push_back(0);
		stride.push_back(1); 
		edge.push_back((*it)->get_size());
	}
	
	for(vector<int32>::const_iterator it = edge.begin(); it!=edge.end(); it++)
                tot_elems *= (*it);

	value_type_t type = v->get_type();

	if(dims.size()==0) //For newly added VDATA only. VDATA has no dimensions.
	{
		eos2_var_data *vdata = (eos2_var_data*)v;
		tot_elems = vdata->get_bytes()/get_type_size(type);
	}

	tot_bytes = tot_elems*get_type_size(type);

	switch(type)
	{
#define HANDLE3(TYPE, type, cast) \
	case DFNT_##TYPE:	\
		{	\
			vector<type> buf(tot_elems); \
			v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); \
			for(size_t i = 0; i<buf.size(); i++) \
			{	\
				if((i != 0) && (i % 6==0) && (i+1 != buf.size())) \
					os << endl << "\t\t";	\
				os << (cast)(buf[i]); \
				if(i+1 != buf.size()) \
					os << ", "; \
			} \
		} \
		break;

#define HANDLE2(TYPE, type) \
        case DFNT_##TYPE:       \
                {       \
                        vector<type> buf(tot_elems); \
                        v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); \
			string str; \
			int32 order = 1; \
			const char* vname = typeid(*v).name(); \
			string s(vname); \
			if(s.find("eos2_var_data")!=string::npos) \
			{ \
				order = (static_cast<eos2_var_data*>(v))->get_order(); \
			} \
			for(size_t i=0; i<=buf.size()-order; i+=order) \
                        { \
				if((i != 0) && (i % 6==0) && (i!= buf.size()-order)) \
                                        os << endl << "\t\t"; \
				for(size_t j=i; j<i+order; j++) \
					if((type)buf[j]>31 && (type)buf[j]<127) \
						str+=(type)buf[j]; \
                                os << "\"" << str << "\""; \
                                if(i!=buf.size()-order) \
                                        os << ","; \
                                str.clear(); \
                        } \
		} \
		break;

#define HANDLE(TYPE, type) \
		HANDLE3(TYPE, type, type)

	//HANDLE(CHAR8, char8);
	HANDLE2(CHAR8,  char8); //For fields in VDATA  whose array size is >1. 
	HANDLE3(INT8,   int8, int);
	HANDLE(INT16,   int16);
	HANDLE(INT32,   int32);
	HANDLE(UCHAR8,  uchar8);
	HANDLE3(UINT8,  uint8, unsigned int);
	HANDLE(UINT16,  uint16);
	HANDLE(UINT32,  uint32);
	HANDLE(FLOAT32, float32);
	HANDLE(FLOAT64, float64);
			
	default:
		throw range_error("Unknown vairable type at" "(" __FILE__ ":" TOSTRING(__LINE__)")" );
#undef HANDLE
#undef HANDLE3
	}
}
