/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to HDF-EOS2 _FillValue attribute.

*****************************************************************************/

#include "eos2_attr_fillvalue.h"
#include <stdexcept>
#include <assert.h>

namespace eoslib 
{

eos2_attr_fillvalue::eos2_attr_fillvalue(const eos2_attr_fillvalue& r): 
    attr(r),
    m_file_info(r.m_file_info),
    m_type(r.m_type),
    m_value(r.m_value)
{
    m_file_info->inc(this);
}

eos2_attr_fillvalue::eos2_attr_fillvalue(
    var *v,
    file_info *f,
    value_type_t type,
    char *fillvalue):
    m_file_info(f),
    attr(v, "_FillValue"),
    m_type(type)
{
    m_file_info->inc(this);
    switch(type)
    {
#define HANDLE(type) \
    case DFNT_##type: \
        m_value.resize(SIZE_##type); \
        std::copy(fillvalue, fillvalue+SIZE_##type, m_value.begin()); \
        break;

        HANDLE(CHAR8);
        HANDLE(INT8);
        HANDLE(INT16);
        HANDLE(INT32);
        HANDLE(UCHAR8);
        HANDLE(UINT8);
        HANDLE(UINT16);
        HANDLE(UINT32);
        HANDLE(FLOAT32);
        HANDLE(FLOAT64);

#undef HANDLE
        default:
            assert(0);
            throw(std::range_error("Wrong fillvalue type"));
    }
}

eos2_attr_fillvalue::~eos2_attr_fillvalue()
{
    m_file_info->dec(this);
}

value_type_t eos2_attr_fillvalue::get_type() const
{
    return m_type;
}

unsigned int eos2_attr_fillvalue::get_num_elements() const
{
    if(m_type == DFNT_CHAR8 || m_type == DFNT_UCHAR8)
        return m_value.size();
    else
        return 1;
}

void eos2_attr_fillvalue::get_value(void *buf) const
{
    std::copy(m_value.begin(), m_value.end(), (char*)buf);
}

attr* eos2_attr_fillvalue::clone() const
{
    eos2_attr_fillvalue *a = new eos2_attr_fillvalue(*this);
    return a;
}

bool eos2_attr_fillvalue::same_obj_test(attr *a) const
{
    eos2_attr_fillvalue *w = dynamic_cast<eos2_attr_fillvalue*>(a);
    //std::cout << typeid(*this).name() << std::endl;
    //std::cout << typeid(*w).name() << std::endl;
    //std::cout << typeid(*a).name() << std::endl;
    //if(typeid(*this) != typeid(*w))
    //	return false;
    if(w==NULL)
        return false;
    if(this->m_type!= w->m_type)
        return false;
    if(this->m_value!= w->m_value)
        return false;
    return true;
}
} // namespace
