/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to attribute access of an 
HDF-EOS2 grid or swath.


*****************************************************************************/

#include "eos2_attr_gdsw.h"
#include "eoslib_types.h"
#include "misc_eos2.h"
#include "eoslib_err.h"
//#include <assert.h>

namespace eoslib 
{

eos2_attr_gdsw::eos2_attr_gdsw(const eos2_attr_gdsw& r):
    attr(r),
    m_file_info(r.m_file_info),
    m_gdsw(r.m_gdsw),
    m_handle(r.m_handle),
    m_eos2name(r.m_eos2name),
    m_typenumber(r.m_typenumber),
    m_count(r.m_count)
{
    m_file_info->inc(this);
    m_handle->inc(this);
}

eos2_attr_gdsw::eos2_attr_gdsw(
    group *g,
    file_info *f,
    eos2_group_type_t gdsw,	// GRID or SWATH
    eos2_handle *handle,
    const std::string& nickname,
    const std::string& eos2name) throw (CannotGetAttrInfoException):
        attr(g, nickname),
        m_file_info(f),
        m_gdsw(gdsw),
        m_handle(handle),
        m_eos2name(eos2name)
{
    m_file_info->inc(this);
    m_handle->inc(this);

    intn ret = -1;
    int32 bytes = 0;
    switch(m_gdsw)
    {
        case GRID:
            ret = GDattrinfo(m_handle->get(), const_cast<char*>(m_eos2name.c_str()), &m_typenumber, &bytes);
            break;
        case SWATH:
            ret = SWattrinfo(m_handle->get(), const_cast<char*>(m_eos2name.c_str()), &m_typenumber, &bytes);
            break;
        default:
            throw std::runtime_error("unsupported HDF-EOS2 object type. Currently only grid or swath is supported. (" __FILE__ ":" TOSTRING(__LINE__)")" );
    } //end of switch

    m_count = bytes / DFKNTsize(m_typenumber);

    if(ret != 0)
        throw CannotGetAttrInfoException(std::string("") + "Cannot get attrinfo of " + get_name() + "(" __FILE__ ":" TOSTRING(__LINE__)")");
}

eos2_attr_gdsw::~eos2_attr_gdsw()
{
    m_file_info->dec(this);
    m_handle->dec(this);
}

value_type_t eos2_attr_gdsw::get_type() const
{
    return m_typenumber;
}

unsigned int eos2_attr_gdsw::get_num_elements() const
{
    return m_count;
}

void eos2_attr_gdsw::get_value(void *buf) const
{
    intn ret = -1;
    switch(m_gdsw)
    {
        case GRID:
            ret = GDreadattr(m_handle->get(), const_cast<char*>(m_eos2name.c_str()), buf);
            break;
        case SWATH:
            ret = SWreadattr(m_handle->get(), const_cast<char*>(m_eos2name.c_str()), buf);
            break;
        default:
            throw std::runtime_error("The HDF-EOS2 type must be either GRID or SWATH for eos2_attr_gdsw::get_value. (" __FILE__ ":" TOSTRING(__LINE__)")" );
            //assert(0);
    };
    if(ret != 0)
        throw CannotReadAttrException(std::string("") + "Cannot read attribute " + get_name() + "(" __FILE__ ":" TOSTRING(__LINE__)")");
}

attr* eos2_attr_gdsw::clone() const
{
    eos2_attr_gdsw *a = new eos2_attr_gdsw(*this);
    return a;
}

int _cb_on_each_gd_attr(char *name, void *arg)
{
    std::pair<
        eos2_group*, 
        std::list<attr*>*> *parg;
		
    parg = (std::pair<eos2_group* , std::list<attr*>* > *)arg;

    eos2_group *g = parg->first;
    std::list<attr*> *pattrs = parg->second;

    //std::cout << "Group: " << g->get_name() << 
    //		", ATTR: " << name << std::endl;
    eos2_attr_gdsw *a = new eos2_attr_gdsw(
        g,
        g->m_file_info,
        GRID,
        g->m_handle,
        name,
        name);

    //std::cout << name << std::endl;
    pattrs->push_back(a);
    return 0;
}

int _cb_on_each_sw_attr(char *name, void *arg)
{
    std::pair<
        eos2_group*, 
        std::list<attr*>*> *parg;
		
    parg = (std::pair<eos2_group* , std::list<attr*>* > *)arg;

    eos2_group *g = parg->first;
    std::list<attr*> *pattrs = parg->second;

    eos2_attr_gdsw *a = new eos2_attr_gdsw(
        g,
        g->m_file_info,
        SWATH,
        g->m_handle,
        name,
        name);

    //std::cout << name << std::endl;
    pattrs->push_back(a);
    return 0;
}

void read_eos2_attr_gd(eos2_group *g, std::list<attr*> *pattrs)
{
    //std::cout << "Reading attributes of grid " << g->get_name() << std::endl;
    std::pair<
        eos2_group*, 
        std::list<attr*>*> arg;
    arg.first= g;
    arg.second = pattrs;

    int32 handle = 0;
    if(g->m_group_type == ROOT)
        handle = g->m_hgdf->get();
    else if(g->m_group_type == GRID)
        handle = g->m_handle->get();
    else
        throw std::runtime_error("The supported group type must be either ROOT or GRID for read_eos2_attr_gd. (" __FILE__ ":" TOSTRING(__LINE__)")" );
//        assert(0);

    eos2util_walk_grid_attrs(handle, _cb_on_each_gd_attr, &arg);
}

void read_eos2_attr_sw(eos2_group *g, std::list<attr*> *pattrs)
{
    std::pair<
        eos2_group*, 
        std::list<attr*>*> arg;
    arg.first= g;
    arg.second = pattrs;

    int32 handle = 0;
    if(g->m_group_type == ROOT)
        handle = g->m_hswf->get();
    else if(g->m_group_type == SWATH)
        handle = g->m_handle->get();
    else
        throw std::runtime_error("The supported group type  must be either ROOT or SWATH for read_eos2_attr_sw. (" __FILE__ ":" TOSTRING(__LINE__)")" );
        //assert(0);

    eos2util_walk_swath_attrs(handle, _cb_on_each_sw_attr, &arg);
}

bool eos2_attr_gdsw::same_obj_test(attr *a) const
{
    eos2_attr_gdsw *w = dynamic_cast<eos2_attr_gdsw*>(a);
    //std::cout << typeid(*this).name() << std::endl;
    //std::cout << typeid(*w).name() << std::endl;
    //std::cout << typeid(*a).name() << std::endl;
    //if(typeid(*this) != typeid(*w))
    //	return false;
    if(w==NULL)
        return false;
    if(this->m_handle!= w->m_handle)
        return false;
    if(this->m_eos2name!= w->m_eos2name)
        return false;
    return true;
}

} // namespace
