/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Category: Headers for the EOS2 module of the eoslib name space.

Description:

This file includes the declarations of methods that access attributes of an HDF-EOS2 grid or swath.


*****************************************************************************/

#ifndef EOS2_ATTR_GDSW_H
#define EOS2_ATTR_GDSW_H

#include "eos2_defs.h"

namespace eoslib
{

//! Atttributes of HDF-EOS2 grids and swaths
/**
 * This class represents the eos2 attributes
 * of grids and swaths in
 * HDF-EOS2 files. 
 * Since the size of the attributes
 * can be large, we do not store the 
 * values internally.
 */
class eos2_attr_gdsw: public attr
{
    public:
        class CannotGetAttrInfoException: public std::runtime_error {
            public: 
                CannotGetAttrInfoException (std::string msg): std::runtime_error(msg) {} }; /* end of CannotGetAttrInfoException */

        class CannotReadAttrException: public std::runtime_error { 
            public: 
                CannotReadAttrException (std::string msg): std::runtime_error(msg) {} }; /* end of CannotReadAttrException */

        eos2_attr_gdsw(
            group *g,
            file_info *f,
            eos2_group_type_t gdsw,	/* GRID or SWATH */
            eos2_handle *handle,
            const std::string& nickname,
            const std::string& eos2name) throw (CannotGetAttrInfoException);

        virtual ~eos2_attr_gdsw();

        virtual value_type_t get_type() const;
        virtual unsigned int get_num_elements() const;
        virtual void get_value(void *buf) const;
        virtual attr *clone() const;
        virtual bool same_obj_test(attr *) const;

    protected:
        file_info *m_file_info;
        eos2_group_type_t m_gdsw;
        eos2_handle *m_handle;
        std::string m_eos2name;

        int32 m_typenumber;
        int32 m_count;
	
    private:
        /* Used by duplicate() */
        eos2_attr_gdsw(const eos2_attr_gdsw&);

        /* disabled */
        eos2_attr_gdsw();
};

/**
 * This function gets the list of all attributes of a given
 * grid. 
 */
void read_eos2_attr_gd(eos2_group *g, std::list<attr*> *pattrs);

/**
 * This function gets the list of all attributes of a given
 * swath. 
 */
void read_eos2_attr_sw(eos2_file *f, int32 swid, std::list<attr*> *pattrs);

} // namespace
#endif

