/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to an HDF-EOS2 group.


*****************************************************************************/

#include <cassert>
#include <iostream>
#include <stdexcept>

#include "eos2_defs.h"
#include "misc_eos2.h"
#include "mem_attr.h"
#include "eoslib_err.h"

namespace eoslib {

std::string EOS2_CALCULATED_LATITUDE = "E2C_latitude";
std::string EOS2_CALCULATED_LONGITUDE = "E2C_longitude";

eos2_group::eos2_group(const char *filename)
    throw(CannotOpenFileException, InvalidEOS2FileException)
    : group("/"), m_hgdf(NULL), m_hswf(NULL), m_handle(NULL), m_group_type(ROOT)
{
    int32 gdfd = GDopen(const_cast<char*>(filename), DFACC_READ);
    if(gdfd  == -1) 
       throw CannotOpenFileException("Cannot open file (" __FILE__ ":" TOSTRING(__LINE__)")" );

    int32 swfd = SWopen(const_cast<char*>(filename), DFACC_READ);
    if(swfd == -1 ){
        GDclose(gdfd);
        throw CannotOpenFileException("Cannot open file (" __FILE__ ":" TOSTRING(__LINE__)")" );
    }

    file_info *fi = NULL;
    try {
        fi = new file_info;
        fi->m_filename = filename;
        set_file_info(fi);

        if(swfd != -1)
        {
            // swath file
            m_hswf= new eos2_handle("m_hswf", swfd, eos2_handle::SWATH_FILE, NULL);
            m_hswf->inc(this);
            read_eos2_attr_sw(this, &m_attrs);
            eos2util_walk_swaths(
                                const_cast<char*>(m_file_info->m_filename.c_str()),
                                 _cb_on_each_swath,
                                 (void*)this);
        } //end of if(swfd != -1)

        if(gdfd != -1)
        {
            // grid file
            m_hgdf= new eos2_handle("m_hgdf", gdfd, eos2_handle::GRID_FILE, NULL);
            m_hgdf->inc(this);
            read_eos2_attr_gd(this, &m_attrs);
            eos2util_walk_grids(
                                const_cast<char*>(m_file_info->m_filename.c_str()),
                                _cb_on_each_grid,
                                (void*)this);
        } // end of if(gdfd != -1)
    }
    catch(...) {
        GDclose(gdfd);
        SWclose(swfd);
        if(fi !=NULL)
            delete fi;
    }
}

eos2_group::eos2_group(eos2_group* parent, eos2_group_type_t type, const char* name)
    throw(GroupNotExistsException)
    : group(name), m_hgdf(NULL), m_hswf(NULL), m_handle(NULL), m_group_type(type), m_eos2name(name)
{
    set_file_info(parent->m_file_info);

    if(m_group_type == GRID)
    {
        int32 eos2id = -1;
        try {
            eos2id = GDattach(
            parent->m_hgdf->get(), const_cast<char*>(m_eos2name.c_str()));
            m_handle = new eos2_handle(std::string("m_handle for ") + name, eos2id, eos2_handle::SINGLE_GRID, parent->m_hgdf);
            m_handle->inc(this);
		
           // Read dims
           int r = eos2util_walk_grid_dims(
                     m_handle->get(), _cb_on_each_gd_dim, (void*)this);

           // This is not ideal since memory leaking may occur, but it is better than generating wrong results.
           // Temporary fix for this release only. A jira ticket has been submitted for the future fix. KY 2013-01-11
           if (-1 == r) 
               throw std::runtime_error("Some HDF-EOS2 dimensions don't have dimension names. (" __FILE__ ":" TOSTRING(__LINE__)")");

           // Read attrs
           read_eos2_attr_gd(this, &m_attrs);

           // Read data fields. Make sure that dim is read before this.
           eos2util_walk_grid_datafields(
                                       m_handle->get(), _cb_on_each_gd_datafield, (void*)this);

           // Add grid latitude and longitude
           _add_gd_ll();
        }
        catch(std::runtime_error e) {
             std::cout<<e.what() <<std::endl;
             GDdetach(eos2id);
             delete m_handle;
             throw;
        }
    
        catch(...) {
            GDdetach(eos2id);
            delete m_handle;
            throw;
        }

    } // end of if(m_group_type == GRID)
    else if(m_group_type == SWATH)
    {
        int eos2id = -1;
        try {
            eos2id = SWattach(
            parent->m_hswf->get(), const_cast<char*>(m_eos2name.c_str()));
            m_handle = new eos2_handle(name, eos2id, eos2_handle::SINGLE_SWATH, parent->m_hswf);
            m_handle->inc(this);

            // Read dims
            int r= eos2util_walk_swath_dims(
                        m_handle->get(), _cb_on_each_sw_dim, (void*)this);

            if (-1 == r) 
                throw std::runtime_error("Some HDF-EOS2 dimensions don't have dimension names. (" __FILE__ ":" TOSTRING(__LINE__)")");

            // Read attrs
            read_eos2_attr_sw(this, &m_attrs);

            // Read data fields. Make sure that dim is read before this.
            r = eos2util_walk_swath_datafields(
                                              m_handle->get(), _cb_on_each_sw_datafield, (void*)this);
            if(-1 == r)
                throw std::runtime_error(" Fail to retrieve swath data fields. (" __FILE__ ":" TOSTRING(__LINE__)")");

            // Read geo fields. Make sure that dim is read before this.
            r = eos2util_walk_swath_geofields(
                                             m_handle->get(), _cb_on_each_sw_geofield, (void*)this);
            if(-1 == r)
                throw std::runtime_error(" Fail to retrieve swath geo fields. (" __FILE__ ":" TOSTRING(__LINE__)")");

        }
        catch(std::runtime_error e) {
            std::cout<<e.what() <<std::endl;
            SWdetach(eos2id);
            delete m_handle;
            throw;
        }
    
        catch(...) {
            SWdetach(eos2id);
            delete m_handle;
            throw;
        }

   } // end of if(m_group_type == SWATH)
}

eos2_group::~eos2_group() 
{
#if 0
    std::cout << "~eos2_group: " << this->get_name() << std::endl;
    if(m_handle)
        std::cout << "\tm_handle: " << m_handle->get_refcnt() << std::endl;
    if(m_hgdf)
        std::cout << "\tm_hgdf: " << m_hgdf->get_refcnt() << std::endl;
    if(m_hswf)
        std::cout << "\tm_hswf: " << m_hswf->get_refcnt() << std::endl;
#endif

    if(m_handle)
        m_handle->dec(this);
    if(m_hgdf)
        m_hgdf->dec(this);
    if(m_hswf)
        m_hswf->dec(this);

    //while(!m_no_use_groups.empty())
    //{
    //	group *g = *(m_no_use_groups.begin());
    //	delete g;
    //	m_no_use_groups.pop_front();
    //}
    // Other items's ref counts will be decremented thrugh
    // the destructor of the base class eoslib_group.
}

int _cb_on_each_grid(char *gridname, void *arg)
{
    eos2_group *parent = (eos2_group*)arg;
    eos2_group *g = new eos2_group(
        parent,
        GRID,
        gridname);
    parent->m_child_groups.push_back(g);

    return 0;
}

int _cb_on_each_swath(char *swathname, void *arg)
{
    eos2_group *parent = (eos2_group*)arg;
    eos2_group *g = new eos2_group(
        parent,
        SWATH,
        swathname);
    parent->m_child_groups.push_back(g);
    return 0;
}

int _cb_on_each_sw_datafield(char *name, void *arg)
{
    eos2_group *g= (eos2_group*)arg;

    std::list<dim*> new_dims;
    eos2_var_data *v = new eos2_var_data(
        g,
        g->m_file_info, 
        SWATH,
        g->m_handle,
        name,
        g->m_dims,
        &new_dims);
    g->m_vars.push_back(v);

    
    // If v is a reduced (dim-mapped) variable, 
    // we also need to add its expaned variable too.
    bool dimmap_reduced = false;

    const std::list<dim*>& dims = v->get_dims_c();
    std::list<dim*>::const_iterator dit;
    for(dit = dims.begin(); dit!=dims.end(); dit++)
    {
        intn r = SWgeomapinfo(g->m_handle->get(),
                                        const_cast<char*>((*dit)->get_name().c_str()));
        if(r==1){
            dimmap_reduced = true;
            break;
        }
    } // end of for

    if(true == dimmap_reduced)
    {
        std::string dm_name = name;

        eos2_var_dm *w = new eos2_var_dm(
            g,
            dm_name,
            v,
            g->m_dims,
            &new_dims);
        g->m_vars.push_back(w);

        // We mark the orig var as dimmap-reduced one
        mem_attr *attr1 = new mem_attr(v, "_dimmap_reduced");
        attr1->set_value("true");
        v->add_attr(attr1);
    } // end  of if(v->is_dimmap_reduced())


    // Process newly-found dims
    while(!new_dims.empty())
    {
        dim* nd = *new_dims.begin();
        g->m_dims.push_back(nd);
        //nd->inc();
        g->m_hidden_dims.push_back(nd);

        new_dims.pop_front();
    } // end of while
    return 0;
}

int _cb_on_each_sw_geofield(char *name, void *arg)
{
    eos2_group *g= (eos2_group*)arg;

    std::list<dim*> new_dims;
    eos2_var_geo *v = new eos2_var_geo(
                                       g,
                                       g->m_file_info, 
                                       g->m_handle,
                                       name,
                                       g->m_dims,
                                       &new_dims);
    g->m_vars.push_back(v);
    
    // We cannot assume that all geo fields are CVs
    // since some CVs may share the same dimension.
    // we have no way to know which CV is the real
    // coordinate for that dimension.
    // Without extra information, we will create
    // a dummy CV for a dimension
    // that is shared by several 1-D variables.
    // KY 2013-01-13
    //v->set_cv(); // All geo fields are cvs.
    

    // If v is a reduced (dim-mapped) variable, 
    // we also need to add its expaned variable too.
    if(v->is_dimmap_reduced())
    {
        std::string dm_name = name;

        eos2_var_dm *w = new eos2_var_dm(
            g,
            dm_name,
            v,
            g->m_dims,
            &new_dims);
        g->m_vars.push_back(w);
        w->set_cv();

        // We mark the orig var as dimmap-reduced one
        mem_attr *attr1 = new mem_attr(v, "_dimmap_reduced");
        attr1->set_value("true");
        v->add_attr(attr1);
    } // end  of if(v->is_dimmap_reduced())

    // Process newly-found dims
    while(!new_dims.empty())
    {
        dim* nd = *new_dims.begin();
        g->m_dims.push_back(nd);
        //nd->inc();
        g->m_hidden_dims.push_back(nd);

        new_dims.pop_front();
    }
    return 0;
}


////////////////////////////////////////////////
// Grid callbacks
////////////////////////////////////////////////

int _cb_on_each_gd_datafield(char *name, void *arg)
{
    eos2_group *g= (eos2_group*)arg;

    std::list<dim*> new_dims;
    eos2_var_data *v = new eos2_var_data(
        g,
        g->m_file_info, 
        GRID,
        g->m_handle,
        name,
        g->m_dims,
        &new_dims);
    g->m_vars.push_back(v);

    // Process newly-found dims
    while(!new_dims.empty())
    {
        dim* nd = *new_dims.begin();
        g->m_dims.push_back(nd);
        //nd->inc();
        g->m_hidden_dims.push_back(nd);

        new_dims.pop_front();
    } // end of while

    return 0;
}

int _cb_on_each_gd_dim(char *name, void *arg)
{
    eos2_group *g= (eos2_group*)arg;
    int32 s = GDdiminfo(g->m_handle->get(), name);
    unsigned int size = (unsigned int) s;

    g->_add_dim(name, size);
    return 0;
}

int _cb_on_each_sw_dim(char *name, void *arg)
{
    eos2_group *g= (eos2_group*)arg;
    int32 s = SWdiminfo(g->m_handle->get(), name);
    unsigned int size = (unsigned int) s;

    g->_add_dim(name, size);
    return 0;
}

eos2_dim* eos2_group::_add_dim(const std::string& name, unsigned int size) 
{
    eos2_dim *d = new eos2_dim(
        this,
        m_file_info,
        m_group_type,
        m_handle,
        name,
        name,
        size);
    this->m_dims.push_back(d);
    //d->inc();

    return d;
}
////////////////////////////////////////////////
// Swath callbacks
////////////////////////////////////////////////

////////////////////////////////////////////////

void eos2_group::_add_gd_ll() 
{
    // dims first
    //assert(m_hidden_dims.size() == 2 || m_hidden_dims.size() == 0); 
    if(m_hidden_dims.size() != 2) 
        if(m_hidden_dims.size() != 0) 
            throw std::runtime_error("The dimension size for added gd should be either 0 or 2. (" __FILE__ ":" TOSTRING(__LINE__)")");

    int32 projcode = -1; //Adding processing of SOM projection [05/17/2012 LD]
    int32 zone = 0;
    int32 sphere = 0;
    float64 params[16];
    GDprojinfo(m_handle->get(), &projcode, &zone, &sphere, params);

    if (GCTP_SOM == projcode) 
        throw std::runtime_error("Conversion to SOM(Space-oblique Mercator) projection files are not supported. (" __FILE__ ":" TOSTRING(__LINE__)")");

    std::list<dim*> dims;
    if(m_hidden_dims.size() == 2) 
    {
        // In this case, dims of lat/lon are YDim and XDim.
        dims = m_hidden_dims;
    } 
    else
    {
        int32 xdimsize = 0;
        int32 ydimsize = 0;
        float64 upleft[2];
        float64 lowright[2];
        intn ret = GDgridinfo(m_handle->get(), &xdimsize, &ydimsize, upleft, lowright);
        if(ret != 0)
            throw std::runtime_error("Cannot get grid information using GDgridinfo()(" __FILE__ ":" TOSTRING(__LINE__)")");

        dim *xdim = NULL;
        dim *ydim = NULL;
        ydim = get_dim_by_name("YDim");
        xdim = get_dim_by_name("XDim");
        if(ydim && xdim)
        {
            // OK
        }
        else if(xdimsize != ydimsize)
        {
            // match by size
            ydim = get_dim_by_size(ydimsize);
            xdim = get_dim_by_size(xdimsize);
        }
        else
        {
            // cannot figure out the dims
            throw std::runtime_error("Cannot differentiate XDim and YDim. They have the same size");
        }
        dims.push_back(ydim);
        dims.push_back(xdim);
    } // end of else

    std::string varname = "latitude";

    eos2_var_gd_ll *lat = new eos2_var_gd_ll(
        this,
        this->m_file_info,
        this->m_handle,
        varname,
        eos2_var_gd_ll::LATITUDE,
        dims);
    mem_attr *lat_attr = new mem_attr(lat, "eoslib");
    lat_attr->set_value("Calculated latitude");
    lat->add_attr(lat_attr);

    m_vars.push_back(static_cast<var*>(lat));

    varname = "longitude";

    eos2_var_gd_ll *lon = new eos2_var_gd_ll(
        this,
        this->m_file_info,
        this->m_handle,
        varname,
        eos2_var_gd_ll::LONGITUDE,
        dims);

    mem_attr *lon_attr = new mem_attr(lon, "eoslib");
    lon_attr->set_value("Calculated longitude");
    lon->add_attr(lon_attr);

    m_vars.push_back(static_cast<var*>(lon));

    // Coordinate variables of XDim and YDim
    /*
    std::list<dim*>::iterator it;
    for(it = dims.begin(); it!= dims.end(); it++)
    {
        eos2_dim* ed = dynamic_cast<eos2_dim*>(*it);
        ed->m_coord_vars.push_back(lat);
        ed->m_coord_vars.push_back(lon);
    }
    */
}

}	// namespace
