/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Category: Header file for the EOS2 module under the eoslib namespace

Description:

This file contains information of an HDF-EOS2 group class.


*****************************************************************************/

#ifndef EOS2_GROUP_H
#define EOS2_GROUP_H

#include <map>
/* #include <gtest/gtest.h> */
#include "eoslib_group.h"
#include "eos2_defs.h"
#include "eoslib_err.h"

namespace eoslib
{

//! HDF-EOS2 group (grid/swath)
/**
 * Class eos2_group implements eosgroup interface and
 * it contains all information needed for accessing a
 * eos2group.
 * Note that we are not showing the real HDF4 hierarchy
 * to the user. Top group is "/". Level two groups are
 * all swaths and grids. There is no group at level three.
 */

class eos2_group : public group
{
    public:
        class CannotOpenFileException: public std::runtime_error { 
            public: 
                CannotOpenFileException (std::string msg): std::runtime_error(msg) {} }; /* end of CannotOpenFileException */

        class InvalidEOS2FileException: public std::runtime_error { 
            public: 
                InvalidEOS2FileException (std::string msg): std::runtime_error(msg) {} }; /* end of InvalidEOS2FileException */

        class GroupNotExistsException: public std::runtime_error { 
            public: 
                GroupNotExistsException (std::string msg): std::runtime_error(msg) {} }; /* GroupNotExistsException */

        /**
         * Root group's m_handle is file handle, which is either swath file or 
         * grid file. Child group's m_handle is grid/swath handle. 
         * Note that this handle depends on the parent's handle.
         */
        eos2_group(const char *filename) throw(CannotOpenFileException, InvalidEOS2FileException); /* root group */

        /**
         * Child group's constructor
         */
        eos2_group(eos2_group* parent, eos2_group_type_t type, const char* name) throw(GroupNotExistsException);

        virtual ~eos2_group();
        typedef eos2_group_type_t group_type_t;

        inline group_type_t get_type() {return m_group_type;}

    public:
        inline std::list<dim*> get_hidden_dims() {return m_hidden_dims;}
	
        inline eos2_handle* get_handle() {return m_handle;}
        inline eos2_handle* get_sw_handle() {return m_hswf;}
        inline eos2_handle* get_gd_handle() {return m_hgdf;}

    private:
        eos2_dim* _add_dim(const std::string& name, unsigned int size);

        /**
         * Add two variables: "latitude" and "longitude" for grid since latitude and longitude are not
         * stored as variables. Instead, their values can be calculated based on parameters stored in the
         * file.
         */
        void _add_gd_ll();

    protected:
        /**
         * Only root group uses m_hgdf and m_hswf;
         */
        eos2_handle *m_hgdf;	
        eos2_handle *m_hswf;	

        /**
         * Only non-root groups use m_handle.
         */
        eos2_handle *m_handle;	

        group_type_t m_group_type;
        std::string m_eos2name;	/* swath or grid name */

        /**
         * Dims for grid coordinate are not returned from GDinqdims()
         * Grid dims (usually, YDim and XDim) are stored in this variable.
         * Invariant: all dims in m_hidden_dims are already in m_dims.
         */
        std::list<dim*> m_hidden_dims;	

        /* All of these are actually eos2_group, eos2_var, eos2_dim, and eos2_attr. */
        friend class eos2_var_data;
        friend class eos2_dim;
        friend class eos2_attr;
        friend int _cb_on_each_grid(char *gridname, void *arg);
        friend int _cb_on_each_swath(char *swathname, void *arg);
        friend int _cb_on_each_sw_datafield(char *name, void *arg);
        friend int _cb_on_each_sw_geofield(char *name, void *arg);
        friend int _cb_on_each_gd_datafield(char *name, void *arg);
        friend int _cb_on_each_sw_dim(char *name, void *arg);
        friend int _cb_on_each_gd_dim(char *name, void *arg);
        friend int _cb_on_each_gd_attr(char *name, void *arg);
        friend int _cb_on_each_sw_attr(char *name, void *arg);
        friend void read_eos2_attr_gd(eos2_group *g, std::list<attr*> *pattrs);
        friend void read_eos2_attr_sw(eos2_group *g, std::list<attr*> *pattrs);

    private:
        eos2_group();	/* No impl */
        eos2_group(const eos2_group&);
};

//Call-back functions to obtain grid and swath info.
int _cb_on_each_grid(char *gridname, void *arg);
int _cb_on_each_swath(char *swathname, void *arg);
int _cb_on_each_sw_datafield(char *name, void *arg);
int _cb_on_each_sw_geofield(char *name, void *arg);
int _cb_on_each_gd_datafield(char *name, void *arg);
int _cb_on_each_gd_dim(char *name, void *arg);


int _cb_on_each_sw_dim(char *name, void *arg);

void read_eos2_attr_gd(eos2_group *g, std::list<attr*> *pattrs);
void read_eos2_attr_sw(eos2_group *g, std::list<attr*> *pattrs);

} // namespace
#endif

