/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of an HDF-EOS2 handle (grid file handle, grid handle, swath file handle and swath handle).


*****************************************************************************/

#include "eos2_handle.h"
#include "eoslib_exception.h"
#include "eoslib_err.h"
#include <cassert>
#include <iostream>


namespace eoslib
{

eos2_handle::eos2_handle(const std::string& name, int32 handle, eos2_handle_type_t type, eos2_handle *dep):
    refcnt(),
    renamable(name),
    m_handle(handle),
    m_type(type)
{
    if(dep)
    {
        dep->inc(this);
        m_deps.push_back(dep);
    }
}

eos2_handle::eos2_handle(const std::string& name, int32 handle, eos2_handle_type_t type, const std::vector<eos2_handle*>& deps):
    refcnt(),
    renamable(name),
    m_handle(handle),
    m_type(type)
{
    std::vector<eos2_handle*>::const_iterator it;
    for(it = deps.begin(); it != deps.end(); it++)
    {
        (*it)->inc(this);
        m_deps.push_back(*it);
    }
}
				

eos2_handle::~eos2_handle()
{
    //std::cout << "Closing eos2_handle " << get_name() << "(" << m_type << ")" << std::endl;
    intn ret = -1;
    switch(m_type)
    {
        case GRID_FILE:
            //std::cout << "GDclose (" << get_name() << ")" << std::endl;
            ret = GDclose(m_handle);
            break;
        case SWATH_FILE:
            //std::cout << "SWclose (" << get_name() << ")" << std::endl;
            ret = SWclose(m_handle);
            break;
        case SINGLE_GRID:
            //std::cout << "GDdetach (" << get_name() << ")" << std::endl;
            ret = GDdetach(m_handle);
            break;
        case SINGLE_SWATH:
            //std::cout << "SWdetach (" << get_name() << ")" << std::endl;
            ret = SWdetach(m_handle);
            break;
        default:
            assert(0);
    } // end of switdh(m_type)

    std::vector<eos2_handle*>::iterator it;
    for(it = m_deps.begin(); it != m_deps.end(); it++)
    {
        (*it)->dec(this);
    }

    if(ret)
        throw std::runtime_error("Cannot close grid/swath (" __FILE__ ":" TOSTRING(__LINE__)")" );
}

int32 eos2_handle::get() const
{
    return m_handle;
}

} // namespace
