/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Category: Header file for the HDF-EOS2 handle of the eoslib namespace

Description:

This file is an HDF-EOS2 handle (grid file handle, grid handle, 
swath file handle and swath handle) definition.


*****************************************************************************/

#ifndef EOS2_HANDLE_H
#define EOS2_HANDLE_H

#include "eoslib_rc.h"
#include <vector>

/**
 * This class manages an eos2 handle (GRID ID, SWATH ID, FILE ID)
 * One ID can be shared by many objects and the number of
 * objects using this eos2_hanle is maintained by the ref_count.
 * When this eos2 handle is not used by any other, 
 * it is destroyed and appropriate close function is EOS2 API
 * is called.
 */

#include <hdf.h>
#include <HdfEosDef.h>
#include <mfhdf.h>

namespace eoslib 
{

//! EOS2-related handles
/**
 * This class holds HDF-EOS2 handles (grid file handle, grid handle,
 * swath file handle and swath handle)
 *
 * There is a dependency among handles. For example, grid handle depends
 * on the grid file handle. If grid file handle is released, grid handle
 * is useless. This class maintains this dependency information and
 * makes sure that a handle which another handle depends on is not
 * released.
 */
class eos2_handle: public refcnt, public renamable
{
     public:
         //! Type of the handle
         typedef enum {GRID_FILE, SWATH_FILE, SINGLE_GRID, SINGLE_SWATH} eos2_handle_type_t;
        /* Constructor */
        /**
         * \param dep The handle which the new handle depends on
         */
        eos2_handle(const std::string& name, int32 handle, eos2_handle_type_t type, eos2_handle *dep);
        /* Constructor */
        /**
         * \param deps The handles which the new handle depends on
         */
        eos2_handle(const std::string& name, int32 handle, eos2_handle_type_t type, const std::vector<eos2_handle*>& deps);
        //! Destructor
        virtual ~eos2_handle();
        /* Getting handle */
        int32 get() const;

    private:
        /* disabled constructors */
        eos2_handle();
        eos2_handle(const eos2_handle&);

        int32 m_handle;
        eos2_handle_type_t m_type;
	/**
         * List of eos2_handles which the current handle is depended on.
	 * e.g., grid handle depends on grid file handle 
         */
        std::vector<eos2_handle*> m_deps;
};

} // namespace

#endif

