/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of an HDF-EOS2
data field.


*****************************************************************************/

#include "eos2_var_data.h"
#include "eos2_defs.h"
#include "misc_util.h"
#include <stdexcept>
#include <assert.h>
#include <string>
#include <cstring>
#include <sstream>
#include <iostream>
#include "eoslib_err.h"
#include "mem_attr.h"

namespace eoslib
{
eos2_var_data::eos2_var_data(
    group *g,
    file_info *f,
    eos2_group_type_t group_type,
    eos2_handle *handle,
    const std::string& name,
    const std::list<dim*>& dims,	// in
    std::list<dim*>* new_dims // out
    ):
        var(g, name),
        m_file_info(f),
        m_group_type(group_type),
        m_handle(handle),
        m_eos2varname(name),
        order(0),
        total_bytes(0)
{
    m_file_info->inc(this);
    m_handle->inc(this);

    intn (*f_fieldinfo)(int32, char *, int32 *, int32 [], int32*, char*);
    intn (*f_getfillvalue)(int32, char*, void *);

    switch(m_group_type)
    {
        case GRID:
            f_fieldinfo = GDfieldinfo;
            f_getfillvalue = GDgetfillvalue;
            break;
        case SWATH:
            f_fieldinfo = SWfieldinfo;
            f_getfillvalue = SWgetfillvalue;
            break;
        default:
            throw std::range_error("A field's group is neither grid nor swath(" __FILE__ ":" TOSTRING(__LINE__)")" );
    } // end of switch(m_group_type)

    intn ret = -1;
    int32 rank = 0;

    // Assume the maximum number of dimension of HDF-EOS grids or swaths is 128.
    // This number should be enough since the maximum number of dimension for NASA EOS grid or swath is 4.
    int32 dimsbuf[128];  	

    // Assume the maximum length of dimension list is 1024. This should be enough for NASA HDF-EOS products.
    std::vector<char> dimlist(1024);	
    ret = f_fieldinfo(m_handle->get(), 
        const_cast<char*>(name.c_str()),
        &rank, dimsbuf, &m_value_type, &dimlist[0]);
	
    if(ret != 0)
        throw std::runtime_error("Canot get fieldinfo (" __FILE__ ":" TOSTRING(__LINE__)")");

    // Verification of dim sizes
    {
        std::vector<std::string> ds;
        util_split_str(&dimlist[0], ",", &ds);
        for(size_t i=0; i<ds.size(); i++)
        {
            dim *d = g->get_dim_by_name(ds[i]);

            if(d==NULL)
            {
                // OK
            }
            else if(d->get_size() == 0)
            {
                assert(dimsbuf[i] > 0);
                d->set_size(dimsbuf[i]);
            }
            else
            {
                assert((ssize_t)(d->get_size()) == dimsbuf[i]);
            }
        }
    } // end of Verification of dim sizes

    // Fillvalue: m_attrs
    {
        char fillvalue[8];
        ret = f_getfillvalue(m_handle->get(), const_cast<char*>(name.c_str()), fillvalue);

        if(ret == 0)
        {
            eos2_attr_fillvalue *fv = new eos2_attr_fillvalue(this, f, m_value_type, fillvalue);
            m_attrs.push_back(fv);
            //fv->inc();
        }
    } // end of Fillvalue: m_attrs

    // Priority for name conflict
    {
        mem_attr *attr2 = new mem_attr(this, "_e2lib_name_conflict_priority");
        std::ostringstream p;
        p << 1;
        attr2->set_value(p.str());
        this->add_attr(attr2);
    } // end of Priority for name conflict

    // dims: m_dims
    {
        std::vector<std::string> vdims;
        util_split_str(&dimlist[0], ",", &vdims);
        for(std::vector<std::string>::iterator it2 = vdims.begin();
            it2 != vdims.end();
            it2++)
        {
            const unsigned int dim_index = distance(vdims.begin(), it2);
            const std::string& ref_dim_name = *it2;

            dim *d = NULL;
            for(std::list<dim*>::const_iterator it = dims.begin();
                it != dims.end(); it++)
            {
                if((*it)->get_name() == ref_dim_name)
                {
                    d = *it;
                    break;
                }
            }
            if(d==NULL)
            {
                // Dimension has not defined in the file.
                // In this case, we add dimension.

                //std::cout << "Var name: " << name << std::endl;
                //std::cout << "Dim name: " << ref_dim_name << std::endl;
                //std::cout.flush();

                eos2_dim *nd = new eos2_dim(
                    g,
                    this->m_file_info,
                    this->m_group_type,
                    this->m_handle,
                    ref_dim_name,
                    ref_dim_name,
                    dimsbuf[dim_index]);

                new_dims->push_back(nd);

                m_dims.push_back(nd);
                //nd->inc();
            } 
            else
            {
                m_dims.push_back(d);
                //d->inc();
            } // end of if(d==NULL)
        } // end of for
    } // end  of dims: m_dims

// The following code is not used.
#if 0
    for(std::list<dim*>::iterator it = m_dims.begin();
        it != m_dims.end(); it++)
    {
        eos2_dim *e2d = dynamic_cast<eos2_dim*>(*it);
        //this->inc();
    }
#endif
}

eos2_var_data::eos2_var_data(const eos2_var_data& r): 
    var(r),
    m_file_info(r.m_file_info),
    m_group_type(r.m_group_type),
    m_handle(r.m_handle),
    m_eos2varname(r.m_eos2varname),
    m_value_type(r.m_value_type),
    order(r.order),
    total_bytes(r.total_bytes)
{
    m_file_info->inc(this);
    m_handle->inc(this);
	
    if(r.total_bytes!=0)
    {
        // For debugging only.
        //double *p = (double*)(r.ptr);
        //std::cout << "ptr[0]=" << (double)p[0] << std::endl;
        ptr = malloc(r.total_bytes);
        memcpy(ptr, r.ptr, r.total_bytes);
    }
}

// This is for the added objects by HDF4 APIs. The 
// current design makes these objects as HDF-EOS2 objects, which
// is conceptually not right. However, to change this
// requires significant efforts. So we still use this
// way. But need to add another member (newname) to
// the "var" class so we can remember the original name of
// the added object. KY 2013-01-04
eos2_var_data::eos2_var_data(
    group *g,
    file_info *f,
    eos2_handle *handle,
    const std::string& name,
    const std::string& origname,
    const std::vector<std::string> vdims,
    int32 *dimsbuf,
    value_type_t type,
    int32 array_size,
    int32 nr_bytes,
    void *p,
    const std::list<dim*>& dims,    // in
    std::list<dim*>* new_dims // out
    ):
        var(g, name,origname),
        m_file_info(f),
        m_handle(handle),
        m_eos2varname(name),
        m_eos2varorigname(origname),
        m_value_type(type),
        order(array_size),
        total_bytes(nr_bytes),
        ptr(p)
{
    eos2_group *eg = dynamic_cast<eos2_group *>(g);

    m_group_type = (eos2_group_type_t)(eg->get_type());

    // The reference count doesn't apply to the added HDF4 objects.
    // So we don't increase the count. Instead, we set the group type 
    // be NONEOS. We also set the disable_clone to be true since
    // these objects are under the root group already, they don't
    // need to be cloned. If these objects are cloned, the resources
    // may not be freed properly and memory leaking may occur.
    //m_file_info->inc(this);
    //m_handle->inc(this);
    m_group_type = NONEOS;
    disable_clone = true;
	
    // Priority for name conflict
    {
        mem_attr *attr2 = new mem_attr(this, "_e2lib_name_conflict_priority");
        std::ostringstream p_stream;
        p_stream << 1;
        attr2->set_value(p_stream.str());
        this->add_attr(attr2);
    } // end of Priority for name conflict

    // dims: m_dims
    {
        for(std::vector<std::string>::const_iterator it2 = vdims.begin(); it2 != vdims.end(); it2++)
        {
            const unsigned int dim_index = distance(vdims.begin(), it2);
            const std::string& ref_dim_name = *it2;

            dim *d = NULL;
            for(std::list<dim*>::const_iterator it = dims.begin(); it != dims.end(); it++)
            {
                if((*it)->get_name() == ref_dim_name)
                {
                    d = *it;
                    break;
                }
            } // end of for
            if(d==NULL)
            {
                // Dimension has not defined in the file. In this case, we add dimension.
                eos2_dim *nd = new eos2_dim(
                    g,
                    this->m_file_info,
                    this->m_group_type,
                    this->m_handle,
                    ref_dim_name,
                    ref_dim_name,
                    dimsbuf[dim_index]);

                    new_dims->push_back(nd);

                    m_dims.push_back(nd);

                    //nd->inc();
            }
            else
            {
                m_dims.push_back(d);

                //d->inc();
            } // end of if(d==NULL)
        } // end of for
    } // end of dims: m_dims

    // Not sure why the following block is there.
#if 0
    for(std::list<dim*>::iterator it = m_dims.begin(); it != m_dims.end(); it++)
    {
        eos2_dim *e2d = dynamic_cast<eos2_dim*>(*it);
        //this->inc();
    }
#endif

    free(dimsbuf);
}

eos2_var_data::~eos2_var_data()
{

    // The reference count needs to be decreased if the object type is NONEOS.
    if(m_group_type !=NONEOS) {
        m_file_info->dec(this);
        m_handle->dec(this);
    }

    while(!m_attrs.empty())
    {
        attr *a = *(m_attrs.begin());
        delete a;
        m_attrs.pop_front();
    }
	
    // We need to free the allocated data buffer.
    if(total_bytes!=0)
        free(ptr);
}

value_type_t eos2_var_data::get_type() const
{
    return m_value_type;
}

void eos2_var_data::get_value(int32 start[], int32 stride[], int32 edge[],
	void *buf) const 
{
    if(total_bytes!=0) //For newly added variables only, which are from HDF4.
    {
        // VDATA only, subsetted of vdata field is not supported. 
        if(m_dims.size()==0) // VDATA only(seems no subset KY 2014-12-5).
        {
            throw std::runtime_error("For an HDF-EOS2 file, the subset of a vdata field added by HDF4 APIs is not supported.(" __FILE__ ":" TOSTRING(__LINE__)")");
            //memcpy(buf, ptr, total_bytes);
        } else //SDS only.
        {
            std::vector<int32> dim_sizes = this->get_dim_sizes();
	
            int rank = dim_sizes.size(); 

            //subset2 needs the pos[], so still use malloc.
            int32 *pos = (int32*)malloc(rank*sizeof(int32)); 
		
            switch(m_value_type)
            {
#define HANDLE(TYPE, type) \
    case DFNT_##TYPE: \
    { \
        std::vector<type> tmp; \
        memcpy((void*)pos, (void*)start, rank*sizeof(int32)); \
        var::subset2<type>( \
            (type*)ptr,  \
            dim_sizes.size(), \
            &dim_sizes[0], \
            start, \
            stride, \
            edge, \
            &tmp, \
            pos, \
            0); \
        std::copy(tmp.begin(), tmp.end(), (type*)buf); \
        break; \
    }
                HANDLE(CHAR8, char8);
                HANDLE(INT8, int8);
                HANDLE(INT16, int16);
                HANDLE(INT32, int32);
                HANDLE(UCHAR8, uchar8);
                HANDLE(UINT8, uint8);
                HANDLE(UINT16, uint16);
                HANDLE(UINT32, uint32);
                HANDLE(FLOAT32, float32);
                HANDLE(FLOAT64, float64);
#undef HANDLE
                default:
                    throw std::range_error("Unknown type(" __FILE__ ":" TOSTRING(__LINE__)")" );
            } // end of switch
		
            free(pos);
        } // end of else //SDS only.

        // For debugging only.	
        //The following snippet for testing Noise in Thermal Detectors in MYD021KM.A2002226.0000.005.2009193222735.hdf.
        /*uint8 *p1 = static_cast<uint8*>(buf);
        uint8 *p2 = static_cast<uint8*>(ptr);
        std::cout << "[eos2_var_data]p1[0]=" << (unsigned int)*p1 << " p1[1]=" << (unsigned int)p1[1] << std::endl;
        std::cout << "[eos2_var_data]p2[0]=" << (unsigned int)*p2 << " p2[1]=" << (unsigned int)p2[1] << std::endl;*/	
        return;
    }

    int (*readfunc)(int32, char*, int32*, int32*, int32*, void*) = NULL;
    switch(m_group_type)
    {
        case GRID:
            readfunc = GDreadfield;
            break;
        case SWATH:
            readfunc = SWreadfield;
            break;
        default:
            throw std::range_error("Variable" +	this->get_name() + "is neither Grid nor Swath variable ("__FILE__ ":" TOSTRING(__LINE__)")");

    } // end of switch

    // XXX. Group type check

    intn r = -1;

    r = readfunc(m_handle->get(), const_cast<char*>(m_eos2varname.c_str()), start, stride, edge, buf);
    if(-1 == r)
        throw std::runtime_error("Cannot read field " +	this->get_name() + " (" __FILE__ ":" TOSTRING(__LINE__)")");
}

var* eos2_var_data::_clone() const
{
    eos2_var_data *w = new eos2_var_data(*this);
    return w;
}

bool eos2_var_data::same_obj_test(var* v) const
{
    eos2_var_data *w = dynamic_cast<eos2_var_data*>(v);
    //std::cout << typeid(*this).name() << std::endl;
    //std::cout << typeid(*w).name() << std::endl;
    //std::cout << typeid(*v).name() << std::endl;
    //if(typeid(*this) != typeid(*w))
    //	return false;
    if(w == NULL)
        return false;
    if(this->m_handle != w->m_handle)
        return false;
    if(this->m_eos2varname != w->m_eos2varname)
        return false;
    return true;
}

} // namespace
