/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file incluides the implementation related to the processing of 
an HDF-EOS2 swath dimension-mapped data field.


*****************************************************************************/

#include "eos2_var_dm.h"
#include "eos2_defs.h"
#include "mem_attr.h"
#include "misc_util.h"
#include <cstring>
#include <sstream>
#include <iostream>
#include <cassert>

namespace eoslib
{

static void get_dimmap(
    std::vector<std::pair<std::string, std::string> > *pdimmap,
    const std::string& dimmapstr)
{
    std::vector<std::string> p;
    util_split_str(dimmapstr, ",", &p);

    for(std::vector<std::string>::iterator it = p.begin(); it != p.end(); it++)
    {
        std::vector<std::string> q;
        util_split_str(*it, "/", &q);

        if(q.size() == 2)
        {
            std::string geo = q[0];
            std::string dat = q[1];
            pdimmap->push_back(std::make_pair(geo, dat));
        }
        else
        {
            assert(0);
        }
    } // end of for
}

// Find index of the reduced dime
static int search_dimmap( 
    const std::vector<std::pair<std::string, std::string> >& dimmap,
    const std::string& reduced_dim)
{
    int i=0;
    for(i=0; i<dimmap.size(); i++)
    {
        if(reduced_dim == dimmap[i].first)
            return i;
    }
    return -1;
}

static dim* find_dim(
    const std::list<dim*>& dims,	
    const std::string& name)
{
    std::list<dim*>::const_iterator it;
    for(it = dims.begin(); it != dims.end(); it++)
    {
        if((*it)->get_name() == name)
            return *it;
    }
    return NULL;
}


//! Expand dimmand field
/**
 *  This function expands a reduced (dimmapped) fields along one dimension.
 *  \param pvals64 original values
 *  \param rank Rank
 *  \param dimsa Original field's dimension
 *  \param dimindex Dimension's index which needs to be expanded
 *  \param ddimsize New size of the expanded dimension
 *  \param offset offset of the dimension map
 *  \param inc increment of the dimension map
 */
template<typename T>
int _expand_dimmap_field(std::vector<T> *pvals64, int32 rank, int32 dimsa[], int dimindex, int32 ddimsize, int32 offset, int32 inc)
{
    int32 offset1 = 0;
    int32 inc1 = 0;
    if(inc<0)
    {
        offset1 = (-1)*offset;
        inc1 = (-1)*inc;
    }		

    if(inc == 0)
        return -1;

    std::vector<T> orig = *pvals64;

    std::vector<int32> pos;
    std::vector<int32> dims;
    std::vector<int32> newdims;
    pos.resize(rank);
    dims.resize(rank);
    for(int i=0; i<rank; i++)
    {
        pos[i]=0;
        dims[i]=dimsa[i];
    }
    newdims=dims;
    newdims[dimindex]=ddimsize;
    dimsa[dimindex]=ddimsize;

    int newsize=1;
    for(int i=0; i<rank; i++)
    {
        newsize *= newdims[i];
    }
    pvals64->clear();

    // The buffer is enlarged for expansion.
    pvals64->resize(newsize);
	
    for(;;)
    {
        // if end
        if(pos[0] == dims[0])
        {
            // we past then end
            break;
        }
        else if(pos[dimindex]==0) 
        {
            // extract 1D values				
            std::vector<T> v;
            for(int i=0; i<dims[dimindex]; i++)
            {
                pos[dimindex]=i;
                v.push_back(orig[var::INDEX_nD_TO_1D(dims, pos)]);
            } // end of for 
            // expand them
            std::vector<T> w;
            for(int32 j=0; j<ddimsize; j++)
            {
                int32 i=(j-offset)/inc;

                if(inc<0)
                    i = offset1 + j*inc1;

                T f;
                if(i*inc+offset == j || inc < 0) // perfect match
                {
                    f = (v[i]);
                }
                else
                {
                    int32 i1 = 0;
                    int32 i2 = 0;
                    int32 j1 = 0; 
                    int32 j2 = 0;
                    if(i<=0)
                    {
                        i1=0;
                        i2=1;
                    }
                    if((unsigned int)i+1>=v.size())
                    {
                        i1=v.size()-2;
                        i2=v.size()-1;
                    }
                    else
                    {
                        i1=i;
                        i2=i+1;
                    } // end  of if((unsigned int)i+1>=
                    j1 = i1 * inc + offset;
                    j2 = i2 * inc + offset;

                    // Using the linear interpolation method.
                    f = (((j-j1)*v[i2] + (j2-j)*v[i1])/(j2-j1));
                } // end of if(i*inc+offset == j || 
                w.push_back(f);
                pos[dimindex]=j;
                (*pvals64)[var::INDEX_nD_TO_1D(newdims, pos)]=f;
            } //end of for(int32 j=0; j<ddimsi
            pos[dimindex]=0;
        } // end of if(pos[0] == 
        // next pos
        pos[rank-1]++;

        for(int i=rank-1; i>0; i--)
        {
            if(pos[i]==dims[i])
            {
                pos[i]=0;
                pos[i-1]++;
            }
        }
    } // end of for(;;)
    return 0;
}

eos2_var_dm::eos2_var_dm(
    group *g,
    const std::string& name,
    const eos2_var_geo *v,
    const std::list<dim*>& dims,	// in
    std::list<dim*>* p_hidden_dims	// in/out
    ):
        var(g, name),
        m_file_info(v->m_file_info),
        m_eos2varname(v->get_name()),
        m_handle(v->m_handle),
        m_value_type(v->m_value_type)					 
{
    m_file_info->inc(this);
    m_handle->inc(this);

    int32 r = -1;
    std::vector<char> dimmapstr(1024);
    std::vector<int32> offset(1024);
    std::vector<int32> increment(1024);
    r = SWinqmaps(m_handle->get(), &dimmapstr[0], &offset[0], &increment[0]);
    if(r == -1) 
            throw std::runtime_error("SWinqmaps failed (" __FILE__ ":" TOSTRING(__LINE__)")");

    std::vector<std::pair<std::string, std::string> > dimmap;
    get_dimmap(&dimmap, &dimmapstr[0]);

    // set m_dims
    const std::list<dim*>& reduced_dims = v->get_dims_c();
    std::list<dim*>::const_iterator dit;
    for(dit = reduced_dims.begin(); dit != reduced_dims.end(); dit++)
    {
        int i = search_dimmap(dimmap, (*dit)->get_name());
        if(i==-1)
        {
            // Not found
            m_dims.push_back(*dit);
            m_reduced_size.push_back((*dit)->get_size());
            m_expanded_size.push_back((*dit)->get_size());
            m_offset.push_back(0);
            m_inc.push_back(0);
        }
        else
        {
            dim *d = find_dim(dims, dimmap[i].second);
            assert(d);
            m_dims.push_back(d);
            m_reduced_size.push_back((*dit)->get_size());
            m_expanded_size.push_back(d->get_size());
            m_offset.push_back(offset[i]);
            m_inc.push_back(increment[i]);
        } // end of if(i==-1)
    } // end of for

    std::ostringstream oss;
    oss << "This is a expanded geo field (dimension map) from ";
    oss << v->get_name();

    mem_attr *attr = new mem_attr(this, "dimmap");
    attr->set_value(oss.str());
    this->add_attr(attr);

    // Priority for name conflict
    {
        mem_attr *attr2 = new mem_attr(this, "_e2lib_name_conflict_priority");
        std::ostringstream p;
        p << 4;
        attr2->set_value(p.str());
        this->add_attr(attr2);
    } // end of Priority for name conflict
}

eos2_var_dm::eos2_var_dm(
    group *g,
    const std::string& name,
    const eos2_var_data *v,
    const std::list<dim*>& dims,	// in
    std::list<dim*>* p_hidden_dims	// in/out
    ):
        var(g, name),
        m_file_info(v->m_file_info),
        m_eos2varname(v->get_name()),
        m_handle(v->m_handle),
        m_value_type(v->m_value_type)					 
{
    m_file_info->inc(this);
    m_handle->inc(this);

    int32 r = -1;
    std::vector<char> dimmapstr(1024);
    std::vector<int32> offset(1024);
    std::vector<int32> increment(1024);
    r = SWinqmaps(m_handle->get(), &dimmapstr[0], &offset[0], &increment[0]);
    //assert(r>0);
    if(r == -1) 
            throw std::runtime_error("SWinqmaps failed (" __FILE__ ":" TOSTRING(__LINE__)")");


    std::vector<std::pair<std::string, std::string> > dimmap;
    get_dimmap(&dimmap, &dimmapstr[0]);

    // set m_dims
    const std::list<dim*>& reduced_dims = v->get_dims_c();
    std::list<dim*>::const_iterator dit;
    for(dit = reduced_dims.begin(); dit != reduced_dims.end(); dit++)
    {
        int i = search_dimmap(dimmap, (*dit)->get_name());
        if(i==-1)
        {
            // Not found
            m_dims.push_back(*dit);
            m_reduced_size.push_back((*dit)->get_size());
            m_expanded_size.push_back((*dit)->get_size());
            m_offset.push_back(0);
            m_inc.push_back(0);
        }
        else
        {
            dim *d = find_dim(dims, dimmap[i].second);
            assert(d);
            m_dims.push_back(d);
            m_reduced_size.push_back((*dit)->get_size());
            m_expanded_size.push_back(d->get_size());
            m_offset.push_back(offset[i]);
            m_inc.push_back(increment[i]);
        } // end of if(i==-1)
    } // end of for

    std::ostringstream oss;
    oss << "This is a expanded data field (dimension map) from ";
    oss << v->get_name();

    mem_attr *attr = new mem_attr(this, "dimmap");
    attr->set_value(oss.str());
    this->add_attr(attr);

    // Priority for name conflict
    {
        mem_attr *attr2 = new mem_attr(this, "_e2lib_name_conflict_priority");
        std::ostringstream p;
        p << 4;
        attr2->set_value(p.str());
        this->add_attr(attr2);
    } // end of Priority for name conflict
}
eos2_var_dm::eos2_var_dm(const eos2_var_dm& r): 
    var(r),
    m_file_info(r.m_file_info),
    m_group_type(r.m_group_type),
    m_handle(r.m_handle),
    m_eos2varname(r.m_eos2varname),
    m_value_type(r.m_value_type),
    m_reduced_size(r.m_reduced_size),
    m_expanded_size(r.m_expanded_size),
    m_offset(r.m_offset),
    m_inc(r.m_inc)
{
    m_file_info->inc(this);
    m_handle->inc(this);
}

eos2_var_dm::~eos2_var_dm()
{
    m_file_info->dec(this);
    m_handle->dec(this);
}

value_type_t eos2_var_dm::get_type() const
{
    return m_value_type;
}

void eos2_var_dm::get_value(int32 start[], int32 stride[], int32 edge[],
    void *buf) const
{
    int32 num=1;
    for(int i=0; i<m_reduced_size.size(); i++)
        num *= m_reduced_size[i];

    intn r = -1;
    if(m_value_type == DFNT_FLOAT32)
    {
        std::vector<float> tmp;
        tmp.resize(num);
        r = SWreadfield(m_handle->get(), const_cast<char*>(m_eos2varname.c_str()), NULL, NULL, NULL, (void*)&tmp[0]);
        if(r <0) 
            throw std::runtime_error("Fail to obtain field values. (" __FILE__ ":" TOSTRING(__LINE__)")" );
 
        std::vector<int32> d = m_reduced_size;

        for(int i=0; i<m_reduced_size.size(); i++)
        {
            if(m_inc[i] != 0)
            {
                _expand_dimmap_field<float>(&tmp, m_reduced_size.size(), &d[0],
                    i, m_expanded_size[i],
                    m_offset[i],
                    m_inc[i]);
            }
        } // end of for

        // Subsetting
        std::vector<float> out;
        subset<float>(
            &tmp[0], 
            (int32)m_expanded_size.size(), 
            &m_expanded_size[0],
            &start[0],
            &stride[0],
            &edge[0],
            &out);
        memcpy(buf, &out[0], sizeof(float) * out.size());
    } // end of if(m_value_type == DFNT_FLOAT32)
    else if(m_value_type == DFNT_FLOAT64)
    {
        std::vector<double> tmp;
        tmp.resize(num);
        r = SWreadfield(m_handle->get(), const_cast<char*>(m_eos2varname.c_str()), NULL, NULL, NULL, (void*)&tmp[0]);
        if(r <0) 
            throw std::runtime_error("Fail to obtain field values. (" __FILE__ ":" TOSTRING(__LINE__)")" );
 
        std::vector<int32> d = m_reduced_size;

        for(int i=0; i<m_reduced_size.size(); i++)
        {
            if(m_inc[i] != 0)
            {
                _expand_dimmap_field<double>(&tmp, m_reduced_size.size(), &d[0],
                    i, m_expanded_size[i],
                    m_offset[i],
                    m_inc[i]);
            }
        } // end of for

        // Subsetting
        std::vector<double> out;
        subset<double>(
            &tmp[0], 
            (int32)m_expanded_size.size(), 
            &m_expanded_size[0],
            start,
            stride,
            edge,
            &out);
        memcpy(buf, &out[0], sizeof(double) * out.size());
    } // end of if(m_value_type == DFNT_FLOAT64)
    else if(m_value_type == DFNT_INT32)
    {
        std::vector<int32> tmp;
        tmp.resize(num);
        r = SWreadfield(m_handle->get(), const_cast<char*>(m_eos2varname.c_str()), NULL, NULL, NULL, (void*)&tmp[0]);
        if(r <0) 
            throw std::runtime_error("Fail to obtain field values. (" __FILE__ ":" TOSTRING(__LINE__)")" );
        std::vector<int32> d = m_reduced_size;

        for(int i=0; i<m_reduced_size.size(); i++)
        {
            if(m_inc[i] != 0)
            {
                _expand_dimmap_field<int32>(&tmp, m_reduced_size.size(), &d[0], i, m_expanded_size[i], m_offset[i], m_inc[i]);
            }
        } // end of for

        // Subsetting
        std::vector<int32> out;
        subset<int32>(&tmp[0],(int32)m_expanded_size.size(), &m_expanded_size[0], start, stride, edge, &out);
        memcpy(buf, &out[0], sizeof(int32) * out.size());
    } // end of if(m_value_type == DFNT_INT32)
    else if(m_value_type == DFNT_UINT32)
    {
        std::vector<uint32> tmp;
        tmp.resize(num);
        r = SWreadfield(m_handle->get(), const_cast<char*>(m_eos2varname.c_str()), NULL, NULL, NULL, (void*)&tmp[0]);
        if(r <0) 
            throw std::runtime_error("Fail to obtain field values.(" __FILE__ ":" TOSTRING(__LINE__)")" );
 
        std::vector<int32> d = m_reduced_size;

        for(int i=0; i<m_reduced_size.size(); i++)
        {
            if(m_inc[i] != 0)
            {
                _expand_dimmap_field<uint32>(&tmp, m_reduced_size.size(), &d[0], i, m_expanded_size[i], m_offset[i], m_inc[i]);
            }
        } // end of for

        // Subsetting
        std::vector<uint32> out;
        subset<uint32>(&tmp[0],(int32)m_expanded_size.size(), &m_expanded_size[0], start, stride, edge, &out);
        memcpy(buf, &out[0], sizeof(uint32) * out.size());
    } // end of if(m_value_type == DFNT_INT16)
    else if(m_value_type == DFNT_INT16)
    {
        std::vector<int16> tmp;
        tmp.resize(num);
        r = SWreadfield(m_handle->get(), const_cast<char*>(m_eos2varname.c_str()), NULL, NULL, NULL, (void*)&tmp[0]);
        if(r <0) 
            throw std::runtime_error("Fail to obtain field values.(" __FILE__ ":" TOSTRING(__LINE__)")" );
        std::vector<int32> d = m_reduced_size;

        for(int i=0; i<m_reduced_size.size(); i++)
        {
            if(m_inc[i] != 0)
            {
                _expand_dimmap_field<int16>(&tmp, m_reduced_size.size(), &d[0], i, m_expanded_size[i], m_offset[i], m_inc[i]);
            }
        } // end of for

        // Subsetting
        std::vector<int16> out;
        subset<int16>(&tmp[0],(int32)m_expanded_size.size(), &m_expanded_size[0], start, stride, edge, &out);
        memcpy(buf, &out[0], sizeof(int16) * out.size());
    } // end of if(m_value_type == DFNT_INT16)
    else if(m_value_type == DFNT_UINT16)
    {
        std::vector<uint16> tmp;
        tmp.resize(num);
        r = SWreadfield(m_handle->get(), const_cast<char*>(m_eos2varname.c_str()), NULL, NULL, NULL, (void*)&tmp[0]);
        if(r <0) 
            throw std::runtime_error("Fail to obtain field values.(" __FILE__ ":" TOSTRING(__LINE__)")" );
 
        std::vector<int32> d = m_reduced_size;

        for(int i=0; i<m_reduced_size.size(); i++)
        {
            if(m_inc[i] != 0)
            {
                _expand_dimmap_field<uint16>(&tmp, m_reduced_size.size(), &d[0], i, m_expanded_size[i], m_offset[i], m_inc[i]);
            }
        } // end of for

        // Subsetting
        std::vector<uint16> out;
        subset<uint16>(&tmp[0],(int32)m_expanded_size.size(), &m_expanded_size[0], start, stride, edge, &out);
        memcpy(buf, &out[0], sizeof(uint16) * out.size());
    } // end of if(m_value_type == DFNT_UINT16)
    else if(m_value_type == DFNT_UINT8)
    {
        std::vector<uint8> tmp;
        tmp.resize(num);
        r = SWreadfield(m_handle->get(), const_cast<char*>(m_eos2varname.c_str()), NULL, NULL, NULL, (void*)&tmp[0]);
        if(r <0) 
            throw std::runtime_error("Fail to obtain field values.(" __FILE__ ":" TOSTRING(__LINE__)")" );
 
        std::vector<int32> d = m_reduced_size;

        for(int i=0; i<m_reduced_size.size(); i++)
        {
            if(m_inc[i] != 0)
            {
                _expand_dimmap_field<uint8>(&tmp, m_reduced_size.size(), &d[0], i, m_expanded_size[i], m_offset[i], m_inc[i]);
            }
        } // end of for

        // Subsetting
        std::vector<uint8> out;
        subset<uint8>(&tmp[0],(int32)m_expanded_size.size(), &m_expanded_size[0], start, stride, edge, &out);
        memcpy(buf, &out[0], sizeof(uint8) * out.size());
    } // end of if(m_value_type == DFNT_INT16)   

    else if(m_value_type == DFNT_INT8)
    {
        std::vector<int8> tmp;
        tmp.resize(num);
        r = SWreadfield(m_handle->get(), const_cast<char*>(m_eos2varname.c_str()), NULL, NULL, NULL, (void*)&tmp[0]);
        if(r <0) 
            throw std::runtime_error("Fail to obtain field values.(" __FILE__ ":" TOSTRING(__LINE__)")" );
        std::vector<int32> d = m_reduced_size;

        for(int i=0; i<m_reduced_size.size(); i++)
        {
            if(m_inc[i] != 0)
            {
                _expand_dimmap_field<int8>(&tmp, m_reduced_size.size(), &d[0], i, m_expanded_size[i], m_offset[i], m_inc[i]);
            }
        } // end of for

        // Subsetting
        std::vector<int8> out;
        subset<int8>(&tmp[0],(int32)m_expanded_size.size(), &m_expanded_size[0], start, stride, edge, &out);
        memcpy(buf, &out[0], sizeof(int8) * out.size());
    } // end of if(m_value_type == DFNT_INT16)   

    else{
        assert(0);
        // throw exxeption
    }
}

var* eos2_var_dm::_clone() const
{
    eos2_var_dm *w = new eos2_var_dm(*this);
    return w;
}

bool eos2_var_dm::same_obj_test(var* v) const
{
    // Not necessary, may be an error.
    assert(0);
	return true;
}

} // namespace
