/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of an HDF-EOS2
grid geo-location field.

*****************************************************************************/

#include "eos2_var_gd_ll.h"
#include "eoslib_err.h"
#include "mem_attr.h"
#include <assert.h>
#include <string>
#include <iostream>
#include <sstream>

namespace eoslib
{

extern void calculate_special_lat_lon(int, int32, int32, float64*, int32*, int32*, int32*); 

eos2_var_gd_ll::eos2_var_gd_ll(const eos2_var_gd_ll& r):
    var(r),
    m_file_info(r.m_file_info),
    m_handle(r.m_handle),
    m_type(r.m_type),
    m_lat(NULL),
    m_lon(NULL),
    m_ydim(0),
    m_xdim(0) 
{
    m_file_info->inc(this);
    m_handle->inc(this);
}

eos2_var_gd_ll::eos2_var_gd_ll(
    group *g,
    file_info *f,
    eos2_handle *handle,
    const std::string& nickname,
    type_t type,
    std::list<dim*>& dims):
        var(g, nickname),
        m_file_info(f),
        m_handle(handle),
        m_type(type),
        m_lat(NULL),
        m_lon(NULL),
        m_ydim(0),
        m_xdim(0) 
{
    m_file_info->inc(this);
    m_handle->inc(this);

    std::list<dim*>::iterator it;
    for(it = dims.begin(); it != dims.end(); it++)
    {
        m_dims.push_back(*it);
        //(*it)->inc();
    }

    // Priority for name conflict
    {
        mem_attr *attr2 = new mem_attr(this, "_e2lib_name_conflict_priority");
        std::ostringstream p;
        p << 3;
        attr2->set_value(p.str());
        this->add_attr(attr2);
    } // end of Priority for name conflict

    this->set_cv();
}

eos2_var_gd_ll::~eos2_var_gd_ll()
{
    m_file_info->dec(this);
    m_handle->dec(this);

    if(m_lat)
        delete [] m_lat;
    if(m_lon)
        delete [] m_lon;
}

value_type_t eos2_var_gd_ll::get_type() const
{
    return DFNT_FLOAT64;
}

/**
 * NOTE that we're not considering XDim/YDim major issues.
 * This may need to be fixed.
 */
void eos2_var_gd_ll::get_value(int32 start[], int32 stride[], int32 edge[],
		void *buf) const
{
    //float64 upleft[2];
    //float64 lowright[2];

    if(!m_lat || !m_lon)
    {
        int32 r = -1;
        /* Retrieve dimensions and X-Y coordinates of corners from 'AerosolParameterAverage' */
        int32 xdim = 0;
        int32 ydim = 0;
        float64 upleft[2];
        float64 lowright[2];
        r = GDgridinfo(m_handle->get(), &xdim, &ydim, upleft, lowright);
        if(-1 == r)
            throw std::runtime_error("GDgridinfo failed(" __FILE__ ":" TOSTRING(__LINE__)")");

        /* Retrieve all GCTP projection information from 'AerosolParameterAverage' */
        int32 projcode = -1;;
        int32 zone = 0;
        int32 sphere = 0;
        float64 params[16];
        r = GDprojinfo(m_handle->get(), &projcode, &zone, &sphere, params);
        if(-1 == r)
            throw std::runtime_error("GDprojinfo failed(" __FILE__ ":" TOSTRING(__LINE__)")");

        /* Retrieve pixel registration information from 'AerosolParameterAverage' */
        int32 pixreg = 0;
        r = GDpixreginfo(m_handle->get(), &pixreg);
        if(-1 == r)
            throw std::runtime_error("GDpixreginfo failed(" __FILE__ ":" TOSTRING(__LINE__)")");

       

        /* Retrieve grid pixel origin from 'AerosolParameterAverage' */
        int32 origin = 0;
        r = GDorigininfo(m_handle->get(), &origin);
        if(-1 == r)
            throw std::runtime_error("GDorigininfo failed(" __FILE__ ":" TOSTRING(__LINE__)")");

        
        /* Allocate and fill row and col. How to fill will be explained later! */
        int32 *row = new int32 [xdim * ydim];
        int32 *col = new int32 [xdim * ydim];
        /* TODO: fill row and col */

        /* Allocate lon and lat. These will be filled by GDij2ll. */

        float64 *lon = new float64 [xdim * ydim];
        float64 *lat = new float64 [xdim * ydim];

        int i = 0;
        int j = 0;
        int k = 0; //y dim major
        for (k = j = 0; j < ydim; ++j) {
            for (i = 0; i < xdim; ++i) {
                row[k] = j;
                col[k] = i;
                ++k;
            }
        } // end of for

        if(projcode==GCTP_GEO && //0 && //Projection=GCTP_GEO 
           (int)upleft[0]==0&&(int)upleft[1]==0&&
           (int)lowright[0]==0&&(int)lowright[1]==0) //Special handling for CERES TRMM's default coordinate values.
        {
            upleft[0] = -180000000.0;
            upleft[1] = 90000000.0;
            lowright[0] = 180000000.0;
            lowright[1] = -90000000.0;
       } // end of  if(projcode==GCTP_GEO

       if(projcode==GCTP_GEO && //0 && //Projection=GCTP_GEO
          ((int)(lowright[0]/1000)==0) &&((int)(upleft[0]/1000)==0) &&
          ((int)(upleft[1]/1000)==0) && ((int)(lowright[1]/1000)==0)) // Special handling for MCD43's small numbers. 
        {
            upleft[0] = upleft[0] * 1000000;
            upleft[1] = upleft[1] * 1000000;
            lowright[0] = lowright[0] * 1000000;
            lowright[1] = lowright[1] * 1000000;
        } // end of if(projcode==GCTP_GEO

        /* Retrieve lon/lat values for 'AerosolParameterAverage' */
        if(projcode!=-1)
        {
            r = GDij2ll(projcode, zone, params, sphere, xdim, ydim, upleft, lowright, xdim * ydim, row, col, lon, lat, pixreg, origin);
         if(-1 == r)
            throw std::runtime_error("GDprojinfo failed(" __FILE__ ":" TOSTRING(__LINE__)")");

        } else  //Special handling for MOD13C2(grid no projection code).
        {
            calculate_special_lat_lon(LATITUDE, xdim, ydim, lat, start, edge, stride);

            calculate_special_lat_lon(LONGITUDE, xdim, ydim, lon, start, edge, stride);

        } //end of if(projcode!=-1)

        // Special handing for GCTP_LAMAZ projection.
        if(projcode==GCTP_LAMAZ)
        {
            float64 *lon2 = new float64 [xdim * ydim];
            float64 *lat2 = new float64 [xdim * ydim];
			
            for(k=0; k<xdim*ydim; k++)
            {
                lon2[k] = lon[k];
                lat2[k] = lat[k];
            }				
			
            for(i=0; i<ydim; i++)
                for(j=0; j<xdim; j++)
                    if(isundef_lat(lat[i*xdim+j]))
                        lat[i*xdim+j]=nearestNeighborLatVal(lat2, i, j, ydim, xdim);
			
            for(i=0; i<ydim; i++)
                for(j=0; j<xdim; j++)
                    if(isundef_lon(lon[i*xdim+j]))
                        lon[i*xdim+j]=nearestNeighborLonVal(lon2, i, j, ydim, xdim);
				
            delete [] lon2;
            delete [] lat2;	
        } // end of if(projcode==GCTP_LAMAZ)

        delete [] row;
        delete [] col;

        if(r==0)
        {
            m_lat = lat;
            m_lon = lon;
            m_ydim = ydim;
            m_xdim = xdim;
        }
        else
            throw std::runtime_error("Cannot calcualte latitude/longitude (" __FILE__ ":" TOSTRING(__LINENO__)")");


        //For AMSR_E_L3_RainGrid_V06_200206, the parameter is 'UpperLeftPointMtrs=(0.000000,70000000.000000) and LowerRightMtrs=(360000000.000000,-70000000.000000)'.  That means longitude ranges from 0 degree to 360 degree. However, many tools expect that the range of longitude is from -180 degree to 180 degree.
        if((int)upleft[0]==0 && (int)lowright[0]==360000000 && m_type==LONGITUDE)
        {
            for(i=start[0]; i<edge[0]; i+=stride[0])
                for(j=start[1]; j<edge[1]; j+=stride[1])
                {
                    if(m_lon[m_xdim * i + j]>180)
                        m_lon[m_xdim * i + j] -= 360.0;
                }
        }
    } // end of if(!m_lat || !m_lon)

    size_t i =0;
    size_t j =0;
    size_t k =0;
    float64 *outptr= (float64*)(buf);
    if(m_type==LATITUDE)
    {
        for(i=start[0]; i<edge[0]; i+=stride[0])
            for(j=start[1]; j<edge[1]; j+=stride[1])
            {
                outptr[k] = m_lat[m_xdim * i + j];
                k++;
            }
    }
    else if(m_type==LONGITUDE)
    {
        for(i=start[0]; i<edge[0]; i+=stride[0])
            for(j=start[1]; j<edge[1]; j+=stride[1])
            {
                outptr[k] = m_lon[m_xdim * i + j];
                k++;
            }
    } // end of else if(m_type==LONGITUDE)
}

var* eos2_var_gd_ll::_clone() const
{
    eos2_var_gd_ll *w = new eos2_var_gd_ll(*this);
    return w;
}

bool eos2_var_gd_ll::same_obj_test(var* v) const
{
    eos2_var_gd_ll *w = dynamic_cast<eos2_var_gd_ll*>(v);
    //if(typeid(*this) != typeid(*w))
    //	return false;
    if(w==NULL)
        return false;
    if(this->m_handle != w->m_handle)
        return false;
    if(this->m_eos2varname != w->m_eos2varname)
        return false;
    return true;
}

} // namespace
