/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of an HDF-EOS2
swath geo-location field.

*****************************************************************************/

#include "eos2_var_geo.h"
#include "eos2_defs.h"
#include "misc_util.h"
#include "eoslib_err.h"
#include "mem_attr.h"
#include <stdexcept>
#include <assert.h>
#include <string>
#include <iostream>
#include <sstream>

namespace eoslib
{

eos2_var_geo::eos2_var_geo(
    group *g,
    file_info *f,
    eos2_handle *handle,
    const std::string& name,
    const std::list<dim*>& dims,	// in
    std::list<dim*>* new_dims // out
    ):
        var(g, name),
        m_file_info(f),
        m_eos2varname(name),
        m_handle(handle)

{
    m_file_info->inc(this);
    m_handle->inc(this);

    intn ret = -1;
    int32 rank = 0;

    // This number should be enough since the maximum number of dimension for NASA EOS grid or swath is 4.
    int32 dimsbuf[128];  	

    // Assume the maximum length of dimension list is 1024. This should be enough for NASA HDF-EOS products.
    std::vector<char> dimlist(1024);

    ret = SWfieldinfo(m_handle->get(), const_cast<char*>(name.c_str()), &rank, dimsbuf, &m_value_type, &dimlist[0]);
    if(ret != 0)
        throw std::runtime_error("Canot get fieldinfo"  "(" __FILE__ ":" TOSTRING(__LINE__)")");

    // Verification of dim sizes
    {
        std::vector<std::string> ds;
        util_split_str(&dimlist[0], ",", &ds);
	
        for(int i=0; i<ds.size(); i++)
        {
            dim *d = g->get_dim_by_name(ds[i]);

            if(d!=NULL) {
            
            
                if(d->get_size() == 0)
                {
                    assert(dimsbuf[i] > 0);
                    d->set_size(dimsbuf[i]);
                }
                else
                {
                    assert(d->get_size() == dimsbuf[i]);
                }
            }
        }
    } // end of Verification of dim sizes

    // Fillvalue: m_attrs
    {
        char fillvalue[8];
        ret = SWgetfillvalue(m_handle->get(), const_cast<char*>(name.c_str()), fillvalue);
        if(ret == 0)
        {
            eos2_attr_fillvalue *fv = new eos2_attr_fillvalue(this, f, m_value_type, fillvalue);
            m_attrs.push_back(fv);
            //fv->inc();
        }
    } // end of Fillvalue: m_attrs

    // dims: m_dims
    {
        std::vector<std::string> vdims;
        util_split_str(&dimlist[0], ",", &vdims);
        for(std::vector<std::string>::iterator it2 = vdims.begin();
            it2 != vdims.end();
            it2++)
        {
            const unsigned int dim_index = distance(vdims.begin(), it2);
            const std::string& ref_dim_name = *it2;

            dim *d = NULL;
            for(std::list<dim*>::const_iterator it = dims.begin(); it != dims.end(); it++)
            {
                if((*it)->get_name() == ref_dim_name)
                {
                    d = *it;
                    break;
                }
	    } // end of for 
            if(d==NULL)
            {
                // Dimension has not defined in the file.
                // In this case, we add dimension.

                //std::cout << "Var name: " << name << std::endl;
                //std::cout << "Dim name: " << ref_dim_name << std::endl;
                //std::cout.flush();

                eos2_dim *nd = new eos2_dim(
                    g,
                    this->m_file_info,
                    this->m_group_type,
                    this->m_handle,
                    ref_dim_name,
                    ref_dim_name,
                    dimsbuf[dim_index]);

                    new_dims->push_back(nd);

                    m_dims.push_back(nd);
                    //nd->inc();
            }
            else
            {
                m_dims.push_back(d);
                //d->inc();
	    } // end of if(d==NULL)
        } // end of for
    } // end of dims: m_dims

#if 0
    for(std::list<dim*>::iterator it = m_dims.begin(); it != m_dims.end(); it++)
    {
        eos2_dim *e2d = dynamic_cast<eos2_dim*>(*it);
        //this->inc();
    }
#endif

    // Priority for name conflict
    {
        mem_attr *attr2 = new mem_attr(this, "_e2lib_name_conflict_priority");
        std::ostringstream p;
        p << 2;
        attr2->set_value(p.str());
        this->add_attr(attr2);
    } // end of Priority for name conflict
}

eos2_var_geo::eos2_var_geo(const eos2_var_geo& r): 
    var(r),
    m_file_info(r.m_file_info),
    m_group_type(r.m_group_type),
    m_handle(r.m_handle),
    m_eos2varname(r.m_eos2varname),
    m_value_type(r.m_value_type)
{
    m_file_info->inc(this);
    m_handle->inc(this);
}

eos2_var_geo::~eos2_var_geo()
{
    m_file_info->dec(this);
    m_handle->dec(this);

    //while(!m_attrs.empty())
    //{
    //  attr *a = *(m_attrs.begin());
    //  delete a;
    //  m_attrs.pop_front();
    //}
}

value_type_t eos2_var_geo::get_type() const
{
    return m_value_type;
}
void eos2_var_geo::get_value(int32 start[], int32 stride[], int32 edge[], void *buf) const
{
    intn r = -1;
    r = SWreadfield(m_handle->get(), const_cast<char*>(m_eos2varname.c_str()), 
        start, stride, edge, buf);
    if(r)
        throw std::runtime_error("Cannot read Swath field " + 
            this->get_name() + " (" __FILE__ ":" TOSTRING(__LINE__)")");
}

var* eos2_var_geo::_clone() const
{
    eos2_var_geo *w = new eos2_var_geo(*this);
    return w;
}

bool eos2_var_geo::same_obj_test(var* v) const
{
    eos2_var_geo *w = dynamic_cast<eos2_var_geo*>(v);
    //std::cout << typeid(*this).name() << std::endl;
    //std::cout << typeid(*w).name() << std::endl;
    //std::cout << typeid(*v).name() << std::endl;
    //if(typeid(*this) != typeid(*w))
    //	return false;
    if(w == NULL)
        return false;
    if(this->m_handle != w->m_handle)
        return false;
    if(this->m_eos2varname != w->m_eos2varname)
        return false;
    return true;
}

bool eos2_var_geo::is_dimmap_reduced() const
{
    const std::list<dim*>& dims = this->get_dims_c();
    std::list<dim*>::const_iterator dit;
    for(dit = dims.begin(); dit!=dims.end(); dit++)
    {
        intn r = SWgeomapinfo(m_handle->get(), 
					const_cast<char*>((*dit)->get_name().c_str()));
        if(r==1)
            return true;
    } // end of for
    return false;
}

} // namespace
