/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Category: Header file for the eoslib namespace

Description:

This file includes a definition of an attribute class.

*****************************************************************************/

#ifndef EOSLIB_ATTR_H
#define EOSLIB_ATTR_H

#include <list>
#include <string>
#include <vector>
#include <iostream>
#include <stdexcept>
#include "eoslib_types.h"
#include "eoslib_defs.h"
#include "eoslib_rc.h"

namespace eoslib
{

//! Attribute class
/*!
 * An attribute is a 1-d array of some primitive data type.
 * For example, a string is represented as a 1D array of DFNT_CHAR8.
 */
class attr: public renamable
{
    public:
        //! Constructor class with initial name. Group attribute.
        inline attr(group *grp, const std::string& name): renamable(name), m_group(grp), m_var(NULL) {}
        //! Constructor class with initial name. Variable attribute.
        inline attr(var *v, const std::string& name): renamable(name), m_group(NULL), m_var(v) {}
        //! Destructor
        inline virtual ~attr() {}

        /*! This function returns the data type
         *  of the attribute. If the attribute is not
         *  initialized, it returns -1.
         */
        virtual value_type_t get_type() const = 0;

        //! Number of elementes in the attribute
        virtual unsigned int get_num_elements() const = 0;

        /*!
         * This function is used to get the value of this 
         * attribute.
         * We assume that buf points to a buffer which 
         * has sufficient space for holding the attribute.
         */
        virtual void get_value(void *buf) const = 0;

        /*!
         * This function is used to get the value of this
         * attribute.
         * Since this uses a vector class as a parameter,
         * no prior memory allocation is needed.
         * This fuction allocates memory space as needed.
         */
        void get_value(std::vector<char> *pbuf) const;

        /**
         * If this attribute has a string value, this
         * function can be used the string.
         * Otherwise, this function throws an exception.
         */
        void get_value_str(std::string *pVal) throw(std::runtime_error);

        /**
         * This function makes another copy of the current
         * attr class instance.
         */
        virtual attr *clone() const = 0;

        /**
         * This function checks whether the given
         * attr is same as the current attr.
         * Equivalence test checks the equivalence of
         * names and values.
         */
        bool is_equivalent_to(attr* r) const;

        /**
         *  This function dumps the name and value of the 
         *  attribute to stdout.
         *  Note: this function doesn't 
         *        return the attribute values.   
         *        It should only be used for internal testing(/internal_test)
         *  \param sp Number of leading spaces
         * 
        */
        void dump_r(int sp) const;

        /**
         * This function checks whether the given attr and the
         * current attr 
         * are the same attribute object in the file or not.
         * If an attr object is clone()-ed, the original object
         * and the cloned one are the "same" object.
         * This function returns true in that case.
         */
        virtual bool same_obj_test(attr *) const = 0;

        /////////////////////////////////////////
        // Static funcs
        /////////////////////////////////////////
	
        //! Getting name of the attribute
        static std::string s_get_name(const attr*a);

        //! Testing whether two attrs are the same objects or not
        static bool s_same_obj_test(attr *a1, attr *a2);

        group *get_group() {return m_group;}
        var *get_var(){return m_var;}

    protected:
        group *m_group;	/* If this attribute belonging to a group, m_group points to it. */
        var *m_var; /* If this attribute belonging to a variable, m_group points to it. */
};
} /* end of namespace */
#endif

