/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file inlcudes the implementation related to the processing of a dimension.

*****************************************************************************/

#include <iostream>
#include <iomanip>
#include <algorithm>

#include "eoslib_defs.h"

namespace eoslib
{

dim::dim(group *g, const std::string& name, unsigned int size):
    renamable(name), m_size(size), m_group(g),disable_clone(false)
{
}

dim::dim(const dim& r):
    renamable(r),
    m_group(r.m_group),
    m_size(r.m_size),
    m_coord_vars(r.m_coord_vars),
    disable_clone(r.disable_clone)

{
}

dim::~dim()
{
    //std::cout << "~dim of " << get_name() << "is called" << std::endl;
}

// For internal testing only
void dim::dump_r(int sp) const
{
    std::cout << std::setfill(' ') << std::setw(sp) << "";
    std::cout << "Dimension: " << this->get_name() << "(" << m_size << ")";
    //std::cout << " (rc: " << get_refcnt() << ")";
    std::cout << std::endl;

    const std::list<var*>& cds = this->get_cv_c();
    std::cout << std::setfill(' ') << std::setw(sp) << "";
    std::cout << "  Coord vars: ";
    for(std::list<var*>::const_iterator it = cds.begin(); it != cds.end(); it++)
        std::cout << (*it)->get_name() << "(" << *it << ") ";
    std::cout << std::endl;
}

bool dim::has_cv_of(var *v) const
{
    std::list<var*>::const_iterator it;
    it = find(m_coord_vars.begin(), m_coord_vars.end(), v);
    return (it != m_coord_vars.end());
}

void dim::replace_cv(var *v1, var *v2)
{
    replace(m_coord_vars.begin(), m_coord_vars.end(), v1, v2);
}

void dim::replace_cv(const std::map<var*, var*>& varmap)
{

    std::list<var*>::iterator it;
    for(it = m_coord_vars.begin(); it != m_coord_vars.end(); it++)
    {
        std::map<var*, var*>::const_iterator im;
        im = varmap.find(*it);
        if(im != varmap.end())
            *it = im->second;
        else
            throw std::runtime_error("Cannot replace coordinate varaible of dim " + this->get_name() + "(" __FILE__ ":" TOSTRING(__LINE__)")" );
    } // end of for
}

void dim::add_cv(var *v)
{
    if(!has_cv_of(v))
        m_coord_vars.push_back(v);
}

void dim::remove_cv(var *v)
{
    if(has_cv_of(v))
        m_coord_vars.remove(v);
}

std::pair<std::string, unsigned int> dim::to_name_size_pair(dim *d)
{
    return std::make_pair(d->get_name(), d->get_size());
}

bool dim::is_equivalent_to(dim* r) const
{
    if(this->get_name() != r->get_name())
        return false;
    if(this->get_size() != r->get_size())
        return false;
	
    if(this->get_cv_c().size() != r->get_cv_c().size())
        return false;

    std::list<var*>::const_iterator it1;
    std::list<var*>::const_iterator it2;
    it1 = this->get_cv_c().begin();
    it2 = r->get_cv_c().begin();

    while(it1 != this->get_cv_c().end())
    {
        if(!(*it1)->is_equivalent_to(*it2))
            return false;
        it1++;
        it2++;
    } // end of while	
    return true;
}

std::string dim::s_get_name(const dim*d)
{
    return d->get_name();
}

bool dim::equivalence_test(dim *d1, dim *d2)
{
    if(d1->get_size() != d2->get_size())
        return false;
    if(!var::s_same_objlist_test(d1->get_cv_c(), d2->get_cv_c()))
        return false;
    return true;
}

} // namespace
