/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************
Category: Header file for the eoslib namespace

Description:

This file includes a definition of class dimension.

*****************************************************************************/


#ifndef EOSLIB_DIM_H
#define EOSLIB_DIM_H

#include <list>
#include <map> 
#include <string> 
#include "eoslib_defs.h"
#include "eoslib_rc.h"

namespace eoslib
{

class var;

//! Dimension class
/*!
 *  This class represents a dimension. Since this class 
 *  defines pure virtual functions, it needs to be 
 *  inherted by other descendent classes.
 *
 *  A dimension has a name, size and coordinate variables.
 */
class dim: public renamable
{
    public:
        //! Constructor
        dim(group *g, const std::string&name, unsigned int size);
        //! Copy-constructor
        dim(const dim& r);
        //! Destructor
        virtual ~dim();

        /////////////////////////////////
        // Size
        /////////////////////////////////
	
        //! Getting size of the dimension
        inline unsigned int get_size() const 
            {return m_size;}
        //! Setting the size of the dimension
        void set_size(unsigned int s)
            {m_size = s;}

        /////////////////////////////////
        // Coordinate variables
        /////////////////////////////////
	
        //! Getting coordinate variables
        std::list<var*>& get_cv() 
            {return m_coord_vars;}

        //! Getting coordinate variables (const)
        const std::list<var*>& get_cv_c() const 
            {return m_coord_vars;}

        //! Checking whether v is one of the coordinate variables of the current dimension
        bool has_cv_of(var *v) const;

        //! Adding a coordinate variable
        void add_cv(var *v);

        //! Removing a coordinate variable
        void remove_cv(var *v);

        /**
         * This function replaces current dim's coordinate variable
         * v1 with v2.
         * If v1 is not the coordinate varible, nothing happens.
         */
        void replace_cv(var *v1, var *v2);

        /**
         * This function replaces all cvs of the current dim
         * with a new one. The mapping is stored in the varmap.
         */
        void replace_cv(const std::map<var*, var*>& varmap);

        /////////////////////////////////
        // Coordinate variables
        /////////////////////////////////

        /** 
         * This function makes another copy of the current dim.
         * Newly created dim has g as its dimension.
         * Note that g still does not have the new dim.
         */
        virtual dim *clone(group *g) const = 0;
	
        /**
         * Dim A in one group is equivalent to another dim B 
         * in another group if
         *  - They have the same name,
         *  - they have the same size,
         *  - Their cv are the same (same obj in the file)
         */
        bool is_equivalent_to(dim *r) const;

        //! Getting the group where the current dimension is in
        group* get_group() const 
            {return m_group;}

        //! Dumping the current dimension to stdout
        void dump_r(int sp) const;

        ////////////////////////////
        // Static
        ////////////////////////////
	
        //! Static version of is_equivalent_to()
        static bool equivalence_test(dim *d1, dim *d2);

        //! This function returns a name-size pair of the given dimension
        static std::pair<std::string, unsigned int> to_name_size_pair(dim *);
	
        //! Getting name of the given dimension
        static std::string s_get_name(const dim*v);
        bool NOT_clone() { return disable_clone;}

    protected:
        unsigned int m_size;
        group *m_group;
        std::list<var*> m_coord_vars;

        // The flag to tell if this dim. should be cloned and moved to the root group.
        bool disable_clone;

    private:
        dim();	// diabled
};
} // namespace
#endif

