/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of a group.

*****************************************************************************/

#include <iomanip>
#include <iostream>
#include <string>
#include <cassert>
#include <stdexcept>
#include <iterator>
#include <algorithm>

#include "eoslib_defs.h"
#include "mem_group.h"
#include "misc_util.h"

namespace eoslib
{

group::group(const std::string& name): 
    renamable(name),
    m_file_info(NULL)	
{
}

group::~group()
{
    //std::cout << "group::~group() of " << get_name() << std::endl;
    // decrement all ref counters
    while(!m_child_groups.empty())
    {
        group *child = *(m_child_groups.begin());
        delete child;
        m_child_groups.pop_front();
    }

    // Remove variables.
    while(!m_vars.empty())
    {
        var *v= *(m_vars.begin());
        delete v;
        m_vars.pop_front();
    }

    // Remove dimensions.
    while(!m_dims.empty())
    {
        dim *v= *(m_dims.begin());
        delete v;
        m_dims.pop_front();
    }

    // Remove attributes.
    while(!m_attrs.empty())
    {
        attr *v= *(m_attrs.begin());
        delete v;
        m_attrs.pop_front();
    }
	
    // Decrement the reference count.
    if(m_file_info)
        m_file_info->dec(this);
} // end of ~group()

file_info* group::get_file_info() const 
{
    return m_file_info;
}

void group::set_file_info(file_info *f) 
{
    if(get_file_info() == NULL)
    {
        m_file_info = f; 
        f->inc(this);
    }
    else
    {
        assert(0);
    }
} // end of set_file_info

group* group::get_child_by_name(const std::string& name)
{
    // XXX. This linear scan can be improved.
    std::list<group*>::iterator it;
    for(it = m_child_groups.begin(); it != m_child_groups.end(); it++)
        if((*it)->get_name() == name)
            return *it;
    return NULL;
}

std::set<var*> group::get_all_cv() const
{
    std::set<var*> all_cv;
    std::list<dim*>::const_iterator id;
    for(id = m_dims.begin(); id != m_dims.end(); id++)
    {
        std::list<var*> dcv = (*id)->get_cv();
        std::list<var*>::iterator icv;
        for(icv = dcv.begin(); icv != dcv.end(); icv++)
            all_cv.insert(*icv);
    }
    return all_cv;
}

void group::get_vars_by_name(const std::string& name, std::list<var*> *pvars)
{
    // XXX. This linear scan can be improved.
    std::list<var*>& vars = get_vars();
    std::list<var*>::iterator it;
    for(it = vars.begin(); it != vars.end(); it++)
        if((*it)->get_name() == name)
            pvars->push_back(*it);
}

dim* group::get_dim_by_name(const std::string& name)
{
    // XXX. This linear scan can be improved.
    std::list<dim*>& dims = get_dims();
    std::list<dim*>::iterator it;
    for(it = dims.begin(); it != dims.end(); it++)
        if((*it)->get_name() == name)
            return *it;
    return NULL;
}

// Note different dimensions may share the same size. 
// This routine
// just returns the last dimension. KY 2014-12-18
dim* group::get_dim_by_size(int size)
{
    // XXX. This linear scan can be improved.
    dim* candidate = NULL;
    std::list<dim*>& dims = get_dims();
    std::list<dim*>::iterator it;
    for(it = dims.begin(); it != dims.end(); it++)
        if((*it)->get_size() == size)
        {
            if(candidate == NULL)
                candidate = *it;
            else
                return NULL;
        }
    return candidate;
}

attr* group::get_attr_by_name(const std::string& name)
{
    // XXX. This linear scan can be improved.
    std::list<attr*>& attrs = get_attrs();
    std::list<attr*>::iterator it;
    for(it = attrs.begin(); it != attrs.end(); it++)
        if((*it)->get_name() == name)
            return *it;
    return NULL;
}

// For internal testing only.
void group::dump_r(unsigned int sp) const
{
    std::cout << std::setfill(' ') << std::setw(sp) << "";
    std::cout << "Group: " << this->get_name();
    //std::cout << " (rc: " << get_refcnt() << ")";
    std::cout << std::endl;
    const std::list<group*>& children = get_child_groups_c();
    const std::list<var*>& vars= get_vars_c();
    const std::list<dim*>& dims= get_dims_c();
    const std::list<attr*>& attrs= get_attrs_c();
	
    for(std::list<group*>::const_iterator it = children.begin(); it != children.end(); it++)
        (*it)->dump_r(sp+2);

    for(std::list<var*>::const_iterator it = vars.begin(); it != vars.end(); it++)
        (*it)->dump_r(sp+2);

    for(std::list<dim*>::const_iterator it = dims.begin(); it != dims.end(); it++)
        (*it)->dump_r(sp+2);

    for(std::list<attr*>::const_iterator it = attrs.begin(); it != attrs.end(); it++)
        (*it)->dump_r(sp+2);
}

// Move items from src to dest. 
/*
void group::move_all(group* dest)
{
    std::list<var*>& dst_vars = dest->get_vars();
    std::list<dim*>& dst_dims = dest->get_dims();
    std::list<attr*>& dst_attrs = dest->get_attrs();
    std::list<group*>& dst_children = dest->get_child_groups();

    std::list<var*>& src_vars = this->get_vars();
    std::list<dim*>& src_dims = this->get_dims();
    std::list<attr*>& src_attrs = this->get_attrs();
    std::list<group*>& src_children = this->get_child_groups();

    dst_vars.splice(dst_vars.end(), src_vars);
    dst_dims.splice(dst_dims.end(), src_dims);
    dst_attrs.splice(dst_attrs.end(), src_attrs);
    dst_children.splice(dst_children.end(), src_children);
}
*/

void group::remove(group* g)
{
    std::list<group*>& children = this->get_child_groups();
    std::list<group*>::iterator it;
    for(it = children.begin(); it != children.end(); it++)
    {
        if(*it == g)
        {
            remove(it);
            return;
        }
    } //end of for
    throw std::runtime_error("Cannot find given group for removal(" __FILE__ ":" TOSTRING(__LINE__)")" );
}

void group::remove(std::list<group*>::iterator it)
{
    m_child_groups.erase(it);
}

void group::get_vars_by_path(const std::string& varpath, group **pg, std::list<var*> *pvars)
{
    if(varpath.size() < 1)
        return;

    if(varpath[0] != '/')
        return;
	
    size_t pos = varpath.find('/', 1);
    if(pos == std::string::npos)
    {
        if(pg!=NULL)
            *pg = this;
        std::string varname = varpath.substr(1);
        get_vars_by_name(varname, pvars);
    }
    else
    {
        std::string childname = varpath.substr(1, pos - 1);
        std::string remaining = varpath.substr(pos);

        group *g = get_child_by_name(childname);
        g->get_vars_by_path(remaining, pg, pvars);
    } // end of if(pos == std::string::npos)
}

group *group::get_group_by_path(const std::string& grouppath, group **parent)
{
    if(grouppath.size() < 1)
        return NULL;

    if(grouppath[0] != '/')
        return NULL;
	
    size_t pos = grouppath.find('/', 1);
    if(pos == std::string::npos)
    {
        if(parent!=NULL)
            *parent = this;
        std::string groupname = grouppath.substr(1);
        return get_child_by_name(groupname);
    }
    else
    {
        std::string childname = grouppath.substr(1, pos - 1);
        std::string remaining = grouppath.substr(pos);

        group *g = get_child_by_name(childname);
        return g->get_group_by_path(remaining, parent);
    } // end of if(pos == std::string::npos)
}

void group::remove_dim(dim *d)
{
    std::list<dim*>& dims = this->get_dims();
    std::list<dim*>::iterator it;
	
    it = find(dims.begin(), dims.end(), d);
    if(it == dims.end())
    {
        // Cannot find v
        //assert(0);
        throw std::runtime_error("Cannot find var for removal(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }
    else
    {
        dims.erase(it);
    } // end of if(itv == vars.end())
}
void group::remove_var(var *v)
{
    std::list<dim*>& dims = this->get_dims();
    std::list<dim*>::iterator it;
    for(it = dims.begin(); it != dims.end(); it++)
    {
        if((*it)->has_cv_of(v))
        {
            // XXX. We may throw exception or just reset the
            // list of coord var.
            throw std::runtime_error("Cannot remove a coordinate variable(" __FILE__ ":" TOSTRING(__LINE__)")" );
        }
    } // end of for
	
    std::list<var*>& vars = this->get_vars();
    std::list<var*>::iterator itv;
    itv = find(vars.begin(), vars.end(), v);
    if(itv == vars.end())
    {
        // Cannot find v
        //assert(0);
        throw std::runtime_error("Cannot find var for removal(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }
    else
    {
        vars.erase(itv);
    } // end of if(itv == vars.end())
}

void group::remove_var_force(var *v)
{
    std::list<var*>& vars = this->get_vars();
    std::list<var*>::iterator itv;
    itv = find(vars.begin(), vars.end(), v);
    if(itv == vars.end())
    {
        // Cannot find v
        //assert(0);
        throw std::runtime_error("Cannot find var for removal(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }
    else
    {
        vars.erase(itv);
    } // end of if(itv == vars.end())
}


void group::replace_cv(var *v1, var *v2)
{
    std::list<dim*>& dims = this->get_dims();
    std::list<dim*>::iterator di;
    for(di = dims.begin(); di != dims.end(); di++)
    {
        (*di)->replace_cv(v1, v2);
    }
}

var *group::find_uniq_1d_var_using(dim *d)
{
    std::list<var*>& vars = get_vars();
    std::list<var*>::iterator it ;

    var *v = NULL;
    for(it = vars.begin(); it != vars.end(); it++)
    {
        std::list<dim*>& dims = (*it)->get_dims();
        if((dims.size() == 1) && (*dims.begin() == d))
        {
            if(v==NULL)
                v = *it;
            else
                return NULL;
        } // end of if
    }
    return v;
}

void group::add_var(var *v)
{
// This is the only place that uses has_dim(). It seems for a debugging purpose.
#ifndef NDEBUG
    std::list<dim*>& dims = v->get_dims();
    std::list<dim*>::iterator it;
    for(it = dims.begin(); it != dims.end(); it++)
    {
        assert(this->has_dim(*it));
    }
#endif

    //The following code prevents the same variable from being added twice.
    // 
    // We found a case that there are two different variables sharing the same name. 
    // HDF-EOS2 assures the uniqueness of the grid/swath name + variable name. So
    // this will only happens when an added coordinate variable shares the same name 
    // as a general variable. For a file AMSR_E_L2A_Brightness..., the variable name
    // Data_Quality (2D) shares the same time as a dimension Data_Quality(1D). 
    // When creating a dummy coordinate variable for Data_Quality(1D), it looks like
    // that the same variable is added twice, but it is not.
    // To distinguish this, we will also compare the dimensionality of these two variables,
    // only when the dimension size and the rank are the same for the two variables,we 
    // will not add the variables. 
    // KY 2013-05-30
    bool found = false;
    for(std::list<var*>::const_iterator i=m_vars.begin(); i!=m_vars.end(); i++)
    {

        if(v->get_name().compare((*i)->get_name())==0)
        {

            std::list<dim*>& temp_dims = v->get_dims();
            std::list<dim*>& curdims = (*i)->get_dims();
            
            if(temp_dims.size() == curdims.size()){ 

                std::list<dim*>::iterator dit = temp_dims.begin();
                std::list<dim*>::iterator cit = curdims.begin();
                found = true;
                while(dit != temp_dims.end() && cit != curdims.end()) {
                    if((*dit)->get_size() != (*cit)->get_size()) {
                        found = false;
                        break;
                    }
                    dit++;
                    cit++;
                }
                
                    
            }

        }
    } // end of for
	
    if(!found)
    {
        m_vars.push_back(v);
    }
}

void group::add_group(group *g)
{
    m_child_groups.push_back(g);

   // std::cout << "[eoslib_group]add_group" << std::endl;
}

void group::add_dim(dim *d)
{
    //The following code prevents the same dim being added twice.
    bool found = false;
    for(std::list<dim*>::const_iterator it=m_dims.begin(); it!=m_dims.end(); it++)
    {
        if(d->get_name().compare((*it)->get_name())==0)
        {
            found = true;
            break;
        }
    } // end of for

    if(!found) 
        m_dims.push_back(d);
}

void group::add_attr(attr *a)
{
    m_attrs.push_back(a);
}

// This method will check if this dimension exists in this group.
//
bool group::has_dim(dim *d)
{
    std::list<dim*>& dims = get_dims();
    std::list<dim*>::iterator it;

    
    //  Check if this dimension exists by just checking the dimensio names. 
    for(it=dims.begin(); it!=dims.end(); it++) 
    {
        std::string str = (*it)->get_name();
        if(str.compare(d->get_name())==0)
            return true;
    }

    // Cannot find a dimension name in this group's dimensions indicates
    // this dimension doesn't exist in this group.
    return false;
}

var* group::copy(group *g, var *v)
{
    //std::cout << "copying " << g->get_name() << "/" << v->get_name() << "to " << this->get_name() << std::endl;
			
    const std::list<dim*>& var_dims = v->get_dims_c();
    std::list<dim*>& dst_group_dims = this->get_dims();

    std::set<std::pair<std::string, unsigned int> > var_dims_p;
    std::set<std::pair<std::string, unsigned int> > dst_group_dims_p;

    transform(var_dims.begin(), var_dims.end(), 
        inserter(var_dims_p, var_dims_p.end()), 
        dim::to_name_size_pair);
    transform(dst_group_dims.begin(), dst_group_dims.end(), 
        inserter(dst_group_dims_p, dst_group_dims_p.begin()), 
        dim::to_name_size_pair);

    std::set<std::pair<std::string, unsigned int> > d;
    set_difference(var_dims_p.begin(), var_dims_p.end(), 
        dst_group_dims_p.begin(), dst_group_dims_p.end(),
        inserter(d, d.begin()));

    if(d.empty())
    {
        // All dims of v are in the destination group.
        // Now we can safely copy v.

        // replace all dims
        std::map<dim*, dim*> dim_conv;
        std::list<dim*>& v_dims = v->get_dims();
        std::list<dim*>::iterator it;
        for(it = v_dims.begin(); it != v_dims.end(); it++)
        {
            dim* newdim = this->get_dim_by_name((*it)->get_name());
            if(newdim)
                dim_conv[*it] = newdim;
            else
            {
                //assert(0);
                //throw std::runtime_error("The dimension doesn't exist in this group(" __FILE__ ":" TOSTRING(__LINE__)")");
                // Just return NULL if cannot copy the variable.
                return NULL;
            }
        } // end of for
        var* w = v->clone_r(this, dim_conv);

        this->add_var(w);
        return w;
    }
    else
    {
        return NULL;
    } // end of if(d.empty())
}

std::list<var*> group::get_vars_r() const
{
    std::list<var*> vars = m_vars;
    std::list<group*>::const_iterator it;
    for(it = m_child_groups.begin(); it != m_child_groups.end(); it++)
    {
        std::list<var*> gv = (*it)->get_vars();
        vars.splice(vars.end(), gv);
    }
    return vars;
}

std::list<dim*> group::get_dims_r() const
{
    std::list<dim*> dims = m_dims;
    std::list<group*>::const_iterator it;
    for(it = m_child_groups.begin(); it != m_child_groups.end(); it++)
    {
        std::list<dim*> gv = (*it)->get_dims();
        dims.splice(dims.end(), gv);
    }
    return dims;
}

/*
void move_group(group* src_parent, group* dst_parent, group *item)
{
    std::list<group*>::iterator it;
    it = std::find(
        src_parent->m_child_groups.begin(),
        src_parent->m_child_groups.end(),
        item);
    if(it != src_parent->m_child_groups.end())
    {
        item->inc();
        dst_parent->m_child_groups.push_back(item);
        src_parent->m_child_groups.erase(it);
        item->dec();
    }
    else
    {
        throw std::runtime_error("Cannot find group for removal(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }
}
*/
} // namespace

