/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Category: Header file for the eoslib name space

Description:

This file includes methods of handling group.

*****************************************************************************/

#ifndef EOSLIB_GROUP_H
#define EOSLIB_GROUP_H

#include <list>
#include <map>
#include <string>
#include "eoslib_defs.h"
#include "eoslib_rc.h"

namespace eoslib
{

//! Group class
/*! 
 *  This class represents a group. In HDF4, a VGroup is a group.
 *  In HDF-EOS2, a grid or a swath is a group.
 *  
 *  Multiple variables under a group can have the same name.
 */
class group: public renamable
{
    public:
        //! Constructor
        group(const std::string& name);
        //! Destructor
        virtual ~group();

        //! Getting child groups
        inline std::list<group*>& get_child_groups()
            {return m_child_groups;}

        //! Getting child groups (const)
        inline const std::list<group*>& get_child_groups_c() const
            {return m_child_groups;}

        //! Getting variable in the current group
        inline std::list<var*>& get_vars()
		{return m_vars;}
        //! Getting variable in the current group (const)
        inline const std::list<var*>& get_vars_c() const
		{return m_vars;}
        //! Getting all coordinate variable in the group
        std::set<var*> get_all_cv() const;

        //! Getting dimensions in the current group
        inline std::list<dim*>& get_dims()
		{return m_dims;}
        //! Getting dimensions in the current group (const)
        inline const std::list<dim*>& get_dims_c() const
		{return m_dims;}
        //! Getting attributes in the current group
        inline std::list<attr*>& get_attrs()
		{return m_attrs;}
        //! Getting attributes in the current group (const)
        inline const std::list<attr*>& get_attrs_c() const
            {return m_attrs;}

        //! Getting all variables in the current and descendent groups
        std::list<var*> get_vars_r() const;
        //! Getting all dimensions in the current and descendent groups
        std::list<dim*> get_dims_r() const;

        //! Getting file_info
        /**
         * Each group maintains a pointer to a file_info struct.
         * This function returns the file_info pointer.
         */
        file_info* get_file_info() const;

        //! Setting file_info
        void set_file_info(file_info *f);

		//! setting a boolean value to indicate if originally this file contains multi-groups
		void set_num_of_orig_group_status(bool num_of_orig_group_status) { have_multi_groups = num_of_orig_group_status;}

		bool get_num_of_orig_group_status() {return have_multi_groups;}
        //! Getting child group by name
        group* get_child_by_name(const std::string& name);
        //! Getting variables by name
        void get_vars_by_name(const std::string& name, std::list<var*> *pvars);
        //! Getting dimension by name
        dim* get_dim_by_name(const std::string& name);
        //! Getting attribute by name
        attr* get_attr_by_name(const std::string& name);

        /**
         * If there is only one dim with the given size,
         * this function returns the dimension.
         * Otherwise, NULL is returned.
         */
        dim* get_dim_by_size(int size);

        /**
         * This function is used to retrive a descendent var.
         * varpath must start with /.
         */
        void get_vars_by_path(const std::string& varpath, group **pg, std::list<var*> *pvars);

        /**
         * This function is used to retrive a descendent group.
         * grouppath must start with /.
         */
        group* get_group_by_path(const std::string& grouppath, group **parent=NULL);

        //! Adding new variable to the current group
        void add_var(var *v);

        //! Adding new child group to the current group
        void add_group(group *g);

        //! Adding new dimension to the current group
        void add_dim(dim *g);

        //! Adding new attribute to the current group
        void add_attr(attr *g);

        //! Removing a child group
        void remove(group* g);

        //! Removing a child group by its iterator
        void remove(std::list<group*>::iterator it);

        void remove_dim(dim* d);

        /**
         * This function removes a variable from the current group.
         * It checks all dims of the current group and 
         * checks whether the deleted var is the coord var of any dim.
         * If so, an exception is thrown.
         */
        void remove_var(var *v);

        /* This function removes a variable if this variable exists.
         * It doesn't check the dims or the coord vars of dims. 
         */
        void remove_var_force(var *v);

        /**
         * This function checks all dimensions and find all
         * the cases that v1 is used as coordinate variables.
         * All such references to v1 are replaced to the ones to v2.
         */
        void replace_cv(var *v1, var *v2);

        /**
         * This function searches the current group for the 1D variables
         * which refers to the given dimension. If there is only one
         * such variable, it is returned. If there is none or multiple,
         * NULL is returned.
         */
        var * find_uniq_1d_var_using(dim *d);

        /**
         * This function checks whether the current group has the given
         * dimension or not.
         */
        bool has_dim(dim *d);

        //inline void no_use(group *g) {m_no_use_groups.push_back(g);};

        //! Dumping the group to stdout(Note attribute valuers are not dumped. */ 
        // Should be used only for the obsolete internal testings(/internal_test)
        void dump_r(unsigned int sp=0) const;

        /**
         * Copy variable v in group g to the current group.
         * Note that the source variable uses dimenstions in the source group.
         * We need dest variables have dimesnions in the dest group.
         * Now we assume that src group and dst group the same dim info.
         * In other words, all dims of v in g are also in the dest group.
         * We check this by dim names. If it is not true, we throw an exception.
         */
        var* copy(group *g, var *v);

        /**
         * This function moves all variables, attributes, dims and sub groups 
         * to another group.
         */
        //void move_all(group* dest);

    protected:
        std::list<group*> m_child_groups;
        std::list<var*> m_vars;
        std::list<dim*> m_dims;
        std::list<attr*> m_attrs;
		bool have_multi_groups;

        /**
         * Groups to be deleted when the current group is deleted.
         */
        //std::list<group*> m_no_use_groups;

        file_info* m_file_info;

    private:
        void _copy_vars_r(group *dest);

        //friend void move_group(group* src_parent, group* dst_parent, group *item);
};
} // namespace
#endif

