/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Category: Header file for the eoslib namespace

Description:

This file includes two class defintions. One class represents a renamable name,
and the other one represents basic reference counting of dynamically allocated 
objects. 

*****************************************************************************/


#ifndef EOSLIB_RC_H
#define EOSLIB_RC_H

#ifdef _MSC_VER
#include <BaseTsd.h>
#pragma warning(disable :4290)
typedef SSIZE_T ssize_t;
#endif
#include <string>
#include <set>
#include <list>

//! A class which has a renamable name.
/**
 *  If a class is a descendent of this class, it can
 *  have name which can be changed later.
 */
class renamable
{
    public:
        //! Constructor
        renamable(const std::string& name): m_name(name) {}
        //! Copy constructor
        renamable(const renamable& r): m_name(r.m_name) {}
        //! Destructor
        virtual ~renamable() {}

        //! Getting name
        const std::string& get_name() const {return m_name;}
        //! Changing name
        void rename(std::string name) {m_name = name;}
    protected:
        std::string m_name;
};

//! Reference-counting class
/**
 * This class implements basic reference counting of
 * dynamically allocated objects.
 * It requires the owner of the reference to be a 
 * renamable class for the ease of debugging.
 */
class refcnt
{
    public:
        refcnt();
        virtual ~refcnt();

        //! Increasing ref count
        void inc(const renamable* owner);
        //! Decreasing ref count
        void dec(const renamable* owner);
        //! Getting ref count
        unsigned int get_refcnt() const ;

    #ifdef DEBUG
        void dump_owners() const;
    #endif
 
    private:
        unsigned int m_refcnt;

    #ifdef DEBUG
        std::set<renamable*> m_owners;
    #endif
};

#endif

