/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of a data field. 

*****************************************************************************/

#include <iostream>
#include <iomanip>
#include <list>
#include <vector>
#include <stdexcept>
#include <cassert>
#include <algorithm>

#include "eoslib_var.h"
#include "eoslib_types.h"
#include "eoslib_err.h"

namespace eoslib 
{

var::var(group *g, const std::string& name): renamable(name), m_group(g),disable_clone(false),var_orig_name(name)
{
}

var::var(const var& r): renamable(r), m_dims(r.m_dims), m_attrs(r.m_attrs),disable_clone(r.disable_clone)
{
}

var::var(group *g, const std::string& name, const std::string& orig_name): renamable(name), m_group(g),disable_clone(false),var_orig_name(orig_name)
{
}


var::~var()
{
    while(!m_attrs.empty())
    {
        attr *d = *m_attrs.begin();
        delete d;
        m_attrs.pop_front();
    }
}

bool var::has_dim_of(dim *d) const
{
    std::list<dim*>::const_iterator it;
    it = find(m_dims.begin(), m_dims.end(), d);
    return (it != m_dims.end());
}


std::vector<int32> var::get_dim_sizes() const
{
    std::vector<int32> ret;

    std::list<dim*>::const_iterator it;
    for(it = m_dims.begin(); it != m_dims.end(); it++)
        ret.push_back((*it)->get_size());
    return ret;
}

void var::get_all(void **pbuf)
{
    const std::list<dim*>& dims = get_dims_c();
    uint32 num_elements = 1;
    std::vector<int32> start;
    std::vector<int32> stride;
    std::vector<int32> edge;
    for(std::list<dim*>::const_iterator it = dims.begin();
        it != dims.end();
        it++)
    {
        num_elements *= (*it)->get_size();
        start.push_back(0);
        stride.push_back(1);
        edge.push_back((*it)->get_size());
    }

    void *buf = NULL;
    value_type_t type = get_type();

    switch(type)
    {
#define HANDLE(TYPE, type) \
    case DFNT_##TYPE: \
        buf = new type [num_elements]; \
        break;

        HANDLE(CHAR8, char8);
        HANDLE(INT8, int8);
        HANDLE(INT16, int16);
        HANDLE(INT32, int32);
        HANDLE(UCHAR8, uchar8);
        HANDLE(UINT8, uint8);
        HANDLE(UINT16, uint16);
        HANDLE(UINT32, uint32);
        HANDLE(FLOAT32, float32);
        HANDLE(FLOAT64, float64);

#undef HANDLE
        default:
            throw std::range_error("Unknown type(" __FILE__ ":" TOSTRING(__LINE__)")" );
    } // end of switch

    get_value(&start[0], &stride[0], &edge[0], buf);
    *pbuf = buf;
}

/*
float32 var::get_float32_at(const void *buf, const std::vector<int32>& pos)
{
    const float32 *b = (const float32*)buf;

    int32 m = 1, next_m = 1, p = 0;
    int i;

    const std::list<dim*>& dims = get_dims_c();
    std::list<dim*>::const_reverse_iterator it = dims.rbegin();

    assert(pos.size() == dims.size());
    for(i=pos.size()-1; i>=0; i--)
    {
        m = next_m;
        next_m = next_m * (*it)->get_size();
        p += m * pos[i];
        it++;
    }
    return b[p];
}
*/

// For internal testing only.
void var::dump_r(int sp, bool bDumpValues) const
{
    std::cout << std::setfill(' ') << std::setw(sp) << "";
    std::cout << "Variable: " << this->get_name() << "(" << this << ")";
    // std::cout << " (rc: " << get_refcnt() << ")";
    std::cout << std::endl;

    { // 1
        std::cout << std::setfill(' ') << std::setw(sp) << "";
        std::cout << "  Type: ";

        value_type_t type = get_type();
        std::cout << get_type_str(type) << std::endl;
    } // end of 1

    { // 2
        std::cout << std::setfill(' ') << std::setw(sp) << "";
        std::cout << "  Dims: ";

        std::cout.flush();
        const std::list<dim*>& ds = this->get_dims_c();
        std::list<dim*>::const_iterator it;
        for(it = ds.begin(); it != ds.end(); it++)
        {
            std::cout << (*it)->get_name() << " ";
        }
        std::cout << std::endl;
    } // end of 2

    { // 3
        const std::list<attr*>& attrs = this->get_attrs_c();
        for(std::list<attr*>::const_iterator it = attrs.begin();
            it != attrs.end(); it++)
        {
            (*it)->dump_r(sp+2);
        }
    } // end of 3

    if(bDumpValues)
    {
        std::cout << std::setfill(' ') << std::setw(sp) << "";
        std::cout << "  Values: ";
        const std::list<dim*>& dims = this->get_dims_c();
        size_t tot_elems= 1;

        std::vector<int32> start;
        std::vector<int32> stride;
        std::vector<int32> edge;

        for(std::list<dim*>::const_iterator it = dims.begin();
            it != dims.end(); it++)
        {
            tot_elems*= (*it)->get_size();
            start.push_back(0);
            stride.push_back(1);
            edge.push_back((*it)->get_size());
        } // end of for

        value_type_t type = this->get_type();
        switch(type)
        {
#define HANDLE3(TYPE, type, cast) \
    case DFNT_##TYPE: \
    { \
        std::vector<type> buf(tot_elems); \
        this->get_value(&start[0], &stride[0], &edge[0], &buf[0]); \
        for(size_t i = 0; i<buf.size(); i++) \
            std::cout << (cast)(buf[i]) << " ";	\
    } \
        break;
#define HANDLE(TYPE, type) \
    HANDLE3(TYPE, type, type)

            HANDLE(CHAR8, char8);
            HANDLE3(INT8, int8, int);
            HANDLE(INT16, int16);
            HANDLE(INT32, int32);
            HANDLE(UCHAR8, uchar8);
            HANDLE3(UINT8, uint8, unsigned int);
            HANDLE(UINT16, uint16);
            HANDLE(UINT32, uint32);
            HANDLE(FLOAT32, float32);
            HANDLE(FLOAT64, float64);
			
            default:
                throw std::range_error("Unknown vairable type at(" __FILE__ ":" TOSTRING(__LINE__)")" );
#undef HANDLE
#undef HANDLE3
        } // end of switch
        std::cout << std::endl;
    } // end of if(bDumpValues)
}

void var::add_attr(attr *a)
{
    m_attrs.push_back(a);
}

void var::remove_attr(attr *a)
{
    m_attrs.remove(a);
    delete a;
}

attr* var::get_attr_by_name(const std::string& name)
{
    std::list<attr*>& attrs = get_attrs();
    std::list<attr*>::iterator it;
    for(it = attrs.begin(); it != attrs.end(); it++) {
        if((*it)->get_name() == name)
            return *it;
    }
    return NULL;
}

void var::set_cv()
{
    std::list<dim*>& dims = get_dims();
    std::list<dim*>::iterator it;
    for(it = dims.begin(); it != dims.end(); it++)
    {
        (*it)->add_cv(this);
    }
}

bool var::is_cv(group* g)
{
    std::set<var*> all_cv = g->get_all_cv();
    if(all_cv.find(this) == all_cv.end())
        return false;
    else
        return true;
}

var* var::clone_r(group *g, const std::map<dim*, dim*>& dimmap) const
{
    var *w = _clone();
    w->m_group = g;
    w->replace_all_dims(dimmap);
    w->_clone_attrs();
    return w;
}

void var::_clone_attrs()
{
    std::list<attr*>::iterator it;
    for(it = m_attrs.begin(); it != m_attrs.end(); it++)
    {
        *it = (*it)->clone();
    }
}

bool var::is_equivalent_to(var *r) const
{
    // dim
    const std::list<dim*> l_dims = this->get_dims_c();
    const std::list<dim*> r_dims = r->get_dims_c();
    if(l_dims.size() != r_dims.size())
        return false;
    std::list<dim*>::const_iterator idl, idr;
    for(idl = l_dims.begin(), idr = r_dims.begin();
        idl != l_dims.end();
        idl++, idr++)
    {
        if(!(*idl)->is_equivalent_to(*idr))
            return false;
    } // end of for

    // values
    std::vector<char> lv, rv;
    this->get_all_values(&lv);
    r->get_all_values(&rv);
    if(lv != rv)
        return false;

    // attrs
    const std::list<attr*> l_attrs = this->get_attrs_c();
    const std::list<attr*> r_attrs = r->get_attrs_c();
    if(l_attrs.size() != r_attrs.size())
        return false;
    std::list<attr*>::const_iterator ial, iar;
    for(ial = l_attrs.begin(), iar = r_attrs.begin();
        ial != l_attrs.end();
        ial++, iar++)
    {
        if(!(*ial)->is_equivalent_to(*iar))
            return false;
    }
    return true;
}

unsigned int var::get_tot_num_elements() const
{
    unsigned int tot=1;
    const std::list<dim*>& dims = get_dims_c();
    std::list<dim*>::const_iterator it;
    for(it = dims.begin(); it != dims.end(); it++)
        tot *= (*it)->get_size();
    return tot;
}

unsigned int var::get_tot_num_bytes() const
{
    return  get_tot_num_elements() * get_type_size(get_type());
}

unsigned int var::get_num_bytes(int32 nr_elems) const
{
    return (nr_elems*get_type_size(get_type()));
}

void var::get_all_values(std::vector<char> *pbuf) const
{
    size_t storage = get_tot_num_elements() * get_type_size(get_type());

    const std::list<dim*>& dims = get_dims_c();
    unsigned int rank = dims.size();
    std::list<dim*>::const_iterator it = dims.begin();

    std::vector<int32> start(rank), stride(rank), edge(rank);
    for(unsigned int i=0; i<rank; i++, it++)
    {
        start[i]=0;
        stride[i]=1;
        edge[i] = (*it)->get_size();
    }
    pbuf->resize(storage);
    get_value(&start[0], &stride[0], &edge[0], &((*pbuf)[0]));
}

std::string var::s_get_name(const var*v)
{
    return v->get_name();
}

bool var::s_same_obj_test(var *v1, var *v2)
{
    return v1->same_obj_test(v2);
}

bool var::s_same_objlist_test(
    const std::list<var *>& l1, 
    const std::list<var *>& l2)
{
    if(l1.size() != l2.size())
        return false;

    std::list<var*>::const_iterator it1, it2;
    it1 = l1.begin();
    it2 = l2.begin();
    for(;it1 != l1.end(); it1++, it2++)
    {
        if(!var::s_same_obj_test(*it1, *it2))
            return false;
    }
    return true;
}

void var::replace_all_dims(const std::map<dim*, dim*>& dimmap)
{
    std::list<dim*>::iterator it;
    for(it = m_dims.begin(); it != m_dims.end(); it++)
    {
        std::map<dim*, dim*>::const_iterator im;
        im = dimmap.find(*it);
        if(im != dimmap.end())
            *it = im->second;
        else
            throw std::runtime_error("Cannot replace dim of var" + this->get_name() + "(" __FILE__ ":" TOSTRING(__LINE__)")" );
    } // end of for
}

int32 var::INDEX_nD_TO_1D(const std::vector<int32>& dims, const std::vector<int32>& pos)
{
    /*
     int a[10][20][30];  // & a[1][2][3] == a + (20*30+1 + 30*2 + 1 *3);
     int b[10][2]; // &b[1][2] == b + (20*1 + 2);
    */
    assert(dims.size() == pos.size());
    int32 sum=0;
    int32 start=1;
    for(unsigned int p=0; p<pos.size(); p++)
    {
        int32 m=1;
        for(unsigned int j=start; j<dims.size(); j++)
            m *= dims[j];
        sum += m * pos[p];
        start ++;
    } // end of for
    return sum;
}

int32 var::INDEX_nD_TO_1D(int32 rank, const int32 dim[], const int32 pos[])
{
    std::vector<int32> vdim(&dim[0], &dim[rank]);
    std::vector<int32> vpos(&pos[0], &pos[rank]);
    return INDEX_nD_TO_1D(vdim, vpos);
}
} // namespace
