/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Category: Header file for the eoslib namespace

Description:

This file includes definitions of a filter class and a filter set class.
These filters are used to make the final output follow the CF conventions.


*****************************************************************************/


#ifndef FILTER_SET_H
#define FILTER_SET_H

#include <string>
#include <set>
#include <utility>
#include "eoslib_group.h"

namespace eoslib {

/**
 * Modifier returns 0 if successful and -1 otherwise.
 */
typedef int (*modifier_t)(group*, void *);

struct filter
{
    std::string m_name;
    modifier_t m_modifier;
    void *m_arg;
};

inline bool NULL_CHECKER(const group *root) { return true; }
inline void NULL_MODIFIER(group *root) {}

///////////////
// Filter set
///////////////
class filter_set
{
    public:
        filter_set();
        ~filter_set();
        void add(std::string name, int priority, modifier_t m, void *arg);
        int apply(group *g) const;
    private:
        std::set<std::pair<int, filter*> > m_filters;
};

//! Define a filter set
#define FILTER_SET(name) filter_set g_filter_set_##name

//! Getting the size of a filter set
#define FILTER_SET_SIZE(name) g_filter_set_##name.size()

//! Adding a filter to a filter set
#define ADD_FILTER(name, fset, priority, mod) \
    class filter_register_##priority##checker##_##mod{	\
        public: \
            filter_register_##priority##checker##_##mod() { \
                g_filter_set_##fset.add(name, priority, mod, NULL); \
            } \
        }; \
        filter_register_##priority##checker##_##mod g_filter_register_##priority##checker##_##mod

inline int _APPLY_FILTER_SET(const filter_set& fs, group *root)
{
    return fs.apply(root);
	
}

#define APPLY_FILTER_SET(name, group) \
    _APPLY_FILTER_SET(g_filter_set_##name, group)

inline int _APPLY_SPECIAL_FILTER_SET(const filter_set& fs, group *root)
{
    return fs.apply(root);
}

#define APPLY_SPECIAL_FILTER_SET(name, group) \
    _APPLY_SPECIAL_FILTER_SET(g_filter_set_##name, group)

} // namespace

#endif
