/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of HDF-EOS2 
files.


*****************************************************************************/

#include <assert.h>
#include <algorithm>
#include <sstream>
#include <stdexcept>
#include <iostream>
#include <string>
#include <list>
#include <map>

#ifdef _MSC_VER
#include <iterator>
#endif

#include "filterset.h"
#include "mem_var.h"
#include "mem_dim.h"
#include "mem_attr.h"
#include "eoslib_err.h"
#include "eos2_defs.h"
#include "eos2_group.h"

#define CFLIB_MAX(A,B) ((A>B)?A:B)
#define CFLIB_MIN(A,B) ((A<B)?A:B)

template <typename K, typename V>
std::list<std::pair<K, V> > pair_list(const std::list<K>& keys, const std::list<V>& values)
{
    std::list<std::pair<K, V> > ret;
    typedef typename std::list<K>::const_iterator KeyIter;
    typedef typename std::list<V>::const_iterator ValueIter;

    KeyIter ik = keys.begin();
    ValueIter iv = values.begin();

    for(; ik != keys.end(); ik++, iv++)
    {
        if(iv != values.end())
            ret.push_back(std::make_pair(*ik,  *iv));
        else
            break;
    }
    return ret;
}

template<class InputIterator, typename T2>
std::list<T2> pick_second(InputIterator start, InputIterator stop)
{
    std::list<T2> ret;
    for(;start != stop; start++)
        ret.push_back(start->second);
    return ret;
}

namespace hdf4 {
extern void add_missing_objects(eoslib::group *);
}

namespace eoslib {
MODISType mtype = MOD_DEFAULT_TYPE; 
//int modis_sg_type = -1;

// Adjust values of latitude and longitude data fields.
extern int CorLatLon (float*, int, int, int);
extern int CorLatLon (double*, int, int, int);

// Remove special characters not allowed by CF convertion.
static std::string remove_special_chars(const std::string&);

// Recursively traverse all items in the tree structure.
/*#define APPLY_RECURSIVELY(func, g, arg) \
{ \
    std::list<group*>::iterator it; \
    std::list<group*>& children = (g)->get_child_groups(); \
    for(it = children.begin(); it != children.end(); it++) \
        (func)(*it, arg); \
}
*/

void apply_recursively(modifier_t func, group *g, void *arg)
{
    std::list<group*>::iterator it;
    std::list<group*>& children = (g)->get_child_groups();
    for(it = children.begin(); it != children.end(); it++)
        (func)(*it, arg);
} 

// Some objects, like VDATA, are not readable by HDF-EOS2 APIs. 
// This function will add these missing items to the corresponding 
// group by utilizing HDF4 APIs.
int add_missing_objects(group *root, void *)
{
    hdf4::add_missing_objects(root);
    return 0;
}

/**
 * This function adds an "origname" attribute to all variables.
 * The added origname attribute contains the full path of the
 * original variable. Since the variable can be renames and
 * moved to a different location, origname is important.
 *
 */
int _add_origname(group *g, const std::string& pathstr)
{
    // For each group
    std::list<group*>::iterator git; 
    std::list<group*>& children = (g)->get_child_groups();

    for(git = children.begin(); git != children.end(); git++) 
    {
        std::string grouppath = pathstr + "/" + (*git)->get_name();
        _add_origname(*git, grouppath);
    }

    // For each variable
    std::list<var*>::iterator vit; 
    std::list<var*>& vars= (g)->get_vars(); 
    for(vit = vars.begin(); vit != vars.end(); vit++) 
    {
        std::string varpath = pathstr + "/" + (*vit)->get_origname();
        std::string orig_var_name = (*vit)->get_origname();
        std::string change_var_name = (*vit)->get_name();

        // If the orig_var_name is the same as the change_var_name, 
        // normal objects. otherwise, special objects. orig_var_name should be used.
        if(orig_var_name != change_var_name) {
            if(orig_var_name[0] !='/')
                 varpath = "/" + orig_var_name;
            else 
                 varpath = orig_var_name;
        }


        mem_attr *m_attr = new mem_attr(*vit, "origname");
        m_attr->set_value(varpath);
        (*vit)->add_attr(m_attr);

        //For debugging only.
        /*std::cout << (*vit)->get_name() << std::endl;
        std::list<attr*>& attrs = (*vit)->get_attrs();
        std::list<attr*>::iterator it;
        for(it=attrs.begin(); it!=attrs.end(); it++)
        {
            (*it)->dump_r(10);
        }
        attr *attribute = (*vit)->get_attr_by_name("FillValue");
        if(attribute != NULL)
        {
            std::string s;
            attribute->get_value_str(&s);
            std::cout << s << std::endl;
        } else
        {
            std::cout << "no such attribute" << std::endl;
        }*/
		
        attr *attribute = (*vit)->get_attr_by_name("long_name");
        if(attribute==NULL) //No "long_name" attribute.
        {
            m_attr = new mem_attr(*vit, "long_name");
            m_attr->set_value(varpath); //(*vit)->get_name());
            (*vit)->add_attr(m_attr);
        }
    } // end of  for(vit = vars.begin 

    return 0;
}

// Add "origname" attribute.
int add_origname(group *root, void *)
{
    return _add_origname(root, "");
}

// If an attribute is named _dimap_reduced, it will be renamed _reduced.
int rename_dimmap_orig(group *g, void *arg)
{
    // Recursive calls
    //APPLY_RECURSIVELY(rename_dimmap_orig, g, arg);
    apply_recursively(rename_dimmap_orig, g, arg);	

    const std::list<var*>& vars = g->get_vars_c();
    std::list<var*>::const_iterator it;
    for(it = vars.begin(); it != vars.end(); it++)
    {
        var *v = (*it);
        attr *a = v->get_attr_by_name("_dimmap_reduced");
        if(a)
        {
            v->rename(v->get_name() + "_reduced");
            v->remove_attr(a);
        }
    } // end of for 
    return 0;
}

// This filter will flatten the group structure, essentially all
// the variables under the non-root group will be moved to the 
// root group.
int flatten_modifier(group *root, void *)
{
    std::list<group*>& children = root->get_child_groups();
    if(root->get_child_groups().size() >1) 
	root->set_num_of_orig_group_status(true);
    else 
	root->set_num_of_orig_group_status(false);

    std::map<dim*, dim*> dim_conv;

    //std::multimap<dim*, dim*> dim_conv_rev;
    std::list<dim*> added_dims;

    std::map<var*, var*> var_conv;
    //std::multimap<var*, var*> var_conv_rev;

    // added_vars seems not to be used in this routine. KY 2013-01-30
    //std::list<var*> added_vars;

    std::map<attr*, attr*> attr_conv;
    std::multimap<attr*, attr*> attr_conv_rev;
    std::list<attr*> added_attrs;

    /////////
    // dims
    /////////
    {

        std::list<dim*> all_dims = root->get_dims_r();
        std::list<std::string> all_dimnames;

        // Copy names of all dimensions to all_dimnames.
        transform(all_dims.begin(), all_dims.end(), back_inserter(all_dimnames), dim::s_get_name);
        
        // Create a pair list of all_dimnames,all_dims
        std::list<std::pair<std::string, dim*> > ndl = pair_list(all_dimnames, all_dims);
        std::multimap<std::string, dim*> dimmap(ndl.begin(), ndl.end());
        typedef std::multimap<std::string, dim*> dimmap_t;
		
        for(dimmap_t::iterator it = dimmap.begin(); it != dimmap.end();)
        {
            const std::string& name = it->first;
            std::pair<dimmap_t::iterator, dimmap_t::iterator> p = dimmap.equal_range(name);
            dimmap_t::iterator it1 = p.first;
            dimmap_t::iterator it2 = p.second;

            std::list<dim*> same_name_dims = pick_second<dimmap_t::iterator, dim*>(it1, it2);
            std::list<dim*> uniq_dims;

            std::unique_copy(
                same_name_dims.begin(),
                same_name_dims.end(),
                back_inserter(uniq_dims),
                dim::equivalence_test
            );
			
            // This if-block checks if the dimension(name, size and has a unique cv)
            if(uniq_dims.size() == 1)
            {
                dim* ud = *(uniq_dims.begin());

                // Some objects are added by the HDF4 APIs, they should not be cloned. Otherwise, memory leaking occurs due to the improper reference count.
                if(false == ud->NOT_clone()) {

                    dim* d = ud->clone(root);

                    //dim* d = new mem_dim(ud->get_name(), ud->get_size());
                    root->add_dim(d);
                    added_dims.push_back(d);

                    // Connect the original dimensions to the same cloned dimension 
                    for(std::list<dim*>::iterator isd=same_name_dims.begin();
                        isd != same_name_dims.end();
                        isd++)
                    {
                        dim_conv[*isd] = d;

                       // The reversed connection(dim_conv_rev) is not used at all, it may be useful for debugging.
                       // Comment out and leave it the line. 
                       // dim_conv_rev.insert(std::make_pair(d, *isd));
                    }
                }
                else {// This else-block may be redundant since dim_conv seems just for the clone case.May remove it later. TEMPKY 2014-11-24

                    for(std::list<dim*>::iterator isd=same_name_dims.begin();
                        isd != same_name_dims.end();
                        isd++)
                    {
                        dim_conv[*isd] = ud;
                      // dim_conv_rev.insert(std::make_pair(ud, *isd));
                    }

                }

            } // end of if(uniq_dims.size() == 1)
            else
            {
                // Will do the following
                // new name = dim name + "_" + gridname
                // duplicate dim (newname)
                // add it to the root
                // add to the conversion map

                for(std::list<dim*>::iterator it = same_name_dims.begin();
                    it != same_name_dims.end();
                    it++)
                {
                    dim *u = *it;

                    if(false == u->NOT_clone()) {

                        dim *w = u->clone(root);

                        // Follow the HDF4 OPeNDAP handler, prepend the path, KY 2013-01-10
                        if (u->get_group()->get_name() !="/") 
                            w->rename(remove_special_chars(u->get_group()->get_name())+"_"+u->get_name()); 
                        //u->get_group()->get_name());
                        //w->rename(u->get_name() + "_" + remove_special_chars(u->get_group()->get_name())); //u->get_group()->get_name());

                        root->add_dim(w);
                        added_dims.push_back(w);
                        dim_conv[u] = w;
                    }
                } // end of for
                //std::cout << "===============" << std::endl;
            } // end of else

            it = it2;
        } //end of for
    } // end of dims
    //root->dump_r(0);

    /////////
    // vars
    /////////
    {
        std::list<var*> all_vars = root->get_vars_r();

        //std::map<var*,
        std::list<std::string> all_varnames;
        transform(all_vars.begin(), all_vars.end(), back_inserter(all_varnames), var::s_get_name);
        std::list<std::pair<std::string, var*> > nvl = pair_list(all_varnames, all_vars);
        std::multimap<std::string, var*> varmap(nvl.begin(), nvl.end());
        typedef std::multimap<std::string, var*> varmap_t;
        for(varmap_t::iterator it = varmap.begin(); it != varmap.end();)
        {
            const std::string& name = it->first;

            std::pair<varmap_t::iterator, varmap_t::iterator> p = varmap.equal_range(name);
            varmap_t::iterator it1 = p.first;
            varmap_t::iterator it2 = p.second;

            std::list<var*> same_name_vars = pick_second<varmap_t::iterator, var*>(it1, it2);
            std::list<var*> uniq_vars;
			
            // s_same_obj_test will compare the variable name,dimensions and values. Essentially
            // it may reduce the redundant variables.
            std::unique_copy(
                same_name_vars.begin(),
                same_name_vars.end(),
                back_inserter(uniq_vars),
                var::s_same_obj_test
            );

            
            // The following code is trying to copy all variables to the root directory. KY 2013-01-30
            if(uniq_vars.size() == 1)// This var is unique
            {
                var* u = *(uniq_vars.begin());

                // Added HDF4 objects should be not cloned. They are always at the top directory.
                if(u->NOT_clone() != true) {
                    var* w = u->clone_r(root, dim_conv);
                    root->add_var(w);
                    //added_vars.push_back(w);

                    for(std::list<var*>::iterator isv=same_name_vars.begin();
                        isv != same_name_vars.end();
                        isv++)
                    {
                        //Build the connections of the vars that have the same names as the cloned var.
                        var_conv[*isv] = w;

                        // var_conv_rev is not used. It may be useful for debugging. Keep the line but comment it out. 
                        //var_conv_rev.insert(std::make_pair(w, *isv));
                    }
                }
                else { // This else block may be redundant since var_conv may be only applied to the cloned case.
                    for(std::list<var*>::iterator isv=same_name_vars.begin();
                        isv != same_name_vars.end();
                        isv++)
                    {
                        var_conv[*isv] = u;

                        // var_conv_rev is not used. It may be useful for debugging. Keep the line but comment it out. 
                        //var_conv_rev.insert(std::make_pair(w, *isv));
                    }
 
                }
            } // end of if(uniq_vars.size() == 1)
            else
            {
                // new name = var name + "_" + gridname
                // duplicate var (newname)
                // add it to the root
                // add to the conversion map
                for(std::list<var*>::iterator it = same_name_vars.begin();
                    it != same_name_vars.end();
                    it++)
                {
                    var *u = *it;
                    if(false == u->NOT_clone()) {

                        var *w = u->clone_r(root, dim_conv);
                        // If the var name is not unique, prepend with the group path by following the HDF4 OPeNDAP handler.
                        // KY 2013-01-10
                        if(u->get_group()->get_name()!="/") 
                            w->rename(remove_special_chars(u->get_group()->get_name())+"_"+u->get_name()); //u->get_group()->get_name());
                        root->add_var(w);
                        var_conv[u] = w;
                    }
                } // end of for

            } // end of else
			
            it = it2;
        } // end of for

    } // end of vars
//    root->dump_r(0);

// Leave the following debugging code for the future enhancement. KY 2013-05-30
#if 0
{
std::cerr<<"total number of variables before flattening CVs is "<<all_vars.size() <<std::endl;

for(std::list<var*>::iterator it = all_vars.begin();
            it != all_vars.end(); it++)
{
std::cerr<<"variable name before flattening CVs is "<<(*it)->get_name() <<std::endl;
}
}
#endif

    /////////
    // CVs
    /////////
    {
        std::list<dim*>::iterator adi;
        for(adi = added_dims.begin(); adi != added_dims.end(); adi++)
        {
            dim *newdim = *adi;
            newdim->replace_cv(var_conv);
	}
    } // end of CVs
    //root->dump_r(0);

// Leave the following debugging code for the future enhancement. KY 2013-05-30
#if 0
{

std::list<var*> all_vars = root->get_vars_r();
std::cerr<<"all variable size "<<all_vars.size() <<std::endl;

std::list<var*> dummy_vars = root->get_vars();
std::cerr<<"variable size just under the root is "<<dummy_vars.size() <<std::endl;

for(std::list<var*>::iterator it = dummy_vars.begin();
            it != dummy_vars.end(); it++)
{
std::cerr<<"variable name under the ROOT after flattening is "<<(*it)->get_name() <<std::endl;
std::list<dim*> dummy_dims = (*it)->get_dims();
for(std::list<dim*>::iterator mit = dummy_dims.begin();
            mit != dummy_dims.end(); mit++)
std::cerr<<"var dim. name under the ROOT after flattening is: "<<(*mit)->get_name() <<std::endl;


}
#endif

    /////////
    // Attributes
    /////////
    {
        std::list<attr*> all_attrs_to_copy;
        for(std::list<group*>::iterator it = children.begin();
            it != children.end(); it++)
        {
            std::list<attr*> tmp = (*it)->get_attrs_c();
            all_attrs_to_copy.splice(all_attrs_to_copy.end(), tmp);
        }
		
        std::list<std::string> all_attrnames;
        transform(all_attrs_to_copy.begin(), all_attrs_to_copy.end(), 
            back_inserter(all_attrnames), attr::s_get_name);
        std::list<std::pair<std::string, attr*> > nvl = 
            pair_list(all_attrnames, all_attrs_to_copy);
        std::multimap<std::string, attr*> attrmap(nvl.begin(), nvl.end());
        typedef std::multimap<std::string, attr*> attrmap_t;
        for(attrmap_t::iterator it = attrmap.begin(); it != attrmap.end();)
        {
            const std::string& name = it->first;

            std::pair<attrmap_t::iterator, attrmap_t::iterator> p = attrmap.equal_range(name);
            attrmap_t::iterator it1 = p.first;
            attrmap_t::iterator it2 = p.second;

            std::list<attr*> same_name_attrs = pick_second<attrmap_t::iterator, attr*>(it1, it2);
            std::list<attr*> uniq_attrs;
			
            std::unique_copy(
                same_name_attrs.begin(),
                same_name_attrs.end(),
                back_inserter(uniq_attrs),
                attr::s_same_obj_test
            );

            if(uniq_attrs.size() == 1)
            {
                attr* u = *(uniq_attrs.begin());
                attr* w = u->clone();
                root->add_attr(w);
                added_attrs.push_back(w);

                for(std::list<attr*>::iterator isv=same_name_attrs.begin();
                    isv != same_name_attrs.end();
                    isv++)
                {
                    attr_conv[*isv] = w;
                    attr_conv_rev.insert(std::make_pair(w, *isv));
                }
            } // end of if(uniq_attrs.size() == 1)
            else
            {
                // new name = attr name + "_" + gridname
                // duplicate attr (newname)
                // add it to the root
				
                for(std::list<attr*>::iterator temp_it = same_name_attrs.begin();
                    temp_it != same_name_attrs.end();
                    temp_it++)
                {
                    attr *u = *temp_it;
                    attr *w = u->clone();
                    if(u->get_group() == NULL)
                    {
                        throw std::runtime_error("Something wrong here(" __FILE__ ":"TOSTRING(__LINE__)")" );
                    }
                    else {
                        // w->rename(u->get_name() + "_" + u->get_group()->get_name());
                         // Follow the HDF4 OPeNDAP handler, prepend the path, KY 2013-01-10
                         w->rename(remove_special_chars(u->get_group()->get_name())+"_"+u->get_name()); //u->get_group()->get_name());

                    }
                    root->add_attr(w);
                    added_attrs.push_back(w);
                } // end of for
            } // end of else
			
            //std::cout << name << ", " << same_name_attrs.size() << ", " << uniq_attrs.size() << std::endl;
            it = it2;
        } // end of for(attrmap_t::iterator
    } // end of Attributes

    while(!root->get_child_groups().empty())
    {
        std::list<group*>::iterator it = root->get_child_groups().begin();
        group *g = *it;
		
        root->remove(it);
        delete g;	
    } // ene of while

    return 0;
}


/**
 * This is an example code and not intended to be used int the
 * final version.
 */
#if 0
/*
int rename_modifier(group *root, void *)
{
    std::list<group*>::iterator it;
    std::list<group*>& children = root->get_child_groups();
    int group_index = 0;
    while(!children.empty())
    {
        // suffix
        std::ostringstream oss;
        oss << "_" << group_index;
        std::string suffix = oss.str();

        group *g = *(children.begin());
        // rename all items in g

        // vars
        for(std::list<var*>::iterator it = g->get_vars().begin();
            it != g->get_vars().end(); it++)
        {
            std::string newname = (*it)->get_name() + suffix;
            (*it)->rename(newname);
        }

        // dims
        for(std::list<dim*>::iterator it = g->get_dims().begin();
            it != g->get_dims().end(); it++)
        {
            std::string newname = (*it)->get_name() + suffix;
            (*it)->rename(newname);
        }

        // attrs
        for(std::list<attr*>::iterator it = g->get_attrs().begin();
            it != g->get_attrs().end(); it++)
        {
            std::string newname = (*it)->get_name() + suffix;
            (*it)->rename(newname);
        }

        g->move_all(root);
        delete g;
		
        children.pop_front();
        group_index++;
    } // end of while
    return 0;
}
*/
#endif


/**
 * Latitude, LatCenter -> latitude
 * Longitude, LonCenter  -> longitude
 *
 * If a group has raw lat/lon variables and
 * calculated lat/lon at the same time,
 * we prefer raw values.
 *
 * If it has calculated lat/lon only, 
 * we need to adjust the names.
 */

static const char *lat_final = "latitude";
static const char *lon_final = "longitude";
static const char* lat_names[] = {"Latitude", "LatCenter", "Lat", "YDim", "center_latitude"};
static const char* lon_names[] = {"Longitude", "LonCenter", "Lon", "XDim", "center_longitude"};
extern std::string EOS2_CALCULATED_LATITUDE; // eos2_group.cpp
extern std::string EOS2_CALCULATED_LONGITUDE; // eos2_group.cpp
const char *EOS2_LOCGRID_LATITUDE = "locgrid_latitude";
const char *EOS2_LOCGRID_LONGITUDE= "locgrid_longitude";

static std::string get_cf_string(const std::string& oldname) {

    std::string name_no_front_slash;
    if ((oldname.size() >1) && ('/' == oldname[0])) 
        name_no_front_slash = oldname.substr(1);
    else 
        name_no_front_slash = oldname;
    return remove_special_chars(name_no_front_slash);
} 
        

/**
 * To convert a string to a CF-compliant name, 
 * it is sufficient to replace non-allowed characters to
 * underscores.
 */
static std::string remove_special_chars(const std::string& oldname)
{
    std::string newname = oldname;

    std::string::iterator it;
    for(it = newname.begin(); it != newname.end(); it++)
    {
        if('0'<= *it && *it <= '9')
            ; // numeric
        else if('A'<= *it && *it <= 'Z')
            ; // capital alpha
        else if('a'<= *it && *it <= 'z')
            ; // lower-case alpha
        else if('_' == *it)
            ; // underscore
        else
            *it = '_';
    } // end of for
    return newname;
}

/**
 *  If a name starts with numeric characters, "_" will be prepended.
 */
int handle_special_chars(group* g, void *arg)
{
    std::list<group*>& groups = g->get_child_groups();
    for(std::list<group*>::iterator it = groups.begin();
        it != groups.end(); it++)
    {
        std::string oldname = (*it)->get_name();
        std::string newname = remove_special_chars(oldname);
        if(newname != oldname)
            (*it)->rename(newname);

        if(isdigit(newname[0]))
        {
            newname = '_' + newname;
            (*it)->rename(newname);
        }
    } // end of for

    std::list<dim*>& dims = g->get_dims();
    for(std::list<dim*>::iterator it = dims.begin();
        it != dims.end(); it++)
    {
        std::string oldname = (*it)->get_name();
        std::string newname = remove_special_chars(oldname);
        if(newname != oldname)
            (*it)->rename(newname);

        if(isdigit(newname[0]))
        {
            newname = '_' + newname;
            (*it)->rename(newname);
        }
    } // end of for

    std::list<attr*>& attrs = g->get_attrs();
    for(std::list<attr*>::iterator it = attrs.begin();
        it != attrs.end(); it++)
    {
        std::string oldname = (*it)->get_name();
        std::string newname = remove_special_chars(oldname);
        if(newname != oldname)
            (*it)->rename(newname);

        if(isdigit(newname[0]))
        {
            newname = '_' + newname;
            (*it)->rename(newname);
        }
    } // end of for

    std::list<var*>& vars = g->get_vars();
    for(std::list<var*>::iterator it = vars.begin();
        it != vars.end(); it++)
    {
        std::string oldname = (*it)->get_name();
        std::string newname = remove_special_chars(oldname);
        if(newname != oldname)
            (*it)->rename(newname);

        if(isdigit(newname[0]))
        {
            newname = '_' + newname;
            (*it)->rename(newname);
        }

        std::list<attr*>& vattrs = (*it)->get_attrs();
        for(std::list<attr*>::iterator temp_it = vattrs.begin();
            temp_it != vattrs.end(); temp_it++)
        {
            std::string oldname = (*temp_it)->get_name();
            std::string newname = remove_special_chars(oldname);
            if(newname != oldname)
                (*temp_it)->rename(newname);

            if(isdigit(newname[0]))
            {
                newname = '_' + newname;
                (*temp_it)->rename(newname);
            }
        } // end of for
    } // end of for

    // Recursive calls
    //APPLY_RECURSIVELY(handle_special_chars, g, arg);
    apply_recursively(handle_special_chars, g, arg);

    return 0;
}

void resolve_name_clashing(const std::list<renamable*>& items)
{
    bool bExecuteMore = true;
    while(bExecuteMore)
    {
        bExecuteMore = false;
        std::multimap<std::string, renamable*> all;
        for(std::list<renamable*>::const_iterator it = items.begin();
            it != items.end(); it++)
        {
            all.insert(std::make_pair((*it)->get_name(), *it));
        }

        std::multimap<std::string, renamable*>::iterator it1, it2, it3;
        for(it1 = all.begin(); it1 != all.end();)
        {
            // It seems that this will only work when the list is an ordered list
            // For example, "A","A","B","C", what if the list is "A","B","C","A"
            // Can this algorithm resolve the name clashing for the late case? KY 2012-12-20
            // Yes, the algorithm still works since multimap is sorted by keys.KY 2012-12-21
            it2 = it1;
            it2 ++;
            if(it2 != all.end() && (it1->first == it2->first))
            {
                bExecuteMore = true;
                it3 = all.upper_bound(it1->first);
                for(int index=0; it1 != it3; it1++, index++)
                {
                    std::string oldname=it1->first;
                    std::ostringstream oss;
                    oss << oldname << "_" << index;
                    it1->second->rename(oss.str());
                } // end of for
            } // end of if(it2 != all.end
            else 
               it1++;
        } // end of for
    } // end of while
}

// Resolve name clashing in groups, attribures, and variables.
int final_name_clashing_check(group* g, void *arg)
{

    // Groups
    {
        std::list<group*>& groups = g->get_child_groups();
        std::list<renamable*> r;
        for(std::list<group*>::iterator it = groups.begin();
            it != groups.end(); it++)
        {
            r.push_back(static_cast<renamable*>(*it));
        }
        resolve_name_clashing(r);
    } // end of Groups

    // Attributes 
    {
        std::list<attr*>& attrs = g->get_attrs();
        std::list<renamable*> r;
        for(std::list<attr*>::iterator it = attrs.begin();
            it != attrs.end(); it++)
        {
            r.push_back(static_cast<renamable*>(*it));
        }
        resolve_name_clashing(r);
    } // end of Attributes

    // Variables 
    {
        std::list<var*>& vars = g->get_vars();
        std::list<renamable*> r;
        for(std::list<var*>::iterator it = vars.begin();
            it != vars.end(); it++)
        {
            r.push_back(static_cast<renamable*>(*it));
        }
        resolve_name_clashing(r);

// Leave the debugging info.
#if 0
for(std::list<var*>::iterator it = vars.begin();
            it != vars.end(); it++)
{
std::cerr<<"variable name after name clashing is "<<(*it)->get_name() <<std::endl;
}
#endif

    } // end of Variables

    // Variable's attributes
    {
        std::list<var*>& vars = g->get_vars();
        for(std::list<var*>::iterator it = vars.begin();
            it != vars.end(); it++)
        {
            std::list<attr*>& attrs = (*it)->get_attrs();
            std::list<renamable*> r;
            for(std::list<attr*>::iterator ia = attrs.begin();
                ia != attrs.end(); ia++)
            {
                r.push_back(static_cast<renamable*>(*ia));
            }
            resolve_name_clashing(r);
        } // end of for
    } // end of Variables's attributes

// Leave the debugging info.
#if 0
std::list<dim*>& fdims = g->get_dims();
for(std::list<dim*>::iterator it = fdims.begin();
            it != fdims.end(); it++)
{
std::cerr<<"clashing final dimension names is "<<(*it)->get_name() <<std::endl;
}           
#endif

    // Recursive calls
    //APPLY_RECURSIVELY(final_name_clashing_check, g, arg);
    apply_recursively(final_name_clashing_check, g, arg);

    return 0;
}

//! Rename lat/lon-related variable names
/**
 *  This function renames all variable names which are useded as
 *  nicknames of lat/lon.
 *
 */
int latlon_rename(group* g, void *arg)
{
    // Recursive calls
    //APPLY_RECURSIVELY(latlon_rename, g, arg);
    apply_recursively(latlon_rename, g, arg);

    std::map<std::string, std::string> rename_map;
    for(int i=0; i<sizeof(lat_names) / sizeof(char*); i++)
        rename_map[lat_names[i]] = lat_final;
    for(int i=0; i<sizeof(lon_names) / sizeof(char*); i++)
        rename_map[lon_names[i]] = lon_final;

    const std::list<var*>& vars = g->get_vars_c();
    std::list<var*>::const_iterator it;
    for(it = vars.begin(); it != vars.end(); it++)
    {
        std::map<std::string, std::string>::const_iterator im;
        im = rename_map.find((*it)->get_name());
        if(im != rename_map.end())
        {
            (*it)->rename(im->second);
            (*it)->set_cv();
        }
    } // end of for
	
    // Leave the following code for the time being. KY 2013-05-31

#if 0
    // If there is a variable named Latitude, LatCenter, etc.,
    // it is renamed to latitude.
    // For longitude, similar procedure is done.
    // Note that, if there are more than one such variable,
    // an exception is thrown.
    {
        var *v_latitude, *v_longitude;
        v_latitude = g->get_var_by_name(lat_final);
        v_longitude = g->get_var_by_name(lon_final);

        if((v_latitude && !v_longitude) || (!v_latitude && v_longitude))
            throw std::runtime_error("Cannot get latitude/longitude information(" __FILE__ ":" TOSTRING(__LINE__)")" );
		
        std::list<var*> v_lat_candidates, v_lon_candidates;
        for(int i=0; i<sizeof(lat_names) / sizeof(char*); i++)
        {
            var *v = g->get_var_by_name(lat_names[i]);
            if(v)
                v_lat_candidates.push_back(v);
        } // end of for
        if(v_lat_candidates.size() == 0)
            ; // OK
        else if(v_lat_candidates.size() == 1)
        {
            (*v_lat_candidates.begin())->rename(lat_final);
        }
        else
        {
            // Conflict. >= 2
            throw std::runtime_error("There are multiple candidate variables for latitude(" __FILE__ ":" TOSTRING(__LINE__)")" );
        }

        for(int i=0; i<sizeof(lon_names) / sizeof(char*); i++)
        {
            var *v = g->get_var_by_name(lon_names[i]);
            if(v)
                v_lon_candidates.push_back(v);
        }
        if(v_lon_candidates.size() == 0)
            ; // OK
        else if(v_lon_candidates.size() == 1)
        {
            (*v_lon_candidates.begin())->rename(lon_final);
        }
        else
        {
            // Conflict. >= 2
            throw std::runtime_error("There are multiple candidate variables for longitude(" __FILE__ ":" TOSTRING(__LINE__)")" );
        }
    }

    // Calculated
    {
        var *v_latitude = NULL;
        var *v_longitude = NULL;
        v_latitude = g->get_var_by_name(lat_final);
        v_longitude = g->get_var_by_name(lon_final); 
        if((v_latitude && !v_longitude) || (!v_latitude && v_longitude))
            throw std::runtime_error("Only one of latitude or longtidue exists(" __FILE__ ":" TOSTRING(__LINE__)")" );

        var *vlg_latitude = NULL;
        var *vlg_longitude = NUL;
        vlg_latitude = g->get_var_by_name(EOS2_LOCGRID_LATITUDE);
        vlg_longitude = g->get_var_by_name(EOS2_LOCGRID_LONGITUDE);
        if((vlg_latitude && !vlg_longitude) || (!vlg_latitude && vlg_longitude))
            throw std::runtime_error("Only one of latitude or longtidue in location grid exists(" __FILE__ ":" TOSTRING(__LINE__)")" );

        var *vc_latitude = NULL;
        var *vc_longitude = NULL;
        vc_latitude = g->get_var_by_name(EOS2_CALCULATED_LATITUDE);
        vc_longitude = g->get_var_by_name(EOS2_CALCULATED_LONGITUDE); 
        if((vc_latitude && !vc_longitude) || (!vc_latitude && vc_longitude))
            throw std::runtime_error("Only one of calculated latitude or calculated longtidue exists(" __FILE__ ":" TOSTRING(__LINE__)")" );

        if(v_latitude)
        {
            v_latitude->set_cv();
            if(vlg_latitude)
            {
                g->remove_var(vlg_latitude);
                delete vlg_latitude;
            }
            if(vc_latitude)
            {
                g->remove_var(vc_latitude);
                delete vc_latitude;
            }

            v_longitude->set_cv();
            if(vlg_longitude)
            {
                g->remove_var(vlg_longitude);
                delete vlg_longitude;
            }
            if(vc_longitude)
            {
                g->remove_var(vc_longitude);
                delete vc_longitude;
            }
        } // end of if(v_latitude)
        else if(vlg_latitude)
        {
            vlg_latitude->set_cv();
            if(vc_latitude)
            {
                g->remove_var(vc_latitude);
                delete vc_latitude;
            }
            vlg_latitude->rename(lat_final);

            vlg_longitude->set_cv();
            if(vc_longitude)
            {
                g->remove_var(vc_longitude);
                delete vc_longitude;
            }
            vlg_longitude->rename(lon_final);
        } // end of else if(vlg_latitude)
        else if(vc_latitude)
        {
            vc_latitude->set_cv();
            vc_latitude->rename(lat_final);

            vc_longitude->set_cv();
            vc_longitude->rename(lon_final);
        } // ene of else if(vc_latitude)
    } // end of Calculated
#endif
    return 0;
}

//! Handle name conflicts among variables w/ the same name in a group
/**
 * This function checks all variables in each group.
 * If there are multiple variables with the same name,
 * all other than the one w/ the top priority are deleted.
 * If there are more than 1 var with the top priority,
 * this function return -1.
 * Otherwise, it return 0.
 */
int remove_redundant_vars(group* g, void *arg)
{
    //APPLY_RECURSIVELY(remove_redundant_vars, g, arg);
    apply_recursively(remove_redundant_vars, g, arg);

    // constructing varinfo
    std::map<std::string, std::map<int, std::list<var*> > > varinfo;
    const std::list<var*>& vars = g->get_vars_c();
    std::list<var*>::const_iterator it;
    for(it = vars.begin(); it != vars.end(); it++)
    {
        var *v = (*it);
        std::string name = (*it)->get_name();
        int priority = 0;

        attr *a = v->get_attr_by_name("_e2lib_name_conflict_priority");
        if(a==NULL)
            continue;

        std::string priority_str;
        a->get_value_str(&priority_str);
        std::istringstream is(priority_str);
        is >> priority;

        std::map<
            std::string, 
            std::map<
                int, 
                std::list<var*> 
            > 
        >::iterator im;
        im = varinfo.find(name);
        if(im == varinfo.end())
        {
            std::map<int, std::list<var*> > m;
            std::list<var*> n;
            n.push_back(v);
            m[priority] = n;
            varinfo[name] = m;
        }
        else
        {
            std::map<int, std::list<var*> >& u = im->second;
            std::map<int, std::list<var*> >::iterator iu;
            iu = u.find(priority);
            if(iu == u.end())
            {
                std::list<var*> tmp;
                tmp.push_back(v);
                u[priority] = tmp;
            }
            else
            {
                iu->second.push_back(v);
            }
        } // end of if(im == varinfo.end())
    } // end of for(it = vars.begin()

    std::map<std::string, std::map<int, std::list<var*> > >::iterator iv;
    for(iv = varinfo.begin(); iv != varinfo.end(); iv++)
    {
        std::map<int, std::list<var*> >::iterator im;
        im = iv->second.begin();
        //int min_priority = im->first;
        if(im->second.size() != 1)
        {
            // There are more than 1 var w/ the min priority.
            return -1;
        }
        var* topvar = *(im->second.begin());

        im++;
        for(; im != iv->second.end(); im++)
        {
            //int pri = im->first;
            std::list<var*>::iterator it3;
            for(it3 = im->second.begin(); it3 != im->second.end(); it3++)
            {
#if 0
                //std::cout << "====" << std::endl;
                //g->dump_r(2);
                // Delete *it3
                //std::cout << "DELETE THIS" << std::endl;
                //(*it3)->dump_r(2, false);
#endif
                g->replace_cv(*it3, topvar); 
                g->remove_var(*it3);

#if 0
                //std::cout << "RESULT" << std::endl;
                //g->dump_r(2);
                // Delete *it3
#endif
            } // end of for(it3 = 

           //Release resource held by removed vars,(HFCFLIB-113) 
            while(false == (im->second).empty()) {
                var* v = *((im->second).begin());
                delete v;
                (im->second).pop_front();
            }

        } // end of for(; im != iv->second
    } // end of for(iv = varinfo.begi

    return 0;
}

/**
 * This function removes _e2lib_name_conflict_prority attribute
 * from all variables.
 */
int remove_priority_attrs(group* g, void *arg)
{
    //APPLY_RECURSIVELY(remove_priority_attrs, g, arg);
    apply_recursively(remove_priority_attrs, g, arg);

    std::list<var*>& vars = g->get_vars();
    std::list<var*>::iterator it; 

    for(it = vars.begin(); it != vars.end(); it++)
    {
        var *v = (*it);
        attr *a = v->get_attr_by_name("_e2lib_name_conflict_priority");
        if(a)
            v->remove_attr(a);
    }
    return 0;
}

/**
 * NOTE that, eos2 files have structmetata, so
 * we can use projection information for quicker condensation.
 *
 * This function checks whether the 2D lat/lon vars
 * can be condensed to 1D variable for grid only. This is posibble
 * only when the grid uses geographic projection.
 */

int condense_2d_latlon(group* g, void *arg)
{

    //APPLY_RECURSIVELY(condense_2d_latlon, g, arg);

    apply_recursively(condense_2d_latlon, g, arg);

    eos2_group *g2 = (eos2_group*) g;
    if(g2->get_type() != GRID)
        return 0;

    int32 projcode = -1;
    if(g->get_name()!="/")
    {
        intn r = GDprojinfo(((eos2_group*)g)->get_handle()->get(), &projcode, NULL, NULL, NULL);
        if(r == -1)
             throw std::runtime_error("Cannot read projection gode in " + g2->get_name() + " (" __FILE__ ":" TOSTRING(__LINE__)")");
    }

    // Only when the projection code is GCTP_GEO or GCTP_CEA, the code needs to be condensed.
    if (projcode != GCTP_GEO && projcode != GCTP_CEA)
        return 0;

    var *v_latitude = NULL;
    var *v_longitude = NULL;
    std::list<var*> vars_lat;
    std::list<var*> vars_lon;

    //g->dump_r(2);
    g->get_vars_by_name(lat_final, &vars_lat);

    if(vars_lat.empty())
        return 0;

    if(vars_lat.size() > 1)
    {

        //std::cout << g->get_name() << std::endl;
        //std::cout << vars_lat.size() << std::endl;

        //assert(0);
        //return -1;
        // It seems that we should not return FAIL here. Just return.
        return 0;
    }
    v_latitude = *(vars_lat.begin());

    g->get_vars_by_name(lon_final, &vars_lon);
    if(vars_lon.empty())
        return 0;
    if(vars_lon.size() > 1)
    {
        //assert(0);
        //return -1;
        // Just return.
        return 0;
    }
    v_longitude = *(vars_lon.begin());

    if(!v_latitude)
        return 0;
    if(!v_longitude)
        return 0;
    if(v_latitude->get_dims().size() != 2)
        return 0;
    if(v_longitude->get_dims().size() != 2)
        return 0;
	
    std::list<dim*>& dims = v_latitude->get_dims();
    if(v_longitude->get_dims() != dims)
        throw std::runtime_error("Latitude and longitude have different dims"
            "(" __FILE__ ":" TOSTRING(__LINE__)")" );

    dim* latdim = *dims.begin();
    dim* londim = *dims.rbegin();
	
    unsigned int ydim, xdim;
    ydim = latdim->get_size();
    xdim = londim->get_size();

    // Latitude
    void *latbuf = NULL;
    void *lonbuf = NULL;
    value_type_t lattype = v_latitude->get_type();
    value_type_t lontype = v_longitude->get_type();

    
    v_latitude->get_all(&latbuf);
    v_longitude->get_all(&lonbuf);

// The following code is redundant, however, we need to check carefully about the lat/lon slicing part. 
// The lat/lon may not always be YDIM major. KY 2013-02-07

//    if(projcode==GCTP_GEO || projcode==GCTP_CEA)
//        goto condense; //condensible
#if 0
    else
        goto condense_out;  // not condensible

    // Latitude
    {
        //v_latitude->get_all(&latbuf);

        if(lattype != DFNT_FLOAT32 && lattype != DFNT_FLOAT64 && lattype != DFNT_INT16)
            throw std::runtime_error("Variable type of latitude is float32 or float64 or int16.(" __FILE__ ":" TOSTRING(__LINE__)")" );

        std::vector<int32> pos1(2), pos2(2);
        // If the comparision cost need to be reduced, we may replace ydim with some
        // small value.
        for(unsigned int i=0; i<ydim; i++)
        {
            pos1[0]=i;
            pos1[1]=0;
            pos2 = pos1;

            if(lattype == DFNT_FLOAT32)
            {
                float32 val1 = v_latitude->get_item_at<float32>(latbuf, pos1);
                for(unsigned int j=1; j<xdim; j++)
                {
                    pos2[1] = j;
                    if(val1 != v_latitude->get_item_at<float32>(latbuf, pos2))
                        goto condense_out; // not condensible
                } // end of for
            } else if(lattype == DFNT_FLOAT64) 
            {
                float64 val1 = v_latitude->get_item_at<float64>(latbuf, pos1);
                for(unsigned int j=1; j<xdim; j++)
                {
                    pos2[1] = j;
                    if(val1 != v_latitude->get_item_at<float64>(latbuf, pos2))
                        goto condense_out; // not condensible
                }

            } else //lattype == DFNT_INT16
            {
                int16 val1 = v_latitude->get_item_at<int16>(latbuf, pos1);
                for(unsigned int j=1; j<xdim; j++)
                {
                    pos2[1] = j;
                    if(val1 != v_latitude->get_item_at<int16>(latbuf, pos2))
                        goto condense_out; // not condensible
                } // end of for
            } // end of else
            /*else // lattype == DFNT_FLOAT64
            {
                float64 val1 = v_latitude->get_item_at<float64>(latbuf, pos1);
                for(unsigned int j=1; j<xdim; j++)
                {
                    pos2[1] = j;
                    if(val1 != v_latitude->get_item_at<float64>(latbuf, pos2))
                        goto condense_out; // not condensible
                }
            }*/
        } // end of for
        // Now v_latitude is condensible
    } // end of Latitude

    // Longitude
    {
        //v_longitude->get_all(&lonbuf);

        if(lontype != DFNT_FLOAT32 && lontype != DFNT_FLOAT64 && lontype != DFNT_INT16)
            throw std::runtime_error("Variable type of longitude is float32 or float64 or int16.(" __FILE__ ":" TOSTRING(__LINE__)")" );

        std::vector<int32> pos1(2), pos2(2);
        for(unsigned int j=0; j<xdim; j++)
        {
            pos1[0]=0;
            pos1[1]=j;
            pos2 = pos1;

            if(lontype == DFNT_FLOAT32)
            {
                float32 val1 = v_longitude->get_item_at<float32>(lonbuf, pos1);
                for(unsigned int i=1; i<ydim; i++)
                {
                    pos2[0] = i;
                    if(val1 != v_longitude->get_item_at<float32>(lonbuf, pos2))
                        goto condense_out; // not condensible
                } // end of for
            } else if(lontype == DFNT_FLOAT64)
            {
                float64 val1 = v_longitude->get_item_at<float64>(lonbuf, pos1);
                for(unsigned int i=1; i<ydim; i++)
                {
                    pos2[0] = i;
                    if(val1 != v_longitude->get_item_at<float64>(lonbuf, pos2))
                        goto condense_out; // not condensible
                } // end of for
            } else //lontype == DFNT_INT16)
            {
                int16 val1 = v_longitude->get_item_at<int16>(lonbuf, pos1);
                for(unsigned int i=1; i<ydim; i++)
                {
                    pos2[0] = i;
                    if(val1 != v_longitude->get_item_at<int16>(lonbuf, pos2))
                        goto condense_out; // not condensible
                } // end of for
            } // end of else
            /*else // lontype == DFNT_FLOAT64
            {
                float64 val1 = v_longitude->get_item_at<float64>(lonbuf, pos1);
                for(unsigned int i=1; i<ydim; i++)
                {
                    pos2[0] = i;
                    if(val1 != v_longitude->get_item_at<float64>(lonbuf, pos2))
                        goto condense_out; // not condensible
                }
            }*/
        } //end of for
        // Now v_longitude is condensible
    } // end oof Longitude 

#endif

//condense:
    // Condensing latitude
    {
        mem_var *condensed_lat = new mem_var(g, lat_final);
        std::list<dim*> ld;
        ld.push_back(latdim);

        std::vector<int32> pos(2);
        pos[0]=pos[1]=0;

        mem_attr *condensed_lat_attr = new mem_attr(condensed_lat, "eoslib");
        condensed_lat_attr->set_value("This 1D variable is condensed from another 2D variable");
        condensed_lat->add_attr(condensed_lat_attr);

        if(lattype == DFNT_FLOAT32)
        {
            std::vector<float32> latslice;
            for(int i=0; i<ydim; i++)
            {
                pos[0] = i;
                latslice.push_back(v_latitude->get_item_at<float32>(latbuf, pos));
            } // end of for
            //condensed_lat->set(lattype, ld, (void*)&latslice[0]);

            attr *myattr =  v_latitude->get_attr_by_name("_FillValue");
            if(myattr!=NULL)
            {
                float32 myfillvalue = 0;
                myattr->get_value(&myfillvalue);
                int ifillvalue = (int) myfillvalue;
                int elms = latdim->get_size();
                int correctedfill = CorLatLon((float*)&latslice[0], 1, elms, ifillvalue);
                if(correctedfill==-1)
                    throw std::range_error("Fill values cannot be replaced.(" __FILE__ ":" TOSTRING(__LINE__)")" );
            } // end of if(myattr!=NULL)

            condensed_lat->set(lattype, ld, (void*)&latslice[0]);
        } // end of if(lattype == DFNT_FLOAT32)
        else // FLOAT64
        {
            std::vector<float64> latslice;
            for(int i=0; i<ydim; i++)
            {
                pos[0] = i;
                latslice.push_back(v_latitude->get_item_at<float64>(latbuf, pos));
            } // end of for
            //Haven't found latitude with double as data type.	
            attr *myattr =  v_latitude->get_attr_by_name("_FillValue");
            if(myattr!=NULL)
            {
                float32 myfillvalue = 0;
                myattr->get_value(&myfillvalue);
                int ifillvalue = (int) myfillvalue;
                int elms = latdim->get_size();
                int correctedfill = CorLatLon((double*)&latslice[0], 1, elms, ifillvalue);
                if(correctedfill==-1)
                    throw std::range_error("Fill values cannot be replaced.(" __FILE__ ":" TOSTRING(__LINE__)")" );
            } // end of if(myattr!=NULL) 
            condensed_lat->set(lattype, ld, (void*)&latslice[0]);
        } // end of else FLOAT64

        g->get_vars().push_back(condensed_lat);

        // Move all attributes
        std::list<attr*>& old_attrs = v_latitude->get_attrs();
        std::list<attr*>& new_attrs = condensed_lat->get_attrs();
        new_attrs.splice(new_attrs.end(), old_attrs);

        // Change coordinate variable of the latdim
        latdim->get_cv().clear();
        latdim->get_cv().push_back(condensed_lat);
    } // end of Condensing latitude

    // Condensing longitude
    {
        mem_var *condensed_lon = new mem_var(g, lon_final);
        std::list<dim*> ld;
        ld.push_back(londim);

        std::vector<int32> pos(2);
        pos[0]=pos[1]=0;

        mem_attr *condensed_lon_attr = new mem_attr(condensed_lon, "eoslib");
        condensed_lon_attr->set_value("This 1D variable is condensed from another 2D variable");
        condensed_lon->add_attr(condensed_lon_attr);

        if(lontype == DFNT_FLOAT32)
        {
            std::vector<float32> lonslice;
            for(int i=0; i<xdim; i++)
            {
                pos[1] = i;
                lonslice.push_back(v_longitude->get_item_at<float32>(lonbuf, pos));
            } // end of for
            condensed_lon->set(lontype, ld, (void*)&lonslice[0]);
        } // end of  if(lontype == DFNT_FLOAT32)
        else // FLOAT64
        {
            std::vector<float64> lonslice;
            for(int i=0; i<xdim; i++)
            {
                pos[1] = i;
                lonslice.push_back(v_longitude->get_item_at<float64>(lonbuf, pos));
            } // end of for
            condensed_lon->set(lontype, ld, (void*)&lonslice[0]);
        } // end of else FLOAT64
        g->get_vars().push_back(condensed_lon);

        // Move all attributes
        std::list<attr*>& old_attrs = v_longitude->get_attrs();
        std::list<attr*>& new_attrs = condensed_lon->get_attrs();
        new_attrs.splice(new_attrs.end(), old_attrs);
        // Change coordinate variable of the londim
        londim->get_cv().clear();
        londim->get_cv().push_back(condensed_lon);
    } // end of Condensing longitude 

    // Remove orig v_latitude, v_longitude
    g->remove_var(v_latitude);
    g->remove_var(v_longitude);

    delete v_latitude;
    delete v_longitude;

    if(latbuf && lattype == DFNT_FLOAT32)
        delete [] (float32*)latbuf;
    else if(latbuf && lattype == DFNT_FLOAT64)
        delete [] (float64*)latbuf;
    if(lonbuf && lontype == DFNT_FLOAT32)
        delete [] (float32*)lonbuf;
    else if(lonbuf && lontype == DFNT_FLOAT64)
        delete [] (float64*)lonbuf;

    return 0;
}

/**
 * Processing related to coordinate variables
 * Goal: Each dim needs to have coordinate variable.
 */
int find_cv_for_each_dim(group* g, void *arg)
{

    //APPLY_RECURSIVELY(find_cv_for_each_dim, g, arg);
    apply_recursively(find_cv_for_each_dim, g, arg);

    // If any variable has "is_cv" attribute,
    // it is considered as a coordinate variable
    // of all dimensions it is referring to.

    // Currently is_cv attribute is not added for any cv variables.
    // This may be useful for future NcML implementation, which one can
    // add a coordinate variable by following this convention. KY 2013-01-10
    {
        std::list<var*>& vars = g->get_vars();
        std::list<var*>::iterator it;
        for(it = vars.begin(); it != vars.end(); it++)
        {
            attr* a = (*it)->get_attr_by_name("is_cv");
            if(a)
                (*it)->set_cv();
            
        } // end of for
    }

    // For the dimensions which have no coordinate variable,
    // we check all 1D variables which uses the dimension.
    // If there is only one such variable, we treat it
    // as the coordinate variable of the dimension.
    {
        std::list<dim*>& dims = g->get_dims();
        std::list<dim*>::iterator it;
        for(it = dims.begin(); it != dims.end(); it++)
        {
            if((*it)->get_cv_c().size() == 0)
            {
                var *v = g->find_uniq_1d_var_using((*it));
                if(v)
                    v->set_cv();
            }
        } // end of for
    }

    // For the dimensions which have no coordinte variable,
    // we create dummy coord vars containing 0,1, 2, 3...
    {
        std::list<dim*>& dims = g->get_dims();
        std::list<dim*>::iterator it;
        for(it = dims.begin(); it != dims.end(); it++)
        {
            if((*it)->get_cv_c().size() == 0)
            {
                // Now we need to follow COARDS for all dimensions. 
                // This includes the third-dimension coordinate variable.
                // KY 2013-01-10
                std::string varname = (*it)->get_name();

                mem_var *v = new mem_var(g, varname);

                // For dump variables under the root,don't clone these variables.
                if(g->get_name() == "/")
                    v->unset_clone();

                std::vector<int32> buf((*it)->get_size());
                for(int i=0; i<buf.size(); i++)
                    buf[i] = i;

                std::list<dim*> tmp_dims;
                tmp_dims.push_back(*it);
                v->set(DFNT_INT32, tmp_dims, (void*)&buf[0]);

                mem_attr *a = new mem_attr(v, "eoslib");
                a->set_value("This is a dummy coordinate variable added by eoslib");
                v->add_attr(a);

                g->add_var(v);
                v->set_cv();
            } // end of if((*it)->get_cv_c(
        } // end of for(it = dims.begin(); i
    }
    return 0;
}

// Some dimensions, which are used by latitude or longitude, are not allowed to have the other cvs except latitude or longitude.
int check_cv(group *g) 
{
    std::set<var*> vars = g->get_all_cv();
    std::set<var*>::iterator it;

    std::set<dim*> dims; //store dims used by latitude or longitude
    std::string varname;
    size_t found = -1;

    /**
     * Find dims used by latitude or longitude.
     */
    for(it=vars.begin(); it!=vars.end(); it++)
    {
        var *v = *it;

        varname= v->get_name();
        found = varname.find(lat_final);
        found = (found==std::string::npos)? varname.find(lon_final): found;
        if (found!=std::string::npos) //dim used by latitude or longitude
        {
            std::list<dim*>& tmp = v->get_dims();
            std::list<dim*>::iterator id;
            for(id=tmp.begin(); id!=tmp.end(); id++)
                dims.insert(*id);				
        } // end of if
    } // end of for(it=	
	
    /**
     * If other cvs, excluding latitude or longitude cvs,
     * use the same dims used by latitude or longitude cvs,
     * remove these cvs from the dims.
     */
    std::set<dim*>::iterator dit;
    for(dit=dims.begin(); dit!=dims.end(); dit++)
    {
        std::list<var*>& cv = (*dit)->get_cv();
        std::list<var*>::iterator cvit;
        std::list<var*> tmp;
        for(cvit=cv.begin(); cvit!=cv.end(); cvit++) //all cvs under dim
        {
            varname= (*cvit)->get_name();
//std::cerr<<"this dim "<<(*dit)->get_name() << " has cvar with the name "<<varname <<std::endl;
            found = varname.find("latitude");
            found = (found==std::string::npos)? varname.find("longitude"): found;
            if (found==std::string::npos) //this cv's name does not contain latitude or longitude
            {
                tmp.push_back(*cvit); 
            }	
        } // end of for
        for(cvit=tmp.begin(); cvit!=tmp.end(); cvit++)
        {
            (*dit)->remove_cv(*cvit);
        }
    } // end of for(dit=dims.begin(	 	
    return 0;
}

//This check is for MOP02-20000303-L2V5.7.1.val.hdf only. Both latitude and longitude are 1D variable and have the same dimension name.
bool check_LatLon1D_Same_Dim(group* g)
{
    bool LatLon1D_Same_Dim = false;
    std::string lat_dim_name="";
    std::string lon_dim_name="";

    std::string lat_name_substr = "latitude";
    std::string lon_name_substr = "longitude";

    std::set<var*> vars = g->get_all_cv();
    std::set<var*>::iterator it;

    for(it=vars.begin(); it!=vars.end(); it++)
    {
        std::string var_name = (*it)->get_name();

		// need TO FIND if it is true for each group
        if(var_name.size() >=lat_name_substr.size()){
            
            if(0 == var_name.compare(var_name.size()-lat_name_substr.size(),lat_name_substr.size(),lat_name_substr))
            {
                std::list<dim*> dims = (*it)->get_dims();
                std::list<dim*>::iterator dit;
                if(dims.size()==1)
                {
                    dit = dims.begin();
                    lat_dim_name = (*dit)->get_name();
					
                }
            } // end of if
        } // end of if

        if(var_name.size() >=lon_name_substr.size()){
            if(0 == var_name.compare(var_name.size()-lon_name_substr.size(),lon_name_substr.size(),lon_name_substr))
            {
                std::list<dim*> dims = (*it)->get_dims();
                std::list<dim*>::iterator dit;
                if(dims.size()==1)
                {
                    dit = dims.begin();
                    lon_dim_name = (*dit)->get_name();	
                }
            } // end of if		
        }// end of if
    } // end of for(it=vars.begin()
	

    if(lat_dim_name.length()!=0 && lon_dim_name.length()!=0)
        if(lat_dim_name==lon_dim_name)// Key to figure out if lat/lon shares the same dimension.
            LatLon1D_Same_Dim = true;

    return LatLon1D_Same_Dim;
}

void check_LatLon1D_Same_Dim_multi_grps(group* g, std::map<std::string,bool> & grp_path_to_ll1d_sdim )
{
    //bool LatLon1D_Same_Dim = false;
    std::string lat_dim_name="";
    std::string lon_dim_name="";

    std::string lat_name_substr = "latitude";
    std::string lon_name_substr = "longitude";

    std::set<var*> vars = g->get_all_cv();
    std::set<var*>::iterator it;
	std::string var_path1="";
	std::string var_path2="";
	std::map<std::string,std::string > varpath_to_latdim;
	std::map<std::string,std::string > varpath_to_londim;
    for(it=vars.begin(); it!=vars.end(); it++)
    {
        std::string var_name = (*it)->get_name();

		// need TO FIND if it is true for each group
        if(var_name.size() >=lat_name_substr.size()){
            
            if(0 == var_name.compare(var_name.size()-lat_name_substr.size(),lat_name_substr.size(),lat_name_substr))
            {
                std::list<dim*> dims = (*it)->get_dims();
                std::list<dim*>::iterator dit;
                if(dims.size()==1)
                {
                    dit = dims.begin();
                    lat_dim_name = (*dit)->get_name();
					var_path1 = var_name.substr(0,var_name.size()-lat_name_substr.size());
					varpath_to_latdim[var_path1] = lat_dim_name;
                }
            } // end of if
        } // end of if

        if(var_name.size() >=lon_name_substr.size()){
            if(0 == var_name.compare(var_name.size()-lon_name_substr.size(),lon_name_substr.size(),lon_name_substr))
            {
                std::list<dim*> dims = (*it)->get_dims();
                std::list<dim*>::iterator dit;
                if(dims.size()==1)
                {
                    dit = dims.begin();
                    lon_dim_name = (*dit)->get_name();	
					var_path2 = var_name.substr(0,var_name.size()-lon_name_substr.size());
					varpath_to_londim[var_path2] = lon_dim_name;
                }
            } // end of if		
        }// end of if
    } // end of for(it=vars.begin()
	

for(std::map<std::string,std::string>::const_iterator i=varpath_to_latdim.begin(); i!=varpath_to_latdim.end();++i) {

	for(std::map<std::string,std::string>::const_iterator j=varpath_to_londim.begin(); j!=varpath_to_londim.end();++j) {

		if((*i).first == (*j).first) {
			// Same dimension name under the same group
			if((*i).second == (*j).second) {
				grp_path_to_ll1d_sdim[(*i).first] = true;
			}
			else 
			    grp_path_to_ll1d_sdim[(*i).first] = false;
		}
	}
}

#if 0
std::cerr<<"lat_dim_name is "<<lat_dim_name <<std::endl;
std::cerr<<"long_dim_name is "<<lon_dim_name <<std::endl;
    if(lat_dim_name.length()!=0 && lon_dim_name.length()!=0)
        if(lat_dim_name==lon_dim_name)// Key to figure out if lat/lon shares the same dimension.
            LatLon1D_Same_Dim = true;

    return LatLon1D_Same_Dim;

#endif
}
// Add "coordinates" attributes
int coordinates_attr(group* g, void *)
{

    /**
     * Some cvs are not real cv. check_cv will remove fake cvs 
     * from dims.
     */
    check_cv(g);
	
//std::cerr<<"Number of group is "<<g->get_child_groups().size() <<std::endl;

    bool multi_grp_status = g->get_num_of_orig_group_status();

// leave for debugging
#if 0
if(g->get_num_of_orig_group_status()) 
   std::cerr<<"original multi-groups"<<std::endl;
else 
   std::cerr<<"Original single-groups" <<std::endl;
#endif


    std::string lat_name_substr ="latitude";
    std::string lon_name_substr ="longitude";

	std::map<std::string,bool> grp_path_to_ll1d_sdim;
	bool LatLon1D_Same_Dim_SG = false;

    // bool LatLon1D_Same_Dim = check_LatLon1D_Same_Dim(g, grp_path_to_ll1d_sdim);
	if(true == multi_grp_status) 
	    check_LatLon1D_Same_Dim_multi_grps(g,grp_path_to_ll1d_sdim);
	else
	    LatLon1D_Same_Dim_SG = check_LatLon1D_Same_Dim(g);

//if(true == LatLon1D_Same_Dim)
//std::cerr<<"LatLon1D_Same_Dim is true"<<std::endl;

    std::list<var*>& vars = g->get_vars();
    std::list<var*>::iterator it;
    for(it = vars.begin(); it != vars.end(); it++)
    {
        var *v = *it;
        std::list<var*> cv;
        std::list<dim*>& dims = v->get_dims();
        std::list<dim*>::iterator id;

        bool has_coordinates = true; //store if var has coordinate attr 
        for(id = dims.begin(); id != dims.end(); id++)
        {
            const std::list<var*>& dcv = (*id)->get_cv_c();

            std::list<var*>::const_iterator icv;
            for(icv = dcv.begin(); icv != dcv.end(); icv++)
            {
                /**
                 * If a dim exists in cv, but not in var, this
                 * cv shouldn't be part of coordinates of var. 
                 */
                const std::list<dim*>& dcvdims = (*icv)->get_dims_c();
                std::list<dim*>::const_iterator dcvit;
                for(dcvit=dcvdims.begin(); dcvit!=dcvdims.end(); dcvit++)
                {
                    if(!v->has_dim_of(*dcvit))
                        break;
                }
                if(dcvit!=dcvdims.end()) 
                {
                    has_coordinates = false;
                    break;
                } 
				
                // If this coordinate variable(*icv) is not in the cv list, 
                // we will generally add this coordinate variable.
                if(find(cv.begin(), cv.end(), *icv) == cv.end())
                {
                    bool LatLon1D_Same_Dim = false;
                    // We provide a short-cut case when there is only one group in the whole file.
                    if(true == multi_grp_status) {
                        for(std::map<std::string,bool>::const_iterator i=grp_path_to_ll1d_sdim.begin(); i!=grp_path_to_ll1d_sdim.end();++i) {
		            if(((*it)->get_name()).find((*i).first)== 0) {
                                LatLon1D_Same_Dim = (*i).second;
                                break;
                            }
                        }
                    }
                    else{
//std::cerr<<"coming to the single-group case"<<std::endl;
                        LatLon1D_Same_Dim = LatLon1D_Same_Dim_SG;
                    }

                    if(false == LatLon1D_Same_Dim)
                    {
                        cv.push_back(*icv);
                    } else {

                        std::string var_name = (*it)->get_name();
                        std::string cvar_name = (*icv)->get_name();
                        bool var_name_is_longitude =
                             ((var_name.size()>= lon_name_substr.size()) &&
                             (0==var_name.compare(var_name.size()-lon_name_substr.size(),
                                                 lon_name_substr.size(),lon_name_substr)))
                             ?true:false;

                        bool cvar_name_is_longitude =
                             ((cvar_name.size()>= lon_name_substr.size()) &&
                             (0==cvar_name.compare(cvar_name.size()-lon_name_substr.size(),
                                                 lon_name_substr.size(),lon_name_substr)))
                             ?true:false;
                        bool cvar_name_is_latitude =
                             ((cvar_name.size()>= lat_name_substr.size()) &&
                             (0==cvar_name.compare(cvar_name.size()-lat_name_substr.size(),
                                                 lat_name_substr.size(),lat_name_substr)))
                             ?true:false;

                        // Check if this variable name and the coordinate variable name are group_name_longitude . 
                        // If they are not, we will add the coordinate variable to the final coordinate variable list. 
                        // This will effectively remove the group_name_longitude from the coordinate variable list. KY 2013-02-06
                        if (false == var_name_is_longitude) {
                            if (false == cvar_name_is_longitude)
                                cv.push_back(*icv);
                        }
                        // The variable is group_name_longitude, we will remove group_name_latitude from the coordinate variable list.
                        else { 
                            if (false == cvar_name_is_latitude)
                            cv.push_back(*icv); 
                        }
                      
                    } // end of if(!LatLon1D_Same_Dim
                } // end of if(find(cv.begin(), c
            } // end of for(icv = dcv.begi
        } //end of for(id = dims.begin();

        /**
         * If has_coordinates is false, skip the following.
         */
        if(!has_coordinates)
            continue;

        // Corrdinate varaibles of variable v is cv
        std::ostringstream oss;
        std::list<var*>::iterator iv;
        for(iv = cv.begin(); iv != cv.end(); iv++)
            oss << (*iv)->get_name() << " ";
        std::string s = oss.str();
//std::cerr<<"coordinate names are "<<s<<std::endl;
        if(s.length() > 0) //Should not print out coordinates="".
        {
            s.resize(s.length()-1);	// trim the last space
	
            mem_attr *attr = new mem_attr(v, "coordinates");
            attr->set_value(s);

            v->add_attr(attr);
        } // end of if(s.length()
    } // end of for(it = vars.begin();

    return 0;
}

// Add "units" and "standard_name" attributes for coordinate variables.
int add_units_attr(group* g, void *arg)
{
    //APPLY_RECURSIVELY(add_units_attr, g, arg);
    apply_recursively(add_units_attr, g, arg);

    std::list<var*>& vars = g->get_vars();
    std::list<var*>::iterator it;
    for(it = vars.begin(); it != vars.end(); it++)
    {
        var *v = *it;

        if(!v->is_cv(g))
            continue;
			
        bool has_units_attr = false;
        attr *units_attr = v->get_attr_by_name("units");

        if(units_attr != NULL)
            has_units_attr = true;

        attr *standard_name_attr = v->get_attr_by_name("standard_name");
        if(standard_name_attr)
            v->remove_attr(standard_name_attr);

        const std::string& name = v->get_name();

        mem_attr *attr;

        //"_reduced" is for the swath dimension map case. KY 2012-12-28
        // We decide to use "level" for the units of non-ll CVs only when
        // no "units" attribute exists for those non-ll CVs. KY 2013-02-14
        if(name == lat_final || name == std::string(lat_final) + "_reduced")
        {
            if(has_units_attr == true) 
                v->remove_attr(units_attr);
            attr = new mem_attr(v,"units");
            attr->set_value("degrees_north");
            v->add_attr(attr);

            mem_attr *myattr = new mem_attr(v, "standard_name");
            myattr->set_value("Latitude");
            v->add_attr(myattr);
        }
        else if(name == lon_final || name == std::string(lon_final) + "_reduced")
        {
            if(has_units_attr == true)
                v->remove_attr(units_attr);
            attr = new mem_attr(v,"units");
            attr->set_value("degrees_east");
            v->add_attr(attr);

            mem_attr *myattr = new mem_attr(v, "standard_name");
            myattr->set_value("Longitude");
            v->add_attr(myattr);
        }
        else {
            if (false == has_units_attr) {
                 mem_attr *temp_attr = new mem_attr(v, "units");
                 temp_attr->set_value("level");
                 v->add_attr(temp_attr);
            }
        }

    } // end of for(it = vars.begin();
   
    return 0;
}

/**
 * If the current file uses a location grid,
 * all variables under the location grid are copied to
 * each normal grid.
 * Note that we need to check checking dimension compatibility.
 * Note this is especially for AIRS version 4-5 grid case 
 *      when lat/lon are located under a specified grid.
 */
//static char *LOCGRID = "location";
const std::string LOCGRID = "location";
int handle_locgrid(group *g, void *arg)
{
    group *loc_grid = g->get_child_by_name(LOCGRID);
    if(loc_grid)
    {
        // If E2C_lat/lon are in loc_grid, rename them
        latlon_rename(loc_grid, NULL);

        // Need to get rid of all calculated lat/lon pairs 
        std::list<var*> removed_vars;

        std::list<group*>& groups = g->get_child_groups();
        for(std::list<group*>::iterator ig = groups.begin();
            ig != groups.end(); ig++)
        {
            std::list<var*>& vars = (*ig)->get_vars();
            std::list<var*>::iterator it;
            for(it = vars.begin();it!=vars.end();it++) {
                attr *a =(*it)->get_attr_by_name("eoslib");
                if(a != NULL) {
                    std::string E2C_ll_attr_value;
                    a->get_value_str(&E2C_ll_attr_value);
                    if("Calculated longitude" == E2C_ll_attr_value){ 
                        removed_vars.push_back(*it);
                        (*ig)->remove_var_force(*it);
                        break;
                    }
                }
            }
        
            for(it = vars.begin();it!=vars.end();it++) {
                attr *a =(*it)->get_attr_by_name("eoslib");
                if(a != NULL) {
                    std::string E2C_ll_attr_value;
                    a->get_value_str(&E2C_ll_attr_value);
                    if("Calculated latitude" == E2C_ll_attr_value) {
                        removed_vars.push_back(*it);
                        (*ig)->remove_var_force(*it);
                        break;
                    }
                }
            }
        }

       //Release removed resources allocated to the removed vars. HFCFLIB-113
       while(false == removed_vars.empty()) {
            var*v = *(removed_vars.begin());
            delete v;
            removed_vars.pop_front();
       }

       // Copy variables in the loc grid to other groups
       std::list<var*>& locvars = loc_grid->get_vars();
       for(std::list<group*>::iterator ig = groups.begin(); 
            ig != groups.end(); ig++)
       {
            if(*ig == loc_grid)
                continue;

            for(std::list<var*>::iterator ilv = locvars.begin();
                ilv != locvars.end(); ilv++)
            {

                //var *copied_var = NULL;
                //copied_var = (*ig)->copy(loc_grid, *ilv);
                (*ig)->copy(loc_grid, *ilv);

            } // end of for(std::list<var*>::iterator
        } // end of for(std::list<group*>::iterator

    } // end of if(loc_grid)
    return 0;
}
/* Helper function to handle MODIS valid_min and valid_max(valid_range)
 ** This function calculates MODIS valid min and valid max
 *
 *\param vmin_ptr        pointer to the final valid_min value
 *\param vmax_ptr        pointer to the final valid_max value
 *\param orig_vmin       original valid_min value
 *\param orig_vmax       original valid_max value
 *\param sotype          MODIS scale offset type(DIV, MUL or EQ)
 *\param sc              scale_factor(should be >0)
 *\param off             add_offset
 *\param has_fv          if having fillvalues
 *\param fvalue          fill value
 *
 */
bool cal_modis_valid_mm(float32*vmin_ptr, float32* vmax_ptr,float32 orig_vmin, float32 orig_vmax, MODISType sotype,double sc, double off,bool has_fv,float32 fvalue) {

    // If sc is less than 0, we will return false. No MODIS valid min or valid max is calculated.
    if (sc <= 0) 
        return false;
    
    float32 temp_vmin = 0.;
    float32 temp_vmax = 0.;
       
    if(MULTI_TYPE == sotype) {

        temp_vmin = sc*(orig_vmin-off);
        temp_vmax = sc*(orig_vmax-off);

    }
    else if(DIV_TYPE == sotype){

        temp_vmin = (orig_vmin-off)/sc;
        temp_vmax = (orig_vmax-off)/sc;

    }
    else  {// equal case.

        temp_vmin = sc*orig_vmin+off;
        temp_vmax = sc*orig_vmax+off;

    }

    // fvalue will be outside the (valid_min, valid_max),no new valid min and max are provided.
    // FillValue is honored.
    if (true == has_fv && (temp_vmin <fvalue && temp_vmax >fvalue)) 
        return false;
    else {
        *vmin_ptr = temp_vmin;
        *vmax_ptr = temp_vmax;
    }
 
    return true;
}
/* Helper function to handle MODIS valid_min and valid_max(valid_range)
** This function calculates the maximum and minimum values of a float32 array 
 *
 *\param float_ptr       pointer to the float32 array
 *\param max_ptr         pointer to the maximum value
 *\param min_ptr         pointer to the minimum value
 *
 */
void get_attr_float_array_max_CFLIB_MIN(attr* float_attr, float32* max_ptr, float32*min_ptr) {

    int num_elems = float_attr->get_num_elements();
    std::vector<char>fabuf;
    float_attr->get_value(&fabuf);
    float temp_attr_val = 0.;
    for (int i = 0; i <num_elems; i++) {
        temp_attr_val = *(float32*)(&fabuf[i*sizeof(float32)]);
        if (temp_attr_val < (*min_ptr))
            *min_ptr = temp_attr_val;
        if (temp_attr_val > (*max_ptr))
            *max_ptr = temp_attr_val;
    }

}
/* Helper function to handle MODIS valid_min and valid_max(valid_range)    
** This function obtains MODIS valid min and valid max from valid_range
 *
 *\param vmin_ptr        pointer to the final valid_min value
 *\param vmax_ptr        pointer to the final valid_max value
 *\param sotype          MODIS scale offset type(DIV, MUL or EQ)
 *\param vr_attr         valid_range attribute
 *\param fv_attr         fillvalue attribute
 *\param sc              scale_factor(should be >0)
 *\param off             add_offset
 *
 * This function assumes that only values of valid_range need to be changed, no sanity check for all input parameters.
 */

int obtain_modis_general_product_valid_mm_value( float32* vmin_ptr,float32* vmax_ptr,MODISType sotype,attr *vr_attr,attr*fv_attr,double sc, double off) {

    bool has_fv = true;
    if (NULL == fv_attr)
        has_fv = false;
    if (NULL == vr_attr) 
        return -1;

    // No need to even check the valid_range if no scale and offset. 
    if (sotype == MOD_DEFAULT_TYPE)
        return 0;

    float32 fvalue = 0.;
    std::vector<char> vbuf;
    vr_attr->get_value(&vbuf); 

    std::vector<char> fbuf;
    if (true == has_fv)
        fv_attr->get_value(&fbuf);

    // Checking the attribute count. It must be 2 for valid_range. 
    if (vr_attr->get_num_elements() != 2) 
        return 0;

    bool has_new_valid_range = false;
    
    switch(vr_attr->get_type())
    {
        case DFNT_UCHAR8:
        {
            uchar8 v1 = *(uchar8*)&vbuf[0]; 
            float32 orig_valid_min = (float)v1; 
            uchar8 v2 = *(uchar8*)&vbuf[1];
            float32 orig_valid_max = (float)v2;

            if( true == has_fv) {
               uchar8 f1 = *(uchar8*)&fbuf[0];
               fvalue = (float)f1;
            }

            has_new_valid_range = cal_modis_valid_mm(vmin_ptr,vmax_ptr,orig_valid_min,orig_valid_max,sotype,sc,off,has_fv,fvalue);

        }
            break;

        case DFNT_INT8:
        {
            int8 v1 = *(int8*)&vbuf[0]; 
            float32 orig_valid_min = (float)v1; 
            int8 v2 = *(int8*)&vbuf[1];
            float32 orig_valid_max = (float)v2;

            if( true == has_fv) {
               int8 f1 = *(int8*)&fbuf[0];
               fvalue = (float)f1;
            }

            has_new_valid_range = cal_modis_valid_mm(vmin_ptr,vmax_ptr,orig_valid_min,orig_valid_max,sotype,sc,off,has_fv,fvalue);

        }
            break;

        case DFNT_UINT8:
        {
            uint8 v1 = *(uint8*)&vbuf[0];
            float32 orig_valid_min = (float)v1; 
            uint8 v2 = *(uint8*)&vbuf[1];
            float32 orig_valid_max = (float)v2;

            if( true == has_fv) {
               uint8 f1 = *(uint8*)&fbuf[0];
               fvalue = (float)f1;
            }

            has_new_valid_range = cal_modis_valid_mm(vmin_ptr,vmax_ptr,orig_valid_min,orig_valid_max,sotype,sc,off,has_fv,fvalue);

        }
            break;

        case DFNT_INT16:
        {
            int16 v1 = *(int16*)&vbuf[0];
            float32 orig_valid_min = (float)v1; 
            int16 v2 = *(int16*)&(vbuf[sizeof(int16)]);
            float32 orig_valid_max = (float)v2;

            if( true == has_fv) {
               int16 f1 = *(int16*)&fbuf[0];
               fvalue = (float)f1;
            }

            has_new_valid_range = cal_modis_valid_mm(vmin_ptr,vmax_ptr,orig_valid_min,orig_valid_max,sotype,sc,off,has_fv,fvalue);

        }
            break;

        case DFNT_UINT16:
        {
            uint16 v1 = *(uint16*)&vbuf[0];
            float32 orig_valid_min = (float)v1; 
            uint16 v2 = *(uint16*)&vbuf[sizeof(uint16)];
            float32 orig_valid_max = (float)v2;

            if( true == has_fv) {
               uint16 f1 = *(uint16*)&fbuf[0];
               fvalue = (float)f1;
            }

            has_new_valid_range = cal_modis_valid_mm(vmin_ptr,vmax_ptr,orig_valid_min,orig_valid_max,sotype,sc,off,has_fv,fvalue);

        }
            break;

        case DFNT_UINT32:
        {
            uint32 v1 = *(uint32*)&vbuf[0];
            float32 orig_valid_min = (float)v1; 
            uint32 v2 = *(uint32*)&vbuf[sizeof(uint32)];
            float32 orig_valid_max = (float)v2;

            if( true == has_fv) {
               uint32 f1 = *(uint32*)&fbuf[0];
               fvalue = (float)f1;
            }

            has_new_valid_range = cal_modis_valid_mm(vmin_ptr,vmax_ptr,orig_valid_min,orig_valid_max,sotype,sc,off,has_fv,fvalue);

        }
            break;

        case DFNT_INT32:
        {
            int32 v1 = *(int32*)&vbuf[0];
            float32 orig_valid_min = (float)v1; 
            int32 v2 = *(int32*)&vbuf[sizeof(int32)];
            float32 orig_valid_max = (float)v2;

            if( true == has_fv) {
               int32 f1 = *(int32*)&fbuf[0];
               fvalue = (float)f1;
            }

            has_new_valid_range = cal_modis_valid_mm(vmin_ptr,vmax_ptr,orig_valid_min,orig_valid_max,sotype,sc,off,has_fv,fvalue);

        }
            break;
        
        case DFNT_FLOAT32:
        {
            float32 orig_valid_min = *(float32*)&vbuf[0]; 
            float32 orig_valid_max = *(float32*)&vbuf[sizeof(float32)];

            if( true == has_fv) {
               fvalue = *(float32*)&fbuf[0];
            }

            has_new_valid_range = cal_modis_valid_mm(vmin_ptr,vmax_ptr,orig_valid_min,orig_valid_max,sotype,sc,off,has_fv,fvalue);

        }
            break;
       
        case DFNT_FLOAT64:
        {
            // In theory, this may lose the precision. However, this case may never occur for MODIS products.
            // If indeed occurs, the precision losing may be neglected for MODIS products. 
            float64 v1 = *(float64*)&(vbuf[0]); 
            float32 orig_valid_min = (float)v1; 
            float64 v2 = *(float64*)&(vbuf[sizeof(float64)]);
            float32 orig_valid_max = (float)v2;

            if( true == has_fv) {
               float64 f1 = *(float64*)&fbuf[0];
               fvalue = (float)f1;
            }

            has_new_valid_range = cal_modis_valid_mm(vmin_ptr,vmax_ptr,orig_valid_min,orig_valid_max,sotype,sc,off,has_fv,fvalue);
        }
            break;
    }//end of switch

    if (true == has_new_valid_range)
        return 1;
    else
        return 0; 
    
}

/* Helper function to handle MODIS valid_min and valid_max(valid_range)
 ** This function obtains general MODIS valid min and valid max and stores them to CF attributes
 *
 *\param var             variable
 *\param sotype          MODIS scale offset type(DIV, MUL or EQ)
 *\param sc              scale_factor(should be >0)
 *\param off             add_offset
 *
 */

int handle_modis_general_product_valid_range(var*var,MODISType sotype,double sc, double off) {

    if (NULL == var) 
        return -1;

    // std::cerr<<"coming into the MODIS general product valid range function" <<std::endl;
    // We encounter a special case that MOD09GA/GHK products have two grids. MODIS needs to apply the EQ rule.
    // So if we find that the origname variable starts with /MODIS_Grid_1km_2D, we will change the rule to the EQU rule.
    attr *attr_origname = var->get_attr_by_name("origname");
    if(attr_origname != NULL) {
        std::string attr_origname_value;
        attr_origname->get_value_str(&attr_origname_value);
        if (0 == attr_origname_value.find("/MODIS_Grid_1km_2D") && DIV_TYPE == sotype)
            sotype = EQU_TYPE;
    }


    if ( sotype == MOD_DEFAULT_TYPE)// No need to do anything.
        return 0; 

    attr* vr_attr = var->get_attr_by_name("valid_range");
    if (vr_attr != NULL) {

        bool change_valid_range = true;
        attr* fv_attr = var->get_attr_by_name("_FillValue");
        if(fv_attr != NULL) {

            // To avoid that the recalculated valid_range covers fill values or maybe other
            // special values, we may need to consider the case when off is 0 in the future.
            // Now we only consider when the fillvalue type is the same as the valid_range type.
            if(fv_attr->get_type() != vr_attr->get_type())
                change_valid_range = false;
        }

        if (true == change_valid_range) {

            float32 vmin = 0;
            float32 vmax = 0;
            int valid_mm_flag = obtain_modis_general_product_valid_mm_value(&vmin,&vmax,sotype,vr_attr,fv_attr,sc,off);

            // mostly valid_mm_flag will never return -1.  Here just have a sanitary check. 
            // "return -1" means something is wrong.
            if (-1 == valid_mm_flag)
                return -1;
            else if (1 == valid_mm_flag){

                mem_attr *vmin_attr = new mem_attr(var,"valid_min");
                vmin_attr->set_value(DFNT_FLOAT32,1,(void*)&vmin);
                var->add_attr(vmin_attr);

                mem_attr *vmax_attr = new mem_attr(var,"valid_max");
                vmax_attr->set_value(DFNT_FLOAT32,1,(void*)&vmax);
                var->add_attr(vmax_attr);
            }            
            // Notice if valid_mm_flag return 0, it means that no new valid_range should be provided and the old valid_range should be removed.
            // This approach may be changed in the future like scale_modis etc.
        }

        var->remove_attr(vr_attr);

    }

    return 0;
}
/* Helper function to handle MODIS valid_min and valid_max(valid_range)
 ** This function calculates either MODIS level1B reflectance or radiance valid min and valid max
 *
 *\param fieldtype          special MODIS field type just to distinguish reflectance from radiance
 *\param var                variable(pointer)
 *
 */
void handle_emissive_reflectance_attrs(int fieldtype, var* var){

    std::string scale_name;
    std::string offset_name;
    std::string unit_name;

    float valid_max = 0;
    float valid_min = 0;


    float scale_max = 0;
    float scale_min = 100000;

    float offset_max = 0;
    float offset_min = 0;


    if (1 == fieldtype) {
        scale_name = "radiance_scales";
        offset_name = "radiance_offsets";
        unit_name = "radiance_units";
    }
    else if ( 2 == fieldtype) {
        scale_name = "reflectance_scales";
        offset_name = "reflectance_offsets";
        unit_name = "reflectance_units";
    }

    attr* sc_attr = var->get_attr_by_name(scale_name);
    if (sc_attr != NULL) {
        if (sc_attr->get_type() != DFNT_FLOAT32) 
            throw std::range_error("Current we only support DFNT_FLOAT32 type for radiance_scales or reflectance_scales.(" __FILE__ ":" TOSTRING(__LINE__)")" );
        //Obtain the maximum and minimum scale_factor from the scale_factor array.
        get_attr_float_array_max_CFLIB_MIN(sc_attr,&scale_max,&scale_min);
    }

    attr* os_attr = var->get_attr_by_name(offset_name);
    if (os_attr != NULL) {
        if (os_attr->get_type() != DFNT_FLOAT32) 
            throw std::range_error("Current we only support DFNT_FLOAT32 type for radiance_scales.(" __FILE__ ":" TOSTRING(__LINE__)")" );
        get_attr_float_array_max_CFLIB_MIN(os_attr,&offset_max,&offset_min);
    }

    attr* ln_attr= var->get_attr_by_name("long_name");

    // Always assume long_name is a string for now
    if (ln_attr != NULL) {

        std::string modify_long_name_value;
        std::string str_removed_from_long_name=" Scaled Integers";

        std::vector<char>lnbuf;
        ln_attr->get_value(&lnbuf);
        std::string orig_long_name_value(lnbuf.begin(),lnbuf.end());

        // std::copy(lnbuf.begin(),lnbuf.end(),orig_long_name_value.begin());
        // std::cerr<<"orig_long_name_value "<<orig_long_name_value <<std::endl;
        if (orig_long_name_value.find(str_removed_from_long_name)!=std::string::npos) {
            if(0 == orig_long_name_value.compare(orig_long_name_value.size()-str_removed_from_long_name.size(), str_removed_from_long_name.size(),str_removed_from_long_name)) {
                modify_long_name_value =
                    orig_long_name_value.substr(0,orig_long_name_value.size()-str_removed_from_long_name.size());
                var->remove_attr(ln_attr);
                mem_attr *nln_attr = new mem_attr(var,"long_name");
                nln_attr->set_value(modify_long_name_value);
                var->add_attr(nln_attr);
                mem_attr *oln_attr = new mem_attr(var,"orig_long_name");
                oln_attr->set_value(orig_long_name_value);
                var->add_attr(oln_attr);
            }

        }
    }

    attr* ru_attr = var->get_attr_by_name(unit_name);

    // Always assume units is a string for now
    if (ru_attr != NULL) {
        std::vector<char>rubuf;
        ru_attr->get_value(&rubuf);
        std::string radiance_units_value(rubuf.begin(),rubuf.end());
        attr* u_attr = var->get_attr_by_name("units");
        if (u_attr != NULL) 
            var->remove_attr(u_attr);
        mem_attr *new_u_attr = new mem_attr(var,"units");
        new_u_attr->set_value(radiance_units_value);
        var->add_attr(new_u_attr);
    }

    // Calculate the final valid_max and valid_min.
    if (scale_min <= 0)
        throw std::range_error("scale factor should always be greater than 0.(" __FILE__ ":" TOSTRING(__LINE__)")" );

    // We want to use obtain_modis_general_product_valid_mm_value to calculate the final valid_min and valid_max.
     // Note we CANNOT use obtain_modis_general_product_valid_range since we have four combinations of
    // the possible valid_min and valid_max[(s_ma,o_ma),(s_mi,o_ma),(s_ma,o_mi),(s_mi,o_mi)].
    // 1) We need to get the valid_range and fillvalue attributes.
    // 2) Then call obtain_modis_general_product_valid_mm_value four times 
    attr* vr_attr = var->get_attr_by_name("valid_range");
    if (vr_attr != NULL) {

        bool change_valid_range = true;
        attr* fv_attr = var->get_attr_by_name("_FillValue");
        if(fv_attr != NULL) {
            if(fv_attr->get_type() != vr_attr->get_type())
                change_valid_range = false;
        }

        if (true == change_valid_range){

            float temp_valid_r1 = 0.;
            float temp_valid_r2 = 0.;
            // Comb. 1
            int valid_range_ret = 
                obtain_modis_general_product_valid_mm_value(&temp_valid_r1,&temp_valid_r2,MULTI_TYPE,vr_attr,fv_attr,scale_max,offset_max);
            if (valid_range_ret > 0) {
                    valid_max = CFLIB_MAX(temp_valid_r1,temp_valid_r2);
                    valid_min = CFLIB_MIN(temp_valid_r1,temp_valid_r2);
            }
            else // The condition is not fit, no need to check further more.
               return; 

            // Comb. 2
            valid_range_ret = obtain_modis_general_product_valid_mm_value(&temp_valid_r1,&temp_valid_r2,MULTI_TYPE,vr_attr,fv_attr,scale_max,offset_min);
            float temp_value = CFLIB_MAX(temp_valid_r1,temp_valid_r2);
            if (valid_max < temp_value)
                valid_max = temp_value;
            temp_value = CFLIB_MIN(temp_valid_r1,temp_valid_r2);
            if (valid_min >temp_value)
                valid_min = temp_value;
            
            // Comb. 3
            valid_range_ret = obtain_modis_general_product_valid_mm_value(&temp_valid_r1,&temp_valid_r2,MULTI_TYPE,vr_attr,fv_attr,scale_min,offset_max);
            temp_value = CFLIB_MAX(temp_valid_r1,temp_valid_r2);
            if (valid_max < temp_value)
                valid_max = temp_value;
            temp_value = CFLIB_MIN(temp_valid_r1,temp_valid_r2);
            if (valid_min >temp_value)
                valid_min = temp_value;
                
            // Comb. 4
            valid_range_ret = obtain_modis_general_product_valid_mm_value(&temp_valid_r1,&temp_valid_r2,MULTI_TYPE,vr_attr,fv_attr,scale_min,offset_min);
            temp_value = CFLIB_MAX(temp_valid_r1,temp_valid_r2);
            if (valid_max < temp_value)
                valid_max = temp_value;
            temp_value = CFLIB_MIN(temp_valid_r1,temp_valid_r2);
            if (valid_min >temp_value)
                valid_min = temp_value;
        }
    }

    // These physical variables should be greater than 0
    if (valid_min < 0)
        valid_min = 0;

    if (valid_min !=0 || valid_max !=0) {
        mem_attr *new_vmi_attr = new mem_attr(var,"valid_min");
        new_vmi_attr->set_value(DFNT_FLOAT32,1,(void*)&valid_min);
        var->add_attr(new_vmi_attr);
    
        mem_attr *new_vma_attr = new mem_attr(var,"valid_max");
        new_vma_attr->set_value(DFNT_FLOAT32,1,(void*)&valid_max);
        var->add_attr(new_vma_attr);

        if (vr_attr != NULL)
            var->remove_attr(vr_attr);
    }

    return;   
   
}
 
/* Helper function to handle MODIS valid_min and valid_max(valid_range)
 ** This function obtains the final MODIS level1B reflectance and radiance valid min and valid max
 *
 *\param var                variable(pointer)
 *\param sotype             MODIS scale offset type(MUL or DIV)
 * Note: this function calls handle_emissive_reflectance_attrs.
 *
 */

int handle_modisl1b_rr_valid_range(std::list<var*>vars,MODISType sotype) {

    // Tackle the build_up of the final CF valid_min and valid_max for MODIS L1B radiance and reflectance data, KY 2012-08-28
    if (sotype == MULTI_TYPE) {

        for(std::list<var*>::iterator it=vars.begin(); it!=vars.end(); it++){

            // Emissive should be at the end of the field name such as "..._Emissive"
            std::string emissive_str = "Emissive";
            std::string RefSB_str = "RefSB";
            bool is_emissive_field = false;
            bool is_refsb_field = false;
            std::string varname = (*it)->get_name();

            if(varname.find(emissive_str)!=std::string::npos) {
                if(0 == varname.compare(varname.size()-emissive_str.size(),emissive_str.size(),emissive_str))
                    is_emissive_field = true;
            }

            if(varname.find(RefSB_str)!=std::string::npos) {
                if(0 == varname.compare(varname.size()-RefSB_str.size(),RefSB_str.size(),RefSB_str))
                    is_refsb_field = true;
            }

            if ((true == is_emissive_field) || (true== is_refsb_field)){

                // Radiance field(Emissive field)
                if(true == is_emissive_field) 
                    handle_emissive_reflectance_attrs(1,*it);

                if(true == is_refsb_field) 
                    handle_emissive_reflectance_attrs(2,*it);

            }
        }
    }

    return 0;
}

// Users can  use NCO to handle the VIP product. This file is a dummy function that should be removed in the future.
// KY 2013-05-31
int handle_modisvip_valid_range(var*var,MODISType sotype,double sc, double off) {

   return 0;

}


// Users can use NCO to handle other special cases. This file is a dummy function that should be removed in the future.
// KY 2013-05-31
int handle_modis_other_special_case(std::list<var*>vars,MODISType sotype) {

   return 0;

}

// For MODIS product only. "scale_factor" and "add_offset" attributes in variables should be equal 1.0 and 0.0 respectively, otherwise "scale_factor" and "add_offset" are renamed as "scale_factor_modis" and "add_offset_modis" correspondingly.
int change_modis_attrs(group *root, void *arg)
{
    // See the eos2_defs.h for detailed information about MODOS products. 
    // Add new information for some products. KY 2013-01-25
    // The reason 0.05Deg is changed to 0_05Deg is because the "." is treated as a special character
    // and the special character has been changed to '_'. 
    std::string multi_type[] = {"L1B", "GEO", "BRDF", "0_05Deg", "Reflectance", "MOD17A2", "North","South","MOD_Swath_Sea_Ice", "MOD_Grid_MOD15A2","MODIS_NACP_LAI"};
    std::string equ_type= "LST";
    std::string div_type[] = {"VI", "1km_2D", "L2g_2d","CMG","MODIS_SWATH_TYPE_L2","MODIS_Grid_500m_2D"};

    // Using global variables like mtype is not the right way. May need to change the way to handle this
    // in the future. KY 2013-01-25
    mtype = MOD_DEFAULT_TYPE; 

    std::list<group*>& groups = root->get_child_groups();
    for(std::list<group*>::iterator it=groups.begin(); it!=groups.end(); it++)
    {
        std::string name = (*it)->get_name();
		
        // Check product type.
        if(name=="mod05" || name=="mod06" || name=="mod07" || name=="mod08" || name=="atml2")
        {
            mtype = MULTI_TYPE;
            break;
        }
	
        if(name.find("MOD")==0 || name.find("mod")==0)
        {
            size_t pos = name.rfind(equ_type);
            if(pos!=std::string::npos && 
               (pos==name.length()-equ_type.length())) {
                mtype = EQU_TYPE;
                break;
            }

            for(int k=0; k<sizeof(multi_type)/sizeof(*multi_type)&&(MOD_DEFAULT_TYPE == mtype); k++)
            {
                pos = name.rfind(multi_type[k]);
                if(pos!=std::string::npos && 
                  (pos==name.length()-multi_type[k].length())){
                    mtype = MULTI_TYPE;
                    break;
                }
            } // end of for

            if (mtype != MOD_DEFAULT_TYPE)
                break;

            for(int k=0; k<sizeof(div_type)/sizeof(*div_type)&&(MOD_DEFAULT_TYPE == mtype); k++)
            {
                pos = name.rfind(div_type[k]);
                if(pos==name.length()-div_type[k].length())
                {
                    mtype = DIV_TYPE;
                    break;
                }
            } // end of for

            if (mtype != MOD_DEFAULT_TYPE)
                break;

            // MOD09GA/GHK products have two grids. MODIS_Grid_1km_2D needs to apply the EQ rule;
            // Other grids needs to apply the DIV rule. The MODIS_Grid_1km_2D has been handled by
            // change_modis_value() in special_flt_cf.cpp. KY 2013-02-11
        } // end of if(name.find("MOD")==0

        if("VIP_CMG_GRID"==name) {
            mtype = DIV_TYPE;
            break;
        }
    } // end of for(std::list<group*>
	
    if(mtype!=MOD_DEFAULT_TYPE) 
    {
        std::list<var*> vars = root->get_vars_r();
        for(std::list<var*>::iterator it=vars.begin(); it!=vars.end(); it++)
        {
            attr *attr1 = (*it)->get_attr_by_name("scale_factor");
            attr *attr2 = (*it)->get_attr_by_name("add_offset");
 
            if(attr1!=NULL) // && attr2!=NULL) //Change the recalculation rule.[04/30/2012 LD]
            {
                double scale_factor = 1;
                double add_offset=0;
                attr1->get_value(&scale_factor);
                if(attr2!=NULL)
                    attr2->get_value(&add_offset);
                if(!(scale_factor==1 && add_offset==0))
                {
                    attr *w1 = attr1->clone();
                    w1->rename("scale_factor_modis");
                    (*it)->add_attr(w1);
                    (*it)->remove_attr(attr1);
				
                    if(attr2!=NULL)
                    {	
                        attr *w2 = attr2->clone();
                        w2->rename("add_offset_modis");
                        (*it)->add_attr(w2);
                        (*it)->remove_attr(attr2);
                    } // end of if

                    // The VIP handling routine will handle all variables in the VIP product.
                    // If not VIP, handle general products.
                    if( 0 == handle_modisvip_valid_range(*it,mtype,scale_factor,add_offset))
                        handle_modis_general_product_valid_range(*it,mtype,scale_factor,add_offset);

                } // end of if(!(scale_factor==1
            } // end of if(attr!=NULL)			
        } // end of for(std::list<var*

        // Handle MODIS level 1B radiance and reflectance valid_range
        handle_modisl1b_rr_valid_range(vars,mtype);
        
        // Handle Other special cases(SCALE_FACTOR to Scale, OFFSET to offset)
        handle_modis_other_special_case(vars,mtype);
        
    } //end of if(mtype!=MOD_DEFAULT_TYPE) 

    return 0;
}

// For 1D latitude or longitude, if coordinate variable name is different from dimension name, dimension name is renamed the same as its coordinate variable name.  
int coards_conventions(group *root, void *arg)
{
    std::map<std::string, std::string> dimnames;

    // A string iset to store dimension names of multiple-dimension latitude and longitude variables
    std::set<std::string>md_latlon_dimnames;
	
    // Find all dimensions used by coordinate variables.
    std::set<var*> cvars = root->get_all_cv();
    for(std::set<var*>::iterator it=cvars.begin(); it!=cvars.end(); it++)
    {
        std::string varname = (*it)->get_name();
        std::list<dim*> dims = (*it)->get_dims();
//std::cerr<<"varname is "<<varname <<std::endl;
        bool is_md_latlon = false;
        
        if(varname.find("latitude")!=std::string::npos ||
           varname.find("longitude")!=std::string::npos)
        {
            if(dims.size()>1) {// Not a 1D case.
                is_md_latlon = true;
                // return 0;
            }
        }
        else if (dims.size() != 1) // All other coordinate variables should be 1-D.
            return -1;

        // Here the algorithm is not efficient since dimname is the same as the varname
        // for most cases. Since usually only 2-3 CVs are inside one file, the performance
        // won't degrade much. Leave it. KY 2013-01-10
        if (false == is_md_latlon) {
            std::string dimname = (*dims.begin())->get_name();
//std::cerr<<"dimname is "<<dimname <<std::endl;
            if(dimnames.find(dimname)==dimnames.end()) {
                dimnames.insert(std::pair<std::string, std::string>(dimname, varname));
//std::cerr<<"dimname,varname pair dimname is "<<dimname << ", varname is "<< varname << std::endl;
			}
        }
        else { // If the lat/lon is not a 1D array, add the dimension names of dims to md lat/lon dimnames set. 
            for(std::list<dim*>::const_iterator dit=dims.begin(); dit!=dims.end(); dit++)
                md_latlon_dimnames.insert((*dit)->get_name());
        }
    } // end of for(std::set<var*>

    //Remove the md lat/lon dimension names from the dimnames list if there is any.
    for (std::set<std::string>::iterator it = md_latlon_dimnames.begin(); it!=md_latlon_dimnames.end();it++) 
        dimnames.erase(*it);

    // Rename dimensin names for all groups.
    const std::list<dim*>& dims= root->get_dims_c();
    for(std::list<dim*>::const_iterator dit=dims.begin(); dit!=dims.end(); dit++)
    {
        std::string dimname = (*dit)->get_name();
        if(dimnames.find(dimname)!=dimnames.end()) // Rename.
            (*dit)->rename(dimnames.find(dimname)->second);
    } // end of for

    return 0;
}

// For internal testing only.
static int dump_group(group *g, void *arg)
{
    g->dump_r(0);
    return 0;
}

//#undef APPLY_RECURSIVELY

//FILTER_SET(cf);
filter_set g_filter_set_cf;

class add_filter 
{
    public:
        add_filter()
        {
            g_filter_set_cf.add("Adding missing objects", 25, add_missing_objects, NULL);
            g_filter_set_cf.add("Adding \"origname\" and \"long_name\" attributes", 30, add_origname, NULL);
            g_filter_set_cf.add("Handling location grid if any", 40, handle_locgrid, NULL);
            g_filter_set_cf.add("Analyzing lat/lon of each group", 50, latlon_rename, NULL);
            g_filter_set_cf.add("Renaming dimmap-reduced variables", 52, rename_dimmap_orig, NULL);
            g_filter_set_cf.add("Remove redundant vars (lat/lon)", 53, remove_redundant_vars, NULL);
            g_filter_set_cf.add("Remove priority attributes", 54, remove_priority_attrs, NULL);
            g_filter_set_cf.add("Changing special characters in names", 59, handle_special_chars, NULL);
            g_filter_set_cf.add("Condensing 2D lat/lon to 1D if possible", 60, condense_2d_latlon, NULL);
            g_filter_set_cf.add("Finding coordinate variables of each dimension", 70, find_cv_for_each_dim, NULL);
            g_filter_set_cf.add("Adding \"units\" and \"standard_name\" attributes to coordinate variables", 75, add_units_attr, NULL);
            g_filter_set_cf.add("Flattening", 100, flatten_modifier, NULL);
            g_filter_set_cf.add("Adding \"coordinates\" attribute", 200, coordinates_attr, NULL);
            g_filter_set_cf.add("Final name clashing check", 130, final_name_clashing_check, NULL);
            //g_filter_set_cf.add("Final name clashing check", 300, final_name_clashing_check, NULL);
            g_filter_set_cf.add("Changing attribute name of MODIS product", 95, change_modis_attrs, NULL);
            //g_filter_set_cf.add("Imposing COARDS conventions for 1D coordinate variables", 350, coards_conventions, NULL);
            g_filter_set_cf.add("Imposing COARDS conventions for 1D coordinate variables", 150, coards_conventions, NULL);
        } // end of add_filter()
};
add_filter af;

//ADD_FILTER("Adding missing objects", cf, 25, add_missing_objects);
//ADD_FILTER("Adding \"origname\" and \"long_name\" attributes", cf, 30, add_origname);
//ADD_FILTER("Handling location grid if any", cf,  40, handle_locgrid);
//ADD_FILTER("Analyzing lat/lon of each group", cf,  50, latlon_rename); 
//ADD_FILTER("Renaming dimmap-reduced variables", cf, 53, rename_dimmap_orig);
//ADD_FILTER("Remove redundant vars (lat/lon)", cf,  53, remove_redundant_vars);
//ADD_FILTER("Remove priority attributes", cf,  54, remove_priority_attrs);

//ADD_FILTER("Changing special characters in names", cf,  59, handle_special_chars);
//ADD_FILTER("Condensing 2D lat/lon to 1D if possible",cf,  60, condense_2d_latlon);
//ADD_FILTER("Finding coordinate variables of each dimension", cf,  70, find_cv_for_each_dim);
//ADD_FILTER("Adding \"units\" and \"standard_name\" attributes to coordinate variables", cf, 75, add_units_attr);
//ADD_FILTER("Flattening", cf, 100, flatten_modifier);
//ADD_FILTER("Adding \"coordinates\" attribute", cf, 200, coordinates_attr);
//ADD_FILTER("Final name clashing check", cf, 300, final_name_clashing_check);
//ADD_FILTER("Changing attribute name of MODIS product", cf, 95, change_modis_attrs);
//ADD_FILTER("Imposing COARDS conventions for 1D coordinate variables", cf, 350, coards_conventions);


// The following macros are for testing only.
/*ADD_FILTER("Before analyzing lat/lon", cf,  45, dump_group)*/
/*ADD_FILTER("Dump", cf, 26, dump_group);*/
/*ADD_FILTER("Before handling locgrid", cf,  35, dump_group);*/
/*ADD_FILTER("dump", cf,  51, dump_group);*/ 
/*ADD_FILTER("Afteranalyzing lat/lon", cf,  55, dump_group);*/
/*ADD_FILTER("Dump", cf,  65, dump_group);*/
/*ADD_FILTER("Dump", cf,  95, dump_group);*/
/*ADD_FILTER("After Flattening", cf,  105, dump_group);*/
/*ADD_FILTER("Renaming", cf, 120, rename_modifier);*/
/*ADD_FILTER("After Flattening", cf,  215, dump_group);*/
/*ADD_FILTER("Final dumping", cf,  998, dump_group);*/

} // namespace

