/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to high-level APIs.


*****************************************************************************/
#include <typeinfo>
#include "h4cf.h"

const string h4cf_get_var_name(var*);

/**
 * Starting point of HDFEOS2 library.
 */
group* root = NULL;

/**
 * Starting point of HDF4 library.
 */
hdf4_file *file = NULL;

/**
 * Starting point of NcML library
 */
ncml_file *ncmlfile = NULL;

/**
 * True if HDF4, false HDFEOS2 or hybrid.
 */
bool apply_h4_filter = false;

bool is4 = false;

void h4cf_open(char *filename)
{

    try {
 
        // Open NcML file. As for this release, this functionality has not yet been implemented. It will be provided in the future release. 
        ncmlfile = new ncml_file(filename);

        // Open file as an HDF4 file.
        file = new hdf4::hdf4_file();
        file->read(filename);
        bool special_eos_no_filter = false;
        //Uncomment this later,STOP KY 2014-12-02
        special_eos_no_filter = file->check_special_eos_no_filters(filename);
#if 0
if(true == special_eos_no_filter)
std::cerr<<"airs version 6" <<std::endl;
else
std::cerr<<"not airs version 6 "<<std::endl;
#endif

        if(true == special_eos_no_filter) {
            file->handle_special_eos_no_filters();
            is4 = true;
        }
        else {

            // Open file as a HDFEOS2 file and apply filtering related to HDFEOS2.
            root = open(const_cast<char*>(filename), CF);

            int len = 0;
            int num_grids = 0;
            int num_swaths = 0;
            num_grids = GDinqgrid(const_cast<char*>(filename), NULL, (int32*)&len);
            num_swaths = SWinqswath(const_cast<char*>(filename), NULL, (int32*)&len);

//std::cerr<<"num_grids is "<<num_grids <<std::endl;
//std::cerr<<"num_swaths is "<<num_swaths <<std::endl;
//
            // If both num_grids and num_swaths are equal to 0, then the openning file is categorized as HDF4.
            is4 = (num_grids==0 && num_swaths==0);
            apply_h4_filter = is4;


//if(is4 == true) 
//std:: cerr<<"this is an HDF4 file "<<std::endl;
    // Apply filtering related bo HDF4.
#if 0
    if(is4)
        file->apply();

#endif
        }
        file->apply(apply_h4_filter);
    }

    catch(runtime_error &e) {

        h4cf_close();
        cout<<e.what() <<endl;
        cout << "Error to call h4cf_open. "<<endl;
        cout << "The program exits abnormally. "<<endl;
        exit(1);
    }
    catch(...) {
        h4cf_close();
        cout<<"Unknown exception caught calling h4cf_open"<<endl;
        exit(1);

    }
    //else 
        //file->release_unused_memory();
}

// The following routine is not declared in the header file and is not throughly tested. 
// Comment it out for 1.0 release. KY 2014-12-17
#if 0
const map<string, string>  h4cf_get_file_attrs_txt()
{
    map<string, string> attrs;
    pair<map<string, string>::iterator,bool> ret;

    if(is4) // For HDF4.
    {
        for(vector<hdf4_attr *>::const_iterator git=file->get_sd()->get_attrs().begin(); git!=file->get_sd()->get_attrs().end(); git++)
        {
            string str = (*git)->get_str_value();
            int len = str.length();
            if(str[0]=='\"') // Remove the heading and trailing quote marks.
                str = str.substr(1, len-1);
            len = str.length();
            if(str[len-1]=='\"')
                str = str.substr(0, len-1);
            attrs.insert(pair<string, string>((*git)->get_name(), str));
        } // end of for

        for(vector<VDATA*>::const_iterator vit=file->get_vds().begin(); vit!=file->get_vds().end(); vit++)
        {
            for(vector<hdf4_attr *>::const_iterator ait = (*vit)->get_attributes().begin(); ait!=(*vit)->get_attributes().end();ait++)
            {

                string str = (*ait)->get_str_value();
                int len = str.length();
                if(str[0]=='\"') // Remove the heading and trailing quote marks.
                    str = str.substr(1, len-1);
                len = str.length();
                if(str[len-1]=='\"')
                    str = str.substr(0, len-1);
                attrs.insert(pair<string, string>((*ait)->get_name(), str));

            }
        }
         
    } else { // For HDFEOS2.
        const list<eoslib::attr*>& tmp = root->get_attrs_c();
        for(list<eoslib::attr*>::const_iterator ai = tmp.begin(); ai != tmp.end(); ai++)
        {
            stringstream oss;
            dump_attr(*ai, oss);

            std::string str = oss.str();
            int len = str.length();
            if(str[0]=='\"')
                str = str.substr(1, len-1);
            len = str.length();
            if(str[len-1]=='\"')
                str = str.substr(0, len-1);
            attrs.insert(pair<string, string>((*ai)->get_name(), str));
        } // end of for
    } // end of else
    return attrs;
}
#endif

const map<string, int> h4cf_get_dims()
{
    map<string, int> dims;
    if(is4) // For HDF4.
    {
        dims = file->get_dims();
    } else { // For HDFEOS2.
        list<dim*> tmp = root->get_dims_c();
        list<dim*>::iterator it;
        for(it=tmp.begin(); it!=tmp.end(); it++)
            dims.insert(pair<string, int>((*it)->get_name(), (*it)->get_size()));	
    } // end of else

    // Define new dimensions for VDATA.
    if(is4)
    {
        for(vector<VDATA*>::const_iterator vit=file->get_vds().begin(); vit!=file->get_vds().end(); vit++)
        {
            for(vector<hdf4_var_vdfield *>::const_iterator vfit=(*vit)->get_fields().begin(); vfit!=(*vit)->get_fields().end(); vfit++)
            {
                string dimname1 = "VDFDim0_" + (*vfit)->get_new_name();
                string dimname2 = "VDFDim1_" + (*vfit)->get_new_name();
                map<string, int>::const_iterator it = dims.find(dimname1);
                if(it==dims.end())
                    dims.insert(pair<string, int>(dimname1, (*vfit)->get_num_rec()));

                // This case is for when vdata field order is >1
                if ((*vfit)->get_order() >1) {
                    if(dims.find(dimname2)==dims.end()) 
                         dims.insert(pair<string, int>(dimname2,(*vfit)->get_order()));
                }
            } // end of for
        } // end of for
    } 

    return dims;
}

const string h4cf_get_dim_name(dim *d)
{
    return d->get_name();
}

const int h4cf_get_dim_size(dim *d)
{
    return d->get_size();
}

const list<var*> h4cf_get_vars()
{
    list<var*> vars =  (is4)? file->get_vars(): root->get_vars_c();
    return vars;
}

// The following routine is not declared in the header file and is not throughly tested.
// Comment it out for 1.0 release.
#if 0
const string h4cf_get_var_str_value(var *v)
{	
    string str = "";
    if(h4cf_get_var_name(v).find("vdf_")==0) // For VDATA.
        if(is4) { // For HDF4.
            str =  ((hdf4_var_vdfield*)v)->get_str_value();
        } else { // For HDFEOS2 
            stringstream oss;
            dump_var(v, oss);
            str = oss.str();		
        } // end of else
    return str;
}
#endif

const string h4cf_get_var_name(var *v)
{
    const char* vname = typeid(*v).name();
    string s(vname);
    if(s.find("hdf4_var_vdfield")!=string::npos) {// For VDATA of HDF4.
        return ((hdf4_var_vdfield*)(v))->get_new_name();
    }
    if(s.find("hdf4_var_sdfield")!=string::npos) {// For SDS of HDF4.
        return ((hdf4_var_sdfield*)(v))->get_new_name();		
    }

    return v->get_name(); // For HDFEOS2.
}

const vector< map<string, int> > h4cf_get_var_dims(var *v)
{
    vector< map<string, int> > vdims;

    if(is4) // For HDF4.
    {

        if((h4cf_get_var_name(v).find("vdata_") == 0) && (h4cf_get_var_name(v).find("vdf_")!=string::npos)) // For VDATA.
        {
            hdf4_var_vdfield *v4 = (hdf4_var_vdfield*)(v);
            string dimname1 = "VDFDim0_" + v4->get_new_name();
            string dimname2 = "VDFDim1_" + v4->get_new_name();

            map<string, int> tmp;
            tmp.insert(pair<string, int>(dimname1, v4->get_num_rec()));

            vdims.push_back(tmp);

            if (v4->get_order() >1) {
                map<string, int> tmp2;
                tmp2.insert(pair<string, int>(dimname2, v4->get_order()));
                vdims.push_back(tmp2);
            }
            return vdims;
        } // end of if

        hdf4_var_sdfield *v4 = (hdf4_var_sdfield*)(v);
        vector<hdf4_dim*> tmp = v4->get_corrected_dimensions();
        for(vector<hdf4_dim*>::iterator it =tmp.begin(); it!=tmp.end(); it++)
        {
            map<string, int> tmp;
            tmp.insert(pair<string, int>((*it)->get_name(), (*it)->get_size()));
            vdims.push_back(tmp);
        }
    } else { // For HDFEOS2.
        const list<dim*> tmp = v->get_dims_c();
        for(list<dim*>::const_iterator it=tmp.begin(); it!=tmp.end(); it++)
        {
            map<string, int> tmp;
            tmp.insert(pair<string, int>((*it)->get_name(), (*it)->get_size()));
            vdims.push_back(tmp);
        }
    } // end of else

    return vdims;
}

const list<attr*> h4cf_get_file_attrs() {

    list<attr*> fattrs;

    if(true == is4) // For HDF4.
    {
        for(vector<hdf4_attr *>::const_iterator git=file->get_sd()->get_attrs().begin(); git!=file->get_sd()->get_attrs().end(); git++)
            fattrs.push_back(*git);

        for(vector<VDATA*>::const_iterator vit=file->get_vds().begin(); vit!=file->get_vds().end(); vit++)
        {
            for(vector<hdf4_attr *>::const_iterator ait = (*vit)->get_attributes().begin(); ait!=(*vit)->get_attributes().end();ait++)
            {
                fattrs.push_back(*ait);
            }
        }
    } else { // For HDFEOS2.
        fattrs = root->get_attrs_c();
    } // end of else
    return fattrs;

}

const list<attr*> h4cf_get_var_attrs(var *v)
{
    list<attr*> vattrs;

    bool found = false;
    if(h4cf_get_var_name(v).find("vdata_")==0) // For VDATA.
        found = true;

    if(is4) // For HDF4.
    {
        if(found) { // For VDATA.
            hdf4_var_vdfield *v4 = (hdf4_var_vdfield*)(v);
            vector<hdf4_attr*> tmp = v4->get_attributes();
            vector<hdf4_attr*>::iterator it;
            for(it=tmp.begin(); it!=tmp.end(); it++) {
                vattrs.push_back(*it);
            }
        } else { // For SDS.
            hdf4_var_sdfield *v4 = (hdf4::hdf4_var_sdfield*)(v);
            change_hdf4_fill_value_attr_type(v4);
               
            vector<hdf4_attr*> tmp = v4->get_attributes();
            vector<hdf4_attr*>::iterator it;
            for(it=tmp.begin(); it!=tmp.end(); it++)
                vattrs.push_back(*it);
        } // end of else SDS
    } 
    else  {// For HDFEOS2.
        // The attribute fillvalue datatype may not be the same as the variable datatype.
        // These need to be corrected.
        value_type_t new_var_type = h4cf_get_var_type(v);
        if (true == check_eos2_fill_value_attr_type(v,new_var_type)) {
            change_eos2_fill_value_attr_type(v,new_var_type);
        }
        vattrs = v->get_attrs_c();
    }

    return vattrs;
}

void h4cf_get_attr_value(vector<char> *vals, attr *a)
{
    hdf4_attr *a4 = NULL;
	
    const char* vname = typeid(*a).name(); 
    string s(vname);
    if(s.find("hdf4")!=std::string::npos) // HDF4 case.
        a4 = (hdf4::hdf4_attr*)(a); 

    (a4==NULL)? a->get_value(vals) : a4->get_value(vals);
}

const string h4cf_get_attr_name(attr *a)
{
    return a->get_name();
}

const h4cf_data_type h4cf_get_attr_type(attr *a)
{
    return a->get_type();
}

const int h4cf_get_attr_count(attr* a)
{
    return a->get_num_elements();
}

const attr* h4cf_get_var_attr_by_name(string aname, var *v)
{

    // Will see if it is necessary to add the file attribute support. KY 2013-02-04
    attr *fvattr = NULL;

    if(h4cf_get_var_name(v).find("vdata_")==0) {// VDATA case.
        hdf4_var_vdfield *v4 = (hdf4_var_vdfield*)(v);
        fvattr = (attr*) (v4->get_attr_by_name(aname));

    }
    else if(true == is4)
    {
        hdf4_var_sdfield *v4 = (hdf4_var_sdfield*)(v);
        fvattr = (attr*)(v4->get_attr_by_name(aname));
    } else {
        fvattr = v->get_attr_by_name(aname);
    }

    return fvattr;
}

void h4cf_get_var_value(vector<char> *vals, var *v)
{
    if(true == is4) {
        if(h4cf_get_var_name(v).find("vdata_")==0) // VDATA case.
        {
            hdf4_var_vdfield *v4 =(hdf4_var_vdfield*)v;
            v4->get_all_values(vals);
        }
        else {
            hdf4_var_sdfield *v4 =(hdf4_var_sdfield*)v;
            v4->get_all_values(vals);
        }
    }
    else {
        v->get_all_values(vals);
        change_modis_value(v,vals,vals->size(),NULL,NULL,NULL);
    }
}

void h4cf_get_var_value(vector<char> *vals, var *v, int32 *start, int32 *stride, int32 *edge)
{
    /// This API is mostly like a C type API, seems unusual for a C++ package KY 2013-01-16 
    int32 tot_elems = 1;
    const vector< map<string, int> > vdims = h4cf_get_var_dims(v);
    for(vector< map<string, int> >::const_iterator vdim_iter = vdims.begin(); vdim_iter != vdims.end(); vdim_iter++)
    {
        // the algorithm may need to be simplified. KY 2012-12-07
        map<string, int> vdim = (*vdim_iter);
        map<string, int >::iterator vdimit = vdim.begin();
        int dimsize = (*vdimit).second;
        int index = distance(vdims.begin(), vdim_iter);
        try {
            if(start[index] <0 || 
               stride[index]<0 || 
               edge[index]  <=0 ||
               start[index]+stride[index]*(edge[index]-1)>dimsize-1 ||
               (0 == stride[index] && edge[index] !=1))
            {
                throw runtime_error("Invalid input parameters in start, stride or edge array. Exit anyway(" __FILE__ ":" TOSTRING(__LINE__)")" );
            }	
        }
        catch(runtime_error &e) {

            h4cf_close();
            cout<<e.what() <<endl;
            cout << "Error to call h4cf_get_var_value. "<<endl;
            cout << "The program exits abnormally. "<<endl;
            exit(1);
        }


        // SDS interface doesn't allow stride to be 0. When stride is 0, essentially only
        // one element is read in that dimension, the stride value doesn't matter, so set stride to 1.
        if (0 == stride[index])
            stride[index] = 1;
        tot_elems *= edge[index];
    } // end of for

    unsigned int storage = v->get_num_bytes(tot_elems);
    vals->resize(storage);
    
    try {

        if(true == is4) { 
        
            if(h4cf_get_var_name(v).find("vdata_")==0) // VDATA case.
            {
                //hdf4_var_vdfield *v4 = (hdf4_var_vdfield*)(v);
                v->get_value(start, stride, edge, (void*)&((*vals)[0]));
            
            }

            else {

                // This casting is also very strange, can we get rid of this? KY 2013-01-16
                hdf4_var_sdfield *v4 = (hdf4_var_sdfield*)(v);

                // I don't understand why this cannot be automatically picked up by get_value()
                // hdf4_var_sdfield is a derived class of var.  KY 2013-01-15
                // The issue is because both HDF4(hdf4_var_sdfield)  and HDF-EOS2(eos2_var_data) are opened in this package.
                // This way is not preferrable. I think the better way is for a pure HDF4 file, the eos2 open should not be called.
                // Since this will involve big changes, I am not going to touch this at this release(June 2013). However,
                // the whole structure indeed needs to be re-investigated after this release.  KY 2013-01-15
                v4->get_value(start, stride, edge, vals);
            }
        } 
        else {

            v->get_value(start, stride, edge, (void*)&((*vals)[0]));

            // Recalculate data values for MODIS products according their "scale_factor" and "add_offset" attributes.
            change_modis_value(v, vals, vals->size(), start, stride, edge);

        } // end of if
    }
    catch(runtime_error &e) {

        h4cf_close();
        cout<<e.what() <<endl;
        cout << "Error to call h4cf_get_var_value. "<<endl;
        cout << "The program exits abnormally. "<<endl;
        exit(1);
    }
}

const h4cf_data_type h4cf_get_var_type(var *v)
{
    h4cf_data_type vtype = v->get_type();
    return get_corrected_var_type(vtype, v);
}

const int h4cf_get_var_rank(var *v)
{
    return h4cf_get_var_dims(v).size();
}

void h4cf_close()
{

    if(ncmlfile != NULL)
        delete ncmlfile;
    if(file != NULL)
        delete file;
    if(root != NULL)
        delete root;
}
