// Copyright by The HDF Group
// All rights reserved.                                                      
//                                                                           
//  This file is part of the H4CF Conversion Toolkit. The full H4CF Conversion
// Toolkit copyright notice including terms governing use, modification, and 
// redistribution, is contained in the file COPYING.     
// COPYING can be found at the root of the source code    
// distribution tree. If you do not have access to this file, you may request
// a copy from eoshelp@hdfgroup.org.     

///////////////////////////////////////////////////////////////////////////////
/// \mainpage The H4CF Conversion Library
/// \version 1.2
///
/// \section Introduction
///
///   The H4CF Conversion Library converts both HDF-EOS2 and HDF4 files by 
/// following the CF conventions. The variables and attributes of the 
/// converted  HDF-EOS2 and HDF4 files are accessible through a set of 
/// high-level APIs described here. 
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
/// \file 
/// Has all functions for the H4CF Conversion Library.
///
/// The \b h4cf.h contains all APIs that user need to know to access the 
/// variables and attributes in HDF4 files following the CF conventions. 
///////////////////////////////////////////////////////////////////////////////
#ifndef H4CF_H
#define H4CF_H

#include "h4cf_header.h"
/// Opens an existing HDF-EOS2 or HDF4 file.
///
/// h4cf_open opens \a filename file and initializes the library. 
/// \param filename name of the file to be opened.
void h4cf_open(char* filename);


/// Retrieves pairs of name and size of dimension in the file.
///
/// \b h4cf_get_dims retrieves the dimension information and returns the pairs 
/// of name and size of dimension in map. The name of dimension follows the CF 
/// conventions.
///
/// For example, if the opening file has two dimensions \c XDim and \c YDim
/// with their size 360 and 180 respectively, the returned map will be
///
/// \li map[XDim] = 360
/// \li map[YDim] = 180
///
/// \returns a map containing dimension definitions. The key value in map is 
/// the name of the dimension and the mapped value is the size of the dimension.
const map<string, int> h4cf_get_dims(); 


/// Retrieves variables in the file.
///
/// \b h4cf_get_vars returns a list of pointers of all variables in the file.
/// \returns a list containing variable pointers.
const list<var*> h4cf_get_vars();

/// Retrieves the name of a variable pointed by \a v.
/// 
/// \b h4cf_get_var_name returns the name of a variable pointed by \a v.
/// The variable name follows the CF conventions.
/// \param v a pointer to a variable
/// \returns a string
const string h4cf_get_var_name(var* v);

/// Retrieves the dimensions of a variable pointed by \a v.
/// 
/// \b h4cf_get_var_dims retrieves the dimensions of a given variable and 
/// returns them in a C++ vector. Each dimension is a pair of name and size. 
/// The name of dimension follows the CF conventions.
///
/// For example, for a variable with two dimensions \c XDim and \c YDim with 
/// their size \c 360 and \c 180 respectively, the returned vector will be:
///
/// \li vector[0] = <XDim, 360>
/// \li vector[1] = <YDim, 180>
/// 
/// \param v a pointer to a variable
/// \returns a vector of maps that have dimension name and size
const vector< map<string, int> > h4cf_get_var_dims(var* v);

/// Retrieves the variable type of a variable pointed by \a v.
/// 
/// \b h4cf_get_var_type returns the data type of a variable pointed by \a v.
///  The data type can be:
/// \li \c CHAR8
/// \li \c UCHAR8
/// \li \c INT8
/// \li \c UINT8
/// \li \c INT16
/// \li \c UINT16
/// \li \c INT32
/// \li \c UINT32
/// \li \c FLOAT
/// \li \c DOUBLE
///
/// \param v a pointer to a variable
/// \returns a data type
const h4cf_data_type h4cf_get_var_type(var* v);


/// Retrieves the rank of a variable pointed by \a v.
///
/// \b h4cf_get_var_rank returns the number of dimensions of a variable 
/// pointed by \a v.
/// For example, if \a v is O3[10][20][30], this function will return 3.
/// 
/// \param v a pointer to a variable
/// \returns the rank of variable
const int h4cf_get_var_rank(var* v);

/// Retrieves data values of a variable pointed by \a v and stores them into 
/// \a buf.
///
/// \b h4cf_get_var_value reads the data values stored in a variable pointed by
/// \a v and saves them into the \a buf vector.
/// 
/// \remark For the first parameter \a buf, a user does not need to 
/// specify its capacity. The storage of vector will be allocated by the 
/// library, and the actual size of vector is equal to the number of bytes the 
/// variable holds.
///
/// \param[out] buf a pointer to store values
/// \param[in] v a pointer to a variable
/// \returns none
void h4cf_get_var_value(vector<char>* buf, var* v);

/// Retrieves subset data values of a variable pointed by \a v and stores them 
/// into\a buf.
///
/// \b h4cf_get_var_value returns the \e subset data values stored in a variable
/// pointed by \a v and saves them into \a buf vector. The subsetting is 
/// controlled by the parameters stored in \a start, \a stride, and \a edge. 
///
/// For example, if \a v has values like:
///
/// \li v[0] = 0
/// \li v[1] = 1
/// \li v[2] = 2
/// \li v[3] = 3
///
/// specifying start[0] = 1, stride[0] = 2, and edge[0] = 2 to this function
/// will return
///
/// \li buf[0] = 1 
/// \li buf[1] = 3. 
///
/// \remark For the first parameter \a buf, user does not need to 
/// specify its capacity. The storage of vector will be allocated by the 
/// library, and the actual size of vector is equal to the number of bytes the 
/// variable holds.
///
/// \param[out] buf a pointer to store values
/// \param[in] v a pointer to a variable
/// \param[in] start a pointer to array containing the position at which this 
/// function will start for each dimension
/// \param[in] stride a pointer to array specifying the interval between the 
/// data values that will be read along each dimension
/// \param[in] edge a pointer to array containing the number of data elements 
/// along each dimension
/// \returns none
void h4cf_get_var_value(vector<char>* buf, var* v, 
                        int32* start, int32* stride, int32* edge);

// This function is good for debugging only.
// It is not thoroughly tested so we'll not make it public yet.
// const string h4cf_get_var_str_value(var* v);

// This function is good for debugging only.
// It is not thoroughly tested so we'll not make it public yet.
// const map<string, string> h4cf_get_file_attrs_txt();

/// Retrieves file attributes.
/// 
/// \b h4cf_get_file_attrs returns the list of attributes.
/// \returns a list of attributes.
const list<attr*> h4cf_get_file_attrs();


/// Retrieves the attributes of a variable pointed by \a v.
/// 
/// \b h4cf_get_var_attrs returns the list of attributes in a variable pointed
/// by \a v.
///
/// \param v a pointer to a variable
/// \returns a list of attributes.
const list<attr*> h4cf_get_var_attrs(var* v);

/// Retrieves the data values of an attribute pointed by \a a and stores them 
/// into \a buf.
/// 
/// \b h4cf_get_attr_value reads the data values stored in an attribute pointed
/// by \a a and saves them into \a buf vector.
///
/// \remark For the first parameter \a buf, user does not need to specify
/// its capacity. The storage of vector will be allocated by the library, and 
/// the actual size of vector is equal to the number of bytes the attribute 
/// holds.
///
/// \param[out] buf a pointer to store values
/// \param[in] a a pointer to an attribute
/// \returns none
void h4cf_get_attr_value(vector<char>* buf, attr* a);

/// Retrieves the name of an attribute pointed by \a a.
/// 
/// \b h4cf_get_attr_name returns the name of an attribute pointed by \a a.
/// The attribute name follows the CF conventions.
///
/// \param a a pointer to an attribute
/// \returns a string
const string h4cf_get_attr_name(attr* a);

/// Retrieves the attribute type of an attribute pointed by \a a.
/// 
/// \b h4cf_get_attr_type returns the data type of an attribute pointed by \a a.
///  The data type can be:
/// \li \c CHAR8
/// \li \c UCHAR8
/// \li \c INT8
/// \li \c UINT8
/// \li \c INT16
/// \li \c UINT16
/// \li \c INT32
/// \li \c UINT32
/// \li \c FLOAT
/// \li \c DOUBLE
///
/// \param a a pointer to an attribute
/// \returns a data type
const h4cf_data_type h4cf_get_attr_type(attr* a);

/// Retrieves the number of elements stored in an attribute pointed by \a a.
///
/// \b h4cf_get_attr_count returns the number of elements stored in an attribute
/// pointed by \a a. 
///
/// For example, if \c coordsys  attribute has type \c CHAR8 and value
/// "Cartesian", the count will be 9. If \c valid_range attribute has 
//// type \c INT8 and value "0, -2", the count will be 2.
///
/// \param a a pointer to an attribute
/// \returns a number of elements.
const int h4cf_get_attr_count(attr* a); 

/// Retrieves the attribute that has \a str name from a variable pointed by 
/// \a v.
///
/// \b h4cf_get_var_attr_by_name returns the pointer to the attribute 
/// in a variable pointed by \a v if the attribute's name matches the \a str 
/// parameter.
///
/// \param str an attribute name to be searched for
/// \param v a pointer to a variable
/// \returns a pointer to the matching attribute if present, otherwise NULL.
const attr* h4cf_get_var_attr_by_name(string str, var* v);


/// Closes the access to the opened file.
///
/// \b h4cf_close terminates access to the opened file by releasing the 
/// resources held by the library. The opened file should be closed by 
/// calling this function when it is no longer needed.
///
/// \returns none
void h4cf_close();

#endif




