/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of an HDF4 
attribute.

*****************************************************************************/
#include <stdint.h>
#include "hdf4_attr.h"

namespace hdf4 {

extern std::string remove_non_printable_chars(const std::string&);

hdf4_attr::hdf4_attr(): attr(new group("non-valid name"), "non-valid name"),type(-1), count(-1) {}
	
hdf4_attr::hdf4_attr(std::string s, int32 t, int c, std::vector<char> v): attr(new group("non-valid name"), "non-valid name")
{
    this->m_name = s;
    this->type = t;
    this->count = c;
    this->value.resize(v.size());
    std::copy(v.begin(), v.end(), value.begin()); 
}
	
hdf4_attr::~hdf4_attr() { 

    if (this->m_group !=NULL)
        delete this->m_group;
}

#define VALID_RANGE(TYPE, CAST, FORMAT) \
    case DFNT_##TYPE: \
    { \
        std::string s; \
        char c[100]; \
        sprintf(c, #FORMAT, *((CAST)&value[0])); \
        s.append(c); \
        s.append(","); \
        sprintf(c, #FORMAT, *((CAST)&value[DFKNTsize(type)])); \
        s.append(c); \
        return s; \
    } \
	break;
		
const std::string hdf4_attr::get_str_value() const
{
    if(m_name.find("valid_range")!=std::string::npos)
    {
        switch(type)
        {
            VALID_RANGE(CHAR8,  char*,     %c);
            VALID_RANGE(UCHAR8, char*,     %c);
            VALID_RANGE(UINT8,  uint8_t*,  %u);
            VALID_RANGE(INT8,   int8_t*,   %d);
            VALID_RANGE(UINT16, uint16_t*, %u);
            VALID_RANGE(INT16,  int16_t*,  %d);
            VALID_RANGE(UINT32, uint32_t*, %u);
            VALID_RANGE(INT32,  int32_t*,  %d);
            VALID_RANGE(FLOAT32,float*,    %8.3e);
            VALID_RANGE(FLOAT64,double*,   %8.3e);
            default:
                throw std::range_error("Attribute's type is unknown(" __FILE__ ":" TOSTRING(__LINE__)")" );
        } // end of switch
        // Check values in "valid_range" attribute.	
        /*std::string s;
        char c[100];
        sprintf(c, "%u", (unsigned int)*((uint8_t*)&value[0]));
        s.append(c);
        s.append(",");
        sprintf(c, "%u", (unsigned int)*((uint8_t*)&value[1]));
        s.append(c);
        return s;*/
    } // end of if

    std::stringstream out;
    switch(type)
    {

//The following commeted code is for testing only.
// Cast vector<char> to a specific data type.

#if 0
/*
#define HANDLE2(TYPE, CAST) \
    case DFNT_##TYPE: \
        out << (CAST)&value[0]; \
        break;

#define HANDLE3(TYPE, T, CAST) \
    case DFNT_##TYPE: \
        out << (CAST)*((T)&value[0]); \
        break;
*/


// Cast vector<char> to a specific data type.
//out << (CAST)&(pbuf)[0]; \

// Obtain attribute value given  vector<char>.
//std::vector<char> pbuf; \
//get_value(&pbuf); \

// Cast vector<char> to a specific data type.
//out << (CAST)*((T)&(pbuf)[0]); \

#endif

        case DFNT_CHAR8: 
        { 
            std::string s; 
            for(size_t pos=0; pos<value.size(); pos++) 
                s+=value[pos]; 
            out << "\""; 
            out << s; 
            out << "\""; 
        } 
            break;
        case DFNT_UCHAR8: 
        { 
            std::string s; 
            for(size_t pos=0; pos<value.size(); pos++) 
                s+=value[pos]; 
            out << "\""; 
            out << s; 
            out << "\""; 
        } 
            break;
        case DFNT_UINT8: 
        { 
            for(int i=0; i<count; i++) 
            { 
                out << (unsigned int)*((uint8_t*)&(value)[i*DFKNTsize(type)]); 
                if(i!=count-1) 
                    out << ","; 
            } 
        } 
            break;
        case DFNT_INT8: 
        { 
            for(int i=0; i<count; i++) 
            { 
                out << (int)*((int8_t*)&(value)[i*DFKNTsize(type)]); 
                if(i!=count-1) 
                    out << ","; 
            } 
        } 
            break;
        case DFNT_UINT16: 
        {
            for(int i=0; i<count; i++) 
            { 
                out << (unsigned int)*((uint16_t*)&(value)[i*DFKNTsize(type)]); 
                if(i!=count-1) 
                    out << ","; 
            } 
        } 
            break;
        case DFNT_INT16: 
        { 
            for(int i=0; i<count; i++) 
            { 
                out << (int)*((int16_t*)&(value)[i*DFKNTsize(type)]); 
                if(i!=count-1) 
                    out << ","; 
            } 
        } 
            break;
        case DFNT_UINT32: 
        { 
            for(int i=0; i<count; i++) 
            { 
                out << (unsigned int)*((uint32_t*)&(value)[i*DFKNTsize(type)]); 
                if(i!=count-1) 
                    out << ","; 
            } 
        } 
            break;
        case DFNT_INT32: 
        { 
            for(int i=0; i<count; i++) 
            { 
                out << (int)*((int32_t*)&(value)[i*DFKNTsize(type)]); 
                if(i!=count-1) 
                    out << ","; 
            } 
        } 
            break;
        case DFNT_FLOAT32: 
        { 
            for(int i=0; i<count; i++) 
            { 
                out << (float)*((float32*)&(value)[i*DFKNTsize(type)]); 
                if(i!=count-1) 
                    out << ","; 
            } 
        } 
            break;		
        case DFNT_FLOAT64: 
        { 
            for(int i=0; i<count; i++) 
            { 
                out << (double)*((float64*)&(value)[i*DFKNTsize(type)]); 
                if(i!=count-1) 
                    out << ","; 
            } 
        } 
            break;

        // Cast vector<char> to a specific data type.
        /*case DFNT_FLOAT32:
            out << *((float*)&value[0]);
            break;
        case DFNT_FLOAT64:
            out << *((double*)&value[0]);
            break;*/
        default:
            throw std::range_error("Attribute's type is unknown(" __FILE__ ":" TOSTRING(__LINE__)")" );
    } // end of switch
    return remove_non_printable_chars(out.str());
}

} // end of namespace
