/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of an 
HDF4 file.


*****************************************************************************/

#include "hdf4_file.h"
#include <cstring>

const char*_FORWARD_SLASH="/";
namespace hdf4 {
hdf4_file::hdf4_file():sptype(OTHERHDF), sd(NULL),EOS2Swathflag(false){}

hdf4_file::~hdf4_file()
{
    if(sdfd!=-1 && sd!=NULL)
        delete sd;
    SDend(sdfd);
    int size = (int)vds.size();
    if(size!=0)
        for(std::vector<VDATA *>::const_iterator i=vds.begin(); i!=vds.end(); i++)
            delete *i;
    Vend(fileid);
    Hclose(fileid);
} 

extern filterset m_filterset; // Store filters for handling NcML.
extern filterqueue m_filterqueue; // Store filters for handling HDF4.

extern std::string remove_special_chars(const std::string&);

//void hdf4_file::read(const std::string &filename) 
void hdf4_file::read(const std::string &filename_r) 
{
    this->filename = filename_r;

    int32 sd_id;
    int32 file_id;

    if((file_id = Hopen(const_cast<char *>(filename_r.c_str()), DFACC_READ,0))==FAIL)
    {
        throw std::runtime_error("Fail to open the HDF4 file with Hopen.(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }
    if ((sd_id = SDstart (const_cast < char *>(filename_r.c_str ()), DFACC_READ)) == -1)
    {
        throw std::runtime_error("Fail to open the HDF4 file with SDstart.(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }
    sdfd   = sd_id;
    fileid = file_id;

    read(sd_id, file_id);
    read(file_id);	

    insert_orig_field_path(sd_id, file_id);
	
    // NcML procssing will be provided in the future release.
    //m_filterset.apply();
}

void hdf4_file::apply(bool use_filter)
{

    if(true == use_filter) 
        check_sd_type();

    m_filterqueue.apply(use_filter);
}

#if 0
void hdf4_file::release_unused_memory() 
{

}
#endif
std::list<var*> hdf4_file::get_vars() 
{
    std::list<var*> vars;
    for(std::vector<hdf4_var_sdfield *>::const_iterator i=sd->sdfields.begin(); i!=sd->sdfields.end(); i++)
        vars.push_back(*i);

    for(std::vector<VDATA *>::const_iterator i=vds.begin(); i!=vds.end(); i++)
    {
        for(std::vector<hdf4_var_vdfield *>::const_iterator j=(*i)->get_fields().begin(); j!=(*i)->get_fields().end(); j++)
        {        
            vars.push_back((*j));
        }
    }
	
    return vars;
}

std::map<std::string, int> hdf4_file::get_dims()
{
    std::map<std::string, int> dims;
    std::pair<std::map<std::string, int>::iterator, bool> ret;

    for(std::set<std::string>::const_iterator i=sd->fulldimnamelist.begin(); i!=sd->fulldimnamelist.end(); ++i)
    {
        int32 dimsize = sd->n1dimnamelist[*i];
        dims.insert(std::pair<std::string, int>(*i, dimsize));
        //dims.push_back(new hdf4_dim(*i, dimsize, 0));
    }

    return dims;
}

#if 0
#define STR_VALUE(TYPE, CAST, FORMAT) \
    case DFNT_##TYPE: \
    { \
        CAST *buf = new CAST[total]; \
        SDreaddata (sds_id, start, NULL, edges, buf); \
        for(int l=0; l<limit; l++) \
        { \
            sprintf(c, #FORMAT, buf[l]); \
            mystr.append(c); \
            if(l!=limit-1) \
                mystr.append(","); \
        } \
        myfield->str_value = mystr; \
        delete[] buf; \
    } \
        break;
#endif
		
void hdf4_file::read(int32 sd_id, int32 file_id) 
{
    int32 n_sds = 0;
    int32 n_sd_attrs = 0;
    int index = 0;
    int32 sds_id = -1;
    int32 dim_sizes[H4_MAX_VAR_DIMS];
    int32 n_sds_attrs = 0;
    char sds_name[H4_MAX_NC_NAME];
    char dim_name[H4_MAX_NC_NAME];
    char attr_name[H4_MAX_NC_NAME];
    int32 dim_size = 0;
    int32 sds_ref = -1;
    int32 dim_type = 0;
    int32 num_dim_attrs = 0;
    int32 attr_value_count = 0;
    intn status = -1;

    sd = new SD();

    if (SDfileinfo (sd_id, &n_sds, &n_sd_attrs) == FAIL)
        throw std::runtime_error("Fail to obtain sdsindex.(" __FILE__ ":" TOSTRING(__LINE__)")" );
        //throw std::runtime_error("SDfileinfo failed.");

    for (index = 0; index < n_sds; index++) 
    {
        hdf4_var_sdfield *myfield = new hdf4_var_sdfield();

        sds_id = SDselect (sd_id, index);
        if (sds_id == FAIL) 
        {
            SDendaccess (sds_id);
            throw std::runtime_error("SDselect Failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
        }

        sds_ref = SDidtoref (sds_id);
        if (sds_ref == FAIL) 
        {
            SDendaccess (sds_id);
            throw std::runtime_error("Fail to obtain SDS reference number.(" __FILE__ ":" TOSTRING(__LINE__)")" );
            //throw std::runtime_error("Cannot obtain SDS reference number.");
        }

        // Obtain object name, rank, size, field type and number of SDS attributes
        status = SDgetinfo (sds_id, sds_name, &myfield->rank, dim_sizes, &myfield->type, &n_sds_attrs);

        if (status == FAIL) 
        {
            SDendaccess (sds_id);
            throw std::runtime_error("SDgetinfo failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
        }
        std::string tempname (sds_name);

        myfield->m_name = tempname;
        myfield->newname = myfield->m_name;
        myfield->sdsref = sds_ref;
        sd->refindexlist[sds_ref] = index;
	
#if 0
        if(0) // For debugging only.
            std::cout << "[" << index << "] " << "m_name=" << myfield->m_name << " ref=" << myfield->sdsref << std::endl;
#endif
	
        // Handle dimensions with original dimension names
        size_t total = 1;
        // int32 *start = (int32 *)malloc(myfield->rank*sizeof(int32));
        // std::fill_n(start, myfield->rank, 0);
        // int32 *edges = (int32 *)malloc(myfield->rank*sizeof(int32));

        for (int dimindex = 0; dimindex < myfield->rank; dimindex++) 
        {
            int dimid = SDgetdimid (sds_id, dimindex);
            if (dimid == FAIL) 
            {
                SDendaccess (sds_id);
                throw std::runtime_error("SDgetdimid failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }	
            status = SDdiminfo (dimid, dim_name, &dim_size, &dim_type, &num_dim_attrs);

            total *= dim_sizes[dimindex];
            //edges[dimindex] = dim_sizes[dimindex];

            if (status == FAIL) 
            {
                SDendaccess (sds_id);
                throw std::runtime_error("SDdiminfo failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }
            std::string dim_name_str = remove_special_chars(dim_name);
            std::string dim_orig_name(dim_name);

            // Since dim_size will be 0 if the dimension is unlimited dimension, so use dim_sizes instead
            hdf4_dim *mydim = new hdf4_dim(dim_name_str, dim_orig_name,dim_sizes[dimindex], dim_type);
#if 0
            if(0) // For debugging only.
                std::cout << "\tdim name=" << mydim->get_name() << " dim size=" << mydim->get_size() << " dim type=" << mydim->get_type() << std::endl;
#endif

            myfield->dims.push_back(mydim);
        } // end of for 

        // Obtain string representation of data values.
#if 0
        /*std::string mystr;
        char c[100];
        int limit = (total>100)?100: total; //total; 
        switch(myfield->type) 
        {
            STR_VALUE(CHAR8,   char,   %c);
            STR_VALUE(UCHAR8,  char,   %c);
            STR_VALUE(UINT8,   uint8,  %u);
            STR_VALUE(INT8,    int8,   %d);
            STR_VALUE(UINT16,  uint16, %u);
            STR_VALUE(INT16,   int16,  %d);
            STR_VLAUE(UINT32,  uint32, %u);
            STR_VALUE(INT32,   int32,  %d);
            STR_VALUE(FLOAT32, float,  %.6f);
            STR_VALUE(FLOAT64, double, %8.3e);
            default:                        
                throw std::range_error("SDS's type is unknown(" __FILE__ ":" TOSTRING(__LINE__)")" );
        }; // end of switch		

        delete[] start;
        delete[] edges;*/ 

        // Read SDreaddata with vector<char>.
        //SDreaddata (sds_id, start, NULL, edges, (VOIDP)&myfield->value[0]);
#endif
        // Handle SDS attributes
        for (int attrindex = 0; attrindex < n_sds_attrs; attrindex++)
        {
            hdf4_attr *myattr = new hdf4_attr();
            status = SDattrinfo (sds_id, attrindex, attr_name, &myattr->type, &attr_value_count);
        	       
            if (status == FAIL) 
            {
                SDendaccess (sds_id);
                throw std::runtime_error("SDattrinfo failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }
            std::string tempname (attr_name);
            myattr->m_name = tempname; //remove_special_chars(tempname);
            myattr->count = attr_value_count;
            myattr->value.resize (attr_value_count * DFKNTsize(myattr->type));
#if 0
            if(0) // For debugging only.
                std::cout << "\tattr name=" << myattr->m_name << " attr type=" << myattr->type << " attr size=" << attr_value_count * sizeof (myattr->type);
#endif

            if (SDreadattr (sds_id, attrindex, &myattr->value[0]) == -1) 
            {
                SDendaccess (sds_id);
                throw std::runtime_error("read SDS attribute failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

#if 0
            if(0) // For debugging only.
            {
                std::string attrvalue (myattr->value.begin(),myattr->value.end ());
                std::cout << " attr value=" << myattr->get_str_value() << std::endl;
            }
#endif
            myfield->attrs.push_back (myattr);
        } // end of for
        SDendaccess (sds_id);
        sd->sdfields.push_back (myfield);
    } // end of for (index = 0; index 

    //Handle SD attributes
    for (int attrindex = 0; attrindex < n_sd_attrs; attrindex++) 
    {
        hdf4_attr *myattr = new hdf4_attr();
        status = SDattrinfo (sd_id, attrindex, attr_name, &myattr->type, &attr_value_count);
        if (status == FAIL)
            throw std::runtime_error("SDattrinfo failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
        std::string tempname (attr_name);
        myattr->m_name = tempname; //remove_special_chars(tempname);
        myattr->count = attr_value_count;
        myattr->value.resize (attr_value_count * DFKNTsize(myattr->type));
        if (SDreadattr (sd_id, attrindex, &myattr->value[0]) == -1)
            throw std::runtime_error("Cannot read SD attribute.(" __FILE__ ":" TOSTRING(__LINE__)")");
        sd->attrs.push_back (myattr);
    } // end of for
    //status = SDend (sd_id);
}

void hdf4_file::read(int32 file_id) 
{
    int status = Vstart(file_id);

    if(status==FAIL)
        throw std::runtime_error("Can't start vdata/vgroup interface.(" __FILE__ ":" TOSTRING(__LINE__)")");

    int num_lone_vdata = VSlone (file_id, NULL, 0);

    if (num_lone_vdata == FAIL)
        throw std::runtime_error("Fail to obtain lone vdata number.(" __FILE__ ":" TOSTRING(__LINE__)")");

    char vdata_class[VSNAMELENMAX];
    char vdata_name[VSNAMELENMAX];

#if 0
    if(0) // For debugging only.
        std::cout << "VSlone=" << num_lone_vdata << std::endl;
#endif

    if (num_lone_vdata > 0) 
    {
        int32 *ref_array = (int32 *) calloc (num_lone_vdata, sizeof (int32));
        if (ref_array == NULL)
            throw std::runtime_error("no enough memory to allocate the buffer.(" __FILE__ ":" TOSTRING(__LINE__)")");
        if (VSlone (file_id, ref_array, num_lone_vdata) == FAIL) 
        {
            free (ref_array);
            throw std::runtime_error ("cannot obtain lone vdata reference arrays.(" __FILE__ ":" TOSTRING(__LINE__)")");
        }

        for (int i=0; i< num_lone_vdata; i++) 
        {
            int32 vdata_id;
            vdata_id = VSattach(file_id, ref_array[i], "r");
            if (vdata_id==FAIL) 
            {
                free(ref_array);
                throw std::runtime_error("Fail to attach Vdata.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            status = VSgetclass (vdata_id, vdata_class);
            if (status==FAIL) 
            {
                VSdetach(vdata_id);
                free(ref_array);
                throw std::runtime_error("Fail to obtain Vdata class.(" __FILE__ ":" TOSTRING(__LINE__)")");
            } // end of if
            if (VSgetname(vdata_id, vdata_name)==FAIL) 
            {
                VSdetach(vdata_id);
                free(ref_array);
                throw std::runtime_error("Fail to obtain Vdata name.(" __FILE__ ":" TOSTRING(__LINE__)")");
            } // end of if

            if (VSisattr(vdata_id)==TRUE
                || !strncmp (vdata_class, _HDF_CHK_TBL_CLASS, strlen(_HDF_CHK_TBL_CLASS))
                || !strncmp (vdata_class, _HDF_SDSVAR, strlen(_HDF_SDSVAR))
                || !strncmp (vdata_class, _HDF_CRDVAR, strlen (_HDF_CRDVAR))
                || !strncmp (vdata_class, DIM_VALS, strlen (DIM_VALS))
                || !strncmp (vdata_class, DIM_VALS01, strlen (DIM_VALS01))
                || !strncmp (vdata_class, RIGATTRCLASS, strlen (RIGATTRCLASS))
                || !strncmp (vdata_name, RIGATTRNAME, strlen (RIGATTRNAME)))
            {   
                status = VSdetach(vdata_id);
                if (status==FAIL) 
                {
                    free(ref_array);
                    throw std::runtime_error("VSdetach failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                }
            } // end of if (VSisattr
            else 
            {
                VDATA *vdataobj = VDATA::read (vdata_id, ref_array[i]);
                for (std::vector< hdf4_var_vdfield * >::const_iterator it_vdf= vdataobj->get_fields ().begin (); it_vdf != vdataobj->get_fields ().end (); it_vdf++) 
                {
                    (*it_vdf)->newname = "vdata_" + vdataobj->newname +"_vdf_" +  (*it_vdf)->m_name ;
                } // end of for

                vds.push_back (vdataobj);

                // To know if the data product is CERES, we need to check Vdata CERE_metadata(CERE_META_NAME). One field name LOCALGRANULEID(CERE_META_FIELD_NAME) includes the product name.
                //if(vdata_name!=NULL && strlen(vdata_name)!=0)
                if(strlen(vdata_name)!=0)
                    check_ceres(vdata_name, vdata_id, ref_array);

                //VSdetach(vdata_id); //Move to VDATA's destructor.
        	//free (ref_array);
            } // end of else
        } // end of for (int i=0; i< num_lone_vdata; 
        free (ref_array);
    } // end of if (num_lone_vdata > 0)
}

// Check if this is an special EOS file that only needs to be handled by HDF4 APIs
// We handle AIRS version 6 level 2 and level 3. Old merra level 3 products.
bool hdf4_file::check_special_eos_no_filters(const std::string & filename )  {

    bool special_eos_no_filters = false;
    bool airs_v6_merra_eos_candidate = false;

    std::string basefilename;
    if(filename.find_last_of("/") !=std::string::npos)
        basefilename = filename.substr(filename.find_last_of("/")+1);
    else
        basefilename = filename;

    bool airs_l3 = false;
    bool airs_l2 = false;
    bool merra_l3 = false;
    if((basefilename.size() > 8) && (basefilename.compare(0,4,"AIRS") == 0)){
        if(basefilename.find(".L3.")!=std::string::npos)
            airs_l3 = true;
        else if(basefilename.find(".L2.")!=std::string::npos)
            airs_l2 = true;
    }
    else if ((basefilename.size() > 5) && (basefilename.compare(0,5,"MERRA") == 0))
        merra_l3 = true;

   
    // Now check if they are real HDF-EOS2 AIRS or MERRA products.
    if(true == airs_l3 || true == airs_l2 || true == merra_l3) {
        for(std::vector<hdf4_attr *>::const_iterator i=this->sd->get_attrs().begin(); i!=this->sd->get_attrs().end(); i++) {
            if((*i)->get_name() == "StructMetadata.0") { 
                airs_v6_merra_eos_candidate = true;
                break;
            }
        }
    }

    if(true == airs_v6_merra_eos_candidate) {
        for (std::vector < hdf4_var_sdfield * >::const_iterator i =this->sd->sdfields.begin (); i != this->sd->sdfields.end (); ++i)
        {
            for (std::vector < hdf4_dim * >::const_iterator k = (*i)->get_dimensions ().begin (); k != (*i)->get_dimensions ().end (); ++k)
            {
                if((*k)->get_type() !=0) {
                    special_eos_no_filters = true;
                    break;
                }
            }
            if(true == special_eos_no_filters)
                break;
        }

    }

    return special_eos_no_filters;

}

// Check if this is an special EOS file that only needs to be handled by HDF4 APIs
void hdf4_file::handle_special_eos_no_filters()  {


    bool airs_l3 = false;
    bool airs_l2 = false;
    bool merra_l3 = false;
    if(this->filename.find("AIRS") !=std::string::npos) {
       if((this->filename).find(".L2.")!=std::string::npos)
           airs_l2 = true;
       else if((this->filename).find(".L3.")!=std::string::npos)
           airs_l3 = true;
    }
    else if(this->filename.find("MERRA") !=std::string::npos) 
         merra_l3 = true;
    else 
        throw std::runtime_error("This is not a MERRA or an AIRS file, please check the file name.(" __FILE__ ":" TOSTRING(__LINE__)")");

    //Remove redundant XDim, YDim, Height etc. fields in the grid
    if(true == merra_l3)
        remove_merra_l3_dup_vars();

    // set of names of dimensions that have dimension scales.
    std::set<std::string> scaled_dname_set;

    // set of names of dimensions that don't have dimension scales.
    std::set<std::string> non_scaled_dname_set;
    std::pair<std::set<std::string>::iterator,bool> ret;

    // For dimensions that don't have dimension scales, a map between dimension name and size.
    std::map<std::string,int> non_scaled_dname_to_size;


    // 1.  Loop through SDS fields and remove suffixes(:???) of the dimension names and the variable names. 
    //     Note: AIRS may have several grids. Their data variables are distingushible. Their dimension scale
    //           variables may be redundant. duplicate variables should be removed.
    // An Example: StdPressureLev:asecending is the same as the StdPressureLev:descending, reduce to StdPressureLev 
    //     In this step, we only remove the suffixes(:), in step 2, those dimension variables are reduced.
    //     We do this way to ensure the non-dimension variables with: are not removed.
    //     Also create scaled dim. name set and non-scaled dim. name set.
    for (std::vector < hdf4_var_sdfield * >::const_iterator i =
            this->sd->sdfields.begin (); i != this->sd->sdfields.end (); ++i) {

        std::string tempname = (*i)->get_name();
        size_t found_colon = tempname.find_first_of(':');

        // some fields have field name like XDIM:grid, the :grid part is redundant. Remove that part.
        if(found_colon!=std::string::npos) 
            (*i)->newname = tempname.substr(0,found_colon);
        else // Otherwise, we want to make the variable new name(with path) the same as the original name.
            (*i)->newname = (*i)->get_name();

        for (std::vector < hdf4_dim * >::const_iterator k =
            (*i)->get_dimensions ().begin ();
            k != (*i)->get_dimensions ().end (); ++k) {

            //Must obtain orig name to find : in the XDim:???
            std::string tempname = (*k)->get_orig_name();
            size_t found_colon = tempname.find_first_of(':');
            if(found_colon!=std::string::npos)
                (*k)->rename(tempname.substr(0,found_colon));

            if(0==(*k)->get_type()) {
                ret = non_scaled_dname_set.insert((*k)->get_name());
                if (true == ret.second)
                    non_scaled_dname_to_size[(*k)->get_name()] = (*k)->get_size();
            }
            else
                scaled_dname_set.insert((*k)->get_name());

        }

    }
#if 0
    for (std::vector < hdf4_var_sdfield * >::const_iterator i =
            file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {
         (*i)->newname = (*i)->get_name();

    }
#endif
   
#if 0
for(set<std::string>::const_iterator sdim_it = scaled_dname_set.begin();
                                    sdim_it !=scaled_dname_set.end();
                                    ++sdim_it) {
cerr<<"scaled dim. name "<<*sdim_it <<endl;

}
#endif

    // For AIRS  level 3 only(MERRA is not affected now, but may be useful in the future.) ****
    // 2. Remove potential redundant CVs
    // For AIRS level 3 version 6 products, many dimension scale variables shared the same value. Physically they are the same.
    // So to keep the performance optimal and to reduce the non-necessary clutter, I remove the duplicate variables.
    // An Example: StdPressureLev:asecending is the same as the StdPressureLev:descending, reduce to StdPressureLev 

    // Make a copy of the scaled-dim name set:scaled-dim-marker
    if(true == airs_l3 || true == merra_l3) {

        std::set<std::string>scaled_dname_set_marker = scaled_dname_set;

        // Loop through all the SDS objects, 
        // If finding a 1-D variable name 
        // b1) in both the scaled-dim name set and the scaled-dim-marker set, 
        //     keep this variable but remove the variable name from the scaled-dim-marker. 
        //     Mark this variable as a CV.(XDim: 2, YDim:1 Others: 3). 
        // b2) In the scaled-dim name set but not in the scaled-dim-marker set, 
        //     remove the variable from the variable vector.
        for (std::vector < hdf4_var_sdfield * >::iterator i =
            this->sd->sdfields.begin (); i != this->sd->sdfields.end (); ++i) {
            if(1 == (*i)->get_rank()) {
                if(scaled_dname_set.find((*i)->get_new_name())!=scaled_dname_set.end()) {
                    if(scaled_dname_set_marker.find((*i)->get_new_name())!=scaled_dname_set_marker.end()) 
                        scaled_dname_set_marker.erase((*i)->get_new_name());
                    else {// Redundant variables
                        delete(*i);
                        this->sd->sdfields.erase(i);
                        // when erasing the iterator, it always goes to the next elment, so move back.
                        i--;
                    }
                }
            }
            // Remove the redundant 2-D Latitude and Longitude under the location group.
            else if( 2 == (*i)->get_rank()) {
                if ("Latitude" == (*i)->get_new_name() ||  "Longitude" == (*i)->get_new_name()) {
                    delete(*i);
                    this->sd->sdfields.erase(i);
                    i--;
                }
            }
            else
                ++i;
        }
    }

#if 0
for(set<std::string>::const_iterator sdim_it = scaled_dname_set.begin();
                                    sdim_it !=scaled_dname_set.end();
                                    ++sdim_it) {
cerr<<"new scaled dim. name "<<*sdim_it <<endl;

}
#endif

    //3. Add potential missing CVs

    // 3.1 Find the true dimensions that don't have dimension scales.
    std::set<std::string>final_non_scaled_dname_set;
    for(std::set<std::string>::const_iterator non_sdim_it = non_scaled_dname_set.begin();
                                    non_sdim_it !=non_scaled_dname_set.end();
                                    ++non_sdim_it) {
        if(scaled_dname_set.find(*non_sdim_it)==scaled_dname_set.end())
            final_non_scaled_dname_set.insert(*non_sdim_it);
    }

    // 3.2 Create the missing CVs based on the non-scaled dimensions.
    for(std::set<std::string>::const_iterator non_sdim_it = final_non_scaled_dname_set.begin();
                                    non_sdim_it !=final_non_scaled_dname_set.end();
                                    ++non_sdim_it) {

        hdf4_var_sdfield *missingfield = new hdf4_var_sdfield ();

        missingfield->type = DFNT_INT32;
        missingfield->rename(*non_sdim_it);
        missingfield->newname = *non_sdim_it;
        missingfield->rank = 1;
        missingfield->fieldtype = 4;
        std::string tempunits = "level";
        missingfield->add_attr_one_str("units", tempunits);
        //missingfield->setUnits("level");
        int32 dimsize = (int32)(non_scaled_dname_to_size[*non_sdim_it]);
        hdf4_dim *dim = new hdf4_dim(*non_sdim_it,*non_sdim_it,dimsize , 0);

        missingfield->dims.push_back (dim);
        std::vector<int32>fieldvalue;
        fieldvalue.resize(dimsize);
        for (int i = 0; i<(int)dimsize;i++)
            fieldvalue[i]=i;
        missingfield->value.resize(sizeof(int) * dimsize);
        memcpy((void*)&(missingfield->value)[0],(const void*)(&fieldvalue[0]),sizeof(int)*dimsize);
        
        this->sd->sdfields.push_back (missingfield);
    }

    // For AIRS level 3 only
    // Change XDim to Longitude and YDim to Latitude for field name and dimension names
    if(true == airs_l3) {
        for (std::vector < hdf4_var_sdfield * >::const_iterator i =
            this->sd->sdfields.begin (); i != this->sd->sdfields.end (); ++i) {

            if(1 ==(*i)->get_rank()){
                if ("XDim" == (*i)->newname)
                    (*i)->newname = "Longitude";
                else if ("YDim" == (*i)->newname)
                    (*i)->newname = "Latitude";
            }

            for (std::vector < hdf4_dim * >::const_iterator k =
                (*i)->get_dimensions ().begin ();
                k != (*i)->get_dimensions ().end (); ++k) {
                if("XDim" == (*k)->get_name())
                    (*k)->rename("Longitude");
                else if ("YDim" == (*k)->get_name())
                    (*k)->rename("Latitude");
            }

        }
    }

    // For AIRS level 2 only
    if(true == airs_l2) {

        bool change_lat_unit = false;
        bool change_lon_unit = false;
        std::string ll_dimname1 = "";
        std::string ll_dimname2 = "";

        // 1. Assign the lat/lon units according to the CF conventions. 
        // 2. Obtain dimension names of lat/lon.
        for (std::vector < hdf4_var_sdfield * >::const_iterator i =
            this->sd->sdfields.begin (); i != this->sd->sdfields.end (); ++i) {

            if(2 == (*i)->get_rank()) {
                if("Latitude" == (*i)->newname){
                    (*i)->fieldtype = 1;
                    change_lat_unit = true;
                    std::string tempunits = "degrees_north";
                    (*i)->add_attr_one_str("units", tempunits);
                    //(*i)->setUnits(tempunits);
                    ll_dimname1 = (*i)->get_dimensions()[0]->get_name();
                    ll_dimname2 = (*i)->get_dimensions()[1]->get_name();

                }
                else if("Longitude" == (*i)->newname) {
                    (*i)->fieldtype = 2;
                    change_lon_unit = true;
                    std::string tempunits = "degrees_east";
                    //(*i)->setUnits(tempunits);
                    (*i)->add_attr_one_str("units", tempunits);
                }
                if((true == change_lat_unit) && (true == change_lon_unit))
                    break;
            }
        }

        // 2. Generate the coordinate attribute
        std::string tempcoordinates = "";
        std::string tempfieldname   = "";
        int tempcount = 0;

        for (std::vector < hdf4_var_sdfield * >::const_iterator i =
            this->sd->sdfields.begin (); i != this->sd->sdfields.end (); ++i) {

            // We don't want to add "coordinates" attributes to all dimension scale variables.
            bool dimscale_var = false;
            //std::string temp_dimname;

            dimscale_var = ((*i)->rank == 1) & (((*i)->newname) == ((*i)->get_dimensions()[0]->get_name()));

            if((0 ==(*i)->fieldtype) && (false == dimscale_var)) {

                tempcount = 0;
                tempcoordinates = "";
                tempfieldname = "";

                // First check if the dimension names of this variable include both ll_dimname1 and ll_dimname2.
                bool has_lldim1 = false;
                bool has_lldim2 = false;
                for (std::vector < hdf4_dim * >::const_iterator j =
                    (*i)->get_dimensions().begin ();
                    j != (*i)->get_dimensions().end (); ++j) {
                    if((*j)->get_name() == ll_dimname1)
                        has_lldim1 = true;
                    else if ((*j)->get_name() == ll_dimname2)
                        has_lldim2 = true;
                    if((true == has_lldim1) && (true == has_lldim2))
                        break;

                }


                if((true == has_lldim1) && (true == has_lldim2)) {
                  for (std::vector < hdf4_dim * >::const_iterator j =
                    (*i)->get_dimensions().begin ();
                    j != (*i)->get_dimensions().end (); ++j) {
                    if((*j)->get_name() == ll_dimname1)
                        tempfieldname = "Latitude";
                    else if ((*j)->get_name() == ll_dimname2)
                        tempfieldname = "Longitude";
                    else
                        tempfieldname = (*j)->get_name();

                    if (0 == tempcount)
                        tempcoordinates = tempfieldname;
                    else
                        tempcoordinates = tempcoordinates + " " + tempfieldname;
                    tempcount++;
                  }
                }
                else {
                  for (std::vector < hdf4_dim * >::const_iterator j =
                    (*i)->get_dimensions().begin ();
                    j != (*i)->get_dimensions().end (); ++j) {
                    if (0 == tempcount)
                       tempcoordinates = (*j)->get_name();
                    else
                        tempcoordinates = tempcoordinates + " " + (*j)->get_name();
                    tempcount++;
                  }

                }
                if(tempcoordinates.size()!=0)
                    (*i)->add_attr_one_str("coordinates", tempcoordinates);

                //(*i)->setCoordinates (tempcoordinates);

            }
        }
    }

    // Add the correcteddims.
    for (std::vector < hdf4_var_sdfield * >::const_iterator i =
            this->sd->sdfields.begin (); i != this->sd->sdfields.end (); ++i) {

        for (std::vector < hdf4_dim * >::const_iterator k =
            (*i)->get_dimensions ().begin ();
            k != (*i)->get_dimensions ().end (); ++k) {
            hdf4_dim* dim = new hdf4_dim((*k)->get_name(),(*k)->get_orig_name(),(*k)->get_size(),(*k)->get_type());
            (*i)->correcteddims.push_back(dim);
        }

    }

    // Generate fulldimnamelist for the function get_dims()
    for (std::vector < hdf4_var_sdfield * >::const_iterator i = this->sd->sdfields.begin (); i != this->sd->sdfields.end (); ++i)
    {
        for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j)
        {
            std::pair < std::set < std::string >::iterator, bool > ret;
            ret = this->sd->fulldimnamelist.insert ((*j)->get_name ());
            // Map from the unique dimension name to its size
            if (ret.second == true)
                this->sd->n1dimnamelist[(*j)->get_name ()] = (*j)->get_size ();
        } // end of for
    } // end of for

    // Remove special characters in the attribute name.(StructMetadata.0 should be StructMetadata_0)
    for(std::vector<hdf4_attr *>::const_iterator i=this->sd->get_attrs().begin(); i!=this->sd->get_attrs().end(); i++) 
        (*i)->rename(remove_special_chars((*i)->get_name())); 

}


void hdf4_file::check_ceres(const char *vdata_name, int32 vdata_id, int32 *ref_array) 
{
    if (!strncmp(vdata_name, CERE_META_NAME, strlen (CERE_META_NAME)))
    {
        char *fieldname = NULL;
        int num_field = VFnfields (vdata_id);
        if (num_field == FAIL)
        {
            free (ref_array);
            throw std::runtime_error("number of fields at Vdata.(" __FILE__ ":" TOSTRING(__LINE__)")");
        } // end of if

        for (int j = 0; j < num_field; j++)
        {
            fieldname = VFfieldname (vdata_id, j);
            if (fieldname == NULL)
            {
                free (ref_array);
                throw std::runtime_error("VFfieldname failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            } // end of if
            if (!strcmp (fieldname, CERE_META_FIELD_NAME)) 
            {
                int32 fieldsize = 0;
                int32 nelms = 0;
                fieldsize = VFfieldesize (vdata_id, j);
                if (fieldsize == FAIL) 
                {
                    free (ref_array);
                    throw std::runtime_error("VFieldsize failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                } // end of if
                nelms = VSelts (vdata_id);
                if (nelms == FAIL) 
                {
                    free (ref_array);
                    throw std::runtime_error("VSelts failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                } // end of if
                char *databuf = (char *) malloc (fieldsize * nelms);
                if (databuf == NULL) 
                {
                    free (ref_array);
                    VSdetach (vdata_id);
                    throw std::runtime_error("No enough memory to allocate buffer.(" __FILE__ ":" TOSTRING(__LINE__)")");
                } // end of if
                if (VSseek (vdata_id, 0) == FAIL) 
                {
                    free (ref_array);
                    free (databuf);
                    throw std::runtime_error("VSeek failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                } // end of if
                if (VSsetfields (vdata_id, CERE_META_FIELD_NAME) == FAIL) 
                {
                    free (ref_array);
                    free (databuf);
                    throw std::runtime_error("VSsetfields failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                } // end of if
                if (VSread(vdata_id, (uint8 *) databuf, 1, FULL_INTERLACE) == FAIL) 
                {
                    free (ref_array);
                    free (databuf);
                    throw std::runtime_error("VSread failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                } // end of if
                if (!strncmp(databuf, CER_AVG_NAME, strlen (CER_AVG_NAME)))
                    sptype = CER_AVG;
                else if (!strncmp(databuf, CER_ES4_NAME, strlen (CER_ES4_NAME)))
                    sptype = CER_ES4;
                else if (!strncmp(databuf, CER_CDAY_NAME, strlen (CER_CDAY_NAME)))
                    sptype = CER_CDAY;
                else if (!strncmp(databuf, CER_CGEO_NAME, strlen (CER_CGEO_NAME)))
                    sptype = CER_CGEO;
                else if (!strncmp(databuf, CER_SRB_NAME, strlen (CER_SRB_NAME)))
                    sptype = CER_SRB;
                else if (!strncmp(databuf, CER_SYN_NAME, strlen (CER_SYN_NAME)))
                    sptype = CER_SYN;
                else if (!strncmp(databuf, CER_ZAVG_NAME, strlen (CER_ZAVG_NAME)))
                    sptype = CER_ZAVG;
                free (databuf);
            } // end of if (!strcmp (fieldname, CERE_M
        } // end of for (int j = 0; j < num_fi
    } // end of if (!strncmp(vdata_name, CERE_M 
}

void hdf4_file::check_sd_type() 
{
     // check the TRMM version 7  cases
    // The default sptype is OTHERHDF.
    // 2A,2B check attribute FileHeader, FileInfo and SwathHeader
    // 3A,3B check attribute FileHeader, FileInfo and GridHeader
    // 3A25 check attribute FileHeader, FileInfo and GridHeader1, GridHeader2
    if (this->sptype == OTHERHDF) {

        int trmm_multi_gridflag = 0;
        int trmm_single_gridflag = 0;
        int trmm_swathflag = 0;

        for (std::vector < hdf4_attr * >::const_iterator i =
            this->sd->get_attrs().begin ();
            i != this->sd->get_attrs().end (); ++i) {
            if ((*i)->get_name() == "FileHeader") {
                trmm_multi_gridflag++;
                trmm_single_gridflag++;
                trmm_swathflag++;
            }
            if ((*i)->get_name() == "FileInfo") {
                trmm_multi_gridflag++;
                trmm_single_gridflag++;
                trmm_swathflag++;
            }
            if ((*i)->get_name() == "SwathHeader") 
                trmm_swathflag++;

            if ((*i)->get_name() == "GridHeader")
                trmm_single_gridflag++;

            else if (((*i)->get_name().find ("GridHeader") == 0) &&
                     (((*i)->get_name()).size() >10))
                trmm_multi_gridflag++;

        }

        if(3 == trmm_single_gridflag) 
            this->sptype = TRMML3S_V7;
        else if(3 == trmm_swathflag) 
            this->sptype = TRMML2_V7;
        else if(trmm_multi_gridflag >3)
            this->sptype = TRMML3M_V7;
            
    }
    // check the TRMM version 6 and MODARNSS/MYDARNSS cases
    if (this->sptype == OTHERHDF) 
    {
        int metadataflag = 0;
        for (std::vector < hdf4_attr * >::const_iterator i = sd->get_attrs ().begin (); i != sd->get_attrs ().end (); ++i) 
        {
            if ((*i)->get_name () == "CoreMetadata.0")
                metadataflag++;
            if ((*i)->get_name () == "ArchiveMetadata.0")
                metadataflag++;
            if ((*i)->get_name () == "StructMetadata.0")
                metadataflag++;
            if ((*i)->get_name ().find ("SubsettingMethod") !=
                std::string::npos)
            metadataflag++;
        } // end of for (std::vector

        if (metadataflag == 4)	// This is a very special MODIS product. It includes StructMetadata.0 but it is not an HDF-EOS2 file. 
            sptype = MODISARNSS;
        if (metadataflag == 2) 
        {	
            // DATA_GRANULE is the TRMM "swath" name; geolocation is the TRMM "geolocation" field.
            for (std::vector < hdf4_var_sdfield * >::const_iterator i = sd->get_fields ().begin (); i != sd->get_fields ().end (); ++i) 
            {
                if (((*i)->get_name () == "geolocation") && (*i)->get_new_name ().find ("DATA_GRANULE") != std::string::npos 
                     && (*i)->get_new_name ().find ("SwathData") != std::string::npos && (*i)->get_rank () == 3) 
                {
                    sptype = TRMML2_V6;
                    break;
                }
            } // end of for
            // This is for TRMM Level 3 3B42 and 3B43 data. The vgroup name is DATA_GRANULE. At least one field is 1440*400 array. The information is obtained from // http://disc.sci.gsfc.nasa.gov/additional/faq/precipitation_faq.shtml#lat_lon
            if (sptype == OTHERHDF) 
            {
                for (std::vector < hdf4_var_sdfield * >::const_iterator i = sd->get_fields ().begin ();	 i != sd->get_fields ().end (); ++i) 
                {
                    if ((*i)->get_new_name ().find ("DATA_GRANULE") !=std::string::npos) 
                    {
                        int loncount = 0;
                        int latcount = 0;
                        for (std::vector < hdf4_dim * >::const_iterator k = (*i)->get_dimensions ().begin (); k != (*i)->get_dimensions ().end (); ++k) 
                        {
                            if ((*k)->get_size () == 1440)
                                loncount++;
                            if ((*k)->get_size () == 400)
                                latcount++;
                        } // end of for
                        if (loncount == 1 && latcount == 1) 
                        {
                            sptype = TRMML3_V6;
                            break;
                        }
                    } // end of if ((*i)->get_new_name
                } // end of for(std::vector < hdf4_var_sdfield
            } // end of if (sptype == OTHERHDF)
        } // end of  if (metadataflag == 2)
    } // end of if (this->sptype == OTHERHDF)

    // Check the OBPG case
    // OBPG includes SeaWIFS,OCTS,CZCS,MODISA,MODIST
    // One attribute "Product Name" includes unique information for each product,
    // For example, SeaWIFS L2 data' "Product Name" is S2003006001857.L2_MLAC 
    // Since we only support Level 2 and Level 3m data, we just check if those characters exist inside the attribute.
    // The reason we cannot support L1A data is lat/lon consists of fill values.
    if (sptype == OTHERHDF) 
    {
        int modisal2flag = 0;
        int modistl2flag = 0;
        int octsl2flag = 0;
        int seawifsl2flag = 0;
        int czcsl2flag = 0;

        int modisal3mflag = 0;
        int modistl3mflag = 0;
        int octsl3mflag = 0;
        int seawifsl3mflag = 0;
        int czcsl3mflag = 0;

        for (std::vector < hdf4_attr * >::const_iterator i = sd->get_attrs ().begin (); i != sd->get_attrs ().end (); ++i) 
        {
            if ((*i)->get_name () == "Product Name") 
            {
                std::string attrvalue ((*i)->get_value ().begin (),  (*i)->get_value ().end ());
                if ((attrvalue.find_first_of ('A', 0) == 0) && (attrvalue.find (".L2", 0) != std::string::npos))
                    modisal2flag++;
                else if ((attrvalue.find_first_of ('A', 0) == 0) && (attrvalue.find (".L3m", 0) != std::string::npos))
                   modisal3mflag++;
                else if ((attrvalue.find_first_of ('T', 0) == 0) && (attrvalue.find (".L2", 0) != std::string::npos))
                    modistl2flag++;
                else if ((attrvalue.find_first_of ('T', 0) == 0) && (attrvalue.find (".L3m", 0) != std::string::npos))
                    modistl3mflag++;
                else if ((attrvalue.find_first_of ('O', 0) == 0) && (attrvalue.find (".L2", 0) != std::string::npos))
                    octsl2flag++;
                else if ((attrvalue.find_first_of ('O', 0) == 0) && (attrvalue.find (".L3m", 0) != std::string::npos))
                    octsl3mflag++;
                else if ((attrvalue.find_first_of ('S', 0) == 0) && (attrvalue.find (".L2", 0) != std::string::npos))
                    seawifsl2flag++;
                else if ((attrvalue.find_first_of ('S', 0) == 0) && (attrvalue.find (".L3m", 0) != std::string::npos))
                    seawifsl3mflag++;
                else if ((attrvalue.find_first_of ('C', 0) == 0) && ((attrvalue.find (".L2", 0) != std::string::npos)|| ((attrvalue.find (".L1A", 0) != std::string::npos))))
                    czcsl2flag++;
                else if ((attrvalue.find_first_of ('C', 0) == 0) && (attrvalue.find (".L3m", 0) != std::string::npos))
                    czcsl3mflag++;
            } // end of if ((*i)->get_name () == "Product 

            if ((*i)->get_name () == "Sensor Name") 
            {
                std::string attrvalue ((*i)->get_value ().begin (), (*i)->get_value ().end ());
                if (attrvalue.find ("MODISA", 0) != std::string::npos) 
                {
                    modisal2flag++;
                    modisal3mflag++;
                } 
                else if (attrvalue.find ("MODIST", 0) != std::string::npos) 
                {
                    modistl2flag++;
                    modistl3mflag++;
                }
                else if (attrvalue.find ("OCTS", 0) != std::string::npos) 
                {
                    octsl2flag++;
                    octsl3mflag++;
                }
                else if (attrvalue.find ("SeaWiFS", 0) != std::string::npos)
                {
                    seawifsl2flag++;
                    seawifsl3mflag++;
                }
                else if (attrvalue.find ("CZCS", 0) != std::string::npos) 
                {
                    czcsl2flag++;
                    czcsl3mflag++;
                }
            } // end of if ((*i)->get_name () == "Sensor

            if ((modisal2flag == 2) || (modisal3mflag == 2)
                || (modistl2flag == 2) || (modistl3mflag == 2)
                || (octsl2flag == 2) || (octsl3mflag == 2)
                || (seawifsl2flag == 2) || (seawifsl3mflag == 2)
                || (czcsl2flag == 2) || (czcsl3mflag == 2))
                break;
        } // end of for (std::vector < hdf4_attr 

        if ((modisal2flag == 2) || (modistl2flag == 2) 
            || (octsl2flag == 2) || (seawifsl2flag == 2) 
            || (czcsl2flag == 2))
            sptype = OBPGL2;

        if ((modisal3mflag == 2) || (modistl3mflag == 2) 
            || (octsl3mflag == 2) || (seawifsl3mflag == 2) 
            || (czcsl3mflag == 2))
            sptype = OBPGL3;

    } // end of if (sptype == 

    if (sptype == OTHERHDF) //Check TRMM 3A46 
    { 
        int32 array[4] = {1, 2, 180, 360};
        int len=sizeof(array)/sizeof(int32);
        for(std::vector < hdf4_var_sdfield * >::const_iterator i = sd->get_fields ().begin ();  i != sd->get_fields ().end (); ++i)
        {
            int nr=0;
            for (std::vector < hdf4_dim * >::const_iterator k = (*i)->get_dimensions ().begin ();  k != (*i)->get_dimensions ().end (); ++k)
                if(array[nr++]!=(*k)->get_size())
                    break;
                if(nr==len)
                    sptype = TRMM_3A46;
        } // end of for
    } // end of if (sptype == OTHER 

#if 0 
    if(0) // For debugging only.
        std::cout << "sptype=" << sptype << std::endl;	
#endif

}

// Obtain the absolute path from root for the given handle.
// file_id: handle to the opening file.
// sd_id: handle to SD interface.
// pobj_ref: handle to variable.
// full_path: return path.
void hdf4_file::obtain_path(int32 file_id, int32 sd_id, char *full_path, int32 pobj_ref) 
{
    int32 vgroup_cid = -1;
    int32 status = -1;
    int i = 0;
    int num_gobjects = 0;
    char cvgroup_name[VGNAMELENMAX];
    char vdata_name[VSNAMELENMAX];
    char vdata_class[VSNAMELENMAX];
    //char sds_name[H4_MAX_NC_NAME];
    int32 vdata_id = -1;
    int32 obj_tag = -1;
    int32 obj_ref = -1;
    char *cfull_path = NULL;
    //int32 dimsizes[H4_MAX_VAR_DIMS];

    cfull_path = (char *) malloc (MAX_FULL_PATH_LEN);
    if (cfull_path == NULL)
        throw std::runtime_error("No enough memory to allocate the buffer(" __FILE__ ":" TOSTRING(__LINE__)")");

    memset(cfull_path,'\0',MAX_FULL_PATH_LEN);

    vgroup_cid = Vattach (file_id, pobj_ref, "r");
    if (vgroup_cid == FAIL) 
    {
        free (cfull_path);
        throw std::runtime_error("Vattach failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
    }
    if (Vgetname (vgroup_cid, cvgroup_name) == FAIL) 
    {
        Vdetach (vgroup_cid);
        free (cfull_path);
        throw std::runtime_error("Vgetname failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
    }
    num_gobjects = Vntagrefs (vgroup_cid);
    if (num_gobjects < 0) 
    {
        Vdetach (vgroup_cid);
        free (cfull_path);
        throw std::runtime_error("Vntagrefs failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
    }


    strncpy(cfull_path,full_path,strlen(full_path));
    strncat(cfull_path,cvgroup_name,strlen(cvgroup_name));
    
    // Insert vgroup attribute mapping routine in the future. KY 2014-11-24
    strncat(cfull_path,_FORWARD_SLASH,strlen(_FORWARD_SLASH));


    // If having a vgroup "Geolocation Fields", we would like to set the EOS2Swath flag.
    std::string temp_str = std::string(cfull_path);
    if (temp_str.find("Geolocation Fields") != std::string::npos)
        if(false == this->EOS2Swathflag)
            this->EOS2Swathflag = true;

    for (i = 0; i < num_gobjects; i++) 
    {
        if (Vgettagref (vgroup_cid, i, &obj_tag, &obj_ref) == FAIL)
        {
            Vdetach (vgroup_cid);
            free (cfull_path);
            throw std::runtime_error("Vgettagref failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
        }
        if (Visvg (vgroup_cid, obj_ref) == TRUE) 
        {
            //strcpy (full_path, cfull_path);
            strncpy(full_path,cfull_path,strlen(cfull_path)+1);
            full_path[strlen(cfull_path)]='\0';
            obtain_path (file_id, sd_id, full_path, obj_ref);
        }
        else if (Visvs (vgroup_cid, obj_ref)) 
        {
            vdata_id = VSattach (file_id, obj_ref, "r");
            if (vdata_id == FAIL) 
            {
                Vdetach (vgroup_cid);
                free (cfull_path);
                throw std::runtime_error("VSattach failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }
            status = VSQueryname (vdata_id, vdata_name);
            if (status == FAIL) 
            {
                Vdetach (vgroup_cid);
                free (cfull_path);
                throw std::runtime_error("VSgetclass failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }
            status = VSgetclass (vdata_id, vdata_class);
            if (status == FAIL) 
            {
                Vdetach (vgroup_cid);
                free (cfull_path);
                throw std::runtime_error("VSgetclass failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            // Obtain the C++ string format of the path.
            std::string temp_str = std::string(cfull_path);

            // Swath 1-D is mapped to Vdata, we need to ignore them.
            // But if vdata is added to a grid, we should not ignore.
            // Since "Geolocation Fields" will always appear before 
            // the "Data Fields", we can use a flag to distinguish 
            // the swath from the grid. Swath includes both "Geolocation Fields"
            // and "Data Fields". Grid only has "Data Fields".
            // KY 2013-01-03
            bool ignore_eos2_geo_vdata = false;
            bool ignore_eos2_data_vdata = false;
            if (temp_str.find("Geolocation Fields") != std::string::npos) 
                ignore_eos2_geo_vdata = true;
            

            // Only ignore "Data Fields" vdata when "Geolocation Fields" vgroup appears.
            if (temp_str.find("Data Fields") != std::string::npos) {
                if (true == this->EOS2Swathflag){
                    ignore_eos2_data_vdata = true;
                }
            }

            if ((true == ignore_eos2_data_vdata)
                ||(true == ignore_eos2_geo_vdata)
                || VSisattr (vdata_id) == TRUE
                || !strncmp (vdata_class, _HDF_CHK_TBL_CLASS,
                            strlen (_HDF_CHK_TBL_CLASS))
                || !strncmp (vdata_class, _HDF_SDSVAR,
                            strlen (_HDF_SDSVAR))
                || !strncmp (vdata_class, _HDF_CRDVAR,
                            strlen (_HDF_CRDVAR))
                || !strncmp (vdata_class, DIM_VALS, strlen (DIM_VALS))
                || !strncmp (vdata_class, DIM_VALS01,
                            strlen (DIM_VALS01))
                || !strncmp (vdata_class, RIGATTRCLASS,
                            strlen (RIGATTRCLASS))
                || !strncmp (vdata_name, RIGATTRNAME,
                            strlen (RIGATTRNAME))) {

                status = VSdetach (vdata_id);
                if (status == FAIL) {
                    Vdetach (vgroup_cid);
                    free (cfull_path);
                    throw std::runtime_error("VSdetach failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                }
            }

            else 
            {
                VDATA *vdataobj = VDATA::read (vdata_id, obj_ref);

                //vdataobj->newname = vdataobj->name + cfull_path; 
                //char *temp_path = (char *) malloc (MAX_FULL_PATH_LEN);
                //strcpy(temp_path, cfull_path);
                //strcat(temp_path, "/");
                //vdataobj->newname = temp_path + vdataobj->name; 
                //free(temp_path); //[04/16/2012 LD]
                vdataobj->newname = cfull_path + vdataobj->name; 

                for (std::vector < hdf4_var_vdfield * >::const_iterator it_vdf =  vdataobj->get_fields ().begin (); it_vdf != vdataobj->get_fields ().end (); it_vdf++) 		
                {
                    (*it_vdf)->newname = "vdata" + vdataobj->newname +"_vdf_" +  (*it_vdf)->m_name ;
                } // end of for (std::vector

                this->vds.push_back (vdataobj);
            }
        }
        else if (obj_tag == DFTAG_NDG || obj_tag == DFTAG_SDG || obj_tag == DFTAG_SD)
        {

            std::string temp_str = std::string(cfull_path);
                    
            if  (sd->refindexlist.find (obj_ref) != sd->refindexlist.end ())
            { 
                //char *temp_path = (char *) malloc (MAX_FULL_PATH_LEN);
                //strcpy(temp_path, cfull_path);
                //strcat(temp_path, "/");
                //sd->sdfields[sd->refindexlist[obj_ref]]->newname = temp_path + sd->sdfields[this->sd->refindexlist[obj_ref]]->m_name; 
                //free(temp_path); //[04/16/2012 LD]
                sd->sdfields[sd->refindexlist[obj_ref]]->newname = cfull_path + sd->sdfields[this->sd->refindexlist[obj_ref]]->m_name; 
            } // end of if (sd-> 
        } // end  of else if (obj_tag == DFTAG_NDG
    } // end of for (i = 0; i < num_ 
    status = Vdetach (vgroup_cid);
    if (status == FAIL) 
    {
        free (cfull_path);
        throw std::runtime_error("Vdetach failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
    }
    free (cfull_path);
}

// Set the variable's newname.
// sd_id: handle to SD interface.
// file_id: handle to the openingfile.
void hdf4_file::insert_orig_field_path(int32 sd_id, int32 file_id) 
{
    int32 status_32 = -1;
    int32 vgroup_id = -1;
    int32 lone_vg_number = -1;
    int32 num_of_lones = 0; 
    int32 *ref_array = NULL; //buffer to hold the ref numbers of lone vgroups   
    int32 num_gobjects = 0;
    int32 obj_ref = -1;
    int32 obj_tag = -1;
    int i = 0;
    //int32 dim_sizes[H4_MAX_VAR_DIMS];

    char vdata_name[VSNAMELENMAX];
    char vdata_class[VSNAMELENMAX];

    char vgroup_name[VGNAMELENMAX];
    char vgroup_class[VGNAMELENMAX];

    char *full_path = NULL;
    char *cfull_path = NULL;
    int32 vdata_id = -1;

    num_of_lones = Vlone (file_id, NULL, 0);
    if (num_of_lones == FAIL)
        throw std::runtime_error("Fail to obtain lone vgroup number.(" __FILE__ ":" TOSTRING(__LINE__)")");

#if 0
    if(0) // For debugging only.
        std::cout << "Vlone=" << num_of_lones << std::endl;
#endif

    if (num_of_lones > 0) 
    {
        ref_array = (int32 *) malloc (sizeof (int32) * num_of_lones);
        if (ref_array == NULL)
            throw std::runtime_error("No enough memory to allocate buffer.(" __FILE__ ":" TOSTRING(__LINE__)")");
        num_of_lones = Vlone (file_id, ref_array, num_of_lones);
        if (num_of_lones == FAIL) 
        {
            free (ref_array);
            throw std::runtime_error("Cannot obtain lone vgroup reference arrays.(" __FILE__ ":" TOSTRING(__LINE__)")");
        }

        for (lone_vg_number = 0; lone_vg_number < num_of_lones; lone_vg_number++) 
        {
            // Attach to the current vgroup 
            vgroup_id = Vattach (file_id, ref_array[lone_vg_number], "r");
            if (vgroup_id == FAIL) 
            {
                free (ref_array);
                throw std::runtime_error("Vattach failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }
            status_32 = Vgetname (vgroup_id, vgroup_name);
            if (status_32 == FAIL) 
            {
                Vdetach (vgroup_id);
                free (ref_array);
                throw std::runtime_error("Vgetname failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }
            status_32 = Vgetclass (vgroup_id, vgroup_class);
            if (status_32 == FAIL) 
            {
                Vdetach (vgroup_id);
                free (ref_array);
                throw std::runtime_error("Vgetclass failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            //Ignore internal HDF groups
            if (strcmp (vgroup_class, _HDF_ATTRIBUTE) == 0
                || strcmp (vgroup_class, _HDF_VARIABLE) == 0
                || strcmp (vgroup_class, _HDF_DIMENSION) == 0
                || strcmp (vgroup_class, _HDF_UDIMENSION) == 0
                || strcmp (vgroup_class, _HDF_CDF) == 0
                || strcmp (vgroup_class, GR_NAME) == 0
                || strcmp (vgroup_class, RI_NAME) == 0) 
            {
                Vdetach(vgroup_id);
                continue;
            }
            num_gobjects = Vntagrefs (vgroup_id);
            if (num_gobjects < 0) 
            {
                Vdetach (vgroup_id);
                free (ref_array);
                throw std::runtime_error("Vntagrefs failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }
            full_path = NULL;
            full_path = (char *) malloc (MAX_FULL_PATH_LEN);
            if (full_path == NULL) 
            {
                Vdetach (vgroup_id);
                free (ref_array);
                throw std::runtime_error("No enough memory to allocate the buffer.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            memset(full_path,'\0',MAX_FULL_PATH_LEN);

            strncpy(full_path,_FORWARD_SLASH,strlen(_FORWARD_SLASH));
            strncat(full_path,vgroup_name,strlen(vgroup_name));

            // Insert vgroup attribute mapping in the future (HFRCFLIB-114)
            strncat(full_path,_FORWARD_SLASH,strlen(_FORWARD_SLASH));

            cfull_path = NULL;
            cfull_path = (char *) malloc (MAX_FULL_PATH_LEN);
            if (cfull_path == NULL) 
            {
                Vdetach (vgroup_id);
                free (ref_array);
                free (full_path);
                throw std::runtime_error("No enough memory to allocate the buffer.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }
            memset(cfull_path,'\0',MAX_FULL_PATH_LEN);
            strncpy (cfull_path, full_path,strlen(full_path));

            // Loop all vgroup objects
            for (i = 0; i < num_gobjects; i++) 
            {
                if (Vgettagref (vgroup_id, i, &obj_tag, &obj_ref) == FAIL) 
                {
                    Vdetach (vgroup_id);
                    free (ref_array);
                    free (full_path);
                    free (cfull_path);
                    throw std::runtime_error("Vgettagref failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                } // end of if

                if (Visvg (vgroup_id, obj_ref) == TRUE) 
                {
                    //strcpy (full_path, cfull_path);
                    strncpy (full_path, cfull_path,strlen(cfull_path)+1);
                    full_path[strlen(cfull_path)]='\0';
                    obtain_path (file_id, sd_id, full_path, obj_ref);
                }
                else if (Visvs (vgroup_id, obj_ref)) 
                {
                    vdata_id = VSattach (file_id, obj_ref, "r");
                    if (vdata_id == FAIL) 
                    {
                        Vdetach (vgroup_id);
                        free (ref_array);
                        free (full_path);
                        free (cfull_path);
                        throw std::runtime_error("VSattach failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                    } // end of if
                    status_32 = VSgetname (vdata_id, vdata_name);
                    if (status_32 == FAIL) 
                    {
                        Vdetach (vgroup_id);
                        free (ref_array);
                        free (full_path);
                        free (cfull_path);
                        throw std::runtime_error("VSgetclass failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                    } // end of if
                    status_32 = VSgetclass (vdata_id, vdata_class);
                    if (status_32 == FAIL) 
                    {
                        Vdetach (vgroup_id);
                        free (ref_array);
                        free (full_path);
                        free (cfull_path);
                        throw std::runtime_error("VSgetclass failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                    } // end of if

                    // Obtain the C++ string format of the path.
                    std::string temp_str = std::string(cfull_path);

                    // Swath 1-D is mapped to Vdata, we need to ignore the mapping from vdata to CF.
                    // But if vdata is added to a grid, we should not ignore.
                    // Since "Geolocation Fields" will always appear before 
                    // the "Data Fields", we can use a flag to distinguish 
                    // the swath from the grid. Swath includes both "Geolocation Fields"
                    // and "Data Fields". Grid only has "Data Fields".
                    // KY 2013-01-03
                    bool ignore_eos2_geo_vdata = false;
                    bool ignore_eos2_data_vdata = false;
                    if (temp_str.find("Geolocation Fields") != std::string::npos) {
                        if(false == this->EOS2Swathflag)
                            this->EOS2Swathflag = true;
                        ignore_eos2_geo_vdata = true;
                    }

                    // Only ignore "Data Fields" vdata when "Geolocation Fields" appears.
                    if (temp_str.find("Data Fields") != std::string::npos) {
                        if (true == this->EOS2Swathflag)
                            ignore_eos2_data_vdata = true;
                    }

                    // Handle vdata 
                    if ((true == ignore_eos2_data_vdata)
                        ||(true == ignore_eos2_geo_vdata)
                        || VSisattr (vdata_id) == TRUE
                        || !strncmp (vdata_class, _HDF_CHK_TBL_CLASS, strlen (_HDF_CHK_TBL_CLASS))
                        || !strncmp (vdata_class, _HDF_SDSVAR, strlen (_HDF_SDSVAR))
                        || !strncmp (vdata_class, _HDF_CRDVAR, strlen (_HDF_CRDVAR))
                        || !strncmp (vdata_class, DIM_VALS, strlen (DIM_VALS))
                        || !strncmp (vdata_class, DIM_VALS01, strlen (DIM_VALS01))
                        || !strncmp (vdata_class, RIGATTRCLASS, strlen (RIGATTRCLASS))
                        || !strncmp (vdata_name, RIGATTRNAME, strlen (RIGATTRNAME))) 
                    {
                        status_32 = VSdetach (vdata_id);
                        if (status_32 == FAIL) 
                        {
                            Vdetach (vgroup_id);
                            free (ref_array);
                            free (full_path);
                            free (cfull_path);
                            throw std::runtime_error("VSdetach failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                        } // end of if
                    } // end of if (VSisattr (vdata_id 
                    else 
                    {
                        VDATA *vdataobj = VDATA::read (vdata_id, obj_ref);
                        vdataobj->newname = full_path + vdataobj->name; 

#if 0
                        if(0) // For debugging only.
                            std::cout << "vdata new name=" << vdataobj->newname << std::endl;
#endif

                        for (std::vector <hdf4_var_vdfield * >::const_iterator it_vdf = vdataobj->get_fields ().begin (); it_vdf != vdataobj->get_fields ().end (); it_vdf++) 
                        {
                            (*it_vdf)->newname = "vdata" + vdataobj->newname +"_vdf_" + (*it_vdf)->m_name;
                        } // end of for

                        vds.push_back (vdataobj);
                        // Here we should detach the vdata id since vdata id is detached in the destructor.
                        // Detaching the vdata here will make the data reading fail.
                        //status_32 = VSdetach (vdata_id);
                        #if 0
                        if (status_32 == FAIL) 
                        {
                            Vdetach (vgroup_id);
                            free (ref_array);
                            free (full_path);
                            free (cfull_path);
                            throw std::runtime_error("VSdetach failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
                        } // end of if (status_32 ==
                        #endif
                    } // end of else
                } // ene of else if (Visvs (vgroup_id, ob

                // These are SDS objects
                else if (obj_tag == DFTAG_NDG || obj_tag == DFTAG_SDG || obj_tag == DFTAG_SD) 
                {
                    // We need to obtain the SDS index; also need to store the new name(object name + full_path).
                    std::string temp_str = std::string(full_path);

                    if (((temp_str.find("Data Fields") == std::string::npos) && 
                        (temp_str.find("Geolocation Fields") == std::string::npos))
                        && 
                        (sd->refindexlist.find (obj_ref) != sd->refindexlist.end ()))
                    {
                        sd->sdfields[sd->refindexlist[obj_ref]]->newname = full_path + sd->sdfields[sd->refindexlist[obj_ref]]->m_name;
				
#if 0
			if(0) // For debugging only.
                            std::cout << "sds new name=" << sd->sdfields[sd->refindexlist[obj_ref]]->newname << std::endl;
#endif
                    } // end of if (sd->refindexlist.find 
                } // end of else if (obj_tag == DFTAG_ND
                else;
            } //end of for (i = 0; i < num_

            free (full_path);
            free (cfull_path);

            status_32 = Vdetach (vgroup_id);
            if (status_32 == FAIL) 
            {
                free (ref_array);
                throw std::runtime_error("Vdetach failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }
        } // end of for (lone_vg_number = 0	
        if (ref_array != NULL)
            free (ref_array);
    } // end of if (num_of_lones > 0) 
}

//MERRA level 3 products put the dimension variables under both the grid and the root level.
// For example, you may find the variable name XDim under both the grid(named EOSGRID) and 
// top level. The top level XDim includes key CF attributes such as
// units. The grid level XDim doesn't have such a unit. So we need to keep the top level XDim. 
void hdf4_file::remove_merra_l3_dup_vars()  {

    hdf4_file* file = this;

    // 1.  Loop through SDS fields and remove suffixes(:???) of the dimension names and the variable names. 
    //     This will effectively remove EOSGRID from the name XDim:EOSGRID.
    for (std::vector < hdf4_var_sdfield * >::const_iterator i =
            file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {

        std::string tempname = (*i)->get_name();
        size_t found_colon = tempname.find_first_of(':');
        if(found_colon!=std::string::npos) {
            (*i)->rename(tempname.substr(0,found_colon));
            (*i)->newname = (*i)->get_name();
        }

        for (std::vector < hdf4_dim * >::const_iterator k =
            (*i)->get_dimensions ().begin ();
            k != (*i)->get_dimensions ().end (); ++k) {

            //Must obtain orig name to find : in the XDim:???
            std::string tempname = (*k)->get_orig_name();
            size_t found_colon = tempname.find_first_of(':');
            if(found_colon!=std::string::npos)
                (*k)->rename(tempname.substr(0,found_colon));

        }

    }

    std::set<std::string>full_var_name_set;
    std::set<std::string>clash_var_name_set;

    // Save the clashed variable names to set clash_var_name_set,we only need to consider the case when rank = 1
    for (std::vector < hdf4_var_sdfield * >::const_iterator i =
            file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {
        if((*i)->get_rank() == 1) {
            std::pair < std::set < std::string >::iterator, bool > ret;
            // You should find two XDim etc..., this is how the clashed variable is found.
            ret = full_var_name_set.insert ((*i)->get_name ());
            if (ret.second == false)
                clash_var_name_set.insert((*i)->get_name());
        }
 
    }

    // Remove the duplicate variable under the grid EOSGrid,the newname should have EOSGrid.
    for (std::vector < hdf4_var_sdfield * >::iterator i =
            file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {
        if((*i)->get_rank() == 1) {
            if(clash_var_name_set.find((*i)->get_name())!=clash_var_name_set.end()) {
                if((*i)->get_new_name().find("EOSGRID")!=std::string::npos) {
                    delete(*i);
                    file->sd->sdfields.erase(i);
                    i--;
                }
            }
        }
    }
}


} //end of namespace
