/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Category: Header file for HDF4 namespace

Description:

This file includes definition of an HDF4 file class. It also includes file-level related method declarations.


*****************************************************************************/

#ifndef HDF4_FILE_H
#define HDF4_FILE_H
#include <string>
#include <iostream>
#include <sstream>
#include "hdf.h"
#include "mfhdf.h"
#include "eoslib_types.h"
#include "eoslib_defs.h"
#include "hdf4_types.h"
#include "hdf4_attr.h"
#include "hdf4_dim.h"
#include "hdf4_filterset.h"
#include "hdf4_filterqueue.h"
#include "hdf4_sds.h"
#include "hdf4_sd.h"
#include "hdf4_vdata.h"

namespace hdf4 {

using namespace eoslib;

class hdf4_file 
{
    public:
        // Default constructor.
        hdf4_file();

        // Default destructor.
        ~hdf4_file();

        // Main interface to read HDF4.
        void read(const std::string&);

        // Check if this is an special EOS file that only needs to be handled by HDF4 APIs
        bool check_special_eos_no_filters(const std::string&) ;

        // Handle Special EOS files that only needs to be handled by HDF4 APIs
        // This routine will bypass all the filters.
        void handle_special_eos_no_filters() ;


        // Check type and apply filters.
        void apply(bool) ;

        // Get vars, dims, and attrs for ncconverter.
        std::list<var*> get_vars();

        // Get dimensions.
        std::map<std::string, int> get_dims();

    private:
        // Read all SDS objects.
        void read(int32, int32) ;

        // Read all SDS objects and save them in group for hybrid file.
        void read(int32, int32 , const group*); 
 
        // Read vdata objects.
        void read(int32);

        // Read vdata and save them in group for hybrid file.
        void read(int32, const group*);

        // Check if the data product is CERES.
        void check_ceres(const char*,  int32, int32*);

        // Check if the data product is TRMM or OBPG.
        void check_sd_type();

        // Obtain the full path of an HDF4 object.
        void obtain_path(int32, int32, char*, int32);

        // Obtain the full path of SDS and vdata.
        void insert_orig_field_path(int32, int32);

        // Remove MERRA duplicate variables
        void remove_merra_l3_dup_vars();
    public:
        // Get pointer to SD holding all SDS objects.
        const SD* get_sd() const
        {
            return sd;
        }

        // Get VDATA variables.
        const std::vector<VDATA*>& get_vds() const
        {
            return vds;
        }

        // Get the opening file name.
        const std::string get_file_name() const
        {
            return filename;
        }

    public:
        // Product type.
        SPType getSPType() const
        { return this->sptype;
        }

        // Store SDS variables.
        SD *sd;

        // Store VDATA variables.
        std::vector< VDATA * > vds;

        // HDF4 file name.
        std::string filename;

        // Get handle to SD.
        int32 get_sd_id() const { return sdfd; }

        // Get handle to file.
        int32 get_file_id() const { return fileid; }

    private:
        // SD handle.
        int32 sdfd;

        // Product type.
        SPType sptype;


        // File handle.
        int32 fileid;

        //A flag that indicates this is an HDF-EOS2 swath for a hybrid HDF-EOS2 file
        bool EOS2Swathflag;
};
} //end of namespae

#endif
