/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of HDF4 files.


*****************************************************************************/

#include <iostream>
#include <cstring>
#include <string>
#include <vector>
#include <set>
#include <map>
#include "misc_util.h"
#include "hdf4_sds.h"
#include "hdf4_vdata_field.h"
#include "hdf4_dim.h"
#include "hdf4_filterset.h"
#include "hdf4_file.h"

using namespace eoslib;

extern hdf4::hdf4_file *file;
extern bool bDumpData;

namespace hdf4 {

extern void generate_dim_names();
extern bool check_cv(hdf4_var_sdfield*);

extern void Handle_NameClashing(std::vector<std::string>&);

std::string tempdimname ="";
std::pair < std::set < std::string >::iterator, bool > ret;
std::string temppath="";
std::set < int32 > fakedimsizeset;
std::pair < std::set < int32 >::iterator, bool > fakedimsizeit;
std::map < int32, std::string > fakedimsizenamelist;
std::map < int32, std::string >::iterator fakedimsizenamelistit;

//bool nameclashflag = false;

// Remove non printable characters, 
std::string remove_non_printable_chars(const std::string& oldname)
{
    std::string newname = oldname;
    std::string::iterator it;
    for(it = newname.begin(); it != newname.end(); it++)
    {
        char c = (*it);
        if(!(c>31 && c<127)) //Printable ASCII characters only.
            (*it) = ' ';
    }	
    return newname;
}


// Remove special characters which are not CF-compliant, and  "_" will be prepanded if a name statrs with numeric characters. 
std::string remove_special_chars(const std::string& oldname)
{
    std::string newname = oldname;
    std::string::iterator it;
    for(it = newname.begin(); it != newname.end(); it++)
    {
        if('0'<= *it && *it <= '9')
            ; // numeric
        else if('A'<= *it && *it <= 'Z')
            ; // capital alpha
        else if('a'<= *it && *it <= 'z')
            ; // lower-case alpha
        else if(*it == '_')
            ; // underscore
        else
            *it = '_';
    }

    //Collapse multiple _.
    int k=0;
    while(k<newname.length()-1)
    {
        if(newname[k]=='_' && newname[k+1]=='_')
            newname.erase(k+1, 1);
        else 
            k++;
    } // end of while

    if(isdigit(newname[0]))
        newname= "_" + newname;

    return newname;
}

// This routine will remove the first '/' from the path since 
// it will be changed to '_', which is redundant for the final name representation.
std::string get_cf_string(const std::string& oldname) {

    std::string name_no_front_slash;
    if ((oldname.size() >1) && ('/' == oldname[0]))
        name_no_front_slash = oldname.substr(1);
    else
        name_no_front_slash = oldname;
    return remove_special_chars(name_no_front_slash);
}

// Add corrected dimensions to SDS variables.
void build_dim_name_list()
{
    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_dimensions ().begin (); j != (*i)->get_dimensions ().end (); ++j) 
        {
            //May treat corrected dimension names as the original dimension names the SAME, CORRECT it in the future.
            if (file->getSPType() != OTHERHDF)
                tempdimname = (*j)->get_name ();
            else
                tempdimname = (*j)->get_name () + temppath;

            hdf4_dim *dim = new hdf4_dim (tempdimname, tempdimname,(*j)->get_size (), (*j)->get_type ());
            (*i)->correcteddims.push_back (dim);
            if (tempdimname.find ("fakeDim") != std::string::npos) 
            {
                fakedimsizeit = fakedimsizeset.insert((*j)->get_size());
                if (fakedimsizeit.second == true) 
                {
                    fakedimsizenamelist[(*j)->get_size ()] = (*j)->get_name ();	//Here we just need the original name since fakeDim is globally generated.
                }
            } // end of if
        } // end of for
    } // end of for
}

// Squeeze fake dimensions with the same size.
void squeeze_fake_dim_names()
{
    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j) 
        {
            if ((*j)->get_name ().find ("fakeDim") != std::string::npos) 
            {
                if (fakedimsizenamelist.find ((*j)->get_size ()) != fakedimsizenamelist.end ()) 
                {
                    //sequeeze the redundant fakeDim with the same size
                    (*j)->rename(fakedimsizenamelist[(*j)->get_size ()]);
                }
                else 
                {
                    std::cout << "The fakeDim name " << (*j)->get_name () << "with the size " <<  (*j)->get_size () << " does not in the fakedimsize list." << std::endl;
                    throw std::runtime_error("Fail to squeeze the fake dimension names.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    
                } // end of else
            } // end of if ((*j
        } // end of for
    } // end of for
}

// Create full dimension name list.
void generate_new_dim_name_set()
{
    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j) 
        {
            std::pair < std::set < std::string >::iterator, bool > temp_ret;
            temp_ret = file->sd->fulldimnamelist.insert ((*j)->get_name ());
            // Map from the unique dimension name to its size
            if (temp_ret.second == true) 
            {
                file->sd->n1dimnamelist[(*j)->get_name ()] = (*j)->get_size ();
            }
        } // end of for
    } // end of for
}

// Add missing coordinate varialbes. They have index values begining with 0.
void add_missing_cv()
{

    // Don't handle OTHERHDF case.
    if(OTHERHDF == file->getSPType()) 
        return;
    for (std::map < std::string, int32 >::const_iterator i = file->sd->n1dimnamelist.begin (); i != file->sd->n1dimnamelist.end (); ++i) 
    {
        if (file->sd->nonmisscvdimnamelist.find ((*i).first) == file->sd->nonmisscvdimnamelist.end ()) 
        {	
            // Create a missing Z-dimension field  
            hdf4_var_sdfield *missingfield = new hdf4_var_sdfield ();

            missingfield->type = DFNT_INT32;
            missingfield->rename((*i).first);
            missingfield->newname = (*i).first;
            missingfield->rank = 1;
            missingfield->fieldtype = 4;
            hdf4_dim *dim = new hdf4_dim ((*i).first, (*i).first,(*i).second, 0);

            missingfield->dims.push_back (dim);
            dim = new hdf4_dim ((*i).first, (*i).first,(*i).second, 0);
            missingfield->correcteddims.push_back (dim);

            std::vector<char> v1;

            // str_value is for dumping the value in string.(Not used by the conversion toolkit.For internal dumper routine)
            std::string s;
            int32 len = (*i).second;	
            for(int k=0; k<len; ++k)
            {	
                std::stringstream ss;
                ss << k; //k+1;
                s.append(ss.str());

                size_t t4 = v1.size();
                v1.resize(t4+4);
                int32 t3 = k; //k+1;
                memcpy((char*)&(v1)[t4], &t3, sizeof(t3));

                if(k!=len-1)
                    s.append(",");		
            } // end of for
            missingfield->str_value = s;
            missingfield->value.resize(v1.size());
            std::copy(v1.begin(), v1.end(), missingfield->value.begin());
            file->sd->sdfields.push_back (missingfield);
        } // end of if (file->sd->nonmissc 
    } // end of for 
}

// All new names are appended with full path, and they are the final names used later. 
// Since NASA does not like this way, we will use the original sds names  for SDS that doesn't have the name clashing. 
void check_name_clashing()
{

    std::set < std::string > fullobjnameset;
    bool has_name_clashing = false;
    for (int i = 0; i < file->sd->sdfields.size (); ++i) 
    {
        ret = fullobjnameset.insert (file->sd->sdfields[i]->get_new_name());//newname);
        if (false == ret.second) 
            has_name_clashing = true;
    } // end of for

    if (true == has_name_clashing) {

        std::vector<std::string> sds_names;
        for (int i = 0; i < file->sd->sdfields.size (); ++i)
            sds_names.push_back(file->sd->sdfields[i]->get_new_name());
        Handle_NameClashing(sds_names);

        // Move the non-clashing names back to the sds's new name.
        for (int i = 0; i < file->sd->sdfields.size (); ++i)
            file->sd->sdfields[i]->newname = sds_names[i];
    }

    fullobjnameset.clear();
    has_name_clashing = false;
    for(std::vector<VDATA *>::const_iterator i=file->vds.begin(); i!=file->vds.end(); i++)
    {
        for(std::vector<hdf4_var_vdfield *>::const_iterator j=(*i)->get_fields().begin(); j!=(*i)->get_fields().end(); j++)
        {	
            ret = fullobjnameset.insert ((*j)->get_new_name()); 
            if(false == ret.second )
                has_name_clashing = true;
        } // end of for
    } // end of for

    if (true == has_name_clashing) {

        std::vector<std::string> vdata_fnames;

        for(std::vector<VDATA *>::const_iterator i=file->vds.begin(); i!=file->vds.end(); i++)
            for(std::vector<hdf4_var_vdfield *>::const_iterator j=(*i)->get_fields().begin(); j!=(*i)->get_fields().end(); j++)
                vdata_fnames.push_back((*j)->get_new_name());
     
        Handle_NameClashing(vdata_fnames);

        // Move the non-clashing names back to the vdata field's new name.
        int k = 0;
        for(std::vector<VDATA *>::const_iterator i=file->vds.begin(); i!=file->vds.end(); i++) {
            for(std::vector<hdf4_var_vdfield *>::const_iterator j=(*i)->get_fields().begin(); j!=(*i)->get_fields().end(); j++) {
                (*j)->newname = vdata_fnames[k];
                k++;
            }
        }
    }
}

// Remove special characters in SDS.
void handle_special_chars_of_sds()
{

    std::string temp1name;
    std::string temp2name;
    std::string tempdim1name;
    std::string tempdim2name;

    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        if ((*i)->fieldtype != 0) 
        {
            if ((*i)->fieldtype == 1 || (*i)->fieldtype == 2) 
            {
                (*i)->newname = get_cf_string((*i)->newname);

                if ((*i)->get_rank () > 2)
                {
                    std::cout << "The lat/lon rank in " << (*i)->get_name() << " should NOT be greater than 2." << std::endl;
                    throw std::runtime_error("lat/lon rank should be 1 or 2.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                }
                else if ((*i)->get_rank () == 2) 
                {
                    // Each lat/lon must be 2-D under the same group.
                    for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j) 
                    {
                        std::map < std::string, std::string >::iterator tempmapit;
                        tempmapit = file->sd->dimcvarlist.find ((*j)->get_name ());
                        if (tempmapit == file->sd->dimcvarlist.end ()) 
                        {
                            file->sd->dimcvarlist[(*j)->get_name()] = (*i)->newname;
                            break;
                        }
                    } // end of for
                } // end of else if ((*i)
                else 
                {			
                    // When rank = 1, must follow COARD conventions. 
                    // Here we don't check name clashing for the performance's reason, the chance of clashing is very,very rare.
                    (*i)->newname = (*i)->get_corrected_dimensions ()[0]->get_name ();
                    file->sd->dimcvarlist[(*i)->get_corrected_dimensions ()[0]->get_name ()] = (*i)->newname;
                } // end of else
            } // end of if ((*i)->fieldtype == 1
        } // end of if ((*i)->fieldtype != 0) 
        else 
        {
            (*i)->newname = get_cf_string((*i)->newname);
            
        } // end of else
    } // end of for (std::vector < hdf4_

    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        // set a flag
        if ((*i)->fieldtype != 0) 
        {
            if ((*i)->fieldtype != 1 && (*i)->fieldtype != 2) 
            {	
                // "Missing" coordinate variables or coordinate variables having dimensional scale data
                if ((*i)->get_rank () > 1)
                {
                    std::cout << "The lat/lon in "  << (*i)->get_name() << "rank should be 1." << std::endl;
                    throw std::runtime_error("The lat/lon rank should be 1.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    //exit(1);
                }

                (*i)->newname = (*i)->get_corrected_dimensions ()[0]->get_name ();
                file->sd->dimcvarlist[(*i)->get_corrected_dimensions ()[0]->get_name ()] = (*i)->newname;
            } // end of if ((*i)->fieldtype != 1 
        } // end of if ((*i)->fieldtype != 0)
    } // end of for

}

// For OBPG L3 products, add "scale_factor" and "add_offset" attributes. They copy values of file attributes of Slope and Intercept. 
void add_attrs_for_obpg_l3m()
{
    if(file->getSPType() != OBPGL3)
        	return;

    bool scaling_function = false; // True if Scaling Equation is available.
    bool base = false; // True if Base is available.

    float32 slope = 0;
    float32 intercept = 0;
 
    for(std::vector<hdf4_attr *>::const_iterator i=file->sd->get_attrs().begin(); i!=file->sd->get_attrs().end(); i++)
    {
        if((*i)->get_name()=="Scaling Equation")
            scaling_function = true;
        if((*i)->get_name()=="Base")
            base = true;
        if((*i)->get_name()=="Slope")
            slope = *((float32*)&((*i)->get_value()[0]));
        if((*i)->get_name()=="Intercept")
            intercept = *((float32*)&((*i)->get_value()[0]));
    } // end of for

    if(scaling_function && !base) //The Scaling Equation is linear. Add "scale_factor" and "add_offset" attributes to each data field.
    {
        for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i)
        {
            if((*i)->fieldtype!=0)
            continue;

            if(!(*i)->has_attr("scale_factor"))
            {
                std::vector<char> fv(4);
                memcpy((char*)&(fv)[0], &slope, sizeof(slope));
                hdf4_attr *myattr = new hdf4_attr("scale_factor", DFNT_FLOAT32, 1, fv);
                (*i)->attrs.push_back(myattr);
            } // end of if
            if(!(*i)->has_attr("add_offset"))
            {
                std::vector<char> fv(4);
                memcpy((char*)&(fv)[0], &intercept, sizeof(intercept));
                hdf4_attr *myattr = new hdf4_attr("add_offset", DFNT_FLOAT32, 1, fv);
                (*i)->attrs.push_back(myattr);
            } // end of if
        } // end of for (std::vector < hdf4_
    } // end of if(scaling_function
}

// Add "origname" attribute.
void add_origname_attr()
{
    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i)
    {
        if(!(*i)->has_attr("origname"))
        {
            (*i)->add_attr_one_str("origname", (*i)->newname);
            //(*i)->add_attr_one_str("origname", (*i)->get_name());
        }	
    } // end of for

    for(std::vector<VDATA *>::const_iterator i=file->vds.begin(); i!=file->vds.end(); i++)
    {
        for(std::vector<hdf4_var_vdfield *>::const_iterator j=(*i)->get_fields().begin(); j!=(*i)->get_fields().end(); j++)
        {
            if(!(*j)->has_attr("origname")) {
                //(*j)->add_attr_one_str("origname", (*j)->newname);
                (*j)->add_attr_one_str("origname", (*j)->get_name());
            }
        } // end of for
    } // end of for
}

// Add "longname" attribute.
void add_longname_attr()
{
    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i)
    {
        if(!(*i)->has_attr("long_name"))
        {
            (*i)->add_attr_one_str("long_name", (*i)->newname); //(*i)->get_name());
        }
    } // end of for
}

// Add "coordinates" attribute to all varialbes, and "units" and "standard_name" attributes to coordinate variables.
void add_coordinates_attr()
{
    generate_dim_names();

    // We will check if this breaks OTHERHDF products. KY 2013-02-22
    if(OTHERHDF == file->getSPType())
        return;

    int tempcount = 0;
    std::map < std::string, std::string >::iterator tempmapit;

    std::string tempcoordinates;
    std::string tempfieldname;
    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        if((*i)->fieldtype!=1 && (*i)->fieldtype!=2)
            if(!check_cv((*i)))
            {
                // For debugging only.
                continue;
            } // end of if	
	
        tempcount = 0;
        tempcoordinates = "";
        tempfieldname = "";

        for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j)
        {
            tempmapit = (file->sd->dimcvarlist).find ((*j)->get_name ());
            if (tempmapit != (file->sd->dimcvarlist).end ())
                tempfieldname = tempmapit->second;
            else
            {
                std::cout << "The dimension with the name " << (*j)->get_name () << " must have corresponding coordinate variable." << std::endl;
                throw std::runtime_error("The dimension name doesn't have the corresponding coordinate variable.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                //exit(1);
            } // end of else

            if (tempcount == 0)
                tempcoordinates = tempfieldname;
            else
                tempcoordinates = tempcoordinates + " " + tempfieldname;
            tempcount++;
        } // end of for
        if(tempcoordinates.length()!=0)
            (*i)->add_attr_one_str("coordinates", tempcoordinates);

        // Add "units" attribute for latitude and longitude
        if ((*i)->fieldtype == 1) // Latitude. 
        {	
            // Add "units" attribute.
            if((*i)->has_attr("units"))
                (*i)->remove_attr_by_name("units");
            (*i)->add_attr_one_str("units", "degrees_north");

            // Add "standard_name" attribute.
            if((*i)->has_attr("standard_name"))
                (*i)->remove_attr_by_name("standard_name");
            (*i)->add_attr_one_str("standard_name", "Latitude");
        } // end of if ((*i)->

        if ((*i)->fieldtype == 2) // Longitude.
        {	
            // Add "units" attribute.
            if((*i)->has_attr("units"))
                (*i)->remove_attr_by_name("units");
            (*i)->add_attr_one_str("units", "degrees_east");

            // Add "standard_name" attribute.
            if((*i)->has_attr("standard_name"))
                (*i)->remove_attr_by_name("standard_name");
            (*i)->add_attr_one_str("standard_name", "Longitude");
        } // end of if

        // Add units for Z-dimension, now it is always "level"
        if ((*i)->fieldtype == 4)
       { 
            std::string tempunits = "level";
            (*i)->add_attr_one_str("units", tempunits);
        } // end of if
    } // end of for  
}

// Remove special characters in VDATA newname.
void handle_special_chars_of_vdata()
{
    std::string temp1name;
    std::string temp2name;

    for (std::vector < VDATA * >::const_iterator i = file->vds.begin (); i != file->vds.end (); ++i) 
    {
        temp1name = (*i)->get_new_name();
        temp2name = remove_special_chars(temp1name);
        (*i)->set_new_name(temp2name);

        // We also need to handle special characters for vdata field names
        for (std::vector < hdf4_var_vdfield * >::const_iterator j = (*i)->get_fields().begin (); j != (*i)->get_fields().end (); ++j) 
        {
            (*j)->newname = remove_special_chars((*j)->newname);
        } // end of for
    } // end of for
}

//Remove special characters for SDS attribute names.
void handle_special_chars_of_sds_attrs()
{

    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i)
    {  
        for(std::vector<hdf4_attr *>::const_iterator j = (*i)->get_attributes().begin(); j!=(*i)->get_attributes().end(); j++)
            (*j)->rename(remove_special_chars((*j)->get_name()));
    }
}

// Remove special characters for file attribute names.
void handle_special_chars_of_file_attrs()
{
    for(std::vector<hdf4_attr *>::const_iterator i=file->sd->get_attrs().begin(); i!=file->sd->get_attrs().end(); i++)
    {
        (*i)->rename(remove_special_chars((*i)->get_name()));
    }
}

// Get string representation of data type.
std::string get_cdl_type_str(value_type_t type)
{
    switch(type)
    {
#define HANDLE(type, cdltype) \
    case DFNT_##type: \
        return #cdltype;

        HANDLE(CHAR8,   char);
        HANDLE(INT8,    byte);
        HANDLE(INT16,   short);
        HANDLE(INT32,   int);
        HANDLE(UCHAR8,  char);
        HANDLE(UINT8,   ubyte);
        HANDLE(UINT16,  ushort);
        HANDLE(UINT32,  uint);
        HANDLE(FLOAT32, float);
        HANDLE(FLOAT64, double);

#undef HANDLE
        default:
            throw std::range_error("Unknown type(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }; // end of switch
}

#if 0
// Output CDL file. For internal testing only.
/*
void dump_cdl()
{
    std::cout << "netCDF " << file->filename << " {" << std::endl;
    std::cout << "dimensions: " << std::endl;
    std::cout << "    ";
    int size = (int)file->sd->fulldimnamelist.size(), nr=0;

    for(std::set<std::string>::const_iterator i=file->sd->fulldimnamelist.begin(); i!=file->sd->fulldimnamelist.end(); ++i)
    {
        ++nr;
        int32 dimsize = file->sd->n1dimnamelist[*i];
        std::cout << *i << " = " << dimsize;
        if(nr!=size)
            std::cout << ", ";
        else
            std::cout << ";" << std::endl;
    } // end of for

    std::cout << "variables:" << std::endl;
    for(std::vector<hdf4_var_sdfield *>::const_iterator i=file->sd->sdfields.begin(); i!=file->sd->sdfields.end(); i++)
    {
        std::cout << "    " << get_cdl_type_str((*i)->type);
        std::cout << "    " << (*i)->newname << "(";
        int size = (*i)->correcteddims.size(), nr=0;
        for(std::vector<hdf4_dim *>::const_iterator j=(*i)->get_corrected_dimensions().begin(); j!=(*i)->get_corrected_dimensions().end(); j++)
        {
            ++nr;
            std::cout << (*j)->get_name();
            if(nr!=size)
                std::cout << ",";
            else
                std::cout << ");" << std::endl;
        } // end of for
        for(std::vector<hdf4_attr *>::const_iterator j=(*i)->attrs.begin(); j!=(*i)->attrs.end(); j++)
        {
            std::cout << "     " << (*i)->newname << ":" << (*j)->get_name() << "=" << remove_non_printable_chars((*j)->get_str_value()) << ";" << std::endl;
        }
    } // end of for

    for(std::vector<VDATA *>::const_iterator i=file->vds.begin(); i!=file->vds.end(); i++)
    {
        for(std::vector<hdf4_var_vdfield *>::const_iterator j=(*i)->get_fields().begin(); j!=(*i)->get_fields().end(); j++) 
        {
            std::cout << "    " << get_cdl_type_str((*j)->type);
            std::cout << "    " << (*j)->newname << "()" << std::endl;
            for(std::vector<hdf4_attr *>::const_iterator k=(*j)->get_attributes().begin(); k!=(*j)->get_attributes().end(); k++)
            {
                std::cout << "     " << (*j)->newname << ":" << (*k)->get_name() << "=" << remove_non_printable_chars((*k)->get_str_value()) << ";" << std::endl; 
            }
        } // end of for
    } // end of for		

    // File attributes.
    for(std::vector<hdf4_attr *>::const_iterator i=file->sd->get_attrs().begin(); i!=file->sd->get_attrs().end(); i++)
    {
        std::cout << ":" << (*i)->get_name() << "=";
        std::cout << (*i)->get_str_value();
        std::cout << ";" << std::endl;
    } // ene of for

    // Data.
    if(bDumpData)
    {
        std::cout << "data:" << std::endl;
        for(std::vector<hdf4_var_sdfield *>::const_iterator i=file->sd->sdfields.begin(); i!=file->sd->sdfields.end(); i++)
        {
            //Only print out cv. For debugging only.
            std::map<std::string, std::string>::iterator it;
            for(it=file->sd->dimcvarlist.begin(); it!=file->sd->dimcvarlist.end(); it++)
            {
                if((*i)->newname.compare((*it).second)==0)
                    break;
            } // end of for
            if(it==file->sd->dimcvarlist.end())
                continue;
			
            //if((*i)->get_name().find("systemNoise")==std::string::npos)
            if((*i)->get_name().find("l3m_data")==std::string::npos)
                continue;
		
            int32 *start = (int32*)malloc((*i)->get_rank()*sizeof(int32));
            int32 *stride = (int32*)malloc((*i)->get_rank()*sizeof(int32));
            int32 *edge = (int32*)malloc((*i)->get_rank()*sizeof(int32));	
            int k = 0;
            for(std::vector<hdf4_dim *>::const_iterator j=(*i)->get_corrected_dimensions().begin(); j!=(*i)->get_corrected_dimensions().end(); j++)
            {
                start[k] = 2; // Testing "systemNoise", "l3m_data", and "tau_670" data fields in 1B21_CSI.990906.10217.KORA.6.HDF, C19860011986008.L3m_8D_CHLO_4.hdf and C1978303124834.L2_MLAC.hdffor subsetting.
                stride[k] = 2; //1;
                edge[k++] = 4; //(*j)->get_size();
            }

            std::cout << "    " << (*i)->newname << "=";
            (*i)->dump_str(file->get_sd_id());
            // Dump data subset.
            // Testing subsetting.
            //(*i)->dump_str(file->get_sd_id(), start, stride, edge);
            std::cout << ";" << std::endl; 
        } // end of for 
	
        for(std::vector<VDATA *>::const_iterator i=file->vds.begin(); i!=file->vds.end(); i++)
        {
            for(std::vector<hdf4_var_vdfield *>::const_iterator j=(*i)->get_fields().begin(); j!=(*i)->get_fields().end(); j++)
            {
                std::cout << "    " << (*j)->get_new_name() << "=" << (*j)->get_str_value() << ";" << std::endl;
            }
        } // end of for
    } // end of if(bDumpData)
    std::cout << "}" << std::endl;
}
*/
#endif
// Add  missing CF attributes for non-CV varibles
//#if 0
void add_missing_cf_attrs() {


    const std::vector<hdf4_var_sdfield* >& spsds = file->sd->sdfields;
    std::vector<hdf4_var_sdfield *>::const_iterator it_g;

    // TRMM version 7 level 3 grid
    if(TRMML3S_V7 == file->getSPType() || TRMML3M_V7 == file->getSPType()) {

        for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){

            if((*it_g)->fieldtype == 0 && (*it_g)->get_type()==DFNT_FLOAT32) {
                float fill_value = -9999.9;
                (*it_g)->add_attr_one_float("_FillValue",fill_value);
            }
        }

        for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){
            if((*it_g)->fieldtype == 0 && (((*it_g)->get_type()==DFNT_INT32) || ((*it_g)->get_type()==DFNT_INT16))) {

                int fill_value_int = -9999;
                short fill_value_short = -9999;
             
                if((*it_g)->get_type()==DFNT_INT32) 
                    (*it_g)->add_attr_one_int32("_FillValue",fill_value_int);
                else if((*it_g)->get_type()==DFNT_INT16) 
                    (*it_g)->add_attr_one_int16("_FillValue",fill_value_short);

            }
        }

        // nlayer for TRMM single grid version 7, the units should be "km"
        if(TRMML3S_V7 == file->getSPType()) {
            for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){

                if((*it_g)->fieldtype == 6 && (*it_g)->get_new_name()=="nlayer") 
                    (*it_g)->add_attr_one_str("units","km");
                else if((*it_g)->fieldtype == 4) {
                    if ((*it_g)->get_new_name()=="nh3" ||
                        (*it_g)->get_new_name()=="ncat3" ||
                        (*it_g)->get_new_name()=="nthrshZO" ||
                        (*it_g)->get_new_name()=="nthrshHB" ||
                        (*it_g)->get_new_name()=="nthrshSRT")
                    {

                        std::string references =
                           "http://pps.gsfc.nasa.gov/Documents/filespec.TRMM.V7.pdf";
                        std::string comment;

                        if((*it_g)->get_new_name()=="nh3") {
                            comment="Index number to represent the fixed heights above the earth ellipsoid,";
                            comment= comment + " at 2, 4, 6 km plus one for path-average.";
                        }

                        else if((*it_g)->get_new_name()=="ncat3") {
                            comment="Index number to represent catgories for probability distribution functions.";
                            comment=comment + "Check more information from the references.";
                        }

                        else if((*it_g)->get_new_name()=="nthrshZO") 
                            comment="Q-thresholds for Zero order used for probability distribution functions.";

                        else if((*it_g)->get_new_name()=="nthrshHB") 
                            comment="Q-thresholds for HB used for probability distribution functions.";

                         else if((*it_g)->get_new_name()=="nthrshSRT") 
                            comment="Q-thresholds for SRT used for probability distribution functions.";
                    
                        (*it_g)->add_attr_one_str("comment",comment);
                        (*it_g)->add_attr_one_str("references",references);

                    }

                }
                
            }

            // 3A26 use special values such as -666, -777,-999 in their fields. 
            // Although the document doesn't provide range for some fields, the meaning of those fields should be greater than 0.
            // So add valid_min = 0 and fill_value = -999 .
            std::string base_filename;
            size_t last_slash_pos = file->filename.find_last_of("/");
            if(last_slash_pos != std::string::npos)
                base_filename = file->filename.substr(last_slash_pos+1);
            if(""==base_filename)
                base_filename = file->filename;
            bool t3a26_flag = ((base_filename.find("3A26")!=std::string::npos)?true:false);
 
            if(true == t3a26_flag) {
                for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){
                    if((*it_g)->fieldtype == 0 && ((*it_g)->get_type()==DFNT_FLOAT32)) {
                        (*it_g)->remove_attr_by_name("_FillValue");
                        float fill_value = -999;
                        (*it_g)->add_attr_one_float("_FillValue",fill_value);
                        float valid_min = 0;
                        (*it_g)->add_attr_one_float("valid_min",valid_min);
                    }
                }
            }

        }

        // nlayer for TRMM single grid version 7, the units should be "km"
        if(TRMML3M_V7 == file->getSPType()) {
            for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){

                if((*it_g)->fieldtype == 4 ) {

                    std::string references ="http://pps.gsfc.nasa.gov/Documents/filespec.TRMM.V7.pdf";
                    if ((*it_g)->get_new_name()=="nh1") {

                        std::string comment="Number of fixed heights above the earth ellipsoid,";
                               comment= comment + " at 2, 4, 6, 10, and 15 km plus one for path-average.";

                        (*it_g)->add_attr_one_str("comment",comment);
                        (*it_g)->add_attr_one_str("references",references);

                    }
                    if ((*it_g)->get_new_name()=="nh3") {

                
                        std::string comment="Number of fixed heights above the earth ellipsoid,";
                               comment= comment + " at 2, 4, 6 km plus one for path-average.";

                        (*it_g)->add_attr_one_str("comment",comment);
                        (*it_g)->add_attr_one_str("references",references);

                    }

                    if ((*it_g)->get_new_name()=="nang") {

                        std::string comment="Number of fixed incidence angles, at 0, 5, 10 and 15 degree and all angles.";
                        references = "http://pps.gsfc.nasa.gov/Documents/ICSVol4.pdf";

                        (*it_g)->add_attr_one_str("comment",comment);
                        (*it_g)->add_attr_one_str("references",references);

                    }

                    if ((*it_g)->get_new_name()=="ncat2") {

                        std::string comment="Second number of categories for histograms (30). "; 
                        comment=comment + "Check more information from the references.";

                        (*it_g)->add_attr_one_str("comment",comment);
                        (*it_g)->add_attr_one_str("references",references);

                    }

                }
            }

        }

    }

    // TRMM level 2 swath
    else if(TRMML2_V7== file->getSPType()) {

        std::string base_filename;
        size_t last_slash_pos = file->filename.find_last_of("/");
        if(last_slash_pos != std::string::npos)
            base_filename = file->filename.substr(last_slash_pos+1);
        if(""==base_filename)
            base_filename = file->filename;

        bool t2b31_flag = ((base_filename.find("2B31")!=std::string::npos)?true:false);
        bool t2a21_flag = ((base_filename.find("2A21")!=std::string::npos)?true:false);
        bool t2a12_flag = ((base_filename.find("2A12")!=std::string::npos)?true:false);
        // 2A23 is temporarily not supported perhaps due to special fill values
        //bool t2a23_flag = ((base_filename.find("2A23")!=std::string::npos)?true:false);
        bool t2a25_flag = ((base_filename.find("2A25")!=std::string::npos)?true:false);
        bool t1c21_flag = ((base_filename.find("1C21")!=std::string::npos)?true:false);
        bool t1b21_flag = ((base_filename.find("1B21")!=std::string::npos)?true:false);
        bool t1b11_flag = ((base_filename.find("1B11")!=std::string::npos)?true:false);
        bool t1b01_flag = ((base_filename.find("1B01")!=std::string::npos)?true:false);

        // Handle scale and offset 
        // group 1: 2B31,2A12,2A21
        if(t2b31_flag || t2a12_flag || t2a21_flag) { 

            // special for 2B31 
            if(t2b31_flag) {
                for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){
         
                    if((*it_g)->fieldtype == 0 && (*it_g)->get_type()==DFNT_INT16) {

                        hdf4_attr* scale_attr= (*it_g)->get_attr_by_name("scale_factor"); 
                        if(scale_attr != NULL) {
                            if(scale_attr->get_type() == DFNT_FLOAT64) {
                                double scale_attr_value = *((double*)((void*)(&((scale_attr->get_value())[0]))));
                                double new_scale_attr_value = 1.0/scale_attr_value;
                                (*it_g)->remove_attr_by_name("scale_factor");
                                (*it_g)->add_attr_one_double("scale_factor",new_scale_attr_value);
                            }
                            else if(scale_attr->get_type() == DFNT_FLOAT32) {
                                float scale_attr_value = *((float*)((void*)(&((scale_attr->get_value())[0]))));
                                float new_scale_attr_value = 1.0/scale_attr_value;
                                (*it_g)->remove_attr_by_name("scale_factor");
                                (*it_g)->add_attr_one_float("scale_factor",new_scale_attr_value);
                            }
                        }
                    }
                }
            }

            // Special for 2A12
            if(t2a12_flag==true) {

                for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){

                    if((*it_g)->fieldtype == 6 && (*it_g)->get_new_name()=="nlayer") {

                        (*it_g)->add_attr_one_str("units","km");

                    }

                    // 
                    if((*it_g)->fieldtype == 0 && (*it_g)->get_type()==DFNT_INT8) {

                        char fill_value_int8 = -99;
                        (*it_g)->add_attr_one_int8("_FillValue",fill_value_int8);

                    }
                }
            }
 
            // for all 2A12,2A21 and 2B31
            // Add fillvalues for float32 and int32.
            for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){
                if((*it_g)->fieldtype == 0 && (*it_g)->get_type()==DFNT_FLOAT32) {
                    float fv = -9999.9;
                    (*it_g)->add_attr_one_float("_FillValue",fv);
                }
            }

            for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){
                if((*it_g)->fieldtype == 0 && (*it_g)->get_type()==DFNT_INT16) {
                    short fv = -9999;
                    (*it_g)->add_attr_one_int16("_FillValue",fv);
                }
            }

        }

        // group 2: 2A21 and 2A25.
        else if(t2a21_flag == true || t2a25_flag == true) {

            // 2A25: handle reflectivity and rain rate scales
            if(t2a25_flag == true) {

                float scale_factor_float = 0.;
                double scale_factor_double = 0.;
                int32 scale_factor_type = -1;

                for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){

                    if((*it_g)->fieldtype == 0 && (*it_g)->get_type()==DFNT_INT16) {

                        bool has_dBZ = false;
                        bool has_rainrate = false;

                        hdf4_attr * units_attr = (*it_g)->get_attr_by_name("units");
                        if(units_attr != NULL){
                            std::string units_value(units_attr->get_value().begin(),units_attr->get_value().end()) ;
                            if("dBZ" == units_value) { 
                                has_dBZ = true;
                            }
                            else if("mm/hr" == units_value){
                                has_rainrate = true;
                            }
                        }
                       
                        hdf4_attr * sf_attr = (*it_g)->get_attr_by_name("scale_factor");
                        if(sf_attr != NULL)
                        {
                            scale_factor_type = sf_attr->get_type();
                            if(DFNT_FLOAT32 == scale_factor_type)
                                scale_factor_float = *((float*)((void*)(&((sf_attr->get_value())[0]))));
                            else if (DFNT_FLOAT64 == scale_factor_type)
                                scale_factor_double = *((double*)((void*)(&((sf_attr->get_value())[0]))));
                        }
                        
                        if(true == has_rainrate || true == has_dBZ) {

                            short valid_min = 0; 
                            short valid_max = 0;
                                 
                            if(true == has_rainrate) {
                            
                                if(scale_factor_type == DFNT_FLOAT32)
                                    valid_max = (short)(300*scale_factor_float);
                                else if(scale_factor_type == DFNT_FLOAT64)
                                    valid_max = (short)(300*scale_factor_double);
                            }
                            else if(true == has_dBZ) {
                                if(scale_factor_type == DFNT_FLOAT32)
                                    valid_max = (short)(80*scale_factor_float);
                                else if(scale_factor_type == DFNT_FLOAT64)
                                    valid_max = (short)(80*scale_factor_double);
                            }

                            (*it_g)->add_attr_one_int16("valid_min",valid_min);
                            if(valid_max !=0)
                                (*it_g)->add_attr_one_int16("valid_max",valid_max);

                            if(scale_factor_type == DFNT_FLOAT64) {
                                double new_scale = 1.0/scale_factor_double;
                                (*it_g)->remove_attr_by_name("scale_factor");
                                (*it_g)->add_attr_one_double("scale_factor",new_scale);
                            }
                               
                            else if(scale_factor_type == DFNT_FLOAT32) {
                                float new_scale = 1.0/scale_factor_float;
                                (*it_g)->remove_attr_by_name("scale_factor");
                                (*it_g)->add_attr_one_float("scale_factor",new_scale);
                            }
                        }

                    }
                }
            }
        }

        // 1B21,1C21 and 1B11
        else if(t1b21_flag || t1c21_flag || t1b11_flag) {

            // 1B21,1C21 scale_factor to CF and valid_range for dBm and dBZ.
            if(t1b21_flag || t1c21_flag) {

                for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){

                    if((*it_g)->fieldtype == 0 && (*it_g)->get_type()==DFNT_INT16) {

                        bool has_dBm = false;
                        bool has_dBZ = false;
                        hdf4_attr *unit_attr = (*it_g)->get_attr_by_name("units");

                        if(unit_attr != NULL){
                            std::string units_value(unit_attr->get_value().begin(),unit_attr->get_value().end()) ;
                            if("dBm" == units_value) { 
                                has_dBm = true;
                            }
                            else if("dBZ" == units_value){
                                has_dBZ = true;
                            }
                        }

                        if(has_dBm == true || has_dBZ == true) {
                            float scale_factor_float = 0.;
                            double scale_factor_double = 0.;
                            short valid_min = 0;
                            short valid_max = 0;
                            int32 scale_factor_type = -1;
                            hdf4_attr * sf_attr = (*it_g)->get_attr_by_name("scale_factor");
                            if(sf_attr != NULL)
                            {
                                scale_factor_type = sf_attr->get_type();
                                if(DFNT_FLOAT32 == scale_factor_type)
                                    scale_factor_float = *((float*)((void*)(&((sf_attr->get_value())[0]))));
                                else if (DFNT_FLOAT64 == scale_factor_type)
                                    scale_factor_double = *((double*)((void*)(&((sf_attr->get_value())[0]))));
                            }
 
                            if(true == has_dBm) {
                                
                                if(DFNT_FLOAT32 == scale_factor_type) {
                                    valid_min = (short)(-120 *scale_factor_float);
                                    valid_max = (short)(-20 *scale_factor_float);
                                }
                                else if(DFNT_FLOAT64 == scale_factor_type) {
                                    valid_min = (short)(-120 *scale_factor_double);
                                    valid_max = (short)(-20 *scale_factor_double);
                                }
                                
                                if(valid_min != 0)
                                    (*it_g)->add_attr_one_int16("valid_min",valid_min);
                                if(valid_max != 0)
                                    (*it_g)->add_attr_one_int16("valid_max",valid_max);
                                      
                            }

                            else if(true == has_dBZ){
                                if(DFNT_FLOAT32 == scale_factor_type) {
                                    valid_min = (short)(-120 *scale_factor_float);
                                    valid_max = (short)(-20 *scale_factor_float);
                                }
                                else if(DFNT_FLOAT64 == scale_factor_type) {
                                    valid_min = (short)(-120 *scale_factor_double);
                                    valid_max = (short)(-20 *scale_factor_double);
 
                                }
                                
                                if(valid_min != 0)
                                    (*it_g)->add_attr_one_int16("valid_min",valid_min);
                                if(valid_max != 0)
                                    (*it_g)->add_attr_one_int16("valid_max",valid_max);
                            }
                        }
                    }
                }
            }

            // For all 1B21,1C21 and 1B11 int16-bit products,change scale to follow CF
            // I find that one 1B21 variable binStormHeight has fillvalue -9999,
            // so add _FillValue -9999 for int16-bit variables.
            for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){

                if((*it_g)->fieldtype == 0 && (*it_g)->get_type()==DFNT_INT16) {

                    int32 scale_factor_type = -1;
                    float scale_factor_float = 0.;
                    double scale_factor_double = 0.;
 
                    hdf4_attr * sf_attr = (*it_g)->get_attr_by_name("scale_factor");
                    if(sf_attr != NULL)
                    {
                        scale_factor_type = sf_attr->get_type();
                        if(DFNT_FLOAT32 == scale_factor_type)
                            scale_factor_float = *((float*)((void*)(&((sf_attr->get_value())[0]))));
                        else if (DFNT_FLOAT64 == scale_factor_type)
                            scale_factor_double = *((double*)((void*)(&((sf_attr->get_value())[0]))));
                    }
 
                    if(scale_factor_type == DFNT_FLOAT64) {
                        double new_scale = 1.0/scale_factor_double;
                        (*it_g)->remove_attr_by_name("scale_factor");
                        (*it_g)->add_attr_one_double("scale_factor",new_scale);
 
                    }
                               
                    else if(scale_factor_type == DFNT_FLOAT32) {
                        float new_scale = 1.0/scale_factor_float;
                        (*it_g)->remove_attr_by_name("scale_factor");
                        (*it_g)->add_attr_one_float("scale_factor",new_scale);
 
                    }

                    short fill_value_short = -9999;
                    std::vector<char>fv_short(2);
                    memcpy((void*)&(fv_short)[0], (const void*)(&fill_value_short), sizeof(short));
                    hdf4_attr* myattr = new hdf4_attr("_FillValue", DFNT_INT16, 1, fv_short);
                    (*it_g)->attrs.push_back(myattr);

                }
            }
        }

        // For 1B01 product, just add the fillvalue.
        else if(t1b01_flag == true) {
            for(it_g = spsds.begin(); it_g != spsds.end(); it_g++){
                if((*it_g)->fieldtype == 0 && (*it_g)->get_type()==DFNT_FLOAT32) {
                    float fv = -9999.9;
                    (*it_g)->add_attr_one_float("_FillValue",fv);

                }
            }
        }

        std::string references ="http://pps.gsfc.nasa.gov/Documents/filespec.TRMM.V7.pdf";
        std::string comment="The h4cf conversion toolkit adds _FillValue, valid_min and valid_max for some TRMM level 1 and level 2 products.";
        comment= comment + " It also changes scale_factor to follow CF conventions. ";

        std::vector<char> comment_v(comment.size());
        std::copy(comment.begin(),comment.end(),comment_v.begin());
        
        hdf4_attr * comment_attr = new hdf4_attr("comment",DFNT_CHAR8,comment.size(),comment_v);

        std::vector<char> references_v(references.size());
        std::copy(references.begin(),references.end(),references_v.begin());
        
        hdf4_attr * references_attr = new hdf4_attr("references",DFNT_CHAR8,references.size(),references_v);

        file->sd->attrs.push_back(comment_attr);
        file->sd->attrs.push_back(references_attr);   
    }

}

// Add latitude and longitude for TRMM L2. 
void prepare_trmm_l2()
{
    if(file->getSPType() != TRMML2_V6)
        return;

    //1. Obtain the geolocation field: type,dimension size and dimension name
    //2. Create latitude and longtiude fields according to the geolocation field.
    std::string tempdimname1, tempdimname2;
    std::string tempnewdimname1, tempnewdimname2;
    std::string temppath2;

    int32 tempdimsize1, tempdimsize2;
    hdf4_var_sdfield *longitude;
    hdf4_var_sdfield *latitude;

    // Create a temporary map from the dimension size to the dimension name
    std::set < int32 > tempdimsizeset;
    std::map < int32, std::string > tempdimsizenamelist;
    std::map < int32, std::string >::iterator tempsizemapit;
    std::pair < std::set < int32 >::iterator, bool > tempsetit;


    // Reduce the fakeDim list. FakeDim is found to be used by TRMM team.
    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j) 
        {
            if (((*j)->get_name ()).find ("fakeDim") == std::string::npos) 
            { //No fakeDim in the string
                tempsetit = tempdimsizeset.insert ((*j)->get_size ());
                if (tempsetit.second == true)
                    tempdimsizenamelist[(*j)->get_size ()] = (*j)->get_name ();
            } // end of if
        } // end of for
    } // end of for

    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        if ((*i)->get_name () == "geolocation") 
        {
            // Obtain the size and the name of the first two dimensions of the geolocation field; make these two dimensions the dimensions of latitude and longtiude.
            (*i)->generate_str_value(file->get_sd_id());
            std::string s  = (*i)->get_str_value();
            std::string s1, s2;
            std::vector<char> v1, v2;
            size_t prepos=0;
            size_t pos=s.find(",");
            int nr=0;
            while(pos!=std::string::npos)
            {
                std::string t1 = s.substr(prepos, pos-prepos);
                const char *t2 = t1.c_str();
                float t3 = atof(t2);
                if(nr/2*2==nr)
                {
                    s1.append(t1); //s.substr(prepos, pos-prepos));
                    s1.append(",");
				
                    size_t t4 = v1.size();
                    v1.resize(t4+4);
                    memcpy((char*)&(v1)[t4], &t3, sizeof(t3));
                } // end of if(nr/2 
                else
                {
                    s2.append(t1); //s.substr(prepos, pos-prepos));
                    s2.append(",");

                    size_t t4 = v2.size();
                    v2.resize(t4+4);
                    memcpy((char*)&(v2)[t4], &t3, sizeof(t3));
                } // end of else
                prepos = pos+1;
                pos = s.find(",", prepos);
                ++nr;
            } // end of while
            s1 = s1.substr(0, s1.size()-1);
            s2 = s2.substr(0, s2.size()-1);

            tempdimname1 = ((*i)->get_dimensions ())[0]->get_name ();
            tempdimsize1 = ((*i)->get_dimensions ())[0]->get_size ();
            tempdimname2 = ((*i)->get_dimensions ())[1]->get_name ();
            tempdimsize2 = ((*i)->get_dimensions ())[1]->get_size ();
		
            tempnewdimname1 = ((*i)->get_corrected_dimensions ())[0]->get_name ();
            tempnewdimname2 = ((*i)->get_corrected_dimensions ())[1]->get_name ();
            latitude = new hdf4_var_sdfield();
            latitude->rename("latitude");
            latitude->rank = 2;
            latitude->sdsref = (*i)->sdsref;
            latitude->type = (*i)->get_type ();
			
            size_t npos = (*i)->newname.size()-(*i)->get_name().size(); // Remove 'geolocation' at the end
            temppath2 = (*i)->newname.substr(0, npos); //.substr (((*i)->get_name()).size ()); 

            //latitude->newname = latitude->get_name() + temppath2;
            latitude->newname = temppath2 + latitude->get_name(); //[04/16/2012 LD]
            latitude->fieldtype = 1;
            latitude->rootfieldname = "geolocation";
            hdf4_dim *dim = new hdf4_dim (tempdimname1, tempdimname1, tempdimsize1, 0);
            latitude->dims.push_back (dim);
            dim = new hdf4_dim (tempdimname2, tempdimname2, tempdimsize2, 0);
            latitude->dims.push_back (dim);
            dim = new hdf4_dim (tempnewdimname1, tempdimname1, tempdimsize1, 0);
            latitude->correcteddims.push_back (dim);
            dim = new hdf4_dim (tempnewdimname2, tempnewdimname2, tempdimsize2, 0);
            latitude->correcteddims.push_back (dim);

            latitude->str_value = s1;

            latitude->value.resize(v1.size());
            std::copy(v1.begin(), v1.end(), latitude->value.begin());

            latitude->add_attr_one_float("_FillValue",-9999.0);

            longitude = new hdf4_var_sdfield ();
            longitude->rename("longitude");
            longitude->rank = 2;
            longitude->sdsref = (*i)->sdsref;
            longitude->type = (*i)->get_type ();
            //longitude->newname = longitude->get_name() + temppath;
            longitude->newname = temppath2 + longitude->get_name(); //[04/16/2012 LD]
            longitude->fieldtype = 2;
            longitude->rootfieldname = "geolocation";
            dim = new hdf4_dim (tempdimname1, tempdimname1, tempdimsize1, 0);
            longitude->dims.push_back (dim);
            dim = new hdf4_dim (tempdimname2, tempdimname2, tempdimsize2, 0);
            longitude->dims.push_back (dim);
            dim = new hdf4_dim (tempnewdimname1, tempnewdimname1, tempdimsize1, 0);
            longitude->correcteddims.push_back (dim);
            dim = new hdf4_dim (tempnewdimname2, tempnewdimname2, tempdimsize2, 0);
            longitude->correcteddims.push_back (dim);

            longitude->str_value = s2;

            longitude->value.resize(v2.size());
            std::copy(v2.begin(), v2.end(), longitude->value.begin());

            longitude->add_attr_one_float("_FillValue",-9999.0);
        } // end of if ((*i)->get_name () == "geolocation")
        else 
        {
            // Use the temp. map (size to name) to replace the name of "fakeDim???" with the dimension name having the same dimension length. This is done only for TRMM. It should be evaluated if this can be applied to other products.
            for (std::vector < hdf4_dim * >::const_iterator k = (*i)->get_corrected_dimensions ().begin (); k != (*i)->get_corrected_dimensions ().end (); ++k) 
            {
                size_t fakeDimpos = ((*k)->get_name ()).find ("fakeDim");
                if (fakeDimpos != std::string::npos) 
                {
                    tempsizemapit = tempdimsizenamelist.find ((*k)->get_size ());
                    if (tempsizemapit != tempdimsizenamelist.end ())
                        (*k)->rename(tempdimsizenamelist[(*k)->get_size ()]);	// Change the dimension name
                } // end of if
            } // end of for
        } // end of else
    } // end of for (std::vector <  

    file->sd->sdfields.push_back (latitude);
    file->sd->sdfields.push_back (longitude);

    // 3. Remove the geolocation field from the field list
    hdf4_var_sdfield *origeo = NULL;

    std::vector < hdf4_var_sdfield * >::iterator toeraseit;
    for (std::vector < hdf4_var_sdfield * >::iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        if ((*i)->get_name () == "geolocation") 
        { // Check the release of dimension and other resources
            toeraseit = i;
            origeo = *i;
            break;
        } // end of if
    } // end of for

    file->sd->sdfields.erase (toeraseit);
    delete (origeo);

    // 4. Create the <dimname,coordinate variable> map from the corresponding dimension names to the latitude and the longitude
    file->sd->nonmisscvdimnamelist.insert (tempnewdimname1);
    file->sd->nonmisscvdimnamelist.insert (tempnewdimname2);
}

// Add latitude and longitude for TRMM L3.
void prepare_trmm_l3()
{
    if(file->getSPType() != TRMML3_V6)
        return;

    float slat = -49.875;
    float slon = -179.875;
    float step = 0.25;

    std::string tempdimname1, tempdimname2;
    std::string tempnewdimname1, tempnewdimname2;
    int latflag = 0;
    int lonflag = 0;

    std::string temppath3;
    hdf4_var_sdfield *latitude = NULL;
    hdf4_var_sdfield *longitude = NULL;

    std::vector<char> v1, v2;

    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        for (std::vector < hdf4_dim * >::const_iterator k = (*i)->get_dimensions ().begin (); k != (*i)->get_dimensions ().end (); ++k) 
        {
            // This dimension has the dimension name
            if ((((*k)->get_name ()).find ("fakeDim")) == std::string::npos)
            {
                size_t npos = (*i)->newname.size()-(*i)->get_name().size(); // Remove 'geolocation' at the end
                temppath3 = (*i)->newname.substr(0, npos); //.substr (((*i)->get_name()).size ()); [04/16/2012]
			
                // The lat/lon formula is from GES DISC web site. http://disc.sci.gsfc.nasa.gov/additional/faq/precipitation_faq.shtml#lat_lon
                if ((*k)->get_size () == 1440 && (*k)->get_type () == 0)
                {
                    //No dimension scale
                    longitude = new hdf4_var_sdfield ();
                    longitude->rename("longitude");
                    longitude->rank = 1;
                    longitude->type = DFNT_FLOAT32;
                    longitude->fieldtype = 2;
                    //longitude->newname = longitude->get_name() + temppath3;
                    longitude->newname = temppath3 + longitude->get_name(); //[04/16/2012 LD]
                    longitude->add_attr_one_float("_FillValue",-9999.0);
                    hdf4_dim *dim = new hdf4_dim ((*k)->get_name (), (*k)->get_orig_name(),(*k)->get_size (), 0);
                    longitude->dims.push_back (dim);
                    tempnewdimname2 = (*k)->get_name ();

                    dim = new hdf4_dim ((*k)->get_name (), (*k)->get_orig_name(), (*k)->get_size (), 0);
                    longitude->correcteddims.push_back (dim);
                    lonflag++;

                    int icount = 0;
                    float sval = slon + step*0;//(int) (offset32[0]);
                    std::string mystr;
                    char c[100];
                    while (icount < (int)(*k)->get_size ())//(int) (count32[0]) 
                    {
                        sprintf(c, "%.6f",  sval + step * (int) /*(step32[0]) **/ icount);
                        mystr.append(c);
                        mystr.append(",");
				
                        //val[icount] = sval + 0.25 * (int) (step32[0]) * icount;
                        size_t t1 = v1.size();
                        v1.resize(t1+4);
                        float t2 = atof(c);
                        memcpy((char*)&(v1)[t1], &t2, sizeof(t2));
  
                        icount++;
                    } // end of while
                    size_t ppp = mystr.find_last_of(",");
                    longitude->str_value = mystr.substr(0, ppp);
                    longitude->value.resize(v1.size());
                    std::copy(v1.begin(), v1.end(), longitude->value.begin());
                } // end of if ((*k)->get_size () == 1440 

                if ((*k)->get_size () == 400 && (*k)->get_type () == 0) 
                {
                    latitude = new hdf4_var_sdfield ();
                    latitude->rename("latitude");
                    latitude->rank = 1;
                    latitude->type = DFNT_FLOAT32;
                    latitude->fieldtype = 1;
                    //latitude->newname = latitude->get_name() + temppath3;
                    latitude->newname = temppath3 + latitude->get_name(); // [04/16/2012 LD]
                    latitude->add_attr_one_float("_FillValue",-9999.0);
                    hdf4_dim *dim = new hdf4_dim ((*k)->get_name (), (*k)->get_orig_name(),(*k)->get_size (), 0);
                    latitude->dims.push_back (dim);
                    tempnewdimname1 = (*k)->get_name ();

                    // We donot need to generate the  unique dimension name based on the full path for all the current  cases we support.
                    dim = new hdf4_dim ((*k)->get_name (), (*k)->get_orig_name(),(*k)->get_size (), 0);
                    latitude->correcteddims.push_back (dim);
                    latflag++;

                    int icount = 0;
                    float sval = slat+step*0; //(int)(offset32[0]);
                    char c[100];
                    std::string mystr;
                    while (icount < (*k)->get_size ()) //{(int) (count32[0])) 
                    {
                        sprintf(c, "%.6f",  sval + step * (int) /*(step32[0]) **/ icount);
                        mystr.append(c);
                        mystr.append(",");

                        //val[icount] = sval + 0.25 * (int) (step32[0]) * icount;
                        size_t t1 = v2.size();
                        v2.resize(t1+4);
                        float t2 = atof(c);
                        memcpy((char*)&(v2)[t1], &t2, sizeof(t2));	
                        icount++;
                    } // end of while
                    size_t ppp = mystr.find_last_of(",");
                    latitude->str_value = mystr.substr(0, ppp);
                    latitude->value.resize(v2.size());
                    std::copy(v2.begin(), v2.end(), latitude->value.begin());
                } // end of if ((*k)->get_size () == 400
            } // end of if ((((*k)->get_name ()).)

            if (latflag == 1 && lonflag == 1)
                break;
        } // end of for (std::vector  

        if (latflag == 1 && lonflag == 1)
            break;				
        // For this case, a field that needs lon and lot must exist. Need to reset the flag to avoid false alarm. For TRMM L3 we can handle, a field that has dimension which size is 400 and 1440 must exist in the file.
        latflag = 0;
        lonflag = 0;
    } // end of for (std::vector  

    if (latflag != 1 || lonflag != 1)
    {
        std::cout << "Either latitude or longitude doesn't exist." <<  "lat. flag= " << latflag << "lon. flag= " << lonflag << std::endl;
        throw std::runtime_error("Either latitude or longitude doesn't exist.(" __FILE__ ":" TOSTRING(__LINE__)")" );
        //exit(1);
    }
    file->sd->sdfields.push_back (latitude);
    file->sd->sdfields.push_back (longitude);

    // Create the <dimname,coordinate variable> map from the corresponding dimension names to the latitude and the longitude
    file->sd->nonmisscvdimnamelist.insert (tempnewdimname1);
    file->sd->nonmisscvdimnamelist.insert (tempnewdimname2);
}

// Add latitude and longitude for TRMM 3A46.
void prepare_trmm_3a46()
{
    if(file->getSPType() != TRMM_3A46)
        return;

    // The orignal dimension in the file is wrong. We have to fix it.
    for(std::vector < hdf4_var_sdfield * >::iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i)
    {
        for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j) {
            if(((*j)->get_name().find("latitude")) == 0)
                (*j)->rename("fakeDim1");
            else if (((*j)->get_name().find("longitude")) == 0)
                (*j)->rename("fakeDim2");
        }
    }

    int32 dimsize1=360;
    int32 dimsize2=180;
    std::string dimname1 = "longitude";
    std::string dimname2 ="latitude";

    std::vector<char> v1;
    std::vector<char> v2;

    hdf4_var_sdfield *longitude = new hdf4_var_sdfield ();
    longitude->rename(dimname1);
    longitude->rank =1;
    longitude->type = DFNT_FLOAT32;
    longitude->fieldtype = 2;
    longitude->newname = longitude->get_name();

    hdf4_dim *dim = new hdf4_dim (dimname1, dimname1,dimsize1, 0);
    longitude->dims.push_back (dim);

    dim = new hdf4_dim (dimname1, dimname1, dimsize1, 0);
    longitude->correcteddims.push_back (dim);

    { // 1
        char c[100];
        std::string mystr;
        for(int k=-179; k<=180; k++)
        {
            sprintf(c, "%d", k);
            mystr.append(c);
            if(k!=180)
                mystr.append(",");

            size_t t1 = v1.size();
            v1.resize(t1+4);
            float t2 = atof(c);
            memcpy((char*)&(v1)[t1], &t2, sizeof(t2));
        } // ene of for
        longitude->str_value = mystr;
        longitude->value.resize(v1.size());
        std::copy(v1.begin(), v1.end(), longitude->value.begin());
    } // end of 1

    hdf4_var_sdfield *latitude = new hdf4_var_sdfield ();
    latitude->rename(dimname2);
    latitude->rank =1;
    latitude->type = DFNT_FLOAT32;
    latitude->fieldtype = 1;
    latitude->newname = latitude->get_name();

    dim = new hdf4_dim (dimname2, dimname2, dimsize2, 0);
    latitude->dims.push_back (dim);

    dim = new hdf4_dim (dimname2, dimname2,dimsize2, 0);
    latitude->correcteddims.push_back (dim);

    { // 2
        char c[100];
        std::string mystr;
        for(int k=-89; k<=90; k++)
        {
            sprintf(c, "%d", k);
            mystr.append(c);
            if(k!=90)
                mystr.append(",");
            size_t t1 = v2.size();
            v2.resize(t1+4);
            float t2 = atof(c);
            memcpy((char*)&(v2)[t1], &t2, sizeof(t2));
        } // end of for
        latitude->str_value = mystr;
        latitude->value.resize(v2.size());
        std::copy(v2.begin(), v2.end(), latitude->value.begin());
    } // end of 2

    file->sd->sdfields.push_back (latitude);
    file->sd->sdfields.push_back (longitude);

    // Set dimname,coordinate variable list
    file->sd->nonmisscvdimnamelist.insert (dimname1);
    file->sd->nonmisscvdimnamelist.insert (dimname2);

    for(std::vector < hdf4_var_sdfield * >::iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i)
    {
        for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j) {

            if((*j)->get_name() != "latitude" && (*j)->get_size() ==180) 
                (*j)->rename("latitude");
            else if ((*j)->get_name() != "longitude" && (*j)->get_size() ==360)
                (*j)->rename("longitude");

        }
    }


// The code of decomposing the variable SSMI doesn't assign the data value. It is
// not worth the effort to fix this. So comment this out. KY 2015-08-27
#if 0
    //Decompose SSM/I data
    for(std::vector < hdf4_var_sdfield * >::iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i)
    {
        if((*i)->get_name().compare("ssmiData")==0)
        {	
            hdf4_var_sdfield *precipitation_rate = new hdf4_var_sdfield();
            precipitation_rate->rename("Precipitation_Rate");
            precipitation_rate->rank = 2;
            precipitation_rate->type = DFNT_FLOAT32;
            precipitation_rate->fieldtype = 0;
            precipitation_rate->newname = precipitation_rate->get_name();

            dim = new hdf4_dim(dimname1, dimname1,dimsize1, 0);
            precipitation_rate->dims.push_back (dim);

            dim = new hdf4_dim(dimname2, dimname2,dimsize2, 0);
            precipitation_rate->dims.push_back (dim);

            dim = new hdf4_dim(dimname1, dimname1,dimsize1, 0);
            precipitation_rate->correcteddims.push_back (dim);

            dim = new hdf4_dim(dimname2, dimname2, dimsize2, 0);
            precipitation_rate->correcteddims.push_back (dim);
	
            hdf4_var_sdfield *number_of_observations = new hdf4_var_sdfield();
            number_of_observations->rename("Number_Of_Observations");
            number_of_observations->rank = 2;
            number_of_observations->type = DFNT_FLOAT32;
            number_of_observations->fieldtype = 0;
            number_of_observations->newname = number_of_observations->get_name();

            dim = new hdf4_dim(dimname1, dimname1,dimsize1, 0);
            number_of_observations->dims.push_back (dim);

            dim = new hdf4_dim(dimname2, dimname2,dimsize2, 0);
            number_of_observations->dims.push_back (dim);

            dim = new hdf4_dim(dimname1, dimname1,dimsize1, 0);
            number_of_observations->correcteddims.push_back (dim);

            dim = new hdf4_dim(dimname2, dimname2,dimsize2, 0);
            number_of_observations->correcteddims.push_back (dim);

            //Precipitation Rate resides from 0-180*360-1, Number of Observations resides from 180*360-2*180*360-1
            (*i)->generate_str_value(file->get_sd_id());
            std::string mystr = (*i)->get_str_value();
std::cerr<<"mystr is "<<mystr <<std::endl;
            size_t prepos=0;
            size_t pos=mystr.find(",");
            int nr=0;
            int total=180*360*1;
            std::string ss;
            while(nr<total && pos!=std::string::npos)
            {
                ss.append(mystr.substr(prepos, pos-prepos));
                ss.append(",");
			
                nr++;
                prepos=pos+1;
                pos = mystr.find(",", prepos);
            } // end of while
            pos = ss.find_last_of(",");
            precipitation_rate->str_value = ss.substr(0, pos);		

            std::string s;
            nr=0;
            while(nr<total && pos!=std::string::npos)
            {
                s.append(mystr.substr(prepos, pos-prepos));
                s.append(",");

                nr++;
                prepos=pos+1;
                pos = mystr.find(",", prepos);
            } // end of while 
            pos = s.find_last_of(",");
            number_of_observations->str_value = s.substr(0, pos);
            delete (*i);		
            file->sd->sdfields.erase (i);
		
            i--;

            file->sd->sdfields.push_back(precipitation_rate);
            file->sd->sdfields.push_back(number_of_observations);

            break;
        } // end of if((*i)->get_name() 
    } // end of for
#endif
}

// TRMM V7 stores grid lat/lon information in an attribute GridHeader.
// We need to parse the attribute to obtain the lat/lon values.
void parser_trmm_v7_gridheader(const std::vector<char>& value, 
                               int& latsize, int&lonsize, 
                               float& lat_start, float& lon_start,
                               float& lat_res, float& lon_res,
                               bool check_reg_orig ){

    //bool cr_reg = false;
    //bool sw_origin = true;
    //float lat_res = 1.;
    //float lon_res = 1.;
    float lat_north = 0.;
    float lat_south = 0.;
    float lon_east = 0.;
    float lon_west = 0.;
     
    // Retrieve the individual element values in the gridheader to a vector string.
    std::vector<std::string> ind_elems;
    std::string sep="\n";
    std::string value_str(value.begin(),value.end());
    util_split_str(value_str,sep,&ind_elems);
     
    // The number of elements in the GridHeader is 9. 
    if(ind_elems.size()!=9)
        throw std::runtime_error("The number of elements in the TRMM level 3 GridHeader is not right.(" __FILE__ ":" TOSTRING(__LINE__)")" );

    if(false == check_reg_orig) {
        if (0 != ind_elems[1].find("Registration=CENTER"))        
            throw std::runtime_error("The TRMM grid registration is not center.(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }
        
    if (0 == ind_elems[2].find("LatitudeResolution")){ 

        size_t equal_pos = ind_elems[2].find_first_of('=');
        if(std::string::npos == equal_pos)
            throw std::runtime_error("Cannot find latitude resolution for TRMM level 3 products(" __FILE__ ":" TOSTRING(__LINE__)")" );
           
        size_t scolon_pos = ind_elems[2].find_first_of(';');
        if(std::string::npos == scolon_pos)
            throw std::runtime_error("Cannot find latitude resolution for TRMM level 3 products(" __FILE__ ":" TOSTRING(__LINE__)")" );
        if (equal_pos < scolon_pos){

            std::string latres_str = ind_elems[2].substr(equal_pos+1,scolon_pos-equal_pos-1);
            lat_res = (float)(strtod(latres_str.c_str(),NULL));
        }
        else 
            throw std::runtime_error(" Latitude resolution is not right for TRMM level 3 products(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }
    else
        throw std::runtime_error(" The TRMM grid LatitudeResolution doesn't exist.(" __FILE__ ":" TOSTRING(__LINE__)")" );


    if (0 == ind_elems[3].find("LongitudeResolution")){ 

        size_t equal_pos = ind_elems[3].find_first_of('=');
        if(std::string::npos == equal_pos)
            throw std::runtime_error("Cannot find longitude resolution for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
           
        size_t scolon_pos = ind_elems[3].find_first_of(';');
        if(std::string::npos == scolon_pos)
            throw std::runtime_error("Cannot find longitude resolution for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
        if (equal_pos < scolon_pos){
            std::string lonres_str = ind_elems[3].substr(equal_pos+1,scolon_pos-equal_pos-1);
            lon_res = (float)(strtod(lonres_str.c_str(),NULL));
        }
        else 
            throw std::runtime_error(" longitude resolution is not right for TRMM level 3 products(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }
    else
        throw std::runtime_error(" The TRMM grid LongitudeResolution doesn't exist.(" __FILE__ ":" TOSTRING(__LINE__)")" );

    if (0 == ind_elems[4].find("NorthBoundingCoordinate")){ 

        size_t equal_pos = ind_elems[4].find_first_of('=');
        if(std::string::npos == equal_pos)
            throw std::runtime_error(" Cannot find north bound coordinate for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
           
        size_t scolon_pos = ind_elems[4].find_first_of(';');
        if(std::string::npos == scolon_pos)
            throw std::runtime_error(" Cannot find north bound coordinate for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
        if (equal_pos < scolon_pos){
            std::string north_bounding_str = ind_elems[4].substr(equal_pos+1,scolon_pos-equal_pos-1);
            lat_north = (float)(strtod(north_bounding_str.c_str(),NULL));
        }
        else 
            throw std::runtime_error("NorthBoundingCoordinate is not right for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
 
    }
    else
        throw std::runtime_error("The TRMM grid NorthBoundingCoordinate doesn't exist. (" __FILE__ ":" TOSTRING(__LINE__)")" );

    if (0 == ind_elems[5].find("SouthBoundingCoordinate")){ 

            size_t equal_pos = ind_elems[5].find_first_of('=');
            if(std::string::npos == equal_pos)
                throw std::runtime_error("Cannot find south bound coordinate for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
           
            size_t scolon_pos = ind_elems[5].find_first_of(';');
            if(std::string::npos == scolon_pos)
                throw std::runtime_error("Cannot find south bound coordinate for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
            if (equal_pos < scolon_pos){
                std::string lat_south_str = ind_elems[5].substr(equal_pos+1,scolon_pos-equal_pos-1);
                lat_south = (float)(strtod(lat_south_str.c_str(),NULL));
            }
            else 
                throw std::runtime_error("South bound coordinate is not right for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
    }
    else
        throw std::runtime_error("The TRMM grid SouthBoundingCoordinate doesn't exist. (" __FILE__ ":" TOSTRING(__LINE__)")" );

    if (0 == ind_elems[6].find("EastBoundingCoordinate")){ 

            size_t equal_pos = ind_elems[6].find_first_of('=');
            if(std::string::npos == equal_pos)
                throw std::runtime_error("Cannot find east bound coordinate for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
           
            size_t scolon_pos = ind_elems[6].find_first_of(';');
            if(std::string::npos == scolon_pos)
                throw std::runtime_error("Cannot find east bound coordinate for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
            if (equal_pos < scolon_pos){
                std::string lon_east_str = ind_elems[6].substr(equal_pos+1,scolon_pos-equal_pos-1);
                lon_east = (float)(strtod(lon_east_str.c_str(),NULL));
            }
            else 
                throw std::runtime_error("east bound coordinate is not right for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
 
    }
    else
        throw std::runtime_error("The TRMM grid EastBoundingCoordinate doesn't exist. (" __FILE__ ":" TOSTRING(__LINE__)")" );

    if (0 == ind_elems[7].find("WestBoundingCoordinate")){ 

            size_t equal_pos = ind_elems[7].find_first_of('=');
            if(std::string::npos == equal_pos)
                throw std::runtime_error("Cannot find west bound coordinate for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
           
            size_t scolon_pos = ind_elems[7].find_first_of(';');
            if(std::string::npos == scolon_pos)
                throw std::runtime_error("Cannot find west bound coordinate for TRMM level 3 products (" __FILE__ ":" TOSTRING(__LINE__)")" );
            if (equal_pos < scolon_pos){
                std::string lon_west_str = ind_elems[7].substr(equal_pos+1,scolon_pos-equal_pos-1);
                lon_west = (float)(strtod(lon_west_str.c_str(),NULL));
//cerr<<"latres str is "<<lon_west_str <<endl;
//cerr<<"latres is "<<lon_west <<endl;
            }
            else 
                throw std::runtime_error("west bound coordinate is not right for TRMM level 3 products. (" __FILE__ ":" TOSTRING(__LINE__)")" );
 
    }
    else
        throw std::runtime_error("The TRMM grid WestBoundingCoordinate doesn't exist. (" __FILE__ ":" TOSTRING(__LINE__)")" );

    if (false == check_reg_orig) {
        if (0 != ind_elems[8].find("Origin=SOUTHWEST")) 
            throw std::runtime_error("The TRMM grid origin is not SOUTHWEST. (" __FILE__ ":" TOSTRING(__LINE__)")" );
    }

    // Since we only treat the case when the Registration is center, so the size should be the 
    // regular number size - 1.
    latsize =(int)((lat_north-lat_south)/lat_res);
    lonsize =(int)((lon_east-lon_west)/lon_res);
    lat_start = lat_south;
    lon_start = lon_west;
}

// A wrapper function to obtain TRMM version 7 parameters to calculate lat/lon fields.
void Obtain_TRMML3S_V7_latlon_paras(int &latsize, int &lonsize,float &lat_start,float &lon_start, float &lat_res, float &lon_res)  {
     
    // No need to check if "GridHeader" etc. exists since this has been checked in the CheckSDType routine.
    for (std::vector < hdf4_attr * >::const_iterator i =
        file->sd->get_attrs ().begin ();
        i != file->sd->get_attrs ().end (); ++i) {
 
        if ((*i)->get_name () == "GridHeader") {
            parser_trmm_v7_gridheader((*i)->get_value(),latsize,lonsize,
                                                 lat_start,lon_start,
                                                 lat_res,lon_res,false);
            break;
        }
    }

}

// Retrieve TRMM version 7 lat/lon names.
bool Obtain_TRMM_V7_latlon_name(const hdf4_var_sdfield* sdfield, const int latsize, 
                                       const int lonsize, std::string& latname, std::string& lonname)  {

    int latname_index = -1;
    int lonname_index = -1;
    for (int temp_index = 0; temp_index <sdfield->get_rank(); ++temp_index) {
        if(-1==latname_index && sdfield->get_corrected_dimensions()[temp_index]->get_size() == latsize) { 
            latname_index = temp_index;
            latname = sdfield->get_corrected_dimensions()[temp_index]->get_name();
        }
        else if (-1 == lonname_index && sdfield->get_corrected_dimensions()[temp_index]->get_size() == lonsize) {
            lonname_index = temp_index;
            lonname = sdfield->get_corrected_dimensions()[temp_index]->get_name();
        }
    }
    return (latname_index + lonname_index == 1);

}

//  Prepare the handling of TRMM version 7 level 2 products. Prepare dims and cvs.
void prepare_trmm_v7_l2() {

    if(file->getSPType() != TRMML2_V7)
        return;

    // 1. Obtain the geolocation field: type,dimension size and dimension name
    // 2. Create latitude and longtiude fields according to the geolocation field.
    std::string tempdimname1, tempdimname2;
    std::string tempnewdimname1, tempnewdimname2;
    //std::string temppath2;

    // Create a temporary map from the dimension size to the dimension name
    std::set < int32 > tempdimsizeset;
    std::map < int32, std::string > tempdimsizenamelist;
    std::map < int32, std::string >::iterator tempsizemapit;
    std::pair < std::set < int32 >::iterator, bool > tempsetit;

    // Reduce the fakeDim list. FakeDim is found to be used by TRMM team.
    // Note: maybe redundant. Reducing FakeDim is handled by build_dim_name_list and suqueeze_dim_name_list. Will evaluate it in the next release.(KY 2014-12-25)
    for (std::vector < hdf4_var_sdfield * >::const_iterator i =
        file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {
        for (std::vector < hdf4_dim * >::const_iterator j =
            (*i)->get_corrected_dimensions ().begin ();
            j != (*i)->get_corrected_dimensions ().end (); ++j) {
            if (((*j)->get_name ()).find ("fakeDim") == std::string::npos) {	//No fakeDim in the string
                tempsetit = tempdimsizeset.insert ((*j)->get_size ());
                if (tempsetit.second == true)
                    tempdimsizenamelist[(*j)->get_size ()] = (*j)->get_name ();
	    }
        }
    }

    // Store the newname with the string of the full path after the first group.
    // The first level of the group hierarchy just stores name like Grid or Swath. It is not useful for CF.
    // The second level stores the real data information such as navigation etc.
    for (std::vector < hdf4_var_sdfield * >::const_iterator i =
        file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {

        std::string temp_name = (*i)->newname.substr(1) ;
        size_t temp_pos = temp_name.find_first_of('/');
        if (temp_pos !=std::string::npos) 
            (*i)->newname = temp_name.substr(temp_pos+1);
    }


    for (std::vector < hdf4_var_sdfield * >::const_iterator i =
         file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {

        if((*i)->get_name() == "Latitude") {
            if((*i)->get_rank() ==2) {

                tempnewdimname1 =
                    ((*i)->get_corrected_dimensions ())[0]->get_name ();
                tempnewdimname2 =
                    ((*i)->get_corrected_dimensions ())[1]->get_name ();
            }

            (*i)->fieldtype = 1;

        }
        else if ((*i)->get_name() == "Longitude") {
            (*i)->fieldtype = 2;

        }
        else {

            // Use the temp. map (size to name) to replace the name of "fakeDim???" with the dimension name having the same dimension length
            // This is done only for TRMM. It should be evaluated if this can be applied to other products.
            for (std::vector < hdf4_dim * >::const_iterator k =
                (*i)->get_corrected_dimensions ().begin ();
                k != (*i)->get_corrected_dimensions ().end (); ++k) {
                size_t fakeDimpos = ((*k)->get_name ()).find ("fakeDim");

                if (fakeDimpos != std::string::npos) {
                    tempsizemapit =
                        tempdimsizenamelist.find ((*k)->get_size ());
                    if (tempsizemapit != tempdimsizenamelist.end ())
                        (*k)->rename(tempdimsizenamelist[(*k)->get_size ()]);// Change the dimension name
                }
            }
        
        }
    }

    // Create the <dimname,coordinate variable> map from the corresponding dimension names to the latitude and the longitude
    if(tempnewdimname1.empty()!=true)
        file->sd->nonmisscvdimnamelist.insert (tempnewdimname1);

    if(tempnewdimname2.empty()!=true)
        file->sd->nonmisscvdimnamelist.insert (tempnewdimname2);

    std::string base_filename;
    size_t last_slash_pos = (file->get_file_name()).find_last_of("/");
    if(last_slash_pos != std::string::npos)
        base_filename = (file->get_file_name()).substr(last_slash_pos+1);
    if(""==base_filename)
       base_filename = file->get_file_name();

    // 2A12 has a 3rd-dimension variable called nlayer. We need to provide it.
    if(base_filename.find("2A12")!=std::string::npos) {

        hdf4_var_sdfield *nlayer = NULL;
        std::string nlayer_name ="nlayer";

        for (std::vector < hdf4_var_sdfield * >::iterator i =
            file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {

            bool has_nlayer = false;

            for (std::vector < hdf4_dim * >::const_iterator k =
                (*i)->get_dimensions ().begin ();
                k != (*i)->get_dimensions ().end (); ++k) {

                if((*k)->get_size() == 28 && (*k)->get_name() == nlayer_name) {
                 
                    nlayer = new hdf4_var_sdfield();
                    nlayer->rename(nlayer_name);
                    nlayer->rank = 1;
                    nlayer->type = DFNT_FLOAT32;
                    nlayer->fieldtype = 6;

                    nlayer->newname = nlayer->get_name() ;
                    hdf4_dim *dim =
                            new hdf4_dim (nlayer->get_name(), nlayer->get_name(),(*k)->get_size (), 0);
                    nlayer->dims.push_back(dim);

                    dim = new hdf4_dim(nlayer->get_name(),nlayer->get_name(),(*k)->get_size(),0);
                    nlayer->correcteddims.push_back(dim);
                
                    // Assign values for the nlayer
                    nlayer->value.resize(sizeof(float)*((*k)->get_size()));
                    std::vector<float>total_val;
                    total_val.resize((*k)->get_size());
                    for (int val_i = 0; val_i<20;val_i++)
                        total_val[val_i] = 0.5*(val_i+1);

                    for (int val_i = 20; val_i<28;val_i++)
                        total_val[val_i] = total_val[19]+(val_i-19);

                    // NOTE: strvalue of the field is not assigned. They are not used in the final output.
                    memcpy((void*)&(nlayer->value)[0],(const void*)(&total_val[0]),sizeof(float)*((*k)->get_size()));

                    has_nlayer = true;
                    break;
                }

            }

            if(true == has_nlayer)
              break;
        }

        if(nlayer !=NULL) {
            file->sd->sdfields.push_back(nlayer);
            file->sd->nonmisscvdimnamelist.insert (nlayer_name);
        }
    }
}

// Prepare TRMM version 7 single grid. Provide dims. and cvs.
void prepare_trmm_v7_l3s() {

    if(file->getSPType() != TRMML3S_V7)
        return;

    for (std::vector < hdf4_var_sdfield * >::iterator i =
        file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {

        //According to GES DISC, the next three variables should be removed from the list.
        if((*i)->get_name() == "InputFileNames") {
            delete (*i);
            file->sd->sdfields.erase(i); 
            i--;
        }
        else if((*i)->get_name() == "InputAlgorithmVersions") {
            delete (*i);
            file->sd->sdfields.erase(i);
            i--;
        }
        else if((*i)->get_name() == "InputGenerationDateTimes") {
            delete (*i);
            file->sd->sdfields.erase(i);
            i--;
        }
        else // Just use SDS names and for performance reasons, change them here.
            (*i)->newname = (*i)->get_name();
    }

    // Add the 3rd dimesnion nlayer.
    hdf4_var_sdfield *nlayer = NULL;
    std::string nlayer_name ="nlayer";

    for (std::vector < hdf4_var_sdfield * >::iterator i =
        file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {

        bool has_nlayer = false;

        for (std::vector < hdf4_dim * >::const_iterator k =
            (*i)->get_dimensions ().begin ();
            k != (*i)->get_dimensions ().end (); ++k) {

            if((*k)->get_size() == 28 && (*k)->get_name() == nlayer_name) {
                 
                nlayer = new hdf4_var_sdfield();
                nlayer->rename(nlayer_name);
                nlayer->rank = 1;
                nlayer->type = DFNT_FLOAT32;
                nlayer->fieldtype = 6;

                nlayer->newname = nlayer->get_name() ;
                hdf4_dim *dim =
                        new hdf4_dim (nlayer->get_name(), nlayer->get_name(),(*k)->get_size (), 0);
                nlayer->dims.push_back(dim);

                dim = new hdf4_dim(nlayer->get_name(),nlayer->get_name(),(*k)->get_size(),0);
                nlayer->correcteddims.push_back(dim);

                // Assign values for the nlayer
                nlayer->value.resize(sizeof(float)*((*k)->get_size()));
                std::vector<float>total_val;
                total_val.resize((*k)->get_size());
                for (int i = 0; i<20;i++)
                    total_val[i] = 0.5*(i+1);

                for (int i = 20; i<28;i++)
                    total_val[i] = total_val[19]+(i-19);

                // NOTE: strvalue of the field is not assigned. They are not used in the final output.
                memcpy((void*)&(nlayer->value)[0],(const void*)(&total_val[0]),sizeof(float)*((*k)->get_size()));
                
                has_nlayer = true;
                break;
            }

        }

        if(true == has_nlayer)
          break;
    }

    if(nlayer !=NULL) {
        file->sd->sdfields.push_back(nlayer);
        file->sd->nonmisscvdimnamelist.insert (nlayer_name);
    }
        
    // Prepare latitude and longitude.
    int latsize = 0;
    int lonsize = 0;
    float lat_start = 0;
    float lon_start = 0.;
    float lat_res = 0.;
    float lon_res = 0.;


    // Obtain lat/lon resolution, starting value and number of elements.
    Obtain_TRMML3S_V7_latlon_paras(latsize,lonsize,lat_start,lon_start,lat_res,lon_res);

    std::string latname;
    std::string lonname;

    // All fields share the same lat/lon names. So we just check one field.
    bool llname_found = false;
    for (std::vector < hdf4_var_sdfield * >::iterator i =
        file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {

        if(2 == (*i)->get_rank()) {

            llname_found = Obtain_TRMM_V7_latlon_name((*i),latsize,lonsize,latname,lonname);
            if (true == llname_found)
                break;

        }
    }

    // Create lat/lon SD fields.
    hdf4_var_sdfield* longitude = new hdf4_var_sdfield ();
    longitude->rename(lonname);
    longitude->rank = 1;
    longitude->type = DFNT_FLOAT32;
    longitude->fieldtype = 2;

    longitude->newname = longitude->get_name();
    hdf4_dim *dim =
                    new hdf4_dim (lonname, lonname,lonsize, 0);
    longitude->dims.push_back (dim);

    dim = new hdf4_dim (lonname, lonname, lonsize, 0);
    longitude->correcteddims.push_back (dim);
    file->sd->sdfields.push_back(longitude);

    // Assign longitude value
    longitude->value.resize(lonsize*sizeof(float));
    std::vector<float>lon_vals;
    lon_vals.resize(lonsize);
    for(int i = 0; i <lonsize; i++)
        lon_vals[i] = lon_start + i*lon_res;

    // NOTE: strvalue of the field is not assigned. They are not used in the final output.
    memcpy((void*)&(longitude->value)[0],(const void*)(&lon_vals[0]),sizeof(float)*lonsize);

    hdf4_var_sdfield* latitude = new hdf4_var_sdfield ();
    latitude->rename(latname);
    latitude->rank = 1;
    latitude->type = DFNT_FLOAT32;
    latitude->fieldtype = 1;

    latitude->newname = latitude->get_name();
    dim = new hdf4_dim (latname, latname,latsize, 0);
    latitude->dims.push_back (dim);

    dim = new hdf4_dim (latname, latname,latsize, 0);
    latitude->correcteddims.push_back (dim);
    file->sd->sdfields.push_back(latitude);

    // Assign latitude value
    latitude->value.resize(latsize*sizeof(float));
    std::vector<float>lat_vals;
    lat_vals.resize(latsize);
    for(int i = 0; i <latsize; i++)
        lat_vals[i] = lat_start + i*lat_res;

    // NOTE: strvalue of the field is not assigned. They are not used in the final output.
    memcpy((void*)&(latitude->value)[0],(const void*)(&lat_vals[0]),sizeof(float)*latsize);

    // Create the <dimname,coordinate variable> map from the corresponding dimension names to the latitude and the longitude
    file->sd->nonmisscvdimnamelist.insert (latname);
    file->sd->nonmisscvdimnamelist.insert (lonname);

    // Now we want to handle the special CVs for 3A26 products. Since these special CVs only apply to the 3A26 products,
    //  we don't want to find them from other products to reduce the performance cost. So here I simply check the filename
    //  At the time of writing the code, the info. can be found at http://pps.gsfc.nasa.gov/Documents/filespec.TRMM.V7.pdf
    std::string base_filename;
    size_t last_slash_pos = (file->get_file_name()).find_last_of("/");
    if(last_slash_pos != std::string::npos)
        base_filename = (file->get_file_name()).substr(last_slash_pos+1);
    if(""==base_filename)
        base_filename = file->get_file_name();

    if(base_filename.find("3A26")!=std::string::npos) {

        bool ZOflag = false;
        bool SRTflag = false;
        bool HBflag = false;
        std::string nthrsh_base_name = "nthrsh";
        std::string nthrsh_zo_name ="nthrshZO";
        std::string nthrsh_hb_name ="nthrshHB";
        std::string nthrsh_srt_name ="nthrshSRT";

        hdf4_var_sdfield* nthrsh_zo = NULL;
        hdf4_var_sdfield* nthrsh_hb = NULL;
        hdf4_var_sdfield* nthrsh_srt = NULL;
        
        for (std::vector < hdf4_var_sdfield * >::iterator i =
            file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {

            if(ZOflag != true) {
                if((*i)->get_name().find("Order")!=std::string::npos) {
                    for (std::vector < hdf4_dim * >::const_iterator k =
                         (*i)->get_dimensions ().begin ();
                         k != (*i)->get_dimensions ().end (); ++k) {

                        if((*k)->get_size() == 6  && (*k)->get_name() == nthrsh_base_name) {
                            nthrsh_zo = new hdf4_var_sdfield();
                            nthrsh_zo->rename(nthrsh_zo_name);
                            nthrsh_zo->rank = 1;
                            nthrsh_zo->type = DFNT_FLOAT32;
                            nthrsh_zo->fieldtype = 6;

                            nthrsh_zo->newname = nthrsh_zo->get_name() ;
                            hdf4_dim *temp_dim =
                                    new hdf4_dim (nthrsh_zo->get_name(), nthrsh_zo->get_name(),(*k)->get_size (), 0);
                            nthrsh_zo->dims.push_back(temp_dim);

                            temp_dim = new hdf4_dim(nthrsh_zo->get_name(),nthrsh_zo->get_name(),(*k)->get_size(),0);
                            nthrsh_zo->correcteddims.push_back(temp_dim);

                            // Assign values for the nthrsh_zo
                            nthrsh_zo->value.resize(sizeof(float)*((*k)->get_size()));
                            std::vector<float>total_val;
                            total_val.resize((*k)->get_size());
                            total_val[0] = 0.1;
                            total_val[1] = 0.2;
                            total_val[2] = 0.3;
                            total_val[3] = 0.5;
                            total_val[4] = 0.75;
                            total_val[5] = 50;

                            // NOTE: strvalue of the field is not assigned. They are not used in the final output.
                            memcpy((void*)&(nthrsh_zo->value)[0],(const void*)(&total_val[0]),sizeof(float)*((*k)->get_size()));
 
                            ZOflag = true;
 
                        }
                    }
                }
                
            }

            else if(SRTflag != true) {
                if((*i)->get_name().find("2A25")!=std::string::npos) {

                    for (std::vector < hdf4_dim * >::const_iterator k =
                         (*i)->get_dimensions ().begin ();
                         k != (*i)->get_dimensions ().end (); ++k) {

                        if((*k)->get_size() == 6  && (*k)->get_name() == nthrsh_base_name) {
                            nthrsh_srt = new hdf4_var_sdfield();
                            nthrsh_srt->rename(nthrsh_srt_name);
                            nthrsh_srt->rank = 1;
                            nthrsh_srt->type = DFNT_FLOAT32;
                            nthrsh_srt->fieldtype = 6;

                            nthrsh_srt->newname = nthrsh_srt->get_name() ;
                            hdf4_dim *temp_dim =
                                    new hdf4_dim (nthrsh_srt->get_name(), nthrsh_srt->get_name(), (*k)->get_size (), 0);
                            nthrsh_srt->dims.push_back(temp_dim);

                            temp_dim = new hdf4_dim(nthrsh_srt->get_name(),nthrsh_srt->get_name(),(*k)->get_size(),0);
                            nthrsh_srt->correcteddims.push_back(temp_dim);

                            // Assign values for the nthrsh_srt
                            nthrsh_srt->value.resize(sizeof(float)*((*k)->get_size()));
                            std::vector<float>total_val;
                            total_val.resize((*k)->get_size());
                            total_val[0] = 1.5;
                            total_val[1] = 1.0;
                            total_val[2] = 0.8;
                            total_val[3] = 0.6;
                            total_val[4] = 0.4;
                            total_val[5] = 0.1;
                            
                            // NOTE: strvalue of the field is not assigned. They are not used in the final output.
                            memcpy((void*)&(nthrsh_srt->value)[0],(const void*)(&total_val[0]),sizeof(float)*((*k)->get_size()));
 
                            SRTflag = true;
 
                        }
                    }
                }
            }
            else if(HBflag != true) {
                if((*i)->get_name().find("hb")!=std::string::npos || (*i)->get_name().find("HB")!=std::string::npos) {

                    for (std::vector < hdf4_dim * >::const_iterator k =
                         (*i)->get_dimensions ().begin ();
                         k != (*i)->get_dimensions ().end (); ++k) {

                        if((*k)->get_size() == 6  && (*k)->get_name() == nthrsh_base_name) {

                            nthrsh_hb = new hdf4_var_sdfield();
                            nthrsh_hb->rename(nthrsh_hb_name);
                            nthrsh_hb->rank = 1;
                            nthrsh_hb->type = DFNT_FLOAT32;
                            nthrsh_hb->fieldtype = 6;

                            nthrsh_hb->newname = nthrsh_hb->get_name() ;
                            hdf4_dim *temp_dim =
                                    new hdf4_dim (nthrsh_hb->get_name(), nthrsh_hb->get_name(),(*k)->get_size (), 0);
                            nthrsh_hb->dims.push_back(temp_dim);

                            temp_dim = new hdf4_dim(nthrsh_hb->get_name(),nthrsh_hb->get_name(),(*k)->get_size(),0);
                            nthrsh_hb->correcteddims.push_back(temp_dim);

                            // Assign values for the nthrsh_srt
                            nthrsh_hb->value.resize(sizeof(float)*((*k)->get_size()));
                            std::vector<float>total_val;
                            total_val.resize((*k)->get_size());
                            total_val[0] = 0.1;
                            total_val[1] = 0.2;
                            total_val[2] = 0.3;
                            total_val[3] = 0.5;
                            total_val[4] = 0.75;
                            total_val[5] = 0.9999;

                            // NOTE: strvalue of the field is not assigned. They are not used in the final output.
                            memcpy((void*)&(nthrsh_hb->value)[0],(const void*)(&total_val[0]),sizeof(float)*((*k)->get_size()));
 
                            HBflag = true;
 
                        }
                    }
                }
            }
        }


        for (std::vector < hdf4_var_sdfield * >::iterator i =
            file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {

            if((*i)->get_name().find("Order")!=std::string::npos && ZOflag == true) {
                for (std::vector < hdf4_dim * >::const_iterator k =
                     (*i)->get_dimensions ().begin ();
                     k != (*i)->get_dimensions ().end (); ++k) {

                    if((*k)->get_size() == 6  && (*k)->get_name() == nthrsh_base_name) {
                       (*k)->rename(nthrsh_zo_name);
                       break;
                    }
                }

                for (std::vector < hdf4_dim * >::const_iterator k =
                    (*i)->get_corrected_dimensions ().begin ();
                    k != (*i)->get_corrected_dimensions ().end (); ++k) {
                    if((*k)->get_size() == 6  && (*k)->get_name() == nthrsh_base_name) {
                       (*k)->rename(nthrsh_zo_name);
                       break;
                    }
                }
 
            }

            else if(((*i)->get_name().find("hb")!=std::string::npos || (*i)->get_name().find("HB")!=std::string::npos)&& HBflag == true) {
                for (std::vector < hdf4_dim * >::const_iterator k =
                     (*i)->get_dimensions ().begin ();
                     k != (*i)->get_dimensions ().end (); ++k) {

                    if((*k)->get_size() == 6  && (*k)->get_name() == nthrsh_base_name) {
                       (*k)->rename(nthrsh_hb_name);
                       break;
                    }
                }

                for (std::vector < hdf4_dim * >::const_iterator k =
                    (*i)->get_corrected_dimensions ().begin ();
                    k != (*i)->get_corrected_dimensions ().end (); ++k) {
                    if((*k)->get_size() == 6  && (*k)->get_name() == nthrsh_base_name) {
                       (*k)->rename(nthrsh_hb_name);
                       break;
                    }
                }
 
            }
            else if(((*i)->get_name().find("2A25")!=std::string::npos) && SRTflag == true) {
                for (std::vector < hdf4_dim * >::const_iterator k =
                     (*i)->get_dimensions ().begin ();
                     k != (*i)->get_dimensions ().end (); ++k) {

                    if((*k)->get_size() == 6  && (*k)->get_name() == nthrsh_base_name) {
                       (*k)->rename(nthrsh_srt_name);
                       break;
                    }
                }

                for (std::vector < hdf4_dim * >::const_iterator k =
                    (*i)->get_corrected_dimensions ().begin ();
                    k != (*i)->get_corrected_dimensions ().end (); ++k) {
                    if((*k)->get_size() == 6  && (*k)->get_name() == nthrsh_base_name) {
                       (*k)->rename(nthrsh_srt_name);
                       break;
                    }
                }
 
            }


        }

        if(nthrsh_zo !=NULL) {
            file->sd->sdfields.push_back(nthrsh_zo);
            file->sd->nonmisscvdimnamelist.insert (nthrsh_zo_name);
        }

        if(nthrsh_hb !=NULL) {
            file->sd->sdfields.push_back(nthrsh_hb);
            file->sd->nonmisscvdimnamelist.insert (nthrsh_hb_name);
        }

        if(nthrsh_srt !=NULL) {
            file->sd->sdfields.push_back(nthrsh_srt);
            file->sd->nonmisscvdimnamelist.insert (nthrsh_srt_name);
        }

    }
    
}

// Prepare TRMM version 7 multiple grids. Provide dims. and cvs.
void prepare_trmm_v7_l3m()  {

    if(file->getSPType() != TRMML3M_V7)
        return;  

    // Remove unuseful variables according to GES DISC.
    for (std::vector < hdf4_var_sdfield * >::iterator i =
        file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) {

        if((*i)->get_name() == "InputFileNames") {
            delete (*i);
            file->sd->sdfields.erase(i); 
            i--;
        }
        else if((*i)->get_name() == "InputAlgorithmVersions") {
            delete (*i);
            file->sd->sdfields.erase(i);
            i--;
        }
        else if((*i)->get_name() == "InputGenerationDateTimes") {
            delete (*i);
            file->sd->sdfields.erase(i);
            i--;
        }
    }

    
#if 0
    NOTE for programming: 
    1. Outer loop: loop global attribute for GridHeader?. Retrieve ? as a number for index.
    1.5. Obtain the lat/lon sizes for this grid.
    The following steps are to retrieve lat/lon names for this grid.
    2. Inner loop: Then loop through the field
    3. Check the field rank, 
        3.1 if the rank is not 2, (if the index is the first index, change the newname to name )
               continue.
        3.2 else {
               3.2.1 Retrieve the index from the field new name(retrieve last path Grid1 then retrieve 1)
               3.2.2 If the index from the field is the same as that from the GridHeader, continue checking 
                     the lat/lon name for this grid as the single grid.
                     change the newname to name.
            }
#endif
    
    // The following code tries to be performance-friendly by looping through the fields and handling the operations
    // as less as I can.
    int first_index = -1;
    for (std::vector < hdf4_attr * >::const_iterator i =
        file->sd->get_attrs ().begin ();
        i != file->sd->get_attrs ().end (); ++i) {
            
        if ((*i)->get_name ().find("GridHeader")==0) {
            std::string temp_name = (*i)->get_name();

            // The size of "GridHeader" is 10, no need to calculate.
            std::string str_num = temp_name.substr(10);
            std::stringstream ss_num(str_num);

            int grid_index;
            ss_num >> grid_index; 

            if ( -1 == first_index)
                first_index = grid_index;
            
            float lat_start = 0.;
            float lon_start = 0.;
            float lat_res = 1.;
            float lon_res = 1.;
            int latsize = 0;
            int lonsize = 0;
            
            parser_trmm_v7_gridheader((*i)->get_value(),latsize,lonsize,
                                                 lat_start,lon_start,
                                                 lat_res, lon_res, false);
          
            std::string latname;
            std::string lonname;
            
            bool llname_found = false;
            for (std::vector < hdf4_var_sdfield * >::iterator temp_it =
                 file->sd->sdfields.begin (); temp_it != file->sd->sdfields.end (); ++temp_it) {
 
                // Just loop the 2-D fields to find the lat/lon size
                if(2 == (*temp_it)->get_rank()) {

                    // If a grid has not been visited, we will check the fields attached to this grid.
                    if ((*temp_it)->newname !=(*temp_it)->get_name()) {

                        std::string temp_field_full_path = (*temp_it)->get_new_name();
                        size_t last_path_pos = temp_field_full_path.find_last_of('/');
                        char str_index = temp_field_full_path[last_path_pos-1];
                        if(grid_index ==(int)(str_index - '0')) {
                            if(llname_found != true) 
                                llname_found = Obtain_TRMM_V7_latlon_name((*temp_it),latsize,lonsize,latname,lonname);
                            (*temp_it)->newname = (*temp_it)->get_name();
                        }
                    }
                }
                else if (first_index == grid_index)
                    (*temp_it)->newname = (*temp_it)->get_name();
            }   

            // Create lat/lon SD fields.
            hdf4_var_sdfield* longitude = new hdf4_var_sdfield ();
            longitude->rename(lonname);
            longitude->rank = 1;
            longitude->type = DFNT_FLOAT32;
            longitude->fieldtype = 2;

            longitude->newname = longitude->get_name();
            hdf4_dim *temp_dim =
                    new hdf4_dim (lonname, lonname,lonsize, 0);
            longitude->dims.push_back (temp_dim);

            temp_dim = new hdf4_dim (lonname, lonname,lonsize, 0);
            longitude->correcteddims.push_back (temp_dim);

            // Assign longitude value
            longitude->value.resize(lonsize*sizeof(float));
            std::vector<float>lon_vals;
            lon_vals.resize(lonsize);
            for(int i = 0; i <lonsize; i++)
                lon_vals[i] = lon_start + i*lon_res;
            // NOTE: strvalue of the field is not assigned. They are not used in the final output.
            memcpy((void*)&(longitude->value)[0],(const void*)(&lon_vals[0]),sizeof(float)*lonsize);

            file->sd->sdfields.push_back(longitude);

            hdf4_var_sdfield* latitude = new hdf4_var_sdfield ();
            latitude->rename(latname);
            latitude->rank = 1;
            latitude->type = DFNT_FLOAT32;
            latitude->fieldtype = 1;

            latitude->newname = latitude->get_name();
            temp_dim = new hdf4_dim (latname, latname,latsize, 0);
            latitude->dims.push_back (temp_dim);

            temp_dim = new hdf4_dim (latname, latname,latsize, 0);
            latitude->correcteddims.push_back (temp_dim);

            // Assign latitude value
            latitude->value.resize(latsize*sizeof(float));
            std::vector<float>lat_vals;
            lat_vals.resize(latsize);
            for(int i = 0; i <latsize; i++)
                lat_vals[i] = lat_start + i*lat_res;

            // NOTE: strvalue of the field is not assigned. They are not used in the final output.
            memcpy((void*)&(latitude->value)[0],(const void*)(&lat_vals[0]),sizeof(float)*latsize);

            file->sd->sdfields.push_back(latitude);
     
            // Create the <dimname,coordinate variable> map from the corresponding dimension names to the latitude and the longitude
            file->sd->nonmisscvdimnamelist.insert (latname);
            file->sd->nonmisscvdimnamelist.insert (lonname);
            
        }  
    }
}

// Add latitude and longitude for OBPG L2.
void prepare_obpg_l2()
{
    if(file->getSPType() != OBPGL2)
        return;

    std::vector<char> v1;

    int num_scan_line_data = 0;
    int num_point_data = 0;

    std::string pixels_per_scan_line_name = "Pixels per Scan Line";
    std::string number_pixels_control_points = "Number of Pixel Control Points";
    std::string tempnewdimname1, tempnewdimname2;

    // 1. Obtain the expanded size of the latitude/longitude
    int nr=0;
    for (std::vector < hdf4_attr * >::const_iterator i = file->sd->get_attrs ().begin (); i != file->sd->get_attrs ().end () && nr<2; ++i) 
    {
        if ((*i)->get_name () == pixels_per_scan_line_name) 
        {
            int *attrvalueptr = (int *) (&((*i)->get_value ()[0]));
            num_scan_line_data = *attrvalueptr;
            ++nr;
        }
        if((*i)->get_name() == number_pixels_control_points)
        {
            int *attrvalueptr = (int *) (&((*i)->get_value ()[0]));
            num_point_data = *attrvalueptr;
            ++nr;
        }
    } // end of for

    pixels_per_scan_line_name = remove_special_chars("Pixels per Scan Line");
    number_pixels_control_points = remove_special_chars("Number of Pixel Control Points");

    //readobpgl2
    bool compmapflag = false;
    if (num_scan_line_data == num_point_data)
        compmapflag = true;

    if(num_point_data == 0)
        throw std::runtime_error("The number of points cannot be zero.(" __FILE__ ":" TOSTRING(__LINE__)")" );

    // 2. Obtain the latitude and longitude information. Assign the new dimension name and the dimension size
    int tempcountllflag = 0;

    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        v1.clear();

        if ((*i)->get_name () == "longitude" || (*i)->get_name () == "latitude")
        {
            if ((*i)->get_name () == "longitude")
                (*i)->fieldtype = 2;
            if ((*i)->get_name () == "latitude")
                (*i)->fieldtype = 1;

            tempcountllflag++;
            if ((*i)->get_rank() != 2)
            {
                std::cout <<"The lat/lon rank must be 2 in " << (*i)->get_name ();
                throw std::runtime_error("The lat/lon rank must be 2 for this field.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                //exit(1);
            }
	
            for (std::vector < hdf4_dim * >::const_iterator k = (*i)->get_dimensions ().begin ();  k != (*i)->get_dimensions ().end (); ++k) 
            {
                if ((*k)->get_name () == number_pixels_control_points) 
                {
                    (*k)->rename(pixels_per_scan_line_name);
                    (*k)->set_size(num_scan_line_data);
                    break;
                }
            } // end of for

            for (std::vector < hdf4_dim * >::const_iterator k = (*i)->get_corrected_dimensions ().begin (); k != (*i)->get_corrected_dimensions ().end (); ++k) 
            {
                if ((*k)->get_name ().find (number_pixels_control_points) != std::string::npos) 
                {
                    (*k)->rename(pixels_per_scan_line_name);
                    (*k)->set_size(num_scan_line_data);
                    if (tempcountllflag == 1)
                        tempnewdimname2 = (*k)->get_name();	
                } else 
                {
                    if (tempcountllflag == 1)
                        tempnewdimname1 = (*k)->get_name();
                } // end of else
            } // end  of for

            int nelms=1;
            for (std::vector < hdf4_dim * >::const_iterator k = (*i)->get_dimensions ().begin ();  k != (*i)->get_dimensions ().end (); ++k)
                nelms *= (*k)->get_size();

            //float32 *val = new float32[nelms];
            std::vector<float32>val;
            val.resize(nelms);

            if(compmapflag)
                continue;

            //readobpgl2
            std::vector<int32>orioffset32;
            std::vector<int32>oricount32;
            std::vector<int32>oristep32;

            int32 myrank = (*i)->get_rank();

            orioffset32.resize(myrank);
            oricount32.resize(myrank);
            oristep32.resize(myrank);

            int tempseg = 0;
            int tempnewseg = 0;
            int total_elm = 0;
	 	
            if (num_scan_line_data % num_point_data == 0)
                tempseg = num_scan_line_data / num_point_data;
            else
                tempseg = num_scan_line_data / num_point_data + 1;
	
            orioffset32[0] = 0;//offset32[0];
            oricount32[0] = (*i)->get_dimensions()[0]->get_size();//count32[0];
            oristep32[0] = 1;//step32[0];
		
            orioffset32[1] = /*offset32[1]*/0 / tempseg;
            if( tempseg > 1) //if ((int) step32[1] < tempseg)
                oristep32[1] = 1;
            else
                oristep32[1] = /*step32[1]*/1 / tempseg;
            tempnewseg = tempseg / /*step32[1]*/1;
            if (/*count32[1]*/(*i)->get_dimensions()[1]->get_size() / tempnewseg < 1)
                oricount32[1] = 2;
            else
                oricount32[1] = 1 + /*count32[1]*/(*i)->get_dimensions()[1]->get_size() / tempnewseg;

            total_elm = 1;
            for (int m = 0; m < myrank; m++)
                total_elm = total_elm * ((int) (oricount32[m]));
		
            int32 sdsindex = SDreftoindex (file->get_sd_id(), (int32) (*i)->sdsref);
            if(sdsindex==-1)
            {
                //std::cout << "SDS index " << sdsindex << " is not right." << std::endl;
                throw std::runtime_error("SDS index cannot be -1.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                //exit(1);
            }
            int32 sdsid = SDselect (file->get_sd_id(), sdsindex);
            if (sdsid<0) 
            {
                //std::cout << "SDselect failed." << std::endl;
                throw std::runtime_error("SDselect failed.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                //exit(1);
            }

            //float32 *orival = new float32[total_elm];
            std::vector<float32>orival;
            orival.resize(total_elm);

            int32 r = SDreaddata (sdsid, &orioffset32[0], &oristep32[0], &oricount32[0], &orival[0]);
            if(r!=0)
            {
                std::cout << "SDreaddata failed" << std::endl;
                throw std::runtime_error("SDreaddata failed.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                //exit(1);
            }	

            // Interpolate according to the fast changing dimension.
            float tempdiff = 0;
            int ii = 0;
            int jj = 0;
            int kk = 0;
            int kk2 = 0;
            int32 realcount2 = oricount32[1] - 1;

            for (ii = 0; ii < (int)(*i)->get_dimensions()[0]->get_size() /*count32[0]*/; ii++) 
            {
                for (jj = 0; jj < (int) realcount2 - 1; jj++) 
                {
                    tempdiff = orival[ii * (int) oricount32[1] + jj + 1] - orival[ii * (int) oricount32[1] + jj];
                    for (kk = 0; kk < tempnewseg; kk++) 
                    {
                        val[ii * (int)(*i)->get_dimensions()[1]->get_size() /*count32[1]*/ + jj * tempnewseg + kk] = orival[ii * (int) oricount32[1] + jj] + tempdiff / tempnewseg * kk;
                    }
                } // end of for
                tempdiff = orival[ii * (int) oricount32[1] + jj + 1] -orival[ii * (int) oricount32[1] + jj];
                // There are three cases: 1. when the oricount is 1 int lastseg = num_scan_line_data - tempnewseg*((int)oricount32[1]-1)-1;
                int lastseg = (int) ((*i)->get_dimensions()[1]->get_size()/*count32[1]*/ - tempnewseg * (int) (realcount2 - 1));
                for (kk2 = 0; kk2 < lastseg; kk2++)
                    val[ii * (int)(*i)->get_dimensions()[1]->get_size() /*count32[1]*/ + jj * tempnewseg + kk2] = orival[ii * (int) oricount32[1] + jj] + tempdiff / lastseg * kk2;
            } // end of for (ii = 0;
            std::string mystr;
            char c[100];
            int icount = 0;
            while (icount < nelms)
            {
                sprintf(c, "%.6f",  val[icount]);
                mystr.append(c);
                mystr.append(",");
			
                size_t t1 = v1.size();
                v1.resize(t1+4);
                float t2 = atof(c);
                memcpy((char*)&(v1)[t1], &t2, sizeof(t2));	
			
                icount++;
            } // end of while
            size_t ppp = mystr.find_last_of(",");
            (*i)->str_value = mystr.substr(0, ppp);
            (*i)->value.resize(v1.size());
            std::copy(v1.begin(), v1.end(), (*i)->value.begin());
        } // end of  if ((*i)->get_name () == "longitude" 
        if (tempcountllflag == 2)
            break;
    } // end of for

    // 3. Create the <dimname,coordinate variable> map from the corresponding dimension names to the latitude and the longitude. Obtain the corrected dimension names for latitude and longitude
    file->sd->nonmisscvdimnamelist.insert (tempnewdimname1);
    file->sd->nonmisscvdimnamelist.insert (tempnewdimname2);
}

// Add latitude and longitude for OBPG L3.
void prepare_obpg_l3()
{
    if(file->getSPType() != OBPGL3)
        return;

    // 1. Obtain the global attributes for calculating latitude and longitude. 
    std::string num_lat_name = "Number of Lines";
    std::string num_lon_name = "Number of Columns";
    std::string lat_step_name = "Latitude Step";
    std::string lon_step_name = "Longitude Step";
    std::string swp_lat_name = "SW Point Latitude";
    std::string swp_lon_name = "SW Point Longitude";

    // Read file attributes to otain the segment
    float32 lat_step = 0;
    float32 lon_step = 0;
    float32 swp_lat = 0;
    float32 swp_lon = 0;

    int32 num_lat = 0;
    int32 num_lon = 0;
    int tempcountllflag = 0;

    for (std::vector < hdf4_attr * >::const_iterator i = file->sd->get_attrs ().begin (); i != file->sd->get_attrs ().end (); ++i) 
    {
        if ((*i)->get_name () == num_lon_name) 
        {
            // Check later if float can be changed to float32
            int *attrvalue = (int *) (&((*i)->get_value ()[0]));
            num_lon = *attrvalue;
            tempcountllflag++;
        }

        if ((*i)->get_name () == num_lat_name) 
        {
            int *attrvalue = (int *) (&((*i)->get_value ()[0]));
            num_lat = *attrvalue;
            tempcountllflag++;
        }
        if ((*i)->get_name () == lat_step_name)
        {
            float32 *attrvalueptr = (float32 *)(&((*i)->get_value ()[0]));
            lat_step = *attrvalueptr;
            tempcountllflag++;
        }
        if((*i)->get_name() == lon_step_name)
        {
            float32 *attrvalueptr = (float32 *)(&((*i)->get_value ()[0]));
            lon_step = *attrvalueptr;
            tempcountllflag++;
        }
        if ((*i)->get_name () == swp_lat_name)
        {
            float32 *attrvalueptr = (float32 *)(&((*i)->get_value ()[0]));
            swp_lat = *attrvalueptr;
            tempcountllflag++;
        }
        if((*i)->get_name() == swp_lon_name)
        {
            float32 *attrvalueptr = (float32 *)(&((*i)->get_value ()[0]));
            swp_lon = *attrvalueptr;
            tempcountllflag++;                
        }	
        if (tempcountllflag == 6)
            break;
    } // end of for

    std::string num_lon_name2 = remove_special_chars(num_lon_name);
    std::string num_lat_name2 = remove_special_chars(num_lat_name);

    std::vector<char> v1;
    std::vector<char> v2;

    // Longitude
    hdf4_var_sdfield *longitude = new hdf4_var_sdfield ();

    longitude->rename("longitude");
    longitude->rank = 1;
    longitude->type = DFNT_FLOAT32;
    longitude->fieldtype = 2;

    // No need to assign fullpath, in this case, only one SDS under one file. If finding other OBPGL3 data, will handle then.
    longitude->newname = longitude->get_name();
    hdf4_dim *dim = new hdf4_dim (num_lon_name2, num_lon_name2,num_lon, 0);

    longitude->dims.push_back (dim);

    // Add the corrected dimension name only to be consistent with general handling of other cases.
    dim = new hdf4_dim (num_lon_name2, num_lon_name2,num_lon, 0);
    longitude->correcteddims.push_back (dim);

    { // 1
        int nelms = (int)num_lon;
        float32 *allval = new float32[num_lon];

        for(int jj = 0; jj < num_lon; jj++)
            allval[jj] = swp_lon + jj * lon_step;

        float32 *val = new float32[nelms];

        for (int kk = 0; kk < nelms; kk++)
            val[kk] = allval[(int) (/*offset[0]*/0 + /*step[0]*/1 * kk)];

        std::string mystr;
        char c[100];
        int icount = 0;
        while (icount < nelms)
        {
            sprintf(c, "%.6f",  val[icount]);
            mystr.append(c);
            mystr.append(",");
	
            size_t t1 = v1.size();
            v1.resize(t1+4);
            float t2 = atof(c);
            memcpy((char*)&(v1)[t1], &t2, sizeof(t2));
	
            icount++;
        } // end of while
        size_t ppp = mystr.find_last_of(",");
        longitude->str_value = mystr.substr(0, ppp);
        longitude->value.resize(v1.size());
        std::copy(v1.begin(), v1.end(), longitude->value.begin());

        delete[](float32 *) val;
        delete[](float32 *) allval;
    } // ene of 1

    hdf4_var_sdfield *latitude = new hdf4_var_sdfield ();

    latitude->rename("latitude");
    latitude->rank = 1;
    latitude->type = DFNT_FLOAT32;
    latitude->fieldtype = 1;

    // No need to assign fullpath, in this case, only one SDS under one file. If finding other OBPGL3 data, will handle then.
    latitude->newname = latitude->get_name();
    dim = new hdf4_dim (num_lat_name2, num_lat_name2,num_lat, 0);
    latitude->dims.push_back (dim);

    // Add the corrected dimension name only to be consistent with general handling of other cases.
    dim = new hdf4_dim (num_lat_name2, num_lat_name2, num_lat, 0);
    latitude->correcteddims.push_back (dim);

    //readobpgl3
    {
        int nelms = (int)num_lat;
        float32 *allval = new float32[num_lat];
        for (int jj = 0; jj < num_lat; jj++)
            allval[jj] = (num_lat - jj - 1) * lat_step + swp_lat;

        float32 *val = new float32[nelms];

        for (int kk = 0; kk < nelms; kk++)
            val[kk] = allval[(int) (/*offset[0]*/0 + /*step[0]*/1 * kk)];

        std::string mystr;
        char c[100];
        int icount = 0;
        while (icount < nelms)
        {
            sprintf(c, "%.6f",  val[icount]);
            mystr.append(c);
            mystr.append(",");
	
            size_t t1 = v2.size();
            v2.resize(t1+4);
            float t2 = atof(c);
            memcpy((char*)&(v2)[t1], &t2, sizeof(t2));

            icount++;
        } // end of while
        size_t ppp = mystr.find_last_of(",");
        latitude->str_value = mystr.substr(0, ppp);
        latitude->value.resize(v2.size());
        std::copy(v2.begin(), v2.end(), latitude->value.begin());

        delete[](float32 *) val;
        delete[](float32 *) allval;
    } // end of readobpgl3

    // The dimension names of the SDS are fakeDim, so need to change them to dimension names of latitude and longitude
    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        if ((*i)->get_rank () != 2) 
        {
            std::cout << "The lat/lon rank must be 2 in " << (*i)->get_name () << std::endl;
            throw std::runtime_error("The lat/lon rank error.(" __FILE__ ":" TOSTRING(__LINE__)")" );
            //exit(1);
        }

        for (std::vector < hdf4_dim * >::const_iterator k = (*i)->get_dimensions ().begin (); k != (*i)->get_dimensions ().end (); ++k) 
        {
            if ((((*k)->get_name ()).find ("fakeDim")) != std::string::npos)
            {
                if ((*k)->get_size () == num_lon)
                    (*k)->rename(num_lon_name2);
                if ((*k)->get_size () == num_lat)
                    (*k)->rename(num_lat_name2);
            } // end of if
        } // end of for
        for (std::vector < hdf4_dim * >::const_iterator k = (*i)->get_corrected_dimensions ().begin (); k != (*i)->get_corrected_dimensions ().end (); ++k) 
        {
            if ((((*k)->get_name ()).find ("fakeDim")) != std::string::npos)
            {
                if ((*k)->get_size () == num_lon)
                    (*k)->rename(num_lon_name2);
                if ((*k)->get_size () == num_lat)
                    (*k)->rename(num_lat_name2);
            } // end of if
        } // end of for
    } // end of for 
    file->sd->sdfields.push_back (latitude);
    file->sd->sdfields.push_back (longitude);

    // Set dimname,coordinate variable list
    file->sd->nonmisscvdimnamelist.insert (num_lat_name2);
    file->sd->nonmisscvdimnamelist.insert (num_lon_name2);
}

// This applies to CERES AVG and SYN(CER_AVG_??? and CER_SYN_??? cases)
// Latitude and longitude are provided; some redundant CO-Latitude and longitude are removed from the final DDS.

static void set_float_value(hdf4_var_sdfield *i)
{
    char c[100];
    i->generate_str_value(file->get_sd_id()); 
    std::string s = i->get_str_value(); 
    size_t prepos=0; 
    size_t pos=s.find(","); 
    std::string astr; 
    std::vector<char> v1; 
    while(pos!=std::string::npos) 
    { 
        float32 v = (float32)atof(s.substr(prepos, pos-prepos).c_str());
        switch (i->fieldtype) 
        { 
            case 1: v=90-v; 
                    break; 
            case 2: v = (v>180.0)? v-360.0: v; 
                    break; 
        } // end of switch 
        sprintf(c, "%.6f", v); 
        astr.append(c); 
        astr.append(","); 
        size_t t1 = v1.size(); 
        v1.resize(t1+4); 
        float t2 = atof(c); 
        memcpy((char*)&(v1)[t1], &t2, sizeof(t2)); 
        prepos = pos+1; 
        pos = s.find(",", prepos); 
    } // end of while 
    pos = astr.find_last_of(",");
    i->str_value = astr.substr(0, pos-1); 
    i->value.resize(v1.size()); 
    std::copy(v1.begin(), v1.end(), i->value.begin()); 
}

static void set_double_value(hdf4_var_sdfield *i)
{
    char c[100];
    i->generate_str_value(file->get_sd_id()); 
    std::string s = i->get_str_value(); 
    size_t prepos=0; 
    size_t pos=s.find(","); 
    std::string astr; 
    std::vector<char> v1; 
    while(pos!=std::string::npos) 
    { 
        float64 v = (float64)atof(s.substr(prepos, pos-prepos).c_str());
        switch (i->fieldtype) 
        { 
            case 1: v=90-v; 
                    break; 
            case 2: v = (v>180.0)? v-360.0: v; 
                    break; 
        } // end of switch 
        sprintf(c, "%8.3e", v); 
        astr.append(c); 
        astr.append(","); 
        size_t t1 = v1.size(); 
        v1.resize(t1+4); 
        float t2 = atof(c); 
        memcpy((char*)&(v1)[t1], &t2, sizeof(t2)); 
        prepos = pos+1; 
        pos = s.find(",", prepos); 
    } // end of while 
    pos = astr.find_last_of(","); 
    i->str_value = astr.substr(0, pos-1); 
    i->value.resize(v1.size()); 
    std::copy(v1.begin(), v1.end(), i->value.begin());
} 

// Add latitude and longitude for CER_AVG or CER_SYN.
void prepare_cer_avg_syn()
{
    if(!(file->getSPType()==CER_AVG || file->getSPType() ==CER_SYN))
        return;

    bool colatflag = false;
    bool lonflag = false;

    std::string tempnewdimname1, tempnewdimname2;
    std::string tempcvarname1, tempcvarname2;


    std::vector < hdf4_var_sdfield * >::iterator beerasedit;

    for (std::vector < hdf4_var_sdfield * >::iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        // This product uses "Colatitude".
        if (((*i)->get_name ()).find ("Colatitude") != std::string::npos) 
        {
            if (!colatflag) 
            {
                if ((*i)->get_rank () != 2)
                {
                    std::cout << "The lat/lon rank must be 2 in " << (*i)->get_name () << std::endl;
                    throw std::runtime_error("The lat/lon rank error.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    //exit(1);
                } // end of if              
                int dimsize0 = (*i)->get_dimensions ()[0]->get_size ();
                int dimsize1 = (*i)->get_dimensions ()[1]->get_size ();

                // The following comparision may not be necessary. For most cases, the C-order first dimension is cooresponding to lat(in 1-D), which is mostly smaller than the dimension of lon(in 2-D). E.g. 90 for lat vs 180 for lon.
                if (dimsize0 < dimsize1) 
                {
                    tempnewdimname1 = (*i)->get_dimensions ()[0]->get_name ();
                    tempnewdimname2 = (*i)->get_dimensions ()[1]->get_name ();
                } 
                else 
                {
                    tempnewdimname1 = (*i)->get_dimensions ()[1]->get_name ();
                    tempnewdimname2 = (*i)->get_dimensions ()[0]->get_name ();
                } 

                colatflag = true;
                (*i)->fieldtype = 1;
                tempcvarname1 = (*i)->get_name ();
			
                switch((*i)->type)
                {
                    case DFNT_FLOAT32: 
                        set_float_value(*i); 
                        break; 
                    case DFNT_FLOAT64: 
                        set_double_value(*i);
                        break; 
                    default:
                        throw std::range_error("SDS's type is unknown(" __FILE__ ":" TOSTRING(__LINE__)")" );
                } // end of switch
            } // end of if (!colatflag)  
            else 
            {         
                //remove the redundant Colatitude field
                delete (*i);
                file->sd->sdfields.erase (i);
                // When erasing the iterator, the iterator will automatically go to the next element, so we need to go back 1 in order not to miss the next element.
                i--;
            } // end of else
        } // end of f (((*i)->get_name ()).find ("Col 
        else if (((*i)->get_name ()).find ("Longitude") != std::string::npos) 
        { 
            if (!lonflag) 
            {
                lonflag = true;
                (*i)->fieldtype = 2;
                tempcvarname2 = (*i)->get_name ();

                switch((*i)->type)
                {
                    case DFNT_FLOAT32: 
                        set_float_value(*i);
                        break; 
                    case DFNT_FLOAT64: 
                        set_double_value(*i);
                        break; 
                    default:
                        throw std::range_error("SDS's type is unknown(" __FILE__ ":" TOSTRING(__LINE__)")" );
                } // end of switch
            } // end of if (! 
            else 
            {
                //remove the redundant Longitude field
                delete (*i);
                file->sd->sdfields.erase (i);
                // When erasing the iterator, the iterator will automatically go to the next element, so we need to go back 1 in order not to miss the next element.
                i--;
            } // end of else
        } // end of else if (((*i)->g
    } // end of for

    file->sd->nonmisscvdimnamelist.insert (tempnewdimname1);
    file->sd->nonmisscvdimnamelist.insert (tempnewdimname2);
}


// Handle CERES ES4 and ISCCP-GEO cases. Essentially the lat/lon need to be condensed to 1-D for the geographic projection.
void prepare_cer_es4_ig()
{
    if(!(file->getSPType()==CER_ES4 || file->getSPType() ==CER_CGEO))
        return;

    std::string tempdimname1;
    std::string tempdimname2;
    int tempdimsize1 = 0;
    int tempdimsize2 = 0;
    int tempdimsize3 = 0;

    std::string tempcvarname1;
    std::string tempcvarname2;
    std::set < std::string > tempdimnameset;
    std::pair < std::set < std::string >::iterator, bool > tempsetit;
    bool cvflag = false;

    // The original latitude is 3-D array; we have to use the dimension name to determine which dimension is the final dimension for 1-D array. "regional colat" and "regional lon" are consistently used in these two CERES cases.

    for (std::vector < hdf4_var_sdfield * >::iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        std::string tempfieldname = (*i)->get_name ();
        if (tempfieldname.find ("Colatitude") != std::string::npos) 
        {
            (*i)->generate_str_value(file->get_sd_id());
            std::string mystr = (*i)->get_str_value();

            // For future error checking.
#if 0
            /*if(file->getSPType()==CER_ES4) //2D Colatitude
            {				
                if((*i)->get_rank()!=2)
                    throw std::range_error("For CER_ES4 case, lat/lon must be 2-D(" __FILE__ ":" TOSTRING(__LINE__)")" );
                if((*i)->get_type()!=DFNT_FLOAT32)
                    throw std::range_error("Datatype is not float, unsupported.(" __FILE__ ":" TOSTRING(__LINE__)")" );
            } else if(file->getSPType()== CER_CGEO) //3D Colatitude
            {
                if((*i)->get_rank()!=3)
                    throw std::range_error("For CER_ISCCP-D2like-GEO case, lat/lon must be 3-D(" __FILE__ ":" TOSTRING(__LINE__)")" );
                if((*i)->get_type()!=DFNT_FLOAT32)
                    throw std::range_error("Datatype is not float, unsupported.(" __FILE__ ":" TOSTRING(__LINE__)")" );
            }*/
#endif

            // They may have more than 2 dimensions, so we have to adjust it.
            for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_dimensions ().begin (); j != (*i)->get_dimensions ().end (); ++j) 
            {	
                if ((((*j)->get_name ()).find (remove_special_chars("regional colat")) != std::string::npos) && !cvflag) 
                {
                    tempsetit = tempdimnameset.insert ((*j)->get_name ());
                    if (tempsetit.second == true) 
                    {
                        tempdimname1 = (*j)->get_name ();
                        tempdimsize1 = (*j)->get_size ();

                        (*i)->fieldtype = 1;
                        (*i)->rank = 1;
                        cvflag = true;
                    } // end of if
                } // end of if ((((*j 
			
                if((((*j)->get_name ()).find (remove_special_chars("regional long"))) != std::string::npos)
                    tempdimsize3 = (*j)->get_size();
            } // end of for 

            if (cvflag) 
            {
                // change the dimension from 3-D to 1-D.
                // Clean up the original dimension vector first
                for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_dimensions ().begin (); j != (*i)->get_dimensions ().end (); ++j)
                    delete (*j);
                for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j)
                    delete (*j);
                (*i)->dims.clear ();
                (*i)->correcteddims.clear ();

                hdf4_dim *dim = new hdf4_dim (tempdimname1, tempdimname1,tempdimsize1, 0);
                (*i)->dims.push_back (dim);
                dim = new hdf4_dim (tempdimname1, tempdimname1, tempdimsize1, 0);
                (*i)->correcteddims.push_back (dim);
                file->sd->nonmisscvdimnamelist.insert (tempdimname1);
                cvflag = false;
		
                std::vector<char> v1;

                std::string s;
                int prepos=0; 
                int pos = mystr.find(",");
                int total=0;
                while(total<tempdimsize1 && pos!=std::string::npos)
                {
                    std::string ss = mystr.substr(prepos, pos-prepos);
                    s.append(ss);//mystr.substr(prepos, pos-prepos));
                    s.append(",");

                    size_t t1 = v1.size();
                    v1.resize(t1+4);
                    float t2 = 90 - atof(ss.c_str());
                    memcpy((char*)&(v1)[t1], &t2, sizeof(t2));	

                    total++;
                    int nr=0;
                    while(nr<tempdimsize3 && pos!=std::string::npos)
                    {
                        prepos = pos+1;
                        pos = mystr.find(",", prepos);
                        nr++;
                    } // end of while
                } // end of while	

                pos = s.find_last_of(",");
                (*i)->str_value = s.substr(0, pos);
                (*i)->value.resize(v1.size());
                std::copy(v1.begin(), v1.end(), (*i)->value.begin());
            } // end of if (cvflag)  
            else 
            {				
                //delete this element from the vector and erase it.
                delete (*i); //no regional colat inside its dim names
                file->sd->sdfields.erase (i);

                // When erasing the iterator, the iterator will automatically go to the next element, so we need to go back 1 in order not to miss the next element.
                i--;
            } // end of else
        } // end of if (tempfieldname.find ("Colatitude" 
        else if (tempfieldname.find ("Longitude") != std::string::npos) 
        { 
            (*i)->generate_str_value(file->get_sd_id());
            std::string mystr = (*i)->get_str_value();

            // For future error checking.
            /*if(file->getSPType()==CER_ES4) //2D Colatitude
            {
                if((*i)->get_rank()!=2)
                    throw std::range_error("For CER_ES4 case, lat/lon must be 2-D(" __FILE__ ":" TOSTRING(__LINE__)")" );
                if((*i)->get_type()!=DFNT_FLOAT32)
                    throw std::range_error("Datatype is not float, unsupported.(" __FILE__ ":" TOSTRING(__LINE__)")" );
            } else if(file->getSPType()== CER_CGEO) //3D Colatitude
            {
                if((*i)->get_rank()!=3)
                    throw std::range_error("For CER_ISCCP-D2like-GEO case, lat/lon must be 3-D(" __FILE__ ":" TOSTRING(__LINE__)")" );
                if((*i)->get_type()!=DFNT_FLOAT32)
                    throw std::range_error("Datatype is not float, unsupported.(" __FILE__ ":" TOSTRING(__LINE__)")" );
            }*/

            for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_dimensions ().begin (); j != (*i)->get_dimensions ().end (); ++j) 
            { 
                if (((*j)->get_name ()).find (remove_special_chars("regional long")) != std::string::npos) 
                { 
                    tempsetit = tempdimnameset.insert ((*j)->get_name ());
                    if (tempsetit.second == true) 
                    {
                        tempdimname2 = (*j)->get_name ();
                        tempdimsize2 = (*j)->get_size ();
                        (*i)->fieldtype = 2;
                        (*i)->rank = 1;
                        cvflag = true;
                        if(file->getSPType()==CER_CGEO)
                            break;
                    } // end of if
                    // Make this the only dimension name of this field
                } // end of if (((*j 
            } // end of for	 
            if (cvflag) 
            { 
                for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_dimensions ().begin (); j != (*i)->get_dimensions ().end (); ++j) 
                    delete (*j);
                for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j) 
                    delete (*j);
                (*i)->dims.clear ();
                (*i)->correcteddims.clear ();

                hdf4_dim *dim = new hdf4_dim (tempdimname2, tempdimname2,tempdimsize2, 0);
                (*i)->dims.push_back (dim);
                dim = new hdf4_dim (tempdimname2, tempdimname2,tempdimsize2, 0);
                (*i)->correcteddims.push_back (dim);
                file->sd->nonmisscvdimnamelist.insert (tempdimname2);
                cvflag = false;

                std::vector<char> v1;

                //2D(CER_ES$) && 3D(CER_CGEO) Longitude
                int prepos=0;
                int pos = mystr.find(",");
                int nr = 0;
                while(nr<tempdimsize2 && pos!=std::string::npos)
                {
                    size_t t1 = v1.size();
                    v1.resize(t1+4);
                    std::string t3 = mystr.substr(prepos, pos-1);
                    float t4 = atof(t3.c_str());
                    if(t4 >180.0) 
                        t4 = t4-360.0;
                    memcpy((char*)&(v1)[t1], &t4, sizeof(t4));

                    prepos = pos+1;
                    pos = mystr.find(",", prepos);
                    nr++;
                } // end of while
                (*i)->str_value = mystr.substr(0, pos);	
                (*i)->value.resize(v1.size());
                std::copy(v1.begin(), v1.end(), (*i)->value.begin());	
            } // end of  if (cvflag) 
            else 
            {				
                //delete this element from the vector and erase it.
                delete (*i);
                file->sd->sdfields.erase (i);
                // When erasing the iterator, the iterator will automatically go to the next element, so we need to go back 1 in order not to miss the next element.
                i--;
            } // end of else
        } // end of else if (tempfieldname.find ("Longitude")  
    } // end of for
}

//Index values are set for the field.
static void set_index_value(hdf4_var_sdfield *latlon, int size)
{
    std::vector<char> v1; 
    std::string ss; 
    char c[100]; 
    for(int ii=0; ii<size; ii++) 
    { 
        sprintf(c, "%d", (ii+1)); 
        ss.append(c); 
        size_t t1 = v1.size(); 
        v1.resize(t1+4); 
        float t2 = atof(c); 
        memcpy((char*)&(v1)[t1], &t2, sizeof(t2)); 
        if(ii!=size-1) 
            ss.append(","); 
    } 
    latlon->str_value = ss; 
    latlon->value.resize(v1.size()); 
    std::copy(v1.begin(), v1.end(), latlon->value.begin()); 
}

// 2-D lat/lon values are set. 
static void set_lat_lon(
    hdf4_var_sdfield *latlon, 
    int dimsize0, int dimsize1,
    float32 **val)
{
    std::vector<char> v1; 
    std::string s; 
    char c[100]; 

    for (int i = 0; i < dimsize0; i++) 
    {
        for (int j = 0; j < dimsize1; j++) 
        { 
            sprintf(c, "%.6f", val[i][j]); 
            s.append(c); 
            s.append(","); 
            size_t t1 = v1.size(); 
            v1.resize(t1+4); 
            float t2 = atof(c); 
            memcpy((char*)&(v1)[t1], &t2, sizeof(t2)); 
        } 
    } 
    size_t pos = s.find_last_of(","); 
    latlon->str_value = s.substr(0, pos);
    latlon->value.resize(v1.size()); 
    std::copy(v1.begin(), v1.end(), latlon->value.begin()); 
}
 
void readcersavgid2(hdf4_var_sdfield *lat, hdf4_var_sdfield *lon)
{
    int dimsize0 = 180;
    int dimsize1 = 360;
    float32 **orival = new float32* [dimsize0]; 
    for(int i=0; i<dimsize0; i++)
        orival[i] = new float32[dimsize1];

    // Following CERES Nested grid
    // URL http://eosweb.larc.nasa.gov/PRODOCS/ceres/SRBAVG/Quality_Summaries/srbavg_ed2d/nestedgrid.html
   
    // Calculate the latitude
    for (int i = 0; i < dimsize0; i++)
        for (int j = 0; j < dimsize1; j++)
            orival[i][j] = 89.5 - i;
		
    set_lat_lon(lat, dimsize0, dimsize1, orival);

    // Calculate the longitude
    int i = 0;
    int j = 0;
    int k = 0;
    int lonextent = 0;
    int latindex_south = 0;
    int latindex_north = 0;
    int latrange = 0;

    // latitude 89-90 (both south and north) 1 value each part
    for (j = 0; j < dimsize1; j++) 
    {
        orival[0][j] = -179.5;
        orival[dimsize0 - 1][j] = -179.5;
    }

    // latitude 80-89 (both south and north) 45 values each part
    // longitude extent is 8
    lonextent = 8;
    // From 89 N to 80 N
    latindex_north = 1;
    latrange = 9;
    for (i = 0; i < latrange; i++)
        for (j = 0; j < (dimsize1 / lonextent); j++)
            for (k = 0; k < lonextent; k++)
                orival[i + latindex_north][j * lonextent + k] = -179.5 + lonextent * j;

    // From 89 S to 80 S
    latindex_south = dimsize0 - 1 - latrange;
    for (i = 0; i < latrange; i++)
        for (j = 0; j < (dimsize1 / lonextent); j++)
            for (k = 0; k < lonextent; k++)
                orival[i + latindex_south][j * lonextent + k] = -179.5 + lonextent * j;

    // From 80 N to 70 N
    latindex_north = latindex_north + latrange;
    latrange = 10;
    lonextent = 4;

    for (i = 0; i < latrange; i++)
        for (j = 0; j < (dimsize1 / lonextent); j++)
            for (k = 0; k < lonextent; k++)
                orival[i + latindex_north][j * lonextent + k] = -179.5 + lonextent * j;

    // From 80 S to 70 S
    latindex_south = latindex_south - latrange;
    for (i = 0; i < latrange; i++)
        for (j = 0; j < (dimsize1 / lonextent); j++)
            for (k = 0; k < lonextent; k++)
                orival[i + latindex_south][j * lonextent + k] = -179.5 + lonextent * j;

    // From 70 N to 45 N
    latindex_north = latindex_north + latrange;
    latrange = 25;
    lonextent = 2;
	
    for (i = 0; i < latrange; i++)
        for (j = 0; j < (dimsize1 / lonextent); j++)
            for (k = 0; k < lonextent; k++)
                orival[i + latindex_north][j * lonextent + k] = -179.5 + lonextent * j;

    // From 70 S to 45 S
    latindex_south = latindex_south - latrange;
    for (i = 0; i < latrange; i++)
        for (j = 0; j < (dimsize1 / lonextent); j++)
            for (k = 0; k < lonextent; k++)
                orival[i + latindex_south][j * lonextent + k] = -179.5 + lonextent * j;

    // From 45 N to 0 N
    latindex_north = latindex_north + latrange;
    latrange = 45;
    lonextent = 1;

    for (i = 0; i < latrange; i++)
        for (j = 0; j < (dimsize1 / lonextent); j++)
            for (k = 0; k < lonextent; k++)
                orival[i + latindex_north][j * lonextent + k] = -179.5 + lonextent * j;

    // From 45 S to 0 S
    latindex_south = latindex_south - latrange;
    for (i = 0; i < latrange; i++)
        for (j = 0; j < (dimsize1 / lonextent); j++)
            for (k = 0; k < lonextent; k++)
                orival[i + latindex_south][j * lonextent + k] = -179.5 + lonextent * j;

    set_lat_lon(lon, dimsize0, dimsize1, orival);

    for(i=0; i<dimsize0; i++)
        delete[] orival[i];
    delete[] orival;
}

// CERES SAVG and CERES ISCCP-IDAY cases.
// We need provide nested CERES grid 2-D lat/lon.
// The lat and lon should be calculated following:
// http://eosweb.larc.nasa.gov/PRODOCS/ceres/SRBAVG/Quality_Summaries/srbavg_ed2d/nestedgrid.html
// The dimension names and sizes are set according to the studies of these files.
void prepare_cer_savg_id()
{
    if(!(file->getSPType()==CER_CDAY || file->getSPType()==CER_SRB))
        return;


    std::string tempdimname1 = remove_special_chars("1.0 deg. regional colat. zones");
    std::string tempdimname2 = remove_special_chars("1.0 deg. regional long. zones");
    std::string tempdimname3 = remove_special_chars("1.0 deg. zonal colat. zones");
    std::string tempdimname4 = remove_special_chars("1.0 deg. zonal long. zones");

    int tempdimsize1 = 180;
    int tempdimsize2 = 360;
    int tempdimsize3 = 180;
    int tempdimsize4 = 1;

    std::string tempnewdimname1, tempnewdimname2;
    std::string tempcvarname1, tempcvarname2;

    hdf4_var_sdfield *latitude = new hdf4_var_sdfield();
    latitude->rename("latitude");
    latitude->rank = 2;
    latitude->type = DFNT_FLOAT32;
    latitude->fieldtype = 1;

    // No need to obtain the full path
    latitude->newname = latitude->get_name();

    hdf4_dim *dim = new hdf4_dim(tempdimname1, tempdimname1,tempdimsize1, 0);
    latitude->dims.push_back (dim);

    dim = new hdf4_dim(tempdimname2, tempdimname2,tempdimsize2, 0);
    latitude->dims.push_back (dim);

    dim = new hdf4_dim(tempdimname1, tempdimname1,tempdimsize1, 0);
    latitude->correcteddims.push_back (dim);

    dim = new hdf4_dim(tempdimname2, tempdimname2,tempdimsize2, 0);
    latitude->correcteddims.push_back (dim);
    file->sd->sdfields.push_back (latitude);

    hdf4_var_sdfield *longitude = new hdf4_var_sdfield();
    longitude->rename("longitude");
    longitude->rank = 2;
    longitude->type = DFNT_FLOAT32;
    longitude->fieldtype = 2;

    // No need to obtain the full path
    longitude->newname = longitude->get_name();

    dim = new hdf4_dim(tempdimname1, tempdimname1,tempdimsize1, 0);
    longitude->dims.push_back (dim);

    dim = new hdf4_dim(tempdimname2, tempdimname2,tempdimsize2, 0);
    longitude->dims.push_back (dim);

    dim = new hdf4_dim(tempdimname1, tempdimname1,tempdimsize1, 0);
    longitude->correcteddims.push_back (dim);

    dim = new hdf4_dim(tempdimname2, tempdimname2,tempdimsize2, 0);
    longitude->correcteddims.push_back (dim);
    file->sd->sdfields.push_back (longitude);

    readcersavgid2(latitude, longitude);

    // For the CER_SRB case, zonal average data is also included. We need only provide the latitude.
    // Note values of latitudez and longitudez are index values,which seem not right. Document in this release. KY 2014-12-25
    if (file->getSPType() == CER_SRB) 
    { 
        hdf4_var_sdfield *latitudez = new hdf4_var_sdfield ();
        latitudez->rename("latitudez");
        latitudez->rank = 1;
        latitudez->type = DFNT_FLOAT32;
        latitudez->fieldtype = 1;
        latitudez->newname = latitudez->get_name();

        dim = new hdf4_dim(tempdimname3, tempdimname3,tempdimsize3, 0);
        latitudez->dims.push_back (dim);

        dim = new hdf4_dim(tempdimname3, tempdimname3,tempdimsize3, 0);
        latitudez->correcteddims.push_back (dim);
        file->sd->sdfields.push_back (latitudez);

        set_index_value(latitudez, tempdimsize3);

        hdf4_var_sdfield *longitudez = new hdf4_var_sdfield ();
        longitudez->rename("longitudez");
        longitudez->rank = 1;
        longitudez->type = DFNT_FLOAT32;
        longitudez->fieldtype = 2;
        longitudez->newname = longitudez->get_name();

        dim = new hdf4_dim(tempdimname4, tempdimname4,tempdimsize4, 0);
        longitudez->dims.push_back (dim);

        dim = new hdf4_dim(tempdimname4, tempdimname4,tempdimsize4, 0);
        longitudez->correcteddims.push_back (dim);
        file->sd->sdfields.push_back (longitudez);

        set_index_value(longitudez, tempdimsize4);
    } // end of if (file->getSPType() == CER_SRB)  

    if (file->getSPType() == CER_SRB) 
    {
        file->sd->nonmisscvdimnamelist.insert (tempdimname3);
        file->sd->nonmisscvdimnamelist.insert (tempdimname4);
    }

    file->sd->nonmisscvdimnamelist.insert (tempdimname1);
    file->sd->nonmisscvdimnamelist.insert (tempdimname2);
}

// Prepare the CER_ZAVG case. This is the zonal average case.
// Only latitude is needed.
void prepare_cer_zavg()
{
    if(file->getSPType()!=CER_ZAVG)
        return;

    std::string tempdimname3 = remove_special_chars("1.0 deg. zonal colat. zones");
    std::string tempdimname4 = remove_special_chars("1.0 deg. zonal long. zones");
    int tempdimsize3 = 180;
    int tempdimsize4 = 1;

    hdf4_var_sdfield *latitudez = new hdf4_var_sdfield ();

    latitudez->rename("latitudez");
    latitudez->rank = 1;
    latitudez->type = DFNT_FLOAT32;
    latitudez->fieldtype = 1;
    latitudez->newname = latitudez->get_name();

    hdf4_dim *dim = new hdf4_dim(tempdimname3, tempdimname3,tempdimsize3, 0);
    latitudez->dims.push_back (dim);

    dim = new hdf4_dim (tempdimname3, tempdimname3,tempdimsize3, 0);
    latitudez->correcteddims.push_back (dim);

    file->sd->sdfields.push_back (latitudez);

    std::vector<char> v1, v2;

    char c[100];
    std::string mystr;
    float32 latstep = 1.0;
    for(int i=0; i<tempdimsize3; i++)
    {
        float32 val =  89.5 - (/*(int) (offset32[0])*/0 + /*((int) (step32[0]))*/1 * i) * latstep;
        sprintf(c, "%.6f", val);
        mystr.append(c);
        mystr.append(",");
        size_t t1 = v1.size();
        v1.resize(t1+4);
        float t2 = atof(c);
        memcpy((char*)&(v1)[t1], &t2, sizeof(t2));
    } // end of for
    size_t pos = mystr.find_last_of(",");
    latitudez->str_value = mystr.substr(0, pos);
    latitudez->value.resize(v1.size());
    std::copy(v1.begin(), v1.end(), latitudez->value.begin());

    hdf4_var_sdfield *longitudez = new hdf4_var_sdfield ();
    longitudez->rename("longitudez");
    longitudez->rank = 1;
    longitudez->type = DFNT_FLOAT32;
    longitudez->fieldtype = 2;
    longitudez->newname = longitudez->get_name();

    dim = new hdf4_dim(tempdimname4, tempdimname4,tempdimsize4, 0);
    longitudez->dims.push_back (dim);

    dim = new hdf4_dim(tempdimname4, tempdimname4,tempdimsize4, 0);
    longitudez->correcteddims.push_back (dim);

    file->sd->sdfields.push_back (longitudez);

    std::string ss;
    for(int ii=0; ii<tempdimsize4; ii++)
    {
        ss.append("0.0");
        size_t t1 = v2.size();
        v2.resize(t1+4);
        float t2 = 0.0;
        memcpy((char*)&(v2)[t1], &t2, sizeof(t2));

        if(ii!=tempdimsize4-1)
            ss.append(",");
    } // end of for
    longitudez->str_value = ss;
    longitudez->value.resize(v2.size());
    std::copy(v2.begin(), v2.end(), longitudez->value.begin());

    file->sd->nonmisscvdimnamelist.insert (tempdimname3);
    file->sd->nonmisscvdimnamelist.insert (tempdimname4);
}

// Set up latitude and longitude for MODIS products like MODARNSS or MYDARNSS. 
void prepare_modisarnss() 
{
    if(file->getSPType() != MODISARNSS)
        return;

    std::set < std::string > tempfulldimnamelist;
    std::pair < std::set < std::string >::iterator, bool > ret;

    std::map < int, std::string > tempsizedimnamelist;

    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        if ((*i)->get_name () == "Latitude")
            (*i)->fieldtype = 1;
        if ((*i)->get_name () == "Longitude") 
        {
            (*i)->fieldtype = 2;
            // Fields associate with lat/lon use different dimension names. To be consistent with other code, use size-dim map to change fields that have the same size as lat/lon to hold the same dimension names.
            for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j) 
            {
                tempsizedimnamelist[(*j)->get_size ()] = (*j)->get_name ();
                file->sd->nonmisscvdimnamelist.insert ((*j)->get_name ());
            }
        } // end of if ((*i)->get_name	 
    } // end of for
 
    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j) 
        {
            // Need to change those dimension names to be the same as lat/lon so that a coordinate variable dimension name map can be built.
            if ((*i)->fieldtype == 0) 
            {
                if ((tempsizedimnamelist.find ((*j)->get_size ())) != tempsizedimnamelist.end ())
                    (*j)->rename(tempsizedimnamelist[(*j)->get_size ()]);
            }
        } // end of for
    } // end of for
}

// Set up latitude and longitude for avhrr of PODAAC product, like avhrr.
void prepare_other_hdf ()
{
    if(file->getSPType() != OTHERHDF)
        return;

    std::set < std::string > tempfulldimnamelist;

    for (std::vector < hdf4_var_sdfield * >::const_iterator i = file->sd->sdfields.begin (); i != file->sd->sdfields.end (); ++i) 
    {
        for (std::vector < hdf4_dim * >::const_iterator j = (*i)->get_corrected_dimensions ().begin (); j != (*i)->get_corrected_dimensions ().end (); ++j) 
        {
            if ((*j)->get_type () != 0) 
            {
                if ((*i)->get_name() == (*j)->get_name () && (*i)->get_rank () == 1)
                {
                    if((*i)->get_name()=="lat")
                        (*i)->fieldtype = 1;
                    else if ((*i)->get_name() =="lon")
                        (*i)->fieldtype = 2;
                    else 
                        (*i)->fieldtype = 3;
                    file->sd->nonmisscvdimnamelist.insert ((*j)->get_name ());
                } // end of if
            } // end of ir
        } // end  of for
    } // end of for

}


filterqueue m_filterqueue;
//FILTERQUEUE();

class filter_hdf4
{
    public:
    filter_hdf4()
    {
        m_filterqueue.add(build_dim_name_list);
        m_filterqueue.add(squeeze_fake_dim_names);
        m_filterqueue.add(prepare_trmm_l2);
        m_filterqueue.add(prepare_trmm_l3);
        m_filterqueue.add(prepare_trmm_3a46);
        m_filterqueue.add(prepare_trmm_v7_l2);
        m_filterqueue.add(prepare_trmm_v7_l3s);
        m_filterqueue.add(prepare_trmm_v7_l3m);
   
        m_filterqueue.add(prepare_other_hdf);
        m_filterqueue.add(prepare_obpg_l2);
        m_filterqueue.add(prepare_obpg_l3);
        m_filterqueue.add(prepare_cer_avg_syn);
        m_filterqueue.add(prepare_cer_es4_ig);
        m_filterqueue.add(prepare_cer_savg_id);
        m_filterqueue.add(prepare_cer_zavg);
        m_filterqueue.add(prepare_modisarnss);
        m_filterqueue.add(add_origname_attr);
        m_filterqueue.add(add_longname_attr);
        m_filterqueue.add(generate_new_dim_name_set);
        m_filterqueue.add(add_missing_cv);
        m_filterqueue.add(add_attrs_for_obpg_l3m);
        //m_filterqueue.add(generate_new_dim_name_set);
        //m_filterqueue.add(add_missing_cv);
        m_filterqueue.add(handle_special_chars_of_sds);
        m_filterqueue.add(handle_special_chars_of_vdata);
        m_filterqueue.add(handle_special_chars_of_file_attrs);
        m_filterqueue.add(handle_special_chars_of_sds_attrs);
        m_filterqueue.add(check_name_clashing);
        m_filterqueue.add(add_coordinates_attr);		
        m_filterqueue.add(add_missing_cf_attrs);
    } // end of constructor	
};
filter_hdf4 fhdf4;

//ADD_METHOD(build_dim_name_list);
//ADD_METHOD(squeeze_fake_dim_names);
//ADD_METHOD(prepare_trmm_l2);
//ADD_METHOD(prepare_trmm_l3);
//ADD_METHOD(prepare_trmm_3a46);
//ADD_METHOD(prepare_other_hdf);
//ADD_METHOD(prepare_obpg_l2);
//ADD_METHOD(prepare_obpg_l3);
//ADD_METHOD(prepare_cer_avg_syn);
//ADD_METHOD(prepare_cer_es4_ig);
//ADD_METHOD(prepare_cer_savg_id);
//ADD_METHOD(prepare_cer_zavg);
//ADD_METHOD(prepare_modisarnss);
//ADD_METHOD(add_origname_attr);
//ADD_METHOD(add_longname_attr);
//ADD_METHOD(check_name_clashing);
//ADD_METHOD(add_attrs_for_obpg_l3m);
//ADD_METHOD(generate_new_dim_name_set);
//ADD_METHOD(add_missing_cv);
//ADD_METHOD(handle_special_chars_of_sds);
//ADD_METHOD(handle_special_chars_of_vdata);
//ADD_METHOD(handle_special_chars_of_file_attrs);
//ADD_METHOD(add_coordinates_attr);

//The following macros are for testing only.
/*ADD_METHOD(dump_cdl);*/
} //end of namespace

