/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of an HDF4 
Scientific Dataset.

*****************************************************************************/

#include "hdf4_sds.h"
#include "hdf4_file.h"
#include <assert.h>
#include <cstring>

extern hdf4::hdf4_file *file;

namespace hdf4 {

hdf4_var_sdfield::hdf4_var_sdfield():var(NULL, "non-valid name"),type(-1), rank(-1), fieldtype(0),sdsref(-1), condenseddim(false) {} 

hdf4_var_sdfield::~hdf4_var_sdfield() 
{
    for(std::vector<hdf4_attr *>::const_iterator i=attrs.begin(); i!=attrs.end(); i++)
        delete *i;
    for(std::vector<hdf4_dim *>::const_iterator i=dims.begin(); i!= dims.end(); i++)
        delete *i;
    for(std::vector<hdf4_dim *>::const_iterator i=correcteddims.begin(); i!=correcteddims.end(); i++)
        delete *i;
}

const bool hdf4_var_sdfield::has_attr(std::string attrname) const
{
    for(std::vector<hdf4_attr *>::const_iterator it=attrs.begin(); it!=attrs.end(); it++)
        if((*it)->get_name()==attrname)
            return true;
    return false;
}

void hdf4_var_sdfield::add_attr_one_str(const std::string &name, std::string attr_value)
{
    std::vector<char> v(attr_value.length());
    copy(attr_value.begin(), attr_value.end(), v.begin());
    hdf4_attr *attr = new hdf4_attr(name, DFNT_CHAR8, attr_value.length(), v);
    attrs.push_back(attr);
}

void hdf4_var_sdfield::add_attr_one_float(const std::string & attr_name, float attr_value)
{
    if(has_attr(attr_name)) //Attribute _FillValue exists.
        return;

    // For testing only
    //char array[4];
    //memcpy(array, &value, sizeof (value));

    std::vector<char> fv(4);
    memcpy((char*)&(fv)[0], &attr_value, sizeof(float));
    hdf4_attr *fvattr = new hdf4_attr(attr_name, DFNT_FLOAT32, 1, fv);
    attrs.push_back(fvattr);
}

void hdf4_var_sdfield::add_attr_one_double(const std::string & attr_name, double attr_value)
{
    if(has_attr(attr_name)) //Attribute _FillValue exists.
        return;

    std::vector<char> fv(sizeof(double));
    memcpy((char*)&(fv)[0], &attr_value, sizeof(double));
    hdf4_attr *fvattr = new hdf4_attr(attr_name, DFNT_FLOAT64, 1, fv);
    attrs.push_back(fvattr);
}

void hdf4_var_sdfield::add_attr_one_int16(const std::string & attr_name, int16 attr_value)

{
    if(has_attr(attr_name)) //Attribute _FillValue exists.
        return;

    std::vector<char> fv(sizeof(short));
    memcpy((char*)&(fv)[0], &attr_value, sizeof(short));
    hdf4_attr *fvattr = new hdf4_attr(attr_name, DFNT_INT16, 1, fv);
    attrs.push_back(fvattr);
}

void hdf4_var_sdfield::add_attr_one_int32(const std::string & attr_name, int32 attr_value)
{
    if(has_attr(attr_name)) //Attribute _FillValue exists.
        return;

    // For testing only
    //char array[4];
    //memcpy(array, &attr_value, sizeof (attr_value));

    std::vector<char> fv(sizeof(int));
    memcpy((char*)&(fv)[0], &attr_value, sizeof(int));
    hdf4_attr *fvattr = new hdf4_attr(attr_name, DFNT_INT32, 1, fv);
    attrs.push_back(fvattr);
}

void hdf4_var_sdfield::add_attr_one_int8(const std::string & attr_name, int8 attr_value)
{
    if(has_attr(attr_name)) //Attribute _FillValue exists.
        return;

    std::vector<char> fv(1);
    fv[0] = attr_value;
    hdf4_attr *fvattr = new hdf4_attr(attr_name, DFNT_INT8, 1, fv);
    attrs.push_back(fvattr);
}

hdf4_attr* hdf4_var_sdfield::get_attr_by_name(const std::string& name) {

    std::vector<hdf4_attr*>& attrs = get_attributes();

    std::vector<hdf4_attr*>::iterator it;
    for(it = attrs.begin(); it != attrs.end(); it++) {
        if((*it)->get_name() == name)
            return *it;
    }
    return NULL;

}

void hdf4_var_sdfield::remove_attr_by_name(std::string name)
{
    bool need_to_erase = false;
    std::vector<hdf4_attr *>::iterator eraseit;
    hdf4_attr* eraseattr = NULL;
    for(std::vector<hdf4_attr *>::iterator i=attrs.begin(); i!=attrs.end(); i++) {
        if((*i)->get_name() == name) 
        {
            need_to_erase = true;
            eraseit = i;
            eraseattr = *i;
            break;
            //attrs.erase(i);
            //delete (*i);
        }
    }

    if (true == need_to_erase) {
        attrs.erase(eraseit);
        if (eraseattr != NULL)
            delete(eraseattr);
    }

    return;
}

value_type_t hdf4_var_sdfield::get_type() const
{
    return type;
}

// Dump partial or all data values.(This function is not throughly tested. It is not used by the conversion toolkit APIs.)
static void print_str(int data_type, int32 sdsid, int32 num_elems, int32 start[], int32 stride[], int32 edge[])
{
    switch(data_type)
    {
        case DFNT_CHAR8:
        case DFNT_UCHAR8: 
        { 
            char *buf = new char[num_elems]; 
            SDreaddata (sdsid, start, stride, edge, (VOIDP)buf); 
            char c[100]; 
            for(int index=0; index<num_elems; index++) 
            { 
                sprintf(c, "%c", buf[index]); 
                std::cout << c; 
                if(index!=num_elems-1) 
                    std::cout << ","; 
            } // end of for 
            delete[] buf; 
        } 
            break;
        case DFNT_UINT8:
        {
            uint8 *buf = new uint8[num_elems]; 
            SDreaddata (sdsid, start, stride, edge, (VOIDP)buf); 
            char c[100]; 
            for(int index=0; index<num_elems; index++) 
            { 
                sprintf(c, "%u", buf[index]); 
                std::cout << c; 
                if(index!=num_elems-1) 
                    std::cout << ","; 
            } 
            delete[] buf; 
        }
            break;
        case DFNT_INT8:
        {
            int8 *buf = new int8[num_elems]; 
            SDreaddata (sdsid, start, stride, edge, (VOIDP)buf); 
            char c[100]; 
            for(int index=0; index<num_elems; index++) 
            { 
                sprintf(c, "%d", buf[index]); 
                std::cout << c; 
                if(index!=num_elems-1) 
                    std::cout << ","; 
            } 
            delete[] buf; 
        }
            break;
        case DFNT_UINT16:
        {
            uint16 *buf = new uint16[num_elems]; 
            SDreaddata (sdsid, start, stride, edge, (VOIDP)buf); 
            char c[100]; 
            for(int index=0; index<num_elems; index++) 
            { 
                sprintf(c, "%u", buf[index]); 
                std::cout << c; 
                if(index!=num_elems-1) 
                    std::cout << ","; 
            } 
            delete[] buf; 
        }
            break;
        case DFNT_INT16:
        {
            int16 *buf = new int16[num_elems]; 
            SDreaddata (sdsid, start, stride, edge, (VOIDP)buf); 
            char c[100]; 
            for(int index=0; index<num_elems; index++) 
            { 
                sprintf(c, "%d", buf[index]); 
                std::cout << c; 
                if(index!=num_elems-1) 
                    std::cout << ","; 
            } 
            delete[] buf; 
        }
            break;
        case DFNT_UINT32:
        {
            uint32 *buf = new uint32[num_elems]; 
            SDreaddata (sdsid, start, stride, edge, (VOIDP)buf); 
            char c[100]; 
            for(int index=0; index<num_elems; index++) 
            { 
                sprintf(c, "%u", buf[index]); 
                std::cout << c; 
                if(index!=num_elems-1) 
                    std::cout << ","; 
            } 
            delete[] buf; 
        }
            break;
        case DFNT_INT32:
        {
            int32 *buf = new int32[num_elems]; 
            SDreaddata (sdsid, start, stride, edge, (VOIDP)buf); 
            char c[100]; 
            for(int index=0; index<num_elems; index++) 
            { 
                sprintf(c, "%d", buf[index]); 
                std::cout << c; 
                if(index!=num_elems-1) 
                    std::cout << ","; 
            } 
            delete[] buf; 
        }
            break;
        case DFNT_FLOAT32:
        {
            float *buf = new float[num_elems]; 
            SDreaddata (sdsid, start, stride, edge, (VOIDP)buf); 
            char c[100]; 
            for(int index=0; index<num_elems; index++) 
            { 
                sprintf(c, "%.6f", buf[index]); 
                std::cout << c; 
                if(index!=num_elems-1) 
                    std::cout << ","; 
            } 
            delete[] buf; 
        }
            break;
        case DFNT_FLOAT64:
        {
            double *buf = new double[num_elems]; 
            SDreaddata (sdsid, start, stride, edge, (VOIDP)buf); 
            char c[100]; 
            for(int index=0; index<num_elems; index++) 
            { 
                sprintf(c, "%8.3e", buf[index]); 
                std::cout << c; 
                if(index!=num_elems-1) 
                    std::cout << ","; 
            } 
            delete[] buf; 
        }
            break;
        default:
            throw std::range_error("SDS's type is unknown(" __FILE__ ":" TOSTRING(__LINE__)")" );
    } // end of switch 
}  

#define DATA_VALUE_2(TYPE, CAST, FORMAT) \
    case DFNT_##TYPE: \
    { \
        CAST *v = new CAST[num_elems]; \
        SDreaddata (sdsid, start, stride, edge, (VOIDP)v); \
        char c[100]; \
        for(int index=0; index<num_elems; index++) \
        { \
            sprintf(c, #FORMAT, v[index]); \
            std::cout << c; \
            if(index!=num_elems-1) \
                std::cout << ","; \
        } \
        delete[] v; \
    } \
        break;

// Print out all data values.
#define DATA_VALUE(TYPE, CAST, FORMAT) \
    case DFNT_##TYPE: \
    { \
        CAST *buf = new CAST[total]; \
        SDreaddata (sdsid, start, stride, edge, (VOIDP)buf); \
        char c[100]; \
        for(int l=0; l<total; l++) \
        { \
            sprintf(c, #FORMAT, buf[l]); \
            std::cout << c; \
            if(l!=total-1) \
                std::cout << ","; \
        } \
        delete[] buf; \
    } \
        break;


// Generate string representation of data values. The values are delimited by ",".
// This function is not throughly tested. It is not used by H4CF conversion toolkit APIs.
void hdf4_var_sdfield::generate_str_value(int32 sdfd)
{
    if(str_value.size()!=0) //Already set.
        return;

    int32 sdsindex = SDreftoindex(sdfd, sdsref);
    int32 sdsid = SDselect(sdfd, sdsindex);

    size_t total = 1;
    
    std::vector<int32>start;
    start.assign(rank,0);
    std::vector<int32>edge;
    edge.resize(rank);
    std::vector<int32>stride;
    stride.resize(rank);
//    int32 *start = (int32 *)malloc(rank*sizeof(int32));
//    std::fill_n(start, rank, 0);
//    int32 *edge = (int32 *)malloc(rank*sizeof(int32));
//    int32 *stride = (int32 *)malloc(rank*sizeof(int32));
    int index=0;
    for(std::vector<hdf4_dim *>::const_iterator i=dims.begin(); i!=dims.end(); i++)
    {
        stride[index] = 1;
        edge[index] = (*i)->get_size();
        total *= edge[index];
        ++index;
    } // end of for
	
    switch(type)
    {
        case DFNT_CHAR8:
        case DFNT_UCHAR8: 
        { 
            char *buf = new char[total]; 
            SDreaddata (sdsid, &start[0], NULL, &edge[0], buf); 
            char c[100]; 
            for(int l=0; l<total; l++) 
            { 
                sprintf(c, "%c", buf[l]); 
                str_value.append(c); 
                if(l!=total-1) 
                    str_value.append(","); 
            } 
            delete[] buf; 
        } 
            break;
        case DFNT_UINT8:
        {
            uint8 *buf = new uint8[total]; 
            SDreaddata (sdsid, &start[0], NULL, &edge[0], buf); 
            char c[100]; 
            for(int l=0; l<total; l++) 
            { 
                sprintf(c, "%u", buf[l]); 
                str_value.append(c); 
                if(l!=total-1) 
                    str_value.append(","); 
            } 
            delete[] buf;
        }
            break;
        case DFNT_INT8:
        {
            int8 *buf = new int8[total]; 
            SDreaddata (sdsid, &start[0], NULL, &edge[0], buf); 
            char c[100]; 
            for(int l=0; l<total; l++) 
            { 
                sprintf(c, "%d", buf[l]); 
                str_value.append(c); 
                if(l!=total-1) 
                    str_value.append(","); 
            } 
            delete[] buf;
        }
            break;
        case DFNT_UINT16:
        {
            uint16 *buf = new uint16[total]; 
            SDreaddata (sdsid, &start[0], NULL, &edge[0], buf); 
            char c[100]; 
            for(int l=0; l<total; l++) 
            { 
                sprintf(c, "%u", buf[l]); 
                str_value.append(c); 
                if(l!=total-1) 
                    str_value.append(","); 
            } 
            delete[] buf;
        }
            break;
        case DFNT_INT16:
        {
            int16 *buf = new int16[total]; 
            SDreaddata (sdsid, &start[0], NULL, &edge[0], buf); 
            char c[100]; 
            for(int l=0; l<total; l++) 
            { 
                sprintf(c, "%d", buf[l]); 
                str_value.append(c); 
                if(l!=total-1) 
                    str_value.append(","); 
            } 
            delete[] buf;
        }
            break;
        case DFNT_UINT32:
        {
            uint32 *buf = new uint32[total]; 
            SDreaddata (sdsid, &start[0], NULL, &edge[0], buf); 
            char c[100]; 
            for(int l=0; l<total; l++) 
            { 
                sprintf(c, "%u", buf[l]); 
                str_value.append(c); 
                if(l!=total-1) 
                    str_value.append(","); 
            } 
            delete[] buf;
        }
            break;
        case DFNT_INT32:
        {
            int32 *buf = new int32[total]; 
            SDreaddata (sdsid, &start[0], NULL, &edge[0], buf); 
            char c[100]; 
            for(int l=0; l<total; l++) 
            { 
                sprintf(c, "%d", buf[l]); 
                str_value.append(c); 
                if(l!=total-1) 
                    str_value.append(","); 
            } 
            delete[] buf;
        }
            break;
        case DFNT_FLOAT32:
        {
            float *buf = new float[total]; 
            SDreaddata (sdsid, &start[0], NULL, &edge[0], buf); 
            char c[100]; 
            for(int l=0; l<total; l++) 
            { 
                sprintf(c, "%.6f", buf[l]); 
                str_value.append(c); 
                if(l!=total-1) 
                    str_value.append(","); 
            } 
            delete[] buf;
        }
            break;
        case DFNT_FLOAT64:
        {
            double *buf = new double[total]; 
            SDreaddata (sdsid, &start[0], NULL, &edge[0], buf); 
            char c[100]; 
            for(int l=0; l<total; l++) 
            { 
                sprintf(c, "%8.3e", buf[l]); 
                str_value.append(c); 
                if(l!=total-1) 
                    str_value.append(","); 
            } 
            delete[] buf;
        }
            break;
        default:
            throw std::range_error("SDS's type is unknown(" __FILE__ ":" TOSTRING(__LINE__)")" );
    } // end of switch

    //delete[] start;
    //delete[] stride;
    //delete[] edge;
}

// For internal testing only.
// Dump partial or all data values.
void hdf4_var_sdfield::dump_str(int32 sdfd, int32 start[], int32 stride[], int32 edge[])
{
    int32 sdsindex = SDreftoindex(sdfd, sdsref);
    int32 sdsid = SDselect(sdfd, sdsindex);

    int32 num_elems = 1;
    for(ssize_t i=0; i <rank; i++)
        num_elems *= edge[i];

    // Dump partial or all data values.
    print_str(type, sdsid, num_elems, start, stride, edge);
}

// For internal testing only.
// Dump all values.
void hdf4_var_sdfield::dump_str(int32 sdfd)
{
    if(str_value.length()!=0)
    {
        std::cout << str_value;
        return;
    }

    int32 sdsindex = SDreftoindex(sdfd, sdsref);
    int32 sdsid = SDselect(sdfd, sdsindex);

    int32 total = 1;
#if 0
    int32 *start = (int32 *)malloc(rank*sizeof(int32));
    std::fill_n(start, rank, 0);
    int32 *edge = (int32 *)malloc(rank*sizeof(int32));
    int32 *stride = (int32 *)malloc(rank*sizeof(int32));
    std::fill_n(stride, rank, 1);
#endif
   std::vector<int32>start;
   start.assign(rank,0);
   std::vector<int32>edge;
   edge.resize(rank);
   std::vector<int32>stride;
   stride.assign(rank,1);
    int index=0;

    for(std::vector<hdf4_dim *>::const_iterator i=dims.begin(); i!=dims.end(); i++)
    {
        edge[index] = (*i)->get_size();
        total *= edge[index];
        ++index;
    } // end of for		
	
    print_str(type, sdsid, total, &start[0], &stride[0], &edge[0]);
		
//    delete[] start;
//    delete[] stride;
//    delete[] edge;

    // For debugging only.
    //std::vector<char> pbuf;
    //get_all_values(&pbuf);

    //int32 start2[2] = {0,0};
    //int32 stride2[2] = {1,1};
    //int32 edge2[2] = {219, 49};
    //std::vector<char> pbuf;
    //get_value(start2, stride2, edge2, &pbuf);
}	

void hdf4_var_sdfield::get_all_values(std::vector<char> *pbuf) const
{

    if(-1 == sdsref ) {

        // Some fields need to be added by the toolkit. These fields
        // include missing CVs such as latitude and longitude in TRMM level 3 files.
        // Their values are assigned when the field is created.
        if(value.size()>0)
            std::copy(value.begin(), value.end(), pbuf->begin());
        else 
            throw std::runtime_error("Fail to obtain field values.(" __FILE__ ":" TOSTRING(__LINE__)")" );
        
    }
    else {

        int32 sdsindex = SDreftoindex(file->get_sd_id(), sdsref);
        if(FAIL == sdsindex) 
            throw std::runtime_error("Fail to obtain sdsindex.(" __FILE__ ":" TOSTRING(__LINE__)")" );

        int32 sdsid = SDselect(file->get_sd_id(), sdsindex);
        if(FAIL == sdsid)
            throw std::runtime_error("Fail to obtain sds ID.(" __FILE__ ":" TOSTRING(__LINE__)")" );

        size_t total = 1;
        std::vector<int32> start(rank,0);
        std::vector<int32> edge;
        edge.resize(rank);
        std::vector<int32>stride(rank,1);
        int index=0;

        for(std::vector<hdf4_dim *>::const_iterator i=dims.begin(); i!=dims.end(); i++)
        {
            edge[index] = (*i)->get_size();
            total *= edge[index];
            ++index;
        } // end of for

        size_t total_bytes = total*DFKNTsize(type);
        pbuf->resize(total_bytes);

        // Some fields need to be added by the toolkit. These fields
        // include missing CVs such as latitude and longitude in TRMM level 3 files.
        // remove this later.
        if(value.size()>0)
            std::copy(value.begin(), value.end(), pbuf->begin());
        else {
            if(SDreaddata (sdsid, &start[0], &stride[0], &edge[0], &((*pbuf)[0])) == FAIL) {
                    SDendaccess(sdsid);
                    throw std::runtime_error("Fail to read sds data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
            }
        }
    }
}
	
void hdf4_var_sdfield::get_value(int32 start[], int32 stride[], int32 edge[], void *buf) const {}
	
void hdf4_var_sdfield::get_value(int32 start[], int32 stride[], int32 edge[], std::vector<char> *buf) const
{

    int32 sdsindex = -1;
    int32 sdsid    = -1;
    bool added_fields = false;

    if(sdsref == -1 && value.size() == 0) 
        throw std::runtime_error("Values are not assigned for fields added by the toolkit.(" __FILE__ ":" TOSTRING(__LINE__)")" );

    if(sdsref == -1)
        added_fields = true;
    else {
        sdsindex = SDreftoindex(file->get_sd_id(), sdsref);
        if(FAIL == sdsindex)
            throw std::runtime_error("Fail to obtain sdsindex.(" __FILE__ ":" TOSTRING(__LINE__)")" );
        sdsid = SDselect(file->get_sd_id(), sdsindex);
        if(FAIL == sdsid)
             throw std::runtime_error("Fail to obtain sds ID.(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }
		
    int32 num_elems = 1;
    for(ssize_t i=0; i <rank; i++)
        num_elems *= edge[i];

    std::vector<hdf4_dim*>  dims = this->get_corrected_dimensions();
    int32 total_size = 1;
    std::vector<int32>dim_sizes;
    dim_sizes.resize(dims.size());
    int dim_index = 0;
    for(std::vector<hdf4_dim *>::const_iterator i=dims.begin(); i!= dims.end(); i++) {
        dim_sizes[dim_index] = (*i)->get_size();
        total_size = total_size*(*i)->get_size();
        dim_index++;
    }
 
    // Obtain the subsetted  data values.
    switch(type)
    {
        case DFNT_CHAR8:
        case DFNT_UCHAR8:
        {

            int32 num_bytes = num_elems*sizeof(char); 
            buf->resize(num_bytes); 

            if (true == added_fields) {
                std::vector<char>vall;
                vall.resize(total_size);
                memcpy((void*)(&vall[0]),(const void*)(&value[0]),value.size());

                std::vector<char>out;
                subset<char>(
                             &vall[0], 
                             dims.size(), 
                             &dim_sizes[0],
                             &start[0],
                             &stride[0],
                             &edge[0],
                             &out);

                 memcpy((void*)&((*buf)[0]),(const void*)(&out[0]),sizeof(char)*out.size());

            }
            else {
                if (FAIL == SDreaddata (sdsid, start, stride, edge, (VOIDP)(&((*buf)[0])))) {
                    SDendaccess(sdsid);
                    throw std::runtime_error("Fail to read SDS data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                }
            }

        }
            break;
        case DFNT_UINT8:
        {
            int32 num_bytes = num_elems*sizeof(uint8); 
            buf->resize(num_bytes); 
            //uint8 *v = new uint8[num_elems]; 
            //std::vector<uint8>v;
            //v.resize(num_elems);
            if (true == added_fields) {
                std::vector<uint8>vall;
                vall.resize(total_size);
                memcpy((void*)(&vall[0]),(const void*)(&value[0]),value.size());

                std::vector<uint8>out;
                subset<uint8>(
                             &vall[0], 
                             dims.size(), 
                             &dim_sizes[0],
                             &start[0],
                             &stride[0],
                             &edge[0],
                             &out);

                 memcpy((void*)&((*buf)[0]),(const void*)(&out[0]),sizeof(uint8)*out.size());
            }
            else {
                if (FAIL == SDreaddata (sdsid, start, stride, edge, (VOIDP)(&((*buf)[0])))) {
                    SDendaccess(sdsid);
                    throw std::runtime_error("Fail to read SDS data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                }
            }

        }
            break;
        case DFNT_INT8:
        {
            int32 num_bytes = num_elems*sizeof(int8); 
            buf->resize(num_bytes); 

            if (true == added_fields) {
                std::vector<int8>vall;
                vall.resize(total_size);
                memcpy((void*)(&vall[0]),(const void*)(&value[0]),value.size());

                std::vector<int8>out;
                subset<int8>(
                             &vall[0], 
                             dims.size(), 
                             &dim_sizes[0],
                             &start[0],
                             &stride[0],
                             &edge[0],
                             &out);

                memcpy((void*)&((*buf)[0]),(const void*)(&out[0]),sizeof(int8)*out.size());

            }
            else {
                if (FAIL == SDreaddata (sdsid, start, stride, edge, (VOIDP)(&((*buf)[0])))) {
                    SDendaccess(sdsid);
                    throw std::runtime_error("Fail to read SDS data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                }

            }

        }
            break;
        case DFNT_UINT16:
        {
            int32 num_bytes = num_elems*sizeof(uint16); 
            buf->resize(num_bytes); 
            if (true == added_fields) {
                std::vector<uint16>vall;
                vall.resize(total_size);
                memcpy((void*)(&vall[0]),(const void*)(&value[0]),value.size());

                std::vector<uint16>out;
                subset<uint16>(
                             &vall[0], 
                             dims.size(), 
                             &dim_sizes[0],
                             &start[0],
                             &stride[0],
                             &edge[0],
                             &out);

                 memcpy((void*)&((*buf)[0]),(const void*)(&out[0]),sizeof(uint16)*out.size());

            }
            else {
                if (FAIL == SDreaddata (sdsid, start, stride, edge, (VOIDP)(&((*buf)[0])))) {
                    SDendaccess(sdsid);
                    throw std::runtime_error("Fail to read SDS data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                }

            }

        }
            break;
        case DFNT_INT16:
        {
            int32 num_bytes = num_elems*sizeof(int16); 
            buf->resize(num_bytes); 
            if (true == added_fields) {
                std::vector<int16>vall;
                vall.resize(total_size);
                memcpy((void*)(&vall[0]),(const void*)(&value[0]),value.size());

                std::vector<int16>out;
                subset<int16>(
                             &vall[0], 
                             dims.size(), 
                             &dim_sizes[0],
                             &start[0],
                             &stride[0],
                             &edge[0],
                             &out);

                 memcpy((void*)&((*buf)[0]),(const void*)(&out[0]),sizeof(int16)*out.size());

            }
            else {
                if (FAIL == SDreaddata (sdsid, start, stride, edge, (VOIDP)(&((*buf)[0])))) {
                    SDendaccess(sdsid);
                    throw std::runtime_error("Fail to read SDS data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                }

            }

        }
            break;
        case DFNT_UINT32:
        {
            int32 num_bytes = num_elems*sizeof(uint32); 
            buf->resize(num_bytes); 
            
            if (true == added_fields) {
                std::vector<uint32>vall;
                vall.resize(total_size);
                memcpy((void*)(&vall[0]),(const void*)(&value[0]),value.size());

                std::vector<uint32>out;
                subset<uint32>(
                             &vall[0], 
                             dims.size(), 
                             &dim_sizes[0],
                             &start[0],
                             &stride[0],
                             &edge[0],
                             &out);

                 memcpy((void*)&((*buf)[0]),(const void*)(&out[0]),sizeof(uint32)*out.size());

            }
            else {
                if (FAIL == SDreaddata (sdsid, start, stride, edge, (VOIDP)(&((*buf)[0])))) {
                    SDendaccess(sdsid);
                    throw std::runtime_error("Fail to read SDS data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                }

            }

        }
            break;
        case DFNT_INT32:
        {
            int32 num_bytes = num_elems*sizeof(int32); 
            buf->resize(num_bytes); 
            if (true == added_fields) {
                std::vector<int32>vall;
                vall.resize(total_size);
                memcpy((void*)(&vall[0]),(const void*)(&value[0]),value.size());

                std::vector<int32>out;
                subset<int32>(
                             &vall[0], 
                             dims.size(), 
                             &dim_sizes[0],
                             &start[0],
                             &stride[0],
                             &edge[0],
                             &out);

                 memcpy((void*)&((*buf)[0]),(const void*)(&out[0]),sizeof(int32)*out.size());

            }
            else {
                if (FAIL == SDreaddata (sdsid, start, stride, edge, (VOIDP)(&((*buf)[0])))) {
                    SDendaccess(sdsid);
                    throw std::runtime_error("Fail to read SDS data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                }

            }

        }
            break;
        case DFNT_FLOAT32:
        {
            int32 num_bytes = num_elems*sizeof(float); 
            buf->resize(num_bytes); 

            if(true == added_fields) {

                std::vector<float>vall;
                vall.resize(total_size);
                memcpy((void*)(&vall[0]),(const void*)(&value[0]),value.size());

                std::vector<float>out;
                subset<float>(
                              &vall[0], 
                              dims.size(), 
                              &dim_sizes[0],
                              &start[0],
                              &stride[0],
                              &edge[0],
                               &out);

                memcpy((void*)&((*buf)[0]),(const void*)(&out[0]),sizeof(float)*out.size());

            }
            else {
                if (FAIL == SDreaddata (sdsid, start, stride, edge, (VOIDP)(&((*buf)[0])))) {
                    SDendaccess(sdsid);
                    throw std::runtime_error("Fail to read SDS data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                }

            }
        }
            break;
        case DFNT_FLOAT64:
        {
            int32 num_bytes = num_elems*sizeof(double); 
            buf->resize(num_bytes); 
            if (true == added_fields) {
                std::vector<double>vall;
                vall.resize(total_size);
                memcpy((void*)(&vall[0]),(const void*)(&value[0]),value.size());

                std::vector<double>out;
                subset<double>(
                             &vall[0], 
                             dims.size(), 
                             &dim_sizes[0],
                             &start[0],
                             &stride[0],
                             &edge[0],
                             &out);

                 memcpy((void*)&((*buf)[0]),(const void*)(&out[0]),sizeof(double)*out.size());

            }
            else {
                if (FAIL == SDreaddata (sdsid, start, stride, edge, (VOIDP)(&((*buf)[0])))) {
                    SDendaccess(sdsid);
                    throw std::runtime_error("Fail to read SDS data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                }
            }

        }
            break;
        default:
            throw std::range_error("SDS's type is unknown(" __FILE__ ":" TOSTRING(__LINE__)")" );
    } // end of switch
}

var* hdf4_var_sdfield::_clone() const
{
    return  NULL;
}

bool hdf4_var_sdfield::same_obj_test(var* v) const
{
    return true;
}
 
} //end of namespace
