/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Category: Header file for the HDF4 name space.

Description:

This file is a definition of an HDF4 Scientific Dataset(SDS) and includes related 
method declarations. The class SD defined in the header file hdf4_sd.h contains a 
class member that is vector of pointer of the instances of the class defined here.

*****************************************************************************/


#ifndef HDF4_SDS_H
#define HDF4_SDS_H

#include "eoslib_types.h"
#include "hdf4_attr.h"
#include "hdf4_dim.h"

using namespace eoslib;

namespace hdf4 {

class hdf4_var_sdfield : public var 
{
    public:
        hdf4_var_sdfield();
        ~hdf4_var_sdfield();

        // Get the name of this field.
        const std::string & get_name () const
        {
            return this->m_name;
        }
        
        // Get the new name of this field.
        const std::string & get_new_name () const
        {
            return this->newname;
        }
        
        // Get the dimension rank of this field.
        const int32 get_rank () const
        {
            return this->rank;
        }
	
        //const std::vector < hdf4_attr * >&get_attributes () const
         std::vector < hdf4_attr * >&get_attributes () 
        {
            return this->attrs;
        }
	
        // Get the list of the corrected dimensions.
        const std::vector < hdf4_dim * >&get_corrected_dimensions () const
        {
            return this->correcteddims;
        }
	
        // Get the pointer to the conrrected dimensions.	
        std::vector < hdf4_dim * >*get_corrected_dimensions_ptr ()
        {
            return &(this->correcteddims);
        }

        // Set the list of the corrected dimensions.
        void set_corrected_dimensions (std::vector < hdf4_dim * >dims)
        {
            correcteddims = dims;
        }

        // Get units.
        const std::string get_units () const
        {
            return this->units;
        }

        // Set units
        void set_units (std::string uni)
        {
            units = uni;
        }

        // Get field data type.
        const int get_field_type () const
        {
            return this->fieldtype;
        }

        // Get reference number.
        int32 get_sds_ref () const
        {
            return this->sdsref;
        }

        // Get dimensions.
        const std::vector < hdf4_dim * >&get_dimensions () const
        {
            return this->dims;
        }

        // Get the string representation of data values.
        const std::string get_str_value() const
        {
            return str_value;
        }

        // Get field values and store the values in the vector user provides.
        void get_all_values(std::vector<char> *pbuf) const;

        // Get attributes by name
        hdf4_attr* get_attr_by_name(const std::string& name);

        // Remove the attribute with the specified name. 
        void remove_attr_by_name(std::string);

        // Add a string attribute with the specified name and value. 
        void add_attr_one_str(const std::string&, std::string);

        // Add an attribute.
        void add_attr(hdf4_attr *attr){ attrs.push_back(attr); }
             

        // Add an attribute of which value is a float number.
        void add_attr_one_float(const std::string &, float);

        // Add an attribute of which value is a double number.
        void add_attr_one_double(const std::string &, double);

        // Add an attribute of which value is a short number.
        void add_attr_one_int16(const std::string &, int16);

        // Add an attribute of which value is an int8 number.
        void add_attr_one_int8(const std::string &, int8);

        // Add an attribute of which value is an int number.
        void add_attr_one_int32(const std::string &, int32);



        // Test whether the attribute with the specified name exists. If so, return true, otherwise false.
        const bool has_attr(std::string) const;

        // Generate string representation of all data values. For internal testing only.
        virtual void generate_str_value(int32);

        // Dump all data values. For internal testing only.
        virtual void dump_str(int32);

        // Dump partial or all data values. For internal testing only.
        virtual void dump_str(int32, int32[], int32[], int32[]);
        virtual value_type_t get_type() const;
        virtual void get_value(int32 start[], int32 stride[], int32 edge[], std::vector<char>*) const;
        virtual void get_value(int32 start[], int32 stride[], int32 edge[], void *buf) const;
        virtual var *_clone() const;
        virtual bool same_obj_test(var* v) const;

    public:
        // Name plus path.
        std::string newname;
        // rank: the number of dimensions.
        // type: data type.
        int32 rank; 
        int32 type;
        // Store attribures.
        std::vector< hdf4_attr * > attrs;
        // Store dimensins.
        std::vector< hdf4_dim *> dims;
        // Store corrected dimensions.
        std::vector< hdf4_dim *> correcteddims;
	
        // This flag will specify the fieldtype.
        // 0 means this field is general field.
        // 1 means this field is lat.
        // 2 means this field is lon.
        // 3 means this field is other dimension variable.
        // 4 means this field is added other dimension variable with nature number.
        int fieldtype;
        // Store value for "units" attribute.
        std::string units;
        // Absolute path from root.
        std::string fullpath;
        // Handle to SDS.
        int32 sdsref;	
        // Whether or not condensable.
        bool condenseddim;
        // Original data field name.
        std::string rootfieldname;
        // String representatoin of field value.
        std::string str_value;
        // Store field value.
        std::vector<char> value;

        friend class SD;
        friend class hdf4_file;
};
} // end of namespace

#endif

