/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the methods processing the add-on HDF4 objects in HDF-EOS2 files.

*****************************************************************************/

#include <iostream>
#include <sstream>
#include <string>
#include <cstring>
#include <map>
#include <list>
#include "hdf4_file.h"
#include "hdf4_attr.h"
#include "hdf4_sds.h"
#include "hdf4_vdata.h"
#include "hdf4_dim.h"
#include "hdf4_attr.h"
#include "eoslib_attr.h"
#include "eoslib_group.h"
#include "eoslib_dim.h"
#include "eos2_var_data.h"
#include "eos2_group.h"
#include "mem_attr.h"

using namespace eoslib;

extern hdf4::hdf4_file *file;

namespace hdf4 {

extern std::string remove_special_chars(const std::string&);

std::set< std::set<std::string> > allnames;

std::map<std::string, hdf4::hdf4_var_sdfield *> ssd;
std::map<std::string, hdf4::VDATA *> svdata;

group *grp = NULL;

// Find the group that the missing variable belongs to.
// Comments added by KY after reviewing this routine.
// This routine cannot correctly identify the group since
// the var. name can have a string that shares a group name.
// var. name = g1_a1, grp. name = g1, however, var.name is 
// located at the root instead of g1.
// To correctly find the group, new_name should be used to
// identify the group with /g1/. This can be done in 
// the future since this routine is not used.
// Now comment out this routine.
// KY 2014-11-26
#if 0
void *find_group(group *g, std::string var_name)
{
    std::string str = remove_special_chars(g->get_name());
    if(var_name.find(str)!=std::string::npos)
    {
        grp = g;
        std::list<group*>& children = g->get_child_groups();
        for(std::list<group*>::iterator git = children.begin(); git != children.end(); git++)
        {
            find_group(*git, var_name);
        }
    }  // end of if
}

#endif

// Adding missing SDS and VDATA variables for hybrid files. These variables are added using HDF4 APIs, therefore HDFEOS2 APIs can not recognize them.
// The current design put all SDS objects under the root group,which is not right. However, changing this approach needs bigger change of the design.
// So leave the current way but remember the full path and the original name.
void add_missing_vars(group *root)
{
    eos2_group *g = (eos2_group *)root;

    if(!ssd.empty()) //Add missing SDS variables.
    {
        for(std::map<std::string, hdf4_var_sdfield *>::const_iterator it=ssd.begin(); it!=ssd.end(); it++)
        {
            int32 rank = ((*it).second)->get_rank();
            std::string origname = ((*it).second)->get_new_name();
            std::string name = ((*it).second)->get_name();
            
            std::vector<hdf4_dim *> dims = ((*it).second)->get_dimensions();
            name += "_NONEOS";
					
            eos2_var_data *v = NULL;

            int32 *dimsbuf =(int32 *)malloc(rank*sizeof(int32));
            std::vector<std::string> vdims;

            int index = 0;
            for(std::vector<hdf4_dim *>::const_iterator j=dims.begin(); j!=dims.end(); j++)
            {
                vdims.push_back((*j)->get_name());
                dimsbuf[index++] = (*j)->get_size();
            }
            std::list<dim*> new_dims;

            // Obtain data value.
            int32 sdsindex = SDreftoindex(file->get_sd_id(), ((*it).second)->get_sds_ref());
            if(FAIL == sdsindex)
                throw std::runtime_error("Fail to obtain sdsindex.(" __FILE__ ":" TOSTRING(__LINE__)")" );

            int32 sdsid = SDselect(file->get_sd_id(), sdsindex);
            if(FAIL == sdsid)
                throw std::runtime_error("Fail to obtain sds ID.(" __FILE__ ":" TOSTRING(__LINE__)")" );

            size_t total = 1;
#if 0
            int32 *start = (int32 *)malloc(rank*sizeof(int32));
            std::fill_n(start, rank, 0);
            int32 *edges = (int32 *)malloc(rank*sizeof(int32));
#endif

            std::vector<int32> start(rank,0);
            std::vector<int32> edges;
            edges.resize(rank);
            index=0;

            for(std::vector<hdf4_dim *>::const_iterator i=dims.begin(); i!=dims.end(); i++)
            {
                edges[index] = (*i)->get_size();
                total *= edges[index];
                ++index;
            }

            
            grp = root;
            //find_group(root, (*it).first);
            eos2_group *grp_for_missing_objs = (eos2_group *)grp;
            int32 total_bytes;
            void *ptr = NULL;
	
            // Better to use vector to replace malloc and free. But since this may need the change of code in other functions, leave it for future release.
            switch(((*it).second)->get_type())
            {
                case DFNT_CHAR8:
                case DFNT_UCHAR8:
                { 
                    char *buf = (char *)malloc(total*sizeof(char)); 
                    total_bytes = total*sizeof(char);
                    if(FAIL == SDreaddata (sdsid, &start[0], NULL, &edges[0], buf)) {
                        SDendaccess(sdsid);
                        throw std::runtime_error("Fail to read sds data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    }
                    ptr = buf;
                }
                    break;
                case DFNT_UINT8:
                {
                    uint8 *buf = (uint8 *)malloc(total*sizeof(uint8)); 
                    total_bytes = total*sizeof(uint8); 
                    if(FAIL == SDreaddata (sdsid, &start[0], NULL, &edges[0], buf)) {
                        SDendaccess(sdsid);
                        throw std::runtime_error("Fail to read sds data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    }
                    ptr = buf;
                }
                    break;
                case DFNT_INT8:
                {
                    int8 *buf = (int8 *)malloc(total*sizeof(int8)); 
                    total_bytes = total*sizeof(int8); 
                    if(FAIL == SDreaddata (sdsid, &start[0], NULL, &edges[0], buf)) {
                        SDendaccess(sdsid);
                        throw std::runtime_error("Fail to read sds data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    }
                    ptr = buf;
                }
                    break;
                case DFNT_UINT16:
                {
                    uint16 *buf = (uint16 *)malloc(total*sizeof(uint16)); 
                    total_bytes = total*sizeof(uint16); 
                    if(FAIL == SDreaddata (sdsid, &start[0], NULL, &edges[0], buf)) {
                        SDendaccess(sdsid);
                        throw std::runtime_error("Fail to read sds data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    }
                    ptr = buf;
                }
                    break;
                case DFNT_INT16:
                {
                    int16 *buf = (int16 *)malloc(total*sizeof(int16)); 
                    total_bytes = total*sizeof(int16); 
                    if(FAIL == SDreaddata (sdsid, &start[0], NULL, &edges[0], buf)) {
                        SDendaccess(sdsid);
                        throw std::runtime_error("Fail to read sds data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    }
                    ptr = buf;
                }
                    break;
                case DFNT_UINT32:
                {
                    uint32 *buf = (uint32 *)malloc(total*sizeof(uint32)); 
                    total_bytes = total*sizeof(uint32); 
                    if(FAIL == SDreaddata (sdsid, &start[0], NULL, &edges[0], buf)) {
                        SDendaccess(sdsid);
                        throw std::runtime_error("Fail to read sds data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    }
                    ptr = buf;
                }
                    break;
                case DFNT_INT32:
                {
                    int32 *buf = (int32 *)malloc(total*sizeof(int32)); 
                    total_bytes = total*sizeof(int32); 
                    if(FAIL == SDreaddata (sdsid, &start[0], NULL, &edges[0], buf)) {
                        SDendaccess(sdsid);
                        throw std::runtime_error("Fail to read sds data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    }
                    ptr = buf;
                }
	            break;
                case DFNT_FLOAT32:
                {
                    float *buf = (float *)malloc(total*sizeof(float)); 
                    total_bytes = total*sizeof(float); 
                    if(FAIL == SDreaddata (sdsid, &start[0], NULL, &edges[0], buf)) {
                        SDendaccess(sdsid);
                        throw std::runtime_error("Fail to read sds data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    }
                    ptr = buf;
                }
                    break;
                case DFNT_FLOAT64:
                {
                    double *buf = (double *)malloc(total*sizeof(double)); 
                    total_bytes = total*sizeof(double);
                    if(FAIL == SDreaddata (sdsid, &start[0], NULL, &edges[0], buf)) {
                        SDendaccess(sdsid);
                        throw std::runtime_error("Fail to read sds data.(" __FILE__ ":" TOSTRING(__LINE__)")" );
                    }
                    ptr = buf;
 
                }
                    break;
                default:
                    throw std::range_error("SDS's type is unknown(" __FILE__ ":" TOSTRING(__LINE__)")" );
            } // end of switch

            v = new eos2_var_data( 
                grp_for_missing_objs, 
                grp_for_missing_objs->get_file_info(), 
                (grp_for_missing_objs->get_handle()!=NULL)? grp_for_missing_objs->get_handle(): ((grp_for_missing_objs->get_sw_handle()!=NULL)?grp_for_missing_objs->get_sw_handle(): grp_for_missing_objs->get_gd_handle()), 
                name, 
                origname,
                vdims, 
                dimsbuf, 
                ((*it).second)->get_type(), 
                1, 
                total_bytes, 
                ptr, 
                grp_for_missing_objs->get_dims(), 
                &new_dims); 

            //free(start);
            //free(edges);

            // Add attributes of missing SDS variables.
            for(std::vector<hdf4_attr *>::const_iterator j=((*it).second)->attrs.begin(); j!=((*it).second)->attrs.end(); j++)
            {
                std::string attrname = (*j)->get_name();
                mem_attr *m_attr = new mem_attr(v, attrname);
                m_attr->set_value((*j)->get_type(), (*j)->get_num_elements(), (char *)&((*j)->get_value()[0]));
                v->add_attr(m_attr);
            } // end of for

            (grp->get_vars()).push_back(v);

            // Process newly found dimensions.
            while(!new_dims.empty())
            {
                dim* nd = *new_dims.begin();
                (g->get_dims()).push_back(nd);
                (g->get_hidden_dims()).push_back(nd);
                new_dims.pop_front();
            } // end of while
        } // end of for 
    } // end if if(!ssd.empty())

    if(!svdata.empty()) //Add missing VDATA variables.
    {
        int nr = 1;
        for(std::map<std::string, VDATA *>::const_iterator i=svdata.begin(); i!=svdata.end(); i++)
        {
            //int32 vdata_id = ((*i).second)->get_vdata_id();
            std::string vdataname = ((*i).second)->get_name();
            grp = root;
            //find_group(root, (*i).first);
            std::ostringstream oss;                                       
            oss << "vdata_name" << "_" << nr++;
	
            mem_attr *attr1 = new mem_attr(/*root*/grp, oss.str());
            attr1->set_value(remove_special_chars(vdataname));
            /*root*/grp->add_attr(attr1);
            std::string s;
            for(std::vector<hdf4_var_vdfield *>::const_iterator j=((*i).second)->get_fields().begin(); j!=((*i).second)->get_fields().end(); j++)
            {
                s.append((*j)->get_name()).append(",");

                std::string name = remove_special_chars((*j)->get_new_name());
                std::string origname = (*j)->get_name();
                eos2_var_data *v = NULL;

                std::vector<std::string> vdims;
                int vdrank = ((*j)->get_order()>1)?2:1;
                int32 *dimsbuf =(int32 *)malloc(vdrank*sizeof(int32));
                std::string dimname1 = "VDFDim0_"+name;
                std::string dimname2 = "VDFDim1_"+name;

         
                if(1==vdrank) {
                    vdims.push_back(dimname1);
                    dimsbuf[0] = (*j)->get_num_rec();
                }
                else {
                    vdims.push_back(dimname1);
                    vdims.push_back(dimname2);
                    dimsbuf[0] = (*j)->get_num_rec();
                    dimsbuf[1] = (*j)->get_order();
                }
                    
 
                std::list<dim*> new_dims;

                eos2_group *grp_for_missing_objs = (eos2_group *)grp;
                int len = (*j)->numrec*(*j)->order;
                void *ptr = NULL;
                int32 tot_bytes = 0;

                // Need to change malloc to vector in the future release KY 2014-12-26
                switch((*j)->get_type())
                {
                    case DFNT_CHAR8: 
                    case DFNT_UCHAR8: 
                    { 
                        char *buf = (char *)malloc(len*sizeof(char)); 
                        tot_bytes = len*sizeof(char); 
                        if((*j)->numrec>1 || (*j)->order>1) 
                        { 
                            int index=0; 
                            for(int i=0; i<tot_bytes; i+=sizeof(char)) 
                                buf[index++]=(char)*((char*)&((*j)->value[i])); 
                        } else 
                            memcpy(buf, &((*j)->value[0]), (*j)->value.size()); 
                        ptr = buf;
                    }
                        break;
                    case DFNT_UINT8:
                    {
                        uint8 *buf = (uint8 *)malloc(len*sizeof(uint8)); 
                        tot_bytes = len*sizeof(uint8); 
                        if((*j)->numrec>1 || (*j)->order>1) 
                        { 
                            int index=0; 
                            for(int i=0; i<tot_bytes; i+=sizeof(uint8)) 
                                buf[index++]=(uint8)*((uint8*)&((*j)->value[i])); 
                        } else 
                            memcpy(buf, &((*j)->value[0]), (*j)->value.size());
                        ptr = buf;
                    }
                        break; 
                    case DFNT_INT8:
                    {
                        int8 *buf = (int8 *)malloc(len*sizeof(int8)); 
                        tot_bytes = len*sizeof(int8); 
                        if((*j)->numrec>1 || (*j)->order>1) 
                        { 
                            int index=0; 
                            for(int i=0; i<tot_bytes; i+=sizeof(int8)) 
                                buf[index++]=(int8)*((int8*)&((*j)->value[i])); 
                        } else 
                            memcpy(buf, &((*j)->value[0]), (*j)->value.size()); 
                        ptr = buf;
                    }
                        break;
                    case DFNT_UINT16:
                    {
                        uint16 *buf = (uint16 *)malloc(len*sizeof(uint16)); 
                        tot_bytes = len*sizeof(uint16); 
                        if((*j)->numrec>1 || (*j)->order>1) 
                        { 
                            int index=0; 
                            for(int i=0; i<tot_bytes; i+=sizeof(uint16)) 
                                buf[index++]=(uint16)*((uint16*)&((*j)->value[i])); 
                        } else 
                            memcpy(buf, &((*j)->value[0]), (*j)->value.size()); 
                        ptr = buf;
                    }
                        break;
                    case DFNT_INT16:
                    {
                        int16 *buf = (int16 *)malloc(len*sizeof(int16)); 
                        tot_bytes = len*sizeof(int16); 
                        if((*j)->numrec>1 || (*j)->order>1) 
                        { 
                            int index=0; 
                            for(int i=0; i<tot_bytes; i+=sizeof(int16)) 
                                buf[index++]=(int16)*((int16*)&((*j)->value[i])); 
                        } else 
                            memcpy(buf, &((*j)->value[0]), (*j)->value.size());
                        ptr = buf;
                    }
	                break;
                    case DFNT_UINT32:
                    {
                        uint32 *buf = (uint32 *)malloc(len*sizeof(uint32)); 
                        tot_bytes = len*sizeof(uint32); 
                        if((*j)->numrec>1 || (*j)->order>1) 
                        { 
                            int index=0; 
                            for(int i=0; i<tot_bytes; i+=sizeof(uint32)) 
                                buf[index++]=(uint32)*((uint32*)&((*j)->value[i])); 
                        } else 
                            memcpy(buf, &((*j)->value[0]), (*j)->value.size());
	                ptr = buf;
                    }
                        break;
                    case DFNT_INT32:
                    {
                        int32 *buf = (int32 *)malloc(len*sizeof(int32)); 
                        tot_bytes = len*sizeof(int32); 
                        if((*j)->numrec>1 || (*j)->order>1) 
                        { 
                            int index=0; 
                            for(int i_count=0; i_count<tot_bytes; i_count+=sizeof(int32)) 
                                buf[index++]=(int32)*((int32*)&((*j)->value[i_count])); 
                        } else 
                            memcpy(buf, &((*j)->value[0]), (*j)->value.size()); 
                        ptr = buf;
                    }
                        break;
                    case DFNT_FLOAT32:
                    {
                        float *buf = (float *)malloc(len*sizeof(float)); 
                        tot_bytes = len*sizeof(float); 
                        if((*j)->numrec>1 || (*j)->order>1) 
                        { 
                            int index=0; 
                            for(int i_count=0; i_count<tot_bytes; i_count+=sizeof(float)) 
                                buf[index++]=(float)*((float*)&((*j)->value[i_count])); 
                        } else 
                            memcpy(buf, &((*j)->value[0]), (*j)->value.size()); 
                        ptr = buf;
                    }
                        break;
                    case DFNT_FLOAT64:
                    {
                        double *buf = (double *)malloc(len*sizeof(double)); 
                        tot_bytes = len*sizeof(double); 
                        if((*j)->numrec>1 || (*j)->order>1) 
                        { 
                            int index=0; 
                            for(int i_count=0; i_count<tot_bytes; i_count+=sizeof(double)) 
                                buf[index++]=(double)*((double*)&((*j)->value[i_count])); 
                        } else 
                            memcpy(buf, &((*j)->value[0]), (*j)->value.size()); 
                        ptr = buf;
                    }
                        break;
                    default:
                        throw std::range_error("Vdata Field's type is unknown(" __FILE__ ":" TOSTRING(__LINE__)")" );
                } // end of switch

                v = new eos2_var_data( 
                    grp_for_missing_objs, 
                    grp_for_missing_objs->get_file_info(), 
                    (grp_for_missing_objs->get_handle()!=NULL)? grp_for_missing_objs->get_handle(): ((grp_for_missing_objs->get_sw_handle()!=NULL)?grp_for_missing_objs->get_sw_handle(): grp_for_missing_objs->get_gd_handle()), 
                    name, 
                    origname,
                    vdims, 
                    dimsbuf, 
                    (*j)->get_type(), 
                    (*j)->order, 
                    tot_bytes, 
                    ptr, //buf, 
                    grp_for_missing_objs->get_dims(), 
                    &new_dims); 

                // NEED to investigate VDATA FIELD DIMENSIONS and if memory leaking will occur. Handle it in the next release. KY 2014-12-25
                (grp->get_vars()).push_back(v);
                // Process newly found dimensions.
                while(!new_dims.empty())
                {
                    dim* nd = *new_dims.begin();
                    (g->get_dims()).push_back(nd);
                    (g->get_hidden_dims()).push_back(nd);
                    new_dims.pop_front();
                } // end of while 
            } // end of for(std::vector<hdf4_v
            mem_attr *attr2 = new mem_attr(/*root*/grp, oss.str()+"_field_name_list");
            attr2->set_value(s.substr(0, s.size()-1));
            /*root*/grp->add_attr(attr2);
        } // end of for 
    } // end of if(!svdata.empty())
}

// Find SDS and VDATA variables which are not recognizable by HDFEOS2 APIs.
void find_missing_vars(group *g, const std::string& pathstr)
{
    // Iterate each group.
    std::list<group*>::iterator git;
    std::list<group*>& children = (g)->get_child_groups();

    for(git = children.begin(); git != children.end(); git++)
    {
        std::string grouppath = (*git)->get_name() + "_" + pathstr;
        find_missing_vars(*git, grouppath);
    }
	
    // Iterate each variable.
    std::list<var*>::iterator vit;
    std::list<var*>& vars= (g)->get_vars();
    for(vit = vars.begin(); vit != vars.end(); vit++)
    {
        
        // Remember the variable path
        std::string varpath = pathstr + "_" + (*vit)->get_name();
        std::string varname = remove_special_chars(varpath);

        // Designer notes. For reference only.
#if 0
        /*size_t pos = varname.find_first_of("d_");
        if(pos==0) //Some variables are renamed by adding prefix d_ in hdfeos2 module. This prefix should be removed before comparison.
            varname = varname.substr(2, varname.size());*/
#endif
	
        // ssd is declared globally to store all variables, the following routine will remove the HDF-EOS2 fields.
        for(std::map<std::string, hdf4_var_sdfield *>::iterator it=ssd.begin(); it!=ssd.end(); it++)
        {
            std::string key = (*it).first;
            if(key==varname) 
            {
                ssd.erase(it);
                break;
            }
        } // end of for
    } // end of for 
}

// Add missing attriutes for SDS variables. These variables are recognizable by HDFEOS2 APIs, but some their attributes are not recognizable by HDFEOS2 APIs, which are added using HDF4 APIs.
void add_missing_var_attrs(group *g, const std::string& pathstr)
{
    // Iterate each group
    std::list<group*>::iterator git;
    std::list<group*>& children = (g)->get_child_groups();

    for(git = children.begin(); git != children.end(); git++)
    {
        std::string grouppath = (*git)->get_name() + "_" + pathstr;
        add_missing_var_attrs(*git, grouppath);
    }

    // Iterate each variable
    std::list<var*>::iterator vit;
    std::list<var*>& vars= (g)->get_vars();
    for(vit = vars.begin(); vit != vars.end(); vit++)
    {
        std::string varpath= pathstr + "_" + (*vit)->get_name(); 
        std::string varname = remove_special_chars(varpath);
		
        for(std::map<std::string, hdf4_var_sdfield *>::const_iterator it=ssd.begin(); it!=ssd.end(); it++)
        {
            std::string key = (*it).first;
            if(key==varname) 
            {
                hdf4_var_sdfield *sdfield = (*it).second;
                for(std::vector<hdf4_attr *>::const_iterator j=sdfield->attrs.begin(); j!=sdfield->attrs.end(); j++)
                {
                    std::string attrname = (*j)->get_name();
                    attr *attribute = (*vit)->get_attr_by_name(attrname);
                    if(attribute==NULL) //No such attribute.
                    {
                        // Get value.
                        mem_attr *m_attr = new mem_attr(*vit, attrname);

                        // Set value.
                        //m_attr->set_value((*j)->get_str());
                        m_attr->set_value((*j)->get_type(), (*j)->get_num_elements(), (char *)&((*j)->get_value()[0]));
                        (*vit)->add_attr(m_attr);
                    } // end of if
                } // end of for
            } // end of if(key==varname)
        } // end of for(std::map< 		
    } // end of for
}

//Add missing file attributes. These file attributes are not recognizable by HDFEOS2 APIs. They are identified by comparing file attributes separately read by using HDFEOS2 APIs and HDF4 APIs. 
void add_missing_file_attrs(group *root)
{
    for(std::vector<hdf4_attr *>::const_iterator i=file->sd->get_attrs().begin(); i!=file->sd->get_attrs().end(); i++)
    {
        std::string attrname = (*i)->get_name();
        mem_attr *m_attr = new mem_attr(root, attrname);
        m_attr->set_value((*i)->get_type(), (*i)->get_num_elements(), (char *)&((*i)->get_value()[0]));
        root->add_attr(m_attr);
    } // end of for
}

// Obtain all SDS and VDATA variables using HDF4 APIs.
void add_missing_objects(group *root) //Prepare for hybrid files.
{
    if(file==NULL)
        return;

    // If the opening file is HDF4, no further processing is taken.
    int len = 0;
    int num_grids = 0;
    int num_swaths = 0;

    //const char *filename = file->get_file_name().c_str();
    std::string filename = file->get_file_name();
    num_grids = GDinqgrid(const_cast<char*>(filename.c_str()), NULL, (int32*)&len);
    num_swaths = SWinqswath(const_cast<char*>(filename.c_str()), NULL, (int32*)&len);
    if(num_grids==0 && num_swaths==0) // HDF4.
        return;

    if(!allnames.empty())
        allnames.clear();
    if(!ssd.empty())
        ssd.clear();
    if(!svdata.empty())
        svdata.clear();
    for(std::vector<hdf4_var_sdfield *>::const_iterator i=file->sd->sdfields.begin(); i!=file->sd->sdfields.end(); i++)
    {
        std::string key = remove_special_chars((*i)->newname);

        //The following string operations guarantee key will exactly match the absolute name of variable exported from hdfeos2 library [05/03/2012 LD].
        // Note in insert_orig_field_path, "Data Fields" and "Geolocation Fields" are ignored from the HDF-EOS2 fields newnames.
        // That's why we need to change the "key" value.
        if('_' == key[0])
            key = key.substr(1);
        std::string str1 = "_Data_Fields";
        size_t pos1 = key.find(str1);
        if(pos1!=std::string::npos)
            key = key.erase(pos1, str1.length());
        else {
            // We also need to ignore "Geolocation Fields". This vgroup is a predefined vgroup in an HDF-EOS2 swath. KY 2013-01-04
            std::string str2 = "_Geolocation_Fields";
            size_t pos2 = key.find(str2);
            if(pos2!=std::string::npos)
                key = key.erase(pos2,str2.length());
        }

        ssd.insert(std::pair<std::string, hdf4_var_sdfield *>(key, (*i)));
    } // end of for

    for(std::vector<VDATA *>::const_iterator i=file->vds.begin(); i!=file->vds.end(); i++)
    {
        std::string str = remove_special_chars((*i)->get_new_name());
        svdata.insert(std::pair<std::string, VDATA *>(str, (*i)));
    }

    // This function must be called beforehand since even HDF-EOS2 variables may miss some attributes.
    add_missing_var_attrs(root, "");
	
    find_missing_vars(root, "");
    add_missing_vars(root);

    add_missing_file_attrs(root);
}

// Save dimension names for either latitude and longitude into a set of set of string(very strange).
// allnames may be {"XDim YDim", "XDim YDim"}. This function is used by add_coordinates_attr in hdf4_flt_cf.cpp.
// Not sure why it is implemented this way. May need to check more in the next release. KY 2014-12-25
void generate_dim_names()
{
    for(std::vector<hdf4_var_sdfield *>::const_iterator i=file->sd->sdfields.begin(); i!=file->sd->sdfields.end(); i++)
    {
        if((*i)->fieldtype==1 || (*i)->fieldtype==2)
        {
            std::set<std::string> dimnames;
            for(std::vector<hdf4_dim *>::const_iterator j=(*i)->get_corrected_dimensions().begin(); j!=(*i)->get_corrected_dimensions().end(); j++)
            {
                dimnames.insert((*j)->get_name());
            }
            allnames.insert(dimnames);
        } // end of if
    } // end of for
}

// Check whether or not a variable has attribute coordinates. If a variable does not have all dimensoins coordinate variable has, this variable is not allowed to have such attribute. 
// I don't understand the logic of this routine. KY 2014-12-25 (Document it and may need to investigate more in the next release).
bool check_cv(hdf4_var_sdfield *var)
{
    std::set< std::string > mynames;
    for(std::vector<hdf4_dim *>::const_iterator j=var->get_corrected_dimensions().begin(); j!=var->get_corrected_dimensions().end(); j++)
    {
        mynames.insert((*j)->get_name());
    }

    for(std::set< std::set<std::string> >::iterator i=allnames.begin(); i!=allnames.end(); i++)
    {
        int nr = 0;
        std::set<std::string>::iterator j;
        for(j=(*i).begin(); j!=(*i).end(); j++)
        {
            if(mynames.find(*j)==mynames.end())
                ++nr;
        }
        // nr will never be>(*i).size(), why using the following check. It seems that if nr!=0, false should return.
        if(nr>0 && nr<(*i).size()) //This var does not have all dims of cv, only part of them.
            return false;
    } // end of for

    return true;
}

// This is one way to generate unique name in a given namelist.
void gen_unique_name(std::string &str,std::set<std::string>& namelist, int&clash_index) {

    std::pair<std::set<std::string>::iterator,bool> ret;
    std::string newstr = "";
    std::stringstream sclash_index;
    sclash_index << clash_index;
    newstr = str + sclash_index.str();

    ret = namelist.insert(newstr);
    if (false == ret.second) {
        clash_index++;
        gen_unique_name(str,namelist,clash_index);
    }
    else
        str = newstr;
}

void
Handle_NameClashing(std::vector<std::string>&newobjnamelist,std::set<std::string>&objnameset) {

    //set<string> objnameset;
    std::pair<std::set<std::string>::iterator,bool> setret;
    std::set<std::string>::iterator iss;

    std::vector<std::string> clashnamelist;
    std::vector<std::string>::iterator ivs;

    std::map<int,int> cl_to_ol;
    int ol_index = 0;
    int cl_index = 0;

    std::vector<std::string>::const_iterator irv;

    for (irv = newobjnamelist.begin(); irv != newobjnamelist.end(); ++irv) {
        setret = objnameset.insert(*irv);
        if (false == setret.second ) {
            clashnamelist.insert(clashnamelist.end(),(*irv));
            cl_to_ol[cl_index] = ol_index;
            cl_index++;
        }
        ol_index++;
    }

    // Now change the clashed elements to unique elements; 
    // Generate the set which has the same size as the original vector.
    for (ivs=clashnamelist.begin(); ivs!=clashnamelist.end(); ivs++) {
        int clash_index = 1;
        std::string temp_clashname = *ivs +'_';
        gen_unique_name(temp_clashname,objnameset,clash_index);
        *ivs = temp_clashname;
    }

    // Now go back to the original vector, make it unique.
    for (unsigned int i =0; i <clashnamelist.size(); i++)
        newobjnamelist[cl_to_ol[i]] = clashnamelist[i];

}

void Handle_NameClashing(std::vector<std::string>&newobjnamelist) {

    std::set<std::string> objnameset;
    Handle_NameClashing(newobjnamelist,objnameset);
}


} //end of namesace
