/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to the processing of HDF vdata field.

*****************************************************************************/

#include "hdf4_vdata_field.h"
#include <cstring>

namespace hdf4 {

hdf4_var_vdfield::hdf4_var_vdfield():var(NULL, "non-valid name"), type(-1), rank(-1), order(-1), numrec(-1), size(-1), vdata_id(-1) {}

hdf4_var_vdfield::~hdf4_var_vdfield()
{
    for(std::vector<hdf4_attr *>::const_iterator i=attrs.begin(); i!=attrs.end(); i++)
        delete *i;
}

hdf4_attr* hdf4_var_vdfield::get_attr_by_name(const std::string& name) {

    std::vector<hdf4_attr*>& attrs = get_attributes();
    std::vector<hdf4_attr*>::iterator it;
    for(it = attrs.begin(); it != attrs.end(); it++) {
        if((*it)->get_name() == name)
            return *it;
    }
    return NULL;

}

const bool hdf4_var_vdfield::has_attr(std::string attrname) const
{
    for(std::vector<hdf4_attr *>::const_iterator it=attrs.begin(); it!=attrs.end(); it++)
        if((*it)->get_name()==attrname)
            return true;
    return false;
}

void hdf4_var_vdfield::add_attr_one_str(std::string attr_name, std::string attr_value)
{
    std::vector<char> v(attr_value.length());
    copy(attr_value.begin(), attr_value.end(), v.begin());
    hdf4_attr *attr = new hdf4_attr(attr_name, DFNT_CHAR8, attr_value.length(), v);
    attrs.push_back(attr);
}
	
void hdf4_var_vdfield::read_attributes(int32 vdata_id_r, int32 fieldindex) 
{
    int32 nattrs = 0;
    int32 attrsize = 0;
    int32 status = -1;

    char attr_name[H4_MAX_NC_NAME];
	
    nattrs = VSfnattrs (vdata_id_r, fieldindex);
    if (nattrs == FAIL)
        throw std::runtime_error("VSfnattrs failed.(" __FILE__ ":" TOSTRING(__LINE__)")");

    if (nattrs > 0) 
    {
        for (int i = 0; i < nattrs; i++) 
        {
            hdf4_attr *attr = new hdf4_attr();
                        	
            status = VSattrinfo (vdata_id_r, fieldindex, i, attr_name, &attr->type, &attr->count, &attrsize);
            std::string tmpname(attr_name);
            attr->m_name = tmpname;

#if 0
            if(0) // For debugging only.
                std::cout << "\t\t\tfield attr name=" << attr->m_name << " type=" << attr->type << " count=" << attr->count << " size=" << attrsize << std::endl;
#endif
            if (status == FAIL) {
                delete attr;
                throw std::runtime_error("VSattrinfo failed.(" __FILE__ ":" TOSTRING(__LINE__)")");
            }
            attr->value.resize (attrsize);
            if (VSgetattr (vdata_id_r, fieldindex, i, &attr->value[0]) == FAIL){
                delete attr;
                throw std::runtime_error("VSgetattr failed .(" __FILE__ ":" TOSTRING(__LINE__)")");
            }
            attrs.push_back (attr);
        } // end of for(int i=0
    } // end of  if (nattrs > 0) 
}

value_type_t hdf4_var_vdfield::get_type() const
{
    return type;
}

// The following function is not used by APIs nor is thoroughly tested. KY 2014-12-26
std::string hdf4_var_vdfield::get_str_value()
{
    if(value.size()==0)
        return "";
		
    std::stringstream out;
    switch(type)
    {
        case DFNT_CHAR8:  // For order(or array size)>1
        { 
            if(numrec>1 || order>1)
            { 
                int len = numrec*order;
                int32 tot_bytes = len*sizeof(char); 
                int index=0; 
                std::vector<char>buf;
                buf.resize(len);
                //char buf[len]; 
                for(int i=0; i<tot_bytes; i+=sizeof(char)) 
                    buf[index++]=(char)*((char*)&(value[i])); 
                std::string a_str; 
                for(int l=0; l<=len-order; l+=order) 
                { 
                    a_str += "\""; 
                    for(size_t k=l; k<l+order; k++) 
                        if((char)buf[k]>31 && (char)buf[k]<127) 
                            a_str += (char)buf[k]; 
                    a_str += "\""; 
                    if(l!=len-order) 
                        	a_str += ","; 
                } 
                out << a_str; 
            } else 
                out << *((char *)&(get_value())[0]); 
        } 
            break;
        case DFNT_UCHAR8: 
        { 
            if(numrec>1 || order>1) 
            { 
                int len = numrec*order;
                int32 tot_bytes = len*sizeof(char); 
                int index=0; 
                std::vector<char>buf;
                buf.resize(len);
                //char buf[len]; 
                for(int i=0; i<tot_bytes; i+=sizeof(char)) 
                    buf[index++]=(char)*((char*)&(value[i])); 
                std::string a_str; 
                char c[100]; 
                for(int l=0; l<len; l++) 
                { 
                    sprintf(c, "%c", buf[l]); 
                    a_str.append(c); 
                    if(l!=len-1) 
                        a_str.append(","); 
                } 
                out << a_str; 
            } else 
                out << *((char *)&value[0]); 
        } 
            break;
        case DFNT_UINT8: 
        { 
            if(numrec>1 || order>1) 
            { 
                int len = numrec*order;
                int32 tot_bytes = len*sizeof(uint8); 
                int index=0; 
                std::vector<uint8> buf;
                buf.resize(len);
                //uint8 buf[len]; 
                for(int i=0; i<tot_bytes; i+=sizeof(uint8)) 
                    buf[index++]=(uint8)*((uint8*)&(value[i])); 
                std::string a_str; 
                char c[100]; 
                for(int l=0; l<len; l++) 
                { 
                    sprintf(c, "%u", buf[l]); 
                    a_str.append(c); 
                    if(l!=len-1) 
                        a_str.append(","); 
                } 
                out << a_str; 
            } else 
                out << *((uint8 *)&value[0]); 
        } 
            break;
        case DFNT_INT8: 
        { 
            if(numrec>1 || order>1) 
            { 
                int len = numrec*order;
                int32 tot_bytes = len*sizeof(int8); 
                int index=0; 
                //int8 buf[len]; 
                std::vector<int8>buf;
                buf.resize(len);
                for(int i=0; i<tot_bytes; i+=sizeof(int8)) 
                    buf[index++]=(int8)*((int8*)&(value[i])); 
                std::string a_str; 
                char c[100]; 
                for(int l=0; l<len; l++) 
                { 
                    sprintf(c, "%d", buf[l]); 
                    a_str.append(c); 
                    if(l!=len-1) 
                        a_str.append(","); 
                } 
                out << a_str; 
            } else 
                out << *((int8 *)&value[0]); 
        } 
            break;
        case DFNT_UINT16: 
        { 
            if(numrec>1 || order>1) 
            { 
                int len = numrec*order;
                int32 tot_bytes = len*sizeof(uint16); 
                int index=0; 
                //uint16 buf[len]; 
                std::vector<uint16>buf;
                buf.resize(len);
                for(int i=0; i<tot_bytes; i+=sizeof(uint16)) 
                    buf[index++]=(uint16)*((uint16*)&(value[i])); 
                std::string a_str; 
                char c[100]; 
                for(int l=0; l<len; l++) 
                { 
                    sprintf(c, "%u", buf[l]); 
                    a_str.append(c); 
                    if(l!=len-1) 
                        a_str.append(","); 
                } 
                out << a_str; 
            } else 
                out << *((uint16 *)&value[0]); 
        } 
            break;
        case DFNT_INT16: 
        { 
            if(numrec>1 || order>1) 
            { 
                int len = numrec*order;
                int32 tot_bytes = len*sizeof(int16); 
                int index=0; 
                //int16 buf[len]; 
                std::vector<int16>buf;
                buf.resize(len);
                for(int i=0; i<tot_bytes; i+=sizeof(int16)) 
                    buf[index++]=(int16)*((int16*)&(value[i])); 
                std::string a_str; 
                char c[100]; 
                for(int l=0; l<len; l++) 
                { 
                    sprintf(c, "%d", buf[l]); 
                    a_str.append(c); 
                    if(l!=len-1) 
                        a_str.append(","); 
                } 
                out << a_str; 
            } else 
            out << *((int16 *)&value[0]); 
        } 
            break;
        case DFNT_UINT32: 
        { 
            if(numrec>1 || order>1) 
            { 
                int len = numrec*order;
                int32 tot_bytes = len*sizeof(uint32); 
                int index=0; 
                //uint32 buf[len]; 
                std::vector<uint32>buf;
                buf.resize(len);
                for(int i=0; i<tot_bytes; i+=sizeof(uint32)) 
                    buf[index++]=(uint32)*((uint32*)&(value[i])); 
                std::string a_str; 
                char c[100]; 
                for(int l=0; l<len; l++) 
                { 
                    sprintf(c, "%u", buf[l]); 
                    a_str.append(c); 
                    if(l!=len-1) 
                        a_str.append(","); 
                } 
                out << a_str; 
            } else 
                out << *((uint32 *)&value[0]); 
        } 
            break;
        case DFNT_INT32: 
        { 
            if(numrec>1 || order>1) 
            { 
                int len = numrec*order;
                int32 tot_bytes = len*sizeof(int32); 
                int index=0; 
                //int32 buf[len]; 
                std::vector<int32>buf;
                buf.resize(len);
                for(int i=0; i<tot_bytes; i+=sizeof(int32)) 
                    buf[index++]=(int32)*((int32*)&(value[i])); 
                std::string a_str; 
                char c[100]; 
                for(int l=0; l<len; l++) 
                { 
                    sprintf(c, "%d", buf[l]); 
                    a_str.append(c); 
                    if(l!=len-1) 
                        a_str.append(","); 
                } 
                out << a_str; 
            } else 
                out << *((int32 *)&value[0]); 
        } 
            break;
        case DFNT_FLOAT32: 
        { 
            if(numrec>1 || order>1) 
            { 
                int len = numrec*order;
                int32 tot_bytes = len*sizeof(float); 
                int index=0; 
                //float buf[len]; 
                std::vector<float>buf;
                buf.resize(len);
                for(int i=0; i<tot_bytes; i+=sizeof(float)) 
                    buf[index++]=(float)*((float*)&(value[i])); 
                std::string a_str; 
                char c[100]; 
                for(int l=0; l<len; l++) 
                { 
                    sprintf(c, "%.6f", buf[l]); 
                    a_str.append(c); 
                    if(l!=len-1) 
                        a_str.append(","); 
                } 
                out << a_str; 
            } else 
                out << *((float *)&value[0]); 
        } 
            break;
        case DFNT_FLOAT64: 
        { 
            if(numrec>1 || order>1) 
            { 
                int len = numrec*order;
                int32 tot_bytes = len*sizeof(double); 
                int index=0; 
                //double buf[len]; 
                std::vector<double>buf;
                buf.resize(len);
                for(int i=0; i<tot_bytes; i+=sizeof(double)) 
                    buf[index++]=(double)*((double*)&(value[i])); 
                std::string a_str; 
                char c[100]; 
                for(int l=0; l<len; l++) 
                { 
                    sprintf(c, "%8.3e", buf[l]); 
                    a_str.append(c); 
                    if(l!=len-1) 
                        a_str.append(","); 
                } 
                out << a_str; 
            } else 
                out << *((double *)&value[0]); 
        } 
            break;
        default:
             throw std::range_error("Vdata Field's type is unknown(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }; // end of switch
    return out.str();
}


// Obtain subsetted values
void hdf4_var_vdfield::get_value(int32 start[], int32 stride[], int32 edge[], void *buf) const {

    int32 r;

    // Seek the position of the starting point
    if (VSseek (this->vdata_id, (int32) start[0]) == -1) {
        VSdetach (this->vdata_id);
        throw std::runtime_error("VSseek failed(" __FILE__ ":" TOSTRING(__LINE__)")");
    }

    // Prepare the vdata field
    if (VSsetfields (this->vdata_id, this->m_name.c_str ()) == -1) {
        VSdetach (this->vdata_id);
        throw std::runtime_error("VSsetfields failed(" __FILE__ ":" TOSTRING(__LINE__)")");
    }

    int32 vdfelms = (this->order) * (1+(edge[0]-1) * stride[0]);
    int nelms = 1;
    if (this->order > 1) {
        nelms = edge[1]*edge[0];
    }
    else 
        nelms = edge[0];

    // Loop through each data type
    switch (this->type) {
        case DFNT_INT8:
        case DFNT_CHAR8:
        {
            std::vector<char>val;
            val.resize(nelms);
            std::vector<char>orival;
            orival.resize(vdfelms);

            // Read the data
            r = VSread (this->vdata_id, (uint8 *) &orival[0], 1+(edge[0] -1)* stride[0],
                FULL_INTERLACE);

            if (r == -1) {
                VSdetach (this->vdata_id);
                throw std::runtime_error("VSread failed(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            // Obtain the subset portion of the data
            if (this->order > 1) {
               for (int i = 0; i < edge[0]; i++)
                    for (int j = 0; j < edge[1]; j++)
                        val[i * edge[1] + j] =
                        orival[i * this->order * stride[0] + start[1] + j * stride[1]];
            }
            else {
                for (int i = 0; i < edge[0]; i++)
                    val[i] = orival[i * stride[0]];
            }

            memcpy(buf,(void*)&val[0],nelms*sizeof(char));

        }

            break;
        case DFNT_UINT8:
        case DFNT_UCHAR8:
        {
            std::vector<unsigned char>val;
            val.resize(nelms);
            std::vector<unsigned char>orival;
            orival.resize(vdfelms);

            r = VSread (vdata_id, (uint8 *) &orival[0], 1+(edge[0] -1)* stride[0], FULL_INTERLACE);
            if (r == -1) {
                VSdetach (vdata_id);
                throw std::runtime_error("VSread failed(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            if (this->order > 1) {
                for (int i = 0; i < edge[0]; i++)
                    for (int j = 0; j < edge[1]; j++)
                        val[i * edge[1] + j] =	orival[i * this->order * stride[0] + start[1] + j * stride[1]];
            }
            else {
                for (int i = 0; i < edge[0]; i++)
                    val[i] = orival[i * stride[0]];
            }

            memcpy(buf,(void*)&val[0],nelms*sizeof(unsigned char));
        }

            break;

        case DFNT_INT16:
        {
            std::vector<int16>val;
            val.resize(nelms);
            std::vector<int16>orival;
            orival.resize(vdfelms);

            r = VSread (vdata_id, (uint8 *) &orival[0], 1+(edge[0] -1)* stride[0],
                FULL_INTERLACE);
            if (r == -1) {
                VSdetach (vdata_id);
                throw std::runtime_error("VSread failed(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            if (this->order > 1) {
                for (int i = 0; i < edge[0]; i++)
                    for (int j = 0; j < edge[1]; j++)
                        val[i * edge[1] + j] =	orival[i * this->order * stride[0] + start[1] + j * stride[1]];
            }
            else {
                for (int i = 0; i < edge[0]; i++)
                    val[i] = orival[i * stride[0]];
            }
            memcpy(buf,(void*)&val[0],nelms*sizeof(int16));
        }
            break;

        case DFNT_UINT16:

        {
            std::vector<uint16>val;
            val.resize(nelms);
            std::vector<uint16>orival;
            orival.resize(vdfelms);

            r = VSread (vdata_id, (uint8 *) &orival[0], 1+(edge[0] -1)* stride[0],
                FULL_INTERLACE);
            if (r == -1) {
                VSdetach (vdata_id);
                throw std::runtime_error("VSread failed(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            if (this->order > 1) {
                for (int i = 0; i < edge[0]; i++)
                    for (int j = 0; j < edge[1]; j++)
                        val[i * edge[1] + j] =	orival[i * this->order * stride[0] + start[1] + j * stride[1]];
            }
            else {
                for (int i = 0; i < edge[0]; i++)
                    val[i] = orival[i * stride[0]];
            }
            memcpy(buf,(void*)&val[0],nelms*sizeof(uint16));

        }

            break;
        case DFNT_INT32:
        {
            std::vector<int32> val;
            val.resize(nelms);
            std::vector<int32>orival;
            orival.resize(vdfelms);

            r = VSread (vdata_id, (uint8 *) &orival[0], 1+(edge[0] -1)* stride[0],
                FULL_INTERLACE);
            if (r == -1) {
                VSdetach (vdata_id);
                throw std::runtime_error("VSread failed(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            if (this->order > 1) {
                for (int i = 0; i < edge[0]; i++)
                    for (int j = 0; j < edge[1]; j++)
                        val[i * edge[1] + j] =	orival[i * this->order * stride[0] + start[1] + j * stride[1]];
            }
            else {
                for (int i = 0; i < edge[0]; i++)
                    val[i] = orival[i * stride[0]];
            }
            memcpy(buf,(void*)&val[0],nelms*sizeof(int32));
        }
            break;

        case DFNT_UINT32:

        {
            std::vector<uint32>val;
            val.resize(nelms);
            std::vector<uint32>orival;
            orival.resize(vdfelms);

            r = VSread (vdata_id, (uint8 *) &orival[0], 1+(edge[0] -1)* stride[0],
                FULL_INTERLACE);
            if (r == -1) {
                VSdetach (vdata_id);
                throw std::runtime_error("VSread failed(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            if (this->order > 1) {
                for (int i = 0; i < edge[0]; i++)
                    for (int j = 0; j < edge[1]; j++)
                        val[i * edge[1] + j] =	orival[i * this->order * stride[0] + start[1] + j * stride[1]];
            }
            else {
                for (int i = 0; i < edge[0]; i++)
                    val[i] = orival[i * stride[0]];
            }
            memcpy(buf,(void*)&val[0],nelms*sizeof(uint32));
        }
            break;
        case DFNT_FLOAT32:
        {
            std::vector<float32>val;
            val.resize(nelms);
            std::vector<float32>orival;
            orival.resize(vdfelms);

            r = VSread (vdata_id, (uint8 *) &orival[0], 1+(edge[0] -1)* stride[0],
                FULL_INTERLACE);
            if (r == -1) {
                VSdetach (vdata_id);
                throw std::runtime_error("VSread failed(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            if (this->order > 1) {
                for (int i = 0; i < edge[0]; i++)
                    for (int j = 0; j < edge[1]; j++)
                        val[i * edge[1] + j] =	orival[i * this->order * stride[0] + start[1] + j * stride[1]];
            }
            else {
                for (int i = 0; i < edge[0]; i++)
                    val[i] = orival[i * stride[0]];
            }
            memcpy(buf,(void*)&val[0],nelms*sizeof(float32));
        }
            break;
        case DFNT_FLOAT64:
        {

            std::vector<float64>val;
            val.resize(nelms);
            std::vector<float64>orival;
            orival.resize(vdfelms);


            r = VSread (vdata_id, (uint8 *) &orival[0], 1+(edge[0] -1)* stride[0],
                FULL_INTERLACE);
            if (r == -1) {
                VSdetach (vdata_id);
                throw std::runtime_error("VSread failed(" __FILE__ ":" TOSTRING(__LINE__)")");
            }

            if (this->order > 1) {
                for (int i = 0; i < edge[0]; i++)
                    for (int j = 0; j < edge[1]; j++)
                        val[i * edge[1] + j] =	orival[i * this->order * stride[0] + start[1] + j * stride[1]];
            }
            else {
                for (int i = 0; i < edge[0]; i++)
                    val[i] = orival[i * stride[0]];
            }
            memcpy(buf,(void*)&val[0],nelms*sizeof(float64));
        }
            break;
        default:
            VSdetach (vdata_id);
            throw std::runtime_error("Datatype is not supported(" __FILE__ ":" TOSTRING(__LINE__)")");

    }

}

var* hdf4_var_vdfield::_clone() const
{
    return  NULL;
}

bool hdf4_var_vdfield::same_obj_test(var* v) const
{
    return true;
}

} //end of namespace
