/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes the implementation related to creating extra variables.


*****************************************************************************/

#include "mem_var.h"
#include "eoslib_types.h"

namespace eoslib
{

mem_var::mem_var(group *g, const std::string& name): var(g, name)
{
}

mem_var::mem_var(const mem_var& r):
    var(r),
    m_type(r.m_type),
    m_buf(r.m_buf)
{
}

mem_var::~mem_var()
{
}

value_type_t mem_var::get_type() const
{
    return m_type;
}

void mem_var::get_value(
    int32 start[], int32 stride[], int32 edge[],
    void *buf) const
{
    std::vector<int32> dim_sizes = this->get_dim_sizes();
//std::cerr<<"coming to get_value subset"<<std::endl;
    switch(m_type)
    {
#define HANDLE(TYPE, type) \
    case DFNT_##TYPE: \
    { \
        std::vector<type> tmp; \
        var::subset<type>( \
        (type*)&m_buf[0],  \
        dim_sizes.size(), \
        &dim_sizes[0], \
        start, \
        stride, \
        edge, \
        &tmp); \
        std::copy(tmp.begin(), tmp.end(), (type*)buf); \
        break; \
    }

        HANDLE(CHAR8, char8);
        HANDLE(INT8, int8);
        HANDLE(INT16, int16);
        HANDLE(INT32, int32);
        HANDLE(UCHAR8, uchar8);
        HANDLE(UINT8, uint8);
        HANDLE(UINT16, uint16);
        HANDLE(UINT32, uint32);
        HANDLE(FLOAT32, float32);
        HANDLE(FLOAT64, float64);

#undef HANDLE
        default:
            throw std::range_error("Unknown type(" __FILE__ ":" TOSTRING(__LINE__)")" );
    } // end of switch
}

void mem_var::set(value_type_t type, const std::list<dim*>& dims,
    const void *buf)
{
    m_type = type;
    m_dims = dims;

    int tot = 1;
    for(std::list<dim*>::const_iterator it = dims.begin();
        it != dims.end();
        it++)
        tot *= (*it)->get_size();

    size_t tot_bytes = tot * get_type_size(type);
    m_buf.resize(tot_bytes);
    std::copy((char*)buf, (char*)buf + tot_bytes, &m_buf[0]);
}

void mem_var::set(value_type_t type, dim* d1,
    const void *buf)
{
    std::list<dim*> dims;
    dims.push_back(d1);
    set(type, dims, buf);
}
void mem_var::set(value_type_t type, dim* d1, dim *d2,
    const void *buf)
{
    std::list<dim*> dims;
    dims.push_back(d1);
    dims.push_back(d2);
    set(type, dims, buf);
}

var* mem_var::_clone() const
{
    mem_var *w = new mem_var(*this);
    return w;
}

bool mem_var::same_obj_test(var* v) const
{
    mem_var *w = dynamic_cast<mem_var*>(v);
    if(w==NULL)
        return false;
    if(this->m_buf != w->m_buf)
        return false;
    return true;
}

} // namespace
