/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes utility methods of implementation for HDF-EOS2 files.


*****************************************************************************/

#include <vector>
#include <hdf.h>
#include <HdfEosDef.h>
#include "misc_eos2.h"
#include "misc_util.h"
#include <assert.h>
#include <iostream>
#include <cstring>

struct func_arg
{
    walk_callback func;
    void *arg;
};

int _eos2util_walk_grsw(const char *filename, int (*func)(char *, void *), void *arg, int32 (*inq)(char * filename, char *swathlist, int32 *strbufsize))
{
    int len = 0;
    int num_grids = 0;
    char *buf = NULL;
    int ret=0;
    //struct func_arg new_arg = {func, arg};

    num_grids = inq((char*)filename, NULL, (int32*)&len);

    if(num_grids <= 0)
        return num_grids;
    if(len <= 0)
        return -1;
    buf = (char*)calloc(len + 1, sizeof(char));
    if(!buf)
        return -1;
    num_grids = inq((char*)filename, buf, (int32*)&len);

    if((num_grids <= 0) ||(len <= 0))
        ret= -1;
    else
        ret = util_walk_str(buf, ",", func, arg);

    if(buf)
        free(buf);
    return ret;
}

// ret: # of walked grids
// -1: error
int eos2util_walk_grids(const char *filename, int (*func)(char *, void *), void *arg)
{
    return _eos2util_walk_grsw(filename, func, arg, GDinqgrid);
}

int eos2util_walk_swaths(const char *filename, int (*func)(char *, void *), void *arg)
{
    return _eos2util_walk_grsw(filename, func, arg, SWinqswath);
}

int _eos2util_walk_grsw_datafields(int32 id, int (*func)(char *, void *), void *arg, int32 (*inq)(int32 swathID, char *fieldlist, int32 rank[], int32 numbertype[]))
{
    char fieldlist[102400]; //Should be enough, review later
    int32 rank[1280];	//Should be enough, review later
    int32 numbertype[1280];	// should be enough, review later
    int32 r = -1;
    //struct func_arg new_arg = {func, arg};

    fieldlist[sizeof(fieldlist)-1]='\0';
    r = inq(id, fieldlist, rank, numbertype);
    assert(fieldlist[sizeof(fieldlist)-1]=='\0');

    if(r==-1)
        return -1;
    else
        return util_walk_str(fieldlist, ",", func, arg);
}

int _eos2util_walk_grsw_dims(int32 id, int (*func)(char *, void *), void *arg,
    int32 (*inq)(int32 swathID, char *dimlist, int32 dims[]))
{
    char dimlist[10240]; // Should be enough, review later
    int32 dims[128];	// Should be enough, review later
    int32 r = -1;

    // Notice that "XDim and YDim" are not included in the dimlist for HDF-EOS grid. 
    // So dimlist may be NULL and dims may be 0. This is normal. KY 2013-02-12
    dimlist[0]='\0';
    dimlist[sizeof(dimlist)-1]='\0';

    r = inq(id, dimlist, dims);
    assert(dimlist[sizeof(dimlist)-1]=='\0');

    if(r==-1)
        return -1;
    else {
        char delim =',';
        if(dimlist[0]!='\0')
           r = eos2util_check_null_dimnames(dimlist,delim);
        if (-1 == r) 
            return -1;
        return util_walk_str(dimlist, ",", func, arg);
    }
}

int eos2util_check_null_dimnames(char *str, char delim) {

    if (NULL == str) { 
        return -1;
    }
    if (delim == str[0] || delim == str[strlen(str)-1])
        return -1;
    else {
        char cont_delim[3];
        cont_delim[0] = delim;
        cont_delim[1] = delim;
        // Pay attention to this, remove this comment later.
        cont_delim[2] ='\0';
        if (strstr(str,cont_delim)!= NULL) 
            return -1;
    }
    return 0;

}

int _eos2util_walk_grsw_attrs(int32 id, int (*func)(char *, void *), void *arg,
    int32 (*inq)(int32 swathID, char *attrlist, int32 *bufsize))
{
    std::vector<char> buf;
    int32 bufsize=0;
    int32 r = -1;

    r = inq(id, NULL, &bufsize);
    if(r==-1)
    {
        assert(0);
        return -1;
    }

    bufsize++;
    buf.resize(bufsize);
    r = inq(id, &buf[0], &bufsize);
    if(r==-1)
    {
        assert(0);
        return -1;
    }

    return util_walk_str(&buf[0], ",", func, arg);
}

int eos2util_walk_grid_datafields(int32 id, int (*func)(char *, void *), void *arg)
{
    return _eos2util_walk_grsw_datafields(id, func, arg, GDinqfields);
}

int eos2util_walk_swath_datafields(int32 id, int (*func)(char *, void *), void *arg)
{
    return _eos2util_walk_grsw_datafields(id, func, arg, SWinqdatafields);
}

int eos2util_walk_swath_geofields(int32 id, int (*func)(char *, void *), void *arg)
{
    return _eos2util_walk_grsw_datafields(id, func, arg, SWinqgeofields);
}

int eos2util_walk_grid_dims(int32 id, int (*func)(char *, void *), void *arg)
{
    return _eos2util_walk_grsw_dims(id, func, arg, GDinqdims);
}

int eos2util_walk_swath_dims(int32 id, int (*func)(char *, void *), void *arg)
{
    return _eos2util_walk_grsw_dims(id, func, arg, SWinqdims);
}

int eos2util_walk_grid_attrs(int32 id, int (*func)(char *, void *), void *arg)
{
    return _eos2util_walk_grsw_attrs(id, func, arg, GDinqattrs);
}

int eos2util_walk_swath_attrs(int32 id, int (*func)(char *, void *), void *arg)
{
    return _eos2util_walk_grsw_attrs(id, func, arg, SWinqattrs);
}


