/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************
Category: Header file for the EOS2 module under the eoslib namespace

Description:

This file includes utility method declarations for HDF-EOS2 files.


*****************************************************************************/


#ifndef MISC_EOS2_H 
#define MISC_EOS2_H 

#include <HdfEosDef.h>

/**
 eos2util_walk_grids visits all grids in a HDF-EOS2 file.

 @param filename Filename
 @param func Callback function to be called at each grid
 @param arg An argument which is provided to the callback function.
 @return If all calls to the callback functions return 0, this function returns 0.
   Otherwise, it stops iteration immediately after a callback function returns
   non-zero value and return it.
*/
int eos2util_walk_grids(const char *filename, int (*func)(char *, void *), void *arg);

/*
 eos2util_walk_grid_datafields visits all data fields in a grid of a HDF-EOS2 file.

 @param gridid Grid ID
 @param func Callback function to be called at each data field
 @param arg An argument which is provided to the callback function.
 @return If all calls to the callback functions return 0, this function returns 0.
   Otherwise, it stops iteration immediately after a callback function returns
   non-zero value and return it.
*/
int eos2util_walk_grid_datafields(int32 gridid, int (*func)(char *, void *), void *arg);

/**
 eos2util_walk_swaths visits all swaths in a HDF-EOS2 file.

 @param filename Filename
 @param func Callback function to be called at each swath
 @param arg An argument which is provided to the callback function.
 @return If all calls to the callback functions return 0, this function returns 0.
   Otherwise, it stops iteration immediately after a callback function returns
   non-zero value and return it.
*/
int eos2util_walk_swaths(const char *filename, int (*func)(char *, void *), void *arg);

/*
 eos2util_walk_swath_datafields visits all data fields in a swath of a HDF-EOS2 file.

 @param swathid Swath ID
 @param func Callback function to be called at each data field
 @param arg An argument which is provided to the callback function.
 @return If all calls to the callback functions return 0, this function returns 0.
   Otherwise, it stops iteration immediately after a callback function returns
   non-zero value and return it.
*/
int eos2util_walk_swath_datafields(int32 swathid, int (*func)(char *, void *), void *arg);

/*
 eos2util_walk_swath_geofields visits all geolocation fields in a swath of a HDF-EOS2 file.

 @param swathid Swath ID
 @param func Callback function to be called at each data field
 @param arg An argument which is provided to the callback function.
 @return If all calls to the callback functions return 0, this function returns 0.
   Otherwise, it stops iteration immediately after a callback function returns
   non-zero value and return it.
*/
int eos2util_walk_swath_geofields(int32 swathid, int (*func)(char *, void *), void *arg);


int eos2util_walk_grid_dims(int32 swathid, int (*func)(char *, void *), void *arg);
int eos2util_walk_swath_dims(int32 swathid, int (*func)(char *, void *), void *arg);
int eos2util_walk_grid_attrs(int32 swathid, int (*func)(char *, void *), void *arg);
int eos2util_walk_swath_attrs(int32 swathid, int (*func)(char *, void *), void *arg);

int eos2util_check_null_dimnames(char*,const char);

#endif
