/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes utility function declarations for splitting a string.

*****************************************************************************/

#ifndef MISC_UTIL_H
#define MISC_UTIL_H

#include <string>
#include <vector>

typedef int (*walk_callback)(char *, void *);

#define UTIL_DEF_ARG_PACK(var, size) void *(var)[size]
#define UTIL_SET_ARG_PACK(var, index, ptr) (var)[index] = (void*)(ptr);
#define UTIL_GET_ARG_PACK(var, index, type) (type)(((void**)(var))[index])

/**
util_walk_str splits a string into many sub-strings and iterates them.

Note that util_walk_str modifies the original string. If the user needs a function
which does not modify the original string, one need to use strdup first.

@param str		str is a long string to be splitted. 
				Note that str is modified after calling util_walk_str
				since NULL characters are inserted.
@param delim	Each character in delim string is a delimiter.
@param func		Pointer to the function to be called for each substring
@param arg		Function argument
@return			If the callback function returns 0 for all sub-strings, util_walk_str
				returns 0. If one callback returns non-zero, util_walk_str stops
				iteration and returns the value.
*/
int util_walk_str(char *str, const char *delim, int (*func)(char*, void*), void *arg);

/**
util_split_str splits a string into many sub-substrings and return them.

Note that (*pentries) should be freed by the caller if it is not-NULL.

@param str		str is a long string to be splitted. 
				Note that str is modified after calling util_walk_str
				since NULL characters are inserted.
@param delim	Each character in delim string is a delimiter.
@param pentries	(OUT) util_split_str allocates the buffer for string
				the pointers to the sub-strings and store the pointer
				to the array of sub-string pointers in (*pentries).
@return			This function returns the number of sub-strings.
				If failed, it returns -1.
*/
int util_split_str(char *str, const char *delim, char ***pentries);

int util_split_str(const std::string& str, 
	const std::string& delim, 
	std::vector<std::string> *ptokens);
#endif
