/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of the H4CF conversion toolkit. The full H4CF conversion*
 * toolkit copyright notice including terms governing use, modification, and *
 * redistribution, is contained in the file COPYING.     *
 * COPYING can be found at the root of the source code    *
 * distribution tree.                                                        *
 * For questions contact eoshelp@hdfgroup.org or help@hdfgroup.org.          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*****************************************************************************

Description:

This file includes utility methods of implementation for HDF4 files.


*****************************************************************************/

#include <iostream>
#include <cstring>
#include "eoslib_var.h"
#include "eos2_var_gd_ll.h"
#include "mem_attr.h"
#include "hdf4_file.h"
#include "special_flt_cf.h"

//#define GET_SCALAR_VALUE(TYPE, CAST) CAST tmpvalue = *(CAST*)&fvbuf[0];
    

namespace eoslib {

//extern int mtype;
extern MODISType mtype;
//extern int modis_sg_type;

// For MODIS (confirmed by level 1B) products, values between 65500(MIN_NON_SCALE_SPECIAL_VALUE)  
// and 65535(MAX_NON_SCALE_SPECIAL_VALUE) are treated as
// special values. These values represent non-physical data values caused by various failures.
// For example, 65533 represents "when Detector is saturated".
bool is_special_value(int32 dtype, float fillvalue, float realvalue) {

    bool ret_value = false;

    if (DFNT_UINT16 == dtype) {

        int fillvalue_int = (int)fillvalue;
        if (MAX_NON_SCALE_SPECIAL_VALUE == fillvalue_int) {
            int realvalue_int = (int)realvalue;
            if (realvalue_int <= MAX_NON_SCALE_SPECIAL_VALUE && realvalue_int >=MIN_NON_SCALE_SPECIAL_VALUE)
                ret_value = true;
        }
    }

    return ret_value;

}

// Change data type from int to float for MODIS products.
value_type_t get_corrected_var_type(value_type_t vtype, var *v)
{
    // Only needs to handle non-float-point and Non-CF-scale-offset MODIS data. 
    // returning DFNT_FLOAT64 is okay.Since in change_modis_value, data will be recalculated with the DFNT_FLOAT64 type. 
    if((MOD_DEFAULT_TYPE == mtype) || (DFNT_FLOAT32 == vtype) || (DFNT_FLOAT64==vtype))
        return vtype;
       
    attr *attr1 = v->get_attr_by_name("scale_factor_modis");
    attr *attr2 = NULL; 
    attr *attr3 = NULL;

    if(attr1==NULL) // || attr2==NULL) [05/02/2012 LD]
    {	
        std::string vname = v->get_name();

        //For variable names containing "Emissive" or none of "RefSB" or "Emissive" like "EV_Band26" in MODIS\MOD021KM.A2000055.0000.005.2010041143816.hdf [05/08/2012 LD].
        attr2 = v->get_attr_by_name("radiance_scales");
        attr3 = v->get_attr_by_name("radiance_offsets");

        if(vname.find("RefSB")!=std::string::npos)
        {
            attr2 = v->get_attr_by_name("reflectance_scales");
            attr3 = v->get_attr_by_name("reflectance_offsets");
        } 
    } // end of if(attr1==NULL)

    // Change data type.
    if(attr1!=NULL || (attr2!=NULL && attr3!=NULL))
    {

        return DFNT_FLOAT32;
    }
    else 
        return vtype;
}

// Recalculate data values for MODIS products based on "scale_factor" and "add_offset" attributes. If these two attributes are not available, data values will not be touched.
/*
 * /params v                           variable
 * /params buf                         buffer to store the value
 * /params nr_bytes                    number of bytes
 * /params start                       the starting indices of the subset
 * /params stride                      the stride of the subset along each dimension
 * /params edge                        the number of elements (for each dim)
 *
 * Note change_modis_value is only used with get_value() in h4cf.cpp, 
 * the start, stride and edge in this function is only used to obtain 
 * scale and offset indices. They are unused since this value is used with get_value(start,stride,edge,value) function
 * at the h4cf.cpp. the subset part of the values is handled in the get_value function already.
 */
void change_modis_value(var *v, std::vector<char> *buf, int32 nr_bytes, int32 *start, int32 *stride, int32 *edge)
{ 
    if(MOD_DEFAULT_TYPE == mtype)
        return;

    // MOD09GA/GHK products have two grids. MODIS_Grid_1km_2D needs to apply the EQ rule;
    // So if we find that the origname variable starts with /MODIS_Grid_1km_2D, we will change the rule to the EQU rule.
    attr *attr_origname = v->get_attr_by_name("origname");
    if(attr_origname != NULL) {
        std::string attr_origname_value;
        attr_origname->get_value_str(&attr_origname_value);
        if (0 == attr_origname_value.find("/MODIS_Grid_1km_2D") && DIV_TYPE == mtype) 
            mtype = EQU_TYPE;
    }
 
    attr *attr1 = v->get_attr_by_name("scale_factor_modis");
    attr *attr2 = v->get_attr_by_name("add_offset_modis"); 
    attr *attr3 = v->get_attr_by_name("_FillValue");

    //Special case is for handling MOD021KM.A2000055.0000.005.2010041143816.hdf or MYD021KM.A2002226.0000.005.2009193222735.hdf only [05/09/2012 LD].
    attr *attr4=NULL, *attr5=NULL;
    bool special_case = false;
    float radiance_scales = 0;
    float radiance_offsets = 0;
    if(attr1==NULL)
    {
        attr4 = v->get_attr_by_name("radiance_scales");
        attr5 = v->get_attr_by_name("radiance_offsets");
        if(attr4!=NULL && attr5!=NULL)
        {
            int nr_values = attr4->get_num_elements();
            if(nr_values==1) 
            {
                attr4->get_value(&radiance_scales);
                attr5->get_value(&radiance_offsets);
                special_case = true;
            }
        } // end of if(attr4!=NULL
    } // end of if(attr1==NULL

    double scale_factor = 0;
    double add_offset=0;

    if(attr1!=NULL || special_case) // && attr2!=NULL) [05/02/2012 LD] 
    {
        if(special_case)
        {
            scale_factor = radiance_scales;
            add_offset = radiance_offsets;
        } else { 
            attr1->get_value(&scale_factor);
            if(attr2!=NULL)
                attr2->get_value(&add_offset);
        }

        value_type_t type = v->get_type();

        float32 fvalue = 0; 
        if(attr3!=NULL) 
            attr3->get_value(&fvalue); 
        //std::vector<float32> corrected_buf;
        float32 *corrected_ptr = NULL;
        int32 nr_elems = 0; 

        switch(type)	
        {
            case DFNT_UCHAR8:
            {
                nr_elems = nr_bytes/sizeof(uchar8); 
                uchar8 *ptr = (uchar8*)&((*buf)[0]); 
                if(type!=get_corrected_var_type(type, v))
                { 
                    //corrected_buf.resize(nr_elems*sizeof(float32)); 
                    //corrected_ptr = (float32*)&(corrected_buf[0]);
                    corrected_ptr = (float32*) malloc(nr_elems*sizeof(float32)); 
                    for(int l=0; l<nr_elems; l++) 
                        corrected_ptr[l] = (float32)ptr[l]; 
                } 
                for(int k=0; k<nr_elems; k++) 
                {
                    if(attr3!=NULL) {
                        if(corrected_ptr!= NULL) {
                            if(fvalue == corrected_ptr[k] || true == is_special_value(type,fvalue,corrected_ptr[k]))
                                continue; 
                            
                        }
                    }
                        if(mtype==MULTI_TYPE) 
                        {
                            if(corrected_ptr!=NULL) 
                                corrected_ptr[k] = (float32)(scale_factor*(corrected_ptr[k]-add_offset)); 
                            else 
                                ptr[k]  = (uchar8)(scale_factor*(ptr[k]-add_offset)); 
                        } else if(mtype==DIV_TYPE) { 
                            if(corrected_ptr!=NULL) 
                                corrected_ptr[k] = (float32)((corrected_ptr[k]-add_offset)/scale_factor); 
                            else 
                                ptr[k] = (uchar8)((ptr[k]-add_offset)/scale_factor); 
                        } else { 
                            if(corrected_ptr!=NULL) 
                                corrected_ptr[k] = (float32)(scale_factor*corrected_ptr[k]+add_offset); 
                            else 
                                ptr[k] = (uchar8)(scale_factor*ptr[k]+add_offset); 
                        } 
                } // end of for  
            } // end of case 
                break;
            case DFNT_INT8:
            {
                nr_elems = nr_bytes/sizeof(int8); 
                int8 *ptr = (int8*)&((*buf)[0]); 
                if(type!=get_corrected_var_type(type, v))
                { 
                    //corrected_buf.resize(nr_elems*sizeof(float32)); 
                    //corrected_ptr = (float32*)&(corrected_buf[0]);
                    corrected_ptr = (float32*) malloc(nr_elems*sizeof(float32)); 
                    for(int l=0; l<nr_elems; l++) 
                        corrected_ptr[l] = (float32)ptr[l]; 
                } 
                for(int k=0; k<nr_elems; k++) 
                { 
                    //if(attr3!=NULL && corrected_ptr[k]==fvalue) 
                    //    continue; 
                    //    Don't handle special values or fill values.
                    if(attr3!=NULL) {
                        if(corrected_ptr!= NULL) {
                            if(fvalue == corrected_ptr[k] || true == is_special_value(type,fvalue,corrected_ptr[k]))
                                continue;
                            
                        }
                    }
                    if(mtype==MULTI_TYPE) 
                    { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*(corrected_ptr[k]-add_offset)); 
                        else 
                            ptr[k]  = (int8)(scale_factor*(ptr[k]-add_offset)); 
                    } else if(mtype==DIV_TYPE) { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)((corrected_ptr[k]-add_offset)/scale_factor);
                        else
                            ptr[k] = (int8)((ptr[k]-add_offset)/scale_factor);
                    } else { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*corrected_ptr[k]+add_offset);
                        else
                            ptr[k] = (int8)(scale_factor*ptr[k]+add_offset);
                    } 
                } 
            }
                break;
            case DFNT_INT16:
            {
                nr_elems = nr_bytes/sizeof(int16); 
                int16 *ptr = (int16*)&((*buf)[0]);
                if(type!=get_corrected_var_type(type, v))
                {
                    //corrected_buf.resize(nr_elems*sizeof(float32));
                    //corrected_ptr = (float32*)&(corrected_buf[0]);
                    corrected_ptr = (float32*) malloc(nr_elems*sizeof(float32));
                    for(int l=0; l<nr_elems; l++)
                        corrected_ptr[l] = (float32)ptr[l];
                } 
                for(int k=0; k<nr_elems; k++) 
                { 
                    //if(attr3!=NULL && corrected_ptr[k]==fvalue) 
                    //    continue; 
                    if(attr3!=NULL) {
                        if(corrected_ptr!= NULL) {
                            if(fvalue == corrected_ptr[k] || true == is_special_value(type,fvalue,corrected_ptr[k]))
                                continue;
                            
                        }
                    }

                    if(mtype==MULTI_TYPE) 
                    {
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*(corrected_ptr[k]-add_offset)); 
                        else 
                            ptr[k]  = (int16)(scale_factor*(ptr[k]-add_offset)); 
                    } else if(mtype==DIV_TYPE) { 
                        if(corrected_ptr!=NULL) 
                           corrected_ptr[k] = (float32)((corrected_ptr[k]-add_offset)/scale_factor); 
                        else 
                           ptr[k] = (int16)((ptr[k]-add_offset)/scale_factor); 
                    } else { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*corrected_ptr[k]+add_offset);
                        else
                            ptr[k] = (int16)(scale_factor*ptr[k]+add_offset); 
                    } 
                } 
            } // end of for
                break;
            case DFNT_INT32:
            {
                nr_elems = nr_bytes/sizeof(int32); 
                int32 *ptr = (int32*)&((*buf)[0]); 
                if(type!=get_corrected_var_type(type, v))
                { 
                    //corrected_buf.resize(nr_elems*sizeof(float32)); 
                    //corrected_ptr = (float32*)&(corrected_buf[0]);
                    corrected_ptr = (float32*) malloc(nr_elems*sizeof(float32)); 
                    for(int l=0; l<nr_elems; l++) 
                        corrected_ptr[l] = (float32)ptr[l]; 
                } 
                for(int k=0; k<nr_elems; k++) 
                { 
//                    if(attr3!=NULL && corrected_ptr[k]==fvalue) 
//                        continue; 
                    if(attr3!=NULL) {
                        if(corrected_ptr!= NULL) {
                            if(fvalue == corrected_ptr[k] || true == is_special_value(type,fvalue,corrected_ptr[k]))
                                continue;
                            
                        }
                    }
                    if(mtype==MULTI_TYPE) 
                    { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*(corrected_ptr[k]-add_offset)); 
                        else 
                            ptr[k]  = (int32)(scale_factor*(ptr[k]-add_offset)); 
                    } else if(mtype==DIV_TYPE) { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)((corrected_ptr[k]-add_offset)/scale_factor);
                        else
                            ptr[k] = (int32)((ptr[k]-add_offset)/scale_factor);

                    } else { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*corrected_ptr[k]+add_offset);
                        else
                            ptr[k] = (int32)(scale_factor*ptr[k]+add_offset);

                    } 
                }  // end of for
            }
                break;
            case DFNT_UINT8:
            {
                nr_elems = nr_bytes/sizeof(uint8); 
                uint8 *ptr = (uint8*)&((*buf)[0]); 
                if(type!=get_corrected_var_type(type, v))
                { 
                    //corrected_buf.resize(nr_elems*sizeof(float32)); 
                    //corrected_ptr = (float32*)&(corrected_buf[0]);
                    corrected_ptr = (float32*) malloc(nr_elems*sizeof(float32)); 
                    for(int l=0; l<nr_elems; l++) 
                        corrected_ptr[l] = (float32)ptr[l]; 
                } 
                for(int k=0; k<nr_elems; k++) 
                { 
                    //if(attr3!=NULL && corrected_ptr[k]==fvalue) 
                    //    continue; 
                    if(attr3!=NULL) {
                        if(corrected_ptr!= NULL) {
                            if(fvalue == corrected_ptr[k] || true == is_special_value(type,fvalue,corrected_ptr[k]))
                                continue;
                            
                        }
                    }

                    if(mtype==MULTI_TYPE) 
                    { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*(corrected_ptr[k]-add_offset)); 
                        else 
                            ptr[k]  = (uint8)(scale_factor*(ptr[k]-add_offset)); 
                    } else if(mtype==DIV_TYPE) { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)((corrected_ptr[k]-add_offset)/scale_factor);
                        else
                            ptr[k] = (uint8)((ptr[k]-add_offset)/scale_factor);

                    } else { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*corrected_ptr[k]+add_offset);
                        else
                            ptr[k] = (uint8)(scale_factor*ptr[k]+add_offset);

                    } 
                } // end of for 
            } // end of case
                break;
            case DFNT_UINT16:
            {
                nr_elems = nr_bytes/sizeof(uint16); 
                uint16 *ptr = (uint16*)&((*buf)[0]); 
                if(type!=get_corrected_var_type(type, v))
                { 
                    //corrected_buf.resize(nr_elems*sizeof(float32)); 
                    //corrected_ptr = (float32*)&(corrected_buf[0]); 
                    corrected_ptr = (float32*) malloc(nr_elems*sizeof(float32));
                    for(int l=0; l<nr_elems; l++) 
                        corrected_ptr[l] = (float32)ptr[l]; 
                } 
                for(int k=0; k<nr_elems; k++) 
                { 
//                    if(attr3!=NULL && corrected_ptr[k]==fvalue) 
//                        continue; 
                    if(attr3!=NULL) {
                        if(corrected_ptr!= NULL) {
                            if(fvalue == corrected_ptr[k] || true == is_special_value(type,fvalue,corrected_ptr[k]))
                                continue;
                            
                        }
                    }
                    if(mtype==MULTI_TYPE) 
                    { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*(corrected_ptr[k]-add_offset)); 
                        else 
                            ptr[k]  = (uint16)(scale_factor*(ptr[k]-add_offset)); 
                    } else if(mtype==DIV_TYPE) { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)((corrected_ptr[k]-add_offset)/scale_factor); 
                        else 
                            ptr[k] = (uint16)((ptr[k]-add_offset)/scale_factor); 
                    } else { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*corrected_ptr[k]+add_offset);
                        else
                            ptr[k] = (uint16)(scale_factor*ptr[k]+add_offset);

                    } 
                } // end of for
            } // end of case
                break;
            case DFNT_UINT32:
            {
                nr_elems = nr_bytes/sizeof(uint32); 
                uint32 *ptr = (uint32*)&((*buf)[0]); 
                if(type!=get_corrected_var_type(type, v))
                { 
                    //corrected_buf.resize(nr_elems*sizeof(float32)); 
                    //corrected_ptr = (float32*)&(corrected_buf[0]);
                    corrected_ptr = (float32*) malloc(nr_elems*sizeof(float32)); 
                    for(int l=0; l<nr_elems; l++) 
                        corrected_ptr[l] = (float32)ptr[l]; 
                } 
                for(int k=0; k<nr_elems; k++) 
                { 
                    //if(attr3!=NULL && corrected_ptr[k]==fvalue) 
                    //    continue; 
                    if(attr3!=NULL) {
                        if(corrected_ptr!= NULL) {
                            if(fvalue == corrected_ptr[k] || true == is_special_value(type,fvalue,corrected_ptr[k]))
                                continue;
                        }
                       
                    }

                    if(mtype==MULTI_TYPE) 
                    { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*(corrected_ptr[k]-add_offset)); 
                        else 
                            ptr[k]  = (uint32)(scale_factor*(ptr[k]-add_offset)); 
                    } else if(mtype==DIV_TYPE) { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)((corrected_ptr[k]-add_offset)/scale_factor);
                        else
                            ptr[k] = (uint32)((ptr[k]-add_offset)/scale_factor);

                    } else { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*corrected_ptr[k]+add_offset);
                        else
                            ptr[k] = (uint32)(scale_factor*ptr[k]+add_offset);

                    } 
                } // end of for 
            }
                break;
            case DFNT_FLOAT32:
            {
                nr_elems = nr_bytes/sizeof(float32); 
                float32 *ptr = (float32*)&((*buf)[0]); 
                if(type!=get_corrected_var_type(type, v))
                { 
                    //corrected_buf.resize(nr_elems*sizeof(float32)); 
                    //corrected_ptr = (float32*)&(corrected_buf[0]);
                    corrected_ptr = (float32*) malloc(nr_elems*sizeof(float32)); 
                    for(int l=0; l<nr_elems; l++) 
                        corrected_ptr[l] = (float32)ptr[l]; 
                } 
                for(int k=0; k<nr_elems; k++) 
                { 
                    //if(attr3!=NULL && corrected_ptr[k]==fvalue) 
                    //    continue; 
                    if(attr3!=NULL) {
                        if(corrected_ptr!= NULL) {
                            if(fvalue == corrected_ptr[k] || true == is_special_value(type,fvalue,corrected_ptr[k]))
                                continue;
                        }

                    }

                    if(mtype==MULTI_TYPE) 
                    { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*(corrected_ptr[k]-add_offset)); 
                        else 
                            ptr[k]  = (float32)(scale_factor*(ptr[k]-add_offset)); 
                    } else if(mtype==DIV_TYPE) { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)((corrected_ptr[k]-add_offset)/scale_factor);
                        else
                            ptr[k] = (float32)((ptr[k]-add_offset)/scale_factor);

                    } else { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*corrected_ptr[k]+add_offset);
                        else
                            ptr[k] = (float32)(scale_factor*ptr[k]+add_offset);

                    } 
                } // end of for 
            } // end of case
                break;
            case DFNT_FLOAT64:
            {
                nr_elems = nr_bytes/sizeof(float64); 
                float64 *ptr = (float64*)&((*buf)[0]); 
                if(type!=get_corrected_var_type(type, v))
                { 
                    //corrected_buf.resize(nr_elems*sizeof(float32)); 
                    //corrected_ptr = (float32*)&(corrected_buf[0]);
                    corrected_ptr = (float32*) malloc(nr_elems*sizeof(float32)); 
                    for(int l=0; l<nr_elems; l++) 
                        corrected_ptr[l] = (float32)ptr[l]; 
                } 
                for(int k=0; k<nr_elems; k++) 
                { 
                    //if(attr3!=NULL && corrected_ptr[k]==fvalue) 
                    //    continue; 
                    if(attr3!=NULL) {
                        if(corrected_ptr!= NULL) {
                            if(fvalue == corrected_ptr[k] || true == is_special_value(type,fvalue,corrected_ptr[k]))
                                continue;
                        }

                    }

                    if(mtype==MULTI_TYPE) 
                    { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*(corrected_ptr[k]-add_offset)); 
                        else 
                            ptr[k]  = (float64)(scale_factor*(ptr[k]-add_offset)); 
                    } else if(mtype==DIV_TYPE) { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)((corrected_ptr[k]-add_offset)/scale_factor);
                        else
                            ptr[k] = (float64)((ptr[k]-add_offset)/scale_factor);

                    } else { 
                        if(corrected_ptr!=NULL) 
                            corrected_ptr[k] = (float32)(scale_factor*corrected_ptr[k]+add_offset);
                        else
                            ptr[k] = (float64)(scale_factor*ptr[k]+add_offset);

                    } 
                } // end of for 
            }
                break;
        } // end of switch
        if(corrected_ptr!=NULL) //corrected_buf.size()!=0) 
        { 
            buf->resize(nr_elems*sizeof(float32)); //corrected_buf.size()); 
            memcpy((void*)&((*buf)[0]), (void*)corrected_ptr, buf->size());//memcpy((void*)&((*buf)[0]), (void*)&(corrected_buf[0]), corrected_buf.size()); 
            //delete[] corrected_ptr;
            free(corrected_ptr);
        }
    } else { // end of if(attr1!=NULL ||
        std::string vname = v->get_name();
        if(vname.find("RefSB")!=std::string::npos)
        {
            attr1 = v->get_attr_by_name("reflectance_scales");
            attr2 = v->get_attr_by_name("reflectance_offsets");
        } else if(vname.find("Emissive")!=std::string::npos)
        {
            attr1 = v->get_attr_by_name("radiance_scales");
            attr2 = v->get_attr_by_name("radiance_offsets");
        }
        attr3 = v->get_attr_by_name("_FillValue");		
        if(attr1!=NULL && attr2!=NULL)
        {
            value_type_t type = attr1->get_type();
            unsigned int nr_values = attr1->get_num_elements();
		
            const std::list<dim*>& dims = v->get_dims_c();
            size_t nr_elems=1;
            size_t step = 0;
            size_t start_index = 0;
            size_t end_index = 0;
            size_t dimindex=0;
            std::list<dim*>::const_iterator it = dims.begin();

            if(nr_values!=(*it)->get_size()) // The number of Z-dimension scale attribute must be equal to the size of first dimension for MOD021KM.A2000055.0000.005.2010041143816.hdf. If not the case, do nothing.
                return;

            it++;
            for(; it != dims.end(); it++)
            {
                if(start==NULL)
                    nr_elems *= (*it)->get_size();
                else 
                       nr_elems *= edge[std::distance(dims.begin(), it)];
            } // end of for

            step = (start==NULL)? 1: stride[dimindex];
			
            start_index = (start==NULL)? 0: start[dimindex];
            end_index = (start==NULL)? nr_values-1: start_index+stride[dimindex]*(edge[dimindex]-1);
	
            value_type_t vtype = v->get_type(); 
	
            float32 fvalue = 0; 
            if(attr3!=NULL) 
                attr3->get_value(&fvalue); 
            //std::vector<float32> corrected_buf; 
            float32 *corrected_ptr = NULL; 
            int32 tot_elems = 0;			

            switch(type)
            {
                case DFNT_FLOAT32:
                {
                    std::vector<float32> scales(nr_values), offsets(nr_values); 
                    attr1->get_value(&scales[0]); 
                    attr2->get_value(&offsets[0]); 
                    if(vtype==DFNT_UINT16)
                    {
                        uint16 *ptr = (uint16*)&((*buf)[0]);
                        tot_elems = nr_bytes/sizeof(uint16); 
                        if(vtype!=get_corrected_var_type(vtype, v)) 
                        {  
                            //corrected_buf.resize(tot_elems*sizeof(float32)); 
                            //corrected_ptr = (float32*)&(corrected_buf[0]);
                            corrected_ptr = (float32*) malloc(tot_elems*sizeof(float32)); 
                            for(int l=0; l<tot_elems; l++) 
                                corrected_ptr[l] = (float32)ptr[l]; 
                        } // end of if 
                        for(size_t k=start_index; k<=end_index; k+=step) 
                        { 
                            for(size_t l=0; l<nr_elems; l++) 
                            { 
                                int index = l+k*nr_elems; 
                                if(attr3!=NULL) { 
                                    if(corrected_ptr!= NULL) {
                                        if(fvalue == corrected_ptr[index] || true == is_special_value(vtype,fvalue,corrected_ptr[index]))
                                            continue; 
                                    }
                                }
                                if(corrected_ptr!=NULL) 
                                    corrected_ptr[index]  = (float32)(scales[k]*(corrected_ptr[index]-offsets[k])); 
                                else 
                                    ptr[index]  = (uint16)(scales[k]*(ptr[index]-offsets[k])); 
                            } // end of for(size_t 
                        } // end of for 
                    } // end of if(vtype==DFNT_UINT16) 
                } // end of case 
                    break;
                case DFNT_FLOAT64:
                {
                    std::vector<float64> scales(nr_values), offsets(nr_values); 
                    attr1->get_value(&scales[0]); 
                    attr2->get_value(&offsets[0]); 
                    if(vtype==DFNT_UINT16)
                    {
                        uint16 *ptr = (uint16*)&((*buf)[0]); 
                        tot_elems = nr_bytes/sizeof(uint16); 
                        if(vtype!=get_corrected_var_type(vtype, v)) 
                        { 
                            //corrected_buf.resize(tot_elems*sizeof(float32)); 
                            //corrected_ptr = (float32*)&(corrected_buf[0]);
                            corrected_ptr = (float32*) malloc(tot_elems*sizeof(float32)); 
                            for(int l=0; l<tot_elems; l++) 
                                corrected_ptr[l] = (float32)ptr[l]; 
                        }  // end of if
                        for(size_t k=start_index; k<=end_index; k+=step) 
                        { 
                            for(size_t l=0; l<nr_elems; l++) 
                            { 
                                int index = l+k*nr_elems; 
                                if(attr3!=NULL) { 
                                    if(corrected_ptr!= NULL) {
                                        if(fvalue == corrected_ptr[index] || true == is_special_value(vtype,fvalue,corrected_ptr[index]))
                                            continue; 
                                    }
                                }
 
                                if(corrected_ptr!=NULL) 
                                    corrected_ptr[index]  = (float32)(scales[k]*(corrected_ptr[index]-offsets[k])); 
                                else 
                                    ptr[index]  = (uint16)(scales[k]*(ptr[index]-offsets[k])); 
                            } // end of for 
                        } // end of for(size_t 
                    } // end of if(vtype==DFNT_UINT16)
                } // end of case
                    break;
            } //end of switch
            if(corrected_ptr!=NULL)
            {
                buf->resize(tot_elems*sizeof(float32));//corrected_buf.size());
                memcpy((void*)&((*buf)[0]), (void*)corrected_ptr, buf->size());//memcpy((void*)&((*buf)[0]), (void*)&(corrected_buf[0]), corrected_buf.size());

                //delete[] corrected_ptr;
                free(corrected_ptr);
            }
        } // end of if(attr1!=NULL && attr2!=NULL) 
    } // end of else
}

// Find first value in latitude or longitude. This case is for handling an AIRS grid file. 
// findfirstfv and CorLatLon are implemented for both float and double datatypes.
// This function assumes that the data values are fillvalues from one point to the 
// end of the data array contiguously.
// It returns the first index of the fill value.
int findfirstfv(float *array, int start, int end, int fillvalue) 
{
    if (start == end || start == (end - 1)) 
    { 
        if (static_cast < int >(array[start]) == fillvalue) 
            return start; 
        else 
            return end; 
    } 
    else 
    { 
        int current = (start + end) / 2; 
        if (static_cast < int >(array[current]) == fillvalue) 
            return findfirstfv (array, start, current, fillvalue); 
        else 
            return findfirstfv (array, current, end, fillvalue); 
    } 
}

// Same as findfirstfv(float*...) except for the double array data type.
int findfirstfv (double * array, int start, int end, int fillvalue) 
{ 
    if (start == end || start == (end - 1)) 
    { 
        if (static_cast < int >(array[start]) == fillvalue) 
            return start; 
        else 
            return end; 
    } 
    else 
    { 
        int current = (start + end) / 2; 
        if (static_cast < int >(array[current]) == fillvalue) 
            return findfirstfv (array, start, current, fillvalue); 
        else 
            return findfirstfv (array, current, end, fillvalue); 
    } 
}

// Adjust latitude longitude values. This only applies to the case
// when fill values occur as the pattern in find_first_fv and the
// latitude and longitude values are in in the arithmetic sequence.
// This function also checks the valid range of lat and lon.
int CorLatLon(float *latlon, int fieldtype, int elms, int fv) 
{ 
    if (elms < 3) 
    { 
        for (int i = 0; i < elms; i++) 
            if ((int) (latlon[i]) == fv) 
                return -1; 
        return 0; 
    } 
    for (int i = 0; i < 3; i++) 
        if ((int) (latlon[i]) == fv) 
            return -1; 
    if ((int) (latlon[elms - 1]) != fv) 
        return 0; 
    float increment = latlon[2] - latlon[1]; 
    int index = 0; 
    index = findfirstfv (latlon, 0, elms - 1, fv); 
    if (index < 2)  
        throw std::range_error("Cannot calculate the fill value.(" __FILE__ ":" TOSTRING(__LINE__)")" ); 
    for (int i = index; i < elms; i++) 
    { 
        latlon[i] = latlon[i - 1] + increment; 
        if (i != (elms - 1) && (fieldtype == 1) && 
            ((float) (latlon[i]) < -90.0 || (float) (latlon[i]) > 90.0)) 
            return -1; 
        if (i != (elms - 1) && (fieldtype == 2) && 
            ((float) (latlon[i]) < -180.0 || (float) (latlon[i]) > 360.0)) 
            return -1; 
    } // end of for
    if (fieldtype == 1) 
    { 
        if ((float) (latlon[elms - 1]) < -90.0) 
            latlon[elms - 1] = -90; 
        if ((float) (latlon[elms - 1]) > 90.0) 
            latlon[elms - 1] = 90; 
    } 
    if (fieldtype == 2) 
    { 
        if ((float) (latlon[elms - 1]) < -180.0) 
            latlon[elms - 1] = -180; 
        if ((float) (latlon[elms - 1]) > 360.0) 
            latlon[elms - 1] = 360; 
    }
    return 0; 
}

// The same as CorLatLon(float*) except that the data type of lat/lon is double.
int CorLatLon(double *latlon,  int fieldtype, int elms, int fv) 
{ 
    if (elms < 3) 
    { 
        for (int i = 0; i < elms; i++) 
            if ((int) (latlon[i]) == fv) 
                return -1; 
        return 0; 
    } 
    for (int i = 0; i < 3; i++) 
        if ((int) (latlon[i]) == fv) 
            return -1; 
    if ((int) (latlon[elms - 1]) != fv) 
        return 0; 
    double increment = latlon[2] - latlon[1]; 
    int index = 0; 
    index = findfirstfv (latlon, 0, elms - 1, fv); 
    if (index < 2)  
        throw std::range_error("Cannot calculate the fill value.(" __FILE__ ":" TOSTRING(__LINE__)")" ); 
    for (int i = index; i < elms; i++) 
    { 
        latlon[i] = latlon[i - 1] + increment; 
        if (i != (elms - 1) && (fieldtype == 1) && 
            ((float) (latlon[i]) < -90.0 || (float) (latlon[i]) > 90.0)) 
            return -1; 
        if (i != (elms - 1) && (fieldtype == 2) && 
            ((float) (latlon[i]) < -180.0 || (float) (latlon[i]) > 360.0)) 
            return -1; 
    } // end of for
    if (fieldtype == 1) 
    { 
        if ((float) (latlon[elms - 1]) < -90.0) 
            latlon[elms - 1] = -90; 
        if ((float) (latlon[elms - 1]) > 90.0) 
            latlon[elms - 1] = 90; 
    } 
    if (fieldtype == 2) 
    { 
        if ((float) (latlon[elms - 1]) < -180.0) 
            latlon[elms - 1] = -180; 
        if ((float) (latlon[elms - 1]) > 360.0) 
            latlon[elms - 1] = 360; 
    }
    return 0; 
}

// Calculate latitude or longitude values for HDFEOS2 special cases that don't have projection codes(MOD13C2).
void calculate_special_lat_lon(int fieldtype, int32 xdim, int32 ydim, float64 *outlatlon, int32 *offset32, int32 *count32, int32 *step32)
{
    int k = 0;
    if(fieldtype==eos2_var_gd_ll::LATITUDE)
    {
        double latstep = 180.0 / ydim;
        for (int i = 0; i < (int) (count32[0]); i++)
            for(int j = 0; j< (int) (count32[1]); j++)
                outlatlon[k++] = 90.0 - latstep * (offset32[0] + i * step32[0]);
    } else {
        double lonstep = 360.0 / xdim;
        for(int j = 0; j < (int) (count32[0]); j++) 
            for (int i = 0; i < (int) (count32[1]); i++)
                outlatlon[k++] = -180.0 + lonstep * (offset32[1] + i * step32[1]);
    }
    return;
}

// Current this function is not used. It is not throughly tested.
// Just leave it here. Perhaps it is useful for the internal tests.(KY 2014-12-23)
// 
void dump_attr(attr* attr, std::ostream &os) throw(std::range_error)
{
    value_type_t type = attr->get_type();
    unsigned int tot_elems = attr->get_num_elements();

    switch(type)
    {
        case DFNT_UCHAR8:
        case DFNT_CHAR8:
        {
            std::string buf;
            attr->get_value_str(&buf);
            os << "\"" << buf << "\"";
        }
            break;
        case DFNT_INT8:
        {
            std::vector<int8> buf(tot_elems); 
            attr->get_value(&buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            { 
               	if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t\t";    
                os << (int)(buf[i]); 
                if(i < buf.size() - 1) 
                    os << ",";      
            } 
        }
            break;
        case DFNT_INT16:
        {
            std::vector<int16> buf(tot_elems); 
            attr->get_value(&buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            { 
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t\t";    
                os << (int16)(buf[i]); 
                if(i < buf.size() - 1) 
                    os << ",";     
            } 
        }
            break;
        case DFNT_INT32:
        {
            std::vector<int32> buf(tot_elems); 
            attr->get_value(&buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            { 
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t\t";    
                os << (int32)(buf[i]); 
                if(i < buf.size() - 1) 
                    os << ",";      
           } 
        }
            break;
        case DFNT_UINT8:
        {
            std::vector<uint8> buf(tot_elems); 
            attr->get_value(&buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            { 
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t\t";    
                os << (unsigned int)(buf[i]); 
                if(i < buf.size() - 1) 
                    os << ",";      
            } 
        }
            break;
        case DFNT_UINT16:
        {
            std::vector<uint16> buf(tot_elems); 
            attr->get_value(&buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            { 
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t\t";    
                os << (uint16)(buf[i]); 
                if(i < buf.size() - 1) 
                    os << ",";      
            } 
        }
            break;	
        case DFNT_UINT32:
        {
            std::vector<uint32> buf(tot_elems); 
            attr->get_value(&buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            { 
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t\t";    
                os << (uint32)(buf[i]); 
                if(i < buf.size() - 1) 
                    os << ",";      
            } 
        }
            break;
        case DFNT_FLOAT32:
        {
            std::vector<float32> buf(tot_elems); 
            attr->get_value(&buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            { 
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t\t";    
                os << (float32)(buf[i]); 
                if(i < buf.size() - 1) 
                    os << ",";      
            } 
        }
            break;
        case DFNT_FLOAT64:
        {
            std::vector<float64> buf(tot_elems); 
            attr->get_value(&buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            { 
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t\t";    
                os << (float64)(buf[i]); 
                if(i < buf.size() - 1) 
                    os << ",";      
            } 
        }
            break;
        default:
            throw std::range_error(std::string("") + "Attribute " + attr->get_name() + " has unknown type (" __FILE__ ":" TOSTRING(__LINE__)")" );
    } // end of switch 
}

// Current this function is not used. It is not throughly tested.
// Just leave it here. Perhaps it is useful for the internal tests.(KY 2014-12-23)
// 
void dump_var(eoslib::var *v, std::ostream &os)
{
    const std::list<dim*>& dims = v->get_dims_c();
    size_t tot_elems= 1;

    std::vector<int32> start;
    std::vector<int32> stride;
    std::vector<int32> edge;

    for(std::list<dim*>::const_iterator it = dims.begin();
        it != dims.end(); it++)
    {
        start.push_back(0);
        stride.push_back(1); 
        edge.push_back((*it)->get_size());
    }

    for(std::vector<int32>::const_iterator it = edge.begin(); it!=edge.end(); it++)
        tot_elems *= (*it);

    value_type_t type = v->get_type();
    if(dims.size()==0) //For newly added VDATA only. VDATA has no dimensions.
    {
        eos2_var_data *vdata = (eos2_var_data*)v;
        tot_elems = vdata->get_bytes()/get_type_size(type);
    }

    switch(type)
    {
        case DFNT_CHAR8: //For fields in VDATA  whose array size is >1
        {
            std::vector<char8> buf(tot_elems); 
            v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); 
            std::string str; 
            int32 order = 1; 
            const char* vname = typeid(*v).name(); 
            std::string s(vname); 
            if(s.find("eos2_var_data")!=std::string::npos) 
                order = (static_cast<eos2_var_data*>(v))->get_order(); 
            for(size_t i=0; i<=buf.size()-order; i+=order) 
            { 
                if((i != 0) && (i % 6==0) && (i!= buf.size()-order)) 
                    os << std::endl << "\t\t"; 
                for(size_t j=i; j<i+order; j++) 
                    if((char8)buf[j]>31 && (char8)buf[j]<127) 
                        str+=(char8)buf[j]; 
                os << "\"" << str << "\""; 
                if(i!=buf.size()-order) 
                    os << ","; 
                str.clear(); 
            } // end of for  	
        } // end of case
            break;
        case DFNT_INT8:
        {
            std::vector<int8> buf(tot_elems); 
            v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            { 
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t";      
                os << (int)(buf[i]); 
                if(i+1 != buf.size()) 
                    os << ", "; 
            } 
        }
            break;
        case DFNT_INT16:
        {
            std::vector<int16> buf(tot_elems); 
            v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            {
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t";      
                os << (int16)(buf[i]); 
                if(i+1 != buf.size()) 
                    os << ", "; 
            } 
        }
            break;
        case DFNT_INT32:
        {
            std::vector<int32> buf(tot_elems); 
            v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            {       
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t";      
                os << (int32)(buf[i]); 
                if(i+1 != buf.size()) 
                    os << ", "; 
            } 
        }
            break;
        case DFNT_UCHAR8:
        {
            std::vector<uchar8> buf(tot_elems); 
            v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            {
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t";      
                os << (int)(buf[i]); 
                if(i+1 != buf.size()) 
                    os << ", "; 
            } 
        }
            break;
        case DFNT_UINT8:
        {
            std::vector<uint8> buf(tot_elems); 
            v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            {
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t";      
                os << (unsigned int)(buf[i]); 
                if(i+1 != buf.size()) 
                    os << ", "; 
            } 
        }
            break;
        case DFNT_UINT16:
        {
            std::vector<uint16> buf(tot_elems); 
            v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            {
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t";      
                os << (uint16)(buf[i]); 
                if(i+1 != buf.size()) 
                    os << ", "; 
            } 
        }
            break;
        case DFNT_UINT32:
        {
            std::vector<uint32> buf(tot_elems); 
            v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            {
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t";      
                os << (uint32)(buf[i]); 
                if(i+1 != buf.size()) 
                    os << ", "; 
            } 
        }
            break;
        case DFNT_FLOAT32:
        {
            std::vector<float32> buf(tot_elems); 
            v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            {       
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t";      
                os << (float32)(buf[i]); 
                if(i+1 != buf.size()) 
                    os << ", "; 
            } 
        }
            break;
        case DFNT_FLOAT64:
        {
            std::vector<float64> buf(tot_elems); 
            v->get_value(&start[0], &stride[0], &edge[0], &buf[0]); 
            for(size_t i = 0; i<buf.size(); i++) 
            { 
                if((i != 0) && (i % 6==0) && (i+1 != buf.size())) 
                    os << std::endl << "\t\t";      
                os << (float64)(buf[i]); 
                if(i+1 != buf.size()) 
                    os << ", "; 
            } 
        }
            break;
        default:
            throw std::range_error("Unknown vairable type at(" __FILE__ ":" TOSTRING(__LINE__)")" );
    } // end of switch
}


// Check if the fillvalue datatype is the same as the variable datatype. Yes return true,no return false.
bool check_eos2_fill_value_attr_type(var*v,value_type_t new_var_type) {

    bool fv_type_not_same = false;
    std::list<attr*>vattrs = v->get_attrs_c();

    for (std::list<attr*>::const_iterator it= vattrs.begin(); it != vattrs.end();++it) {
        if((*it)->get_name() =="_FillValue") {
            if ((*it)->get_type() != new_var_type)
                fv_type_not_same = true;
            break;
        }
    }
    return fv_type_not_same;

}

// Change the fillvalue datatype if the variable datatype is different than the _FillValue attribute datatype.
void change_eos2_fill_value_attr_type(var*v,value_type_t vtype) {
     
    attr *fvattr = v->get_attr_by_name("_FillValue");

    if(fvattr == NULL)
        throw std::range_error("Fillvalue must exist for this routine. (" __FILE__ ":" TOSTRING(__LINE__)")" );
    if(fvattr->get_num_elements() != 1) 
        throw std::range_error("Currently we only support one element fillvalue. (" __FILE__ ":" TOSTRING(__LINE__)")" );

    value_type_t fvtype = fvattr->get_type();
    std::vector<char> fvbuf;
    fvattr->get_value(&fvbuf);

    // std::cerr<<"fill value original : "<<*(int16*)&fvbuf[0] <<std::endl;
    v->remove_attr(fvattr);
    mem_attr *nfvattr = new mem_attr(v,"_FillValue");

    switch(vtype) {

        case DFNT_UCHAR8:
        case DFNT_UINT8:
        {
            uint8 f1 = 0;
            if (DFNT_INT8 == fvtype) 
                f1 = (uint8)(*(int8*)&fvbuf[0]);   
            else if (DFNT_INT16 == fvtype)
                f1 = (uint8)(*(int16*)&fvbuf[0]);    
            else if (DFNT_UINT16 == fvtype)
                f1 = (uint8)(*(uint16*)&fvbuf[0]);    
            else if (DFNT_INT32 == fvtype)
                f1 = (uint8)(*(int32*)&fvbuf[0]);    
            else if (DFNT_UINT32 == fvtype)
                f1 = (uint8)(*(uint32*)&fvbuf[0]);    
            else if (DFNT_FLOAT32 == fvtype)
                f1 = (uint8)(*(float32*)&fvbuf[0]);    
            else if (DFNT_FLOAT64 == fvtype)
                f1 = (uint8)(*(float64*)&fvbuf[0]);    
            nfvattr->set_value(DFNT_UINT8,1,(void*)&f1);
        }
        break;

        case DFNT_INT8:
        {
            int8 f1 = 0;
            if (DFNT_UINT8 == fvtype) 
                f1 = (int8)(*(uint8*)&fvbuf[0]);   
            else if (DFNT_INT16 == fvtype)
                f1 = (int8)(*(int16*)&fvbuf[0]);    
            else if (DFNT_UINT16 == fvtype)
                f1 = (int8)(*(uint16*)&fvbuf[0]);    
            else if (DFNT_INT32 == fvtype)
                f1 = (int8)(*(int32*)&fvbuf[0]);    
            else if (DFNT_UINT32 == fvtype)
                f1 = (int8)(*(uint32*)&fvbuf[0]);    
            else if (DFNT_FLOAT32 == fvtype)
                f1 = (int8)(*(float32*)&fvbuf[0]);    
            else if (DFNT_FLOAT64 == fvtype)
                f1 = (int8)(*(float64*)&fvbuf[0]);    
            nfvattr->set_value(DFNT_INT8,1,(void*)&f1);
        }
        break;

        case DFNT_UINT16:
        {
            uint16 f1 = 0;
            if (DFNT_UINT8 == fvtype) 
                f1 = (uint16)(*(uint8*)&fvbuf[0]);   
            else if (DFNT_INT8 == fvtype) 
                f1 = (uint16)(*(int8*)&fvbuf[0]);   
            else if (DFNT_INT16 == fvtype)
                f1 = (uint16)(*(int16*)&fvbuf[0]);    
            else if (DFNT_INT32 == fvtype)
                f1 = (uint16)(*(int32*)&fvbuf[0]);    
            else if (DFNT_UINT32 == fvtype)
                f1 = (uint16)(*(uint32*)&fvbuf[0]);    
            else if (DFNT_FLOAT32 == fvtype)
                f1 = (uint16)(*(float32*)&fvbuf[0]);    
            else if (DFNT_FLOAT64 == fvtype)
                f1 = (uint16)(*(float64*)&fvbuf[0]);    
            nfvattr->set_value(DFNT_UINT16,1,(void*)&f1);
        }
        break;

        case DFNT_INT16:
        {
            int16 f1 = 0;
            if (DFNT_UINT8 == fvtype) 
                f1 = (int16)(*(uint8*)&fvbuf[0]);   
            else if (DFNT_INT8 == fvtype) 
                f1 = (int16)(*(int8*)&fvbuf[0]);   
            else if (DFNT_UINT16 == fvtype)
                f1 = (int16)(*(uint16*)&fvbuf[0]);    
            else if (DFNT_INT32 == fvtype)
                f1 = (int16)(*(int32*)&fvbuf[0]);    
            else if (DFNT_UINT32 == fvtype)
                f1 = (int16)(*(uint32*)&fvbuf[0]);    
            else if (DFNT_FLOAT32 == fvtype)
                f1 = (int16)(*(float32*)&fvbuf[0]);    
            else if (DFNT_FLOAT64 == fvtype)
                f1 = (int16)(*(float64*)&fvbuf[0]);    
            nfvattr->set_value(DFNT_INT16,1,(void*)&f1);
        }
        break;


        case DFNT_UINT32:
        {
            uint32 f1 = 0;
            if (DFNT_UINT8 == fvtype) 
                f1 = (uint32)(*(uint8*)&fvbuf[0]);   
            else if (DFNT_INT8 == fvtype) 
                f1 = (uint32)(*(int8*)&fvbuf[0]);   
            else if (DFNT_UINT16 == fvtype)
                f1 = (uint32)(*(uint16*)&fvbuf[0]);
            else if (DFNT_INT16 == fvtype)
                f1 = (uint32)(*(int16*)&fvbuf[0]);    
            else if (DFNT_INT32 == fvtype)
                f1 = (uint32)(*(int32*)&fvbuf[0]);    
            else if (DFNT_FLOAT32 == fvtype)
                f1 = (uint32)(*(float32*)&fvbuf[0]);    
            else if (DFNT_FLOAT64 == fvtype)
                f1 = (uint32)(*(float64*)&fvbuf[0]);    
            nfvattr->set_value(DFNT_UINT32,1,(void*)&f1);
        }
        break;

        case DFNT_INT32:
        {
            int32 f1 = 0;
            if (DFNT_UINT8 == fvtype) 
                f1 = (int32)(*(uint8*)&fvbuf[0]);   
            else if (DFNT_INT8 == fvtype) 
                f1 = (int32)(*(int8*)&fvbuf[0]);   
            else if (DFNT_UINT16 == fvtype)
                f1 = (int32)(*(uint16*)&fvbuf[0]);
            else if (DFNT_INT16 == fvtype)
                f1 = (int32)(*(int16*)&fvbuf[0]);    
            else if (DFNT_UINT32 == fvtype)
                f1 = (int32)(*(uint32*)&fvbuf[0]);    
            else if (DFNT_FLOAT32 == fvtype)
                f1 = (int32)(*(float32*)&fvbuf[0]);    
            else if (DFNT_FLOAT64 == fvtype)
                f1 = (int32)(*(float64*)&fvbuf[0]);    
            nfvattr->set_value(DFNT_INT32,1,(void*)&f1);
        }
        break;

        case DFNT_FLOAT32:
        {
            float32 f1 = 0;
            if (DFNT_UINT8 == fvtype) 
                f1 = (float32)(*(uint8*)&fvbuf[0]);   
            else if (DFNT_INT8 == fvtype) 
                f1 = (float32)(*(int8*)&fvbuf[0]);   
            else if (DFNT_UINT16 == fvtype)
                f1 = (float32)(*(uint16*)&fvbuf[0]);
            else if (DFNT_INT16 == fvtype)
                f1 = (float32)(*(int16*)&fvbuf[0]);    
            else if (DFNT_UINT32 == fvtype)
                f1 = (float32)(*(uint32*)&fvbuf[0]);    
            else if (DFNT_INT32 == fvtype)
                f1 = (float32)(*(int32*)&fvbuf[0]);    
            else if (DFNT_FLOAT64 == fvtype)
                f1 = (float32)(*(float64*)&fvbuf[0]);    
            nfvattr->set_value(DFNT_FLOAT32,1,(void*)&f1);
        }
        break;

        case DFNT_FLOAT64:
        {
            float64 f1 = 0;
            if (DFNT_UINT8 == fvtype) 
                f1 = (float64)(*(uint8*)&fvbuf[0]);   
            else if (DFNT_INT8 == fvtype) 
                f1 = (float64)(*(int8*)&fvbuf[0]);   
            else if (DFNT_UINT16 == fvtype)
                f1 = (float64)(*(uint16*)&fvbuf[0]);
            else if (DFNT_INT16 == fvtype)
                f1 = (float64)(*(int16*)&fvbuf[0]);    
            else if (DFNT_UINT32 == fvtype)
                f1 = (float64)(*(uint32*)&fvbuf[0]);    
            else if (DFNT_INT32 == fvtype)
                f1 = (float64)(*(int32*)&fvbuf[0]);    
            else if (DFNT_FLOAT32 == fvtype)
                f1 = (float64)(*(float32*)&fvbuf[0]);    
            nfvattr->set_value(DFNT_FLOAT64,1,(void*)&f1);
        }
        break;
        default:
            delete nfvattr;
            throw std::range_error("Unsupported type(" __FILE__ ":" TOSTRING(__LINE__)")" );
    }

    v->add_attr(nfvattr);
#if 0
    switch(vtype) {
#define GET_FILLVALUE_ATTR_VALUE(TYPE, CAST1,CAST2) \
    case DFNT_##TYPE: \
    { \
        CAST tmpvalue = *(CAST*)&fvbuf[0]; \
std::cerr<<"fill value: "<<tmpvalue <<std::endl; \
        nfvattr->set_value(DFNT_##TYPE,1,(void*)&tmpvalue);\
    } \
    break;
                        // No need to have CHAR8 and INT8 since we are using macros.
                        GET_FILLVALUE_ATTR_VALUE(INT8, int8);
                        GET_FILLVALUE_ATTR_VALUE(UINT8, uint8);
                        GET_FILLVALUE_ATTR_VALUE(INT16, int16);
                        GET_FILLVALUE_ATTR_VALUE(UINT16, uint16);
                        GET_FILLVALUE_ATTR_VALUE(INT32, int32);
                        GET_FILLVALUE_ATTR_VALUE(UINT32, uint32);
                        GET_FILLVALUE_ATTR_VALUE(FLOAT32, float32);
                        GET_FILLVALUE_ATTR_VALUE(FLOAT64, float64);

                    };
#undef GET_FILLVALUE_ATTR_VALUE
#endif
            

}

} //end of namespace

namespace hdf4 {

//The following code is not used.
#if 0
bool check_hdf4_fill_value_attr_type(hdf4_var_sdfield* v4){

    return v4->has_attr("_FillValue");

}
#endif

// Change HDF4 fill value type.
void change_hdf4_fill_value_attr_type(hdf4_var_sdfield* v4){

    bool change_fill_value_type = false;
    value_type_t fvtype;
    std::vector<char> fvbuf;
    hdf4_attr *nfvattr;
    std::vector<hdf4_attr *> sds_attrs = v4->get_attributes();
    std::vector<hdf4_attr *>::iterator it;
    //hdf4_attr *oriattr = NULL;
    //std::vector<hdf4_attr *>::iterator toeraseit;

    for (it = sds_attrs.begin();it!=sds_attrs.end();++it) {
        if((*it)->get_name() == "_FillValue") {
            if(v4->get_type() != (*it)->get_type()) {
                if((*it)->get_num_elements() != 1)
                    throw std::range_error("Currently we only support one element fillvalue. (" __FILE__ ":" TOSTRING(__LINE__)")" );
                (*it)->get_value(&fvbuf);
                fvtype = (*it)->get_type();
                   //toeraseit = it;
                   //oriattr = *it;
                  // delete(*it);
                   //sds_attrs.erase(it);
                change_fill_value_type = true;
                break;
            }
        }
    }

    
    if (true == change_fill_value_type) {

        v4->remove_attr_by_name("_FillValue");
        switch(v4->get_type()) {
            case DFNT_UCHAR8:
            case DFNT_UINT8:
            {
                uint8 f1 = 0;
                if (DFNT_INT8 == fvtype) 
                    f1 = (uint8)(*(int8*)&fvbuf[0]);   
                else if (DFNT_INT16 == fvtype)
                    f1 = (uint8)(*(int16*)&fvbuf[0]);    
                else if (DFNT_UINT16 == fvtype)
                    f1 = (uint8)(*(uint16*)&fvbuf[0]);    
                else if (DFNT_INT32 == fvtype)
                    f1 = (uint8)(*(int32*)&fvbuf[0]);    
                else if (DFNT_UINT32 == fvtype)
                    f1 = (uint8)(*(uint32*)&fvbuf[0]);    
                else if (DFNT_FLOAT32 == fvtype)
                    f1 = (uint8)(*(float32*)&fvbuf[0]);    
                else if (DFNT_FLOAT64 == fvtype)
                    f1 = (uint8)(*(float64*)&fvbuf[0]);    
                std::vector<char>nfvbuf(sizeof(uint8));
                memcpy((char*)&(nfvbuf)[0],&f1,sizeof(uint8));
                nfvattr = new hdf4_attr("_FillValue",DFNT_UINT8,1,nfvbuf);
            }
            break;

            case DFNT_INT8:
            {
                int8 f1 = 0;
                if (DFNT_UINT8 == fvtype) 
                    f1 = (int8)(*(uint8*)&fvbuf[0]);   
                else if (DFNT_INT16 == fvtype)
                    f1 = (int8)(*(int16*)&fvbuf[0]);    
                else if (DFNT_UINT16 == fvtype)
                    f1 = (int8)(*(uint16*)&fvbuf[0]);    
                else if (DFNT_INT32 == fvtype)
                    f1 = (int8)(*(int32*)&fvbuf[0]);    
                else if (DFNT_UINT32 == fvtype)
                    f1 = (int8)(*(uint32*)&fvbuf[0]);    
                else if (DFNT_FLOAT32 == fvtype)
                    f1 = (int8)(*(float32*)&fvbuf[0]);    
                else if (DFNT_FLOAT64 == fvtype)
                    f1 = (int8)(*(float64*)&fvbuf[0]);    

                std::vector<char>nfvbuf(sizeof(int8));
                memcpy((char*)&(nfvbuf)[0],&f1,sizeof(int8));
                nfvattr = new hdf4_attr("_FillValue",DFNT_INT8,1,nfvbuf);

            }
            break;

            case DFNT_UINT16:
            {
                uint16 f1 = 0;
                if (DFNT_UINT8 == fvtype) 
                    f1 = (uint16)(*(uint8*)&fvbuf[0]);   
                else if (DFNT_INT8 == fvtype) 
                    f1 = (uint16)(*(int8*)&fvbuf[0]);   
                else if (DFNT_INT16 == fvtype)
                    f1 = (uint16)(*(int16*)&fvbuf[0]);    
                else if (DFNT_INT32 == fvtype)
                    f1 = (uint16)(*(int32*)&fvbuf[0]);    
                else if (DFNT_UINT32 == fvtype)
                    f1 = (uint16)(*(uint32*)&fvbuf[0]);    
                else if (DFNT_FLOAT32 == fvtype)
                    f1 = (uint16)(*(float32*)&fvbuf[0]);    
                else if (DFNT_FLOAT64 == fvtype)
                    f1 = (uint16)(*(float64*)&fvbuf[0]);    

                std::vector<char>nfvbuf(sizeof(uint16));
                memcpy((char*)&(nfvbuf)[0],&f1,sizeof(uint16));
                nfvattr = new hdf4_attr("_FillValue",DFNT_UINT16,1,nfvbuf);

            }
            break;

            case DFNT_INT16:
            {
                int16 f1 = 0;
                if (DFNT_UINT8 == fvtype) 
                    f1 = (int16)(*(uint8*)&fvbuf[0]);   
                else if (DFNT_INT8 == fvtype) 
                    f1 = (int16)(*(int8*)&fvbuf[0]);   
                else if (DFNT_UINT16 == fvtype)
                    f1 = (int16)(*(uint16*)&fvbuf[0]);    
                else if (DFNT_INT32 == fvtype)
                    f1 = (int16)(*(int32*)&fvbuf[0]);    
                else if (DFNT_UINT32 == fvtype)
                    f1 = (int16)(*(uint32*)&fvbuf[0]);    
                else if (DFNT_FLOAT32 == fvtype)
                    f1 = (int16)(*(float32*)&fvbuf[0]);    
                else if (DFNT_FLOAT64 == fvtype)
                    f1 = (int16)(*(float64*)&fvbuf[0]);    
                std::vector<char>nfvbuf(sizeof(int16));
                memcpy((char*)&(nfvbuf)[0],&f1,sizeof(int16));
                nfvattr = new hdf4_attr("_FillValue",DFNT_INT16,1,nfvbuf);


            }
            break;


            case DFNT_UINT32:
            {
                uint32 f1 = 0;
                if (DFNT_UINT8 == fvtype) 
                    f1 = (uint32)(*(uint8*)&fvbuf[0]);   
                else if (DFNT_INT8 == fvtype) 
                    f1 = (uint32)(*(int8*)&fvbuf[0]);   
                else if (DFNT_UINT16 == fvtype)
                    f1 = (uint32)(*(uint16*)&fvbuf[0]);
                else if (DFNT_INT16 == fvtype)
                    f1 = (uint32)(*(int16*)&fvbuf[0]);    
                else if (DFNT_INT32 == fvtype)
                    f1 = (uint32)(*(int32*)&fvbuf[0]);    
                else if (DFNT_FLOAT32 == fvtype)
                    f1 = (uint32)(*(float32*)&fvbuf[0]);    
                else if (DFNT_FLOAT64 == fvtype)
                    f1 = (uint32)(*(float64*)&fvbuf[0]);    
                std::vector<char>nfvbuf(sizeof(uint32));
                memcpy((char*)&(nfvbuf)[0],&f1,sizeof(uint32));
                nfvattr = new hdf4_attr("_FillValue",DFNT_UINT32,1,nfvbuf);


            }
            break;

            case DFNT_INT32:
            {
                int32 f1 = 0;
                if (DFNT_UINT8 == fvtype) 
                    f1 = (int32)(*(uint8*)&fvbuf[0]);   
                else if (DFNT_INT8 == fvtype) 
                    f1 = (int32)(*(int8*)&fvbuf[0]);   
                else if (DFNT_UINT16 == fvtype)
                    f1 = (int32)(*(uint16*)&fvbuf[0]);
                else if (DFNT_INT16 == fvtype)
                    f1 = (int32)(*(int16*)&fvbuf[0]);    
                else if (DFNT_UINT32 == fvtype)
                    f1 = (int32)(*(uint32*)&fvbuf[0]);    
                else if (DFNT_FLOAT32 == fvtype)
                    f1 = (int32)(*(float32*)&fvbuf[0]);    
                else if (DFNT_FLOAT64 == fvtype)
                    f1 = (int32)(*(float64*)&fvbuf[0]);    
                std::vector<char>nfvbuf(sizeof(int32));
                memcpy((char*)&(nfvbuf)[0],&f1,sizeof(int32));
                nfvattr = new hdf4_attr("_FillValue",DFNT_INT32,1,nfvbuf);


            }
            break;

            case DFNT_FLOAT32:
            {
                float32 f1 = 0;
                if (DFNT_UINT8 == fvtype) 
                    f1 = (float32)(*(uint8*)&fvbuf[0]);   
                else if (DFNT_INT8 == fvtype) 
                    f1 = (float32)(*(int8*)&fvbuf[0]);   
                else if (DFNT_UINT16 == fvtype)
                    f1 = (float32)(*(uint16*)&fvbuf[0]);
                else if (DFNT_INT16 == fvtype)
                    f1 = (float32)(*(int16*)&fvbuf[0]);    
                else if (DFNT_UINT32 == fvtype)
                    f1 = (float32)(*(uint32*)&fvbuf[0]);    
                else if (DFNT_INT32 == fvtype)
                    f1 = (float32)(*(int32*)&fvbuf[0]);    
                else if (DFNT_FLOAT64 == fvtype)
                    f1 = (float32)(*(float64*)&fvbuf[0]);    
                std::vector<char>nfvbuf(sizeof(float32));
                memcpy((char*)&(nfvbuf)[0],&f1,sizeof(float32));
                nfvattr = new hdf4_attr("_FillValue",DFNT_FLOAT32,1,nfvbuf);


            }
            break;

            case DFNT_FLOAT64:
            {
                float64 f1 = 0;
                if (DFNT_UINT8 == fvtype) 
                    f1 = (float64)(*(uint8*)&fvbuf[0]);   
                else if (DFNT_INT8 == fvtype) 
                    f1 = (float64)(*(int8*)&fvbuf[0]);   
                else if (DFNT_UINT16 == fvtype)
                    f1 = (float64)(*(uint16*)&fvbuf[0]);
                else if (DFNT_INT16 == fvtype)
                    f1 = (float64)(*(int16*)&fvbuf[0]);    
                else if (DFNT_UINT32 == fvtype)
                    f1 = (float64)(*(uint32*)&fvbuf[0]);    
                else if (DFNT_INT32 == fvtype)
                    f1 = (float64)(*(int32*)&fvbuf[0]);    
                else if (DFNT_FLOAT32 == fvtype)
                    f1 = (float64)(*(float32*)&fvbuf[0]);    
                std::vector<char>nfvbuf(sizeof(float64));
                memcpy((char*)&(nfvbuf)[0],&f1,sizeof(float64));
                nfvattr = new hdf4_attr("_FillValue",DFNT_FLOAT64,1,nfvbuf);


            }
            break;
            default:
                throw std::range_error("Unsupported type(" __FILE__ ":" TOSTRING(__LINE__)")" );
        }
        v4->add_attr(nfvattr);
    }

    return;
}

}
#if 0
#define SET_NEW_FILLVALUE_ATTR(TYPE, CAST) \
    case DFNT_##TYPE: \
    { \
        CAST tmpvalue = *(CAST*)&fvbuf[0]; \
        std::vector<char> fvbuf(sizeof(CAST));\
        memcpy((char*)&(fvbuf)[0],&tmpvalue,sizeof(CAST));\
        nfvattr = new hdf4_attr("_FillValue",DFNT_##TYPE,sizeof(CAST),fvbuf);\
    } \
    break;
 // No need to have CHAR8 and INT8 since we are using macros.
                        SET_NEW_FILLVALUE_ATTR(INT8, int8);
                        SET_NEW_FILLVALUE_ATTR(UINT8, uint8);
                        SET_NEW_FILLVALUE_ATTR(INT16, int16);
                        SET_NEW_FILLVALUE_ATTR(UINT16, uint16);
                        SET_NEW_FILLVALUE_ATTR(INT32, int32);
                        SET_NEW_FILLVALUE_ATTR(UINT32, uint32);
                        SET_NEW_FILLVALUE_ATTR(FLOAT32, float32);
                        SET_NEW_FILLVALUE_ATTR(FLOAT64, float64);

                    };
#undef SET_NEW_FILLVALUE_ATTR
#endif
           
#if 0
        v4->add_attr(nfvattr);
        }

    return;
}

}
#endif










