// Copyright by The HDF Group
// All rights reserved.                                                      
//                                                                           
// This file is part of the H4CF Conversion Toolkit. The full H4CF Conversion
// Toolkit copyright notice including terms governing use, modification, and 
// redistribution, is contained in the file COPYING.     
// COPYING can be found at the root of the source code    
// distribution tree. If you do not have access to this file, you may request
// a copy from eoshelp@hdfgroup.org.     
#include "h4cf.h"
#include <iostream>
#include <string>
#include <map>
#include <vector>
#include <list>

// This API test reads corner values from a variable and prints out the
// values.


void write_char8(vector<char> vals)
{
    // Print out the values.
    int n =  vals.size() / sizeof(char8);
    int i = 0;
    for (i=0; i < n ; i ++){
        cout << (char8)*(char8*)&(vals[i*sizeof(char8)]) << endl;
    }
}

void write_uchar8(vector<char> vals)
{
    // Print out the values.
    int n =  vals.size() / sizeof(uchar8);
    int i = 0;
    for (i=0; i < n ; i ++){
        cout << (uchar8)*(uchar8*)&(vals[i*sizeof(uchar8)]) << endl;
    }
}


void write_int8(vector<char> vals)
{
    // Print out the values.
    int n =  vals.size() / sizeof(int8);
    int i = 0;
    for (i=0; i < n ; i ++){
        // int8 is same as signed char for stream.
        // Thus, cast it as a number.
        cout 
            << static_cast<int16_t>((int8)*(int8*)&(vals[i*sizeof(int8)])) 
            << endl;
    }
}


void write_uint8(vector<char> vals)
{
    // Print out the values.
    int n =  vals.size() / sizeof(uint8);
    int i = 0;
    for (i=0; i < n ; i ++){
        // int8 is same as signed char for stream.
        // Thus, cast it as a number.
        cout 
            << static_cast<int16_t>((uint8)*(uint8*)&(vals[i*sizeof(uint8)]))
            << endl;
    }
}


void write_int16(vector<char> vals)
{
    // Print out the values.
    int n =  vals.size() / sizeof(int16);
    int i = 0;
    for (i=0; i < n ; i ++){
        cout << (int16)*(int16*)&(vals[i*sizeof(int16)]) << endl;
    }
}


void write_uint16(vector<char> vals)
{
    // Print out the values.
    int n =  vals.size() / sizeof(uint16);
    int i = 0;
    for (i=0; i < n ; i ++){
        cout << (uint16)*(uint16*)&(vals[i*sizeof(uint16)]) << endl;
    }
}

void write_int32(vector<char> vals)
{
    // Print out the values.
    int n =  vals.size() / sizeof(int32);
    int i = 0;
    for (i=0; i < n ; i ++){
        cout << (int32)*(int32*)&(vals[i*sizeof(int32)]) << endl;
    }
}


void write_uint32(vector<char> vals)
{
    // Print out the values.
    int n =  vals.size() / sizeof(uint32);
    int i = 0;
    for (i=0; i < n ; i ++){
        cout << (uint32)*(uint32*)&(vals[i*sizeof(uint32)]) << endl;
    }
}


void write_float32(vector<char> vals)
{
    // Print out the values.
    int n =  vals.size() / sizeof(float32);
    int i = 0;
    for (i=0; i < n ; i ++){
        cout << (float32)*(float32*)&(vals[i*sizeof(float32)]) << endl;
    }
}

void write_float64(vector<char> vals)
{
    // Print out the values.
    int n =  vals.size() / sizeof(float64);
    int i = 0;
    for (i=0; i < n ; i ++){
        cout << (float64)*(float64*)&(vals[i*sizeof(float64)]) << endl;
    }
}

void write_values(list<var*>::const_iterator it)
{
    vector<char> vals;

    const int rank = h4cf_get_var_rank((*it));

    if(0 == rank){
        cout << "The variable has a 0 rank." << endl;
        return;
    }
    else{
        cout << "Rank = " << rank << endl;
    }

    int32* start = new int32[rank];
    int32* stride = new int32[rank];
    int32* edge = new int32[rank];

    const vector< map<string, int> > vdims =  h4cf_get_var_dims((*it));        

    int i = 0;

    for(vector< map<string, int> >::const_iterator vdim_iter = vdims.begin();
        vdim_iter != vdims.end();
        vdim_iter++){
        int size = 0;
        map<string, int> dims = (*vdim_iter);
        for(map<string, int>::const_iterator dim_iter = dims.begin(); 
            dim_iter != dims.end(); 
            dim_iter++){
            cout << "Dimension name:" 
                 << (*dim_iter).first
                 << " size:" 
                 << (*dim_iter).second
                 << endl;
            size = (*dim_iter).second;

        } // for(each dim map);

        start[i] = 0;
        stride[i] = size-1;
        if (0 == stride[i])
            edge[i] = 1;           
        else 
            edge[i] = 2;
        i++;

    } // for(each dim vector);


    if (i != rank){
        cerr << "Rank and dimension size doesn't match." << endl;
    }


    h4cf_get_var_value(&vals, ((*it)), start, stride, edge);

    h4cf_data_type type = h4cf_get_var_type((*it));


    switch(type){

    case DFNT_CHAR8: 
        write_char8(vals);
        break;

    case DFNT_UCHAR8: 
        write_uchar8(vals);
        break;

    case DFNT_INT8: 
        write_int8(vals);
        break;

    case DFNT_UINT8: 
        write_uint8(vals);
        break;
   
    case DFNT_INT16:
        write_int16(vals);
        break;            

    case DFNT_UINT16:
        write_uint16(vals);
        break;             

    case DFNT_INT32:
        write_int32(vals);
        break;            

    case DFNT_UINT32:
        write_uint32(vals);
        break;            


    case DFNT_FLOAT32:
        write_float32(vals);
        break;

    case DFNT_FLOAT64:
        write_float64(vals);
        break;

    default:
        cerr << "Unknown type:type=" << type << endl;
        break;
    }
    delete[] start;
    delete[] stride;
    delete[] edge;
}

int main(int argc, char* argv[])
{
    if (argc > 3){
        cerr << "h4cf_get_var_value file_name [dataset_name] "<< endl;
        exit(1);
    }

    // Open file and initialize library.
    h4cf_open(argv[1]); 

    // Retrieve a list of variables.
    const list<var*> vars = h4cf_get_vars();

    // Count the number of variables.
    for(list<var*>::const_iterator it=vars.begin(); 
        it!=vars.end(); 
        it++) {

        if(argc == 3) {
            if(h4cf_get_var_name((*it)) == argv[2]) {
                write_values(it);
            }
        }
        else{
            cout << h4cf_get_var_name((*it)) << endl;
            write_values(it);
            
        }

    }
    // Close the file.
    h4cf_close();
    return 0;
}
