// Copyright by The HDF Group
// All rights reserved.                                                      
//                                                                           
// This file is part of the H4CF Conversion Toolkit. The full H4CF Conversion
// Toolkit copyright notice including terms governing use, modification, and 
// redistribution, is contained in the file COPYING.     
// COPYING can be found at the root of the source code    
// distribution tree. If you do not have access to this file, you may request
// a copy from help@hdfgroup.org.     

// Description:
//
// This file demonstrates how to use APIs provided by h4cflib to generate 
// netCDF files.
//
#include <list>

#include "netcdf.h"
#include "h4cf.h"

// Handle errors by printing an error message and exit with a non-zero status.
#define ERR(e) {printf("Error: %s\n", nc_strerror(e)); return 2;}

// Handle attributes.
#define HANDLE_ATTR_TYPE(htype, ctype, nctype, ncfunc) case DFNT_##htype: {if((retval = nc_put_att(ncid, varid, (*vattr_iter)->get_name().c_str(), nctype, (*vattr_iter)->get_num_elements(), (void*)&vattr_var[0]))) ERR(retval); break; }

// Handle variables.
#define HANDLE_VAR_TYPE(htype, ctype) case DFNT_##htype:  {if ((retval = nc_def_var(ncid, var_name.c_str(), ctype, size, vdim_var, &varid))) ERR(retval); break;}

// Define dimensions using netCDF API.
int ncconverter_define_dims(map<string, int>, int*, int);


// Define variable.
int ncconverter_define_var(var*, int, int*,  int&, int);

// Get dimension ID defined in method ncconverter_define_dims.
void ncconverter_get_var_dim_def(vector< map<string, int> >, map<string, int>, int*, int*);

// Write variable attributes.
int ncconverter_write_var_attrs(list<attr*>, int, int);


// Write variable values using netCDF API.
int ncconverter_write_vars(list<var*>, int*, int);

// Write file attributes using netCDF API.
int ncconverter_write_file_attrs(list<attr*>, int);


int main(int argc, char* argv[])
{
    if ( argc < 2) // At lease two input parameters.
	{
            cout << "Usage: h4tonccf [hdf file name] or h4tonccf [hdf file name] [netcdf file name] \nNote: If the netcdf file name is not specified, the hdf file name will be used instead with .nc as extension." << endl;
            return 1;
	} else {
        string hdf_file;
        string nc_file;
        char* hdf_file_name = NULL;
        char* nc_file_name = NULL;

        hdf_file = argv[1];

        if(argc > 2) // Three input parameters with netcdf file name.
            nc_file = argv[2];
        else { // Two input parameters without netcdf file name.
            int lastindex = hdf_file.find_last_of(".");
            string basename = hdf_file.substr(0, lastindex);
            nc_file = basename.append(".nc");
        }	

        hdf_file_name = (char*) hdf_file.c_str();
        nc_file_name = (char*) nc_file.c_str();
        h4cf_open(hdf_file_name);
	   
        int ncid = 0;
        int retval = 0;

#ifndef NC4
        if ((retval = nc_create(nc_file_name,  NC_CLOBBER | NC_64BIT_OFFSET, 
                                &ncid))) 
            ERR(retval);
#else
        if ((retval = nc_create(nc_file_name, NC_CLOBBER | NC_NETCDF4, 
                                &ncid))) 
            ERR(retval);
#endif
		
        int old_fill_mode;
        if ((retval = nc_set_fill(ncid, NC_NOFILL, &old_fill_mode))) 
            ERR(retval);
		
        // Get dimensions from library.
        map<string, int> dims = h4cf_get_dims();

        // Define dimensions read from library.
        int* dim_var = 0;
        dim_var = new int[dims.size()];
        ncconverter_define_dims(dims, dim_var, ncid);
	   
        // Get variables from the library.
        const list<var*> pvars = h4cf_get_vars();
        int* vars = 0;
        vars = new int[pvars.size()];
        list<var*>::const_iterator var_iter;
        int n=0;
        for(var_iter = pvars.begin(); var_iter != pvars.end(); var_iter++)
            {
                // Find the corresponding dimensions.
                const vector< map<string, int> > vdims = 
                    h4cf_get_var_dims(*var_iter);
                int* vdim_var = 0;
                vdim_var = new int[vdims.size()];
                ncconverter_get_var_dim_def(vdims, dims, dim_var, vdim_var);
                int &varid = vars[n];

                ncconverter_define_var((*var_iter), vdims.size(), vdim_var, 
                                       varid, ncid);
	 
#ifdef NC4
                // netCDF 4 only.	
                // nc_def_var_deflate() doesn't work on scalar variable.
                if(vdims.size() > 0){
                    if((retval = nc_def_var_deflate(ncid, vars[n], 1, 1, 2))) 
                        ERR(retval);
                }
#endif
                // Define attributes for variables.
                const list<attr*> vattrs = h4cf_get_var_attrs(*var_iter);
                if(!vattrs.empty())
                    ncconverter_write_var_attrs(vattrs, vars[n], ncid);		
                n++;
                delete [] vdim_var;
            }
        // Write file attributes.
        const list<attr*> file_attrs = h4cf_get_file_attrs();
        ncconverter_write_file_attrs(file_attrs, ncid);
        // End of definition mode.
        if ((retval = nc_enddef(ncid)))	ERR(retval);
        // Write variable values.
        ncconverter_write_vars(pvars, vars, ncid);
        // Close the file. 
        if ((retval = nc_close(ncid))) ERR(retval);
		
        delete [] dim_var;
        delete [] vars;

        h4cf_close();

        cout << "Done with writing netcdf file "<< nc_file_name << "." << endl;

        //        if(argc==2)
        //            delete [] nc_file_name;

        return 0;
    }
}

int ncconverter_define_dims(map<string, int> dims, int *dim_var, int ncid)
{
    map<string, int>::const_iterator dim_iter;

    int k=0, retval;

    for(dim_iter = dims.begin(); dim_iter != dims.end(); dim_iter++)
        {
            std::string dim_name = (*dim_iter).first;
            if ((retval = nc_def_dim(ncid, dim_name.c_str(), (*dim_iter).second, &dim_var[k++])))
                ERR(retval);
        }

    return 0;
}

void ncconverter_get_var_dim_def(vector< map<string, int> > vdims, map<string, int> dims, int *dim_var, int *vdim_var)
{
    int m=0;
    for(vector< map<string, int> >::iterator vdim_iter = vdims.begin(); vdim_iter != vdims.end(); vdim_iter++)
        {
            map<string, int> vdim = (*vdim_iter);
            map<string, int >::iterator vdimit = vdim.begin(); 
            string dimname = (*vdimit).first; 

            int j = 0;
            map<string, int>::const_iterator dim_iter;
            for(dim_iter = dims.begin(); dim_iter != dims.end(); dim_iter++)
                {
                    if(dimname == (*dim_iter).first)
                        {
                            vdim_var[m] = dim_var[j];
                            m++;
                            break;
                        }
                    j++;
                }
        }
}

int ncconverter_define_var(var *v, int size, int *vdim_var,  int& varid, int ncid)
{
    int retval;
    string var_name = h4cf_get_var_name(v);

    switch(h4cf_get_var_type(v))
	{
            HANDLE_VAR_TYPE(CHAR8,   NC_CHAR);
            HANDLE_VAR_TYPE(INT8,    NC_BYTE);
            HANDLE_VAR_TYPE(INT16,   NC_SHORT);
            HANDLE_VAR_TYPE(INT32,   NC_INT);

#ifdef NC4
            HANDLE_VAR_TYPE(UCHAR8,  NC_UBYTE);
            HANDLE_VAR_TYPE(UINT8,   NC_UBYTE);
            HANDLE_VAR_TYPE(UINT16,  NC_USHORT);
            HANDLE_VAR_TYPE(UINT32,  NC_UINT);
#else
            // UCHAR8, UINT8, and UINT16 are converted to NC_SHORT, NC_SHORT,
            // and NC_INT respectively as net-CDF3 does not provide support for 
            // these data types.
            HANDLE_VAR_TYPE(UCHAR8,  NC_SHORT); 
            HANDLE_VAR_TYPE(UINT8,   NC_SHORT); 
            HANDLE_VAR_TYPE(UINT16,  NC_INT);   
            HANDLE_VAR_TYPE(UINT32,  NC_INT);
#endif
            HANDLE_VAR_TYPE(FLOAT32, NC_FLOAT);
            HANDLE_VAR_TYPE(FLOAT64, NC_DOUBLE);
        default:
            throw std::range_error("Unknown type(" __FILE__ ":" TOSTRING(__LINE__)")" );
	}
    return 0;
}	

#ifdef NC4
int 
ncconverter_write_var_attrs_nc4(list<attr*> vattrs, int varid, int ncid)
{
    int retval = 0;
    vector<char>  vattr_var;
    list<attr*>::const_iterator vattr_iter;

    for(vattr_iter = vattrs.begin(); vattr_iter != vattrs.end(); vattr_iter++)
        {
            h4cf_get_attr_value(&vattr_var, (*vattr_iter));

            switch(h4cf_get_attr_type(*vattr_iter))
                {
                    HANDLE_ATTR_TYPE(CHAR8,   char, NC_CHAR, nc_put_att_text);
                    HANDLE_ATTR_TYPE(INT8,    signed char, NC_BYTE, nc_put_att_schar);
                    HANDLE_ATTR_TYPE(INT16,   short, NC_SHORT, nc_put_att_short);
                    HANDLE_ATTR_TYPE(INT32,   int, NC_INT, nc_put_att_int);
                    HANDLE_ATTR_TYPE(UCHAR8,  char, NC_CHAR, nc_put_att_text);
                    HANDLE_ATTR_TYPE(UINT8,   unsigned char, NC_UBYTE, nc_put_att_uchar);
                    HANDLE_ATTR_TYPE(UINT16,  unsigned short, NC_USHORT, nc_put_att_ushort);
                    HANDLE_ATTR_TYPE(UINT32,  unsigned int, NC_UINT, nc_put_att_uint);

                    HANDLE_ATTR_TYPE(FLOAT32, float, NC_FLOAT, nc_put_att_float);
                    HANDLE_ATTR_TYPE(FLOAT64, double, NC_DOUBLE, nc_put_att_double);
                default:
                    throw std::range_error("Unknown type(" __FILE__ ":" TOSTRING(__LINE__)")" );
        	}
	}
    return retval;
}
#else
int 
ncconverter_write_var_attrs_nc3(list<attr*> vattrs, int varid, int ncid)
{
    int retval;
    list<attr*>::const_iterator vattr_iter;

    for(vattr_iter = vattrs.begin(); vattr_iter != vattrs.end(); vattr_iter++)
        {
            vector<char>  vattr_var;
            h4cf_get_attr_value(&vattr_var, (*vattr_iter));
//			cerr<<"pass h4cf_get_attr_value "<<endl;	
//cerr<<"attr value vector size is "<<vattr_var.size() <<endl;
            switch(h4cf_get_attr_type(*vattr_iter))
                {
                    HANDLE_ATTR_TYPE(CHAR8, char, NC_CHAR, 
                                     nc_put_att_text);
                    HANDLE_ATTR_TYPE(INT8, signed char, NC_BYTE, 
                                     nc_put_att_schar);
                    HANDLE_ATTR_TYPE(INT16, short, NC_SHORT, 
                                     nc_put_att_short);
                    HANDLE_ATTR_TYPE(INT32,int, NC_INT, 
                                     nc_put_att_int);
                    HANDLE_ATTR_TYPE(FLOAT32, float, NC_FLOAT, 
                                     nc_put_att_float);
                    HANDLE_ATTR_TYPE(FLOAT64, double, NC_DOUBLE, 
                                     nc_put_att_double);
                    // Overflow may occur for this type. See documentation.
                    HANDLE_ATTR_TYPE(UINT32,  int, NC_INT,
                                     nc_put_att_int);

                    // Convert the unsigned types with twice bigger type.
                    case DFNT_UCHAR8: 
                        {
                            vector<int16> tmp;
                            tmp.resize(vattr_var.size()); 
                            for(unsigned int k=0; k<tmp.size(); k++) 
                                tmp[k] = static_cast<int16>
                                    ((uint8)*(uint8*)&(vattr_var[k*sizeof(uint8)]));

                            if((retval = 
                                nc_put_att(ncid, varid, 
                                           (*vattr_iter)->get_name().c_str(), 
                                           NC_SHORT, 
                                           (*vattr_iter)->get_num_elements(), 
                                           (void*)&tmp[0])))
                                ERR(retval);
                            break; 
                        }

                    case DFNT_UINT8:
                        {
                            vector<int16> tmp;
                            tmp.resize(vattr_var.size()); 
                            for(unsigned int k=0; k<tmp.size(); k++)
                                tmp[k] = static_cast<int16>
                                    ((uint8)*(uint8*)&(vattr_var[k*sizeof(uint8)]));

                            if((retval = 
                                nc_put_att(ncid, varid, 
                                           (*vattr_iter)->get_name().c_str(), 
                                           NC_SHORT, 
                                           (*vattr_iter)->get_num_elements(), 
                                           (void*)&tmp[0])))
                                ERR(retval);
                            break; 
                        }

                    case DFNT_UINT16: 
                        { 
                            vector<int32> tmp;
							uint16* vattr_var_ptr = (uint16 *)(&vattr_var[0]);
							if(vattr_var.size()/2 *2 != vattr_var.size())
								throw std::range_error("uint16 data stored in vector<char> should be even." "(" __FILE__ ":" TOSTRING(__LINE__)")" );
                            tmp.resize(vattr_var.size()/2);
//cerr<<"attr size is "<<tmp.size() <<endl;
                            for(unsigned int k=0; k<tmp.size(); k++) {
								tmp[k] = static_cast<int32>(*vattr_var_ptr);
								vattr_var_ptr++;
							}
                                //tmp[k] = static_cast<int>
                                  //  ((uint16)*(uint16*)&(vattr_var[k*sizeof(uint16)]));
//cerr<<" after allocating the value " <<endl;
                            if((retval = 
                                nc_put_att(ncid, varid, 
                                           (*vattr_iter)->get_name().c_str(), 
                                           NC_INT,
                                           (*vattr_iter)->get_num_elements(), 
                                           (void*)&tmp[0])))
                                ERR(retval);
                            break; 
                        }

                    
                default:
                    throw std::range_error("Unknown type" "(" __FILE__ ":" TOSTRING(__LINE__)")" );
        	} // switch()
	} // for()
    return 0;

}
#endif

int ncconverter_write_var_attrs(list<attr*> vattrs, int varid, int ncid)
{
    int retval = 0;
#ifdef NC4
    retval = ncconverter_write_var_attrs_nc4(vattrs, varid, ncid);
#else
    retval = ncconverter_write_var_attrs_nc3(vattrs, varid, ncid);
#endif
    return retval;
}

int ncconverter_write_file_attrs(const list<attr*> vattrs, int ncid)
{
    return ncconverter_write_var_attrs(vattrs, NC_GLOBAL, ncid);
}

int ncconverter_write_vars(list<var*> pvars, int *vars, int ncid)
{
    int n=0, retval;
    vector<char> var_val;
    for(list<var*>::const_iterator var_iter = pvars.begin(); 
        var_iter != pvars.end(); var_iter++)
        {

            h4cf_get_var_value(&var_val, (*var_iter));
            h4cf_data_type vtype = h4cf_get_var_type((*var_iter));
#if !defined(NC4)
            // As netCDF-3 has no unsigned char or unsigned int, we covert 
            // these types to int with double size. Therefore unsigned char8
            //  and unsigned int8 will become NC_SHORT, and unsigned int16
            // will become NC_INT.
            if(vtype==DFNT_UCHAR8 || 
               vtype==DFNT_UINT8  || 
               vtype==DFNT_UINT16)
		{
                    vector<char> tmp(var_val);
                    var_val.resize(var_val.size()*2); // Double size.
                    if(vtype==DFNT_UCHAR8) // UCHAR8
			{
                            uchar8 *ptr1 = (uchar8*)&(tmp[0]);
                            int16 *ptr2 = (int16*)&(var_val[0]);
                            for(unsigned int k=0; k<tmp.size(); k++)
                                ptr2[k] = ptr1[k];
			} else  if(vtype==DFNT_UINT8) {// UINT8
                        uint8 *ptr1 = (uint8*)&(tmp[0]);
                        int16 *ptr2 = (int16*)&(var_val[0]);
                        for(unsigned int k=0; k<tmp.size(); k++)
                            ptr2[k] = ptr1[k];
                    } else { // UINT16
                        uint16 *ptr1 = (uint16*)&(tmp[0]);
                        int32 *ptr2 = (int32*)&(var_val[0]);
                        for(unsigned int k=0; k<tmp.size()/2; k++)
                            ptr2[k] = ptr1[k];
                    }
		}
#endif
            if(var_val.empty())
		{
                    cerr << "h4cf_get_var_value() returned empty data." << endl;
		} 
            else 
                {
                    if((retval = nc_put_var(ncid, vars[n], (void*)&var_val[0])))
                        ERR(retval);
                }
            n++;
        } // for each variable.
    return 0;
}

