/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by  The HDF Group and                                           *
 *               The Board of Trustees of the University of Illinois.        *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of H4H5TOOLS. The full H4H5TOOLS copyright notice,      *
 * including terms governing use, modification, and redistribution, is       *
 * contained in the COPYING file, which can be found at the root of the      *
 * source code distribution tree. The full H4H5TOOLS copyright notice can    *
 * also be found at https://www.hdfgroup.org/licenses.  If you do not have   *
 * access to either file, you may request a copy from help@hdfgroup.org.     *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*-----------------------------------------------------------------------------
 * File:    H4TOH5Iprivate.h
 * Purpose: header file for ID API
 *---------------------------------------------------------------------------*/

/* avoid re-inclusion */
#ifndef _H4TOH5Iprivate_H
#define _H4TOH5Iprivate_H

#include "H4TOH5Ipublic.h" /*include Public Definitions	     */

/* Private headers needed by this file */

/* Default sizes of the hash-tables for various atom groups */
#define H4TOH5I_HASHSIZE             64
/*
 * Function for freeing objects. This function will be called with an object
 * ID group number (object type) and a pointer to the object. The function
 * should free the object and return non-negative to indicate that the object
 * can be removed from the ID group. If the function returns negative
 * (failure) then the object will remain in the ID group.
 */
typedef int (*H4TOH5I_free_t)(void*);

/* Type of the function to compare objects & keys */
typedef int (*H4TOH5I_search_func_t)(void *obj, const void *key);

/* Atom information structure used */
typedef struct H4TOH5I_id_info_t {
    hid_t	id;		/* ID for this info			    */
    h4toh5_uint	count;		/* ref. count for this atom		    */
    void	*obj_ptr;	/* pointer associated with the atom	    */
    struct H4TOH5I_id_info_t *next;	/* link to next atom (in case of hash-clash)*/
} H4TOH5I_id_info_t;

/* ID group structure used */
typedef struct {
    h4toh5_uint	count;		/*# of times this group has been initialized*/
    h4toh5_uint	reserved;	/*# of IDs to reserve for constant IDs	    */
    h4toh5_uint	wrapped;	/*whether the id count has wrapped around   */
    size_t	hash_size;	/*sizeof the hash table to store the IDs in */
    h4toh5_uint	ids;		/*current number of IDs held		    */
    h4toh5_uint	nextid;		/*ID to use for the next atom		    */
    H4TOH5I_free_t	free_func;	/*release object method	    		    */
    H4TOH5I_id_info_t **id_list;	/*pointer to an array of ptrs to IDs	    */
} H4TOH5I_id_group_t;

/* Private Functions in H4TOH5I.c */
/*__DLL__*/ int H4TOH5I_init_group(H4TOH5I_type_t grp, size_t hash_size, h4toh5_uint reserved,
			    H4TOH5I_free_t func);
hid_t H4TOH5I_register(H4TOH5I_type_t grp, void *object);
 void *H4TOH5I_object(hid_t id);
 H4TOH5I_type_t H4TOH5I_get_type(hid_t id);
void *H4TOH5I_remove(hid_t id);
 void *H4TOH5I_search(H4TOH5I_type_t grp, H4TOH5I_search_func_t func,
			 const void *key);
int  H4TOH5I_inc_ref(hid_t id);
int  H4TOH5I_dec_ref(hid_t id);
#endif
