/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group (THG).                                         *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of H4H5TOOLS. The full H4H5TOOLS copyright notice,      *
 * including terms governing use, modification, and redistribution, is       *
 * contained in the COPYING file, which can be found at the root of the      *
 * source code distribution tree. The copyright notice can also be found     *
 * at https://www.hdfgroup.org/licenses.  If you do not have access to       *
 * either file, you may request a copy from help@hdfgroup.org.               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifdef HAVE_LIBHDFEOS
#include "hdf.h"
#include "mfhdf.h"
#include "HdfEosDef.h"


#define err_ret(s)  return _err(__FILE__, __LINE__, "", "", "")
#define err1_ret(s)  return _err(__FILE__, __LINE__, s, "", "")
#define err2_ret(s1, s2)  return _err(__FILE__, __LINE__, s1, s2, "")
#define err3_ret(s1, s2, s3)  return _err(__FILE__, __LINE__, s1, s2, s3)
#define suicide()  do { _err(__FILE__, __LINE__, "assertion failure", "committing suicide", ""); exit(1); } while(0)

static int _err(const char *fname, int line, const char *s1, const char *s2, const char *s3)
{
  fprintf(stderr, "failed at %s:%d - %s %s %s\n", fname, line, s1, s2, s3);
#ifdef _MSC_VER
#ifdef _WIN32
#ifdef _DEBUG
  DebugBreak();
#endif
#endif
#endif
  return 1;
}

static int32 GDopen_const(const char *fname, intn access)
{
  return GDopen((char *)fname, access);
}

static int32 GDcreate_const(int32 id, const char *gridname, int32 xdim, int32 ydim, const float64 ul[], const float64 lr[])
{
  return GDcreate(id, (char *)gridname, xdim, ydim, (float64 *)ul, (float64 *)lr);
}

static intn GDdefproj_const(int32 id, int32 a, int32 b, int32 c, const float64 d[])
{
  return GDdefproj(id, a, b, c, (float64 *)d);
}

static intn GDdefdim_const(int32 id, const char *dimname, int32 size)
{
  return GDdefdim(id, (char *)dimname, size);
}

static intn GDdeffield_const(int32 id, const char *a, const char *b, int32 c, int32 d)
{
  return GDdeffield(id, (char *)a, (char *)b, c, d);
}

static intn GDsetfillvalue_const(int32 id, const char *a, const void *b)
{
  return GDsetfillvalue(id, (char *)a, (VOIDP)b);
}

static intn GDwriteattr_const(int32 id, const char *a, int32 b, int32 c, const VOIDP d)
{
  return GDwriteattr(id, (char *)a, b, c, (VOIDP)d);
}

static intn GDwritefield_const(int32 id, const char *a, const int32 b[], const int32 c[], const int32 d[], const VOIDP e)
{
  return GDwritefield(id, (char *)a, (int32 *)b, (int32 *)c, (int32 *)d, (VOIDP)e);
}

static int32 SWopen_const(const char *a, intn b)
{
  return SWopen((char *)a, b);
}

static int32 SWcreate_const(int32 a, const char *b)
{
  return SWcreate(a, (char *)b);
}

static intn SWdefdim_const(int32 a, const char *b, int32 c)
{
  return SWdefdim(a, (char *)b, c);
}

static intn SWdefdimmap_const(int32 a, const char *b, const char *c, int32 d, int32 e)
{
  return SWdefdimmap(a, (char *)b, (char *)c, d, e);
}

static intn SWwriteattr_const(int32 id, const char *a, int32 b, int32 c, const VOIDP d)
{
  return SWwriteattr(id, (char *)a, b, c, (VOIDP)d);
}

static intn SWdefgeofield_const(int32 a, const char *b, const char *c, int32 d, int32 e)
{
  return SWdefgeofield(a, (char *)b, (char *)c, d, e);
}

static intn SWsetfillvalue_const(int32 a, const char *b, const void *c)
{
  return SWsetfillvalue(a, (char *)b, (VOIDP)c);
}

static intn SWwritefield_const(int32 a, const char *b, const int32 *c, const int32 *d, const int32 *e, const VOIDP f)
{
  return SWwritefield(a, (char *)b, (int32 *)c, (int32 *)d, (int32 *)e, (VOIDP)f);
}

static intn SWdefdatafield_const(int32 a, const char *b, const char *c, int32 d, int32 e)
{
  return SWdefdatafield(a, (char *)b, (char *)c, d, e);
}


#define MAX_DIMS  5
#define MAX_FIELDS  20
#define MAX_GRIDS  15
#define MAX_SWATHS  15
#define MAX_DIMMAPS  5
#define MAX_BUFFER  5000
#define MAX_ATTRS  5

/* HDF-EOS2 has the following definition to check the validity of type
 *
 *  uint16          good_number[10] = {3, 4, 5, 6, 20, 21, 22, 23, 24, 25};
 *  UCHAR8, CHAR8, FLOAT32, FLOAT64, INT8, UINT8, INT16, UINT16, INT32, UINT32
 */
static const uint16 eostypes[10] = { 3, 4, 5, 6, 20, 21, 22, 23, 24, 25 };

#define UNIVERSAL_FILLVALUE  62
static const uint8 fillvalue_UCHAR8 = UNIVERSAL_FILLVALUE;
static const int8 fillvalue_CHAR8 = UNIVERSAL_FILLVALUE;
static const float32 fillvalue_FLOAT32 = UNIVERSAL_FILLVALUE;
static const float64 fillvalue_FLOAT64 = UNIVERSAL_FILLVALUE;
static const uint8 fillvalue_INT8 = UNIVERSAL_FILLVALUE;
static const int8 fillvalue_UINT8 = UNIVERSAL_FILLVALUE;
static const uint16 fillvalue_INT16 = UNIVERSAL_FILLVALUE;
static const int16 fillvalue_UINT16 = UNIVERSAL_FILLVALUE;
static const uint32 fillvalue_INT32 = UNIVERSAL_FILLVALUE;
static const int32 fillvalue_UINT32 = UNIVERSAL_FILLVALUE;
static const void * eosfillvalues[10] = {
  &fillvalue_UCHAR8,
  &fillvalue_CHAR8,
  &fillvalue_FLOAT32,
  &fillvalue_FLOAT64,
  &fillvalue_INT8,
  &fillvalue_UINT8,
  &fillvalue_INT16,
  &fillvalue_UINT16,
  &fillvalue_INT32,
  &fillvalue_UINT32,
};

#define IMPLEMENT_TYPES()	      \
  HANDLE_TYPE(DFNT_UCHAR8, uint8)     \
  HANDLE_TYPE(DFNT_CHAR8, int8)	      \
  HANDLE_TYPE(DFNT_FLOAT32, float32)  \
  HANDLE_TYPE(DFNT_FLOAT64, float64)  \
  HANDLE_TYPE(DFNT_INT8, uint8)	      \
  HANDLE_TYPE(DFNT_UINT8, int8)	      \
  HANDLE_TYPE(DFNT_INT16, uint16)     \
  HANDLE_TYPE(DFNT_UINT16, int16)     \
  HANDLE_TYPE(DFNT_INT32, uint32)     \
  HANDLE_TYPE(DFNT_UINT32, int32)

static void fill_data(const int32 *dimsizes, int32 datatype, int usefillvalue, void *data)
{
  int32 npoints = 1;
  int32 typesize;
  int i;

  typesize = DFKNTsize(datatype);
  for (i = 0; i < MAX_DIMS; ++i) {
    if (dimsizes[i] == 0) break;
    npoints *= dimsizes[i];
  }
  if (npoints * typesize > MAX_BUFFER) suicide();

  for (i = 0; i < npoints; ++i) {
    switch (datatype) {
#define HANDLE_TYPE(eostype, ctype) case eostype: { ((ctype *)data)[i] = i % 2 == 0 ? i + 1 : (usefillvalue ? UNIVERSAL_FILLVALUE : i + 1); } break;
      IMPLEMENT_TYPES()
#undef HANDLE_TYPE
      default:
	suicide();
	break;
    }
  }
}

static void fill_attr(int32 length, int32 datatype, void *data)
{
  int i;
  int32 typesize;

  typesize = DFKNTsize(datatype);
  if (length * typesize > MAX_BUFFER) suicide();
  for (i = 0; i < length; ++i) {
    switch (datatype) {
#define HANDLE_TYPE(eostype, ctype) case eostype: { ((ctype *)data)[i] = i + 1; } break;
      IMPLEMENT_TYPES()
#undef HANDLE_TYPE
      default:
	suicide();
	break;
    }
  }
}

typedef struct {
  const char *fieldname;
  int32 datatype;
  const char *dimnames;
  int32 dimsizes[MAX_DIMS];
  int use_fillvalue;
  const void *fillvalue;
} grid_field_t;

typedef struct {
  const char *dimname;
  int32 dimsize;
} grid_dim_t;

typedef struct {
  const char *attrname;
  int32 length;
} grid_attr_t;

typedef struct {
  const char *gridname;
  int32 xdimsize;
  int32 ydimsize;
  int32 projcode;
  int32 attrtype;
  grid_dim_t dims[MAX_DIMS];
  grid_field_t fields[MAX_FIELDS];
  grid_attr_t attrs[MAX_ATTRS];
} eos_grid_t;

typedef struct {
  const char *filename;
  eos_grid_t grids[MAX_GRIDS];
} grid_file_t;

typedef struct {
  int32 zonecode;
  int32 projcode;
  int32 spherecode;
  int32 origincode;
  float64 projparam[16];
  float64 upleft[2];
  float64 lowright[2];
} grid_proj_t;

static void fill_grid_proj(grid_proj_t *gridinfo, int32 projcode)
{
  int i;

  for (i = 0; i < 16; ++i)
    gridinfo->projparam[i] = 0;

  switch (projcode) {
    case GCTP_GEO:
      gridinfo->projcode = GCTP_GEO;
      gridinfo->zonecode = 0;
      gridinfo->spherecode = 0;
      gridinfo->origincode = 0;
      gridinfo->upleft[0] = EHconvAng(0.0, HDFE_DEG_DMS);
      gridinfo->upleft[1] = EHconvAng(30.0, HDFE_DEG_DMS);
      gridinfo->lowright[0] = EHconvAng(15.0, HDFE_DEG_DMS);
      gridinfo->lowright[1] = EHconvAng(20.0, HDFE_DEG_DMS);
      break;
    case GCTP_PS:
      gridinfo->projcode = GCTP_PS;
      gridinfo->zonecode = 0;
      gridinfo->spherecode = 3;
      gridinfo->origincode = HDFE_GD_LR;
      gridinfo->projparam[5] = 90000000.00;
      gridinfo->upleft[0] = 0;
      gridinfo->upleft[1] = 0;
      gridinfo->lowright[0] = 0;
      gridinfo->lowright[1] = 0;
      break;
    case GCTP_UTM:
      gridinfo->projcode = GCTP_UTM;
      gridinfo->zonecode = 40;
      gridinfo->spherecode = 0;
      gridinfo->origincode = 0;
      gridinfo->upleft[0] = 210584.50041;
      gridinfo->upleft[1] = 3322395.95445;
      gridinfo->lowright[0] = 813931.10959;
      gridinfo->lowright[1] = 2214162.53278;
      break;
    default:
      suicide();
  }
}

static int grid_generic(const grid_file_t *gridfile)
{
  int32 fileid;
  int data[MAX_BUFFER];
  int i, j;

  if ((fileid = GDopen_const(gridfile->filename, DFACC_CREATE)) == -1) err2_ret("cannot create", gridfile->filename);
  for (i = 0; i < MAX_GRIDS; ++i) {
    const eos_grid_t *grid = &gridfile->grids[i];
    grid_proj_t proj;
    int32 gdid;
    if (!grid->gridname) break;

    fill_grid_proj(&proj, grid->projcode);
    if ((gdid = GDcreate_const(fileid, grid->gridname, grid->xdimsize, grid->ydimsize, proj.upleft, proj.lowright)) == -1) err2_ret("cannot create", grid->gridname);
    if (GDdefproj_const(gdid, proj.projcode, proj.zonecode, proj.spherecode, proj.projparam) == -1) err1_ret("define projection");
    if (GDdeforigin(gdid, proj.origincode)) err1_ret("define origin");

    for (j = 0; j < MAX_DIMS; ++j) {
      const grid_dim_t *dim = &grid->dims[j];
      if (!dim->dimname) break;

      if (GDdefdim_const(gdid, dim->dimname, dim->dimsize) == -1) err2_ret("define dim", dim->dimname);
    }
    for (j = 0; j < MAX_ATTRS; ++j) {
      const grid_attr_t *attr = &grid->attrs[j];
      if (!attr->attrname) break;

      fill_attr(attr->length, grid->attrtype, data);
      if (GDwriteattr_const(gdid, attr->attrname, grid->attrtype, attr->length, data) == -1) err2_ret("write", attr->attrname);
    }
    for (j = 0; j < MAX_FIELDS; ++j) {
      const grid_field_t *field = &grid->fields[j];
      if (!field->fieldname) break;

      if (GDdeffield_const(gdid, field->fieldname, field->dimnames, field->datatype, HDFE_NOMERGE) == -1) err2_ret("define field", field->fieldname);
      if (field->use_fillvalue) {
	if (GDsetfillvalue_const(gdid, field->fieldname, field->fillvalue) == -1) err2_ret("set fillvalue", field->fieldname);
      }
      fill_data(field->dimsizes, field->datatype, field->use_fillvalue, data);
      if (GDwritefield_const(gdid, field->fieldname, NULL, NULL, NULL, data) == -1) err2_ret("write", field->fieldname);
    }
    GDdetach(gdid);
  }
  GDclose(fileid);
  return 0;
}

typedef struct {
  const char *fieldname;
  int32 datatype;
  const char *dimnames;
  int32 dimsizes[MAX_DIMS];
  int use_fillvalue;
  const void *fillvalue;
} swath_field_t;

typedef struct {
  const char *dimname;
  int32 dimsize;
} swath_dim_t;

typedef struct {
  const char *geoname;
  const char *dataname;
  int32 offset;
  int32 increment;
} swath_dimmap_t;

typedef struct {
  const char *attrname;
  int32 length;
} swath_attr_t;

typedef struct {
  const char *swathname;
  int32 attrtype;
  swath_dim_t dims[MAX_DIMS];
  swath_dimmap_t dimmaps[MAX_DIMMAPS];
  swath_field_t geofields[MAX_FIELDS];
  swath_field_t datafields[MAX_FIELDS];
  swath_attr_t attrs[MAX_ATTRS];
} eos_swath_t;

typedef struct {
  const char *filename;
  eos_swath_t swaths[MAX_SWATHS];
} swath_file_t;

static int32 get_dimsizen(const eos_swath_t *swath, const char *dimname, int len)
{
  int i;

  for (i = 0; i < MAX_DIMS; ++i) {
    const swath_dim_t *dim = &swath->dims[i];
    if (!dim->dimname) break;

    if (strncmp(dim->dimname, dimname, len) == 0)
      return dim->dimsize;
  }
  suicide();
  return -1;
}

static int32 get_dimsize(const eos_swath_t *swath, const char *dimname)
{
  return get_dimsizen(swath, dimname, (int)strlen(dimname));
}

static void validate_swath(const swath_file_t *swathfile)
{
  int i, j, k;

  for (i = 0; i < MAX_SWATHS; ++i) {
    const eos_swath_t *swath = &swathfile->swaths[i];
    if (!swath->swathname) break;

    /* dimension maps */
    for (j = 0; j < MAX_DIMS; ++j) {
      const swath_dimmap_t *dimmap = &swath->dimmaps[j];
      int32 geosize, datasize;
      if (!dimmap->geoname) break;

      geosize = get_dimsize(swath, dimmap->geoname);
      datasize = get_dimsize(swath, dimmap->dataname);

      /* The "offset" tells how far along a data dimension you must travel to find the */
      /* first point to have a corresponding entry along the geolocation dimension. */
      if (dimmap->offset > 0)
	datasize -= dimmap->offset;
      else if (dimmap->offset < 0)
	geosize -= -dimmap->offset;

      /* The "increment" tells how many points to travel along the data dimension before */
      /* the next point is found for which there is a corresponding entry along the geolocation dimension. */
      if (dimmap->increment > 1) {
	/* datasize = ( (geosize - 1) * increment, geosize * increment ] */
	if (!(((geosize - 1) * dimmap->increment < datasize) &&
	       (datasize <= geosize * dimmap->increment))) suicide();
      }
      else if (dimmap->increment < -1) {
	/* geosize = ( (datasize - 1) * increment, datasize * increment ] */
	if (!(((datasize - 1) * -dimmap->increment < geosize) &&
	       (geosize <= datasize * -dimmap->increment))) suicide();
      }
    }

    /* geo fields, data fields */
    for (j = 0; j < 2; ++j) {
      const swath_field_t *fields = j == 0 ? swath->geofields : swath->datafields;
      for (k = 0; k < MAX_FIELDS; ++k) {
	const swath_field_t *field = &fields[k];
	int dimnamelength, l, m, n = 0;
	if (!field->fieldname) break;

	dimnamelength = (int)strlen(field->dimnames);
	for (l = m = 0; m <= dimnamelength; ++m) {
	  if ((m == dimnamelength && dimnamelength) || field->dimnames[m] == ',') {
	    int32 dimsize = get_dimsizen(swath, field->dimnames + l, m - l);
	    if (dimsize != field->dimsizes[n++]) suicide();

	    l = m + 1;
	  }
	}
      }
    }
  }
}

static int swath_generic(const swath_file_t *swathfile)
{
  int32 fileid;
  int data[MAX_BUFFER];
  float geo[MAX_BUFFER];
  int i, j;

  validate_swath(swathfile);

  if ((fileid = SWopen_const(swathfile->filename, DFACC_CREATE)) == -1) err2_ret("cannot create", swathfile->filename);
  for (i = 0; i < MAX_SWATHS; ++i) {
    const eos_swath_t *swath = &swathfile->swaths[i];
    int32 swid;
    if (!swath->swathname) break;

    if ((swid = SWcreate_const(fileid, swath->swathname)) == -1) err2_ret("cannot create", swath->swathname);

    for (j = 0; j < MAX_DIMS; ++j) {
      const swath_dim_t *dim = &swath->dims[j];
      if (!dim->dimname) break;

      if (SWdefdim_const(swid, dim->dimname, dim->dimsize) == -1) err2_ret("define dim", dim->dimname);
    }
    for (j = 0; j < MAX_DIMMAPS; ++j) {
      const swath_dimmap_t *dimmap = &swath->dimmaps[j];
      if (!dimmap->geoname) break;

      if (SWdefdimmap_const(swid, dimmap->geoname, dimmap->dataname, dimmap->offset, dimmap->increment) == -1) err3_ret("define dimmap", dimmap->geoname, dimmap->dataname);
    }
    for (j = 0; j < MAX_ATTRS; ++j) {
      const swath_attr_t *attr = &swath->attrs[j];
      if (!attr->attrname) break;

      fill_attr(attr->length, swath->attrtype, data);
      if (SWwriteattr_const(swid, attr->attrname, swath->attrtype, attr->length, data) == -1) err2_ret("write", attr->attrname);
    }
    for (j = 0; j < MAX_FIELDS; ++j) {
      const swath_field_t *field = &swath->geofields[j];
      if (!field->fieldname) break;

      if (SWdefgeofield_const(swid, field->fieldname, field->dimnames, field->datatype, HDFE_NOMERGE) == -1) err2_ret("define field", field->fieldname);
      if (field->use_fillvalue) {
	if (SWsetfillvalue_const(swid, field->fieldname, field->fillvalue) == -1) err2_ret("set fillvalue", field->fieldname);
      }
      fill_data(field->dimsizes, field->datatype, field->use_fillvalue, geo);
      if (SWwritefield_const(swid, field->fieldname, NULL, NULL, NULL, geo) == -1) err2_ret("write", field->fieldname);
    }
    for (j = 0; j < MAX_FIELDS; ++j) {
      const swath_field_t *field = &swath->datafields[j];
      if (!field->fieldname) break;

      if (SWdefdatafield_const(swid, field->fieldname, field->dimnames, field->datatype, HDFE_NOMERGE) == -1) err2_ret("define field", field->fieldname);
      if (field->use_fillvalue) {
	if (SWsetfillvalue_const(swid, field->fieldname, field->fillvalue) == -1) err2_ret("set fillvalue", field->fieldname);
      }
      fill_data(field->dimsizes, field->datatype, field->use_fillvalue, data);
      if (SWwritefield_const(swid, field->fieldname, NULL, NULL, NULL, data) == -1) err2_ret("write", field->fieldname);
    }
    SWdetach(swid);
  }
  SWclose(fileid);
  return 0;
}

int main(int argc, char **argv)
{
#define ASSERT_SUCCESS(x)  \
  if (x) { fprintf(stderr, "failed to generate %s\n", eos.filename); return 1; } \
  else { printf("generated %s\n", eos.filename); }

#pragma region GENERIC {{{
#pragma region Type {{{
#pragma region Grid {{{
  {
    const char *gridname[] = { "uchar8", "char8", "float32", "float64", "int8", "uint8", "int16", "uint16", "int32", "uint32" };
    int i;
    eos_grid_t grid =
	{ "", 5, 7, GCTP_GEO, 0,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", 0, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { "field1_fill", 0, "XDim,YDim", { 5, 7 }, 1, 0 },
	    { 0, 0, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "attr1", 3 },
	  }
	};
    grid_file_t eos;
    memset(&eos, 0, sizeof(eos));
    eos.filename = "generic_type_grid.hdf";

    for (i = 0; i < sizeof(eostypes) / sizeof(eostypes[0]); ++i) {
      eos.grids[i] = grid;
      eos.grids[i].gridname = gridname[i];
      eos.grids[i].attrtype = eostypes[i];
      eos.grids[i].fields[0].datatype = eostypes[i];
      eos.grids[i].fields[1].datatype = eostypes[i];
      eos.grids[i].fields[1].fillvalue = eosfillvalues[i];
    }
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}

#pragma region Swath {{{
  {
    const char *swathname[] = { "uchar8", "char8", "float32", "float64", "int8", "uint8", "int16", "uint16", "int32", "uint32" };
    int i;
    eos_swath_t swath =
	{ "", 0,
	  {
	    { "dim0", 3 },
	    { "dim1", 5 },
	    { "dim2", 7 },
	  },
	  {
	    { 0, 0, 0, 0 },
	  },
	  {
	    { "Time", DFNT_FLOAT32, "dim0", { 3 }, 0, 0 },
	    { "Longitude", DFNT_FLOAT32, "dim1,dim2", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "dim1,dim2", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", 0, "dim0", { 3 }, 0, 0 },
	    { "field1_fill", 0, "dim0", { 3 }, 1, 0 },
	    { "field2", 0, "dim1,dim2", { 5, 7 }, 0, 0 },
	    { "field2_fill", 0, "dim1,dim2", { 5, 7 }, 1, 0 },
	    { 0, 0, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "attr1", 3 },
	  }
	};
    swath_file_t eos;
    memset(&eos, 0, sizeof(eos));
    eos.filename = "generic_type_swath.hdf";

    for (i = 0; i < sizeof(eostypes) / sizeof(eostypes[0]); ++i) {
      eos.swaths[i] = swath;
      eos.swaths[i].swathname = swathname[i];
      eos.swaths[i].attrtype = eostypes[i];
      eos.swaths[i].datafields[0].datatype = eostypes[i];
      eos.swaths[i].datafields[1].datatype = eostypes[i];
      eos.swaths[i].datafields[1].fillvalue = eosfillvalues[i];
      eos.swaths[i].datafields[2].datatype = eostypes[i];
      eos.swaths[i].datafields[3].datatype = eostypes[i];
      eos.swaths[i].datafields[3].fillvalue = eosfillvalues[i];
    }
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

#pragma region Shared name {{{
#pragma region Grid {{{
  {
    /* It turns out that this is not a valid file because
     * every field should refer to at least XDim and YDim.
     */
    grid_file_t eos =
    { "grid_sharedname.hdf",
      {
	{ "grid1", 5, 7, GCTP_GEO, DFNT_INT32,
	  {
	    { "dim0", 9 },
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "dim0", { 9 }, 0, 0 },
	    { "XDim", DFNT_INT32, "XDim", { 5 }, 0, 0 },
	    { "YDim", DFNT_INT32, "YDim", { 7 }, 0, 0 },
	    { "dim0", DFNT_INT32, "dim0", { 9 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	},
	{ "grid2", 5, 7, GCTP_UTM, DFNT_INT32,
	  {
	    { "dim0", 9 },
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "dim0", { 9 }, 0, 0 },
	    { "XDim", DFNT_INT32, "XDim", { 5 }, 0, 0 },
	    { "YDim", DFNT_INT32, "YDim", { 7 }, 0, 0 },
	    { "dim0", DFNT_INT32, "dim0", { 9 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}

#pragma region Swath {{{
  {
    swath_file_t eos =
    { "swath_sharedname.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "dim0", 5 },
	    { "dim1", 7 },
	    { "dim2", 9 },
	  },
	  {
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "dim0,dim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "dim0,dim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "dim0,dim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "dim0,dim1", { 5, 7 }, 0, 0 },
	    { "dim2", DFNT_INT32, "dim2", { 9 } , 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

#pragma region Bad name {{{
  /* HDF-EOS2 allows slashes in names, but netCDF-4 does not */
#pragma region Grid {{{
  {
    grid_file_t eos =
    { "grid_badname.hdf",
      {
	{ "grid1/slash", 5, 7, GCTP_GEO, DFNT_INT32,
	  {
	    { "dim0/slash", 3 },
	    { 0, 0 },
	  },
	  {
	    { "field1/slash", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { "field2/slash", DFNT_INT32, "dim0/slash", { 3 }, 0, 0 },
	    { "field3/slash", DFNT_INT32, "XDim,YDim,dim0/slash", { 5, 7, 3 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "attr1/slash", 3 },
	    { 0, 0 },
	  }
	},
	{ "grid2", 5, 7, GCTP_UTM, DFNT_INT32,
	  {
	    { "dim0/slash", 3 },
	    { 0, 0 },
	  },
	  {
	    { "field1/slash", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { "field2/slash", DFNT_INT32, "dim0/slash", { 3 }, 0, 0 },
	    { "field3/slash", DFNT_INT32, "XDim,YDim,dim0/slash", { 5, 7, 3 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "attr1/slash", 3 },
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}

#pragma region Swath {{{
  {
    swath_file_t eos =
    { "swath_badname.hdf",
      {
	{ "swath1/slash", DFNT_INT32,
	  {
	    { "dim0/slash", 5 },
	    { "dim1/slash", 7 },
	  },
	  {
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "dim0/slash,dim1/slash", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "dim0/slash,dim1/slash", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "dim0/slash,dim1/slash", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "dim0/slash,dim1/slash", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "attr1/slash", 3 },
	    { 0, 0 },
	  }
	},
	{ "swath2/slash", DFNT_INT32,
	  {
	    { "geodim0_slash", 4 },
	    { "geodim1_slash", 6 },
	    { "datadim0_slash", 5 },
	    { "datadim1_slash", 7 },
	  },
	  {
	    /* When dimension maps are used, slash cannot be used because this is the delimiter.
	     * I can parse more smartly, but isn't it overkill? */
	    { "geodim0_slash", "datadim0_slash", 1, 1 },
	    { "geodim1_slash", "datadim1_slash", 1, 1 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0_slash,geodim1_slash", { 4, 6 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0_slash,geodim1_slash", { 4, 6 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0_slash,datadim1_slash", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0_slash,datadim1_slash", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "attr1/slash", 3 },
	    { 0, 0 },
	  }
	}
      }
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}
#pragma endregion }}}

#pragma region GRID {{{
#pragma region GEO {{{
#pragma region V(XDim, YDim) GCTP_GEO {{{
  {
    grid_file_t eos =
    { "grid_geo_xy.hdf",
      {
	{ "grid1", 5, 7, GCTP_GEO, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region V(YDim, XDim) GCTP_GEO {{{
  {
    grid_file_t eos =
    { "grid_geo_yx.hdf",
      {
	{ "grid1", 5, 7, GCTP_GEO, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "YDim,XDim", { 7, 5 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region V(Time, XDim, YDim) GCTP_GEO {{{
  {
    grid_file_t eos =
    { "grid_geo_timexy.hdf",
      {
	{ "grid1", 5, 7, GCTP_GEO, DFNT_INT32,
	  {
	    { "Time", 3 },
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "Time,XDim,YDim", { 3, 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region V(Time, YDim, XDim) GCTP_GEO {{{
  {
    grid_file_t eos =
    { "grid_geo_timeyx.hdf",
      {
	{ "grid1", 5, 7, GCTP_GEO, DFNT_INT32,
	  {
	    { "Time", 3 },
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "Time,YDim,XDim", { 3, 7, 5 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

#pragma region UTM {{{
#pragma region V(XDim, YDim) GCTP_UTM (2-D) {{{
  {
    grid_file_t eos =
    { "grid_utm_xy.hdf",
      {
	{ "grid1", 5, 7, GCTP_UTM, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region V(YDim, XDim) GCTP_UTM (2-D) {{{
  {
    grid_file_t eos =
    { "grid_utm_yx.hdf",
      {
	{ "grid1", 5, 7, GCTP_UTM, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "YDim,XDim", { 7, 5 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region V(Time, XDim, YDim) GCTP_UTM (2-D) {{{
  {
    grid_file_t eos =
    { "grid_utm_timexy.hdf",
      {
	{ "grid1", 5, 7, GCTP_UTM, DFNT_INT32,
	  {
	    { "Time", 3 },
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "Time,XDim,YDim", { 3, 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region V(Time, YDim, XDim) GCTP_UTM (2-D) {{{
  {
    grid_file_t eos =
    { "grid_utm_timeyx.hdf",
      {
	{ "grid1", 5, 7, GCTP_UTM, DFNT_INT32,
	  {
	    { "Time", 3 },
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "Time,YDim,XDim", { 3, 7, 5 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

#pragma region Multiple variables {{{
#pragma region GEO {{{
#pragma region V(XDim, YDim) W(XDim, YDim) GCTP_GEO {{{
  {
    grid_file_t eos =
    { "grid_geo_xyxy.hdf",
      {
	{ "grid1", 5, 7, GCTP_GEO, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region V(YDim, XDim) W(YDim, XDim) GCTP_GEO {{{
  {
    grid_file_t eos =
    { "grid_geo_yxyx.hdf",
      {
	{ "grid1", 5, 7, GCTP_GEO, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "YDim,XDim", { 7, 5 }, 0, 0 },
	    { "field2", DFNT_INT32, "YDim,XDim", { 7, 5 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region V(XDim, YDim) W(YDim, XDim) GCTP_GEO (conflict) {{{
  {
    grid_file_t eos =
    { "grid_geo_xyyx.hdf",
      {
	{ "grid1", 5, 7, GCTP_GEO, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "YDim,XDim", { 7, 5 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region no fields GCTP_GEO (cannot guess) {{{
  {
    grid_file_t eos =
    { "grid_geo_nil.hdf",
      {
	{ "grid1", 5, 7, GCTP_GEO, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

#pragma region UTM {{{
#pragma region V(XDim, YDim) W(XDim, YDim) GCTP_UTM (2-D) {{{
  {
    grid_file_t eos =
    { "grid_utm_xyxy.hdf",
      {
	{ "grid1", 5, 7, GCTP_UTM, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region V(YDim, XDim) W(YDim, XDim) GCTP_UTM (2-D) {{{
  {
    grid_file_t eos =
    { "grid_utm_yxyx.hdf",
      {
	{ "grid1", 5, 7, GCTP_UTM, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "YDim,XDim", { 7, 5 }, 0, 0 },
	    { "field2", DFNT_INT32, "YDim,XDim", { 7, 5 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region V(XDim, YDim) W(YDim, XDim) GCTP_UTM (conflict) {{{
  {
    grid_file_t eos =
    { "grid_utm_xyyx.hdf",
      {
	{ "grid1", 5, 7, GCTP_UTM, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "XDim,YDim", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "YDim,XDim", { 7, 5 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma region no fields GCTP_UTM (cannot guess) {{{
  {
    grid_file_t eos =
    { "grid_utm_nil.hdf",
      {
	{ "grid1", 5, 7, GCTP_UTM, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}
#pragma endregion }}}

#pragma region Multiple Grids {{{
  {
    grid_file_t eos =
    { "grid_multgrids.hdf",
      {
	{ "grid1", 1, 3, GCTP_GEO, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "XDim,YDim", { 1, 3 }, 0, 0 },
	    { "field2", DFNT_INT32, "XDim,YDim", { 1, 3 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "Attribute1", 1 },
	    { "Attribute2", 3 },
	  }
	},
	{ "grid2", 5, 7, GCTP_UTM, DFNT_INT32,
	  {
	    { 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "YDim,XDim", { 7, 5 }, 0, 0 },
	    { "field2", DFNT_INT32, "YDim,XDim", { 7, 5 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "Attribute1", 5 },
	    { "Attribute2", 7 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(grid_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

#pragma region SWATH {{{
#pragma region dimension map {{{
#pragma region without dimmap {{{
  {
    swath_file_t eos =
    { "swath_basic.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "dim0", 5 },
	    { "dim1", 7 },
	  },
	  {
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "dim0,dim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "dim0,dim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "dim0,dim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "dim0,dim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}

#pragma region without scaling {{{
#pragma region dimmap (offset: 0, increment: 1) {{{
  {
    swath_file_t eos =
    { "swath_0101.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 5 },
	    { "geodim1", 7 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", 0, 1 },
	    { "geodim1", "datadim1", 0, 1 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: 1, increment: 1) {{{
  {
    swath_file_t eos =
    { "swath_1111.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 4 },
	    { "geodim1", 6 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", 1, 1 },
	    { "geodim1", "datadim1", 1, 1 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 4, 6 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 4, 6 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: 2, increment: 1) {{{
  {
    swath_file_t eos =
    { "swath_2121.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 3 },
	    { "geodim1", 5 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", 2, 1 },
	    { "geodim1", "datadim1", 2, 1 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 3, 5 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 3, 5 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: -1, increment: 1) {{{
  {
    swath_file_t eos =
    { "swath_n11n11.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 6 },
	    { "geodim1", 8 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", -1, 1 },
	    { "geodim1", "datadim1", -1, 1 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 6, 8 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 6, 8 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: -1, increment: 1) geodim is not bigger than datadim {{{
  {
    swath_file_t eos =
    { "swath_n11n11_tight.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 5 },
	    { "geodim1", 7 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", -1, 1 },
	    { "geodim1", "datadim1", -1, 1 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: -2, increment: 1) {{{
  {
    swath_file_t eos =
    { "swath_n21n21.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 7 },
	    { "geodim1", 9 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", -2, 1 },
	    { "geodim1", "datadim1", -2, 1 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 7, 9 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 7, 9 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

#pragma region scaling (increment > 1) {{{
#pragma region dimmap (offset: 0, increment: 2) {{{
  {
    swath_file_t eos =
    { "swath_0202.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 3 },
	    { "geodim1", 4 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", 0, 2 },
	    { "geodim1", "datadim1", 0, 2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 3, 4 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 3, 4 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: 1, increment: 2) {{{
  {
    swath_file_t eos =
    { "swath_1212.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 2 },
	    { "geodim1", 3 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", 1, 2 },
	    { "geodim1", "datadim1", 1, 2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 2, 3 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 2, 3 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: 2, increment: 2) {{{
  {
    swath_file_t eos =
    { "swath_2222.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 2 },
	    { "geodim1", 3 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", 2, 2 },
	    { "geodim1", "datadim1", 2, 2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 2, 3 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 2, 3 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: -1, increment: 2) {{{
  {
    swath_file_t eos =
    { "swath_n12n12.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 4 },
	    { "geodim1", 5 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", -1, 2 },
	    { "geodim1", "datadim1", -1, 2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 4, 5 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 4, 5 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: -2, increment: 2) {{{
  {
    swath_file_t eos =
    { "swath_n22n22.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 5 },
	    { "geodim1", 6 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", -2, 2 },
	    { "geodim1", "datadim1", -2, 2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 6 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 6 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

#pragma region scaling (increment < -1) {{{
#pragma region dimmap (offset: 0, increment: -2) {{{
  {
    swath_file_t eos =
    { "swath_0n20n2.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 9 },
	    { "geodim1", 13 }, 
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", 0, -2 },
	    { "geodim1", "datadim1", 0, -2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 9, 13 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 9, 13 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: 1, increment: -2) {{{
  {
    swath_file_t eos =
    { "swath_1n21n2.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 7 },
	    { "geodim1", 11 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", 1, -2 },
	    { "geodim1", "datadim1", 1, -2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 7, 11 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 7, 11 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: 2, increment: -2) {{{
  {
    swath_file_t eos =
    { "swath_2n22n2.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 6 },
	    { "geodim1", 10 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", 2, -2 },
	    { "geodim1", "datadim1", 2, -2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 6, 10 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 6, 10 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: -1, increment: -2) {{{
  {
    swath_file_t eos =
    { "swath_n1n2n1n2.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 10 },
	    { "geodim1", 14 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", -1, -2 },
	    { "geodim1", "datadim1", -1, -2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 10, 14 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 10, 14 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma region dimmap (offset: -2, increment: -2) {{{
  {
    swath_file_t eos =
    { "swath_n2n2n2n2.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "geodim0", 11 },
	    { "geodim1", 15 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", -2, -2 },
	    { "geodim1", "datadim1", -2, -2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 11, 15 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 11, 15 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "field2", DFNT_INT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

#pragma region multiple geofields for a datafield {{{
#pragma region one data field without dimmap {{{
#pragma region qual, datadim0, datadim1 {{{
  {
    swath_file_t eos =
    { "swath_qdd.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "qualdim", 3 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "qualdim,datadim0,datadim1", { 3, 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}

#pragma region datadim0, qual, datadim1 {{{
  {
    swath_file_t eos =
    { "swath_dqd.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "qualdim", 3 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,qualdim,datadim1", { 5, 3, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}

#pragma region datadim0, datadim1, qual {{{
  {
    swath_file_t eos =
    { "swath_ddq.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "qualdim", 3 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "datadim0,datadim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1,qualdim", { 5, 7, 3 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

#pragma region one data field with a dimmap {{{
#pragma region qual, geodim0, geodim1 {{{
  {
    swath_file_t eos =
    { "swath_qgg.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "qualdim", 3 },
	    { "geodim0", 5 },
	    { "geodim1", 7 },
	    { "datadim0", 10 },
	    { "datadim1", 14 },
	  },
	  {
  	    { "geodim0", "datadim0", 0, 2 },
	    { "geodim1", "datadim1", 0, 2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "qualdim,datadim0,datadim1", { 3, 10, 14 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}

#pragma region geodim0, qual, geodim1 {{{
  {
    swath_file_t eos =
    { "swath_gqg.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "qualdim", 3 },
	    { "geodim0", 5 },
	    { "geodim1", 7 },
	    { "datadim0", 10 },
	    { "datadim1", 14 },
	  },
	  {
  	    { "geodim0", "datadim0", 0, 2 },
	    { "geodim1", "datadim1", 0, 2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,qualdim,datadim1", { 10, 3, 14 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}

#pragma region geodim0, geodim1, qual {{{
  {
    swath_file_t eos =
    { "swath_ggq.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "qualdim", 3 },
	    { "geodim0", 5 },
	    { "geodim1", 7 },
	    { "datadim0", 10 },
	    { "datadim1", 14 },
	  },
	  {
  	    { "geodim0", "datadim0", 0, 2 },
	    { "geodim1", "datadim1", 0, 2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field1", DFNT_INT32, "datadim0,datadim1,qualdim", { 10, 14, 3 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

#pragma region Qual, Longitude, Latitude {{{
  {
    swath_file_t eos =
    { "swath_tll.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "qualdim", 3 },
	    { "geodim0", 5 },
	    { "geodim1", 7 },
	    { "datadim0", 5 },
	    { "datadim1", 7 },
	  },
	  {
	    { "geodim0", "datadim0", 0, 1 },
	    { "geodim1", "datadim1", 0, 1 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field01", DFNT_INT32, "qualdim", { 3 }, 0, 0 },
	    { "field02", DFNT_INT32, "geodim0", { 5 }, 0, 0 },
	    { "field03", DFNT_INT32, "geodim1", { 7 }, 0, 0 },
	    { "field04", DFNT_INT32, "datadim0", { 5 }, 0, 0 },
	    { "field05", DFNT_INT32, "datadim1", { 7 }, 0, 0 },
	    { "field06", DFNT_INT32, "qualdim,geodim0", { 3, 5 }, 0, 0 },
	    { "field07", DFNT_INT32, "qualdim,geodim1", { 3, 7 }, 0, 0 },
	    { "field08", DFNT_INT32, "qualdim,datadim0", { 3, 5 }, 0, 0 },
	    { "field09", DFNT_INT32, "qualdim,datadim1", { 3, 7 }, 0, 0 },
	    { "field10", DFNT_INT32, "geodim0,qualdim", { 5, 3 }, 0, 0 },
	    { "field11", DFNT_INT32, "geodim1,qualdim", { 7, 3 }, 0, 0 },
	    { "field12", DFNT_INT32, "datadim0,qualdim", { 5, 3 }, 0, 0 },
	    { "field13", DFNT_INT32, "datadim1,qualdim", { 7, 3 }, 0, 0 },
	    { "field14", DFNT_INT32, "qualdim,geodim0,geodim1", { 3, 5, 7 }, 0, 0 },
	    { "field15", DFNT_INT32, "qualdim,datadim0,datadim1", { 3, 5, 7 }, 0, 0 },
	    { "field16", DFNT_INT32, "geodim0,qualdim,geodim1", { 5, 3, 7 }, 0, 0 },
	    { "field17", DFNT_INT32, "datadim0,qualdim,datadim1", { 5, 3, 7 }, 0, 0 },
	    { "field18", DFNT_INT32, "geodim0,geodim1,qualdim", { 5, 7, 3 }, 0, 0 },
	    { "field19", DFNT_INT32, "datadim0,datadim1,qualdim", { 5, 7, 3 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}

#pragma region Qual, Longitude, Latitude with scaling {{{
  {
    swath_file_t eos =
    { "swath_tll02.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "qualdim", 3 },
	    { "geodim0", 5 },
	    { "geodim1", 7 },
	    { "datadim0", 10 },
	    { "datadim1", 14 },
	  },
	  {
	    { "geodim0", "datadim0", 0, 2 },
	    { "geodim1", "datadim1", 0, 2 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field01", DFNT_INT32, "qualdim", { 3 }, 0, 0 },
	    { "field02", DFNT_INT32, "geodim0", { 5 }, 0, 0 },
	    { "field03", DFNT_INT32, "geodim1", { 7 }, 0, 0 },
	    { "field04", DFNT_INT32, "datadim0", { 10 }, 0, 0 },
	    { "field05", DFNT_INT32, "datadim1", { 14 }, 0, 0 },
	    { "field06", DFNT_INT32, "qualdim,geodim0", { 3, 5 }, 0, 0 },
	    { "field07", DFNT_INT32, "qualdim,geodim1", { 3, 7 }, 0, 0 },
	    { "field08", DFNT_INT32, "qualdim,datadim0", { 3, 10 }, 0, 0 },
	    { "field09", DFNT_INT32, "qualdim,datadim1", { 3, 14 }, 0, 0 },
	    { "field10", DFNT_INT32, "geodim0,qualdim", { 5, 3 }, 0, 0 },
	    { "field11", DFNT_INT32, "geodim1,qualdim", { 7, 3 }, 0, 0 },
	    { "field12", DFNT_INT32, "datadim0,qualdim", { 10, 3 }, 0, 0 },
	    { "field13", DFNT_INT32, "datadim1,qualdim", { 14, 3 }, 0, 0 },
	    { "field14", DFNT_INT32, "qualdim,geodim0,geodim1", { 3, 5, 7 }, 0, 0 },
	    { "field15", DFNT_INT32, "qualdim,datadim0,datadim1", { 3, 10, 14 }, 0, 0 },
	    { "field16", DFNT_INT32, "geodim0,qualdim,geodim1", { 5, 3, 7 }, 0, 0 },
	    { "field17", DFNT_INT32, "datadim0,qualdim,datadim1", { 10, 3, 14 }, 0, 0 },
	    { "field18", DFNT_INT32, "geodim0,geodim1,qualdim", { 5, 7, 3 }, 0, 0 },
	    { "field19", DFNT_INT32, "datadim0,datadim1,qualdim", { 10, 14, 3 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { 0, 0 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}
#pragma endregion }}}

#pragma region Multiple Swaths {{{
  {
    swath_file_t eos =
    { "swath_multswaths.hdf",
      {
	{ "swath1", DFNT_INT32,
	  {
	    { "qualdim", 2 },
	    { "geodim0", 4 },
	    { "geodim1", 6 },
	    { "datadim0", 4 },
	    { "datadim1", 6 },
	  },
	  {
	    { "geodim0", "datadim0", 0, 1 },
	    { "geodim1", "datadim1", 0, 1 },
	    { 0, 0 },
	  },
	  {
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 4, 6 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 4, 6 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field01", DFNT_INT32, "qualdim", { 2 }, 0, 0 },
	    { "field02", DFNT_INT32, "datadim0,datadim1", { 4, 6 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "Attribute1", 2 },
	    { "Attribute2", 4 },
	  }
	},
	{ "swath2", DFNT_INT32,
	  {
	    { "qualdim", 3 },
	    { "geodim0", 5 },
	    { "geodim1", 7 },
	  },
	  {
	    { 0, 0 },
	  },
	  {
	    { "Qual", DFNT_FLOAT32, "qualdim", { 3, }, 0, 0 },
	    { "Longitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { "Latitude", DFNT_FLOAT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "field01", DFNT_INT32, "qualdim", { 3 }, 0, 0 },
	    { "field02", DFNT_INT32, "geodim0,geodim1", { 5, 7 }, 0, 0 },
	    { 0, DFNT_INT32, 0, { 0, }, 0, 0 },
	  },
	  {
	    { "Attribute1", 3 },
	    { "Attribute2", 5 },
	  }
	}
      },
    };
    ASSERT_SUCCESS(swath_generic(&eos))
  }
#pragma endregion }}}
#pragma endregion }}}

  return 0;
}

#else
int main(int argc, char **argv)
{
  return 0;
}
#endif
/* vim:set ts=8 sw=2 sts=2 cindent fdm=marker: */
