# CCR v2.0.0 Sprint 1: Remove Legacy Compression Algorithms

## Overview
This document provides a comprehensive plan for removing ZStandard, Bit Grooming, GRANULARBR, and BITROUND algorithms from CCR, leaving only BZIP2 and LZ4. This is a major breaking change that will significantly simplify the codebase and reduce dependencies.

**Total Estimated Duration:** 10-13 days  
**Target:** Remove 4 compression algorithms (~75% of current algorithms)  
**Impact:** 200+ code references across the codebase

## Sprint Goals
- Remove all ZStandard compression functionality
- Remove all Bit Grooming compression functionality
- Remove all GRANULARBR compression functionality
- Remove all BITROUND compression functionality
- Update build systems and configuration files
- Update tests and documentation
- Ensure remaining functionality (BZIP2, LZ4) continues to work

## Implementation Phases

### Phase 1: Remove ZStandard Components
**Duration:** 2-3 days | **Priority:** High

#### Step 1.1: Remove ZStandard HDF5 Plugin Directory
```bash
# Remove entire ZSTANDARD plugin directory
rm -rf hdf5_plugins/ZSTANDARD/
```

#### Step 1.2: Update Core Library Files
**File:** `src/ccr.c` (41 matches)
- [ ] Remove ZStandard algorithm constants
- [ ] Remove ZStandard compression/decompression functions
- [ ] Remove ZStandard initialization code
- [ ] Remove ZStandard error handling
- [ ] Update algorithm enumeration

**File:** `include/ccr.h`
- [ ] Remove ZStandard constants and macros
- [ ] Remove ZStandard function declarations
- [ ] Update algorithm enum definitions

**File:** `fsrc/ccr.F90` (18 matches)
- [ ] Remove ZStandard Fortran interface functions
- [ ] Remove ZStandard constants
- [ ] Update Fortran module exports

#### Step 1.3: Remove ZStandard Tests
```bash
# Remove test files
rm -f test/tst_zstandard.c
rm -f test_h5/tst_h_zstandard.c
rm -f test_h5/tst_zstandard_size.c
rm -f ftest/ftst_ccr_zstandard.F90
```

#### Step 1.4: Update Build Configuration
**File:** `configure.ac` (10 matches)
- [ ] Remove ZStandard dependency checks
- [ ] Remove ZStandard configuration options
- [ ] Remove ZStandard library linking

**File:** `hdf5_plugins/configure.ac`
- [ ] Remove ZSTANDARD subdirectory references
- [ ] Update plugin list

**File:** `hdf5_plugins/Makefile.am`
- [ ] Remove ZSTANDARD from SUBDIRS
- [ ] Remove ZSTANDARD build targets

**Files:** Test Makefiles
- [ ] `test/Makefile.am`: Remove tst_zstandard references
- [ ] `test_h5/Makefile.am`: Remove ZStandard test references
- [ ] `ftest/Makefile.am`: Remove ftst_ccr_zstandard references

#### Step 1.5: Validation
- [ ] Build test: `./configure && make`
- [ ] Verify no ZStandard references remain: `grep -ri zstandard . --exclude-dir=.git`

### Phase 2: Remove Bit Grooming Components
**Duration:** 2-3 days | **Priority:** High

#### Step 2.1: Remove Bit Grooming HDF5 Plugin Directory
```bash
# Remove entire BITGROOM plugin directory
rm -rf hdf5_plugins/BITGROOM/
```

#### Step 2.2: Update Core Library Files
**File:** `src/ccr.c` (75 matches)
- [ ] Remove Bit Grooming algorithm constants
- [ ] Remove Bit Grooming compression/decompression functions
- [ ] Remove Bit Grooming initialization code
- [ ] Remove Bit Grooming error handling
- [ ] Update algorithm enumeration

**File:** `include/ccr.h`
- [ ] Remove Bit Grooming constants and macros
- [ ] Remove Bit Grooming function declarations
- [ ] Update algorithm enum definitions

**File:** `fsrc/ccr.F90` (18 matches)
- [ ] Remove Bit Grooming Fortran interface functions
- [ ] Remove Bit Grooming constants
- [ ] Update Fortran module exports

#### Step 2.3: Remove Bit Grooming Tests
```bash
# Remove test files
rm -f test/tst_bitgroom.c
rm -f test_h5/tst_h_bitgroom.c
rm -f ftest/ftst_ccr_bitgroom.F90
```

#### Step 2.4: Update Build Configuration
**File:** `configure.ac` (13 matches)
- [ ] Remove Bit Grooming dependency checks
- [ ] Remove Bit Grooming configuration options
- [ ] Remove Bit Grooming library linking

**File:** `hdf5_plugins/configure.ac`
- [ ] Remove BITGROOM subdirectory references
- [ ] Update plugin list

**File:** `hdf5_plugins/Makefile.am`
- [ ] Remove BITGROOM from SUBDIRS
- [ ] Remove BITGROOM build targets

#### Step 2.5: Validation
- [ ] Build test: `./configure && make`
- [ ] Verify no Bit Grooming references remain: `grep -ri bitgroom . --exclude-dir=.git`

### Phase 3: Remove GRANULARBR Components
**Duration:** 2-3 days | **Priority:** High

#### Step 3.1: Remove GRANULARBR HDF5 Plugin Directory
```bash
# Remove entire GRANULARBR plugin directory
rm -rf hdf5_plugins/GRANULARBR/
```

#### Step 3.2: Update Core Library Files
**File:** `src/ccr.c` (39 matches)
- [ ] Remove GRANULARBR algorithm constants
- [ ] Remove GRANULARBR compression/decompression functions
- [ ] Remove GRANULARBR initialization code
- [ ] Remove GRANULARBR error handling
- [ ] Update algorithm enumeration

**File:** `include/ccr.h`
- [ ] Remove GRANULARBR constants and macros
- [ ] Remove GRANULARBR function declarations
- [ ] Update algorithm enum definitions

**File:** `fsrc/ccr.F90` (13 matches)
- [ ] Remove GRANULARBR Fortran interface functions
- [ ] Remove GRANULARBR constants
- [ ] Update Fortran module exports

#### Step 3.3: Remove GRANULARBR Tests
```bash
# Remove test files
rm -f test/tst_granularbr.c
rm -f ftest/ftst_ccr_granularbr.F90
```

#### Step 3.4: Update Build Configuration
**File:** `configure.ac` (10 matches)
- [ ] Remove GRANULARBR dependency checks
- [ ] Remove GRANULARBR configuration options
- [ ] Remove GRANULARBR library linking

**File:** `hdf5_plugins/configure.ac`
- [ ] Remove GRANULARBR subdirectory references
- [ ] Update plugin list

**File:** `hdf5_plugins/Makefile.am`
- [ ] Remove GRANULARBR from SUBDIRS
- [ ] Remove GRANULARBR build targets

#### Step 3.5: Validation
- [ ] Build test: `./configure && make`
- [ ] Verify no GRANULARBR references remain: `grep -ri granularbr . --exclude-dir=.git`

### Phase 4: Remove BITROUND Components
**Duration:** 1-2 days | **Priority:** High

#### Step 4.1: Remove BITROUND HDF5 Plugin Directory
```bash
# Remove entire BITROUND plugin directory
rm -rf hdf5_plugins/BITROUND/
```

#### Step 4.2: Update Core Library Files
**File:** `src/ccr.c` (25 matches)
- [ ] Remove BITROUND algorithm constants
- [ ] Remove BITROUND compression/decompression functions
- [ ] Remove BITROUND initialization code
- [ ] Remove BITROUND error handling
- [ ] Update algorithm enumeration

**File:** `include/ccr.h`
- [ ] Remove BITROUND constants and macros
- [ ] Remove BITROUND function declarations
- [ ] Update algorithm enum definitions

**File:** `fsrc/ccr.F90` (6 matches)
- [ ] Remove BITROUND Fortran interface functions
- [ ] Remove BITROUND constants
- [ ] Update Fortran module exports

#### Step 4.3: Update Build Configuration
**File:** `configure.ac` (17 matches)
- [ ] Remove BITROUND dependency checks
- [ ] Remove BITROUND configuration options
- [ ] Remove BITROUND library linking

**File:** `hdf5_plugins/configure.ac`
- [ ] Remove BITROUND subdirectory references
- [ ] Update plugin list

**File:** `hdf5_plugins/Makefile.am`
- [ ] Remove BITROUND from SUBDIRS
- [ ] Remove BITROUND build targets

#### Step 4.4: Validation
- [ ] Build test: `./configure && make`
- [ ] Verify no BITROUND references remain: `grep -ri bitround . --exclude-dir=.git`

### Phase 5: Update Configuration and Metadata Files
**Duration:** 1 day | **Priority:** Medium

#### Step 5.1: Update Configuration Files
**File:** `ccr-config.in`
- [ ] Remove ZStandard/Bit Grooming/GRANULARBR/BITROUND flags
- [ ] Update supported algorithms list
- [ ] Update version information

**File:** `libccr.settings.in`
- [ ] Remove removed algorithm references
- [ ] Update algorithm list template

**File:** `include/ccr_meta.h.in`
- [ ] Update version metadata
- [ ] Remove algorithm-specific metadata

#### Step 5.2: Update Test Scripts
**File:** `test/run_tests.sh.in`
- [ ] Remove ZStandard test calls
- [ ] Remove Bit Grooming test calls
- [ ] Remove GRANULARBR test calls
- [ ] Remove BITROUND test calls

**File:** `test/run_benchmarks.sh.in`
- [ ] Remove removed algorithm benchmarks
- [ ] Update benchmark suite

**File:** `test/run_par_tests.sh.in`
- [ ] Remove parallel tests for removed algorithms

**File:** `ftest/run_tests.sh.in`
- [ ] Remove Fortran tests for removed algorithms

#### Step 5.3: Validation
- [ ] Build test: `./configure && make`
- [ ] Test script validation: `./test/run_tests.sh`

### Phase 6: Update CI/CD and Documentation
**Duration:** 1 day | **Priority:** Medium

#### Step 6.1: Update GitHub Actions Workflows
**Directory:** `.github/workflows/`

For each `.yml` file:
- [ ] `build_all.yml`: Update actions/cache@v2 → actions/cache@v4, remove removed algorithm tests
- [ ] `build_asan.yml`: Update actions/cache@v2 → actions/cache@v4, remove removed algorithm tests
- [ ] `build_par.yml`: Update actions/cache@v2 → actions/cache@v4, remove removed algorithm tests
- [ ] `build_par_2.yml`: Update actions/cache@v2 → actions/cache@v4, remove removed algorithm tests
- [ ] `lz4.yml`: Update actions/cache@v2 → actions/cache@v4
- [ ] `macos_all.yml`: Update actions/cache@v2 → actions/cache@v4, remove removed algorithm tests
- [ ] `minimal.yml`: Update actions/cache@v2 → actions/cache@v4, remove removed algorithm tests
- [ ] `minimal_2.yml`: Update actions/cache@v2 → actions/cache@v4, remove removed algorithm tests
- [ ] `multifilter.yml`: Update actions/cache@v2 → actions/cache@v4, remove removed algorithm tests
- [ ] `multifilter_asan.yml`: Update actions/cache@v2 → actions/cache@v4, remove removed algorithm tests

#### Step 6.2: Update Documentation
**File:** `README.md`
- [ ] Remove references to removed algorithms
- [ ] Update algorithm list to show only BZIP2 and LZ4
- [ ] Update examples and usage instructions

**File:** `ChangeLog`
- [ ] Add v2.0.0 entry with breaking change notice
- [ ] Document removed algorithms
- [ ] Note migration path

#### Step 6.3: Validation
- [ ] Documentation review
- [ ] CI workflow syntax validation

### Phase 7: Validation and Testing
**Duration:** 1-2 days | **Priority:** High

#### Step 7.1: Build System Validation
- [ ] Clean build: `make distclean && ./configure && make`
- [ ] Parallel build: `make -j$(nproc)`
- [ ] HDF5 plugin build verification
- [ ] Installation test: `make install DESTDIR=/tmp/ccr-test`

#### Step 7.2: Functional Testing
- [ ] Run remaining test suites: `make check`
- [ ] Test BZIP2 functionality extensively
- [ ] Test LZ4 functionality extensively
- [ ] Validate Fortran interface for remaining algorithms
- [ ] Test HDF5 plugin functionality for BZIP2 and LZ4
- [ ] Cross-platform testing (if available)

#### Step 7.3: Performance Validation
- [ ] Run benchmark tests: `./test/run_benchmarks.sh`
- [ ] Establish new performance baseline
- [ ] Verify no regressions in remaining algorithms
- [ ] Document performance characteristics

#### Step 7.4: Integration Testing
- [ ] Test with sample climate data
- [ ] Verify HDF5 filter chain functionality
- [ ] Test error handling for removed algorithms

### Phase 8: Final Documentation Updates
**Duration:** 0.5 days | **Priority:** Low

#### Step 8.1: Update Design Documentation
- [ ] Update `docs/design.md` with final architecture
- [ ] Document simplified algorithm set
- [ ] Update API documentation

#### Step 8.2: Create Migration Guide
- [ ] Create detailed migration guide for users
- [ ] Document breaking changes
- [ ] Provide code examples for BZIP2/LZ4 usage
- [ ] Update any remaining documentation

#### Step 8.3: Final Validation
- [ ] Complete documentation review
- [ ] Verify all references updated
- [ ] Final build and test run

## Risk Assessment

### High Risk Items
- **Major Breaking API Changes**: Removal of 4 algorithms will break existing code using these features
- **Build System Complexity**: CCR has complex autotools configuration that needs careful updating
- **HDF5 Plugin Dependencies**: Need to ensure plugin system still works correctly
- **Extensive Code Changes**: Large number of references across many files (200+ total matches)

### Mitigation Strategies
- Thorough testing at each phase before proceeding
- Maintain backup branches for rollback capability
- Test builds on multiple platforms
- Validate remaining functionality extensively

## Post-Sprint Checklist
- [ ] All tests passing
- [ ] Documentation updated
- [ ] CI/CD pipelines green
- [ ] Performance benchmarks documented
- [ ] Migration guide complete
- [ ] Ready for v2.0.0 release preparation

## Rollback Plan
If issues arise during implementation:
1. Return to backup branch: `git checkout backup-before-v2.0.0-sprint-1`
2. Analyze specific failure points
3. Create targeted fixes on feature branches
4. Resume implementation from stable checkpoint

## Success Criteria
- [ ] All ZStandard code and references removed
- [ ] All Bit Grooming code and references removed
- [ ] All GRANULARBR code and references removed
- [ ] All BITROUND code and references removed
- [ ] Build system works with only BZIP2 and LZ4
- [ ] All remaining tests pass (100% success rate)
- [ ] No performance regressions in BZIP2/LZ4
- [ ] Clean grep results for removed algorithms
- [ ] CI/CD pipelines updated and passing
- [ ] Documentation accurately reflects new state
- [ ] Migration guide created for users

## Dependencies
- No external dependencies for this sprint
- Requires coordination with any downstream users about breaking changes

## Notes
- This is a major breaking change that will require version bump to 2.0.0
- Consider deprecation warnings in a 1.x release before this sprint
- LZ4 enhancement (Sprint 2) should focus on optimization and validation
- Total estimated duration: 10-13 days
- This removes approximately 75% of the compression algorithms, significantly simplifying the codebase
