#!/usr/bin/env python
"""
This script will mark for culling the periodic rows/columns at the top, bottom,
left, and right of a mesh generated by periodic_hex.  It creates a new field
called cullCell and marks those locations with 1s.  This mesh can then be
run through mesh_conversion_tools/MpasCellCuller.x to remove those boundary
rows/columns, leaving a non-periodic planar mesh.

The logic to do this is to mark the max and min rows and the two max and min 
columns, so it is specific to a periodic_hex mesh.  If it were wished to 
make this tool general, it could calculate cell-to-cell distances between
all neighboring cells and mark for culling any cell pairs that have 
distances greater than, say, half of the range in x/y values of the entire mesh.
"""

from __future__ import print_function
import sys
import netCDF4
import numpy as np

from optparse import OptionParser


print("== running mark_periodic_boundaries_for_culling.py")
print("   Gathering information.  (Invoke with --help for more details. All arguments are optional)")
parser = OptionParser()
parser.description = "This script takes an MPAS grid file and marks the edge rows and columns for culling, e.g., to remove periodicity."
parser.add_option("-f", "--file", dest="inFile", help="MPAS grid file name used as input.", default="grid.nc", metavar="FILENAME")
parser.add_option("--keepx", dest="keepx", help="Retain periodicity in x-direction.", action='store_true', default=False)
parser.add_option("--keepy", dest="keepy", help="Retain periodicity in y-direction.", action='store_true', default=False)
parser.add_option("--remove_extra_y", dest="extray", help="Remove extra row in y-direction to create an odd number of rows (which is not allowed by periodic_hex.  The extra row is removed from the north side of the mesh.", action='store_true', default=False)
for option in parser.option_list:
    if option.default != ("NO", "DEFAULT"):
        option.help += (" " if option.help else "") + "[default: %default]"
options, args = parser.parse_args()

print('') # make a space in stdout before further output


# ===============================================

fin = netCDF4.Dataset(options.inFile, 'r+')

# Get info from input file
xCell = fin.variables['xCell'][:]
yCell = fin.variables['yCell'][:]
nCells = len(fin.dimensions['nCells'])

if 'cullCell' in fin.variables:
  cullCell = fin.variables['cullCell']
else:
  cullCell = fin.createVariable('cullCell', fin.variables['xCell'].dtype, ('nCells',))

cullCell_local = np.zeros( (nCells,) )

# For a periodic hex, the upper and lower rows need to be marked
if not options.keepy:
    print('y-periodic cells marked for culling')
    cullCell_local[np.nonzero(yCell == yCell.min())] = 1
    cullCell_local[np.nonzero(yCell == yCell.max())] = 1

if options.extray:
   if options.keepy:
      print("--remove_extra_y cannot be used with --keepy.  --remove_extra_y will be ignored.")
   else:
      print("An extra row along the north will be marked for culling to leave an odd number of row.")
      cullCell_local[np.nonzero(yCell == np.unique(yCell[:])[-2])] = 1  # second to north row

# For a periodic hex the leftmost and rightmost *TWO* columns need to be marked
if not options.keepx:
    print('x-periodic cells marked for culling')
    unique_Xs=np.array(sorted(list(set(xCell[:]))))
    cullCell_local[np.nonzero(xCell == unique_Xs[0])] = 1
    cullCell_local[np.nonzero(xCell == unique_Xs[1])] = 1
    cullCell_local[np.nonzero(xCell == unique_Xs[-1])] = 1
    cullCell_local[np.nonzero(xCell == unique_Xs[-2])] = 1

cullCell[:] = cullCell_local

fin.close()
print("Marking complete.  Don't forget to run MpasCellCuller!")
