"""
This script has functions that are needed to post-process and write state
output variables from ISMIP6 simulations.
The input files (i.e., MALI output files) need to have been
concatenated to have yearly data, which can be done using 'ncrcat' command
before using this script.
"""

from netCDF4 import Dataset
import xarray as xr
import numpy as np
from datetime import date
import shutil
import os, sys


def process_state_vars(inputfile_state, tmp_file):
    """
    inputfile_state: output file copy from MALI simulations
    tmp_file: temporary file name
    inputfile_temperature: output temperature file from MALI simulations
    """

    inputfile_state_vars = xr.open_dataset(inputfile_state, engine="netcdf4", decode_cf=False)
    del inputfile_state_vars.daysSinceStart.attrs['units'] # need this line to prevent xarray from reading daysSinceStart as a timedelta type and corrupting values after about 250 years

    # get the mesh description data
    nCells = inputfile_state_vars.dims['nCells']
    nTime = inputfile_state_vars.dims['Time']
    nLayer = inputfile_state_vars.dims['nVertLevels']
    nInterface = nLayer + 1  # inputfile_state_vars.dims['nVertInterfaces']
    cellMask = inputfile_state_vars['cellMask'][:, :]
    basalTemperature = inputfile_state_vars['basalTemperature'][:, :]
    betaSolve = inputfile_state_vars['betaSolve'][:, :]

    inputfile_state_vars['litempbotfl'] = basalTemperature * (cellMask[:, :] & 4) / 4
    inputfile_state_vars['litempbotgr'] = basalTemperature * (1 - (cellMask[:, :] & 4) / 4)

    uxsurf = inputfile_state_vars['uReconstructX'][:, :, 0]
    uysurf = inputfile_state_vars['uReconstructY'][:, :, 0]
    uxbase = inputfile_state_vars['uReconstructX'][:, :, nInterface - 1]
    uybase = inputfile_state_vars['uReconstructY'][:, :, nInterface - 1]
    inputfile_state_vars['uReconstructX_sfc'] = uxsurf
    inputfile_state_vars['uReconstructY_sfc'] = uysurf
    inputfile_state_vars['uReconstructX_base'] = uxbase
    inputfile_state_vars['uReconstructY_base'] = uybase

    inputfile_state_vars['upperSurface'] = np.maximum(0.0, inputfile_state_vars['upperSurface'])

    inputfile_state_vars['sftflf'] = (cellMask[:, :] & 4) / 4 * (cellMask[:, :] & 2) / 2  # floating and dynamic
    inputfile_state_vars['sftgrf'] = ((cellMask[:, :] * 0 + 1) - (cellMask[:, :] & 4) / 4) * (cellMask[:, :] & 2) / 2  # grounded: not-floating & dynamic
    inputfile_state_vars['sftgif'] = (cellMask[:, :] & 2) / 2  # grounded: dynamic ice
    inputfile_state_vars['strbasemag'] = betaSolve[:, :] * ((uxbase[:, :]) ** 2 + (uybase[:, :]) ** 2) **0.5 \
        * (3600.0 * 24.0 * 365.0) \
        * (cellMask[:, :] * 0 + 1 - (cellMask[:, :] & 4) / 4) * (cellMask[:, :] & 2) / 2

    inputfile_state_vars.to_netcdf(tmp_file)
    inputfile_state_vars.close()


def write_netcdf_2d_state_vars(mali_var_name, ismip6_var_name, var_std_name,
                               var_units, var_varname, remapped_mali_outputfile,
                               ismip6_grid_file, exp, output_path):
    """
    mali_var_name: variable name on MALI side
    ismip6_var_name: variable name required by ISMIP6
    var_std_name: standard variable name
    var_units: variable units
    var_varname: variable variable name
    remapped_mali_outputfile: mali state file remapped on the ISMIP6 grid
    ismip6_grid_file: original ISMIP6 file
    exp: experiment name
    output_path: output path to which the output files will be saved
    """

    data_ismip6 = Dataset(ismip6_grid_file, 'r')
    var_x = data_ismip6.variables['x'][:]
    var_y = data_ismip6.variables['y'][:]

    data = Dataset(remapped_mali_outputfile, 'r')
    data.set_auto_mask(False)
    simulationStartTime = data.variables['simulationStartTime'][:].tostring().decode('utf-8').strip().strip('\x00')
    simulationStartDate = simulationStartTime.split("_")[0]
    daysSinceStart = data.variables['daysSinceStart'][:]
    var_sftgif = data.variables['sftgif'][:, :, :]
    var_sftgrf = data.variables['sftgrf'][:, :, :]
    var_sftflf = data.variables['sftflf'][:, :, :]
    var_mali = data.variables[mali_var_name][:,:,:]
    var_mali[np.where(abs(var_mali + 1e34) < 1e33)] = np.NAN
    timeSteps, latN, lonN = np.shape(var_mali)

    dataOut = Dataset(f'{output_path}/{ismip6_var_name}_AIS_DOE_MALI_{exp}.nc',
                      'w', format='NETCDF4_CLASSIC')
    dataOut.createDimension('time', timeSteps)
    dataOut.createDimension('x', lonN)
    dataOut.createDimension('y', latN)
    dataValues = dataOut.createVariable(ismip6_var_name, 'd',
                                       ('time', 'y', 'x'), fill_value=np.NAN)
    xValues = dataOut.createVariable('x', 'd', ('x'))
    yValues = dataOut.createVariable('y', 'd', ('y'))
    timeValues = dataOut.createVariable('time', 'd', ('time'))
    timeValues[:] = daysSinceStart
    AUTHOR_STR = 'Matthew Hoffman, Trevor Hillebrand, Holly Kyeore Han'
    DATE_STR = date.today().strftime("%d-%b-%Y")

    for i in range(timeSteps):
        if ismip6_var_name == 'sftgif':
            dataValues[i, :, :] = var_mali[i, :, :]
        else:
            if ismip6_var_name == 'litempbotgr':
                mask = var_sftgrf[i, :, :]
            elif ismip6_var_name == 'litempbotfl':
                mask = var_sftflf[i, :, :]
            elif ismip6_var_name == 'topg':
                mask = np.ones(var_mali.shape[1:])  # don't mask topg
            else:
                mask = var_sftgif[i, :, :]
            tmp = var_mali[i, :, :]
            tmp[mask == 0] = np.NAN
            dataValues[i, :, :] = tmp

    for i in range(latN):
        xValues[i] = var_x[i]

    for i in range(lonN):
        yValues[i] = var_y[i]

    dataValues.standard_name = var_std_name
    dataValues.units = var_units
    timeValues.units = f'days since {simulationStartDate}'
    timeValues.calendar = 'noleap'
    timeValues.standard_name = 'time'
    timeValues.long_name = 'time'
    xValues.units = 'm'
    xValues.standard_name = 'x'
    xValues.long_name = 'x'
    yValues.units = 'm'
    yValues.standard_name = 'y'
    yValues.long_name = 'y'
    dataOut.AUTHORS = AUTHOR_STR
    dataOut.MODEL = 'MALI (MPAS-Albany Land Ice model)'
    dataOut.GROUP = 'Los Alamos National Laboratory, Department of Energy'
    dataOut.VARIABLE = var_varname
    dataOut.DATE = DATE_STR
    dataOut.close()


def generate_output_2d_state_vars(file_remapped_mali_state,
                                  ismip6_grid_file, exp, output_path):
    """
    file_remapped_mali_state: output files on mali mesh remapped
    on the ismip6 grid
    ismip6_grid_file: ismip6 original file
    exp: ISMIP6 experiment name
    output_path: path to which the final output files are saved
    """


    # ----------- lithk ------------------
    write_netcdf_2d_state_vars('thickness','lithk', 'land_ice_thickness',
                               'm', 'Ice thickness',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- orog ------------------
    write_netcdf_2d_state_vars('upperSurface','orog', 'surface_altitude', 'm',
                               'Surface elevation',
                               file_remapped_mali_state,
                               ismip6_grid_file,exp, output_path)

    # ----------- base ------------------
    write_netcdf_2d_state_vars('lowerSurface','base', 'base_altitude', 'm',
                               'Base elevation',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- topg ------------------
    write_netcdf_2d_state_vars('bedTopography','topg', 'bedrock_altitude', 'm',
                               'Bedrock elevation',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- hfgeoubed------------------
    # Note: even though this is a flux variable, we are taking a snapshot of it
    # as it does not change with time #Uncomment the function all once basalHeatFlux is outputted in the output stream
    # write_netcdf_2d_state_vars('basalHeatFlux', 'hfgeoubed',
    #                            'upward_geothermal_heat_flux_in_land_ice',
    #                            'W m-2', 'Geothermal heat flux',
    #                            file_remapped_mali_state, ismip6_grid_file,
    #                            exp, output_path)

    # ----------- xvelsurf ------------------
    write_netcdf_2d_state_vars('uReconstructX_sfc', 'xvelsurf',
                               'land_ice_surface_x_velocity',
                               'm s-1', 'Surface velocity in x',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # -----------yxvelsurf ------------------
    write_netcdf_2d_state_vars('uReconstructY_sfc', 'yvelsurf',
                               'land_ice_surface_y_velocity',
                               'm s-1', 'Surface velocity in x',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- xvelbase ------------------
    write_netcdf_2d_state_vars('uReconstructX_base', 'xvelbase',
                               'land_ice_basal_x_velocity',
                               'm s-1', 'Basal velocity in x',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- yvelbase ------------------
    write_netcdf_2d_state_vars('uReconstructY_base', 'yvelbase',
                               'land_ice_basal_y_velocity',
                               'm s-1', 'Basal velocity in y',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- zvelsurf & zvelbase ------------------
        # ISMIP6 requires these variables, but MALI does not output them.
        # So, we are not processing/writing these variables out.

    # ----------- xvelmean ------------------
    write_netcdf_2d_state_vars('xvelmean', 'xvelmean',
                               'land_ice_vertical_mean_x_velocity',
                               'm s-1', 'Mean velocity in x',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- yvelmean ------------------
    write_netcdf_2d_state_vars('yvelmean', 'yvelmean',
                               'land_ice_vertical_mean_y_velocity',
                               'm s-1', 'Mean velocity in y',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- litemptop ------------------
    write_netcdf_2d_state_vars('surfaceTemperature', 'litemptop',
                               'temperature_at_top_of_ice_sheet_model', 'K',
                               'Surface temperature',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- litempbotgr ------------------
    write_netcdf_2d_state_vars('litempbotgr', 'litempbotgr',
                               'temperature_at_base_of_ice_sheet_model', 'K',
                               'Basal temperature beneath grounded ice sheet',
                               file_remapped_mali_state,
                               ismip6_grid_file,exp, output_path)

    # ----------- litempbotfl ------------------
    write_netcdf_2d_state_vars('litempbotfl', 'litempbotfl',
                               'temperature_at_base_of_ice_sheet_model', 'K',
                               'Basal temperature beneath floating ice shelf',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- strbasemag ------------------
    write_netcdf_2d_state_vars('strbasemag', 'strbasemag',
                               'land_ice_basal_drag ', 'Pa',
                               'Basal drag',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- sftgif ------------------
    write_netcdf_2d_state_vars('sftgif','sftgif',
                               'land_ice_area_fraction', '1',
                               'Land ice area fraction',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- sftgrf ------------------
    write_netcdf_2d_state_vars('sftgrf', 'sftgrf',
                               'grounded_ice_sheet_area_fraction', '1',
                               'Grounded ice sheet area fraction',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)

    # ----------- sftflf ------------------
    write_netcdf_2d_state_vars('sftflf','sftflf',
                               'floating_ice_shelf_area_fraction', '1',
                               'Floating ice shelf area fraction',
                               file_remapped_mali_state,
                               ismip6_grid_file, exp, output_path)


def generate_output_1d_vars(global_stats_file, exp, output_path=None):
    """
    This code processes both state and flux 1D variables
    global_stats_file: MALI globalStats.nc output file
    exp: ISMIP6 experiment number
    output_path:
    """

    if not os.path.exists(output_path):
        output_path = os.getcwd()

    AUTHOR_STR = 'Matthew Hoffman, Trevor Hillebrand, Holly Kyeore Han'
    DATE_STR = date.today().strftime("%d-%b-%Y")

    data = Dataset(global_stats_file, 'r')
    nt_in = len(data.dimensions['Time'])
    xtime = data.variables['xtime'][:, :]
    daysSinceStart = data.variables['daysSinceStart'][:]
    dt = data.variables['deltat'][:]
    simulationStartTime = data.variables['simulationStartTime'][:].tostring().decode('utf-8').strip().strip('\x00')
    simulationStartDate = simulationStartTime.split("_")[0]
    if simulationStartDate[5:10] != '01-01':
        sys.exit("Error: simulationStartTime for globalStats file is not on Jan. 1.")
    refYear = int(simulationStartDate[0:4])
    decYears = refYear + daysSinceStart/365.0
    endYr = decYears[-1]
    if endYr != np.round(endYr):
        sys.exit("Error: end year not an even year in globalStats file.")

    # Determine processed time levels for state and flux fields
    # The historical state fields should include the initial time (Jan. 1).
    # Projection state fields should not include the initial time (Jan. 1)
    # of the projection because it's a restart from the historical.
    # Flux fields should never use the Jan. 1 time level at the start of the
    # year as part of the averaging.
    # For year conventions here, for state fields, the year is the snapshot at
    # the start of the year, e.g., state year 2000 means the snapshot at Jan. 1, 2000.
    # For flux fields, the years is the calendar year being averaged over,
    # e.g., flux year 2000 is the average between Jan. 1, 2000, and Jan. 1, 2001.
    # Note this year convention differs from the first column in table in A2.3.2 at
    # https://www.climate-cryosphere.org/wiki/index.php?title=ISMIP6-Projections2300-Antarctica#A2.3.3_Table_A1:_Variable_request_for_ISMIP6
    # but that year indexing convention ultimately doesn't matter because the
    # time coordinates in these files uses units of days since a reference date,
    # and it does not use a year indexing convention at all.
    if decYears[0] == np.round(decYears[0]):
        # The initial time level will only be on an even year (Jan. 1)
        # for the hist run.  In that case, we want to include that initial
        # even year in the state processing.  We also want the state snapshot
        # at the final (even) year in the output.
        # The flux processing should start with the first year, which covers a
        # full 12 months.  We exclude the final year, which is just a Jan. 1 posting.
        years_state = np.arange(decYears[0], endYr + 1)
        years_flux = np.arange(decYears[0], endYr)
    else:
        # For projection runs, the first state snapshot we want is the first Jan. 1,
        # which we be the first even year after the initial time in the file.
        # For flux files, the first full year we want to process is the year of the
        # first time level in the file.  As with hist, we exclude the final year,
        # which is just a Jan. 1 posting.
        years_state = np.arange(np.ceil(decYears[0]), endYr + 1)
        years_flux = np.arange(np.floor(decYears[0]), endYr)
    nt_state = len(years_state)
    nt_flux = len(years_flux)
    print(f'For state processing, using start year={years_state[0]} and end year={years_state[-1]}.')
    print(f'For flux  processing, using start year={years_flux[0]} and end year={years_flux[-1]}.')

    # read in state variables
    vol = data.variables['totalIceVolume'][:]
    vaf = data.variables['volumeAboveFloatation'][:]
    gia = data.variables['groundedIceArea'][:]
    fia = data.variables['floatingIceArea'][:]

    # read in flux variables over which yearly average will be taken
    smb = data.variables['totalSfcMassBal'][:]
    bmb = data.variables['totalBasalMassBal'][:]
    # clean out some garbage values we can't account for
    ind = np.nonzero(bmb>1.0e18)[0]
    if len(ind) > 0:
        print(f"WARNING: Found {len(ind)} values of totalBasalMassBal>1.0e18")
        bmb[ind] = np.nan
    ind = np.nonzero(bmb<-1.0e18)[0]
    if len(ind) > 0:
        print(f"WARNING: Found {len(ind)} values of totalBasalMassBal<-1.0e18")
        bmb[ind] = np.nan
    bmbFlt = data.variables['totalFloatingBasalMassBal'][:]
    # clean out some garbage values we can't account for
    ind = np.nonzero(bmbFlt>1.0e18)[0]
    if len(ind) > 0:
        print(f"WARNING: Found {len(ind)} values of totalFloatingBasalMassBal>1.0e18")
        bmbFlt[ind] = np.nan
    ind = np.nonzero(bmbFlt<-1.0e18)[0]
    if len(ind) > 0:
        print(f"WARNING: Found {len(ind)} values of totalFloatingBasalMassBal<-1.0e18")
        bmbFlt[ind] = np.nan
    cfx = data.variables['totalCalvingFlux'][:]
    fmfx = data.variables['totalFaceMeltingFlux'][:]
    gfx = data.variables['groundingLineFlux'][:]

    # initialize 1D variables that will store data value on the
    # January 1st of each year
    vol_snapshot = np.zeros(nt_state) * np.nan
    vaf_snapshot = np.zeros(nt_state) * np.nan
    gia_snapshot = np.zeros(nt_state) * np.nan
    fia_snapshot = np.zeros(nt_state) * np.nan
    days_snapshot = np.zeros(nt_state) * np.nan
    smb_avg = np.zeros(nt_flux) * np.nan
    bmb_avg = np.zeros(nt_flux) * np.nan
    bmbFlt_avg = np.zeros(nt_flux) * np.nan
    cfx_avg = np.zeros(nt_flux) * np.nan
    cfmfx_avg = np.zeros(nt_flux) * np.nan
    gfx_avg = np.zeros(nt_flux) * np.nan
    days_min = np.zeros(nt_flux) * np.nan
    days_max = np.zeros(nt_flux) * np.nan

    # this is for the state variables
    for i in range(nt_state):
        ind_snap = np.where(decYears==years_state[i])[0]

        vol_snapshot[i] = vol[ind_snap]
        vaf_snapshot[i] = vaf[ind_snap]
        gia_snapshot[i] = gia[ind_snap]
        fia_snapshot[i] = fia[ind_snap]
        days_snapshot[i] = daysSinceStart[ind_snap]

        if decYears[ind_snap] == endYr:
            break

    # this is for the flux variables
    for i in range(nt_flux):
        ind_avg = np.where(np.logical_and(decYears > years_flux[i],
                                          decYears <= (years_flux[i] + 1.0)))[0]
        smbi = smb[ind_avg]
        bmbi = bmb[ind_avg]
        bmbFlti = bmbFlt[ind_avg]
        cfxi = cfx[ind_avg]
        cfmfxi = cfx[ind_avg] + fmfx[ind_avg]
        gfxi = gfx[ind_avg]
        dti  = dt[ind_avg]

        # take the average of the flux variables
        smb_avg[i] = np.nansum(smbi * dti) / np.nansum(dti)
        bmb_avg[i] = np.nansum(bmbi * dti) / np.nansum(dti)
        bmbFlt_avg[i] = np.nansum(bmbFlti * dti) / np.nansum(dti)
        cfx_avg[i] = np.nansum(cfxi * dti) / np.nansum(dti)
        cfmfx_avg[i] = np.nansum(cfmfxi * dti) / np.nansum(dti)
        gfx_avg[i] = np.nansum(gfxi * dti) / np.nansum(dti)
        days_min[i] = (years_flux[i] - refYear) * 365.0
        days_max[i] = (years_flux[i] + 1.0 - refYear) * 365.0

        if decYears[ind_avg][-1] == endYr:
            break

    # -------------- lim ------------------
    data_scalar = Dataset(f'{output_path}/lim_AIS_DOE_MALI_{exp}.nc', 'w', format='NETCDF4_CLASSIC')
    data_scalar.createDimension('time', nt_state)
    limValues = data_scalar.createVariable('lim', 'd', ('time'))
    timeValues = data_scalar.createVariable('time', 'd', ('time'))
    for i in range(nt_state):
        limValues[i] = vol_snapshot[i] * 910
        timeValues[i] = days_snapshot[i]
    timeValues.units = f'days since {simulationStartDate}'
    timeValues.calendar = 'noleap'
    timeValues.standard_name = 'time'
    timeValues.long_name = 'time'
    limValues.standard_name = 'land_ice_mass'
    limValues.units = 'kg'
    data_scalar.AUTHORS = AUTHOR_STR
    data_scalar.MODEL = 'MALI (MPAS-Albany Land Ice model)'
    data_scalar.GROUP= 'Los Alamos National Laboratory'
    data_scalar.VARIABLE = 'Total ice mass'
    data_scalar.DATE = DATE_STR
    data_scalar.close()

    # -------------- limnsw ------------------
    data_scalar = Dataset(f'{output_path}/limnsw_AIS_DOE_MALI_{exp}.nc', 'w', format='NETCDF4_CLASSIC')
    data_scalar.createDimension('time', nt_state)
    limnswValues = data_scalar.createVariable('limnsw', 'd', ('time'))
    timeValues = data_scalar.createVariable('time', 'd', ('time'))
    for i in range(nt_state):
        limnswValues[i] = vaf_snapshot[i] * 910
        timeValues[i] = days_snapshot[i]
    timeValues.units = f'days since {simulationStartDate}'
    timeValues.calendar = 'noleap'
    timeValues.standard_name = 'time'
    timeValues.long_name = 'time'
    limnswValues.standard_name = 'land_ice_mass_not_displacing_sea_water'
    limnswValues.units = 'kg'
    data_scalar.AUTHORS = AUTHOR_STR
    data_scalar.MODEL = 'MALI (MPAS-Albany Land Ice model)'
    data_scalar.GROUP = 'Los Alamos National Laboratory'
    data_scalar.VARIABLE = 'Mass above floatation'
    data_scalar.DATE = DATE_STR
    data_scalar.close()

    # -------------- iareagr ------------------
    data_scalar = Dataset(f'{output_path}/iareagr_AIS_DOE_MALI_{exp}.nc', 'w', format='NETCDF4_CLASSIC')
    data_scalar.createDimension('time', nt_state)
    iareagrValues = data_scalar.createVariable('iareagr', 'd', ('time'))
    timeValues = data_scalar.createVariable('time', 'd', ('time'))
    for i in range(nt_state):
        iareagrValues[i] = gia_snapshot[i]
        timeValues[i] = days_snapshot[i]
    timeValues.units = f'days since {simulationStartDate}'
    timeValues.calendar = 'noleap'
    timeValues.standard_name = 'time'
    timeValues.long_name = 'time'
    iareagrValues.standard_name = 'grounded_ice_sheet_area'
    iareagrValues.units = 'm2'
    data_scalar.AUTHORS= AUTHOR_STR
    data_scalar.MODEL= 'MALI (MPAS-Albany Land Ice model)'
    data_scalar.GROUP = 'Los Alamos National Laboratory'
    data_scalar.VARIABLE = 'Grounded ice area'
    data_scalar.DATE = DATE_STR
    data_scalar.close()

    # -------------- iareafl ------------------
    data_scalar = Dataset(f'{output_path}/iareafl_AIS_DOE_MALI_{exp}.nc', 'w', format='NETCDF4_CLASSIC')
    data_scalar.createDimension('time', nt_state)
    iareaflValues = data_scalar.createVariable('iareafl', 'd', ('time'))
    timeValues = data_scalar.createVariable('time', 'd', ('time'))
    for i in range(nt_state):
        iareaflValues[i] = fia_snapshot[i]
        timeValues[i] = days_snapshot[i]
    timeValues.units = f'days since {simulationStartDate}'
    timeValues.calendar = 'noleap'
    timeValues.standard_name = 'time'
    timeValues.long_name = 'time'
    iareaflValues.standard_name = 'floating_ice_shelf_area'
    iareaflValues.units = 'm2'
    data_scalar.AUTHORS= AUTHOR_STR
    data_scalar.MODEL= 'MALI (MPAS-Albany Land Ice model)'
    data_scalar.GROUP = 'Los Alamos National Laboratory'
    data_scalar.VARIABLE = 'Floating ice area'
    data_scalar.DATE = DATE_STR
    data_scalar.close()

    # -------------- tendacabf: this is a flux var
    data_scalar = Dataset(f'{output_path}/tendacabf_AIS_DOE_MALI_{exp}.nc', 'w', format='NETCDF4_CLASSIC')
    data_scalar.createDimension('time', nt_flux)
    tendacabfValues = data_scalar.createVariable('tendacabf', 'd', ('time'))
    timeValues = data_scalar.createVariable('time', 'd', ('time'))
    data_scalar.createDimension('bnds', 2)
    timebndsValues = data_scalar.createVariable('time_bnds', 'd', ('time', 'bnds'))
    for i in range(nt_flux):
        tendacabfValues[i] = smb_avg[i] / 31536000.0
        timeValues[i] = (days_min[i] + days_max[i]) / 2.0
        timebndsValues[i, 0] = days_min[i]
        timebndsValues[i, 1] = days_max[i]
    timeValues.units = f'days since {simulationStartDate}'
    timeValues.calendar = 'noleap'
    timeValues.standard_name = 'time'
    timeValues.long_name = 'time'
    tendacabfValues.standard_name = 'tendency_of_land_ice_mass_due_to_surface_mass_balance'
    tendacabfValues.units = 'kg s-1'
    data_scalar.AUTHORS= AUTHOR_STR
    data_scalar.MODEL= 'MALI (MPAS-Albany Land Ice model)'
    data_scalar.GROUP = 'Los Alamos National Laboratory'
    data_scalar.VARIABLE = 'Total SMB flux'
    data_scalar.DATE = DATE_STR
    data_scalar.close()

    # -------------- tendlibmassbf: this is a flux var
    data_scalar = Dataset(f'{output_path}/tendlibmassbf_AIS_DOE_MALI_{exp}.nc', 'w', format='NETCDF4_CLASSIC')
    data_scalar.createDimension('time', nt_flux)
    tendlibmassbfValues = data_scalar.createVariable('tendlibmassbf', 'd', ('time'))
    timeValues = data_scalar.createVariable('time', 'd', ('time'))
    data_scalar.createDimension('bnds', 2)
    timebndsValues = data_scalar.createVariable('time_bnds', 'd', ('time', 'bnds'))
    for i in range(nt_flux):
        tendlibmassbfValues[i] = bmb_avg[i] / 31536000
        timeValues[i] = (days_min[i] + days_max[i]) / 2.0
        timebndsValues[i, 0] = days_min[i]
        timebndsValues[i, 1] = days_max[i]
    timeValues.units = f'days since {simulationStartDate}'
    timeValues.calendar = 'noleap'
    timeValues.standard_name = 'time'
    timeValues.long_name = 'time'
    tendlibmassbfValues.standard_name = 'tendency_of_land_ice_mass_due_to_basal_mass_balance '
    tendlibmassbfValues.units = 'kg s-1'
    data_scalar.AUTHORS= AUTHOR_STR
    data_scalar.MODEL= 'MALI (MPAS-Albany Land Ice model)'
    data_scalar.GROUP = 'Los Alamos National Laboratory'
    data_scalar.VARIABLE = 'Total BMB flux'
    data_scalar.DATE = DATE_STR
    data_scalar.close()

    # -------------- tendlibmassbffl: this is a flux var
    data_scalar = Dataset(f'{output_path}/tendlibmassbffl_AIS_DOE_MALI_{exp}.nc', 'w',
                          format='NETCDF4_CLASSIC')
    data_scalar.createDimension('time', nt_flux)
    tendlibmassbfflValues = data_scalar.createVariable('tendlibmassbffl', 'd', ('time'))
    timeValues = data_scalar.createVariable('time', 'd', ('time'))
    data_scalar.createDimension('bnds', 2)
    timebndsValues = data_scalar.createVariable('time_bnds', 'd', ('time', 'bnds'))
    for i in range(nt_flux):
        tendlibmassbfflValues[i] = bmbFlt_avg[i] / 31536000
        timeValues[i] = (days_min[i] + days_max[i]) / 2.0
        timebndsValues[i, 0] = days_min[i]
        timebndsValues[i, 1] = days_max[i]
    timeValues.units = f'days since {simulationStartDate}'
    timeValues.calendar = 'noleap'
    timeValues.standard_name = 'time'
    timeValues.long_name = 'time'
    tendlibmassbfflValues.standard_name = 'tendency_of_land_ice_mass_due_to_basal_mass_balance'
    tendlibmassbfflValues.units = 'kg s-1'
    data_scalar.AUTHORS= AUTHOR_STR
    data_scalar.MODEL= 'MALI (MPAS-Albany Land Ice model)'
    data_scalar.GROUP = 'Los Alamos National Laboratory'
    data_scalar.VARIABLE = 'Total BMB flux beneath floating ice'
    data_scalar.DATE = DATE_STR
    data_scalar.close()

    # -------------- tendlicalvf: this is a flux var
    data_scalar = Dataset(f'{output_path}/tendlicalvf_AIS_DOE_MALI_{exp}.nc', 'w', format='NETCDF4_CLASSIC')
    data_scalar.createDimension('time', nt_flux)
    tendlicalvfValues = data_scalar.createVariable('tendlicalvf', 'd', ('time'))
    timeValues = data_scalar.createVariable('time', 'd', ('time'))
    data_scalar.createDimension('bnds', 2)
    timebndsValues = data_scalar.createVariable('time_bnds', 'd', ('time', 'bnds'))
    for i in range(nt_flux):
        tendlicalvfValues[i] = -cfx_avg[i] / 31536000
        timeValues[i] = (days_min[i] + days_max[i]) / 2.0
        timebndsValues[i, 0] = days_min[i]
        timebndsValues[i, 1] = days_max[i]
    timeValues.units = f'days since {simulationStartDate}'
    timeValues.calendar = 'noleap'
    timeValues.standard_name = 'time'
    timeValues.long_name = 'time'
    tendlicalvfValues.standard_name = 'tendency_of_land_ice_mass_due_to_calving'
    tendlicalvfValues.units = 'kg s-1'
    data_scalar.AUTHORS= AUTHOR_STR
    data_scalar.MODEL= 'MALI (MPAS-Albany Land Ice model)'
    data_scalar.GROUP = 'Los Alamos National Laboratory'
    data_scalar.VARIABLE = 'Total calving flux'
    data_scalar.DATE = DATE_STR
    data_scalar.close()

    # -------------- tendlifmassbf: this is a flux var
    data_scalar = Dataset(f'{output_path}/tendlifmassbf_AIS_DOE_MALI_{exp}.nc', 'w', format='NETCDF4_CLASSIC')
    data_scalar.createDimension('time', nt_flux)
    tendlicalvfValues = data_scalar.createVariable('tendlifmassbf', 'd', ('time'))
    timeValues = data_scalar.createVariable('time', 'd', ('time'))
    data_scalar.createDimension('bnds', 2)
    timebndsValues = data_scalar.createVariable('time_bnds', 'd', ('time', 'bnds'))
    for i in range(nt_flux):
        tendlicalvfValues[i] = -cfmfx_avg[i] / 31536000
        timeValues[i] = (days_min[i] + days_max[i]) / 2.0
        timebndsValues[i, 0] = days_min[i]
        timebndsValues[i, 1] = days_max[i]
    timeValues.units = f'days since {simulationStartDate}'
    timeValues.calendar = 'noleap'
    timeValues.standard_name = 'time'
    timeValues.long_name = 'time'
    tendlicalvfValues.standard_name = 'tendency_of_land_ice_mass_due_to_calving_and_ice_front_melting'
    tendlicalvfValues.units = 'kg s-1'
    data_scalar.AUTHORS= AUTHOR_STR
    data_scalar.MODEL= 'MALI (MPAS-Albany Land Ice model)'
    data_scalar.GROUP = 'Los Alamos National Laboratory'
    data_scalar.VARIABLE = 'Total calving and ice front melting flux'
    data_scalar.DATE = DATE_STR
    data_scalar.close()

    # -------------- tendligroundf: this is a flux var
    data_scalar = Dataset(f'{output_path}/tendligroundf_AIS_DOE_MALI_{exp}.nc', 'w', format='NETCDF4_CLASSIC')
    data_scalar.createDimension('time', nt_flux)
    tendligroundfValues = data_scalar.createVariable('tendligroundf', 'd', ('time'))
    timeValues = data_scalar.createVariable('time', 'd', ('time'))
    data_scalar.createDimension('bnds', 2)
    timebndsValues = data_scalar.createVariable('time_bnds', 'd', ('time', 'bnds'))
    for i in range(nt_flux):
        tendligroundfValues[i] = gfx_avg[i] / 31536000
        timeValues[i] = (days_min[i] + days_max[i]) / 2.0
        timebndsValues[i, 0] = days_min[i]
        timebndsValues[i, 1] = days_max[i]
    timeValues.units = f'days since {simulationStartDate}'
    timeValues.calendar = 'noleap'
    timeValues.standard_name = 'time'
    timeValues.long_name = 'time'
    tendligroundfValues.standard_name = 'tendency_of_grounded_ice_mass'
    tendligroundfValues.units = 'kg s-1'
    data_scalar.AUTHORS= AUTHOR_STR
    data_scalar.MODEL= 'MALI (MPAS-Albany Land Ice model)'
    data_scalar.GROUP = 'Los Alamos National Laboratory'
    data_scalar.VARIABLE = 'Total grounding line flux'
    data_scalar.DATE = DATE_STR
    data_scalar.close()

    data.close()
