      SUBROUTINE SFLX (
     I ICE,DT,Z,NSOIL,SLDPTH,
     I LWDN,SOLDN,SFCPRS,PRCP,SFCTMP,TH2,Q2,SFCSPD,Q2SAT,DQSDT2,
     I VEGTYP,SOILTYP,SLOPETYP,
     I SHDFAC,PTU,TBOT,ALB,SNOALB,
     2 CMC,T1,STC,SMC,SH2O,SNOWH,SNEQV,ALBEDO,CH,CM,
     O ETP,ETA,H,S,RUNOFF1,RUNOFF2,Q1,SNMAX,
     O SOILW,SOILM, SMCWLT,SMCDRY,SMCREF,SMCMAX )
C
      IMPLICIT NONE
CC
C ----------------------------------------------------------------------
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC PURPOSE:  SUB-DRIVER FOR "NOAH/OSU LSM" FAMILY OF PHYSICS SUBROUTINES
CC           FOR A SOIL/VEG/SNOWPACK LAND-SURFACE MODEL TO UPDATE SOIL 
CC           MOISTURE, SOIL ICE, SOIL TEMPERATURE, SKIN TEMPERATURE, 
CC           SNOWPACK WATER CONTENT, SNOWDEPTH, AND ALL TERMS
CC           OF THE SURFACE ENERGY BALANCE AND SURFACE WATER
CC           BALANCE (EXCLUDING INPUT ATMOSPHERIC FORCINGS OF 
CC           DOWNWARD RADIATION AND PRECIP)
CC
CC  VERSION 2.2 07 FEBRUARY 2001
CC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC
C ----------------------------------------------------------------------
C ------------    FROZEN GROUND VERSION     ----------------------------
C     ADDED STATES: SH2O(NSOIL) - UNFROZEN SOIL MOISTURE
C                   SNOWH       - SNOW DEPTH
C
C ----------------------------------------------------------------------
C
C NOTE ON SNOW STATE VARIABLES:
C   SNOWH = actual physical snow depth in m
C   SNEQV = liquid water-equivalent snow depth in m
C            (time-dependent snow density is obtained from SNEQV/SNOWH)
C
C NOTE ON ALBEDO FRACTIONS:
C   Input:
C     ALB    = BASELINE SNOW-FREE ALBEDO, FOR JULIAN DAY OF YEAR 
C   	       (USUALLY FROM TEMPORAL INTERPOLATION OF MONTHLY MEAN VALUES)
C   	       (CALLING PROG MAY OR MAY NOT INCLUDE DIURNAL SUN ANGLE EFFECT)
C     SNOALB = UPPER BOUND ON MAXIMUM ALBEDO OVER DEEP SNOW
C   	       (E.G. FROM ROBINSON AND KUKLA, 1985, J. CLIM. & APPL. METEOR.)
C   Output:
C     ALBEDO = COMPUTED ALBEDO WITH SNOWCOVER EFFECTS 
C   	      (COMPUTED USING ALB, SNOALB, SNEQV, AND SHDFAC->green veg frac)
C
C   		 ARGUMENT LIST IN THE CALL TO SFLX
C
C ----------------------------------------------------------------------
C 1. CALLING STATEMENT
C
C     SUBROUTINE SFLX
C    I (ICE,DT,Z,NSOIL,SLDPTH,
C    I LWDN,SOLDN,SFCPRS,PRCP,SFCTMP,TH2,Q2,Q2SAT,DQSDT2,
C    I VEGTYP,SOILTYP,SLOPETYP,
C    I SHDFAC,PTU,TBOT,ALB,SNOALB,
C    I SFCSPD,
C    2 CMC,T1,STC,SMC,SH2O,SNOWH,SNEQV,CH,CM,
C    O ETP,ETA,H,S,RUNOFF1,RUNOFF2,Q1,SNMAX,ALBEDO,
C    O SOILW,SOILM,SMCWLT,SMCDRY,SMCREF,SMCMAX)
C
C 2. INPUT (denoted by "I" in column six of argument list at top of routine)
C                  ### GENERAL PARAMETERS ###
C
C          ICE: SEA-ICE FLAG  (=1: SEA-ICE, =0: LAND)
C           DT: TIMESTEP (SEC)
C               (DT SHOULD NOT EXCEED 3600 SECS, RECOMMEND 1800 SECS OR LESS)
C            Z: HEIGHT (M) ABOVE GROUND OF ATMOSPHERIC FORCING VARIABLES
C        NSOIL: NUMBER OF SOIL LAYERS  
C              (at least 2, and not greater than parameter NSOLD set below)
C       SLDPTH: THE THICKNESS OF EACH SOIL LAYER (M) 
C
C                  ### ATMOSPHERIC VARIABLES ###
C
C         LWDN: LW DOWNWARD RADIATION (W M-2; POSITIVE, not net longwave)
C        SOLDN: SOLAR DOWNWARD RADIATION (W M-2; POSITIVE, not net shortwave)
C       SFCPRS: PRESSURE AT HEIGHT Z ABOVE GROUND (PASCALS)
C         PRCP: PRECIP RATE (KG M-2 S-1) (note, this is a rate)
C       SFCTMP: AIR TEMPERATURE (K) AT HEIGHT Z ABOVE GROUND 
C          TH2: AIR POTENTIAL TEMPERATURE (K) AT HEIGHT Z ABOVE GROUND 
C           Q2: MIXING RATIO AT HEIGHT Z ABOVE GROUND (KG KG-1)
C       SFCSPD: WIND SPEED (M S-1) AT HEIGHT Z ABOVE GROUND
C        Q2SAT: SAT MIXING RATIO AT HEIGHT Z ABOVE GROUND (KG KG-1)
C       DQSDT2: SLOPE OF SAT SPECIFIC HUMIDITY CURVE AT T=SFCTMP (KG KG-1 K-1)
C
C                  ### CANOPY/SOIL CHARACTERISTICS ###
C
C       VEGTYP: VEGETATION TYPE (INTEGER INDEX)
C       SOILTYP: SOIL TYPE (INTEGER INDEX)
C     SLOPETYP: CLASS OF SFC SLOPE (INTEGER INDEX)
C       SHDFAC: AREAL FRACTIONAL COVERAGE OF GREEN VEGETATION (range 0.0-1.0)
C          PTU: PHOTO THERMAL UNIT (PLANT PHENOLOGY FOR ANNUALS/CROPS)
C              (not yet used, but passed to REDPRM for future use in veg parms)
C         TBOT: BOTTOM SOIL TEMPERATURE (LOCAL YEARLY-MEAN SFC AIR TEMPERATURE)
C          ALB: BACKROUND SNOW-FREE SURFACE ALBEDO (FRACTION)
C       SNOALB: ALBEDO UPPER BOUND OVER DEEP SNOW (FRACTION)
C
C 3. STATE VARIABLES: BOTH INPUT AND OUTPUT
C			 (NOTE: OUTPUT USUALLY MODIFIED FROM INPUT BY PHYSICS)
C
C      (denoted by "2" in column six of argument list at top of routine)
C
C       !!! ########### STATE VARIABLES ##############  !!!
C
C         CMC: CANOPY MOISTURE CONTENT (M)
C          T1: GROUND/CANOPY/SNOWPACK) EFFECTIVE SKIN TEMPERATURE (K)
C
C  STC(NSOIL): SOIL TEMP (K)
C  SMC(NSOIL): TOTAL SOIL MOISTURE CONTENT (VOLUMETRIC FRACTION)
C SH2O(NSOIL): UNFROZEN SOIL MOISTURE CONTENT (VOLUMETRIC FRACTION)
C               NOTE: FROZEN SOIL MOISTURE = SMC - SH2O
C
C       SNOWH: SNOW DEPTH (M)
C       SNEQV: WATER-EQUIVALENT SNOW DEPTH (M)
C               NOTE: SNOW DENSITY = SNEQV/SNOWH
C      ALBEDO: SURFACE ALBEDO INCLUDING SNOW EFFECT (UNITLESS FRACTION)
C          CH: SFC EXCH COEF FOR HEAT AND MOISTURE (M S-1)
C          CM: SFC EXCH COEF FOR MOMENTUM (M S-1)
C              NOTE: CH AND CM ARE TECHNICALLY CONDUCTANCES SINCE THEY
C              HAVE BEEN MULTIPLIED BY THE WIND SPEED.
C
C 4. OUTPUT (denoted by "O" in column six of argument list at top of routine)
C
C	NOTE-- SIGN CONVENTION OF SFC ENERGY FLUXES BELOW IS: NEGATIVE IF
C            SINK OF ENERGY TO SURFACE
C
C          ETP: POTENTIAL EVAPORATION (W M-2)
C          ETA: ACTUAL LATENT HEAT FLUX (W M-2: NEGATIVE, IF UP FROM SURFACE)
C            H: SENSIBLE HEAT FLUX (W M-2: NEGATIVE, IF UPWARD FROM SURFACE)
C            S: SOIL HEAT FLUX (W M-2: NEGATIVE, IF DOWNWARD FROM SURFACE)
C      RUNOFF1: SURFACE RUNOFF (M S-1), NOT INFILTRATING THE SURFACE
C      RUNOFF2: SUBSURFACE RUNOFF (M S-1), DRAINAGE OUT BOTTOM OF LAST SOIL LYR
C           Q1: EFFECTIVE MIXING RATIO AT GRND SFC (KG KG-1)
C               (NOTE: Q1 IS NUMERICAL EXPENDIENCY FOR EXPRESSING ETA
C                     EQUIVALENTLY IN A BULK AERODYNAMIC FORM)
C        SNMAX: SNOW MELT (M) (WATER EQUIVALENT)
C        SOILW: AVAILABLE SOIL MOISTURE IN ROOT ZONE (UNITLESS FRACTION BETWEEN
C               SOIL SATURATION AND WILTING POINT)
C        SOILM: TOTAL SOIL COLUMN MOISTURE CONTENT (M) (FROZEN + UNFROZEN)
C
C           FOR DIAGNOSTIC PURPOSES, RETURN SOME PRIMARY PARAMETERS NEXT
C			(SET IN ROUTINE REDPRM)
C
C       SMCWLT: WILTING POINT (VOLUMETRIC)
C       SMCDRY: DRY SOIL MOISTURE THRESHOLD WHERE DIRECT EVAP FRM TOP LYR ENDS
C       SMCREF: SOIL MOISTURE THRESHOLD WHERE TRANSPIRATION BEGINS TO STRESS
C       SMCMAX: POROSITY, I.E. SATURATED VALUE OF SOIL MOISTURE

      INTEGER NSOLD
      PARAMETER (NSOLD = 20)
C
      LOGICAL SNOWNG
      LOGICAL FRZGRA
      LOGICAL SATURATED
C
      INTEGER K
      INTEGER KZ
      INTEGER ICE
      INTEGER NSOIL,VEGTYP,SOILTYP,NROOT
      INTEGER SLOPETYP
C
      REAL ALBEDO
      REAL ALB
      REAL B
      REAL BETA
      REAL CFACTR
C..................CH IS SFC EXCHANGE COEF FOR HEAT/MOIST
C..................CM IS SFC MOMENTUM DRAG (NOT NEEDED IN SFLX)
      REAL CH
      REAL CM
C
      REAL CMC
      REAL CMCMAX
      REAL CP
      REAL CSNOW
      REAL CSOIL
      REAL CZIL
      REAL DEW
      REAL DF1
      REAL DKSAT
      REAL DT
      REAL DWSAT
      REAL DQSDT2
      REAL DSOIL
      REAL DTOT
      REAL DRIP
      REAL EC
      REAL EDIR
      REAL ETT
      REAL EXPSNO
      REAL EXPSOI
      REAL EPSCA
      REAL ETA
      REAL ETP
      REAL EDIR1
      REAL EC1
      REAL ETT1
      REAL F
      REAL F1
      REAL FLX1
      REAL FLX2
      REAL FLX3
      REAL FXEXP
      REAL FRZX
      REAL H
      REAL HS
      REAL KDT
      REAL LWDN
      REAL LVH2O
      REAL PC
      REAL PRCP
      REAL PTU
      REAL PRCP1
      REAL PSISAT
      REAL Q1
      REAL Q2
      REAL Q2SAT
      REAL QUARTZ
      REAL R
      REAL RCH
      REAL REFKDT
      REAL RR
      REAL RTDIS (NSOLD)
      REAL RUNOFF1
      REAL RUNOFF2
      REAL RGL
      REAL RUNOF
      REAL RIB
      REAL RUNOFF3
      REAL RSMAX
      REAL RC
      REAL RCMIN
      REAL RSNOW
      REAL SNDENS
      REAL SNCOND 
      REAL S
      REAL SBETA
      REAL SFCPRS
      REAL SFCSPD
      REAL SFCTMP
      REAL SHDFAC
      REAL SH2O(NSOIL)
      REAL SLDPTH(NSOIL)
      REAL SMCDRY
      REAL SMCMAX
      REAL SMCREF
      REAL SMCWLT
      REAL SMC(NSOIL)
      REAL SNEQV
      REAL SNOWH
      REAL SNOFAC
      REAL SN_NEW
      REAL SLOPE
      REAL SNUP
      REAL SALP
      REAL SNOALB
      REAL STC(NSOIL)
      REAL SOLDN
      REAL SNMAX
      REAL SOILM
      REAL SOILW
      REAL SOILWM
      REAL SOILWW
      REAL T1
      REAL T1V
      REAL T24
      REAL T2V
      REAL TBOT
      REAL TH2
      REAL TH2V
      REAL TOPT
      REAL TFREEZ
      REAL XLAI
      REAL Z
      REAL ZBOT
      REAL Z0
      REAL ZSOIL(NSOLD)
C
      PARAMETER ( TFREEZ = 273.15      )
      PARAMETER ( LVH2O  = 2.501000E+6 )
      PARAMETER ( R      = 287.04      )
      PARAMETER ( CP     = 1004.5      )
      
C
C COMMON BLK "RITE" CARRIES DIAGNOSTIC QUANTITIES FOR PRINTOUT,
C BUT IS NOT INVOLVED IN MODEL PHYSICS AND IS NOT PRESENT IN
C PARENT MODEL THAT CALLS SFLX
C
      COMMON/RITE/ BETA,DRIP,EC,EDIR,ETT,FLX1,FLX2,FLX3,RUNOF,
     &             DEW,RIB,RUNOFF3

C   INITIALIZATION

C	write(6,*) 'inside SFLX: '

      RUNOFF1 = 0.0
      RUNOFF2 = 0.0
      RUNOFF3 = 0.0
      SNMAX = 0.0
C
C  THE VARIABLE "ICE" IS A FLAG DENOTING SEA-ICE CASE 

      IF(ICE .EQ. 1) THEN

C SEA-ICE LAYERS ARE EQUAL THICKNESS AND SUM TO 3 METERS
        DO KZ = 1, NSOIL
          ZSOIL(KZ)=-3.*FLOAT(KZ)/FLOAT(NSOIL)
        END DO

      ELSE

C CALCULATE DEPTH (NEGATIVE) BELOW GROUND FROM TOP SKIN SFC TO 
C BOTTOM OF EACH SOIL LAYER.
C NOTE:!!! SIGN OF ZSOIL IS NEGATIVE (DENOTING BELOW GROUND)
        ZSOIL(1)=-SLDPTH(1)
        DO KZ = 2, NSOIL
          ZSOIL(KZ)=-SLDPTH(KZ)+ZSOIL(KZ-1)
        END DO

      ENDIF
         
C ----------------------------------------------------------------------
CC
CC   NEXT IS CRUCIAL CALL TO SET THE LAND-SURFACE PARAMETERS, 
CC   INCLUDING SOIL-TYPE AND VEG-TYPE DEPENDENT PARAMETERS.
CC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC
C	write(6,*) 'call REDPRM'
        CALL REDPRM(VEGTYP,SOILTYP, SLOPETYP, 
     +    CFACTR, CMCMAX, RSMAX, TOPT, REFKDT, KDT, SBETA,
     O    SHDFAC, RCMIN, RGL, HS, ZBOT, FRZX, PSISAT, SLOPE, 
     +    SNUP, SALP, B, DKSAT, DWSAT, SMCMAX, SMCWLT, SMCREF,
     O    SMCDRY, F1, QUARTZ, FXEXP, RTDIS, SLDPTH, ZSOIL,
     +    NROOT, NSOIL, Z0, CZIL, XLAI, CSOIL, PTU)
C
C	write(6,*) 'return REDPRM'
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC
CC  NEXT CALL ROUTINE SFCDIF TO CALCULATE 
CC    THE SFC EXCHANGE COEF (CH) FOR HEAT AND MOISTURE
CC
CC  NOTE  NOTE  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
CC    
CC          COMMENT OUT CALL SFCDIF, IF SFCDIF ALREADY CALLED
CC          IN CALLING PROGRAM (SUCH AS IN COUPLED ATMOSPHERIC MODEL)
CC
CC  NOTE !!  DO NOT CALL SFCDIF UNTIL AFTER ABOVE CALL TO REDPRM, 
CC             IN CASE ALTERNATIVE VALUES OF ROUGHNESS LENGTH (Z0) AND 
CC              ZILINTINKEVICH COEF (CZIL) ARE SET THERE VIA NAMELIST I/O
CC
CC   NOTE !! ROUTINE SFCDIF RETURNS A CH THAT REPRESENTS THE WIND SPD
CC          TIMES THE "ORIGINAL" NONDIMENSIONAL "Ch" TYPICAL IN LITERATURE.
CC          HENCE THE CH RETURNED FROM SFCDIF HAS UNITS OF M/S.
CC          THE IMPORTANT COMPANION COEFFICIENT OF CH, CARRIED HERE AS "RCH",
CC          IS THE CH FROM SFCDIF TIMES AIR DENSITY AND PARAMETER "CP".
CC         "RCH" IS COMPUTED IN "CALL PENMAN". RCH RATHER THAN CH IS THE 
C          COEFF USUALLY INVOKED LATER IN EQNS.
CC
CC   NOTE !! SFCDIF ALSO RETURNS THE SURFACE EXCHANGE COEFFICIENT FOR
C            MOMENTUM, CM, ALSO KNOWN AS THE SURFACE DRAGE COEFFICIENT,
C            BUT CM IS NOT USED HERE
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
    
C ----------------------------------------------------------------------
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C CALC VIRTUAL TEMPS AND VIRTUAL POTENTIAL TEMPS NEEDED BY 
C SUBROUTINES SFCDIF AND PENMAN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      T2V  = SFCTMP * (1.0 + 0.61 * Q2 )
c comment out below 2 lines if CALL SFCDIF is commented out, i.e. in
c the coupled model
c      T1V  =     T1 * (1.0 + 0.61 * Q2 )
c      TH2V =    TH2 * (1.0 + 0.61 * Q2 )
C
C      CALL SFCDIF ( Z, Z0, T1V, TH2V, SFCSPD, CZIL, CM, CH )

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C  INITIALIZE MISC VARIABLES.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      SNOWNG = .FALSE.
      FRZGRA = .FALSE.

C IF SEA-ICE CASE,        ASSIGN DEFAULT WATER-EQUIV SNOW ON TOP
      IF(ICE .EQ. 1) THEN
        SNEQV = 0.01
        SNOWH = 0.05
      ENDIF
C
C IF INPUT SNOWPACK IS NONZERO, THEN COMPUTE SNOW DENSITY "SNDENS"
C AND SNOW THERMAL CONDUCTIVITY "SNCOND"
C (NOTE THAT CSNOW IS A FUNCTION SUBROUTINE)
C
      IF(SNEQV .EQ. 0.0) THEN
        SNDENS = 0.0
        SNOWH = 0.0
        SNCOND = 1.0
      ELSE
        SNDENS=SNEQV/SNOWH
        SNCOND = CSNOW (SNDENS) 
      ENDIF

C ----------------------------------------------------------------------
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     DETERMINE IF ITS PRECIPITATING AND WHAT KIND OF PRECIP IT IS.
C     IF IT'S PRCPING AND THE AIR TEMP IS COLDER THAN 0 C, IT'S SNOWING!
C     IF ITS PRCPING AND THE AIR TEMP IS WARMER THAN 0 C, BUT THE GRND
C     TEMP IS COLDER THAN 0 C, FREEZING RAIN IS PRESUMED TO BE FALLING.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      IF ( PRCP .GT. 0.0 ) THEN
        IF ( SFCTMP .LE. TFREEZ ) THEN
          SNOWNG = .TRUE.
        ELSE
          IF ( T1 .LE. TFREEZ ) FRZGRA = .TRUE.
        ENDIF
      ENDIF

C ----------------------------------------------------------------------
C If either prcp flag is set, determine new snowfall (converting prcp
C rate from kg m-2 s-1 to a liquid equiv snow depth in meters) and add
C it to the existing snowpack.
C Note that since all precip is added to snowpack, no precip infiltrates
C into the soil so that PRCP1 is set to zero.
      IF ( ( SNOWNG ) .OR. ( FRZGRA ) ) THEN
        SN_NEW = PRCP * DT * 0.001
        SNEQV = SNEQV + SN_NEW
        PRCP1 = 0.0
C ----------------------------------------------------------------------
C Update snow density based on new snowfall, using old and new snow.
      CALL SNOW_NEW (SFCTMP,SN_NEW,SNOWH,SNDENS)
C --- debug ------------------------------------------------------------
c      SNDENS = 0.2
c      SNOWH = SNEQV/SNDENS
C --- debug ------------------------------------------------------------
C ----------------------------------------------------------------------
C Update snow thermal conductivity
      SNCOND = CSNOW (SNDENS) 
C ----------------------------------------------------------------------

      ELSE
C
C PRECIP IS LIQUID (RAIN), HENCE SAVE IN THE PRECIP VARIABLE THAT
C LATER CAN WHOLELY OR PARTIALLY INFILTRATE THE SOIL (ALONG WITH 
C ANY CANOPY "DRIP" ADDED TO THIS LATER)
C
        PRCP1 = PRCP

      ENDIF

C ----------------------------------------------------------------------
C Thermal conductivity for sea-ice case
      IF (ICE .EQ. 1) THEN
        DF1=2.2
      ELSE
C
C NEXT CALCULATE THE SUBSURFACE HEAT FLUX, WHICH FIRST REQUIRES
C CALCULATION OF THE THERMAL DIFFUSIVITY.  TREATMENT OF THE
C LATTER FOLLOWS THAT ON PAGES 148-149 FROM HEAT TRANSFER IN 
C COLD CLIMATES, BY V. J. LUNARDINI (PUBLISHED IN 1981 
C BY VAN NOSTRAND REINHOLD CO.) I.E. TREATMENT OF TWO CONTIGUOUS 
C "PLANE PARALLEL" MEDIUMS (NAMELY HERE THE FIRST SOIL LAYER 
C AND THE SNOWPACK LAYER, IF ANY). THIS DIFFUSIVITY TREATMENT 
C BEHAVES WELL FOR BOTH ZERO AND NONZERO SNOWPACK, INCLUDING THE 
C LIMIT OF VERY THIN SNOWPACK.  THIS TREATMENT ALSO ELIMINATES
C THE NEED TO IMPOSE AN ARBITRARY UPPER BOUND ON SUBSURFACE 
C HEAT FLUX WHEN THE SNOWPACK BECOMES EXTREMELY THIN.
C
C ----------------------------------------------------------------------
C FIRST CALCULATE THERMAL DIFFUSIVITY OF TOP SOIL LAYER, USING
C BOTH THE FROZEN AND LIQUID SOIL MOISTURE, FOLLOWING THE 
C SOIL THERMAL DIFFUSIVITY FUNCTION OF PETERS-LIDARD ET AL.
C (1998,JAS, VOL 55, 1209-1224), WHICH REQUIRES THE SPECIFYING
C THE QUARTZ CONTENT OF THE GIVEN SOIL CLASS (SEE ROUTINE REDPRM)
C
        CALL TDFCND ( DF1, SMC(1),QUARTZ,SMCMAX,SH2O(1) )
C ----------------------------------------------------------------------
C NEXT ADD SUBSURFACE HEAT FLUX REDUCTION EFFECT FROM THE 
C OVERLYING GREEN CANOPY, ADAPTED FROM SECTION 2.1.2 OF 
C PETERS-LIDARD ET AL. (1997, JGR, VOL 102(D4))
C
        DF1 = DF1 * EXP(SBETA*SHDFAC)
      ENDIF
C ----------------------------------------------------------------------
C FINALLY "PLANE PARALLEL" SNOWPACK EFFECT FOLLOWING 
C V.J. LINARDINI REFERENCE CITED ABOVE. NOTE THAT DTOT IS
C COMBINED DEPTH OF SNOWDEPTH AND THICKNESS OF FIRST SOIL LAYER
C
      DSOIL = -(0.5 * ZSOIL(1))

      IF (SNEQV .EQ. 0.) THEN
        S = DF1 * (T1 - STC(1) ) / DSOIL
      ELSE
        DTOT = SNOWH + DSOIL
        EXPSNO = SNOWH/DTOT
        EXPSOI = DSOIL/DTOT
c 1. harmonic mean (series flow)
c     DF1 = (SNCOND*DF1)/(EXPSOI*SNCOND+EXPSNO*DF1)
c 2. arithmetic mean (parallel flow)
c     DF1 = EXPSNO*SNCOND + EXPSOI*DF1
c 3. geometric mean (intermediate between 
c                     harmonic and arithmetic mean)
        DF1 = (SNCOND**EXPSNO)*(DF1**EXPSOI)
C ----------------------------------------------------------------------
C CALCULATE SUBSURFACE HEAT FLUX, S, FROM FINAL THERMAL DIFFUSIVITY
C OF SURFACE MEDIUMS, DF1 ABOVE, AND SKIN TEMPERATURE AND TOP 
C MID-LAYER SOIL TEMPERATURE
        S = DF1 * (T1 - STC(1) ) / DTOT
      ENDIF

C ----------------------------------------------------------------------
C Update albedo, except over sea-ice
      IF (ICE .EQ. 0) THEN

C ----------------------------------------------------------------------
C NEXT IS TIME-DEPENDENT SURFACE ALBEDO MODIFICATION DUE TO 
C TIME-DEPENDENT SNOWDEPTH STATE AND TIME-DEPENDENT CANOPY GREENNESS
      
c      IF ( (SNEQV .EQ. 0.0) .OR. (ALB .GE. SNOALB) ) THEN
        IF (SNEQV .EQ. 0.0) THEN
          ALBEDO = ALB

        ELSE
C ----------------------------------------------------------------------
C SNUP IS VEG-CLASS DEPENDENT SNOWDEPTH THRESHHOLD (SET IN ROUTINE
C REDPRM)WHERE MAX SNOW ALBEDO EFFECT IS FIRST ATTAINED
          IF (SNEQV .LT. SNUP) THEN
            RSNOW = SNEQV/SNUP
            SNOFAC = 1. - ( EXP(-SALP*RSNOW) - RSNOW*EXP(-SALP))
          ELSE
            SNOFAC = 1.0
          ENDIF
C ----------------------------------------------------------------------
C SNOALB IS ARGUMENT REPRESENTING MAXIMUM ALBEDO OVER DEEP SNOW,
C AS PASSED INTO SFLX, AND ADAPTED FROM THE SATELLITE-BASED MAXIMUM 
C SNOW ALBEDO FIELDS PROVIDED BY D. ROBINSON AND G. KUKLA 
C (1985, JCAM, VOL 24, 402-411)

          ALBEDO = ALB + (1.0-SHDFAC)*SNOFAC*(SNOALB-ALB) 
          IF (ALBEDO .GT. SNOALB) ALBEDO=SNOALB
        ENDIF

      ELSE
C ----------------------------------------------------------------------
C albedo over sea-ice
          ALBEDO = 0.60
          SNOFAC = 1.0
      ENDIF

C ----------------------------------------------------------------------
C  CALCULATE TOTAL DOWNWARD RADIATION (SOLAR PLUS LONGWAVE)
C  NEEDED IN PENMAN EP SUBROUTINE THAT FOLLOWS
          
          F = SOLDN*(1.0-ALBEDO) + LWDN

C ----------------------------------------------------------------------
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALL PENMAN SUBROUTINE TO CALCULATE POTENTIAL EVAPORATION (ETP)
C     (AND OTHER PARTIAL PRODUCTS AND SUMS SAVE IN COMMON/RITE FOR 
C       LATER CALCULATIONS)
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

       CALL PENMAN ( SFCTMP,SFCPRS,CH,T2V,TH2,PRCP,F,T24,S,Q2,
     &              Q2SAT,ETP,RCH,EPSCA,RR,SNOWNG,FRZGRA,DQSDT2)
C
C following old constraint is disabled
C.....IF(SATURATED) ETP = 0.0

C ----------------------------------------------------------------------
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALL CANRES TO CALCULATE THE CANOPY RESISTANCE AND CONVERT IT 
C     INTO PC IF MORE THAN TRACE AMOUNT OF CANOPY GREENNESS FRACTION
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

c      IF(SHDFAC .GT. 1.E-6) THEN
c make this threshold consistent with the one in SMFLX for TRANSP
c and EC(anopy)
      IF(SHDFAC .GT. 0.) THEN
      
C  FROZEN GROUND EXTENSION: TOTAL SOIL WATER "SMC" WAS REPLACED 
C  BY UNFROZEN SOIL WATER "SH2O" IN CALL TO CANRES BELOW
C      
        CALL CANRES(SOLDN,CH,SFCTMP,Q2,SFCPRS,SH2O,ZSOIL,NSOIL,
     &            SMCWLT,SMCREF,RCMIN,RC,PC,NROOT,Q2SAT,DQSDT2,
     &            TOPT,RSMAX,RGL,HS,XLAI)

      ENDIF

C ----------------------------------------------------------------------
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C      NOW DECIDE MAJOR PATHWAY BRANCH TO TAKE DEPENDING ON WHETHER
C      SNOWPACK EXISTS OR NOT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        IF ( SNEQV .EQ. 0.0 ) THEN

          CALL NOPAC ( ETP, ETA, PRCP, SMC, SMCMAX, SMCWLT,
     &                 SMCREF,SMCDRY, CMC, CMCMAX, NSOIL, DT, SHDFAC,
     &                 SBETA,Q1,Q2,T1,SFCTMP,T24,TH2,F,F1,S,STC,
     &                 EPSCA, B, PC, RCH, RR,  CFACTR,
     +                 SH2O, SLOPE, KDT, FRZX, PSISAT, ZSOIL,
     &                 DKSAT, DWSAT, TBOT, ZBOT, RUNOFF1,RUNOFF2,
     &                 RUNOFF3, EDIR1, EC1, ETT1,NROOT,ICE,RTDIS,
     &                 QUARTZ, FXEXP,CSOIL)

        ELSE

          CALL SNOPAC ( ETP,ETA,PRCP,PRCP1,SNOWNG,SMC,SMCMAX,SMCWLT,
     &                SMCREF, SMCDRY, CMC, CMCMAX, NSOIL, DT, 
     &                SBETA,Q1,DF1,
     &                Q2,T1,SFCTMP,T24,TH2,F,F1,S,STC,EPSCA,SFCPRS,
c     &                B, PC, RCH, RR, CFACTR, SALP, SNEQV,
     &                B, PC, RCH, RR, CFACTR, SNOFAC, SNEQV,SNDENS,
     +                SNOWH, SH2O, SLOPE, KDT, FRZX, PSISAT, SNUP,
     &                ZSOIL, DWSAT, DKSAT, TBOT, ZBOT, SHDFAC,RUNOFF1,
     &                RUNOFF2,RUNOFF3,EDIR1,EC1,ETT1,NROOT,SNMAX,ICE,
     &                RTDIS,QUARTZ, FXEXP,CSOIL)
        
        ENDIF

C ----------------------------------------------------------------------
C   PREPARE SENSIBLE HEAT (H) FOR RETURN TO PARENT MODEL
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

          H = -(CH * CP * SFCPRS)/(R * T2V) * ( TH2 - T1 )
          
C ----------------------------------------------------------------------
C  CONVERT UNITS AND/OR SIGN OF TOTAL EVAP (ETA), POTENTIAL EVAP (ETP),
C  SUBSURFACE HEAT FLUX (S), AND RUNOFFS FOR WHAT PARENT MODEL EXPECTS
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C  CONVERT ETA FROM KG M-2 S-1 TO W M-2
C
      ETA = ETA*LVH2O
      ETP = ETP*LVH2O

C CONVERT THE SIGN OF SOIL HEAT FLUX SO THAT:
C         S>0: WARM THE SURFACE  (NIGHT TIME)
C         S<0: COOL THE SURFACE  (DAY TIME)

      S=-1.0*S      
C
C  CONVERT RUNOFF3 (INTERNAL LAYER RUNOFF FROM SUPERSAT) FROM M TO M S-1
C  AND ADD TO SUBSURFACE RUNOFF/DRAINAGE/BASEFLOW
C
      RUNOFF3 = RUNOFF3/DT
      RUNOFF2 = RUNOFF2+RUNOFF3
C
C TOTAL COLUMN SOIL MOISTURE IN METERS (SOILM) AND ROOT-ZONE 
C SOIL MOISTURE AVAILABILITY (FRACTION) RELATIVE TO POROSITY/SATURATION

      SOILM=-1.0*SMC(1)*ZSOIL(1)
      
      DO K = 2, NSOIL
        SOILM=SOILM+SMC(K)*(ZSOIL(K-1)-ZSOIL(K))
      END DO
      SOILWM=-1.0*(SMCMAX-SMCWLT)*ZSOIL(1)
      SOILWW=-1.0*(SMC(1)-SMCWLT)*ZSOIL(1)
      DO K = 2, NROOT
        SOILWM=SOILWM+(SMCMAX-SMCWLT)*(ZSOIL(K-1)-ZSOIL(K))
        SOILWW=SOILWW+(SMC(K)-SMCWLT)*(ZSOIL(K-1)-ZSOIL(K))
      END DO
      SOILW=SOILWW/SOILWM
C
      RETURN
      END
      SUBROUTINE CANRES(SOLAR,CH,SFCTMP,Q2,SFCPRS,SMC,ZSOIL,NSOIL,
     &                  SMCWLT,SMCREF,RCMIN,RC,PC,NROOT,Q2SAT,DQSDT2, 
     &                  TOPT,RSMAX,RGL,HS,XLAI)

      IMPLICIT NONE

C ######################################################################
C                        SUBROUTINE CANRES
C                        -----------------
C       THIS ROUTINE CALCULATES THE CANOPY RESISTANCE WHICH DEPENDS ON
C       INCOMING SOLAR RADIATION, AIR TEMPERATURE, ATMOSPHERIC WATER
C       VAPOR PRESSURE DEFICIT AT THE LOWEST MODEL LEVEL, AND SOIL
C       MOISTURE (PREFERABLY UNFROZEN SOIL MOISTURE RATHER THAN TOTAL)
C ----------------------------------------------------------------------
C        SOURCE:  JARVIS (1976), JACQUEMIN AND NOILHAN (1990 BLM)
C ----------------------------------------------------------------------
C ----------------------------------------------------------------------
C        INPUT:  SOLAR: INCOMING SOLAR RADIATION
C                CH:     SURFACE EXCHANGE COEFFICIENT FOR HEAT AND MOISTURE
C                SFCTMP: AIR TEMPERATURE AT 1ST LEVEL ABOVE GROUND
C                Q2:     AIR HUMIDITY AT 1ST LEVEL ABOVE GROUND
C                Q2SAT:  SATURATION AIR HUMIDITY AT 1ST LEVEL ABOVE GROUND
C                DQSDT2: SLOPE OF SATURATION HUMIDITY FUNCTION WRT TEMP
C                SFCPRS: SURFACE PRESSURE
C                SMC:    VOLUMETRIC SOIL MOISTURE 
C                ZSOIL:  SOIL DEPTH (NEGATIVE SIGN, AS IT IS BELOW GROUND)
C                NSOIL:  NO. OF SOIL LAYERS
C                NROOT:  NO. OF SOIL LAYERS IN ROOT ZONE (1.LE.NROOT.LE.NSOIL)
C                XLAI:   LEAF AREA INDEX
C                SMCWLT: WILTING POINT
C                SMCREF: REFERENCE SOIL MOISTURE
C                        (WHERE SOIL WATER DEFICIT STRESS SETS IN)
C
C RCMIN, RSMAX, TOPT, RGL, HS: CANOPY STRESS PARAMETERS SET IN SUBR REDPRM
C
C  (SEE EQNS 12-14 AND TABLE 2 OF SEC. 3.1.2 OF 
C       CHEN ET AL., 1996, JGR, VOL 101(D3), 7251-7268)               
C
C        OUTPUT:  PC: PLANT COEFFICIENT
C                 RC: CANOPY RESISTANCE
C ----------------------------------------------------------------------
C ######################################################################

      INTEGER   NSOLD
      PARAMETER (NSOLD = 20)

      INTEGER K
      INTEGER NROOT
      INTEGER NSOIL

      REAL SIGMA, RD, CP, SLV
      REAL SOLAR, CH, SFCTMP, Q2, SFCPRS 
      REAL SMC(NSOIL), ZSOIL(NSOIL), PART(NSOLD) 
      REAL SMCWLT, SMCREF, RCMIN, RC, PC, Q2SAT, DQSDT2
      REAL TOPT, RSMAX, RGL, HS, XLAI, RCS, RCT, RCQ, RCSOIL, FF
      REAL P, QS, GX, TAIR4, ST1, SLVCP, RR, DELTA

      PARAMETER (SIGMA=5.67E-8, RD=287.04, CP=1004.5, SLV=2.501000E6)

      RCS = 0.0
      RCT = 0.0
      RCQ = 0.0
      RCSOIL = 0.0
      RC = 0.0

C ----------------------------------------------------------------------
C CONTRIBUTION DUE TO INCOMING SOLAR RADIATION
C ----------------------------------------------------------------------

CC/98/01/05/..disgard old version assuming fixed LAI=1
CC...........FF = 0.55*2.0*SOLAR/RGL

      FF = 0.55*2.0*SOLAR/(RGL*XLAI)
      RCS = (FF + RCMIN/RSMAX) / (1.0 + FF)
      RCS = MAX(RCS,0.0001)

C ----------------------------------------------------------------------
C CONTRIBUTION DUE TO AIR TEMPERATURE AT FIRST MODEL LEVEL ABOVE GROUND
C ----------------------------------------------------------------------

      RCT = 1.0 - 0.0016*((TOPT-SFCTMP)**2.0)
      RCT = MAX(RCT,0.0001)

C ----------------------------------------------------------------------
C CONTRIBUTION DUE TO VAPOR PRESSURE DEFICIT AT FIRST MODEL LEVEL.
C ----------------------------------------------------------------------

c      P = SFCPRS
      QS = Q2SAT
C RCQ EXPRESSION FROM SSIB 
      RCQ = 1.0/(1.0+HS*(QS-Q2))
      RCQ = MAX(RCQ,0.01)

C ----------------------------------------------------------------------
C CONTRIBUTION DUE TO SOIL MOISTURE AVAILABILITY.
C DETERMINE CONTRIBUTION FROM EACH SOIL LAYER, THEN ADD THEM UP.
C ----------------------------------------------------------------------

      GX = (SMC(1) - SMCWLT) / (SMCREF - SMCWLT)
      IF (GX .GT. 1.) GX = 1.
      IF (GX .LT. 0.) GX = 0.

C####   USING SOIL DEPTH AS WEIGHTING FACTOR
      PART(1) = (ZSOIL(1)/ZSOIL(NROOT)) * GX

C#### USING ROOT DISTRIBUTION AS WEIGHTING FACTOR
CC      PART(1) = RTDIS(1) * GX
      
      DO K = 2, NROOT
        GX = (SMC(K) - SMCWLT) / (SMCREF - SMCWLT)
        IF (GX .GT. 1.) GX = 1.
        IF (GX .LT. 0.) GX = 0.
C####   USING SOIL DEPTH AS WEIGHTING FACTOR        
        PART(K) = ((ZSOIL(K)-ZSOIL(K-1))/ZSOIL(NROOT)) * GX

C#### USING ROOT DISTRIBUTION AS WEIGHTING FACTOR
CC         PART(K) = RTDIS(K) * GX 
               
      END DO

      DO K = 1, NROOT
        RCSOIL = RCSOIL+PART(K)
      END DO

      RCSOIL = MAX(RCSOIL,0.0001)

C ----------------------------------------------------------------------
C         DETERMINE CANOPY RESISTANCE DUE TO ALL FACTORS.
C         CONVERT CANOPY RESISTANCE (RC) TO PLANT COEFFICIENT (PC).
C ----------------------------------------------------------------------

CC/98/01/05/........RC = RCMIN/(RCS*RCT*RCQ*RCSOIL)
      RC = RCMIN/(XLAI*RCS*RCT*RCQ*RCSOIL)
          
      TAIR4 = SFCTMP**4.
      ST1 = (4.*SIGMA*RD)/CP
      SLVCP = SLV/CP
      RR = ST1*TAIR4/(SFCPRS*CH) + 1.0
      DELTA = SLVCP*DQSDT2
      
      PC = (RR+DELTA)/(RR*(1.+RC*CH)+DELTA)
      
      RETURN
      END
      FUNCTION CSNOW ( DSNOW )

      IMPLICIT NONE

      REAL C
      REAL DSNOW
      REAL CSNOW
      REAL UNIT

      PARAMETER ( UNIT=0.11631 ) 
                                         
C   ####  SIMULATION OF TERMAL SNOW CONDUCTIVITY                   
C   ####  SIMULATION UNITS OF CSNOW IS CAL/(CM*HR* C) 
C   ####  AND IT WILL BE RETURND IN W/(M* C)
C   ####  BASIC VERSION IS DYACHKOVA EQUATION                                

C #####   DYACHKOVA EQUATION (1960), FOR RANGE 0.1-0.4

      C=0.328*10**(2.25*DSNOW)
      CSNOW=UNIT*C

C #####    DE VAUX EQUATION (1933), IN RANGE 0.1-0.6
C       CSNOW=0.0293*(1.+100.*DSNOW**2)
      
C     #####   E. ANDERSEN FROM FLERCHINGER
C     CSNOW=0.021+2.51*DSNOW**2        
      
      RETURN                                                      
      END
      FUNCTION DEVAP ( ETP1, SMC, ZSOIL, SHDFAC, SMCMAX, B,
     &                 DKSAT, DWSAT, SMCDRY, SMCREF, SMCWLT, FXEXP)

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    NAME:  DIRECT EVAPORATION (DEVAP) FUNCTION  VERSION: N/A
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      REAL B
      REAL DEVAP
      REAL DKSAT
      REAL DWSAT
      REAL ETP1
      REAL FX
      REAL FXEXP
      REAL SHDFAC
      REAL SMC
      REAL SMCDRY
      REAL SMCMAX
      REAL ZSOIL
      REAL SMCREF
      REAL SMCWLT

Cmp      FX = ( (SMC - SMCDRY) / (SMCMAX - SMCDRY) )**FXEXP
      FX = (ABS((SMC - SMCDRY) / (SMCMAX - SMCDRY)))**FXEXP

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     FX > 1 REPRESENTS DEMAND CONTROL
C     FX < 1 REPRESENTS FLUX CONTROL
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      FX = MAX ( MIN ( FX, 1. ) ,0. )

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     ALLOW FOR THE DIRECT-EVAP-REDUCING EFFECT OF SHADE
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DEVAP = FX * ( 1.0 - SHDFAC ) * ETP1

      RETURN
      END
      FUNCTION FRH2O(TKELV,SMC,SH2O,SMCMAX,B,PSIS)

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC  PURPOSE:  CALCULATE AMOUNT OF SUPERCOOLED LIQUID SOIL WATER CONTENT
CC  IF TEMPERATURE IS BELOW 273.15K (T0).  REQUIRES NEWTON-TYPE ITERATION
CC  TO SOLVE THE NONLINEAR IMPLICIT EQUATION GIVEN IN EQN 17 OF
CC  KOREN ET AL. (1999, JGR, VOL 104(D16), 19569-19585).
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C New version (FEB 2001): much faster and more accurate newton iteration
c achieved by first taking log of eqn cited above -- less than 4
c (typically 1 or 2) iterations achieves convergence.  Also, explicit
c 1-step solution option for special case of parameter Ck=0, which reduces
c the original implicit equation to a simpler explicit form, known as the
c "Flerchinger Eqn". Improved handling of solution in the limit of
c freezing point temperature T0.
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C INPUT:
C
C   TKELV.........Temperature (Kelvin)
C   SMC...........Total soil moisture content (volumetric)
C   SH2O..........Liquid soil moisture content (volumetric)
C   SMCMAX........Saturation soil moisture content (from REDPRM)
C   B.............Soil type "B" parameter (from REDPRM)
C   PSIS..........Saturated soil matric potential (from REDPRM)
C
C OUTPUT:
C   FRH2O.........supercooled liquid water content.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      REAL B
      REAL BLIM
      REAL BX
      REAL CK
      REAL DENOM
      REAL DF
      REAL DH2O
      REAL DICE
      REAL DSWL
      REAL ERROR
      REAL FK
      REAL FRH2O
      REAL GS
      REAL HLICE
      REAL PSIS
      REAL SH2O
      REAL SMC
      REAL SMCMAX
      REAL SWL
      REAL SWLK
      REAL TKELV
      REAL T0

      INTEGER NLOG
      INTEGER KCOUNT

      PARAMETER (CK=8.0)
C      PARAMETER (CK=0.0)
      PARAMETER (BLIM=5.5)
C      PARAMETER (BLIM=7.0)
      PARAMETER (ERROR=0.005)

      PARAMETER (HLICE=3.335E5)
      PARAMETER (GS = 9.81)
      PARAMETER (DICE=920.0)
      PARAMETER (DH2O=1000.0)
      PARAMETER (T0=273.15)

C  ###   LIMITS ON PARAMETER B: B < 5.5  (use parameter BLIM)  ####
C  ###   SIMULATIONS SHOWED IF B > 5.5 UNFROZEN WATER CONTENT  ####
C  ###   IS NON-REALISTICALLY HIGH AT VERY LOW TEMPERATURES    ####
C##################################################################
C
      BX = B
      IF ( B .GT. BLIM ) BX = BLIM
C------------------------------------------------------------------

C INITIALIZING ITERATIONS COUNTER AND ITERATIVE SOLUTION FLAG.
      NLOG=0
      KCOUNT=0

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C  IF TEMPERATURE NOT SIGNIFICANTLY BELOW FREEZING (T0), SH2O = SMC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      IF (TKELV .GT. (T0 - 1.E-3)) THEN

        FRH2O=SMC

      ELSE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
       IF (CK .NE. 0.0) THEN

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCC OPTION 1: ITERATED SOLUTION FOR NONZERO CK CCCCCCCCCCC
CCCCCCCCCCCC IN KOREN ET AL, JGR, 1999, EQN 17 CCCCCCCCCCCCCCCCC
C
C INITIAL GUESS FOR SWL (frozen content)
        SWL = SMC-SH2O
C KEEP WITHIN BOUNDS.
        IF(SWL .GT. (SMC-0.02)) THEN
CMBEK          SWL=SMC-0.02
          SWL=MAX(0.,SMC-0.02)
        ENDIF
CMBEK        IF(SWL .LT. 0.)  THEN
CMBEK          SWL=0.
CMBEK        ENDIF
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C  START OF ITERATIONS
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        DO WHILE (NLOG .LT. 10 .AND. KCOUNT .EQ. 0)
         NLOG = NLOG+1
         DF = ALOG(( PSIS*GS/HLICE ) * ( ( 1.+CK*SWL )**2. ) *
     &        ( SMCMAX/(SMC-SWL) )**BX) - ALOG(-(TKELV-T0)/TKELV)
         DENOM = 2. * CK / ( 1.+CK*SWL ) + BX / ( SMC - SWL )
         SWLK = SWL - DF/DENOM
         DSWL=ABS(SWLK-SWL)
         SWL=SWLK
C KEEP WITHIN BOUNDS.
         IF(SWL .GT. (SMC-0.02)) THEN
CMBEK           SWL=SMC-0.02
           SWL=MAX(0.,SMC-0.02)
         ENDIF
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC IF MORE THAN 10 ITERATIONS, USE EXPLICIT METHOD (CK=0 APPROX.)
CC WHEN DSWL LESS OR EQ. ERROR, NO MORE ITERATIONS REQUIRED.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
         IF ( DSWL .LE. ERROR )  THEN
           KCOUNT=KCOUNT+1
         END IF
        END DO
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C  END OF ITERATIONS
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C LOWER BOUND FOR SWL (ICE CONTENT)
CMBEK        IF(SWL .LT. 0.)  THEN
CMBEK          SWL=0.
CMBEK        ENDIF
C UPPER BOUND FOR SWL ALREADY APPLIED WITHIN DO-BLOCK
        FRH2O = SMC - SWL
C
CCCCCCCCCCCCCCCCCCCCCCCC END OPTION 1 CCCCCCCCCCCCCCCCCCCCCCCCCCC

       ENDIF

       IF (KCOUNT .EQ. 0) THEN
         Print*,'Flerchinger used. Iterations=',NLOG

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCC OPTION 2: EXPLICIT SOLUTION FOR FLERCHINGER EQ. i.e. CK=0 CCCCCCCC
CCCCCCCCCCCCC IN KOREN ET AL., JGR, 1999, EQN 17  CCCCCCCCCCCCCCC
C
        FK=(((HLICE/(GS*(-PSIS)))*((TKELV-T0)/TKELV))**(-1/BX))*SMCMAX
        IF (FK .LT. 0.02) FK = 0.02
        FRH2O = MIN ( FK, SMC )
C
CCCCCCCCCCCCCCCCCCCCCCCCC END OPTION 2 CCCCCCCCCCCCCCCCCCCCCCCCCC

       ENDIF

      ENDIF

      RETURN
      END 
      SUBROUTINE HRT ( RHSTS,STC,SMC,SMCMAX,NSOIL,ZSOIL,YY,ZZ1,
     +                 TBOT, ZBOT, PSISAT, SH2O, DT, B,
     +                 F1, DF1, QUARTZ, CSOIL)

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE THE RIGHT HAND SIDE OF THE TIME TENDENCY
CC    =======   TERM OF THE SOIL THERMAL DIFFUSION EQUATION.  ALSO TO
CC              COMPUTE ( PREPARE ) THE MATRIX COEFFICIENTS FOR THE
CC              TRI-DIAGONAL MATRIX OF THE IMPLICIT TIME SCHEME.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      INTEGER NSOLD
      PARAMETER ( NSOLD = 20 )

      INTEGER I
      INTEGER K
      INTEGER NSOIL

C DECLARE WORK ARRAYS NEEDED IN TRI-DIAGONAL IMPLICIT SOLVER

      REAL AI    ( NSOLD )
      REAL BI    ( NSOLD )
      REAL CI    ( NSOLD )

C DECLARE SPECIFIC HEAT CAPACITIES

      REAL CAIR
      REAL CH2O
      REAL CICE
      REAL CSOIL

      REAL DDZ
      REAL DDZ2
      REAL DENOM
      REAL DF1
      REAL DF1N
      REAL DF1K
      REAL DTSDZ
      REAL DTSDZ2
      REAL F1
      REAL HCPCT
      REAL QUARTZ
      REAL QTOT
      REAL RHSTS ( NSOIL )
      REAL S
      REAL SMC   ( NSOIL )

      REAL SH2O  ( NSOIL )
      REAL SMCMAX
            
      REAL STC   ( NSOIL )
      REAL TBOT
      REAL ZBOT
      REAL YY
      REAL ZSOIL ( NSOIL )
      REAL ZZ1

      REAL T0, TSURF, PSISAT, DT, B, SICE, TBK, TSNSR, TBK1

      REAL SNKSRC
C
      COMMON /ABCI/ AI, BI, CI
C
      PARAMETER ( T0   = 273.15  )

C SET SPECIFIC HEAT CAPACITIES OF AIR, WATER, ICE, SOIL MINERAL       

      PARAMETER ( CAIR =1004.0   )
      PARAMETER ( CH2O = 4.2E6   )
      PARAMETER ( CICE = 2.106E6 )
C.....PARAMETER ( CSOIL=1.26E6   )
C.....
C NOTE: CSOIL NOW SET IN ROUTINE REDPRM AND PASSED IN

C+++++++++++++ BEGIN SECTION FOR TOP SOIL LAYER +++++++++++++++++++++

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALC THE HEAT CAPACITY OF THE TOP SOIL LAYER
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      HCPCT = SH2O(1)*CH2O + (1.0-SMCMAX)*CSOIL + (SMCMAX-SMC(1))*CAIR
     +        + ( SMC(1) - SH2O(1) )*CICE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALC THE MATRIX COEFFICIENTS AI, BI, AND CI FOR THE TOP LAYER
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DDZ = 1.0 / ( -0.5 * ZSOIL(2) )
      AI(1) = 0.0
      CI(1) =  ( DF1 * DDZ ) / ( ZSOIL(1) * HCPCT )
      BI(1) = -CI(1) + DF1 / ( 0.5 * ZSOIL(1) * ZSOIL(1)*HCPCT*ZZ1)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALC THE VERTICAL SOIL TEMP GRADIENT BTWN THE 1ST AND 2ND SOIL
C     LAYERS.  THEN CALCULATE THE SUBSURFACE HEAT FLUX. USE THE TEMP
C     GRADIENT AND SUBSFC HEAT FLUX TO CALC "RIGHT-HAND SIDE TENDENCY
C     TERMS", OR "RHSTS", FOR TOP SOIL LAYER.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      DTSDZ = ( STC(1) - STC(2) ) / ( -0.5 * ZSOIL(2) )
      S = DF1 * ( STC(1) - YY ) / ( 0.5 * ZSOIL(1) * ZZ1 )
      RHSTS(1) = ( DF1 * DTSDZ - S ) / ( ZSOIL(1) * HCPCT )

C NEXT, SET TEMP "TSURF" AT TOP OF SOIL COLUMN (FOR USE IN FREEZING
C SOIL PHYSICS LATER IN FUNCTION SUBROUTINE SNKSRC). IF SNOWPACK 
C CONTENT IS ZERO, THEN EXPRESSION BELOW GIVES TSURF = SKIN TEMP.
C IF SNOWPACK IS NONZERO (HENCE ARGUMENT ZZ1=1), THEN EXPRESSION
C BELOW YIELDS SOIL COLUMN TOP TEMPERATURE UNDER SNOWPACK.
C
      TSURF = ( YY + ( ZZ1 - 1 ) * STC(1) ) / ZZ1
C
C NEXT CAPTURE THE VERTICAL DIFFERENCE OF THE HEAT FLUX AT TOP 
C AND BOTTOM OF FIRST SOIL LAYER FOR USE IN HEAT FLUX CONSTRAINT 
C APPLIED TO POTENTIAL SOIL FREEZING/THAWING IN ROUTINE SNKSRC
C
      QTOT = S - DF1*DTSDZ

C
C CALCULATE TEMPERATURE AT BOTTOM INTERFACE OF 1ST SOIL LAYER 
C FOR USE LATER IN FCN SUBROUTINE SNKSRC
C
      CALL TBND ( STC(1), STC(2), ZSOIL, ZBOT, 1, NSOIL,TBK)
C
C CALCULATE FROZEN WATER CONTENT IN 1ST SOIL LAYER. 
C
      SICE = SMC(1) - SH2O(1)
C
C IF FROZEN WATER PRESENT OR ANY OF LAYER-1 MID-POINT OR BOUNDING
C INTERFACE TEMPERATURES BELOW FREEZING, THEN CALL SNKSRC TO
C COMPUTE HEAT SOURCE/SINK (AND CHANGE IN FROZEN WATER CONTENT)
C DUE TO POSSIBLE SOIL WATER PHASE CHANGE
C
      IF ( (SICE .GT. 0.) .OR. (TSURF .LT. T0) .OR.
     &     (STC(1) .LT. T0) .OR. (TBK .LT. T0) ) THEN
 
       TSNSR = SNKSRC ( TSURF, STC(1),TBK, SMC(1), SH2O(1), 
     +           ZSOIL, NSOIL, SMCMAX, PSISAT, B, DT, 1, QTOT )

       RHSTS(1) = RHSTS(1) - TSNSR / ( ZSOIL(1) * HCPCT )

      ENDIF
 
C ++++++++++++++ THIS ENDS SECTION FOR TOP SOIL LAYER ++++++++++++++
            
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     INITIALIZE DDZ2
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DDZ2 = 0.0

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     LOOP THRU THE REMAINING SOIL LAYERS, REPEATING THE ABOVE PROCESS
C(EXCEPT SUBSFC OR "GROUND" HEAT FLUX NOT REPEATED IN LOWER LAYERS)
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DF1K = DF1
      DO K = 2, NSOIL

C       CALC THIS SOIL LAYERS HEAT CAPACITY

        HCPCT = SH2O(K)*CH2O +(1.0-SMCMAX)*CSOIL +(SMCMAX-SMC(K))*CAIR
     +        + ( SMC(K) - SH2O(K) )*CICE
C
        IF ( K .NE. NSOIL ) THEN

C+++++++ THIS SECTION FOR LAYER 2 OR GREATER, BUT NOT LAST LAYER +++++

C CALCULATE THERMAL DIFFUSIVITY FOR THIS LAYER

           CALL TDFCND ( DF1N, SMC(K),QUARTZ,SMCMAX,SH2O(K))

C CALC THE VERTICAL SOIL TEMP GRADIENT THRU THIS LAYER

           DENOM = 0.5 * ( ZSOIL(K-1) - ZSOIL(K+1) )
           DTSDZ2 = ( STC(K) - STC(K+1) ) / DENOM

C CALC THE MATRIX COEF, CI, AFTER CALCNG ITS PARTIAL PRODUCT

           DDZ2 = 2. / (ZSOIL(K-1) - ZSOIL(K+1))
           CI(K) = -DF1N * DDZ2 / ((ZSOIL(K-1) - ZSOIL(K)) * HCPCT)

C CALCULATE TEMP AT BOTTOM OF LAYER

           CALL TBND ( STC(K),STC(K+1),ZSOIL,ZBOT,K,NSOIL,TBK1 )

        ELSE
C+++++++++++++ SPECIAL CASE OF BOTTOM SOIL LAYER +++++++++++++++++++++

C CALCULATE THERMAL DIFFUSIVITY FOR THIS LAYER

           CALL TDFCND ( DF1N, SMC(K),QUARTZ,SMCMAX,SH2O(K))

C CALC THE VERTICAL SOIL TEMP GRADIENT THRU THIS LAYER

           DENOM = .5 * (ZSOIL(K-1) + ZSOIL(K)) - ZBOT
           DTSDZ2 = (STC(K)-TBOT) / DENOM

C....SET MATRIX COEF, CI TO ZERO IF BOTTOM LAYER 

           CI(K) = 0.

C CALCULATE TEMP AT BOTTOM OF LAST LAYER

           CALL TBND ( STC(K), TBOT, ZSOIL, ZBOT, K, NSOIL,TBK1 )

        END IF
C+++++++++++++ THIS ENDS SPECIAL CODE FOR BOTTOM LAYER +++++++++

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CALC RHSTS FOR THIS LAYER AFTER CALCNG A PARTIAL PRODUCT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        DENOM = ( ZSOIL(K) - ZSOIL(K-1) ) * HCPCT
        RHSTS(K) = ( DF1N * DTSDZ2 - DF1K * DTSDZ ) / DENOM

        QTOT = -1.0*DENOM*RHSTS(K)

        SICE = SMC(K) - SH2O(K)

      IF ( (SICE .GT. 0.) .OR. (TBK .LT. T0) .OR.
     &     (STC(K) .LT. T0) .OR. (TBK1 .LT. T0) ) THEN

       TSNSR = SNKSRC ( TBK, STC(K),TBK1, SMC(K), SH2O(K), 
     +           ZSOIL, NSOIL, SMCMAX, PSISAT, B, DT, K, QTOT)

       RHSTS(K) = RHSTS(K) - TSNSR / DENOM

      ENDIF 
C -------------------------------------------------------------------
      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CALC MATRIX COEFS, AI, AND BI FOR THIS LAYER.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        AI(K) = - DF1 * DDZ / ((ZSOIL(K-1) - ZSOIL(K)) * HCPCT)
        BI(K) = -(AI(K) + CI(K))

C RESET VALUES OF DF1, DTSDZ, DDZ, AND TBK FOR LOOP TO NEXT SOIL LYR

        TBK   = TBK1
        DF1K  = DF1N
        DTSDZ = DTSDZ2
        DDZ   = DDZ2
C   
      END DO

      RETURN
      END
      SUBROUTINE HRTICE (RHSTS,STC,NSOIL,ZSOIL,YY,ZZ1,DF1)

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE THE RIGHT HAND SIDE OF THE TIME TENDENCY
CC    =======   TERM OF THE SOIL THERMAL DIFFUSION EQUATION IN THE CASE
CC              OF SEA-ICE PACK.  ALSO TO COMPUTE ( PREPARE ) THE
CC              MATRIX COEFFICIENTS FOR THE TRI-DIAGONAL MATRIX OF 
CC              THE IMPLICIT TIME SCHEME.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      INTEGER NSOLD
      PARAMETER ( NSOLD = 20 )

      INTEGER K
      INTEGER NSOIL

      REAL AI    ( NSOLD )
      REAL BI    ( NSOLD )
      REAL CI    ( NSOLD )

      REAL DDZ
      REAL DDZ2
      REAL DENOM
      REAL DF1
      REAL DTSDZ
      REAL DTSDZ2
      REAL HCPCT
      REAL RHSTS ( NSOIL )
      REAL S
      REAL STC   ( NSOIL )
      REAL TBOT
      REAL YY
      REAL ZBOT
      REAL ZSOIL ( NSOIL )
      REAL ZZ1
C
      COMMON /ABCI/ AI, BI, CI

C THE INPUT ARGUMENT DF1 A UNIVERSALLY CONSTANT VALUE OF
C SEA-ICE THERMAL DIFFUSIVITY, SET IN ROUTINE SNOPAC AS
C  DF1 = 2.2

C SET LOWER BOUNDARY DEPTH AND BOUNDARY TEMPERATURE OF 
C UNFROZEN SEA WATER AT BOTTOM OF SEA ICE PACK.  ASSUME 
C ICE PACK IS OF NSOIL LAYERS SPANNING A UNIFORM CONSTANT
C ICE PACK THICKNESS AS DEFINED IN ROUTINE SFLX

      ZBOT = ZSOIL(NSOIL)
      TBOT = 271.16

C SET A NOMINAL UNIVERSAL VALUE OF THE SEA-ICE SPECIFIC HEAT CAPACITY
      
      HCPCT=1880.0*917.0

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALC THE MATRIX COEFFICIENTS AI, BI, AND CI FOR THE TOP LAYER
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DDZ = 1.0 / ( -0.5 * ZSOIL(2) )
      AI(1) = 0.0
      CI(1) =  ( DF1 * DDZ ) / ( ZSOIL(1) * HCPCT )
      BI(1) = -CI(1) + DF1/( 0.5 * ZSOIL(1) * ZSOIL(1) * HCPCT * ZZ1)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALC THE VERTICAL SOIL TEMP GRADIENT BTWN THE TOP AND 2ND SOIL
C     LAYERS.  RECALC/ADJUST THE SOIL HEAT FLUX.  USE THE GRADIENT
C     AND FLUX TO CALC RHSTS FOR THE TOP SOIL LAYER.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DTSDZ = ( STC(1) - STC(2) ) / ( -0.5 * ZSOIL(2) )
      S = DF1 * ( STC(1) - YY ) / ( 0.5 * ZSOIL(1) * ZZ1 )
      RHSTS(1) = ( DF1 * DTSDZ - S ) / ( ZSOIL(1) * HCPCT )

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     INITIALIZE DDZ2
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DDZ2 = 0.0

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     LOOP THRU THE REMAINING SOIL LAYERS, REPEATING THE ABOVE PROCESS
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DO K = 2, NSOIL

        IF ( K .NE. NSOIL ) THEN

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C         CALC THE VERTICAL SOIL TEMP GRADIENT THRU THIS LAYER.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

          DENOM = 0.5 * ( ZSOIL(K-1) - ZSOIL(K+1) )
          DTSDZ2 = ( STC(K) - STC(K+1) ) / DENOM

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C         CALC THE MATRIX COEF, CI, AFTER CALCNG ITS PARTIAL PRODUCT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

          DDZ2 = 2. / (ZSOIL(K-1) - ZSOIL(K+1))
          CI(K) = -DF1 * DDZ2 / ((ZSOIL(K-1) - ZSOIL(K)) * HCPCT)

        ELSE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C         CALC THE VERTICAL SOIL TEMP GRADIENT THRU THE LOWEST LAYER
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

          DTSDZ2 = (STC(K)-TBOT)/(.5 * (ZSOIL(K-1) + ZSOIL(K))-ZBOT)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C         SET MATRIX COEF, CI TO ZERO
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

          CI(K) = 0.
        END IF

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CALC RHSTS FOR THIS LAYER AFTER CALCNG A PARTIAL PRODUCT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        DENOM = ( ZSOIL(K) - ZSOIL(K-1) ) * HCPCT
        RHSTS(K) = ( DF1 * DTSDZ2 - DF1 * DTSDZ ) / DENOM

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CALC MATRIX COEFS, AI, AND BI FOR THIS LAYER.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        AI(K) = - DF1 * DDZ / ((ZSOIL(K-1) - ZSOIL(K)) * HCPCT)
        BI(K) = -(AI(K) + CI(K))

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       RESET VALUES OF DTSDZ AND DDZ FOR LOOP TO NEXT SOIL LYR
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        DTSDZ = DTSDZ2
        DDZ   = DDZ2

      END DO

      RETURN
      END
      SUBROUTINE HSTEP ( STCOUT, STCIN, RHSTS, DT, NSOIL )

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE/UPDATE THE SOIL TEMPERATURE FIELD.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      INTEGER NSOLD
      PARAMETER ( NSOLD = 20 )

      INTEGER K
      INTEGER NSOIL

      REAL AI    ( NSOLD )
      REAL BI    ( NSOLD )
      REAL CI    ( NSOLD )
      REAL CIin  ( NSOLD )
      REAL DT
      REAL RHSTS   ( NSOIL )
      REAL RHSTSin ( NSOIL )
      REAL STCOUT  ( NSOIL )
      REAL STCIN   ( NSOIL )
     
C
      COMMON /ABCI/ AI, BI, CI

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CREATE FINITE DIFFERENCE VALUES FOR USE IN ROSR12 ROUTINE
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DO K = 1 , NSOIL
        RHSTS(K) = RHSTS(K) * DT
        AI(K) = AI(K) * DT
        BI(K) = 1. + BI(K) * DT
        CI(K) = CI(K) * DT
      END DO

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     COPY VALUES FOR INPUT VARIABLES BEFORE CALL TO ROSR12
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      DO K = 1 , NSOIL
         RHSTSin(K) = RHSTS(K)
      END DO
      DO K = 1 , NSOLD
         CIin(K) = CI(K)
      END DO
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     SOLVE THE TRI-DIAGONAL MATRIX EQUATION
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      CALL ROSR12 ( CI,AI,BI,CIin,RHSTSin,RHSTS,NSOIL )

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALC/UPDATE THE SOIL TEMPS USING MATRIX SOLUTION
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DO K = 1 , NSOIL
         STCOUT(K) = STCIN(K) + CI(K)
      END DO

      RETURN
      END
      SUBROUTINE NOPAC ( ETP, ETA, PRCP, SMC, SMCMAX, SMCWLT,
     &                   SMCREF,SMCDRY,CMC,CMCMAX, NSOIL, DT, SHDFAC,
     &                   SBETA,
     &                   Q1, Q2, T1, SFCTMP, T24, TH2, F, F1, S, STC,
     &                   EPSCA, B, PC, RCH, RR,  CFACTR, 
     &                   SH2O, SLOPE, KDT, FRZFACT, PSISAT, ZSOIL,
     &                   DKSAT, DWSAT, TBOT, ZBOT, RUNOFF1, RUNOFF2,
     &                   RUNOFF3, EDIR1, EC1, ETT1, NROOT, ICE,RTDIS,
     &                   QUARTZ, FXEXP,CSOIL)


      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE SOIL MOISTURE AND HEAT FLUX VALUES AND UPDATE
CC    =======   SOIL MOISTURE CONTENT AND SOIL HEAT CONTENT VALUES FOR
CC              THE CASE WHEN NO SNOW PACK IS PRESENT.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      INTEGER ICE
      INTEGER NROOT
      INTEGER NSOIL

      REAL B
      REAL BETA
      REAL CFACTR
      REAL CMC
      REAL CMCMAX
      REAL CP
      REAL CSOIL
      REAL DEW
      REAL DF1
      REAL DKSAT
      REAL DRIP
      REAL DT
      REAL DWSAT
      REAL EC
      REAL EDIR
      REAL EPSCA
      REAL ETA
      REAL ETA1
      REAL ETP
      REAL ETP1
      REAL ETT
      REAL F
      REAL F1
      REAL FXEXP
      REAL FLX1
      REAL FLX2
      REAL FLX3
      REAL KDT
      REAL PC
      REAL PRCP
      REAL PRCP1
      REAL Q2
      REAL RCH
      REAL RIB
      REAL RR
      REAL RTDIS (NSOIL)
      REAL RUNOFF,RUNOXX3
      REAL S
      REAL SBETA
      REAL SFCTMP
      REAL SHDFAC
      REAL SIGMA
      REAL SMC   ( NSOIL )
      REAL SH2O  ( NSOIL )
      REAL SMCDRY
      REAL SMCMAX
      REAL SMCREF
      REAL SMCWLT
      REAL STC   ( NSOIL )
      REAL T1
      REAL T24
      REAL TBOT
      REAL ZBOT
      REAL TH2
      REAL YY
      REAL YYNUM
      REAL ZSOIL ( NSOIL )
      REAL ZZ1

      REAL Q1, SLOPE, FRZFACT, PSISAT, RUNOFF1, RUNOFF2, RUNOFF3
      REAL EDIR1, EC1, ETT1, QUARTZ

      COMMON/RITE/ BETA,DRIP,EC,EDIR,ETT,FLX1,FLX2,FLX3,RUNOFF,
     &             DEW,RIB,RUNOXX3

      PARAMETER(CP=1004.5, SIGMA=5.67E-8)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     EXECUTABLE CODE BEGINS HERE.....
C     CONVERT ETP FROM KG M-2 S-1 TO MS-1 AND INITIALIZE DEW.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      PRCP1 = PRCP * 0.001
      ETP1 = ETP * 0.001
      DEW = 0.0

      IF ( ETP .GT. 0.0 ) THEN

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CONVERT PRCP FROM  KG M-2 S-1  TO  M S-1
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

           CALL SMFLX ( ETA1,SMC,NSOIL,CMC,ETP1,DT,PRCP1,ZSOIL,
     +          SH2O, SLOPE, KDT, FRZFACT,
     &          SMCMAX,B,PC,SMCWLT,DKSAT,DWSAT,SMCREF,SHDFAC,
     &          CMCMAX,SMCDRY,CFACTR, RUNOFF1,RUNOFF2, RUNOFF3, 
     &          EDIR1, EC1, ETT1, SFCTMP,Q2,NROOT,RTDIS, FXEXP)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CONVERT MODELED EVAPOTRANSPIRATION FM  M S-1  TO  KG M-2 S-1
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        ETA = ETA1 * 1000.0

      ELSE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       IF ETP < 0, ASSUME DEW FORMS (TRANSFORM ETP1 INTO DEW
C       AND REINITIALIZE ETP1 TO ZERO)
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        DEW = -ETP1
        ETP1 = 0.0

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CONVERT PRCP FROM  KG M-2 S-1  TO  M S-1  AND ADD DEW AMT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        PRCP1 = PRCP1 + DEW
C
      CALL SMFLX ( ETA1,SMC,NSOIL,CMC,ETP1,DT,PRCP1,ZSOIL,
     +          SH2O, SLOPE, KDT, FRZFACT,
     &          SMCMAX,B,PC,SMCWLT,DKSAT,DWSAT,SMCREF,SHDFAC,
     &          CMCMAX,SMCDRY,CFACTR, RUNOFF1,RUNOFF2, RUNOFF3, 
     &          EDIR1, EC1, ETT1, SFCTMP, Q2, NROOT,RTDIS, FXEXP)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CONVERT MODELED EVAPOTRANSPIRATION FM  M S-1  TO  KG M-2 S-1
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

        ETA = ETA1 * 1000.0

      ENDIF

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     BASED ON ETP AND E VALUES, DETERMINE BETA
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      IF ( ETP .LE. 0.0 ) THEN
        BETA = 0.0
        IF ( ETP .LT. 0.0 ) THEN
          BETA = 1.0
          ETA = ETP
        ENDIF
      ELSE
        BETA = ETA / ETP
      ENDIF

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C    GET SOIL THERMAL DIFFUXIVITY/CONDUCTIVITY FOR TOP SOIL LYR,
C    CALC. ADJUSTED TOP LYR SOIL TEMP AND ADJUSTED SOIL FLUX, THEN
C    CALL SHFLX TO COMPUTE/UPDATE SOIL HEAT FLUX AND SOIL TEMPS.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      CALL TDFCND ( DF1, SMC(1),QUARTZ,SMCMAX,SH2O(1) )

C VEGETATION GREENNESS FRACTION REDUCTION IN SUBSURFACE HEAT FLUX 
C VIA REDUCTION FACTOR, WHICH IS CONVENIENT TO APPLY HERE TO THERMAL 
C DIFFUSIVITY THAT IS LATER USED IN HRT TO COMPUTE SUB SFC HEAT FLUX
C (SEE ADDITIONAL COMMENTS ON VEG EFFECT SUB-SFC HEAT FLX IN 
C  ROUTINE SFLX)

      DF1 = DF1 * EXP(SBETA*SHDFAC)

C COMPUTE INTERMEDIATE TERMS PASSED TO ROUTINE HRT (VIA ROUTINE 
C SHFLX BELOW) FOR USE IN COMPUTING SUBSURFACE HEAT FLUX IN HRT

      YYNUM = F - SIGMA * T24
      YY = SFCTMP + (YYNUM/RCH+TH2-SFCTMP-BETA*EPSCA) / RR
      ZZ1 = DF1 / ( -0.5 * ZSOIL(1) * RCH * RR ) + 1.0

      CALL SHFLX ( S,STC,SMC,SMCMAX,NSOIL,T1,DT,YY,ZZ1,ZSOIL,TBOT,
     +             ZBOT, SMCWLT, PSISAT, SH2O,
     &             B,F1,DF1, ICE, 
     &             QUARTZ,CSOIL)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     SET FLX1, AND FLX3 TO ZERO SINCE THEY ARE NOT USED.  FLX2
C     WAS SIMILARLY INITIALIZED IN THE PENMAN ROUTINE.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      FLX1 = 0.0
      FLX3 = 0.0
C
      RETURN
      END
      SUBROUTINE PENMAN(SFCTMP,SFCPRS,CH,T2V,TH2,PRCP,F,T24,S,Q2,
     &                  Q2SAT,ETP,RCH,EPSCA,RR,SNOWNG,FRZGRA,DQSDT2)

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE POTENTIAL EVAPORATION FOR THE CURRENT POINT.
CC    =======   VARIOUS PARTIAL SUMS/PRODUCTS ARE ALSO CALCULATED AND
CC              PASSED BACK TO THE CALLING ROUTINE FOR LATER USE.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      LOGICAL SNOWNG
      LOGICAL FRZGRA

      REAL A
      REAL BETA
      REAL CH
      REAL CP
      REAL CPH2O
      REAL CPICE
      REAL DELTA
      REAL DEW
      REAL DRIP
      REAL EC
      REAL EDIR
      REAL ELCP
      REAL EPSCA
      REAL ETP
      REAL ETT
      REAL F
      REAL FLX1
      REAL FLX2
      REAL FLX3
      REAL FNET
      REAL LSUBC
      REAL LSUBF
      REAL PRCP
      REAL Q2
      REAL Q2SAT
      REAL R
      REAL RAD
      REAL RCH
      REAL RHO
      REAL RIB
      REAL RR
      REAL RUNOFF,RUNOXX3
      REAL S
      REAL SFCPRS
      REAL SFCTMP
      REAL SIGMA
      REAL T24
      REAL T2V
      REAL TH2
      REAL DQSDT2

      COMMON/RITE/ BETA,DRIP,EC,EDIR,ETT,FLX1,FLX2,FLX3,RUNOFF,
     &             DEW,RIB,RUNOXX3

      PARAMETER(CP=1004.6,CPH2O=4.218E+3,CPICE=2.106E+3,R=287.04,
     &   ELCP=2.4888E+3,LSUBF=3.335E+5,LSUBC=2.501000E+6,SIGMA=5.67E-8)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     EXECUTABLE CODE BEGINS HERE...
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      FLX2 = 0.0

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     PREPARE PARTIAL QUANTITIES FOR PENMAN EQUATION.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DELTA = ELCP * DQSDT2
      T24 = SFCTMP * SFCTMP * SFCTMP * SFCTMP
      RR = T24 * 6.48E-8 / ( SFCPRS * CH ) + 1.0
      RHO = SFCPRS / ( R * T2V )
      RCH = RHO * CP * CH

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     ADJUST THE PARTIAL SUMS / PRODUCTS WITH THE LATENT HEAT
C     EFFECTS CAUSED BY FALLING PRECIPITATION.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      IF ( .NOT. SNOWNG ) THEN
        IF ( PRCP .GT. 0.0 ) RR = RR + CPH2O * PRCP / RCH
      ELSE
        RR = RR + CPICE * PRCP / RCH
      ENDIF

      FNET = F - SIGMA * T24 - S

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     INCLUDE THE LATENT HEAT EFFECTS OF FRZNG RAIN CONVERTING TO
C     ICE ON IMPACT IN THE CALCULATION OF FLX2 AND FNET.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      IF ( FRZGRA ) THEN
        FLX2 = -LSUBF * PRCP
        FNET = FNET - FLX2
      ENDIF

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     FINISH PENMAN EQUATION CALCULATIONS.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      RAD = FNET / RCH + TH2 - SFCTMP
      A = ELCP * ( Q2SAT - Q2 )
      EPSCA = ( A * RR + RAD * DELTA ) / ( DELTA + RR )
      ETP = EPSCA * RCH / LSUBC

      RETURN
      END
      SUBROUTINE REDPRM(VEGTYP, SOILTYP, SLOPETYP,
     +     CFACTR, CMCMAX, RSMAX, TOPT, REFKDT, KDT, SBETA,
     +     SHDFAC, RCMIN, RGL, HS, ZBOT, FRZX, PSISAT, SLOPE,
     +     SNUP, SALP, B, DKSAT, DWSAT, SMCMAX, SMCWLT, SMCREF,
     +     SMCDRY, F1, QUARTZ, FXEXP, RTDIS, SLDPTH, ZSOIL,
     +     NROOT, NSOIL, Z0, CZIL, LAI, CSOIL, PTU)


      IMPLICIT NONE

C  This subroutine internally sets (defaults), or optionally reads-in
c  via namelist I/O, all the soil and vegetation parameters
C  required for the execusion of the NOAH - LSM
c 
c optional non-default parameters can be read in, accommodating up
C  to 30 soil, veg, or slope classes, if the default max number of 
C  soil, veg, and/or slope types is reset.

c future upgrades of routine REDPRM must expand to incorporate some
c of the empirical parameters of the frozen soil and snowpack physics
c (such as in routines FRH2O, SNOWPACK, and SNOW_NEW) not yet set in 
c  this REDPRM routine, but rather set in lower level subroutines

C  Set maximum number of soil-, veg-, and slopetyp in data statement

      INTEGER MAX_SOILTYP
      INTEGER MAX_VEGTYP
      INTEGER MAX_SLOPETYP
      PARAMETER (MAX_SOILTYP  = 30)
      PARAMETER (MAX_VEGTYP   = 30)
      PARAMETER (MAX_SLOPETYP = 30)

C  Number of defined soil-, veg-, and slopetyps used

      INTEGER DEFINED_VEG
      INTEGER DEFINED_SOIL
      INTEGER DEFINED_SLOPE
      DATA DEFINED_VEG/13/
      DATA DEFINED_SOIL/9/
      DATA DEFINED_SLOPE/9/

C  SET-UP SOIL PARAMETERS FOR GIVEN SOIL TYPE
C  INPUT: SOLTYP: SOIL TYPE (INTEGER INDEX)
C  OUTPUT: SOIL PARAMETERS:

C    MAXSMC: MAX SOIL MOISTURE CONTENT (POROSITY)
C    REFSMC: REFERENCE SOIL MOISTURE (ONSET OF SOIL MOISTURE
C            STRESS IN TRANSPIRATION)
C    WLTSMC: WILTING PT SOIL MOISTURE CONTENTS
C    DRYSMC: AIR DRY SOIL MOIST CONTENT LIMITS
C    SATPSI: SATURATED SOIL POTENTIAL
C    SATDK:  SATURATED SOIL HYDRAULIC CONDUCTIVITY
C    BB:     THE 'B' PARAMETER
C    SATDW:  SATURATED SOIL DIFFUSIVITY
C    F11:    USED TO COMPUTE SOIL DIFFUSIVITY/CONDUCTIVITY
C    QUARTZ:  SOIL QUARTZ CONTENT
C
C SOIL TYPES   ZOBLER (1986)      COSBY ET AL (1984) (quartz cont.(1))
C  1        COARSE            LOAMY SAND         (0.82)
C  2        MEDIUM            SILTY CLAY LOAM    (0.10)
C  3        FINE              LIGHT CLAY         (0.25)
C  4        COARSE-MEDIUM     SANDY LOAM         (0.60)
C  5        COARSE-FINE       SANDY CLAY         (0.52)
C  6        MEDIUM-FINE       CLAY LOAM          (0.35)
C  7        COARSE-MED-FINE   SANDY CLAY LOAM    (0.60)
C  8        ORGANIC           LOAM               (0.40)
C  9        GLACIAL LAND ICE  LOAMY SAND         (NA using 0.82)

      REAL BB(MAX_SOILTYP)
      REAL DRYSMC(MAX_SOILTYP)
      REAL F11(MAX_SOILTYP)
      REAL MAXSMC(MAX_SOILTYP)
      REAL REFSMC(MAX_SOILTYP)
      REAL SATPSI(MAX_SOILTYP)
      REAL SATDK(MAX_SOILTYP)
      REAL SATDW(MAX_SOILTYP)
      REAL WLTSMC(MAX_SOILTYP)
      REAL QTZ(MAX_SOILTYP)

      REAL B
      REAL DKSAT
      REAL DWSAT
      REAL SMCMAX
      REAL SMCWLT
      REAL SMCREF
      REAL SMCDRY
      REAL PTU
      REAL F1
      REAL QUARTZ
      REAL REFSMC1
      REAL WLTSMC1

      DATA MAXSMC/0.421, 0.464, 0.468, 0.434, 0.406, 0.465,
     &            0.404, 0.439, 0.421, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000/
      DATA SATPSI/0.04, 0.62, 0.47, 0.14, 0.10, 0.26,
     &            0.14, 0.36, 0.04, 0.00, 0.00, 0.00,
     &            0.00, 0.00, 0.00, 0.00, 0.00, 0.00,
     &            0.00, 0.00, 0.00, 0.00, 0.00, 0.00,
     &            0.00, 0.00, 0.00, 0.00, 0.00, 0.00/
      DATA SATDK /1.41E-5, 0.20E-5, 0.10E-5, 0.52E-5, 0.72E-5,
     &            0.25E-5, 0.45E-5, 0.34E-5, 1.41E-5, 0.00,
     &            0.00   , 0.00   , 0.00   , 0.00   , 0.00,
     &            0.00   , 0.00   , 0.00   , 0.00   , 0.00,
     &            0.00   , 0.00   , 0.00   , 0.00   , 0.00,
     &            0.00   , 0.00   , 0.00   , 0.00   , 0.00/
      DATA BB    /4.26,  8.72, 11.55, 4.74, 10.73,  8.17,
     &            6.77,  5.25,  4.26, 0.00,  0.00,  0.00,
     &            0.00,  0.00,  0.00, 0.00,  0.00,  0.00,
     &            0.00,  0.00,  0.00, 0.00,  0.00,  0.00,
     &            0.00,  0.00,  0.00, 0.00,  0.00,  0.00/
      DATA QTZ   /0.82, 0.10, 0.25, 0.60, 0.52, 0.35,
     &            0.60, 0.40, 0.82, 0.00, 0.00, 0.00,
     &            0.00, 0.00, 0.00, 0.00, 0.00, 0.00,
     &            0.00, 0.00, 0.00, 0.00, 0.00, 0.00,
     &            0.00, 0.00, 0.00, 0.00, 0.00, 0.00/

C The following 5 parameters are derived later in REDPRM.f 
C from the soil data, and are just given here for reference 
C and to force static storage allocation
C Dag Lohmann, Feb. 2001

      DATA REFSMC/0.283, 0.387, 0.412, 0.312, 0.338, 0.382,
     &            0.315, 0.329, 0.283, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000/
      DATA WLTSMC/0.029, 0.119, 0.139, 0.047, 0.100, 0.103,
     &            0.069, 0.066, 0.029, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000/
      DATA DRYSMC/0.029, 0.119, 0.139, 0.047, 0.100, 0.103,
     &            0.069, 0.066, 0.029, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
     &            0.000, 0.000, 0.000, 0.000, 0.000, 0.000/
      DATA SATDW /5.71E-6, 2.33E-5, 1.16E-5, 7.95E-6, 1.90E-5,
     &            1.14E-5, 1.06E-5, 1.46E-5, 5.71E-6, 0.00,
     &            0.00   , 0.00   , 0.00   , 0.00   , 0.00,
     &            0.00   , 0.00   , 0.00   , 0.00   , 0.00,
     &            0.00   , 0.00   , 0.00   , 0.00   , 0.00,
     &            0.00   , 0.00   , 0.00   , 0.00   , 0.00/
      DATA F11  /-0.999, -1.116, -2.137, -0.572, -3.201, -1.302,
     &           -1.519, -0.329, -0.999,  0.000,  0.000,  0.000,
     &            0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
     &            0.000,  0.000,  0.000,  0.000,  0.000,  0.000,
     &            0.000,  0.000,  0.000,  0.000,  0.000,  0.000/

C#######################################################################

C  SET-UP VEGETATION PARAMETERS FOR A GIVEN VEGETAION TYPE
C
C  INPUT: VEGTYP = VEGETATION TYPE (INTEGER INDEX)
C  OUPUT: VEGETATION PARAMETERS
C         SHDFAC: VEGETATION GREENNESS FRACTION
C         RCMIN:  MIMIMUM STOMATAL RESISTANCE
C         RGL:    PARAMETER USED IN SOLAR RAD TERM OF
C                 CANOPY RESISTANCE FUNCTION
C         HS:     PARAMETER USED IN VAPOR PRESSURE DEFICIT TERM OF
C                 CANOPY RESISTANCE FUNCTION
C         SNUP:   THRESHOLD SNOW DEPTH (IN WATER EQUIVALENT M) THAT
C                 IMPLIES 100% SNOW COVER
C
C  SSIB VEGETATION TYPES (DORMAN AND SELLERS, 1989; JAM)
C
C   1:   BROADLEAF-EVERGREEN TREES  (TROPICAL FOREST)
C   2:   BROADLEAF-DECIDUOUS TREES
C   3:   BROADLEAF AND NEEDLELEAF TREES (MIXED FOREST)
C   4:   NEEDLELEAF-EVERGREEN TREES
C   5:   NEEDLELEAF-DECIDUOUS TREES (LARCH)
C   6:   BROADLEAF TREES WITH GROUNDCOVER (SAVANNA)
C   7:   GROUNDCOVER ONLY (PERENNIAL)
C   8:   BROADLEAF SHRUBS WITH PERENNIAL GROUNDCOVER
C   9:   BROADLEAF SHRUBS WITH BARE SOIL
C  10:   DWARF TREES AND SHRUBS WITH GROUNDCOVER (TUNDRA)
C  11:   BARE SOIL
C  12:   CULTIVATIONS (THE SAME PARAMETERS AS FOR TYPE 7)
C  13:   GLACIAL (THE SAME PARAMETERS AS FOR TYPE 11)

      INTEGER NROOT_DATA(MAX_VEGTYP)
      REAL    RSMTBL(MAX_VEGTYP)
      REAL    RGLTBL(MAX_VEGTYP)
      REAL    HSTBL(MAX_VEGTYP)
      REAL    SNUPX(MAX_VEGTYP)
      REAL    Z0_DATA(MAX_VEGTYP)
      REAL    LAI_DATA(MAX_VEGTYP)

      INTEGER NROOT
      REAL    SHDFAC
      REAL    RCMIN
      REAL    RGL
      REAL    HS
      REAL    FRZFACT
      REAL    PSISAT
      REAL    SNUP
      REAL    Z0
      REAL    LAI

      DATA NROOT_DATA /4,4,4,4,4,4,3,3,3,2,3,3,2,0,0,
     *                 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0/
      DATA RSMTBL /150.0, 100.0, 125.0, 150.0, 100.0, 70.0,
     *              40.0, 300.0, 400.0, 150.0, 400.0, 40.0,
     *             150.0,   0.0,   0.0,   0.0,   0.0,  0.0,
     *               0.0,   0.0,   0.0,   0.0,   0.0,  0.0,
     *               0.0,   0.0,   0.0,   0.0,   0.0,  0.0/
      DATA RGLTBL /30.0,  30.0,  30.0,  30.0,  30.0,  65.0,
     *            100.0, 100.0, 100.0, 100.0, 100.0, 100.0,
     *            100.0,   0.0,   0.0,   0.0,   0.0,   0.0,
     *              0.0,   0.0,   0.0,   0.0,   0.0,   0.0,
     *              0.0,   0.0,   0.0,   0.0,   0.0,   0.0/
      DATA HSTBL /41.69, 54.53, 51.93, 47.35,  47.35, 54.53,
     *            36.35, 42.00, 42.00, 42.00,  42.00, 36.35,
     *            42.00,  0.00,  0.00,  0.00,   0.00,  0.00,
     *             0.00,  0.00,  0.00,  0.00,   0.00,  0.00,
     *             0.00,  0.00,  0.00,  0.00,   0.00,  0.00/
      DATA SNUPX  /0.080, 0.080, 0.080, 0.080, 0.080, 0.080,
     *             0.040, 0.040, 0.040, 0.040, 0.025, 0.040,
     *             0.025, 0.000, 0.000, 0.000, 0.000, 0.000,
     *             0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
     *             0.000, 0.000, 0.000, 0.000, 0.000, 0.000/
      DATA Z0_DATA /2.653, 0.826, 0.563, 1.089, 0.854, 0.856,
     *              0.035, 0.238, 0.065, 0.076, 0.011, 0.035,
     *              0.011, 0.000, 0.000, 0.000, 0.000, 0.000,
     *              0.000, 0.000, 0.000, 0.000, 0.000, 0.000,
     *              0.000, 0.000, 0.000, 0.000, 0.000, 0.000/
c      DATA LAI_DATA /3.0, 3.0, 3.0, 3.0, 3.0, 3.0,
c     *               3.0, 3.0, 3.0, 3.0, 3.0, 3.0,
c     *               3.0, 0.0, 0.0, 0.0, 0.0, 0.0,
c     *               0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
c     *               0.0, 0.0, 0.0, 0.0, 0.0, 0.0/
      DATA LAI_DATA /4.0, 4.0, 4.0, 4.0, 4.0, 4.0,
     *               4.0, 4.0, 4.0, 4.0, 4.0, 4.0,
     *               4.0, 0.0, 0.0, 0.0, 0.0, 0.0,
     *               0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
     *               0.0, 0.0, 0.0, 0.0, 0.0, 0.0/

C#######################################################################

C  CLASS PARAMETER 'SLOPETYP' WAS INCLUDED TO ESTIMATE
C  LINEAR RESERVOIR COEFFICIENT 'SLOPE' TO THE BASEFLOW RUNOFF
C  OUT OF THE BOTTOM LAYER. LOWEST CLASS (SLOPETYP=0)MEANS
C  HIGHEST SLOPE PARAMETER= 1
C  DEFINITION OF SLOPETYP FROM 'ZOBLER' SLOPE TYPE
C  SLOPE CLASS      PERCENT SLOPE
C  1                0-8
C  2                8-30
C  3                > 30
C  4                0-30
C  5                0-8 & > 30
C  6                8-30 & > 30
C  7                0-8, 8-30, > 30
C  9                GLACIAL ICE
C  BLANK            OCEAN/SEA
C  NOTE:  CLASS 9 FROM 'ZOBLER' FILE SHOULD BE REPLACED BY 8
C  AND 'BLANK'  9

      REAL SLOPE
      REAL SLOPE_DATA(MAX_SLOPETYP)
      DATA SLOPE_DATA /0.1,  0.6, 1.0, 0.35, 0.55, 0.8,
     *                 0.63, 0.0, 0.0, 0.0,  0.0,  0.0,
     *                 0.0 , 0.0, 0.0, 0.0,  0.0,  0.0,
     *                 0.0 , 0.0, 0.0, 0.0,  0.0,  0.0,
     *                 0.0 , 0.0, 0.0, 0.0,  0.0,  0.0/

C#######################################################################

C  Set namelist file name

      CHARACTER*50 NAMELIST_NAME

C#######################################################################

C SET UNIVERSAL PARAMETERS (NOT DEPENDENT ON SOIL, VEG, SLOPE TYPE)

      INTEGER VEGTYP
      INTEGER SOILTYP
      INTEGER SLOPETYP

      INTEGER NSOIL
      INTEGER I

      INTEGER BARE
      DATA    BARE /11/

      LOGICAL LPARAM
      DATA    LPARAM /.TRUE./

      LOGICAL LFIRST
      DATA    LFIRST /.TRUE./

C  Parameter used to calculate roughness length of heat
      REAL CZIL, CZIL_DATA
      DATA CZIL_DATA /0.2/

C  Parameter used to caluculate vegetation effect on soil heat flux
      REAL SBETA, SBETA_DATA
      DATA SBETA_DATA /-2.0/

C BARE SOIL EVAPORATION EXPONENT USED IN DEVAP

      REAL FXEXP, FXEXP_DATA
      DATA FXEXP_DATA /2.0/

C Soil heat capacity [J/m^3/K]

      REAL CSOIL, CSOIL_DATA
      DATA CSOIL_DATA /1.26E+6/

C  SPECIFY SNOW DISTRIBUTION SHAPE PARAMETER
C  SALP   - SHAPE PARAMETER OF DISTRIBUTION FUNCTION
C  OF SNOW COVER. FROM ANDERSONS DATA (HYDRO-17)
C  BEST FIT IS WHEN SALP = 2.6
      REAL SALP, SALP_DATA
      DATA SALP_DATA /2.6/

C  KDT IS DEFINED BY REFERENCE REFKDT AND DKSAT
C  REFDK=2.E-6 IS THE SAT. DK. VALUE FOR THE SOIL TYPE 2
      REAL REFDK, REFDK_DATA
      DATA REFDK_DATA /2.0E-6/

      REAL REFKDT, REFKDT_DATA
      DATA REFKDT_DATA /3.0/

      REAL KDT
      REAL FRZX

C  FROZEN GROUND PARAMETER, FRZK, DEFINITION
C  FRZK IS ICE CONTENT THRESHOLD ABOVE WHICH FROZEN SOIL IS IMPERMEABLE
C  REFERENCE VALUE OF THIS PARAMETER FOR THE LIGHT CLAY SOIL (TYPE=3)
C  FRZK = 0.15 M
      REAL FRZK, FRZK_DATA
      DATA FRZK_DATA /0.15/

      REAL RTDIS(NSOIL)
      REAL SLDPTH(NSOIL)
      REAL ZSOIL(NSOIL)

C  Set two canopy water parameters
      REAL CFACTR, CFACTR_DATA
      REAL CMCMAX, CMCMAX_DATA
      DATA CFACTR_DATA /0.5/
      DATA CMCMAX_DATA /0.5E-3/

C  Set max. stomatal resistance
      REAL RSMAX, RSMAX_DATA
      DATA RSMAX_DATA /5000.0/

C  Set optimum transpiration air temperature
      REAL TOPT, TOPT_DATA
      DATA TOPT_DATA /298.0/

C  Specify depth[m] of lower boundary soil temperature
      REAL ZBOT, ZBOT_DATA
      DATA ZBOT_DATA /-3.0/

C#######################################################################

C  Namelist definition

      NAMELIST /SOIL_VEG/ SLOPE_DATA, RSMTBL, RGLTBL, HSTBL, SNUPX,
     &     BB, DRYSMC, F11, MAXSMC, REFSMC, SATPSI, SATDK, SATDW,
     &     WLTSMC, QTZ, LPARAM, ZBOT_DATA, SALP_DATA, CFACTR_DATA,
     &     CMCMAX_DATA, SBETA_DATA, RSMAX_DATA, TOPT_DATA,
     &     REFDK_DATA, FRZK_DATA, BARE, DEFINED_VEG, DEFINED_SOIL,
     &     DEFINED_SLOPE, FXEXP_DATA, NROOT_DATA, REFKDT_DATA, Z0_DATA,
     &     CZIL_DATA, LAI_DATA, CSOIL_DATA

C  Read namelist file to override default parameters
C  only once.

      IF (LFIRST) THEN
         OPEN(58, FILE = 'namelist_filename.txt')
C NAMELIST_NAME must be 50 characters or less.
         READ(58,'(A)') NAMELIST_NAME
         CLOSE(58)
         WRITE(*,*) 'Namelist Filename is ', NAMELIST_NAME
Clnx         OPEN(59, FILE = NAMELIST_NAME)
 50      CONTINUE
Clnx            READ(59, SOIL_VEG, END=100)
Clnx         IF (LPARAM) GOTO 50
 100     CONTINUE
C         CLOSE(59)
c         WRITE(*,NML=SOIL_VEG)
         LFIRST = .FALSE.
         IF (DEFINED_SOIL .GT. MAX_SOILTYP) THEN
            WRITE(*,*) 'Warning: DEFINED_SOIL too large in namelist'
            STOP 222
         END IF
         IF (DEFINED_VEG .GT. MAX_VEGTYP) THEN
            WRITE(*,*) 'Warning: DEFINED_VEG too large in namelist'
            STOP 222
         END IF
         IF (DEFINED_SLOPE .GT. MAX_SLOPETYP) THEN
            WRITE(*,*) 'Warning: DEFINED_SLOPE too large in namelist'
            STOP 222
         END IF

         DO I = 1, DEFINED_SOIL
            SATDW(I)  = BB(I)*SATDK(I)*(SATPSI(I)/MAXSMC(I))
            F11(I)    = ALOG10(SATPSI(I)) + BB(I)*ALOG10(MAXSMC(I)) + 2.0
            REFSMC1   = MAXSMC(I)*(5.79E-9/SATDK(I))
     &                                    **(1.0/(2.0*BB(I)+3.0))
            REFSMC(I) = REFSMC1 + (MAXSMC(I)-REFSMC1) / 3.0
            WLTSMC1   = MAXSMC(I) * (200.0/SATPSI(I))**(-1.0/BB(I))
            WLTSMC(I) = WLTSMC1 - 0.5 * WLTSMC1
C Current version DRYSMC values that equate to WLTSMC
C Future version could let DRYSMC be independently set via namelist 
            DRYSMC(I) = WLTSMC(I)
         END DO

      END IF

      IF (SOILTYP .GT. DEFINED_SOIL) THEN
         WRITE(*,*) 'Warning: too many soil types'
         STOP 333
      END IF
      IF (VEGTYP .GT. DEFINED_VEG) THEN
         WRITE(*,*) 'Warning: too many veg types'
         STOP 333
      END IF
      IF (SLOPETYP .GT. DEFINED_SLOPE) THEN
         WRITE(*,*) 'Warning: too many slope types'
         STOP 333
      END IF

C  SET-UP UNIVERSAL PARAMETERS 
C (NOT DEPENDENT ON SOILTYP, VEGTYP OR SLOPETYP)
      ZBOT   = ZBOT_DATA
      SALP   = SALP_DATA
      CFACTR = CFACTR_DATA
      CMCMAX = CMCMAX_DATA
      SBETA  = SBETA_DATA
      RSMAX  = RSMAX_DATA
      TOPT   = TOPT_DATA
      REFDK  = REFDK_DATA
      FRZK   = FRZK_DATA
      FXEXP  = FXEXP_DATA
      REFKDT = REFKDT_DATA
      CZIL   = CZIL_DATA
      CSOIL  = CSOIL_DATA

C  SET-UP SOIL PARAMETERS
      B       = BB(SOILTYP)
      SMCDRY  = DRYSMC(SOILTYP)
      F1      = F11(SOILTYP)
      SMCMAX  = MAXSMC(SOILTYP)
      SMCREF  = REFSMC(SOILTYP)
      PSISAT  = SATPSI(SOILTYP)
      DKSAT   = SATDK(SOILTYP)
      DWSAT   = SATDW(SOILTYP)
      SMCWLT  = WLTSMC(SOILTYP)
      QUARTZ  = QTZ(SOILTYP)
      FRZFACT = (SMCMAX / SMCREF) * (0.412 / 0.468)
      KDT     = REFKDT * DKSAT/REFDK

C  TO ADJUST FRZK PARAMETER TO ACTUAL SOIL TYPE: FRZK * FRZFACT

      FRZX = FRZK * FRZFACT

C  SET-UP VEGETATION PARAMETERS
      NROOT = NROOT_DATA(VEGTYP)
      SNUP  = SNUPX(VEGTYP)
      RCMIN = RSMTBL(VEGTYP)
      RGL   = RGLTBL(VEGTYP)
      HS    = HSTBL(VEGTYP)
      Z0    = Z0_DATA(VEGTYP)
      LAI   = LAI_DATA(VEGTYP)
      IF(VEGTYP .EQ. BARE) SHDFAC = 0.0

      IF (NROOT .GT. NSOIL) THEN
         WRITE(*,*) 'Warning: too many root layers'
         STOP 333
      END IF

C  CALCULATE ROOT DISTRIBUTION
C  PRESENT VERSION ASSUMES UNIFORM DISTRIBUTION BASED ON SOIL LAYERS

      DO I=1,NROOT
         RTDIS(I) = -SLDPTH(I)/ZSOIL(NROOT)
      END DO

C  SET-UP SLOPE PARAMETER
      SLOPE = SLOPE_DATA(SLOPETYP)
C
      RETURN
      END
      SUBROUTINE ROSR12 ( P, A, B, C, D, DELTA, NSOIL )

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO INVERT (SOLVE) THE TRI-DIAGONAL MATRIX PROBLEM SHOWN
CC    =======   BELOW:
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      INTEGER K
      INTEGER KK
      INTEGER NSOIL
      
      REAL P     (NSOIL)
      REAL A     (NSOIL)
      REAL B     (NSOIL)
      REAL C     (NSOIL)
      REAL D     (NSOIL)
      REAL DELTA (NSOIL)
      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     INITIALIZE EQN COEF C FOR THE LOWEST SOIL LAYER.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      C(NSOIL) = 0.0

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     SOLVE THE COEFS FOR THE 1ST SOIL LAYER
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      P(1) = -C(1) / B(1)
      DELTA(1) = D(1) / B(1)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     SOLVE THE COEFS FOR SOIL LAYERS 2 THRU NSOIL
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DO K = 2 , NSOIL
        P(K) = -C(K) * ( 1.0 / (B(K) + A (K) * P(K-1)) )
        DELTA(K) = (D(K)-A(K)*DELTA(K-1))*(1.0/(B(K)+A(K)*P(K-1)))
      END Do

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     SET P TO DELTA FOR LOWEST SOIL LAYER.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      P(NSOIL) = DELTA(NSOIL)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     ADJUST P FOR SOIL LAYERS 2 THRU NSOIL
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DO K = 2 , NSOIL
         KK = NSOIL - K + 1
         P(KK) = P(KK) * P(KK+1) + DELTA(KK)
      END DO

      RETURN
      END
      SUBROUTINE SHFLX(S,STC,SMC,SMCMAX,NSOIL,T1,DT,YY,ZZ1,ZSOIL,TBOT,
     +     ZBOT, SMCWLT, PSISAT, SH2O, B,F1,DF1,ICE,QUARTZ,CSOIL)
      
      IMPLICIT NONE
      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  UPDATE THE TEMPERATURE STATE OF THE SOIL COLUMN BASED ON
CC              THE THERMAL DIFFUSION EQUATION AND UPDATE THE FROZEN SOIL
CC              MOISTURE CONTENT BASED ON THE TEMPERATURE.
CC      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      INTEGER NSOLD
      PARAMETER ( NSOLD = 20 )

      INTEGER I
      INTEGER ICE
      INTEGER IFRZ
      INTEGER NSOIL

      REAL B
      REAL DF1
      REAL CSOIL
      REAL DT
      REAL F1
      REAL PSISAT
      REAL QUARTZ
      REAL RHSTS ( NSOLD )
      REAL S
      REAL SMC   ( NSOIL )
      REAL SH2O  ( NSOIL )
      REAL SMCMAX
      REAL SMCWLT
      REAL STC	(NSOIL)
      REAL STCF	(NSOLD)
      REAL T0
      REAL T1
      REAL TBOT
      REAL ZBOT
      REAL YY
      REAL ZSOIL ( NSOIL )
      REAL ZZ1

      PARAMETER ( T0 = 273.15)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     HRT ROUTINE CALCS THE RIGHT HAND SIDE OF THE SOIL TEMP DIF EQN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      IF(ICE.EQ.1) THEN

C..SEA-ICE CASE

         CALL HRTICE(RHSTS,STC,NSOIL,ZSOIL,YY,ZZ1,DF1)

         CALL HSTEP (STCF,STC,RHSTS,DT,NSOIL)
         
      ELSE

C..LAND-MASS CASE

         CALL HRT(RHSTS,STC,SMC,SMCMAX,NSOIL,ZSOIL,YY,ZZ1,TBOT,
     +        ZBOT, PSISAT, SH2O, DT,
     +        B,F1,DF1,QUARTZ,CSOIL)
         
         CALL HSTEP(STCF,STC,RHSTS,DT,NSOIL)

      ENDIF

      DO I = 1,NSOIL
         STC(I)  = STCF(I)
      END DO
      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     IN THE NO SNOWPACK CASE (VIA ROUTINE NOPAC BRANCH,) UPDATE THE
C     GRND (SKIN) TEMPERATURE HERE IN RESPONSE TO THE UPDATED SOIL 
C     TEMPERATURE PROFILE ABOVE.
C (NOTE: INSPECTION OF ROUTINE SNOPAC SHOWS THAT T1 BELOW IS A DUMMY
C     VARIABLE ONLY, AS SKIN TEMPERATURE IS UPDATED DIFFERENTLY
C     IN ROUTINE SNOPAC) 
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      
      T1 = (YY + (ZZ1 - 1.0) * STC(1)) / ZZ1

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALC THE SFC SOIL HEAT FLUX
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

c      S = DF1 * (STC(1) - T1) / (0.5 * ZSOIL(1))

      RETURN
      END
      SUBROUTINE SMFLX ( ETA1,SMC,NSOIL,CMC,ETP1,DT,PRCP1,ZSOIL,
     &     SH2O, SLOPE, KDT, FRZFACT,
     &     SMCMAX,B,PC,SMCWLT,DKSAT,DWSAT,SMCREF,SHDFAC,CMCMAX,
     &     SMCDRY,CFACTR, RUNOFF1,RUNOFF2, RUNOFF3, EDIR1, EC1, 
     &     ETT1, SFCTMP,Q2,NROOT,RTDIS, FXEXP)


      IMPLICIT NONE

C ------------    FROZEN GROUND VERSION    --------------------------
C   NEW STATES ADDED: SH2O, AND FROZEN GROUD CORRECTION FACTOR, FRZFACT
C   AND PARAMETER SLOPE 
C

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE SOIL MOISTURE FLUX.  THE SOIL MOISTURE
CC    =======   CONTENT (SMC - A PER UNIT VOLUME MEASUREMENT) IS A
CC              DEPENDENT VARIABLE THAT IS UPDATED WITH PROGNOSTIC EQNS.
CC              THE CANOPY MOISTURE CONTENT (CMC) IS ALSO UPDATED.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      INTEGER NSOLD
      PARAMETER ( NSOLD = 20 )
      INTEGER K
      INTEGER NSOIL
      REAL B
      REAL BETA
      REAL CFACTR
      REAL CMC
      REAL CMCMAX
      REAL DEW
      REAL DKSAT
      REAL DRIP
      REAL DT
      REAL DWSAT
      REAL EC
      REAL EDIR
      REAL ET     ( NSOLD )
      REAL ETA1
      REAL ETP1
      REAL ETT
      REAL EXCESS
      REAL FXEXP
      REAL FLX1
      REAL FLX2
      REAL FLX3
      REAL KDT
      REAL PC
      REAL PCPDRP
      REAL PRCP1
      REAL RHSCT
      REAL RHSTT  ( NSOLD )
      REAL RIB
      REAL RTDIS (NSOIL)
      REAL RUNOF
      REAL RUNOFF,RUNOXX3
      REAL SHDFAC
      REAL SMC    ( NSOIL )

C ---------------    FROZEN GROUND VERSION     ---------------------
      
      REAL SH2O   ( NSOIL )
      REAL SICE   ( NSOLD )
      REAL SH2OA  ( NSOLD )
      REAL SH2OFG ( NSOLD )
C -------------------------------------------------------------------
           
      REAL SMCDRY
      REAL SMCMAX
      REAL SMCREF
      REAL SMCWLT
      REAL TRHSCT
      REAL ZSOIL  ( NSOIL )

C Temperature criteria for snowfall TFREEZ should have 
C same value as in SFLX.f
      REAL TFREEZ
      PARAMETER (TFREEZ = 273.15)

      REAL SLOPE, FRZFACT, RUNOFF1, RUNOFF2, RUNOFF3, EDIR1, EC1
      REAL ETT1, SFCTMP, Q2, DUMMY, CMC2MS, DEVAP

      INTEGER NROOT, I

      COMMON/RITE/ BETA,DRIP,EC,EDIR,ETT,FLX1,FLX2,FLX3,RUNOF,
     &     DEW,RIB,RUNOXX3

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     EXECUTABLE CODE BEGINS HERE....IF THE POTENTIAL EVAPOTRANS-
C     PIRATION IS GREATER THAN ZERO...
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      DUMMY=0.
      EDIR = 0.
      EC = 0.
      ETT = 0.
      DO K = 1, NSOIL
         ET ( K ) = 0.
      END DO
      
C ----------------------------------------------------------------------
      IF ( ETP1 .GT. 0.0 ) THEN

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       RETRIEVE DIRECT EVAPORATION FROM SOIL SURFACE
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

C ----------------------------------------------------------------------
C call this function only if veg cover not complete
C --------------     FROZEN GROUND VERSION     ---------------------
C   SMC STATES WERE REPLACED BY SH2O STATES
C
        IF (SHDFAC .LT. 1.) THEN
          EDIR = DEVAP ( ETP1, SH2O(1), ZSOIL(1), SHDFAC, SMCMAX,
     &      B, DKSAT, DWSAT, SMCDRY,SMCREF, SMCWLT, FXEXP)
        ENDIF
C ----------------------------------------------------------------------
C       INITIALIZE PLANT TOTAL TRANSPIRATION, RETRIEVE PLANT
C       TRANSPIRATION, AND ACCUMULATE IT FOR ALL SOIL LAYERS.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

c        ETT = 0.
         
        IF(SHDFAC.GT.0.0) THEN
        
C ----------------------------------------------------------------------
C --------------     FROZEN GROUND VERSION     ---------------------
C   SMC STATES WERE REPLACED BY SH2O STATES
C
          CALL TRANSP ( ET,NSOIL,ETP1,SH2O,CMC,ZSOIL,SHDFAC,SMCWLT,
     &      CMCMAX,PC,CFACTR,SMCREF,SFCTMP,Q2,NROOT,RTDIS)
          
          DO K = 1 , NSOIL
            ETT = ETT + ET ( K )
          END DO
c move this ENDIF after canopy evap calcs since CMC=0 for SHDFAC=0
c        ENDIF

C ----------------------------------------------------------------------
C       CALCULATE CANOPY EVAPORATION
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
ccc If statements to avoid TANGENT LINEAR problems near CMC=zero
          IF (CMC .GT. 0.0) THEN
            EC = SHDFAC * ( ( CMC / CMCMAX ) ** CFACTR ) * ETP1
          ELSE
            EC = 0.0
          ENDIF
C ----------------------------------------------------------------------
C########  EC SHOULD BE LIMITED BY THE TOTAL AMOUNT OF AVAILABLE
C          WATER ON THE CANOPY. MODIFIED BY F.CHEN ON 10/18/94
C########
          CMC2MS = CMC / DT
          EC = MIN ( CMC2MS, EC )
        ENDIF
      ENDIF

C ----------------------------------------------------------------------
C     TOTAL UP EVAP AND TRANSP TYPES TO OBTAIN ACTUAL EVAPOTRANSP
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      EDIR1=EDIR
      EC1=EC
      ETT1=ETT
      
      ETA1 = EDIR + ETT + EC
      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     COMPUTE THE RIGHT HAND SIDE OF THE CANOPY EQN TERM ( RHSCT )
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      RHSCT = SHDFAC * PRCP1 - EC

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CONVERT RHSCT (A RATE) TO TRHSCT (AN AMT) AND ADD IT TO EXISTING
C     CMC. IF RESULTING AMT EXCEEDS MAX CAPACITY, IT BECOMES DRIP
C     AND WILL FALL TO THE GRND.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DRIP = 0.
      TRHSCT = DT * RHSCT
      EXCESS =  CMC + TRHSCT
      IF ( EXCESS .GT. CMCMAX ) DRIP = EXCESS - CMCMAX

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     PCPDRP IS THE COMBINED PRCP1 AND DRIP (FROM CMC) THAT
C     GOES INTO THE SOIL
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      PCPDRP = (1. - SHDFAC) * PRCP1 + DRIP / DT

C      PRINT*,' ################ SMLX ##################'
C      PRINT*,' PCPDRP=', PCPDRP, ' EDIR=', EDIR,' ET=', ET,
C     *      'SMC(1)=', SMC(1), 'SMC(2)=', SMC(2), ' PRCP1=', PRCP1,
C     *      'DRIP = ', DRIP / DT

C ---------------     FROZEN GROUND VERSION     --------------------
C    STORE ICE CONTENT AT EACH SOIL LAYER BEFORE CALLING SRT & SSTEP
C
      DO I = 1,NSOIL
         SICE(I) = SMC(I) - SH2O(I)
      END DO
C ------------------------------------------------------------------
            
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALL SUBROUTINES SRT AND SSTEP TO SOLVE THE SOIL MOISTURE
C     TENDENCY EQUATIONS. 
C
C  IF THE INFILTRATING PRECIP RATE IS NONTRIVIAL,
C
C    (WE CONSIDER NONTRIVIAL TO BE A PRECIP TOTAL OVER THE TIME STEP 
C     EXCEEDING ONE ONE-THOUSANDTH OF THE WATER HOLDING CAPACITY OF 
C     THE FIRST SOIL LAYER)
C 
C  THEN CALL THE SRT/SSTEP SUBROUTINE PAIR TWICE IN THE MANNER OF 
C    TIME SCHEME "F" (IMPLICIT STATE, AVERAGED COEFFICIENT)
C    OF SECTION 2 OF KALNAY AND KANAMITSU (1988, MWR, VOL 116, 
C    PAGES 1945-1958)TO MINIMIZE 2-DELTA-T OSCILLATIONS IN THE 
C    SOIL MOISTURE VALUE OF THE TOP SOIL LAYER THAT CAN ARISE BECAUSE
C    OF THE EXTREME NONLINEAR DEPENDENCE OF THE SOIL HYDRAULIC 
C    DIFFUSIVITY COEFFICIENT AND THE HYDRAULIC CONDUCTIVITY ON THE
C    SOIL MOISTURE STATE
C
C  OTHERWISE CALL THE SRT/SSTEP SUBROUTINE PAIR ONCE IN THE MANNER OF
C    TIME SCHEME "D" (IMPLICIT STATE, EXPLICIT COEFFICIENT) 
C    OF SECTION 2 OF KALNAY AND KANAMITSU
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C PCPDRP IS UNITS OF KG/M**2/S OR MM/S, ZSOIL IS NEGATIVE DEPTH IN M 
C......IF ( PCPDRP .GT. 0.0 ) THEN

      IF ( (PCPDRP*DT) .GT. (0.001*1000.0*(-ZSOIL(1))*SMCMAX) ) THEN

C ---------------    FROZEN GROUND VERSION       ---------------------
C    SMC STATES REPLACED BY SH2O STATES IN SRT SUBR.
C    SH2O & SICE STATES INCLUDED IN SSTEP SUBR.
C    FROZEN GROUND CORRECTION FACTOR, FRZFACT, ADDED
C    ALL WATER BALANCE CALCULATIONS USING UNFROZEN WATER
C
         CALL SRT ( RHSTT,RUNOFF,EDIR,ET,SH2O,SH2O,NSOIL,PCPDRP,ZSOIL,
     &        DWSAT,DKSAT,SMCMAX, B, RUNOFF1, 
     +        RUNOFF2,DT,SMCWLT,SLOPE,KDT,FRZFACT, SICE)
         
         CALL SSTEP ( SH2OFG,SH2O,DUMMY,RHSTT,RHSCT,DT,NSOIL,SMCMAX,
     &        CMCMAX, RUNOFF3, ZSOIL, SMC, SICE )
         
         DO K = 1, NSOIL
            SH2OA(K) = ( SH2O(K) + SH2OFG(K) ) * 0.5
         END DO
        
         CALL SRT ( RHSTT,RUNOFF,EDIR,ET,SH2O,SH2OA,NSOIL,PCPDRP,ZSOIL,
     &        DWSAT,DKSAT,SMCMAX, B, RUNOFF1,
     +        RUNOFF2,DT,SMCWLT,SLOPE,KDT,FRZFACT, SICE)
         
         CALL SSTEP ( SH2O,SH2O,CMC,RHSTT,RHSCT,DT,NSOIL,SMCMAX,
     &        CMCMAX, RUNOFF3, ZSOIL,SMC,SICE)
         
      ELSE
         
         CALL SRT ( RHSTT,RUNOFF,EDIR,ET,SH2O,SH2O,NSOIL,PCPDRP,ZSOIL,
     &        DWSAT,DKSAT,SMCMAX, B, RUNOFF1,
     +        RUNOFF2,DT,SMCWLT,SLOPE,KDT,FRZFACT, SICE)
         
         
         CALL SSTEP ( SH2O,SH2O,CMC,RHSTT,RHSCT,DT,NSOIL,SMCMAX,
     &        CMCMAX, RUNOFF3, ZSOIL,SMC,SICE)
         
      ENDIF
      
      RUNOF = RUNOFF
      RETURN
      END
      FUNCTION SNKSRC ( TUP,TM,TDN, SMC, SH2O, ZSOIL,NSOIL,
     +     SMCMAX, PSISAT, B, DT, K, QTOT) 
      
      IMPLICIT NONE
      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE SINK/SOURCE TERM OF THE TERMAL DIFFUSION
CC    =======   EQUATION. (SH2O) IS AVAILABLE LIQUED WATER.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      INTEGER  K
      INTEGER  NSOIL
      
      REAL B
      REAL DF
      REAL DFH2O
      REAL DFICE
      REAL DH2O
      REAL DT
      REAL DZ
      REAL DZH
      REAL FREE
      REAL FRH2O
      REAL HLICE
      REAL PSISAT
      REAL QTOT
      REAL SH2O
      REAL SMC
      REAL SMCMAX
      REAL SNKSRC
      REAL T0
      REAL TAVG
      REAL TDN
      REAL TM
      REAL TUP
      REAL TZ
      REAL X0
      REAL XDN
      REAL XH2O
      REAL XUP
      REAL ZSOIL (NSOIL)

      PARAMETER (HLICE=3.3350E5)
      PARAMETER (DH2O =1.0000E3)
      PARAMETER (  T0 =2.7315E2)
      
      IF(K.EQ.1) THEN
        DZ=-ZSOIL(1)
      ELSE
        DZ=ZSOIL(K-1)-ZSOIL(K)
      ENDIF

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALCULATE POTENTIAL REDUCTION OF LIQUED WATER CONTENT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      
      XH2O=QTOT*DT/(DH2O*HLICE*DZ) + SH2O
      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     ESTIMATE UNFROZEN WATER AT TEMPERATURE TAVG,
C     AND CHECK IF CALCULATED WATER CONTENT IS REASONABLE 
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC 
        
C  ####   NEW CALCULATION OF AVERAGE TEMPERATURE (TAVG)   ##########
C  ####   IN FREEZING/THAWING LAYER USING UP, DOWN, AND MIDDLE   ###
C  ####   LAYER TEMPERATURES (TUP, TDN, TM)               ##########
   
      DZH=DZ*0.5

      IF (TUP .LT. T0) THEN

        IF (TM .LT. T0) THEN

          IF (TDN .LT. T0) THEN

C           *** TUP, TM, TDN < T0 ***

            TAVG = (TUP + 2.0*TM + TDN)/ 4.0
            
          ELSE

C           *** TUP & TM < T0,  TDN >= T0 ***

            X0 = (T0 - TM) * DZH / (TDN - TM)
            TAVG = 0.5 * (TUP*DZH+TM*(DZH+X0)+T0*(2.*DZH-X0)) / DZ
                       
          ENDIF      

        ELSE
        
          IF (TDN .LT. T0) THEN

C           *** TUP < T0, TM >= T0, TDN < T0 ***

            XUP  = (T0-TUP) * DZH / (TM-TUP)
            XDN  = DZH - (T0-TM) * DZH / (TDN-TM)
            TAVG = 0.5 * (TUP*XUP+T0*(2.*DZ-XUP-XDN)+TDN*XDN) / DZ

          ELSE

C           *** TUP < T0, TM >= T0, TDN >= T0 ***

            XUP  = (T0-TUP) * DZH / (TM-TUP)
            TAVG = 0.5 * (TUP*XUP+T0*(2.*DZ-XUP)) / DZ
                      
          ENDIF   
        
        ENDIF

      ELSE

        IF (TM .LT. T0) THEN

          IF (TDN .LT. T0) THEN

C           *** TUP >= T0, TM < T0, TDN < T0 ***

            XUP  = DZH - (T0-TUP) * DZH / (TM-TUP)
            TAVG = 0.5 * (T0*(DZ-XUP)+TM*(DZH+XUP)+TDN*DZH) / DZ
                      
          ELSE

C           *** TUP >= T0, TM < T0, TDN >= T0 ***

            XUP  = DZH - (T0-TUP) * DZH / (TM-TUP)
            XDN  = (T0-TM) * DZH / (TDN-TM)
            TAVG = 0.5 * (T0*(2.*DZ-XUP-XDN)+TM*(XUP+XDN)) / DZ
                                   
          ENDIF   

        ELSE

          IF (TDN .LT. T0) THEN

C           *** TUP >= T0, TM >= T0, TDN < T0 ***

            XDN  = DZH - (T0-TM) * DZH / (TDN-TM)
            TAVG = (T0*(DZ-XDN)+0.5*(T0+TDN)*XDN) / DZ
                 
          ELSE

C           *** TUP >= T0, TM >= T0, TDN >= T0 ***

            TAVG = (TUP + 2.0*TM + TDN) / 4.0
                      
          ENDIF           

        ENDIF

      ENDIF                      

      FREE=FRH2O(TAVG, SMC, SH2O, SMCMAX, B, PSISAT )

      IF ( XH2O .LT. SH2O .AND. XH2O .LT. FREE) THEN 
         IF ( FREE .GT. SH2O ) THEN
              XH2O = SH2O
          ELSE
              XH2O = FREE
          ENDIF
      ENDIF
              
      IF ( XH2O .GT. SH2O .AND. XH2O .GT. FREE )  THEN
         IF ( FREE .LT. SH2O ) THEN
              XH2O = SH2O
          ELSE
              XH2O = FREE
          ENDIF
      ENDIF 

      IF(XH2O .LT. 0. ) XH2O=0.
      IF(XH2O .GT. SMC) XH2O=SMC

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALCULATE SINK/SOURCE TERM AND REPLACE PREVIOUS WATER CONTENT 
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
     
      SNKSRC=-DH2O*HLICE*DZ*(XH2O-SH2O)/DT

      SH2O=XH2O
      
77    RETURN
      END
      SUBROUTINE SNOPAC (ETP,ETA,PRCP,PRCP1,SNOWNG,SMC,SMCMAX,SMCWLT,
     &  SMCREF, SMCDRY, CMC, CMCMAX, NSOIL, DT, SBETA, Q1, DF1,
     &  Q2,T1,SFCTMP,T24,TH2,F,F1,S,STC,EPSCA,SFCPRS,
c     &  B, PC, RCH, RR, CFACTR, SALP, ESD,
     &  B, PC, RCH, RR, CFACTR, SNCOVER, ESD, SNDENS,
     +  SNOWH, SH2O, SLOPE, KDT, FRZFACT, PSISAT,SNUP,
     &  ZSOIL, DWSAT, DKSAT, TBOT, ZBOT, SHDFAC, RUNOFF1,
     &  RUNOFF2,RUNOFF3,EDIR1,EC1,ETT1,NROOT,SNMAX,ICE,
     &  RTDIS,QUARTZ, FXEXP,CSOIL)

      IMPLICIT NONE

C ----------------------------------------------------------------------
CC    PURPOSE:  TO CALCULATE SOIL MOISTURE AND HEAT FLUX VALUES & UPDATE
CC    =======   SOIL MOISTURE CONTENT AND SOIL HEAT CONTENT VALUES FOR
CC              THE CASE WHEN A SNOW PACK IS PRESENT.
C ----------------------------------------------------------------------

      INTEGER ICE
      INTEGER NROOT
      INTEGER NSOIL

      LOGICAL SNOWNG

      REAL B
      REAL BETA
      REAL CFACTR
      REAL CMC
      REAL CMCMAX
      REAL CP
      REAL CPH2O
      REAL CPICE
      REAL CSOIL
      REAL DENOM
      REAL DEW
      REAL DF1
      REAL DKSAT
      REAL DRIP
      REAL DSOIL
      REAL DTOT
      REAL DT
      REAL DWSAT
      REAL EC
      REAL EDIR
      REAL EPSCA
      REAL ESD
      REAL EXPSNO
      REAL EXPSOI
      REAL ETA
      REAL ETA1
      REAL ETP
      REAL ETP1
      REAL ETP2
      REAL ETT
      REAL EX
      REAL EXPFAC
      REAL F
      REAL FXEXP
      REAL FLX1
      REAL FLX2
      REAL FLX3
      REAL F1
      REAL KDT
      REAL LSUBF
      REAL LSUBC
      REAL LSUBS
      REAL PC
      REAL PRCP
      REAL PRCP1
      REAL Q1
      REAL Q2
      REAL RCH
      REAL RIB
      REAL RR
      REAL RTDIS   ( NSOIL )
      REAL RUNOFF
      REAL S
      REAL SBETA
      REAL S1
      REAL SFCTMP
      REAL SHDFAC
      REAL SIGMA
      REAL SMC     ( NSOIL )
      REAL SH2O    ( NSOIL )
      REAL SMCDRY
      REAL SMCMAX
      REAL SMCREF
      REAL SMCWLT
      REAL SNMAX
      REAL SNOWH
      REAL STC     ( NSOIL )
      REAL T1
      REAL T11
      REAL T12
      REAL T12A
      REAL T12B
      REAL T24
      REAL TBOT
      REAL ZBOT
      REAL TH2
      REAL YY
      REAL ZSOIL( NSOIL )
      REAL ZZ1
C
      REAL TFREEZ, SALP, SFCPRS, SLOPE, FRZFACT, PSISAT, SNUP
      REAL RUNOFF1, RUNOFF2, RUNOFF3,RUNOXX3
      REAL EDIR1, EC1, ETT1, QUARTZ
      REAL SNDENS, SNCOND, RSNOW, SNCOVER, QSAT, ETP3, SEH, T14
      REAL CSNOW

      COMMON/RITE/ BETA,DRIP,EC,EDIR,ETT,FLX1,FLX2,FLX3,RUNOFF,
     &  DEW,RIB,RUNOXX3
     
      PARAMETER(CP=1004.5,CPH2O=4.218E+3,CPICE=2.106E+3,
     &  LSUBF=3.335E+5,LSUBC=2.501000E+6,LSUBS=2.83E+6,SIGMA=5.67E-8)

      PARAMETER ( TFREEZ = 273.15)

C ----------------------------------------------------------------------
C EXECUTABLE CODE BEGINS HERE...
C CONVERT POTENTIAL EVAP (ETP) FROM KG M-2 S-1 TO M S-1 AND THEN TO AN
C AMOUNT (M) GIVEN TIMESTEP (DT) AND CALL IT AN EFFECTIVE SNOWPACK
C REDUCTION AMOUNT, ETP2 (M).  THIS IS THE AMOUNT THE SNOWPACK WOULD BE
C REDUCED DUE TO EVAPORATION FROM THE SNOW SFC DURING THE TIMESTEP.
C EVAPORATION WILL PROCEED AT THE POTENTIAL RATE UNLESS THE SNOW DEPTH
C IS LESS THAN THE EXPECTED SNOWPACK REDUCTION.
C IF SEAICE (ICE=1), BETA REMAINS=1.
C ----------------------------------------------------------------------
      PRCP1 = PRCP1*0.001

      ETP2 = ETP * 0.001 * DT
      BETA = 1.0
      IF(ICE .NE. 1) THEN
        IF (ESD .LT. ETP2) THEN
          BETA = ESD / ETP2
        ENDIF
      ENDIF

C ----------------------------------------------------------------------
C IF ETP<0 (DOWNWARD) THEN DEWFALL (=FROSTFALL IN THIS CASE).
C ----------------------------------------------------------------------
      DEW = 0.0
      IF (ETP .LT. 0.0) THEN
        DEW = -ETP * 0.001
      ENDIF

C ----------------------------------------------------------------------
C If precip is falling, calculate heat flux from snow sfc to newly
C accumulating precip.  Note that this reflects the flux appropriate for
C the not-yet-updated skin temperature (T1).  Assumes temperature of the
C snowfall striking the gound is =SFCTMP (lowest model level air temp).
C ----------------------------------------------------------------------
      FLX1 = 0.0
      IF ( SNOWNG ) THEN
        FLX1 = CPICE * PRCP * ( T1 - SFCTMP )
      ELSE
        IF (PRCP .GT. 0.0) FLX1 = CPH2O * PRCP * (T1 - SFCTMP)
      ENDIF
      DSOIL = -(0.5 * ZSOIL(1))
      DTOT = SNOWH + DSOIL

C ----------------------------------------------------------------------
C Calculate an 'effective snow-grnd sfc temp' (T12) based on heat fluxes
C between the snow pack and the soil and on net radiation.
C Include FLX1 (precip-snow sfc) and FLX2 (freezing rain latent heat)
C fluxes.  FLX1 from above, FLX2 brought in via COMMOM block RITE.
C FLX2 reflects freezing rain latent heat flux using T1 calculated in
C PENMAN.
C ----------------------------------------------------------------------
      DENOM = 1.0 + DF1 / ( DTOT * RR * RCH )
      T12A = ((F - FLX1 - FLX2 - SIGMA * T24) /
     &       RCH+TH2-SFCTMP-BETA*EPSCA) / RR
      T12B = DF1 * STC(1) / ( DTOT * RR * RCH )
      T12 = (SFCTMP + T12A + T12B ) / DENOM      

C ----------------------------------------------------------------------
C IF THE 'EFFECTIVE SNOW-GRND SFC TEMP' IS AT OR BELOW FREEZING, NO SNOW
C MELT WILL OCCUR.  SET THE SKIN TEMP TO THIS EFFECTIVE TEMP AND SET THE
C EFFECTIVE PRECIP TO ZERO.
C ----------------------------------------------------------------------
      IF (T12 .LE. TFREEZ) THEN
        ESD = MAX(0.0, ESD-ETP2)

cggg    update snow depth.
        snowh = esd / sndens
cggg

        T1 = T12
C ----------------------------------------------------------------------
C Update soil heat flux (S) using new skin temperature (T1)
        S = DF1 * ( T1 - STC(1) ) / ( DTOT )
        FLX3 = 0.0
        EX = 0.0
        SNMAX = 0.0

C ----------------------------------------------------------------------
C IF THE 'EFFECTIVE SNOW-GRND SFC TEMP' IS ABOVE FREEZING, SNOW MELT
C WILL OCCUR.  CALL THE SNOW MELT RATE,EX AND AMT, SNMAX.  REVISE THE
C EFFECTIVE SNOW DEPTH.  REVISE THE SKIN TEMP BECAUSE IT WOULD HAVE CHGD
C DUE TO THE LATENT HEAT RELEASED BY THE MELTING. CALC THE LATENT HEAT
C RELEASED, FLX3. SET THE EFFECTIVE PRECIP, PRCP1 TO THE SNOW MELT RATE,
C EX FOR USE IN SMFLX.  ADJUSTMENT TO T1 TO ACCOUNT FOR SNOW PATCHES.
C ----------------------------------------------------------------------
      ELSE
c        IF ( (SNUP .GT. 0.0) .AND. (ESD .LT. SNUP) ) THEN
c turn off this block below since SNCOVER is calculated (as SNOFAC) in
C SFLX and now passed to SNOPAC
c        IF (ESD .LT. SNUP) THEN
c          RSNOW = ESD / SNUP
c          SNCOVER = 1.- (EXP(-SALP*RSNOW)-RSNOW*EXP(-SALP))
c        ELSE
c          SNCOVER = 1.
c        ENDIF  
        T1 = TFREEZ * SNCOVER + T12 * ( 1.0 - SNCOVER )
        QSAT = (0.622*6.11E2)/(SFCPRS-0.378*6.11E2)
        ETP = RCH*(QSAT-Q2)/CP
        ETP2 = ETP*0.001*DT
        BETA = 1.0
	
C ----------------------------------------------------------------------
C IF POTENTIAL EVAP (SUBLIMATION) GREATER THAN DEPTH OF SNOWPACK.
C BETA<1
C ----------------------------------------------------------------------
        IF ( ESD .LE. ETP2 ) THEN
          BETA = ESD / ETP2
          ESD = 0.0

cggg      snow pack has sublimated, set depth to zero
          snowh = 0.0
cggg

          SNMAX = 0.0
          EX = 0.0
C ----------------------------------------------------------------------
C Update soil heat flux (S) using new skin temperature (T1)
          S = DF1 * ( T1 - STC(1) ) / ( DTOT )
	  
C ----------------------------------------------------------------------
C POTENTIAL EVAP (SUBLIMATION) LESS THAN DEPTH OF SNOWPACK, BETA=1.
C SNOWPACK (ESD) REDUCED BY POT EVAP RATE
C ETP3 (CONVERT TO FLUX)
C UPDATE SOIL HEAT FLUX BECAUSE T1 PREVIOUSLY CHANGED.
C SNOWMELT REDUCTION DEPENDING ON SNOW COVER
C IF SNOW COVER LESS THAN 5% NO SNOWMELT REDUCTION
C ----------------------------------------------------------------------
        ELSE
c          ESD = MAX(0.0, ESD-ETP2)
          ESD = ESD-ETP2

cggg      snow pack reduced by sublimation, reduce snow depth
          snowh = esd / sndens
cggg

          ETP3 = ETP*LSUBC
          S = DF1 * ( T1 - STC(1) ) / ( DTOT )
          SEH = RCH*(T1-TH2)
          T14 = T1*T1
          T14 = T14*T14
          FLX3 = F - FLX1 - FLX2 - SIGMA*T14 - S - SEH - ETP3
          IF(FLX3.LE.0.0) FLX3=0.0
          EX = FLX3*0.001/LSUBF
C ----------------------------------------------------------------------
C Does below fail to match the melt water with the melt energy?
          IF ( SNCOVER .GT. 0.05) EX = EX * SNCOVER
          SNMAX = EX * DT
        ENDIF
        
C ----------------------------------------------------------------------
C SNMAX.LT.ESD
C ELSE
C ----------------------------------------------------------------------
c        IF(SNMAX.LT.ESD) THEN
C The 1.E-6 value represents a snowpack depth threshold value (0.1 mm)
C below which we choose not to retain any snowpack, and instead include
C it in snowmelt.
        IF(SNMAX.LT.ESD-1.E-6) THEN
          ESD = ESD - SNMAX

cggg      snow melt reduced snow pack, reduce snow depth
          snowh = esd / sndens
cggg

        ELSE
          EX = ESD/DT
          SNMAX = ESD
          ESD = 0.0

cggg      snow melt exceeds snow depth
          snowh = 0.0
cggg

          FLX3 = EX*1000.0*LSUBF
        ENDIF
        PRCP1 = PRCP1 + EX

      ENDIF
         
C ----------------------------------------------------------------------
C SET THE EFFECTIVE POTNL EVAPOTRANSP (ETP1) TO ZERO SINCE SNOW CASE SO
C SURFACE EVAP NOT CALCULATED FROM EDIR, EC, OR ETT IN SMFLX (BELOW).
C IF SEAICE (ICE=1) SKIP CALL TO SMFLX.
C SMFLX RETURNS SOIL MOISTURE VALUES AND PRELIMINARY VALUES OF
C EVAPOTRANSPIRATION.  IN THIS, THE SNOW PACK CASE, THE PRELIM VALUES
C (ETA1) ARE NOT USED IN SUBSEQUENT CALCULATION OF EVAP.
C NEW STATES ADDED: SH2O, AND FROZEN GROUND CORRECTION FACTOR
C EVAP EQUALS POTENTIAL EVAP UNLESS BETA<1.
C ----------------------------------------------------------------------
      ETP1 = 0.0
      IF (ICE .NE. 1) THEN
        CALL SMFLX ( ETA1,SMC,NSOIL,CMC,ETP1,DT,PRCP1,ZSOIL,
     +    SH2O, SLOPE, KDT, FRZFACT,
     &    SMCMAX,B,PC,SMCWLT,DKSAT,DWSAT,
     &    SMCREF,SHDFAC,CMCMAX,SMCDRY,CFACTR,RUNOFF1,RUNOFF2,
     &    RUNOFF3, EDIR1, EC1, ETT1,SFCTMP,Q2,NROOT,RTDIS,
     &    FXEXP)

      ENDIF
      ETA = BETA*ETP

C ----------------------------------------------------------------------
C THE 'ADJUSTED TOP SOIL LYR TEMP' (YY) AND THE ADJUSTED SOIL HEAT
C FLUX (ZZ1) ARE SET TO THE TOP SOIL LYR TEMP, AND 1, RESPECTIVELY.
C THESE ARE CLOSE-ENOUGH APPROXIMATIONS BECAUSE THE SFC HEAT FLUX TO BE
C COMPUTED IN SHFLX WILL EFFECTIVELY BE THE FLUX AT THE SNOW TOP
C SURFACE.  T11 IS A DUMMY ARGUEMENT SINCE WE WILL NOT USE ITS VALUE AS
C REVISED BY SHFLX.
C ----------------------------------------------------------------------
      ZZ1 = 1.0
      YY = STC(1)-0.5*S*ZSOIL(1)*ZZ1/DF1
      T11 = T1

C ----------------------------------------------------------------------
C SHFLX WILL CALC/UPDATE THE SOIL TEMPS.  NOTE:  THE SUB-SFC HEAT FLUX 
C (S1) AND THE SKIN TEMP (T11) OUTPUT FROM THIS SHFLX CALL ARE NOT USED 
C IN ANY SUBSEQUENT CALCULATIONS. RATHER, THEY ARE DUMMY VARIABLES HERE 
C IN THE SNOPAC CASE, SINCE THE SKIN TEMP AND SUB-SFC HEAT FLUX ARE 
C UPDATED INSTEAD NEAR THE BEGINNING OF THE CALL TO SNOPAC.
C ----------------------------------------------------------------------

      CALL SHFLX(S1,STC,SMC,SMCMAX,NSOIL,T11,DT,YY,ZZ1,ZSOIL,TBOT,
     +  ZBOT, SMCWLT, PSISAT, SH2O,
     &  B,F1,DF1,ICE, 
     &  QUARTZ,CSOIL)
      
C ----------------------------------------------------------------------
C SNOW DEPTH AND DENSITY ADJUSTMENT BASED ON SNOW COMPACTION.
C YY is assumed to be the soil temperture at the top of the soil column.
C ----------------------------------------------------------------------
      IF (ESD .GT. 0.) THEN
C --- debug ------------------------------------------------------------
c     write(6,*) 'SNOPAC1:ESD,SNOWH,SNDENS=',ESD,SNOWH,SNDENS
C --- debug ------------------------------------------------------------
        CALL SNOWPACK(ESD,DT,SNOWH,SNDENS,T1,YY)
C --- debug ------------------------------------------------------------
c        SNDENS = 0.2
c        SNOWH = ESD/SNDENS
C --- debug ------------------------------------------------------------
C --- debug ------------------------------------------------------------
c     write(6,*) 'SNOPAC2:ESD,SNOWH,SNDENS=',ESD,SNOWH,SNDENS
C --- debug ------------------------------------------------------------
      ELSE
        ESD = 0.
        SNOWH = 0.
        SNDENS = 0.
        SNCOND = 1.
      ENDIF

C ----------------------------------------------------------------------
      RETURN
      END
      SUBROUTINE SNOWPACK ( W,DTS,HC,DS,TSNOW,TSOIL )

      IMPLICIT NONE

C ##############################################################
C ##  SUBROUTINE TO CALCULATE COMPACTION OF SNOWPACK  UNDER  ###
C ##  CONDITIONS OF INCREASING SNOW DENSITY, AS OBTAINED     ###
C FROM AN APPROXIMATE SOLUTION OF E. ANDERSONS DIFFERENTIAL ###
C     EQUATION (3.29), NOAA TECHNICAL REPORT NWS 19,         ###
C                 BY   VICTOR KOREN   03/25/95               ###
C ##############################################################

C ##############################################################
C  W      IS A WATER EQUIVALENT OF SNOW, IN M                ###
C  DTS    IS A TIME STEP, IN SEC                             ###
C  HC     IS A SNOW DEPTH, IN M                              ###
C  DS     IS A SNOW DENSITY, IN G/CM3                        ###
C  TSNOW  IS A SNOW SURFACE TEMPERATURE, K                   ###
C  TSOIL  IS A SOIL SURFACE TEMPERATURE, K                   ###
C      SUBROUTINE WILL RETURN NEW VALUES OF H AND DS         ###
C ##############################################################

      INTEGER IPOL
      INTEGER J

      REAL C1, C2, HC, W, DTS, DS, TSNOW, TSOIL, H, WX
      REAL DT, TSNOWX, TSOILX, TAVG, B, DSX, DW
      REAL PEXP
      REAL WXX

      PARAMETER (C1=0.01, C2=21.0)

C ##  CONVERSION INTO SIMULATION UNITS   ######################### 

      H=HC*100.
      WX=W*100.
      DT=DTS/3600.
      TSNOWX=TSNOW-273.15
      TSOILX=TSOIL-273.15

C ##  CALCULATING OF AVERAGE TEMPERATURE OF SNOW PACK              ###

      TAVG=0.5*(TSNOWX+TSOILX)                                    

C ##  CALCULATING OF SNOW DEPTH AND DENSITY AS A RESULT OF COMPACTION
C              DS=DS0*(EXP(B*W)-1.)/(B*W)
C              B=DT*C1*EXP(0.08*TAVG-C2*DS0)
C NOTE: B*W IN DS EQN ABOVE HAS TO BE CAREFULLY TREATED 
C NUMERICALLY BELOW
C ##  C1 IS THE FRACTIONAL INCREASE IN DENSITY (1/(CM*HR)) 
C ##  C2 IS A CONSTANT (CM3/G) KOJIMA ESTIMATED AS 21 CMS/G

      IF(WX .GT. 1.E-2) THEN
        WXX = WX
      ELSE
        WXX = 1.E-2
      ENDIF
      B=DT*C1*EXP(0.08*TAVG-C2*DS)

C.........DSX=DS*((DEXP(B*WX)-1.)/(B*WX))
C--------------------------------------------------------------------
C  The function of the form (e**x-1)/x imbedded in above expression
C  for DSX was causing numerical difficulties when the denominator "x"
C  (i.e. B*WX) became zero or approached zero (despite the fact that
C  the analytical function (e**x-1)/x has a well defined limit as 
C  "x" approaches zero), hence below we replace the (e**x-1)/x 
C  expression with an equivalent, numerically well-behaved 
C  polynomial expansion.
C 
C  Number of terms of polynomial expansion, and hence its accuracy, 
C  is governed by iteration limit "ipol".
C       ipol greater than 9 only makes a difference on double
C             precision (relative errors given in percent %).
C        ipol=9, for rel.error <~ 1.6 e-6 % (8 significant digits)
C        ipol=8, for rel.error <~ 1.8 e-5 % (7 significant digits)
C        ipol=7, for rel.error <~ 1.8 e-4 % ...

      ipol = 4
      PEXP = 0.
      do j = ipol,1,-1
c        PEXP = (1. + PEXP)*B*WX/real(j+1) 
        PEXP = (1. + PEXP)*B*WXX/real(j+1) 
      end do 
      PEXP = PEXP + 1.
C
      DSX=DS*(PEXP)
C                     above line ends polynomial substitution

      IF(DSX .GT. 0.40) DSX=0.40
C ----------------------------------------------------------------------
C mbek - April 2001
C Set lower limit on snow density, rather than just previous value.
c         IF(DSX .LT. 0.05) DSX=DS
      IF(DSX .LT. 0.05) DSX=0.05

      DS=DSX

C ##  UPDATE OF SNOW DEPTH AND DENSITY DEPENDING ON LIQUID WATER
C ##  DURING SNOWMELT. ASSUMED THAT 13% OF LIQUID WATER CAN BE STORED
C ##  IN SNOW PER DAY DURING SNOWMELT TILL SNOW DENSITY 0.40

c         IF((TSNOWX .GE. 0.) .AND. (H .NE. 0.)) THEN
      IF (TSNOWX .GE. 0.) THEN
        DW=0.13*DT/24.
        DS=DS*(1.-DW)+DW
        IF(DS .GT. 0.40) DS=0.40
      ENDIF
C ----------------------------------------------------------------------
C Calculate snow depth (cm) from snow water equivalent and snow density.
      H=WX/DS
C ----------------------------------------------------------------------
C Change snow depth units to meters
      HC=H*0.01

      RETURN
      END
      SUBROUTINE SNOW_NEW ( T,P,HC,DS )

      IMPLICIT NONE
      
C ----------------------------------------------------------------------
C CALCULATING SNOW DEPTH AND DENSITITY TO ACCOUNT FOR THE NEW SNOWFALL
C T - AIR TEMPERATURE, K
C P - NEW SNOWFALL, M
C HC - SNOW DEPTH, M
C DS - SNOW DENSITY
C NEW VALUES OF SNOW DEPTH & DENSITY WILL BE RETURNED
      REAL HC
      REAL T 
      REAL P
      REAL DS
      REAL H
      REAL PX
      REAL TX
      REAL DS0
      REAL HNEW
c
      REAL ESD
      
C ----------------------------------------------------------------------
C CONVERSION INTO SIMULATION UNITS      
      H=HC*100.
      PX=P*100.
      TX=T-273.15
      
C ----------------------------------------------------------------------
C CALCULATING NEW SNOWFALL DENSITY DEPENDING ON TEMPERATURE
C EQUATION FROM GOTTLIB L. 'A GENERAL RUNOFF MODEL FOR SNOWCOVERED'
C 'AND GLACIERIZED BASIN', 6TH NORDIC HYDROLOGICAL CONFERENCE,
C VEMADOLEN, SWEDEN, 1980, 172-177PP.
C-----------------------------------------------------------------------
      IF(TX .LE. -15.) THEN
        DS0=0.05
      ELSE                                                      
        DS0=0.05+0.0017*(TX+15.)**1.5
      ENDIF
      
C ----------------------------------------------------------------------
C ADJUSTMENT OF SNOW DENSITY DEPENDING ON NEW SNOWFALL      
      HNEW=PX/DS0
      DS=(H*DS+HNEW*DS0)/(H+HNEW)
      H=H+HNEW
      HC=H*0.01
      
C ----------------------------------------------------------------------
      RETURN
      END
      SUBROUTINE SRT (RHSTT,RUNOFF,EDIR,ET,SH2O,SH2OA,NSOIL,PCPDRP,
     &                 ZSOIL,DWSAT,DKSAT,SMCMAX,B, RUNOFF1, 
     +                 RUNOFF2,DT,SMCWLT,SLOPE,KDT,FRZX,SICE)      


      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE THE RIGHT HAND SIDE OF THE TIME TENDENCY
CC    =======   TERM OF THE SOIL WATER DIFFUSION EQUATION.  ALSO TO
CC              COMPUTE ( PREPARE ) THE MATRIX COEFFICIENTS FOR THE
CC              TRI-DIAGONAL MATRIX OF THE IMPLICIT TIME SCHEME.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      
      INTEGER NSOLD
      PARAMETER ( NSOLD = 20 )

      INTEGER CVFRZ      
      INTEGER IALP1
      INTEGER IOHINF
      INTEGER J
      INTEGER JJ      
      INTEGER K
      INTEGER KS
      INTEGER NSOIL

	REAL AI     ( NSOLD )
      REAL B
      REAL BI     ( NSOLD )
      REAL CI     ( NSOLD )
      REAL DMAX   ( NSOLD )
      REAL DDZ
      REAL DDZ2
      REAL DENOM
      REAL DENOM2
      REAL DKSAT
      REAL DSMDZ
      REAL DSMDZ2
      REAL DWSAT
      REAL EDIR
      REAL ET     ( NSOIL )
      REAL INFMAX
      REAL KDT
      REAL MXSMC
      REAL MXSMC2
      REAL NUMER
      REAL PCPDRP
      REAL PDDUM
      REAL RHSTT  ( NSOIL )
      REAL RUNOFF
      
      REAL SH2O   ( NSOIL )
      REAL SH2OA  ( NSOIL )
      REAL SICE   ( NSOIL )
      REAL SICEMAX
      
      REAL SMCMAX
      REAL WCND
      REAL WCND2
      REAL WDF
      REAL WDF2
      REAL ZSOIL  ( NSOIL )

      REAL RUNOFF1, RUNOFF2, DT, SMCWLT, SLOPE, FRZX, DT1
      REAL SMCAV, DICE, DD, VAL, DDT, PX, FCR, ACRT, SUM
      REAL SSTT, SLOPX

C
      COMMON /ABCI/ AI, BI, CI

C -----------     FROZEN GROUND VERSION    -------------------------
C   REFERENCE FROZEN GROUND PARAMETER, CVFRZ, IS A SHAPE PARAMETER OF
C   AREAL DISTRIBUTION FUNCTION OF SOIL ICE CONTENT WHICH EQUALS 1/CV.
C   CV IS A COEFFICIENT OF SPATIAL VARIATION OF SOIL ICE CONTENT. 
C   BASED ON FIELD DATA CV DEPENDS ON AREAL MEAN OF FROZEN DEPTH, AND IT
C   CLOSE TO CONSTANT = 0.6 IF AREAL MEAN FROZEN DEPTH IS ABOVE 20 CM.
C   THAT IS WHY PARAMETER CVFRZ = 3 (INT{1/0.6*0.6})  
C
C   Current logic doesnt allow CVFRZ be bigger than 3
        PARAMETER ( CVFRZ = 3 )
C ------------------------------------------------------------------
     
C      PRINT*,'in SRT, Declaration -----------------------'
C      PRINT*,'NSOIL=' , NSOIL
C      PRINT*,'NSOLD=' , NSOLD
        
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     DETERMINE RAINFALL INFILTRATION RATE AND RUNOFF
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

C
C ##INCLUDE THE INFILTRATION FORMULE FROM SCHAAKE AND KOREN MODEL
C
CC    MODIFIED BY Q DUAN
CC      
      IOHINF=1

C Let SICEMAX be the greatest, if any, frozen water content within 
c soil layers.
      SICEMAX = 0.0
      DO KS=1,NSOIL
       IF (SICE(KS) .GT. SICEMAX) SICEMAX = SICE(KS)
      END DO

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     DETERMINE RAINFALL INFILTRATION RATE AND RUNOFF
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      PDDUM = PCPDRP
      RUNOFF1 = 0.0
      IF ( PCPDRP .NE. 0.0 ) THEN

CC++  MODIFIED BY Q. DUAN, 5/16/94

C        IF (IOHINF .EQ. 1) THEN
  
          DT1 = DT/86400.
          SMCAV = SMCMAX - SMCWLT
          DMAX(1)=-ZSOIL(1)*SMCAV

C -----------     FROZEN GROUND VERSION    ------------------------
C
          DICE = -ZSOIL(1) * SICE(1)
C-------------------------------------------------------------------
          
          DMAX(1)=DMAX(1)*(1.0 - (SH2OA(1)+SICE(1)-SMCWLT)/SMCAV)
          DD=DMAX(1)
      DO KS=2,NSOIL
          
C -----------     FROZEN GROUND VERSION    ------------------------
C
           DICE = DICE + ( ZSOIL(KS-1) - ZSOIL(KS) ) * SICE(KS)
C------------------------------------------------------------------- 
         
           DMAX(KS)=(ZSOIL(KS-1)-ZSOIL(KS))*SMCAV
           DMAX(KS)=DMAX(KS)*(1.0 - (SH2OA(KS)+SICE(KS)-SMCWLT)/SMCAV)
           DD=DD+DMAX(KS)
      END DO
CC .....VAL = (1.-EXP(-KDT*SQRT(DT1)))
C IN BELOW, REMOVE THE SQRT IN ABOVE
          VAL = (1.-EXP(-KDT*DT1))
          DDT = DD*VAL
          PX = PCPDRP*DT  
          IF(PX.LT.0.0) PX = 0.0
          INFMAX = (PX*(DDT/(PX+DDT)))/DT
          
C -----------     FROZEN GROUND VERSION    --------------------------
C    REDUCTION OF INFILTRATION BASED ON FROZEN GROUND PARAMETERS
C
         FCR = 1. 
         IF ( DICE .GT. 1.E-2) THEN 
           ACRT = CVFRZ * FRZX / DICE 
           SUM = 1.
           IALP1 = CVFRZ - 1 
           DO J = 1,IALP1
              K = 1
              DO JJ = J+1, IALP1
                K = K * JJ
              END DO   
              SUM = SUM + (ACRT ** ( CVFRZ-J)) / FLOAT (K) 
           END DO 
           FCR = 1. - EXP(-ACRT) * SUM 
         END IF 
         INFMAX = INFMAX * FCR
C -------------------------------------------------------------------

C ############    CORRECTION OF INFILTRATION LIMITATION    ##########
C     IF INFMAX .LE. HYDROLIC CONDUCTIVITY ASSIGN INFMAX THE 
C     VALUE OF HYDROLIC CONDUCTIVITY
C
C         MXSMC = MAX ( SH2OA(1), SH2OA(2) ) 
        MXSMC = SH2OA(1)

C      PRINT*,'SRT, BEFORE WDFCND - 1 ------------------------------'
C      PRINT*,'MXSMC,SMCMAX=' , MXSMC,SMCMAX
C      PRINT*,'B,DKSAT,DWSAT=' , B,DKSAT,DWSAT

      CALL WDFCND ( WDF,WCND,MXSMC,SMCMAX,B,DKSAT,DWSAT,
     &               SICEMAX )

            INFMAX = MAX(INFMAX, WCND)
            INFMAX= MIN(INFMAX,PX)

C      PRINT*,'SRT, AFTER WDFCND - 1 ------------------------------'
C      PRINT*,'WDF,WCND=' , WDF,WCND
C      PRINT*,'MXSMC,SMCMAX=' , MXSMC,SMCMAX
C      PRINT*,'B,DKSAT,DWSAT=' , B,DKSAT,DWSAT
 
C
          IF ( PCPDRP .GT. INFMAX ) THEN
            RUNOFF1 = PCPDRP - INFMAX
            PDDUM = INFMAX
          END IF

      END IF
C
C TO AVOID SPURIOUS DRAINAGE BEHAVIOR IDENTIFIED BY P. GRUNMANN,
C FORMER APPROACH IN LINE BELOW REPLACED WITH NEW APPROACH IN 2ND LINE
C...MXSMC = MAX( SH2OA(1), SH2OA(2) )
        MXSMC =  SH2OA(1)

C      PRINT*,'SRT, BEFORE WDFCND - 2'
C      PRINT*,'MXSMC,SMCMAX=' , MXSMC,SMCMAX
C      PRINT*,'B,DKSAT,DWSAT=' , B,DKSAT,DWSAT

      CALL WDFCND ( WDF,WCND,MXSMC,SMCMAX,B,DKSAT,DWSAT,
     &SICEMAX )

C      PRINT*,'SRT, AFTER WDFCND - 2'
C      PRINT*,'WDF,WCND=' , WDF,WCND
C      PRINT*,'MXSMC,SMCMAX=' , MXSMC,SMCMAX
C      PRINT*,'B,DKSAT,DWSAT=' , B,DKSAT,DWSAT
 
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALC THE MATRIX COEFFICIENTS AI, BI, AND CI FOR THE TOP LAYER
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DDZ = 1. / ( -.5 * ZSOIL(2) )
      AI(1) = 0.0
      BI(1) = WDF * DDZ / ( -ZSOIL(1) )
      CI(1) = -BI(1)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALC RHSTT FOR THE TOP LAYER AFTER CALCING THE VERTICAL SOIL
C     MOISTURE GRADIENT BTWN THE TOP AND NEXT TO TOP LAYERS.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DSMDZ = ( SH2O(1) - SH2O(2) ) / ( -.5 * ZSOIL(2) )
      RHSTT(1) = (WDF * DSMDZ + WCND - PDDUM + EDIR + ET(1))/ZSOIL(1)
      SSTT = WDF * DSMDZ + WCND + EDIR + ET(1)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     INITIALIZE DDZ2
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DDZ2 = 0.0

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     LOOP THRU THE REMAINING SOIL LAYERS, REPEATING THE ABV PROCESS
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DO K = 2 , NSOIL
         DENOM2 = ( ZSOIL(K-1) - ZSOIL(K) )
         IF ( K .NE. NSOIL ) THEN
            SLOPX = 1.
C
C AGAIN, TO AVOID SPURIOUS DRAINAGE BEHAVIOR IDENTIFIED BY P. GRUNMANN,
C FORMER APPROACH IN LINE BELOW REPLACED WITH NEW APPROACH IN 2ND LINE
C....MXSMC2 = MAX ( SH2OA(K), SH2OA(K+1) )
            MXSMC2 =  SH2OA(K)

C      PRINT*,'SRT, BEFORE WDFCND - 3'
C      PRINT*,'MXSMC2,SMCMAX=' , MXSMC2,SMCMAX
C      PRINT*,'B,DKSAT,DWSAT=' , B,DKSAT,DWSAT
C      PRINT*,'K=' , K

            CALL WDFCND ( WDF2,WCND2,MXSMC2,SMCMAX,B,DKSAT,DWSAT,
     &           SICEMAX )

C      PRINT*,'SRT, AFTER WDFCND - 3'
C      PRINT*,'WDF2,WCND2=' , WDF2,WCND2
C      PRINT*,'MXSMC2,SMCMAX=' , MXSMC2,SMCMAX
C      PRINT*,'B,DKSAT,DWSAT=' , B,DKSAT,DWSAT
 
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CALC SOME PARTIAL PRODUCTS FOR LATER USE IN CALCNG RHSTT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

            DENOM = ( ZSOIL(K-1) - ZSOIL(K+1) )
            DSMDZ2 = ( SH2O(K) - SH2O(K+1) ) / ( DENOM * 0.5 )

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C         CALC THE MATRIX COEF, CI, AFTER CALCNG ITS PARTIAL PRODUCT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

            DDZ2 = 2.0 / DENOM
            CI(K) = -WDF2 * DDZ2 / DENOM2
         ELSE

C   SLOPE OF BOTTOM LAYER IS INTRODUCED     ############
C
            SLOPX = SLOPE
C--------------------------------------------------------
          
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C         RETRIEVE THE SOIL WATER DIFFUSIVITY AND HYDRAULIC
C         CONDUCTIVITY FOR THIS LAYER
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC


C      PRINT*,'SRT, BEFORE WDFCND - 4'
C      PRINT*,'SH2OA(NSOIL),SMCMAX=' , SH2OA(NSOIL),SMCMAX
C      PRINT*,'B,DKSAT,DWSAT=' , B,DKSAT,DWSAT
C      PRINT*,'K=' , K
 
            CALL WDFCND ( WDF2,WCND2,SH2OA(NSOIL),SMCMAX,
     &           B,DKSAT,DWSAT,SICEMAX )

C      PRINT*,'SRT, AFTER WDFCND - 4'
C      PRINT*,'WDF2,WCND2=' , WDF2,WCND2
C      PRINT*,'SH2OA(NSOIL),SMCMAX=' , SH2OA(NSOIL),SMCMAX
C      PRINT*,'B,DKSAT,DWSAT=' , B,DKSAT,DWSAT
 
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C         CALC A PARTIAL PRODUCT FOR LATER USE IN CALCNG RHSTT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

            DSMDZ2 = 0.0

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C         SET MATRIX COEF CI TO ZERO
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

            CI(K) = 0.0
         END IF

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CALC RHSTT FOR THIS LAYER AFTER CALCNG ITS NUMERATOR
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

         NUMER = (WDF2 * DSMDZ2) + SLOPX * WCND2 - (WDF * DSMDZ) 
     +        - WCND + ET(K)
         RHSTT(K) = NUMER / (-DENOM2)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CALC MATRIX COEFS, AI, AND BI FOR THIS LAYER
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

         AI(K) = -WDF * DDZ / DENOM2
         BI(K) = -( AI(K) + CI(K) )

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       RESET VALUES OF WDF, WCND, DSMDZ, AND DDZ FOR LOOP TO NEXT LYR
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

         IF(K.EQ.NSOIL) THEN
C############### RUNOFF2: GROUND WATER RUNOFF ###########
            RUNOFF2 = SLOPX * WCND2
         ENDIF

         IF ( K .NE. NSOIL ) THEN
            WDF = WDF2
            WCND = WCND2
            DSMDZ = DSMDZ2
            DDZ = DDZ2
         END IF
      END DO

C      PRINT*,'SRT, final Runoff'
C      PRINT*,'RUNOFF1=' , RUNOFF1
C      PRINT*,'RUNOFF2=' , RUNOFF2
 
      RETURN
      END
      SUBROUTINE SSTEP ( SH2OOUT, SH2OIN, CMC, RHSTT, RHSCT, DT,
     &     NSOIL, SMCMAX, CMCMAX, RUNOFF3, ZSOIL,SMC,SICE )

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE/UPDATE THE SOIL MOISTURE CONTENT VALUES
CC    =======   AND THE CANOPY MOISTURE CONTENT VALUES.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      INTEGER NSOLD
      PARAMETER ( NSOLD = 20 )
C
      INTEGER I
      INTEGER K 
      INTEGER KK11
      INTEGER NSOIL

      REAL AI     ( NSOLD )
      REAL BI     ( NSOLD )
      REAL CI     ( NSOLD )
      REAL CIin   ( NSOLD )
      REAL CMC
      REAL CMCMAX
      REAL DT
      REAL RHSCT
      REAL RHSTT   ( NSOIL )
      REAL RHSTTin ( NSOIL )
      REAL SH2OIN  ( NSOIL )
      REAL SH2OOUT ( NSOIL )
      REAL SICE    ( NSOIL )
      REAL SMC     ( NSOIL )
      REAL SMCMAX
      REAL ZSOIL(NSOIL)

      REAL RUNOFF3, RUNOFS, WPLUS, DDZ, STOT, WFREE, DPLUS
C
      COMMON /ABCI/ AI, BI, CI

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CREATE 'AMOUNT' VALUES OF VARIABLES TO BE INPUT TO THE
C     TRI-DIAGONAL MATRIX ROUTINE.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DO K = 1 , NSOIL
        RHSTT(K) = RHSTT(K) * DT
        AI(K) = AI(K) * DT
        BI(K) = 1. + BI(K) * DT
        CI(K) = CI(K) * DT
      END DO

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     COPY VALUES FOR INPUT VARIABLES BEFORE CALL TO ROSR12
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      DO K = 1 , NSOIL
         RHSTTin(K) = RHSTT(K)
      END DO
      DO K = 1 , NSOLD
         CIin(K) = CI(K)
      END DO
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALL ROSR12 TO SOLVE THE TRI-DIAGONAL MATRIX
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      CALL ROSR12 ( CI, AI, BI, CIin, RHSTTin, RHSTT, NSOIL )

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     SUM THE PREVIOUS SMC VALUE AND THE MATRIX SOLUTION TO GET A
C     NEW VALUE.  MIN ALLOWABLE VALUE OF SMC WILL BE 0.02.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

C   ################## RUNOFF3: Runoff within soil layers #######

      RUNOFS = 0.0
      WPLUS = 0.0
      RUNOFF3 = 0.
      DDZ = - ZSOIL(1)
      
      DO K = 1 , NSOIL
         IF ( K .NE. 1 ) DDZ = ZSOIL(K - 1) - ZSOIL(K)
         SH2OOUT(K) = SH2OIN(K) + CI(K) + WPLUS / DDZ
        
C      PRINT*,'IN sstep'
C      PRINT*,'SH2OOUT=', SH2OOUT
        
         STOT = SH2OOUT(K) + SICE(K)
         IF ( STOT .GT. SMCMAX ) THEN
            IF ( K .EQ. 1 ) THEN
               DDZ = -ZSOIL(1)
            ELSE
               KK11 = K - 1
               DDZ = -ZSOIL(K) + ZSOIL(KK11)
            END IF
            WPLUS = ( STOT - SMCMAX ) * DDZ
         ELSE
            WPLUS = 0.
         END IF
         SMC(K) = MAX ( MIN( STOT, SMCMAX ), 0.02 )
         SH2OOUT(K) = MAX ( (SMC(K) - SICE(K)), 0.0 )
      END DO

C  ###  V. KOREN   9/01/98    ######
C     WATER BALANCE CHECKING UPWARD

      IF(WPLUS .GT. 0.) THEN
       DO I=NSOIL-1,1,-1
        IF(I .EQ. 1) THEN
         DDZ=-ZSOIL(1)
        ELSE
         DDZ=-ZSOIL(I)+ZSOIL(I-1)
        ENDIF
        WFREE=(SMCMAX-SH2OOUT(I)-SICE(I))*DDZ
        DPLUS=WFREE-WPLUS
        IF(DPLUS .GE. 0.) THEN
         SH2OOUT(I)=SH2OOUT(I)+WPLUS/DDZ
         SMC(I)=SH2OOUT(I)+SICE(I)
         WPLUS=0.
           
        ELSE
         SH2OOUT(I)=SH2OOUT(I)+WFREE/DDZ
         SMC(I)=SH2OOUT(I)+SICE(I)
         WPLUS=-DPLUS
        ENDIF
       END DO
30     RUNOFF3=WPLUS
      ENDIF

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C  UPDATE CANOPY WATER CONTENT/INTERCEPTION (CMC).  CONVERT RHSCT TO 
C  AN 'AMOUNT' VALUE AND ADD TO PREVIOUS CMC VALUE TO GET NEW CMC.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      CMC = CMC + DT * RHSCT
      IF (CMC .LT. 1.E-20) CMC=0.0
      CMC = MIN(CMC,CMCMAX)

      RETURN
      END
      SUBROUTINE TBND (TU, TB, ZSOIL, ZBOT, K, NSOIL, TBND1)

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC   PURPOSE:   CALCULATE TEMPERATURE ON THE BOUNDARY OF THE LAYER
CC   =======    BY INTERPOLATION OF THE MIDDLE LAYER TEMPERATURES
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      INTEGER NSOIL
      INTEGER K

      REAL TBND1
      REAL T0
      REAL TU
      REAL TB
      REAL ZB
      REAL ZBOT
      REAL ZUP
      REAL ZSOIL (NSOIL)

      PARAMETER (T0=273.15)

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC   USE SURFACE TEMPERATURE ON THE TOP OF THE FIRST LAYER
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      
      IF(K .EQ. 1) THEN
        ZUP=0.
      ELSE
        ZUP=ZSOIL(K-1)
      ENDIF

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC   USE DEPTH OF THE CONSTANT BOTTOM TEMPERATURE WHEN INTERPOLATE
CC   TEMPERATURE INTO THE LAST LAYER BOUNDARY
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      
      IF(K .EQ. NSOIL) THEN
        ZB=2.*ZBOT-ZSOIL(K)
      ELSE
        ZB=ZSOIL(K+1)
      ENDIF

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC   LINEAR INTERPOLATION BETWEEN THE AVERAGE LAYER TEMPERATURES
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      
      TBND1 = TU+(TB-TU)*(ZUP-ZSOIL(K))/(ZUP-ZB)
      
      RETURN
      END
      SUBROUTINE TDFCND ( DF, SMC, Q,  SMCMAX, SH2O)

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE THERMAL DIFFUSIVITY AND CONDUCTIVITY OF
CC    =======   THE SOIL FOR A GIVEN POINT AND TIME.
CC
CC    VERSION:  PETERS-LIDARD APPROACH (PETERS-LIDARD et al., 1998)
CC    =======
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

       REAL DF
       REAL GAMMD
       REAL THKDRY
       REAL AKE
       REAL THKICE
       REAL THKO
       REAL THKQTZ
       REAL THKSAT
       REAL THKS
       REAL THKW
       REAL Q
       REAL SATRATIO
       REAL SH2O
       REAL SMC
       REAL SMCMAX
       REAL XU
       REAL XUNFROZ


C WE NOW GET QUARTZ AS AN INPUT ARGUMENT (SET IN ROUTINE REDPRM):
C        DATA QUARTZ /0.82, 0.10, 0.25, 0.60, 0.52, 
C     &              0.35, 0.60, 0.40, 0.82/

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     IF THE SOIL HAS ANY MOISTURE CONTENT COMPUTE A PARTIAL SUM/PRODUCT
C     OTHERWISE USE A CONSTANT VALUE WHICH WORKS WELL WITH MOST SOILS
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C  
C
C  THKW ......WATER THERMAL CONDUCTIVITY
C  THKQTZ ....THERMAL CONDUCTIVITY FOR QUARTZ
C  THKO ......THERMAL CONDUCTIVITY FOR OTHER SOIL COMPONENTS
C  THKS ......THERMAL CONDUCTIVITY FOR THE SOLIDS COMBINED(QUARTZ+OTHER)
C  THKICE ....ICE THERMAL CONDUCTIVITY
C  SMCMAX ....POROSITY (= SMCMAX)
C  Q .........QUARTZ CONTENT (SOIL TYPE DEPENDENT)
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C USE AS IN PETERS-LIDARD, 1998 (MODIF. FROM JOHANSEN, 1975).
C
C                                  PABLO GRUNMANN, 08/17/98
C REFS.:
C      FAROUKI, O.T.,1986: THERMAL PROPERTIES OF SOILS. SERIES ON ROCK 
C              AND SOIL MECHANICS, VOL. 11, TRANS TECH, 136 PP.
C      JOHANSEN, O., 1975: THERMAL CONDUCTIVITY OF SOILS. PH.D. THESIS,
C              UNIVERSITY OF TRONDHEIM,
C      PETERS-LIDARD, C. D., ET AL., 1998: THE EFFECT OF SOIL THERMAL 
C              CONDUCTIVITY PARAMETERIZATION ON SURFACE ENERGY FLUXES
C              AND TEMPERATURES. JOURNAL OF THE ATMOSPHERIC SCIENCES,
C              VOL. 55, PP. 1209-1224.
C 
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

C  NEEDS PARAMETERS
C POROSITY(SOIL TYPE):
C      POROS = SMCMAX
C SATURATION RATIO:
      SATRATIO = SMC/SMCMAX
C      print *, 'SATRATIO=',SATRATIO
C     PARAMETERS  W/(M.K)
      THKICE = 2.2
      THKW = 0.57
      THKO = 2.0
C      IF (Q .LE. 0.2) THKO = 3.0
      THKQTZ = 7.7
C  SOLIDS CONDUCTIVITY      
      THKS = (THKQTZ**Q)*(THKO**(1.- Q))
C      print *, 'THKS = ',THKS
C  UNFROZEN FRACTION (FROM 1.0, I.E., 100%LIQUID, TO 0.0 (100% FROZEN))
      XUNFROZ=(SH2O + 1.E-9)/(SMC + 1.E-9)
C      print *, '   '
C      print *, 'XUNFROZ = ',XUNFROZ
C      print *, '    '
C  UNFROZEN VOLUME FOR SATURATION (POROSITY*XUNFROZ)
      XU=XUNFROZ*SMCMAX 
C  SATURATED THERMAL CONDUCTIVITY
      THKSAT = THKS**(1.-SMCMAX)*THKICE**(SMCMAX-XU)*THKW**(XU)
C      print *, 'THKSAT = ',THKSAT
C  DRY DENSITY IN KG/M3
      GAMMD = (1. - SMCMAX)*2700.
C      print *, 'GAMMD = ',GAMMD
C  DRY THERMAL CONDUCTIVITY IN W.M-1.K-1
      THKDRY = (0.135*GAMMD + 64.7)/(2700. - 0.947*GAMMD)
C      print *, 'THKDRY = ',THKDRY
C RANGE OF VALIDITY FOR THE KERSTEN NUMBER
      IF ( SATRATIO .GT. 0.1 ) THEN

C    KERSTEN NUMBER (FINE FORMULA, AT LEAST 5% OF PARTICLES<(2.E-6)M)
           IF ( (XUNFROZ + 0.0005) .LT. SMC ) THEN
C    FROZEN
              AKE = SATRATIO
           ELSE
C    UNFROZEN
              AKE = LOG10(SATRATIO) + 1.0
           ENDIF

      ELSE
        
C USE K = KDRY
        AKE = 0.0
C        print *, 'AKE (ELSE) = ',AKE
      ENDIF
C  THERMAL CONDUCTIVITY

       DF = AKE*(THKSAT - THKDRY) + THKDRY

      RETURN
      END
      SUBROUTINE TRANSP (ET,NSOIL,ETP1,SMC,CMC,ZSOIL,SHDFAC,SMCWLT,
     &      CMCMAX,PC,CFACTR,SMCREF,SFCTMP,Q2,NROOT,RTDIS)

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE TRANSPIRATION FROM THE VEGTYP FOR THIS PT.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      INTEGER I
      INTEGER K
      INTEGER NSOIL
      INTEGER NROOT

      REAL CFACTR
      REAL CMC
      REAL CMCMAX
      REAL ET    ( NSOIL )
      REAL ETP1
      REAL ETP1A
      REAL GX (7)
C.....REAL PART ( NSOIL )
      REAL PC
      REAL RTDIS ( NSOIL )
      REAL SHDFAC
      REAL SMC   ( NSOIL )
      REAL SMCREF
      REAL SMCWLT
      REAL ZSOIL ( NSOIL )

      REAL SFCTMP, Q2, SGX, DENOM, RTX

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       INITIALIZE  PLANT TRANSP TO ZERO FOR ALL SOIL LAYERS.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      DO K = 1, NSOIL
         ET(K) = 0.
      END DO

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       CALC AN 'ADJUSTED' POTNTL TRANSPIRATION
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

ccc If statements to avoid TANGENT LINEAR problems near zero
      IF (CMC .NE. 0.0) THEN
      ETP1A = SHDFAC * PC * ETP1 * (1.0 - (CMC /CMCMAX) ** CFACTR)
      ELSE
      ETP1A = SHDFAC * PC * ETP1
      ENDIF
      
      SGX = 0.0
      DO I = 1, NROOT
         GX(I) = ( SMC(I) - SMCWLT ) / ( SMCREF - SMCWLT )
         GX(I) = MAX ( MIN ( GX(I), 1. ), 0. )
         SGX = SGX + GX (I)
      END DO
      SGX = SGX / NROOT
      
      DENOM = 0.
      DO I = 1,NROOT
         RTX = RTDIS(I) + GX(I) - SGX
         GX(I) = GX(I) * MAX ( RTX, 0. )
         DENOM = DENOM + GX(I)
      END DO   
      IF ( DENOM .LE. 0.0) DENOM = 1.
      
      DO I = 1, NROOT
         ET(I) = ETP1A * GX(I) / DENOM
      END DO 

C ABOVE CODE ASSUMES A VERTICALLY UNIFORM ROOT DISTRIBUTION
C
C CODE BELOW TESTS A VARIABLE ROOT DISTRIBUTION
C
C     ET(1) = ( ZSOIL(1) / ZSOIL(NROOT) ) * GX * ETP1A
C        ET(1) = ( ZSOIL(1) / ZSOIL(NROOT) ) * ETP1A
C
C ###  USING ROOT DISTRIBUTION AS WEIGHTING FACTOR
C     ET(1) = RTDIS(1) * ETP1A
C         ET(1) =  ETP1A*PART(1)
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     LOOP DOWN THRU THE SOIL LAYERS REPEATING THE OPERATION ABOVE,
C     BUT USING THE THICKNESS OF THE SOIL LAYER (RATHER THAN THE
C     ABSOLUTE DEPTH OF EACH LAYER) IN THE FINAL CALCULATION.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      
C     DO 10 K = 2, NROOT
C     GX = ( SMC(K) - SMCWLT ) / ( SMCREF - SMCWLT )
C     GX = MAX ( MIN ( GX, 1. ), 0. )
C     TEST CANOPY RESISTANCE
C     GX = 1.0
C     ET(K) = ((ZSOIL(K)-ZSOIL(K-1))/ZSOIL(NROOT))*GX*ETP1A
C       ET(K) = ((ZSOIL(K)-ZSOIL(K-1))/ZSOIL(NROOT))*ETP1A
C###  USING ROOT DISTRIBUTION AS WEIGHTING FACTOR
C       ET(K) = RTDIS(K) * ETP1A
C         ET(K) = ETP1A*PART(K)
C     10    CONTINUE
      
      RETURN
      END
      SUBROUTINE WDFCND ( WDF,WCND,SMC,SMCMAX,B,DKSAT,DWSAT,
     &                         SICEMAX )

      IMPLICIT NONE

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CC    PURPOSE:  TO CALCULATE SOIL WATER DIFFUSIVITY AND SOIL
CC    =======   HYDRAULIC CONDUCTIVITY.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      REAL B
      REAL DKSAT
      REAL DWSAT
      REAL EXPON
      REAL FACTR1
      REAL FACTR2
      REAL SICEMAX
      REAL SMC
      REAL SMCMAX
      REAL VKwgt
      REAL WCND
      REAL WDF

C      PRINT*,'------------ in WDFCND -------------------------------'
C      PRINT*,'BEFORE WDFCND'
C      PRINT*,'B=',B
C      PRINT*,'DKSAT=',DKSAT
C      PRINT*,'DWSAT=',DWSAT
C      PRINT*,'EXPON=',EXPON
C      PRINT*,'FACTR2=',FACTR2
C      PRINT*,'SMC=',SMC
C      PRINT*,'SMCMAX=',SMCMAX
C      PRINT*,'WCND=',WCND
C      PRINT*,'WDF=',WDF
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     CALC THE RATIO OF THE ACTUAL TO THE MAX PSBL SOIL H2O CONTENT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      SMC = SMC
      SMCMAX = SMCMAX
      FACTR1 = 0.2 / SMCMAX
      FACTR2 = SMC / SMCMAX

CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     PREP AN EXPNTL COEF AND CALC THE SOIL WATER DIFFUSIVITY
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      EXPON = B + 2.0
      WDF = DWSAT * FACTR2 ** EXPON

C FROZEN SOIL HYDRAULIC DIFFUSIVITY.  VERY SENSITIVE TO THE VERTICAL
C GRADIENT OF UNFROZEN WATER. THE LATTER GRADIENT CAN BECOME VERY
C EXTREME IN FREEZING/THAWING SITUATIONS, AND GIVEN THE RELATIVELY 
C FEW AND THICK SOIL LAYERS, THIS GRADIENT SUFFERES SERIOUS 
C TRUNCTION ERRORS YIELDING ERRONEOUSLY HIGH VERTICAL TRANSPORTS OF
C UNFROZEN WATER IN BOTH DIRECTIONS FROM HUGE HYDRAULIC DIFFUSIVITY.  
C THEREFORE, WE FOUND WE HAD TO ARBITRARILY CONSTRAIN WDF 
C
C version D_10cm: ........  FACTR1 = 0.2/SMCMAX
C Weighted approach...................... Pablo Grunmann, 09/28/99.
      IF (SICEMAX .GT. 0.0)  THEN
      VKwgt=1./(1.+(500.*SICEMAX)**3.)
      WDF = VKwgt*WDF + (1.- VKwgt)*DWSAT*FACTR1**EXPON
C      PRINT*,'______________________________________________'
C      PRINT*,'Weighted approach:'
C      PRINT*,'  SICEMAX       VKwgt              Dwgt'
C      PRINT*,SICEMAX,  VKwgt, 1.-VKwgt
C      PRINT*,'______________________________________________'
      ENDIF
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C     RESET THE EXPNTL COEF AND CALC THE HYDRAULIC CONDUCTIVITY
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

      EXPON = ( 2.0 * B ) + 3.0
      WCND = DKSAT * FACTR2 ** EXPON

C      PRINT*,' WDFCND Results --------------------------------'
C      PRINT*,'B=',B
C      PRINT*,'DKSAT=',DKSAT
C      PRINT*,'DWSAT=',DWSAT
C      PRINT*,'EXPON=',EXPON
C      PRINT*,'FACTR2=',FACTR2
C      PRINT*,'SMC=',SMC
C      PRINT*,'SMCMAX=',SMCMAX
C      PRINT*,'WCND=',WCND
C      PRINT*,'WDF=',WDF
C      PRINT*,' SMC         WDF           WCND             B'
C      PRINT*,SMC,WDF,WCND,B

      RETURN
      END
