#!/bin/sh

#####################################################################
# config.sh
#
#   Configuration script. It is assumed the following directory
# structure for the sources/support libraries:
#
#   sc170/
#
#          grads.c   grads.h   ...
#
#   $mach/
#
#        hdf/      include/      lib/      netcdf/       X11/
#        gd/	   libpng/      zlib/
#
# where $mach = alpha, hpux, irix6, ... is one the GrADS supported 
# platforms. Notice that the X11/ subdirectory has NOT yet been
# included in most supp libs. For additional information consult file
# ~gradsdev/Supplibs/src/supplibs_tree.txt
#
# REVISION HISTORY:
#
# 03Oct02  joew Added darwin target for Mac OS/X
#
# 30Apr98  da Silva  Major revision to conform to
#                    new supp libs. Tested on:
#                      sunos4, site=dao
#                      hpux,   site=dao
#                      alpha
#                      linux5
#                      irix5
#                      irix6
#                      win32
#                      msdos
#
# 05May98  da Silva  Revised alpha rules (-non_shared moved from cco to ladd);
#                    added -luser32 to win32 to cope with new versions
#                    of the ImageMagick library; added -ldpstk to limg on AIX
#                    per Hoop's request.
#
# 16Oct2000   Hoop   Added support for GXPNG into GrADS option
#
#
# 24Jun2002   Komine Added support for 64-bit machine for hpux11
#
######################################################################



##################################################
#
#	defaults  
#
##################################################

gver="`cat VERSION`"	       # grads version

gradsbin="xgrads"              # name of grads binary
ladd=""	                       # additional libraries
endian=1                       # arch
isacray=0                      # cray
ishp64=0                       # hpux11
bindir="../bin"                # bin dir in mv.sh script
usrdir="/usr/local/grads/bin"  # bin dir in mv.usr.sh script

sdf=1                          # use sdf
netcdf_unidata=1               # use unidata netcdf libraries

rl=1                           # use readline libs
lats=1                         # build lats
gui=1                          # build in gui
###imgxa=0                     # build in XA image tool interface
imgim=1                        # build in ImageMagick image tool interface
img=1                          # 0 = no ; 1 = ImageMagic ; 2 = XA
make=1                         # do the makes
gxpng=1			       # build in GXPNG code into GrADS
			       # change to 0 to make default w/o

MORECMD="more"                 # more command, varies believe it or not
moveit="no"                    # move after make
mach="none"                    # name of machine
site="none"                    # name of site
setcc=0                        # set the C compiler at the command line
ldopts=""                      # additional load options
a4flag=0                       # A4 setting in gxps and gxeps
dods=0			       # dods default
dodsflg=0		       # dods default
gadods=0		       # gadods default
cplusplus="g++"                # needed to link with dods

xroot=""                       # X installation root directory, e.g., /usr
xlib=" -lXext -lX11"           # vanilla xlib
xtaw="-lXaw -lXmu -lXt"        # toolkit + Athena widgets 
xsmice="-lSM -lICE "           # shared memory/ICE X11R6 libraries

win32e=0                       # indicates whether Win32 uses XLIBEMU
libm="-lm"                     # whether libm should be included

##################################################
#
#   sub routines - check for gcc and help
#
##################################################

:
dowehaveit() { 
    PROG=$1 ; shift
    IFS=" "
    for DIR in $@ ; do
        if [ -f ${DIR}/${PROG} ] ; then
	    return 1
        fi
    done
    return 0
}

:
lshelp() { 

echo " "
echo "GrADS Configuration script -- create a Makefile"
echo " "
echo " "
echo "Usage: config.sh -arch=PLATFORM [ -site=SITE ] [ -movem | -nosdf | -noreadline "
echo "            | -hdf   | -nolats | -nogui | -clean | -imgim | -noimg | -a4 "
echo "            [ -CC=COMPILER -usrdir=USRDIR -bindir=BINDIR -gradsbin=GRADSBIN ]"
echo "		  | [ -gxpng | -nogxpng ]"
echo " "
echo "  Options with settings:"
echo " "
echo "  -arch=PLATFORM     : what machine you are building on"  
echo "        PLATFORM     = win32 | win32e | msdos | sunos4 | sol55 (sunos5.5) | sol2 (sunos5.[2-4]) | "
echo "                       irix4 (SGI) | irix5 (SGI) | irix6 (SGI)"
echo "                       linux | linux5 (Red Hat 5) linux6 (RH 6/7) | hpux | alpha"
echo "                       aix (IBM) | nec | c90 | j90 | hpux11" 
echo "  -site=SITE         : where you are building to set local options"
echo "        SITE         = [defaults to basic set ups for that platform]"
echo "  -usrdir=USRDIR     : where the mv.usr.sh script puts binaries"
echo "          USRDIR     = [default: /usr/local/grads/bin]"
echo "  -bindir=BINDIR     : local bin dir where mv.sh and mv.x.sh put binaries" 
echo "          BINDIR     = [default: ../bin]"
echo "  -gradsbin=GRADSBIN : name of grads binary"
echo "            GRADSBIN = [default: xgrads]"
echo "  -CC=COMPILER       : compiler to overide defaults based on -arch and -site"
echo "      COMPILER       = [default: gcc or cc]"
echo " "
echo "  Switches:"
echo " "
echo "  -movem             = mv the binaries"
### echo "  -imgxa             = build with XA image output interface"
echo "  -imgim             = build with ImageMagick image output interface"
echo "  -noimg             = build with NO image output interface"
echo "  -nosdf             = build without the SDF libraries"
echo "  -noreadline        = build without the the readline libraries"
echo "  -nolats            = build without the LATS library"
echo "  -hdf               = build SDF using HDF libraries"
echo "  -nogui             = build without the GUI routines"
echo "  -clean             = clean the *.o files"
echo "  -nomake            = do NOT make; just create the Makefile"
echo "  -gxpng             = build GXPNG code into GrADS"
echo "  -nogxpng           = do NOT build GXPNG code into GrADS"
echo "  -a4                = make the default page size in gxps and gxeps A4; otherwise letter"
echo "  -dods              = enable DODS"
echo "  -nodods            = disable DODS"
echo "  -gadods            = enable gadods interface (DODS station data)"
echo "  -nogadods          = disable gadods interface"
echo " "

}

##################################################
#
#  process command line options
#
##################################################


for arg in $* ; do
    case $arg in
        -arch=*)
	    mach=`echo $arg | sed -e 's/-arch=//'`
        ;;
	-site=*)
	    site=`echo $arg | sed -e 's/-site=//'`
	;;
	-clean)
	    rm *.o
	    exit 0
        ;;
	-movem)
	    moveit="yes"
	;;
	-noimg)                                  # if no img
	    imgxa=0
	    imgim=0
            img=0
	;;
        -gxpng)
	    gxpng=1
        ;;
	-nogxpng)
	    gxpng=0
        ;;
	-nosdf)                                  # if no SDF
	    sdf=0
            netcdf_unidata=0                     #
	;;
	-nolats)                                 # if no lats use the NCSA netcdf
	    lats=0
            # netcdf_unidata=0                   # ams: no need for this                 
	;;
	-hdf)
	    sdf=1                                # if no lats use the NCSA netcdf
	    # lats=0                             # ams: no need for this                 
            netcdf_unidata=0                 
	;;
###        -imgxa)
###           img=2
###        ;;                 
        -imgim)
           img=1
        ;;                 
	-nogui)                                  #
	    gui=0
	;;
	-nomake)                                  #
	    make=0
	;;
	-a4)                                     #
	    a4flag=1     
	;;
	-dods)                                     #
	   dods=1    
 	   dodsflg=1 
	;;
	-nodods)                                     #
	   dods=0     
	   dodsflg=0
	;;
	-gadods)                                     #
	   gadods=1
	;;
	-nogadods)                                     #
	   gadods=0     
	;;
	-nonetcdf_unidata)
	    netcdf_unidata=0
	;;
	-CC=*)
	    ccname=`echo $arg | sed -e 's/-CC=//'`
            setcc=1
	;;
	-usrdir=*)
	    usrdir=`echo $arg | sed -e 's/-usrdir=//'`
	;;
	-bindir=*)
	    bindir=`echo $arg | sed -e 's/-bindir=//'`
	;;
	-gradsbin=*)
	    gradsbin=`echo $arg | sed -e 's/-gradsbin=//'`
	;;
	-noreadline)
	    rl=0
	;;
	*)
            lshelp $0
            echo "ERROR: unrecognized command line pararmeter to config.sh"
            exit 0
        ;;
    esac
done

if [ "$mach" = "none" ] ; then
    lshelp $0
    echo "Please Specify PLATFORM with -arch=PLATFORM"
    exit 1
fi

if [ "$mach" = "win32e" ] ; then
     mach="win32"
     win32e=1  # a variant of win32
     imgxa=0
     imgim=0
     img=0
     gui=0
#    dods=0
fi

#############################################################
#
# set parameters for the distribution creation shell script
#
#############################################################

distdir="GrADS-$gver"
distdirbin="GrADS-$gver/bin"
distdirdoc="doc"
distnametar="GrADS-$gver.$mach.bin.tar"
distnametargz="GrADS-$gver.$mach.bin.tar.gz"
distnametarZ="GrADS-$gver.$mach.bin.tar.Z"
distlinkgrads="gradshdf"
distlinkgrads="gradsnc"

#############################################################
#
# Warning!  On HPs at least, this will reset argv for the script to be that
# of the dowehaveit function call.
#
#############################################################

SAVEIFS="${IFS}"
IFS=:
dowehaveit gcc $PATH
HAVEGCC=$?
if [ $HAVEGCC -eq 1 -a $setcc -eq 0 ] ; then
    echo "gcc is available..."
fi
IFS="${SAVEIFS}"

##################################################
#
#  set the architecture (64 or 32 bit)
#
##################################################

case $mach in
    c90)
# will have to add newest SGIs, some SUNs sometime, and maybe some DECs
        arch=64
    ;;
    j90)
        arch=64
# will have to add newest SGIs, some SUNs sometime, and maybe some DECs
    ;;
#kk 020624 --- s
    hpux11)
        arch=64
    ;;
#kk 020624 --- e
    *)
        arch=32
    ;;
esac


echo
if [ "$site" = "none" -a $make -eq 1 ] ; then
    echo "Making GrADS Makefile and DOING the make for..."
    echo "a $mach which is a ${arch}-bit machine."
elif [ $make -eq 1 ] ; then
    echo "Making GrADS Makefile and DOING the make for..."
    echo "a "$mach" at "$site" which is a "$arch"-bit machine"
elif [ $make -eq 0 ] ; then
    echo "Making GrADS Makefile ONLY for..."
    echo "a "$mach" at "$site" which is a "$arch"-bit machine"
fi
echo


##################################################
#
#  SDF set up
#
##################################################

osdf="OBJSSDF"
if [ $sdf -eq 1 ] ; then
     ISDF="-I../$mach/hdf/include -I../$mach//netcdf/include"
     LSDF="../$mach/hdf/lib/libmfhdf.a ../$mach/hdf/lib/libdf.a ../$mach/hdf/lib/libjpeg.a ../$mach/zlib/libz.a ../$mach/netcdf/lib/libudunits.a"
     if [ $netcdf_unidata -eq 1 ] ; then
          ISDF="-I../$mach/netcdf/include"
          LSDF="../$mach/netcdf/lib/libnetcdf.a ../$mach/netcdf/lib/libudunits.a"
     fi
#----------------------------------------------------------------------
#
#  dods set up
#
#----------------------------------------------------------------------
     if [ $dods -eq 1 ] ; then
	 ISDF="-I../$mach/DODS/include -I../$mach/netcdf/include"
	 LSDF="-L../$mach/DODS/lib -L../$mach/DODS/packages-3.1.1/lib"
	 if [ $gadods -eq 1 ] ; then
	     ogadods="dodstn.o"
	     ISDF="$ISDF -I../$mach/gadods/include"
	     LSDF="$LSDF -L../$mach/gadods/lib  -lnc-dods -lgadods -ldap++ -lnc-dods -lgadods -ldap++"
	 else 
	     LSDF="$LSDF -lnc-dods -ldap++ -lnc-dods -ldap++"
	 fi
	 LSDF="$LSDF -lwww -lrx  ../$mach/netcdf/lib/libudunits.a"
     fi
#
#  no sdf but LATS
#
elif [ $lats -eq 1 ] ; then
    ISDF="-I../$mach/hdf/include -I../$mach//netcdf/include"
    LSDF="../$mach/hdf/lib/libmfhdf.a ../$mach/hdf/lib/libdf.a ../$mach/hdf/lib/libjpeg.a ../$mach/zlib/libz.a ../$mach/netcdf/lib/libudunits.a"
    if [ $netcdf_unidata -eq 1 ] ; then
         ISDF="-I../$mach/netcdf/include"
         LSDF=="../$mach/netcdf/lib/libnetcdf.a ../$mach/netcdf/lib/libudunits.a"
    fi
    osdf=""
#
# no sdf, no LATS
#
else

    ISDF=""
    LSDF=""
    osdf=""

fi


##################################################
#
#  NETCDF setup for GrADS-LATS interface
#
##################################################

if [ $netcdf_unidata -eq 1 ] ; then
  nclats="latsnc.o"  
  netcdf_ncsa=0
else
  nclats="latsnc.o"  
  netcdf_ncsa=1
fi



##################################################
#
#  LATS interface set up
#
##################################################

if [ $lats -eq 1 ] ; then
    olats="OBJSLATS"
else
    olats="OBJSNOLATS"
fi

##################################################
#
#  readline setup
#
##################################################

if [ $rl -eq 1 ] ; then
      lrl="../$mach/lib/libreadline.a -ltermcap "
      irl="-I../$mach/include"
else
        irl=""
        lrl=""
fi

##################################################
#
#    GUI setup
#
##################################################

if [ $gui -eq 1 ] ; then

    igui="-I../$mach/include"
    lgui="../$mach/lib/libfreq.a ../$mach/lib/libsx.a"
    ogui="gagui.o gsgui.o"

else

    igui=""
    lgui=""
    ogui=""

fi

##################################################
#
#  ImageMagick Setup.
#
#  TO DO: Restore XA support when these become 
#         part of the supp libs.
#
##################################################

if [ $img -eq 1 ] ; then
     oimg="gaimg.o"
     iimg="-I../$mach/include/magick "
     limg="../$mach/lib/libMagick.a ../$mach/hdf/lib/libdf.a ../$mach/hdf/lib/libjpeg.a ../$mach/zlib/libz.a"
     if [ $sdf -eq 1 -a $netcdf_unidata -eq 0 ] ; then
          iimg="-I../$mach/include/magick"
 	  limg="../$mach/lib/libMagick.a "
     fi
else
     oimg=""
     iimg=""
     limg=""
fi

##################################################
#
#   GXPNG into GrADS setup
#
##################################################
if [ $gxpng -eq 1 ] ; then
    ogxpng="gd13gif.o"
    igxpng="-DGIFSUPPORT -I../$mach/gd -I../$mach/libpng -I../$mach/zlib"
    lgxpng="../$mach/gd/libgd.a ../$mach/libpng/libpng.a ../$mach/zlib/libz.a"
else
    ogxpng=""
    igxpng=""
    lgxpng=""
fi

##################################################
#
#  describe the features of this creation of GrADS
#
##################################################

if [ $lats -eq 1 -a $netcdf_unidata -eq 1 -a $sdf -eq 1 ] ; then
  gdesc="LATS=GRIB_netCDF__SDF_READ=unidata_netCDF"

elif [ $lats -eq 1 -a $netcdf_ncsa -eq 1 -a $sdf -eq 1 ] ; then
  gdesc="LATS=GRIB_NCSA_HDF__SDF_READ=NCSA_netCDF_HDF"

elif [ $lats -eq 0 -a $netcdf_unidata -eq 1 -a $sdf -eq 1 ] ; then
  gdesc="LATS=n/a__SDF_READ=unidata_netCDF"

elif [ $lats -eq 0 -a $netcdf_unidata -eq 0 -a $sdf -eq 1 ] ; then
  gdesc="LATS=n/a__SDF_READ=NCSA_netCDF_HDF"

elif [ $lats -eq 0 -a $sdf -eq 0 ] ; then
  gdesc="LATS=n/a__SDF_read=n/a"

elif [ $lats -eq 1 -a $sdf -eq 0 ] ; then
  gdesc="LATS=GRIB__SDF_read=n/a"

else

  echo " "
  echo "!!!!!!!!!!"
  echo "invalid GrADS make, sayoonara"
  echo "!!!!!!!!!!"
  echo " "
  exit 1

fi
if [ $gxpng -eq 1 ] ; then
    gdesc="${gdesc}=GXPNG"
fi

##################################################
#
#  Main loop:
#
#    32-bit arch set up for platforms and sites
#
#
##################################################

if [ $arch = "32" ] ; then

#   MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS 
#
#       MS-DOS with DJGPP (gcc) compiler and XLIB emulation (XLIBEMU)
#    
#   MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS MSDOS 

  if [ $mach = "msdos" ] ; then

    cc="gcc"
    cco="-O -DXLIBEMU" 
    endian=0

    rl=0         # no FULL readline support in MSDOS yet
    lrl="../$mach/lib/history.o "
    gui=0        # no Athena gui either
    img=0        # no IM support for now
    imgim=0

    xlib="-L../msdos/lib -lgvwm -lX11 -lgrx20"
    xtaw=""                        # toolkit + Athena widgets 
    xsmice=""                      # shared memory/ICE X11R6 libraries

  fi


# WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 
#
#     Windows 95/NT with CYGWIN and X11R6.4 libraries
#
# WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 WIN32 

  if [ $mach = "win32" ] ; then

    cc="gcc"
    cco="-O -I." 
    cplusplus=g++-2  # default g++ 3 does not work with our supplibs
    endian=0
    ladd="-luser32"

    xroot="/usr/X11R6"
    xlib="$xlib -lXpm"

    if [ $rl -eq 1 ] ; then
	lrl="-lreadline"
    fi

    if [ $sdf -eq 1 ] ; then
      LSDF="$LSDF ../$mach/netcdf/lib/libudport.a"
    fi

    if [ $dods -eq 1 ] ; then
      LSDF="$LSDF -lrpclib"
    fi

  fi

# WIN32e WIN32e WIN32e WIN32e WIN32e WIN32e WIN32e WIN32e WIN32e WIN32e WIN32e 
#
#     Windows 95/NT with CYGWIN and XLIBEMU/GRX
#
# WIN32e WIN32e WIN32e WIN32e WIN32e WIN32e WIN32e WIN32e WIN32e WIN32e WIN32 

  if [ $win32e -eq 1 ] ; then

    cco="-O -I. -DXLIBEMU32" 
    xlib=""
    ladd="-L../win32/xlibemu/lib -lX11 -lgrx20 -lm  -mwindows"

    xroot="../win32/xlibemu"

  fi


#   SUNOS4 SUNOS4 SUNOS4 SUNOS4 SUNOS4 SUNOS4 SUNOS4 
#
#       SunOS 4.1.x
#
#   SUNOS4 SUNOS4 SUNOS4 SUNOS4 SUNOS4 SUNOS4 SUNOS4 

  if [ $mach = "sunos4" ] ; then

    if [ $HAVEGCC -eq 1 ] ; then
        cc="gcc"
        ldopts="-static"
    else
        cc="acc"
    fi
    cco="-O"

    case $site in
        dao)
    	    cc="gcc -static"
            xroot="/usr/X11R6"
            if [ $sdf -eq 1 -a $netcdf_unidata -eq 1 ] ; then
		 LSDF="$LSDF -ludport"  # to resolve strerror()
            fi
        ;;
        fnmoc)
            xroot="/usr/X11R5"
            xsmice=""
        ;;
        llnl)
    	    cc="acc"
            xroot="/usr/X11R5"
            xsmice=""
            distlinkgrads="gradsnc"
        ;;
    esac

  fi


#   HPUX HPUX HPUX HPUX HPUX HPUX HPUX 
#
#
#   HPUX HPUX HPUX HPUX HPUX HPUX HPUX 

  if [ $mach = "hpux" ] ; then

# Something is wrong with Bourne shell functions on HP.  The argv
# of the function becomes the argv of the script, and, worse, it says
# we have gcc when we don't.  So, assume no gcc.
#    if [ $HAVEGCC -eq 1 ] ; then
#        cc="gcc"
#    else
#        cc="cc"
#    fi

    cc="cc"
    cco="-O"              # "-O -Aa" 

    case $site in
        dao)
	    cc="gcc"
	    cco="-O "
	    ldopts="-static /usr/lib/libdld.sl"
	    xroot="/usr/local/X11R5"
            xsmice=""
        ;;
        nhc)
	    cc="c89"
	    cco="-O"
	    xroot="/usr/X11R6"
        ;;
        nersc)
	    cc=gcc
	    cco="-O" 
	    ldopts="-static"
	    xroot="/usr/X11R6"
        ;;
        jma)
            cc=gcc
            cco="-O"
            xroot="/usr/contrib/X11R6"
            xsmice=""
	   a4flag=1
        ;;
        nrl)
            cc=gcc
            cco="-O"
            xroot="/usr/contrib/X11R5"
            xsmice=""
        ;;
    esac

  fi


#   NEC NEC NEC NEC NEC NEC NEC 
#
#    nec at jamstec.go.jp (mstdiox)
#    UNIX_SV mstdiox 4.2MP 1.release.1023.06:21 R4000 r4000
#
#   NEC NEC NEC NEC NEC NEC NEC 

  if [ $mach = "nec" ] ; then

    cc="cc"
    cco="-O -Aa" 

    xroot="/usr/neccsc"

    case $site in
        jamstec)
	    cc=cc
	    cco="-KOlimit" 
	    ldopts=""
        ;;
    esac

  fi

#   IRIX? IRIX? IRIX? IRIX? IRIX? IRIX? IRIX? 
#
#	irix4 and irix5 and irix6
#
#   IRIX? IRIX? IRIX? IRIX? IRIX? IRIX? IRIX? 

  if [ $mach = "irix4" ] ; then

    if [ $HAVEGCC -eq 1 ] ; then
        cc="gcc"
    else
        cc="cc"
        rl=1
        irl="-I../include"
        lrl="-L../lib -lreadline -ltermcap"
    fi

    cco=""

    case $site in
        ncep_irix4)
            irl="-I/export/sgi80/wd20mf/lib/readline -L/export/sgi80/wd20mf/lib/readline/sgi"
            lrl="-L/export/sgi80/wd20mf/lib/readline/sgi -lreadline -ltermcap"
            lrl="-L../lib -lreadline -ltermcap"
	;;
	*)
#
#	defaults on Steve Lord's machine
#
            irl="-I../include"
            lrl="-L../lib -lreadline -ltermcap"
            xroot="/usr"
	    ladd="-lsun"
	    xsmice=""
            distlinkgrads="gradsc"
        ;;
    esac

    
  fi

  if [ "$mach" = "irix5" ] ; then 

    if [ $HAVEGCC -eq 1 ] ; then 
        cc="gcc"
	cco="-O"
        ldopts="-static"
    else
        cc="cc"
        cco="-O "
    fi

  fi

  if [ "$mach" = "irix6" ] ; then 

    cc="cc -n32"                  # no gcc on irix6 for now
    cco="-O "
    xroot="/usr/lib32"

    case $site in

      tenki)
        cc="gcc"                  # no gcc on irix6 for now
        cco="-O "
        xroot="/usr/lib32"
	;;

      mri)
        cc="cc -n32"                  # no gcc on irix6 for now
        cco="-O "
        xroot="/usr/lib32"
	a4flag=1
	;;

      ecmwf)
        cc="cc -n32"                  # no gcc on irix6 for now
        cco="-O "
        xroot="/usr/lib32"
	a4flag=1
	;;
      esac

  fi




#   ALPHA ALPHA ALPHA ALPHA ALPHA ALPHA ALPHA 
#
#    DEC alpha
#
#   ALPHA ALPHA ALPHA ALPHA ALPHA ALPHA ALPHA 

  if [ $mach = "alpha" ] ; then

    endian=0

#   if [ $HAVEGCC -eq 1 ] ; then
#       cc="gcc"
#	cco=""
#       ldopts="-static"
#   else
        cc="cc"
        cco="-ieee_with_no_inexact -O3"
        ldopts="-non_shared"
#   fi

    limg="$limg -ldpstk -ldps"
    ladd="-ldnet_stub"

    case $site in
        cola)
	    lx="../lib/libX11.a -ldnet_stub"
	    ix="-I/usr/include/X11"
	;;
        mri)
	    lx="-L/usr/lib -lX11 -ldnet_stub"
	    ix="-I/usr/include/X11"
            lsmice=""

	;;
	*)
	    lx="-L/usr/lib -lXext -lX11"
	    ix="-I/usr/include/X11"
	;;
    esac

  fi

#   SOL2 SOL2 SOL2 SOL2 SOL2 SOL2 SOL2 
#
#    solaris 2 (solaris 5.[2-4])
#
#   SOL2 SOL2 SOL2 SOL2 SOL2 SOL2 SOL2 

  if [ $mach = "sol2" ] ; then

    if [ $HAVEGCC -eq 1 ] ; then
        cc="gcc"
	cco=""
        ldopts="-static"
    else
        cc="cc"
	cco=""
	
    fi
    if [ $setcc -eq 1 ] ; then
        echo "doing readline on solaris without gcc"
	cco=""
    fi

    xroot="/usr/openwin"
    ladd="-lsocket -lnsl"
    xsmice=""

    case $site in
        fnmoc)
	    cc="cc"
	;;
        nrl)
	    cc="cc"
	;;
        nersc)
	    DYNOPT="-dynamic"
	    cc="gcc"
	    xroot"/usr"
        ;;
    esac

  fi

#   SOL55 SOL55 SOL55 SOL55 SOL55 SOL55 SOL55 
#
#    solaris 5.5 (sunos 5.5)
#
#   SOL55 SOL55 SOL55 SOL55 SOL55 SOL55 SOL55 

  if [ $mach = "sol55" ] ; then

    if [ $HAVEGCC -eq 1 ] ; then
        cc="gcc"
	cco=""
#        ldopts="-static"
    else
        cc="cc"
	cco=""
	
    fi
    if [ $setcc -eq 1 ] ; then
        echo "doing readline on solaris without gcc"
	cco=""
    fi

    xroot="/usr/openwin"
    xsmice="-lSM -lICE"
    ladd="-lsocket -lnsl -lw"

    case $site in
        fnmoc)
	    cc="cc"
	;;
        eorc)
	    cc="gcc"
	    dods=1
	    if [ $dodsflg -eq 0 ] ; then
	          dods=0
	    fi			
	;;
        nersc)
	    cc="gcc"
	    xroot="/usr"
	;;
        llnl)
	    cc="gcc"
	    xroot="/usr/local/X11R6.3"
            distlinkgrads="gradsnc"
	    dods=1
	    if [ $dodsflg -eq 0 ] ; then
	          dods=0
	    fi			
	;;
        cdc)
	    xroot="../$mach"
	    cc="cc"
	    cco=-g
#	    cco="-Bstatic"
	    ladd="-lsocket -lnsl -lw"
	    dods=1
	    if [ $dodsflg -eq 0 ] ; then
	          dods=0
	    fi			
	;;
    esac

  fi



#   AIX AIX AIX AIX AIX AIX AIX 
#
#    IBM AIX workstations
#
#   AIX AIX AIX AIX AIX AIX AIX 

  if [ $mach = "aix" ] ; then

    cco=""
    endian=1
    if [ $HAVEGCC -eq 1 ] ; then
        cc="gcc"
        ldopts="-static"
    else
        cc="cc"
        rl=1
    fi
    limg="$limg -ldpstk"

    case $site in
       aixcu)
           cc="xlc"
           cco="-O -D_ALL_SOURCE -U_NO_PROTO"
           xroot="../$mach"
           xsmice=""
        ;;
        *)
#
#	defaults for cumulus.gcn.ou.edu (AIX 2 3 )
#
	    cco="-O"
	    cc=cc
	    lx="-L/usr/lib/X11 -lX11"
	    ix="-I/usr/include/X11"
	    lsmice="-lSM -lICE"
        ;;
    esac

  fi



#   LINUX LINUX LINUX LINUX LINUX LINUX LINUX LINUX LINUX LINUX  
#
#	linux or linux5
#
#   LINUX LINUX LINUX LINUX LINUX LINUX LINUX LINUX LINUX LINUX  


  if [ $mach = "linux" ] ; then

    endian=0

    if [ $HAVEGCC -eq 1 ] ; then
        cc="gcc"
	ldopts="-static"
    else
        cc="cc"
	if [ $rl -eq 1 ] ; then
	    echo "Cannot build with readline on non-Sun without gcc."
        fi
        rl=0
        irl=""
        lrl=""
    fi
    cco=""
    ladd="-lbsd"

    xroot="/usr/X11R6"

    case $site in
        ncep)
            xroot="../$mach"
            distlinkgrads="gradsc"
        ;;
        navy)
            xroot="../$mach"
        ;;
    esac


  fi

  if [ $mach = "linux5" ] ; then

    endian=0

    cc="gcc"
    ldopts="-static"
    cco=" -O "

    xroot="/usr/X11R6"

  fi


  if [ $mach = "linux6" ] ; then

    endian=0

    cc="gcc"
#   ldopts="-static"
    cco=" -O "

    xroot="/usr/X11R6"

#    ladd="-L../$mach/lib -ltiff -lttf -lpng"

    case $site in
#
# 20001116 - current linux - rh7.0 can't link statically with earlier built libraries; disable
#
        llnl)
            ldopts=""
#
# 20010314 -- seems to work now
#
            ldopts="-static"
	    dods=1
	    if [ $dodsflg -eq 0 ] ; then
	          dods=0
	    fi			
        ;;
    esac

#----------------------------------------------------------------------
#  turn off static for dods
#----------------------------------------------------------------------

     if [ $dods -eq 1 ] ; then
        ldopts=""
     fi

  fi

# DARWIN  DARWIN DARWIN DARWIN DARWIN DARWIN DARWIN DARWIN DARWIN
#  Mac OS/X 10.1
# DARWIN DARWIN DARWIN DARWIN DARWIN DARWIN DARWIN DARWIN DARWIN
  
  if [ $mach = "darwin" ] ; then
    if [ $img -eq 1 ] ; then
      echo No ImageMagick support as yet. Use \"-noimg\". ; exit
    fi
    if [ $gui -eq 1 ] ; then 
      echo No GUI support as yet. Use \"-nogui\". ; exit
    fi
    cc=cc
    darwin=`uname -r`
    # note - this if statement will break when darwin hits 10.0..
    if [ $darwin "<" "6.0"  ] ; then
	echo "Darwin version $darwin is < 6.0; using c++"
	cplusplus="c++"
    else 
        echo "Darwin version $darwin is >= 6.0; using g++2"
	cplusplus="g++2"
    fi
    cco="-I."
    echo "Creating dummy malloc.h to satisfy cpp..."
    touch malloc.h
    xroot=/usr/X11R6
    libm=""
    if [ $rl -eq 1 ] ; then
      lrl="../$mach/readline/lib/libreadline.a -ltermcap"
      irl="-I ../$mach/readline/include"
    fi

    if [ $sdf -eq 1 ] ; then
       if [ $netcdf_unidata -eq 0 ] ; then
           # echo No HDF support as yet. Omit \"-hdf\". ; exit
 	echo doing hdf
       fi
       LSDF="$LSDF ../$mach/lib/libudport.a"
       if [ $dods -eq 1 ] ; then
           LSDF="../$mach/dods/lib/inst.o $LSDF"
           OURLD=c++
       fi
    fi
  fi

# CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY 
#
#  non-32 bit - crays
#
# CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY CRAY 


else

  isacray=1
  ishp64=0

  if [ $HAVEGCC -eq 1 -a $setcc -eq 0 ] ; then
    cc="gcc"
  else
    cc="cc"
  fi
  cco=""

  xroot="/usr"

#kk 020624 --- s
#   HPUX11 HPUX11 HPUX11 HPUX11 HPUX11 HPUX11 HPUX11 
#
#
#   HPUX11 HPUX11 HPUX11 HPUX11 HPUX11 HPUX11 HPUX11 

  if [ $mach = "hpux11" ] ; then
    ishp64=1
    isacray=0
    cc="cc"
    cco="-O +DD64 -I/usr/contrib/X11R6/include"
    ldopts="+DD64 -lnsl"
    lrl="../$mach/readline/libreadline.a ../$mach/lib/libtermcap.a"

    case $site in
      *)
        ;;
    esac

  fi
#kk 020624 --- e

  if [ "$mach" = "j90" ] ; then

    case $site in
      fnmoc)
        xsmice=""
      ;;
    esac
  fi
  
  if [ $mach = "c90" ] ; then

        xroot="/usr"
	xsmice=""

  fi
  
fi


#.............................. end of $mach dependent section  .................................


##################################################
#
#    Standard include search path: supp lib
#
##################################################

cco="$cco -I../$mach/include"


##################################################
#
#    X11 Setup 
#
##################################################

if [ "$xroot" = "" ] ; then
     ix=""
     lx=""
else
     ix="-I$xroot/include"
     lx="-L$xroot/lib"
fi

if [ $gui -eq 1 ] ; then
     lx="-L../$mach/lib -lfreq -lsx $lx $xtaw $xsmice $xlib"
elif [ $img -eq 1 ] ; then
     lx="$lx $xsmice $xlib"
else
     lx="$lx $xlib"
fi


##################################################
#
#    No GUI setup
#
##################################################

if [ $gui -eq 0 ] ; then
  ogui=""
  igui=""
  lgui=""
fi

##################################################
#
#    No Imagemagick setup
#
##################################################

if [ $img -eq 0 ] ; then
  oimg=""
  iimg=""
  limg=""
fi


##################################################
#
#  No readline setup
#
##################################################

if [ $rl -eq 0 -a "$mach" != "msdos" ] ; then
    irl=""
    lrl=""
fi

#DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD
#
#    echo the build parameters
#
#DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD

if [ $setcc -eq 1 ] ; then
  cc=$ccname
fi

if [ "$mach" = "linux5" -o "$mach" = "linux6" ] ; then
  buildinfo="Built on `date '+%D'` by `whoami` (`uname -sr`)"
elif [ "$mach" = "win32"  ] ; then
  buildinfo="Built on `date '+%D'` by $USER (`uname -sr`)"
else
  buildinfo="Built on `date '+%D'` by `whoami` (`uname -srv`)"
fi

echo "static char *buildinfo = \"$buildinfo\" ;" >buildinfo.h
if [ $img -eq 1 -o $img -eq 2 ] ; then
  DDIR='-DBYTEORDER='"$endian"' -DGRADS_CRAY='$isacray' -DGRADS_HP64='$ishp64' -DUSEIMG='$img' -DUSELATS='$lats' -DGXPNG='$gxpng' -DUSEGUI='$gui' -DUSESDF='$sdf' -DUSEHDF='$netcdf_ncsa' -DUSEDODS='$dods' -DREADLINE='$rl' -DGRADS_VERSION=\\"'$gver'\\" -DGRADS_DESC=\\"'$gdesc'\\" -DUSEGADODS='$gadods
  DDIR2=' -DBYTEORDER='"$endian"' -DGRADS_CRAY='$isacray' -DGRADS_HP64='$ishp64' -DUSEIMG='$img' -DUSELATS='$lats' -DGXPNG='$gxpng' -DUSEGUI='$gui' -DUSESDF='$sdf' -DUSEDODS='$dods' -DUSEHDF='$netcdf_ncsa' -DREADLINE='$rl' -DGRADS_VERSION=\\"'$gver'\\" -DGRADS_DESC=\\"'$gdesc'\\" -DUSEGADODS='$gadods
else
  DDIR='-DBYTEORDER='"$endian"' -DGRADS_CRAY='$isacray' -DGRADS_HP64='$ishp64' -DUSEIMG=0 -DUSELATS='$lats' -DGXPNG='$gxpng' -DUSEGUI='$gui' -DUSESDF='$sdf' -DUSEDODS='$dods' -DUSEHDF='$netcdf_ncsa' -DREADLINE='$rl' -DGRADS_VERSION=\\"'$gver'\\" -DGRADS_DESC=\\"'$gdesc'\\" -DUSEGADODS='$gadods
  DDIR2=' -DBYTEORDER='"$endian"' -DGRADS_CRAY='$isacray' -DGRADS_HP64='$ishp64' -DUSEIMG=0 -DUSELATS='$lats' -DGXPNG='$gxpng' -DUSEGUI='$gui' -DUSESDF='$sdf' -DUSEDODS='$dods' -DUSEHDF='$netcdf_ncsa' -DREADLINE='$rl' -DGRADS_VERSION=\\"'$gver'\\" -DGRADS_DESC=\\"'$gdesc'\\" -DUSEGADODS='$gadods
fi

#----------------------------------------------------------------------
#
#	now set the a4 option
#
#----------------------------------------------------------------------

if [ $a4flag -eq 1 ] ; then
   DDIR="$DDIR -DA4PAPER"  
   DDIR2="$DDIR2 -DA4PAPER"  
fi

echo "DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD"
echo "DD"
echo "DD descriptors of this build:"
echo "DD"
echo "DD   grads binary name:     $gradsbin"
echo "DD   internal grads desc:   $gdesc"
echo "DD   grads version:         $gver"
echo "DD   usrdir:                $usrdir"
echo "DD   bindir:                $bindir"
echo "DD   mach opts:             $arch $endian $isacray"
echo "DD   compiler:              $cc"
echo "DD   X opts:                $lx $ix"
echo "DD   readline opts:         $rl $lrl $irl"
echo "DD   a4 opts:               $a4flag"
echo "DD   SDF opts:              $sdf"
echo "DD   SDF library:           $LSDF"
echo "DD   SDF objects:           $osdf"
echo "DD   LATS netcdf:           $olats"
echo "DD   LATS CCOPTS:           $cco"
echo "DD   GXPNG:                 $gxpng"
echo "DD   DDIR (cc directives):  $DDIR"


cco2="$cco $DDIR2"
echo "cco2 set to ${cco2}."
cco="$cco $DDIR"

#MMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMM
#
#	set up the make file (Makefile) by subsitituting
#	options in the template Makefile.sh.in
#
#MMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMM
if [ $dods -eq 1 ] ; then
    OURLD=$cplusplus
else
    OURLD=$cc
fi

sed -e "s,%LIBX%,${limg} ${lx},g" \
    -e "s,\%INCX\%,${iimg} ${ix},g" \
    -e "s,\%GRADSBIN\%,${gradsbin},g" \
    -e "s,\%OBJSSDF\%,${osdf},g" \
    -e "s,\%COMPILER\%,${cc},g" \
    -e "s,\%LDOPTS\%,${ldopts},g" \
    -e "s,\%CCOPTS\%,${cco} ${ISDF},g" \
    -e "s,\%CCOPTS2\%,${cco2} ${ISDF},g" \
    -e "s,\%INCRL\%,${irl},g" \
    -e "s,\%LIBA\%,${ladd},g" \
    -e "s,\%INCLATS\%,${ilats},g" \
    -e "s,\%LIBLATS\%,${llats},g" \
    -e "s,\%INCGUI\%,${igui},g" \
    -e "s,\%LIBGUI\%,${lgui},g" \
    -e "s,\%OBJSGUI\%,${ogui},g" \
    -e "s,\%LATSNCOBJ\%,${nclats},g" \
    -e "s,\%OBJSLATS\%,${olats},g" \
    -e "s,\%OBJSIMG\%,${oimg},g" \
    -e "s,\%OBJSGADODS\%,${ogadods},g" \
    -e "s,\%OGXPNG\%,${ogxpng},g" \
    -e "s,\%IGXPNG\%,${igxpng},g" \
    -e "s,\%LGXPNG\%,${lgxpng},g" \
    -e "s,\%OURLD%,${OURLD},g" \
    -e "s,\%LIBM%,${libm},g" \
    -e "s,\%LIBRL\%,${lrl},g"  \
    -e "s,\%LIBSDF\%,${LSDF},g" \
< Makefile.sh.in > Makefile


#LLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLL
#
#	set up the appropriate "netcdf" libraries
#
#LLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLL

if [ $netcdf_unidata -eq 1 -a $sdf -eq 1 ] ; then

     echo "DD   Using support NetCDF libraries" 

elif  [ $netcdf_unidata -eq 0 -a $sdf -eq 1 ] ; then

     echo "DD   Using support MFHDF libraries" 

else

echo "DD   NOT USING NETCDF/MFHDF IN THIS BUILD....."

fi

echo "DD"
echo "DDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDDD"


#SSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSS
#
#	create the mv shell and mv the new binaries
#
#SSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSS

cat > mv.sh << E-O-F
#!/bin/sh

mv xgrads    $bindir/grads.$gver.$mach
mv xgribmap  $bindir/gribmap.$gver.$mach
mv xgxps     $bindir/gxps.$gver.$mach
mv xgxtran   $bindir/gxtran.$gver.$mach
mv xgribscan $bindir/gribscan.$gver.$mach
mv xstnmap   $bindir/stnmap.$gver.$mach
mv gxeps     $bindir/
mv wgrib     $bindir/
exit
E-O-F

chmod a+x mv.sh

cat > mv.usr.sh << E-O-F
#!/bin/sh

mv xgrads    $usrdir/grads
mv xgribmap  $usrdir/gribmap
mv xgxps     $usrdir/gxps
mv xgxtran   $usrdir/gxtran
mv xgribscan $usrdir/gribscan
mv xstnmap   $usrdir/stnmap
mv gxeps     $usrdir/
mv wgrib     $usrdir/
exit
E-O-F

chmod a+x mv.usr.sh

cat > mv.usr.win32.sh << E-O-F
#!/bin/sh

mv xgrads.exe    $usrdir/grads.exe
mv xgribmap.exe  $usrdir/gribmap.exe
mv xgxps.exe     $usrdir/gxps.exe
mv xgxtran.exe   $usrdir/gxtran.exe
mv xgribscan.exe $usrdir/gribscan.exe
mv xstnmap.exe   $usrdir/stnmap.exe
mv gxeps.exe     $usrdir/
mv wgrib.exe     $usrdir/
exit
E-O-F

chmod a+x mv.usr.win32.sh

cat > mv.x.sh << E-O-F
#!/bin/sh
mv xgrads    $bindir/xgrads
mv xgribmap  $bindir/xgribmap
mv xgxps     $bindir/xgxps
mv xgxtran   $bindir/xgxtran
mv xgribscan $bindir/xgribscan
mv xstnmap   $bindir/xstnmap
mv gxeps     $bindir/
mv wgrib     $bindir/
exit
E-O-F

chmod a+x mv.x.sh

###################################################
#
#  tar.dist.bin.sh - grads binaries
#
###################################################

cat > tar.dist.bin.sh << E-O-F
#!/bin/sh

rm -r ../dist/$distdir
 
mkdir ../dist/$distdir
mkdir ../dist/$distdirbin

cp gradsnc    ../dist/$distdirbin/gradsnc
cp gradshdf   ../dist/$distdirbin/gradshdf
cp gradsc     ../dist/$distdirbin/gradsc

cp xgribmap  ../dist/$distdirbin/gribmap
cp xgxps     ../dist/$distdirbin/gxps
cp xgxtran   ../dist/$distdirbin/gxtran
cp xgribscan ../dist/$distdirbin/gribscan
cp xstnmap   ../dist/$distdirbin/stnmap
cp gxeps     ../dist/$distdirbin/
cp gxpng     ../dist/$distdirbin/
cp wgrib     ../dist/$distdirbin/

cd ../dist

cp $distdirdoc/README.dist $distdir/README
cp $distdirdoc/README.dist.bin $distdirbin/README

#
#  create the symbolic link between grads and gradshdf
#  NOTE: On win32 cannot do symlink() on executables. The frontend
#        mechanism will take care of this.
#
if [ "$mach" != "win32" ] ; then
	cd $distdirbin
	ln -s $distlinkgrads grads
fi

#
#  create the tar file
#
cd ../..

tar -cvf $distnametar $distdir/*

compress -v $distnametar

rm -i -r ../dist/$distdir
 
exit
E-O-F

chmod a+x tar.dist.bin.sh


#MMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMM
#
#	do the make
#
#MMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMM

if [ $make -eq 0 ] ; then
   echo " "
   echo The Makefile has been prepared, please enter
   echo "   make gx            to create grads executable"
   echo "   make gxutils       to create the utilities"
   echo " "
   exit 0
else
   make gx
   make gxutils
fi


if [ $moveit = "yes" ] ; then
  ./mv.sh
fi

exit
