/*  Copyright (C) 1988-2005 by Brian Doty and the Institute
                  of Global Environment and Society (IGES).  

    See file COPYRIGHT for more information.   */

/*
 * 
 * dodstn.c: interface to gadods library, for reading remote station data
 *
 * to do:
 * 
 *  - as in BUFR datafiles, any given coordinate may occur in
 *  either header or profile of DODS data. for example, in an EPIC
 *  time series, lat/lon/lev are in the header, time is in the
 *  profile. the loop that builds garpt structures needs to handle
 *  this.
 *
 *  - queries to the EPIC system do not send the time constraints;
 *  a function needs to be written to convert floating point grid
 *  time into epic integer-format absolute time, and print that
 *  out into the constraint string.
 *  
 *  the following features will reduce unnecessary use of server resorces, 
 *  by making it quicker and easier to figure out where the data is 
 *  located in a station dataset:
 * 
 *  - ideally there should be a way to request just coordinate data,
 *  without a data variable. 
 *
 * - dodpfi should check for some kind of metadata fields that set
 *   lat/lon/lev to reasonable values if present. 
 *
 * - one could even go further, and have attributes for average
 *   number of stations, average profile length etc. in order to
 *   get a sense of how many data points are in the dataset.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>

/* If autoconfed, only include malloc.h when it's presen */
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#else /* undef HAVE_CONFIG_H */

#include <malloc.h>

#endif /* HAVE_CONFIG_H */


#include <stdio.h>
#include <string.h>
#include <math.h>
#include "grads.h"
#include "gadods.h"

/* returns dimension index (0,1,2,3 or -1 for no match) associated
   with the varid, by searching the dodinf structure */
int doddim(int varid, struct gafile *pfi) {
  int i;
  for (i = 0; i < 5; i++) {
    if (varid == pfi->dodinf[i]) {
      return i;
    }
  }
  return -1;
}

/* gets the index of variable named either name1 or if not found,
   name2 */
int dodgvar(char *name1, char *name2, struct gafile *pfi) {
  int var = gadods_d_varindex(pfi->dhandle, name1);
  if (var < 0) {
    var = gadods_d_varindex(pfi->dhandle, name2);
  }
  if (var < 0) {
    var = -999;
  }
  /*   printf("joew: var %s is %d\n", name1, var); */
  return var;
}

/* handle EPIC time format, which is a 64-bit integer representing 
*  milliseconds since 01-01-1970, broken into two 32-bit pieces */
double epict2gr(double val, double val2, struct gafile *pfi) {
  double tim;
  long ltim;
  struct dt dtbase, dtval;
  
  dtbase.yr = 1970;
  dtbase.mo = 1;
  dtbase.dy = 1;
  dtbase.hr = 0;
  dtbase.mn = 0;
  
  tim = ldexp(val,32) + val2; /* millisecs since 01-01-1970 */
  tim = (tim / 60 / 1000); /* minutes since 01-01-1970 */
  ltim = tim;
  
  dtval.yr = 0;
  dtval.mo = 0;
  dtval.mn = (ltim % 60);
  dtval.hr = (ltim / 60) % 24;
  dtval.dy = (ltim / 60 / 24);
  
  timadd(&dtbase, &dtval);

/*   printf("Decoded EPIC time: %d-%d-%d %d:%d\n",  */
/* 	 dtval.yr,dtval.mo,dtval.dy,dtval.hr,dtval.mn); */
  
  return (double)t2gr(pfi->abvals[3], &dtval);
  
  
}


/* builds URL for sending a query to epic, which uses generic DODS
 * constraint clauses (i.e. '&varname>value') instead of GDS functions
 * bounds() and stid()
 */
void epicqstr(char * buf, struct gastn * stn) {

  char * next, * stid, * ret;
  const char * name;
  struct gafile * pfi;
  int * dodinf;
  int i;
  GADODS_DATASET handle;

  pfi = stn->pfi;
  handle = pfi->dhandle;
  dodinf = pfi->dodinf;
  stid = stn->stid;

  next = buf;

  if (stn->sflag) {
    name = gadods_d_varname(handle, dodinf[4]);
    sprintf(next, "&%s=", name, stid);
    next += strlen(next);

    for (i = 0; i < 8; i++) {
      if (stid[i] == ' ') break;
      (*next) = stid[i];
      next++;
    }
    (*next) = '\0';
  } else {

    name = gadods_d_varname(handle, dodinf[0]);
    sprintf(next, "&%s>=%f&%s<=%f", name, stn->dmin[0], name, stn->dmax[0]);
    next += strlen(next);

    name = gadods_d_varname(handle, dodinf[1]);
    sprintf(next, "&%s>=%f&%s<=%f", name, stn->dmin[1], name, stn->dmax[1]);
    next += strlen(next);

  }

  if (stn->pvar->levels) {
    name = gadods_d_varname(handle, dodinf[2]);
    sprintf(next, "&%s>=%f&%s<=%f", name, stn->dmax[2], name, stn->dmin[2]);
    next += strlen(next);
  }
}

/*  Open a DODS station data set and fill in the gafile
    info from metadata from the server */

int dodpfi (char *url, struct gafile *pfi) {
  struct gavar *pvar;
  struct dt tdef,dt1,dt2;
  int i,isvert,nvars,nivars,size,lcnt,len,gotfill,tvar,tminid,tsizeid,tstepid;
  const char *name, * longname;
  GADODS_DATASET handle;
  GADODS_STATUS rc;
  float v1,v2,*vals;
  double time,fill;
  char *pos;


  rc = gadods_open(url, &handle);

  if (rc!=0) {
    gaprnt (0,"Open Error on DODS URL\n");
    return (99);
  }

  nvars = gadods_d_numvars(handle);
  if (nvars<1) {
    gaprnt (0,"Open error:  DODS URL is not a station dataset\n");
    gadods_close(handle,1);
    return(99);
  }

  /* save handle, url, and dataset title */

  pfi->type = 2;
  pfi->dhandle = handle;    /* qqq this could be a problem */

  len = 0;
  while (*(url+len) && len<4095) {
    pfi->name[len] = *(url+len);
    pfi->dnam[len] = *(url+len);
    len++;
  }
  pfi->name[len] = '\0';  pfi->dnam[len] = '\0';

  name = gadods_d_title(handle);
  if (name) {
    len = 0;
    while (*(name+len) && len<510) {
      pfi->title[len] = *(name+len);
      len++;
    }
    pfi->title[len] = '\0';
  } else {
    /* empty string for title by default */
    pfi->title[0] = '\0';
  }
  
  /* search for coordinate variables */
  pfi->dodinf[0] = dodgvar("lon", "longitude", pfi);
  pfi->dodinf[1] = dodgvar("lat", "latitude", pfi);
  pfi->dodinf[2] = dodgvar("lev", "depth", pfi);
  pfi->dodinf[3] = dodgvar("time", "time", pfi);
  pfi->dodinf[4] = dodgvar("stid", "_id", pfi);
  tvar = pfi->dodinf[3];

  /* search for data variables */
  nivars = gadods_d_numlivars(handle);

  size = nvars * (sizeof(struct gavar) + 7 );
  pvar = (struct gavar *)malloc(size);
  if (pvar==NULL) {
    gaprnt (0,"Memory allocation error in dodpfi\n");
    gadods_close(handle,1);
    return(99);
  }
  pfi->pvar1 = pvar;
  lcnt = 0;
  gotfill = 0;
  for (i=0; i<nvars; i++) {
    isvert = 0;
    if (i>=nivars) isvert = 1;
    name = gadods_d_varname(handle,i);
    longname = gadods_d_attrstr(handle, i, 
				gadods_d_attrindex(handle, i, "long_name"));
    if (!longname) longname = name;
    if (doddim(i, pfi) == -1) {
      if (!gotfill) {
	fill = -9.99e33;
	if (gadods_d_fill(handle,i,&fill) == GADODS_SUCCESS) {
	  pfi->undef = fill;
	  if (SETMISS) {
	    pfi->ulow = fabs(pfi->undef/EPSILON);
	    pfi->uhi = pfi->undef + pfi->ulow;
	    pfi->ulow = pfi->undef - pfi->ulow;
	  }
	  gotfill = 1;
	}
      }

      pvar->offset = i;
      pvar->units[0] = 99;
      pvar->levels = isvert;

      len = 0;
      while (*(name+len) && len < 16) {
        pvar->abbrv[len] = tolower(*(name+len));
        len++;
      }
      pvar->abbrv[len] = '\0';

      len = 0;
      while (*(longname+len) && len < 128) {
        pvar->varnm[len] = (*(longname+len));
        len++;
      }
      pvar->varnm[len] = '\0'; 

      pvar++;
      lcnt++;
    }
  }
  pfi->vnum = lcnt;
  pfi->ivnum = nivars-4; 
  pfi->lvnum = lcnt - pfi->ivnum;
  if (pfi->lvnum>0 && pfi->dodinf[2]<0) goto leverr;

  /* Parse tdef info provided by server */

  tminid = gadods_d_attrindex(handle,tvar, "grads_size");
  tsizeid = gadods_d_attrindex(handle,tvar, "grads_min");
  tstepid = gadods_d_attrindex(handle,tvar, "grads_step");

  if ((tminid >= 0) && (tsizeid >= 0) && (tstepid >= 0)) {
    name = gadods_d_attrstr(handle,tvar, tminid);
    if ( (pos = intprs((char *)name,&(pfi->dnum[3])))==NULL) goto tdeferr;
    
    name = gadods_d_attrstr(handle,tvar,tsizeid);
    tdef.yr = -1000;
    tdef.mo = -1000;
    tdef.dy = -1000;
    if ( (pos = adtprs((char *)name,&tdef,&dt1))==NULL) goto tdeferr;
    if (dt1.yr == -1000 || dt1.mo == -1000.0 || dt1.dy == -1000) goto tdeferr;
    
    name = gadods_d_attrstr(handle,tvar,tstepid);
    if ( (pos = rdtprs((char *)name,&dt2))==NULL) goto tdeferr;
    v1 = (dt2.yr * 12) + dt2.mo;
    v2 = (dt2.dy * 1440) + (dt2.hr * 60) + dt2.mn;
    if( (v1 == 0) && (v2 == 0) ) goto tdeferr;

  } else {
    /* If no tdef info, use default time grid - daily, starting at UNIX epoch */
    dt1.yr = 1970;
    dt1.mo = 1;
    dt1.dy = 1;
    dt1.hr = 0;
    dt1.mn = 0;
    v1 = 0;
    v2 = 1440;
  }
    
  /* The info we just collected gets hung off the pfi block
     as the time conversion constants */

  vals = (float *)malloc(sizeof(float)*8);
  if (vals==NULL) goto tdeferr;
  *(vals) = dt1.yr;
  *(vals+1) = dt1.mo;
  *(vals+2) = dt1.dy;
  *(vals+3) = dt1.hr;
  *(vals+4) = dt1.mn;
  *(vals+5) = v1;
  *(vals+6) = v2;
  *(vals+7) = -999.9;
  pfi->grvals[3] = vals;
  pfi->abvals[3] = vals;
  pfi->linear[3] = 1;

  return (0);

tdeferr:
  gaprnt (0,"Invalid tdef info from server; error in dodpfi\n");
  gadods_close(handle,1);
  return(99);
leverr:
  gaprnt (0,"Invalid lev info from server; error in dodpfi\n");
  gadods_close(handle,1);
  return(99);
}



/* Obtain data to satisfy the request described in the 
 *    gastn block 
 *
 */
int dodget(struct gastn *stn) {
  struct gavar *pvar;
  struct garpt *rpt;
  struct dt dt;
  struct gafile * pfi;
  int * dodinf;
  int rptinfo[5];
  GADODS_DATASET handle;
  GADODS_STATUS rc;
  GADODS_STN_QUERY *query;
  GADODS_RPTCOL r_handle;
  double lon,lat,lev,val, val2, time,fill;
  int nreps,nlevs,i,j,k,num, rptdatavar, gotepic;
  char tchmn[20],tchmx[20],stid[10];
  const char *stid2, *varnm;
  char extra[8192];

  pfi = stn->pfi;
  handle = pfi->dhandle;
  dodinf = pfi->dodinf;
  query = gadods_sq_new(handle);
  if (query==NULL) { 
    gaprnt (0,"Memory allocation error in dodget\n");
    return (99);
  }

  /* select variables to request */

  pvar = stn->pvar;

  for (i = 0; i < 5; i++) {
    if (i == 2 && pvar->levels == 0) continue;
    query->varflags[dodinf[i]] = 1;
  }
  query->varflags[pvar->offset] = 1;

  /* set query constraints */

  if (gadods_d_varlen(handle, dodinf[3]) > 1) {
    gotepic = 1;
    epicqstr(extra, stn);
    query->extra = extra;
    
  } else {
    gotepic = 0;

    query->minlon = stn->dmin[0];
    query->maxlon = stn->dmax[0];
    query->minlat = stn->dmin[1];
    query->maxlat = stn->dmax[1];
    query->minlev = stn->dmin[2];
    query->maxlev = stn->dmax[2];
    gr2t (stn->tvals, stn->tmin, &dt);
    gat2ch (&dt, 4, tchmn);   /* should be 5 */
    query->mintime = tchmn;
    gr2t (stn->tvals, stn->tmax, &dt);
    gat2ch (&dt, 4, tchmx);
    query->maxtime = tchmx;
    if (stn->sflag) {
      for (i=0; i<8; i++) stid[i] = stn->stid[i];
      i = 0;
      while (stid[i]!=' ' && i<8) i++;
      stid[i] = '\0';
      query->stid = stid;
    } else {
      query->stid = NULL;
    }
    query->usebounds = 1;

  }

  gaprnt(2, "gadods: requesting ");
  gaprnt(2, (char*)gadods_sq_url(query));
  gaprnt(2, "\n");

  rc = gadods_sq_send(query, &r_handle);


  if (rc) {
    gaprnt (0,"DODS data retrieval error\n");
    gadods_sq_free(query);
    return(99);
  }

  /* indices of coordinate variables in report will differ from
   * indices in original dataset, since some vars are missing */
  for (i = 0; i < 5; i++) {
    if (i == 2 && pvar->levels == 0) continue;
    varnm = gadods_d_varname(handle, dodinf[i]);
    rptinfo[i] = gadods_r_varindex(r_handle, varnm);
  }

  varnm = gadods_d_varname(handle, pvar->offset);
  rptdatavar = gadods_r_varindex(r_handle, varnm);
  
  nreps = gadods_r_numrpts(r_handle);

  num = 0;
  for (i = 0; i < nreps; i++) {

    /* get "header" info (lat/lon/time coordinates) */
    
    gadods_r_valdbl(r_handle, i, 0, rptinfo[0], 0, &lon); 
    gadods_r_valdbl(r_handle, i, 0, rptinfo[1], 0, &lat); 

    if (gotepic) {

      gadods_r_valdbl(r_handle, i, 0, rptinfo[4], 0, &val); 
      sprintf(stid, "%d", (int)val);
      
      gadods_r_valdbl(r_handle, i, 0, rptinfo[3], 0, &val); 
      gadods_r_valdbl(r_handle, i, 1, rptinfo[3], 0, &val2); 

      time = epict2gr(val, val2, pfi);

    } else {
      stid2 = gadods_r_valstr(r_handle, i, 0, rptinfo[4], 0);
      for (j = 0; j < 8; j++) {
	if (stid2[j] == '\0') break;
	stid[j] = stid2[j];
      }
      gadods_r_valdbl(r_handle, i, 0, rptinfo[3], 0, &time); 
    }

    /* pad station id with spaces */
    for (j = 0; j < 8; j++) {
      if (stid[j] == '\0') break;
    }
    while (j < 8) {
      stid[j] = ' ';
      j++;
    }
    stid[9] = '\0';

    /* get surface value or vertical profile */

    if (pvar->levels==0) {
      gadods_r_valdbl(r_handle, i, 0, rptdatavar, 0, &val);
      rpt = gaarpt(stn);
      for (k=0; k<8; k++) rpt->stid[k] = *(stid+k);  
      rpt->lon = lon;
      rpt->lat = lat;
      rpt->tim = time;
      rpt->lev = -9.99e33;
      rpt->val = val;
/* 	printf("%s (%g, %g) (@ %g) = %g\n",  */
/* 	       stid, rpt->lon, rpt->lat, rpt->tim, rpt->val); */
      num++;
    } else {
      nlevs = gadods_r_numlev(r_handle, i);
      for (j=0; j<nlevs; j++) {
	gadods_r_valdbl(r_handle, i, j, rptinfo[2], 0, &lev);
        gadods_r_valdbl(r_handle, i, j, rptdatavar, 0, &val);
        rpt = gaarpt(stn);
        for (k=0; k<8; k++) rpt->stid[k] = *(stid+k);  /* assume blanks? */
        rpt->lon = lon;
        rpt->lat = lat;
        rpt->tim = time;
        rpt->lev = lev;
        rpt->val = val;
/* 	printf("%s (%g, %g) [%g] (@ %g) = %g\n",  */
/* 	       stid, rpt->lon, rpt->lat, rpt->lev, rpt->tim, rpt->val); */
        num++;
      }
    }
  }
  stn->rnum = num;
  gadods_r_free(r_handle);
  gadods_sq_free(query);
  return (0);
}

/*  Close gadods data set */

void dodclo (struct gafile *pfi) {
GADODS_DATASET handle;

  if (pfi->dhandle == -999) return;
  handle = pfi->dhandle;
  gadods_close(handle,1);
  pfi->dhandle = -999;
}
