/*  Copyright (C) 1988-2005 by Brian Doty and the Institute
                  of Global Environment and Society (IGES).  

    See file COPYRIGHT for more information.   */

/* Authored by B. Doty and Jennifer Adams */

#ifdef HAVE_CONFIG_H
#include <config.h>

/* If autoconfed, only include malloc.h when it's presen */
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif

#else /* undef HAVE_CONFIG_H */

#include <malloc.h>

#endif /* HAVE_CONFIG_H */

#include "grads.h"
#include <math.h>
#include <stdio.h>
#if USESDF == 1
#include <netcdf.h>
#if USEHDF == 1
#include <mfhdf.h>
#endif 
#endif

/* Do garead function prototype here */
int garead (off_t, int, float *);

/* Do gafcorlf function prototype here */
off_t gafcorlf (int, int, int, int);

/* global struct for warning level setting */
extern struct gamfcmn mfcmn;

int cvhdr (unsigned char *, struct rpthdr *);
int cvflt (unsigned char *, int);

static char pout[256];    /* Build error msgs here */

/* Global pointers for this file */

static struct gafile *pfi;
static struct gagrid *pgr;
static struct gavar *pvr;
static struct gaindx *pindx;
static int timerr;
static int msgflg=1;

/* GRIB I/O caching.  GRIB data is chached, as well as the bit
   maps, if present.  Sometimes the expanded bit map is cached. */

static char *cache;           /* I/O cache for GRIB */
static char *bcache;          /* Bit map cache */
static int cflag=0;           /* cache flag */
static int bcflag=0;          /* Bit cache flag */
static int bpsav = -999;      /* Bit cache pointer */
static int bssav = -999;      /* Bit cache size */
static int *bpcach;           /* expanded bit map cache */

/* Station data I/O caching.  We will cache fairly small I/O
   requests that fit within the specified size buffer.  If the buffer
   gets overfilled, we just forget the caching.  */

static int scflg = 0;         /* Anything cached? */
static int scuca = 0;         /* Can use cache for this request */
static int scerr = 0;         /* Buffer full? */
static int scok;              /* Ok to fill buffer */
static int scpnt;             /* Current cache offset */
static int scseq;             /* File sequence of last request */
static struct gastn scstn;    /* Previous request */
static char *scbuf=NULL;      /* Cache */
      /* Size of cache */
#define SCNUM 50000


/* Routine resets flag to allow warning in regards to interpolation */

void gaiomg () {
  msgflg = 1;
}

/* Routine to obtain a grid.  The addresses of the gagrid
   structure is passed to this routine.  The storage for the
   grid is obtained and the grid is filled with data.                 */

int gaggrd (struct gagrid *pgrid) {
  int fipnt;
  float *gr;
  int x,i,id,jd,d[4],dx[4];
  float ulo, uhi, undef;
  int incr,rc,dflag;
  int size,ssz;
#if USESDF == 1
  /* prototype for SDF reader */
  int gagsdf(struct gagrid *gridptr, float grid[]) ;
#endif
  
  if (cflag) free(cache);
  cache = NULL;
  cflag = 0;
  if (bcflag) {
    free(bcache);
    free(bpcach);
  }
  bcache = NULL;
  bpcach = NULL;
  bcflag = 0;
  bssav = -999;
  bpsav = -999;


  pgr = pgrid;
  pvr = pgr->pvar;
  pfi = pgr->pfile;
  timerr = 0;
  if (pfi->idxflg) pindx = pfi->pindx;

  if (pfi->ppflag && msgflg) {
    gaprnt (3,"Notice:  Automatic Grid Interpolation Taking Place\n");
    msgflg = 0;
  }

  if (pfi->type==4) {
    rc = gagdef();
    return (rc);
  }

  /* Check dimensions we were given */

  if (pgr->idim < -1 || pgr->idim > 3 ||
      pgr->jdim < -1 || pgr->jdim > 3 ||
      ( pgr->idim == -1 && pgr->jdim!=-1 ) ) {
    sprintf (pout,"Internal logic check 16:  %i %i  \n", pgr->idim,
            pgr->jdim);
    gaprnt (0,pout);
    return (16);
  }

  /* Calc sizes and get storage for the grid */

  id = pgr->idim;
  jd = pgr->jdim;
  if (id > -1)  pgr->isiz = pgr->dimmax[id] - pgr->dimmin[id] + 1;
  else pgr->isiz = 1;
  if (jd > -1)  pgr->jsiz = pgr->dimmax[jd] - pgr->dimmin[jd] + 1;
  else pgr->jsiz = 1;

  size = pgr->isiz*pgr->jsiz;
  if (size>1) {
    ssz  = size * sizeof(float);
    gr = (float *)malloc(ssz);
    if (gr==NULL) {
      gaprnt (0,"Memory Allocation Error:  grid storage \n");
      return (1);
    }
    pgr->grid = gr;
  } else {
    pgr->grid = &(pgr->rmin);
    gr = pgr->grid;
  }

  /* Handle predefined variable */

  if (pvr->levels<-900) {
    rc = gagpre();
    return (rc);
  }

  for (i=0; i<4; i++) {d[i] = pgr->dimmin[i]; dx[i] = pfi->dnum[i];}
  dx[2] = pvr->levels;
  if (dx[2]==0) {        
    if (id==2 || jd==2) goto nozdat;
    dx[2] = 1;
    d[2] = 1;
  }

  incr = pgr->isiz;

  /* If X does not vary, make sure the X coordinate is normalized.    */

  if (id!=0 && pfi->wrap) {
    x=pgr->dimmin[0];
    while (x<1) x=x+dx[0];
    while (x>dx[0]) x=x-dx[0];
    pgr->dimmin[0]=x;
    pgr->dimmax[0]=x;
    d[0] = x;
  }

  /* If any of the non-varying dimensions are out of bounds of the
     file dimension limits, then we have a grid of missing data.
     Check for this.                                                  */

  for (i=0; i<4; i++) {
    if (id!=i && jd!=i &&
       (d[i]<1 || d[i]>dx[i]) ) goto nodat;
  }

#if USESDF == 1
/* If we don't have a regular FILE*, but we do have an IO_STD*, use SDF read */
/*  if (!(pfi->infile) && (pfi->sdf_ptr)) { */
  if (pfi->is_a_SDF) {
    rc = gagsdf(pgr, gr);

    if (rc == 0) {  
      if (SETMISS) {
	undef = pgr->undef;
	ulo = fabs(undef/EPSILON);
	uhi = undef + ulo;
	ulo = undef - ulo;       	
	for (i = 0; i < size; i++) {
	  if (*gr > ulo && *gr < uhi) {
	    *gr = undef;
	  }
	  gr++;
	}
      }
    }
    return rc;
  }
#endif

  /* Handle case where X varies.                                      */

  dflag = 0;
  if ( id == 0 ) {
    if (jd<0) jd = 1;
    for (d[jd]=pgr->dimmin[jd]; d[jd]<=pgr->dimmax[jd]; d[jd]++) {
      if (d[jd]<1 || d[jd]>dx[jd]) {
        for (i=0; i<incr; i++) *(gr+i) = pgr->undef;
      } else {
        rc = gagrow(gr, d);
        if (rc > 0 ) return (1);
        if (rc==0) dflag=1;
      }
      gr += incr;
    }
    if (!dflag) goto nodatmsg;
    return (0);
  }

  /* Handle cases where X does not vary.  We will have to read
     each point in the grid seperately.                               */

  if (jd<0) {
    if (id<0) { id=0; jd=1; }
    else jd=0;
  }

  for (d[jd]=pgr->dimmin[jd]; d[jd]<=pgr->dimmax[jd]; d[jd]++) {
    if (d[jd]<1 || d[jd]>dx[jd]) {
      for (i=0; i<incr; i++,gr++) *gr = pgr->undef;
    } else {
      for (d[id]=pgr->dimmin[id]; d[id]<=pgr->dimmax[id]; d[id]++) {
        if (d[id]<1 || d[id]>dx[id]) *gr = pgr->undef;
        else {
          rc = garrow (d[0], d[1], d[2], d[3], 1, gr);
          if (rc != 0 ) return (1);
          dflag=1;
        }
        gr++;
      }
    }
  }
  if (!dflag) goto nodatmsg;
  return (0);

nozdat:
  if(mfcmn.warnflg>0) {
    gaprnt (1,"Data Request Warning:  Varying Z dimension environment...\n");
    gaprnt (1,"  but the requested variable has no Z dimension\n");
    gaprnt (2,"  Entire grid contents are set to missing data \n");
  }
  for (i=0; i<size; i++,gr++) *gr = pgr->undef;  /* yeah, i duped it */
  return (-1);

nodat:
  for (i=0; i<size; i++,gr++) *gr = pgr->undef;

nodatmsg:
  if(mfcmn.warnflg>0) {
    gaprnt (1,"Data Request Warning:  Request beyond file limits\n");
    gaprnt (2,"  Entire grid contents are set to missing data \n");
    sprintf (pout,"  Dimension ranges are:  X = %i %i  Y = %i %i ",
	     d[0],dx[0],d[1],dx[1]);
    gaprnt (2,pout);
    sprintf (pout," Z = %i %i  T = %i %i \n",d[2],dx[2],d[3],dx[3]);
    gaprnt (2,pout);
  }
  return (-1);

}


/* gagrow gets a row of data from the file.  The row of data can
   be 'wrapped' if the x direction of the grid spans the globe.       */


int gagrow ( float *gr, int *d ) {
int fpnt;
int rc,i,x,j;
int y,z,t;


  y = *(d+1);
  z = *(d+2);
  t = *(d+3);

  /* If the needed data is within the bounds of the file dimensions
     then read the data directly.                                     */

  if (pgr->dimmin[0] >= 1 && pgr->dimmax[0] <= pfi->dnum[0]) {
    rc = garrow (pgr->dimmin[0], y, z, t,
                      (pgr->dimmax[0]-pgr->dimmin[0]+1), gr);
    if (rc != 0 ) return (1);
    return (0);
  }

  /* If the file does not wrap, then read the data directly, if
     possible.  If the requested data lies outside the file's bounds,
     fill in with missing data where appropriate.                   */

  if (!pfi->wrap) {
    if ( pgr->dimmin[0]>=1 && pgr->dimmax[0]<=pfi->dnum[0] ) {
      rc = garrow (pgr->dimmin[0], y, z, t,
                    (pgr->dimmax[0]-pgr->dimmin[0]+1), gr);
      if (rc != 0 ) return (1);
      return (0);
    }

    for (i=0; i<pgr->isiz; i++) *(gr+i) = pgr->undef;
    if (pgr->dimmin[0]<1 && pgr->dimmax[0]<1 ) return (-1);
    if (pgr->dimmin[0]>pfi->dnum[0] &&
        pgr->dimmax[0]>pfi->dnum[0] ) return (-1);
    i = 1 - pgr->dimmin[0];
    if (i>0) gr+=i;
    i = 1;
    if (pgr->dimmin[0]>1) i = pgr->dimmin[0];
    j = pgr->dimmax[0];
    if (j > pfi->dnum[0]) j = pfi->dnum[0];
    j = 1 + (j - i);
    rc = garrow (i, y, z, t, j, gr);
    if (rc != 0 ) return (1);
    return (0);
  }

  /* When the file wraps, we read the entire row into the row buffer, and
     copy the values as needed into locations in the requested row.    */
  rc = garrow (1, y, z, t, pfi->dnum[0], pfi->rbuf);
  if (rc != 0 ) return (1);

  for (x=pgr->dimmin[0];x<=pgr->dimmax[0];x++) {
    i=x;
    while (i<1) i = i + pfi->dnum[0];
    while (i>pfi->dnum[0]) i = i-(pfi->dnum[0]);    /* Best way??? */
    *gr = *((pfi->rbuf)+i-1);
    gr++;
  }
  return (0);
}

long gafcor ( int x, int y, int z, int t) {
off_t pos;
long ltmpz,ltmpy,ltmpt;
int yy,zz;
int levs;

  levs=pvr->levels;
  if(levs == 0) levs=1;
  if (pfi->tlpflg) {
    t = t + pfi->tlpst;
    if (t > pfi->dnum[3]) t = t - pfi->dnum[3];
  }

  if (pfi->yrflg) yy = pfi->dnum[1] - y;
  else yy = y-1;

  if (pfi->zrflg) {
    if (pvr->levels==0) zz=0;
    else zz = pvr->levels-z;
  } else zz = z-1;

  if(pvr->var_t) {
    pos = (t-1)*(pfi->gsiz)*levs +
      pvr->offset +
	zz*(pfi->gsiz) +
	  yy*(pfi->dnum[0]) +
	    (x-1);
  } else {
    ltmpt=(long)(t-1)*((long)pfi->tsiz);
    ltmpz=zz*(pfi->gsiz)*((long)pvr->var_z); 
    ltmpy=yy*(pfi->dnum[0]);
    pos = ltmpt+
    (long)pvr->offset +
      ltmpz +
        ltmpy +
          ((long)x-1); 
  }

  if (pfi->xyhdr) pos = pos + pfi->xyhdr;
  if (pfi->thdr) pos = pos + pfi->thdr;
  return (pos);
}



off_t gafcorlf (int x, int y, int z, int t) {
off_t pos;
off_t ltmpz,ltmpy,ltmpt;
off_t yy,zz;
off_t levs;
off_t xl, yl, zl, tl;

  xl = x;
  yl = y;
  zl = z;
  tl = t;
  levs=(off_t)pvr->levels;
  if(levs == 0) levs=1;
  if (pfi->tlpflg) {
    tl = tl + (off_t)pfi->tlpst;
    if (tl > (off_t)pfi->dnum[3]) tl = tl - (off_t)pfi->dnum[3];
  }

  if (pfi->yrflg) {
    yy = (off_t)pfi->dnum[1] - yl;
  }
  else {
    yy = yl - 1;
  }

  if (pfi->zrflg) {
    if (levs==0) {
      zz = 0;
    }
    else {
      zz = levs - zl;
    }
  } 
  else {
    zz = zl - 1;
  }
  if (pvr->var_t) {
    pos = (tl-1)*(off_t)(pfi->gsiz)*levs +
      (off_t)pvr->offset +
	zz*(off_t)(pfi->gsiz) +
	  yy*(off_t)(pfi->dnum[0]) +
	    (xl-1);
  } else {
    ltmpt=(tl-1)*((off_t)pfi->tsiz);
    ltmpz=zz*(off_t)(pfi->gsiz)*((off_t)pvr->var_z); 
    ltmpy=yy*((off_t)pfi->dnum[0]);
    pos = ltmpt +
      (off_t)pvr->offset +
        ltmpz +
          ltmpy +
            (xl-1); 
  }

  if (pfi->xyhdr) pos = pos + (off_t)pfi->xyhdr;
  if (pfi->thdr)  pos = pos + (off_t)pfi->thdr;
  return (pos);
}

long gafcyx ( int x, int y, int z, int t) {
off_t pos;
int yy,zz;

  if (pfi->tlpflg) {
    t = t + pfi->tlpst;
    if (t > pfi->dnum[3]) t = t - pfi->dnum[3];
  }

  if (pfi->yrflg) yy = pfi->dnum[0] - y;
  else yy = y-1;

  if (pfi->zrflg) {
    if (pvr->levels==0) zz=0;
    else zz = pvr->levels-z;
  } else zz = z-1;

  if(pvr->var_t) {
    pos = (t-1)*(pfi->gsiz)*(pvr->levels) +
      pvr->offset +
	zz*(pfi->gsiz) +
	  (x-1)*(pfi->dnum[1]) +
	    yy;
  } else {
    pos = (t-1)*(pfi->tsiz) +
      pvr->offset +
	zz*(pfi->gsiz)*(pvr->var_z) +
	  (x-1)*(pfi->dnum[1]) +
	    yy;
  }

  if (pfi->xyhdr) pos = pos + pfi->xyhdr;
  if (pfi->thdr) pos = pos + pfi->thdr;
  return (pos);
}


int gardyx (off_t fpos, int len, float *gr) {
char *ch1,*ch2,*ch3,*ch4,cc1,cc2;
int rc,i,j;
off_t pos;
pos = fpos;

for (i=1;i<=len;i++) {

  rc = fseeko(pfi->infile, pos*sizeof(float)+pfi->fhdr, 0);
  if (rc!=0) {
    gaprnt (0,"Low Level I/O Error:  Seek error on data file \n");
    sprintf (pout,"%d  rc=%ld pos=%ld pfi->fhdr =%ld \n",__LINE__,rc,pos,pfi->fhdr);
    gaprnt (0,pout);
    sprintf (pout,"  Data file name = %s \n",pfi->name);
    gaprnt (0,pout);
    sprintf (0,"  Error occurred when seeking to byte %ld \n",fpos);
    gaprnt (0,pout);
    return (1);
  }
  rc = fread (gr, sizeof(float), 1, pfi->infile);
  if (rc<1) {
    gaprnt (0,"Low Level I/O Error:  Read error on data file \n");
    sprintf (pout,"  Data file name = %s \n",pfi->name);
    gaprnt (0,pout);
    sprintf (pout,"  Error reading %i bytes at location %ld \n",
	     len, fpos);
    gaprnt (0,pout);
    return (1);
  }

  /* Do byte swapping if needed */

  if (pfi->bswap) {
    ch1 = (char *)gr;
    ch2 = ch1+1;
    ch3 = ch2+1;
    ch4 = ch3+1;
    for (i=0; i<len; i++) {
      cc1 = *ch1;
      cc2 = *ch2;
      *ch1 = *ch4;
      *ch2 = *ch3;
      *ch3 = cc2;
      *ch4 = cc1;
      ch1+=4; ch2+=4; ch3+=4; ch4+=4;
    }
  }

  /* Set missing data values to exact value if specified */
  if (SETMISS) {
    for (j=0;j<len;j++) {
      if (*gr >= pfi->ulow && *gr <= pfi->uhi) *gr = pgr->undef;
      gr++;
    }
  }
  pos=pos+pfi->dnum[1];

}

return (0);

}


/*  Basic read of a row of data elements -- a row is always
    in the X direction, which for grads binary is the fastest
    varying dimension */

int garrow (int x, int y, int z, int t, int len, float *gr) {
  int rc,i=0,tt,oflg;
  off_t fposlf;

  tt = t;
  if (pfi->tmplat) {
    tt = gaopfn(t,&oflg,pfi);
    if (tt==-99999) return(1);
    if (tt==-88888) {
      for (i=0; i<len; i++) *(gr+i) = pfi->undef;
      return (0);
    }
    if (oflg) bpsav = -999;  /* Force new bit map cache if new file opened */
  }
  
  if (pfi->ppflag) {               /* Preprojected (pdef) */
    rc = gaprow (x, y, z, t, tt, len, gr);
    return (rc);
  }
  
#if USESDF == 1
  if (pfi->ncflg==1) {              /* netcdf */
    rc = ganrow(x,y,z,tt,len,gr);
    return(rc);
  }
#endif

# if USEHDF == 1
  if (pfi->ncflg==2) {              /* HDF-SDS */
    rc = gahrow(x,y,z,tt,len,gr);
    return(rc);
  }
#endif
  
  if (pfi->idxflg) {               /* Indexed (grib) */
    rc = gairow (x, y, z, t, i, len, gr);
    return (rc);
  }
  
  if(pvr->y_x) {            
    fposlf = gafcyx (x,y,z,tt);
    rc = gardyx (fposlf, len, gr);
  } else {                      /* if none of the above... binary */
    fposlf = gafcorlf (x,y,z,tt);
    rc = garead (fposlf, len, gr);
  }
  return (rc);
}
  
int garead (off_t fpos, int len, float *gr) {
char *ch1,*ch2,*ch3,*ch4,cc1,cc2;
unsigned char *uch1,*uch2,ucc1,ucc2;
int rc,i;
int j,cnt,ival,*ig;
float *fg;

unsigned char *igr;
unsigned char *cgr;
signed char chsign;
unsigned char chunsign;
off_t ffpos;

  if(pvr->dfrm == 1) {
    ffpos = fpos*(off_t)sizeof(char)+(off_t)pfi->fhdr;
  } else if(pvr->dfrm == 2 || pvr->dfrm == -2 ) {
    ffpos = fpos*2ll + (off_t)pfi->fhdr;
  } else if(pvr->dfrm == 4) {
    ffpos = fpos*(off_t)sizeof(int)+(off_t)pfi->fhdr;
  } else if(pfi->cray_ieee) {
    ffpos =  fpos*4ll+(off_t)pfi->fhdr;
  } else {
    ffpos = fpos*(off_t)sizeof(float)+(off_t)pfi->fhdr;
  }

  rc = fseeko(pfi->infile, ffpos, 0);

  if (rc!=0) {
    gaprnt (0,"Low Level I/O Error:  Seek error on data file \n");
    sprintf (pout,"  Data file name = %s \n",pfi->name);
    gaprnt (0,pout);
    sprintf (pout,"%d rc=%d pos=%ld pfi->fhdr =%d\n",__LINE__,rc,fpos,pfi->fhdr);
    gaprnt (0,pout);
    sprintf (0,"  Error occurred when seeking to byte %ld \n",fpos);
    gaprnt (0,pout);
    return (1);
  }

  if(pvr->dfrm == 1) {

    j = len*sizeof(char);
    igr = (unsigned char *)malloc(len*sizeof(char));
    if (igr==NULL) {
      gaprnt (0,"Memory Allocation Error:  char grid storage \n");
      return (1);
    }

    rc = fread (igr, 1, len, pfi->infile);
    for(i=0;i<len;i++) {
      *(gr+i) = (float)(*(igr+i));
    }
    free(igr);

/*mf 961127 handle integer*2 for NCEP CPC  does byteswapping too mf*/

  } else if(pvr->dfrm == 2 || pvr->dfrm == -2 ) {

    j = len*2;
    cgr = (unsigned char *)malloc(len*2);
    if (cgr==NULL) {
      gaprnt (0,"Memory Allocation Error:  integer*2 storage \n");
      return (1);
    }

    rc = fread (cgr, 2, len, pfi->infile);
    cnt=0;

/*mf - 961127 - byteswapping mf*/

    if(pfi->bswap) {
      uch1 = cgr;
      uch2 = uch1+1;
      for (i=0; i<len; i++) {
	ucc1 = *uch1;
	ucc2 = *uch2;
	*uch1 = ucc2;
	*uch2 = ucc1;
	uch1+=2; uch2+=2;
      }
    }

/*mf - 961127 - signed integer*2 mf*/

    if(pvr->dfrm == -2) {

      for(i=0;i<len;i++) {
	ival=(int)(*(cgr+cnt)*256) + (int)((*(cgr+cnt+1))) - 65536*(*(cgr+cnt)>>7);
	*(gr+i) = (float)ival;
	cnt+=2;
      }

/*mf - 961127 - unsigned integer*2 mf*/

    } else {

      for(i=0;i<len;i++) {
	ival=(int)(*(cgr+cnt)*256) + (int)((*(cgr+cnt+1)));
	*(gr+i) = (float)ival;
	cnt+=2;
      }

    }

    free(cgr);

  } else if(pvr->dfrm == 4) {

    j = len*sizeof(int);
    ig = (int *)malloc(len*sizeof(int));
    if (ig==NULL) {
      gaprnt (0,"Memory Allocation Error:  integer*4 storage \n");
      return (1);
    }

    rc = fread (ig, sizeof(int), len, pfi->infile);

    for(i=0;i<len;i++) {
      if (pfi->bswap) {
	ch1 = (char *)(ig+i);
	ch2 = ch1+1;
	ch3 = ch2+1;
	ch4 = ch3+1;
	cc1 = *ch1;
	cc2 = *ch2;
	*ch1 = *ch4;
	*ch2 = *ch3;
	*ch3 = cc2;
	*ch4 = cc1;
      }
      *(gr+i) = (float)(*(ig+i));
    }
    free(ig);

  } else if(pfi->cray_ieee) {

/*mf 970112 -- use wesley ebisuzaki's routines ---mf*/

    cgr = (unsigned char *)malloc(len*4);
    if (cgr==NULL) {
      gaprnt (0,"Memory Allocation Error:  32-bit grid storage on cray \n");
      sprintf (pout,"len*4 = %d\n",len*4);
      gaprnt (0,pout);
      return (1);
    }

    rc = fread (cgr, 4, len, pfi->infile);
    if (rc<len) {
      gaprnt (0,"Low Level I/O Error:  Read error on 32-bit grid data file on cray \n");
      sprintf (pout,"  Data file name = %s \n",pfi->name);
      gaprnt (0,pout);
      sprintf (pout,"  Error reading %i bytes at location %li \n",
	       len, fpos);
      gaprnt (0,pout);
      free(cgr);
      return (1);
    }

    for(i=0;i<len;i++) {

      if (pfi->bswap) {
	ch1 = (char *)cgr;
	ch2 = ch1+1;
	ch3 = ch2+1;
	ch4 = ch3+1;
	cc1 = *ch1;
	cc2 = *ch2;
	*ch1 = *ch4;
	*ch2 = *ch3;
	*ch3 = cc2;
	*ch4 = cc1;
      }

      *(gr+i) = ieee2flt(cgr);
      cgr+=4;

    }

    free(cgr);

  } else {
    /* standard direct access read */
    rc = fread (gr, sizeof(float), len, pfi->infile);
  }

  if (rc<len) {
    gaprnt (0,"Low Level I/O Error:  Read error on data file \n");
    sprintf (pout,"  Data file name = %s \n",pfi->name);
    gaprnt (0,pout);
    sprintf (pout,"  Error reading %i bytes at location %li \n",
            len, fpos);
    gaprnt (0,pout);
    return (1);
  }

  /* Do byte swapping if needed, only if !CRAY  and !cray_ieee*/

  if (pfi->bswap && !GRADS_CRAY && ( pvr->dfrm != 4 ) && ( abs(pvr->dfrm) != 2 ) && ( pvr->dfrm != 1 ) && !pfi->cray_ieee ) {
    ch1 = (char *)gr;
    ch2 = ch1+1;
    ch3 = ch2+1;
    ch4 = ch3+1;
    for (i=0; i<len; i++) {
      cc1 = *ch1;
      cc2 = *ch2;
      *ch1 = *ch4;
      *ch2 = *ch3;
      *ch3 = cc2;
      *ch4 = cc1;
      ch1+=4; ch2+=4; ch3+=4; ch4+=4;
    }
  }

  /* Set missing data values to exact value if specified */
  if (SETMISS) {
    for (i=0;i<len;i++) {
      if (*gr >= pfi->ulow && *gr <= pfi->uhi)  *gr = pgr->undef;
      gr++;
    }
  }

  return (0);
}

/* Handle a station data request */

int gagstn (struct gastn *stn) {
  struct garpt *rpt;
  struct stninf sts;
  struct rpthdr ghdr, *hdr;
  int i,j,k,rc,ii,flag,tim,fnum,rtot,rdw,nsiz;
  int dpos,selflg,oflg;
  off_t fpos;
  int sizhdrf,sizhdrd,sizf,sizd,idum;
  float lnmin,lnmax,ltmin,ltmax,hlon;
  char ch1[16],ch2[16],*ch;
  char rec[256];


  stn->rpt = NULL;
  for (i=0; i<BLKNUM; i++) {
    stn->blks[i] = NULL;
  }

  if (stn->pfi->bufrflg) {         /* bufrstn */
    rc = getbufr(stn);             /* bufrstn */
    return (rc);                   /* bufrstn */
  }

#if USEGADODS
  if (stn->pfi->dhandle > -999) {  /* dodstn */
    rc = dodget(stn);              /* dodstn */
    return (rc);                   /* dodstn */
  }
#endif

  /* Determine cache situation */
  if (scbuf==NULL && !scerr) {
    scbuf = (char *)malloc(SCNUM);
    if (scbuf==NULL) {
      gaprnt (0,"Memory allocation error:  Stn data cache buffer\n");
      gaprnt (0,"  Station data cache disabled\n");
      scerr = 1;
    }
    scflg = 0;
  }

  scuca = 0;
  scpnt = 0;
  if (!scerr) scok = 1;
  if (scflg && stn->pfi->fseq != -999 && scseq == stn->pfi->fseq &&
      scstn.pfi==stn->pfi && scstn.idim==stn->idim &&
      scstn.jdim==stn->jdim && scstn.tmin==stn->tmin &&
      scstn.tmax==stn->tmax && scstn.rflag==stn->rflag &&
      scstn.sflag==stn->sflag) {
    rc = 1;
    if (stn->rflag && scstn.radius!=stn->radius) rc = 0;
    if (stn->sflag) {
      for (i=0; i<8; i++) if (scstn.stid[i]!=stn->stid[i]) rc=0;
    } else {
      if (scstn.dmin[0]!=stn->dmin[0]) rc = 0;
      if (scstn.dmin[1]!=stn->dmin[1]) rc = 0;
      if (scstn.dmax[0]!=stn->dmax[0]) rc = 0;
      if (scstn.dmax[1]!=stn->dmax[1]) rc = 0;
    }
    if (rc) {
      scuca = 1;
      scok = 0;
    }
  }

  pvr = stn->pvar;
  pfi = stn->pfi;
  hdr = &ghdr;

  lnmin = stn->dmin[0]; lnmax = stn->dmax[0];
  ltmin = stn->dmin[1]; ltmax = stn->dmax[1];
  if (stn->rflag) {
    lnmin = lnmin - stn->radius;
    lnmax = lnmax + stn->radius;
    ltmin = ltmin - stn->radius;
    ltmax = ltmax + stn->radius;
  }
  stn->rnum = 0;

/* set size of the file and data hdr */
  sizhdrf = sizeof(struct rpthdr);
  sizhdrd = sizeof(struct rpthdr);
  if(pfi->cray_ieee) {
    sizhdrf=8+((sizeof(struct rpthdr)-8)/2);
  }

  /* Loop through times looking for appropriate stations */
  for (tim=stn->tmin; tim<=stn->tmax; tim++) {
    if (tim<1) continue;
    if (tim > pfi->dnum[3]) break;

    if (!scuca && pfi->tmplat) {
      rc = gaopfn(tim,&oflg,pfi);
      if (rc==-99999) goto err;
      if (rc==-88888) {
        if (scok) {
          hdr->nlev = 0;
          gacstn((char *)hdr, NULL, 0, sizhdrd);
        }
        continue;
      }
    }

    /* Loop through stations for this time looking for valid reports */
    if (!scuca) {
      fpos = *(pfi->tstrt+tim-1);
      rc = gasstn(fpos);
      if (rc) goto err;
    }

    while (1) {
      /* get the header */
      if (scuca) {                             /* from the cache */
	gagcst (sizhdrd, (char *)hdr);
      } else {                                 /* from the file */
        if (pfi->seqflg) {
          rc = garstn(4,(char *)(&rdw),fpos);  /*mf changed 4 to sizeof(int) */
          if (rc) goto err;
          if (pfi->bswap) gabswp((float *)(&rdw),1);
	  if(pfi->cray_ieee) {
	    idum=be_int2int((unsigned char*)(&rdw));
	    rdw=idum;
	  }

        }

        rc = garstn (sizhdrf, (char *)hdr, fpos);
        if (rc) goto err;
        if (pfi->bswap) gahswp(hdr);
	if(pfi->cray_ieee) {
	  memcpy(rec,hdr,sizhdrf);
	  rc=cvhdr((unsigned char *)rec,hdr);
	}
      }

      if (hdr->nlev==0) break;   /* END OF DATA CHECK */

      /* Left justify the station id if there are leading blanks */

      j = 0;
      while (j<7 && hdr->id[0]==' ') {
        for (i=0; i<7; i++) hdr->id[i] = hdr->id[i+1];
        hdr->id[7] = ' ';
        j++;
      }

      /* Determine if we want to read the data portion of this report */
      selflg = 1;
      if (stn->sflag) {
        getwrd (ch1,hdr->id,8);
        lowcas(ch1);
        getwrd(ch2,stn->stid,8);
        if (!cmpwrd(ch1,ch2)) selflg = 0;
      } else {
        hlon = hdr->lon;
        if (hlon<lnmin) hlon+=360.0;
        if (hlon>lnmax) hlon-=360.0;
        if (hlon<lnmin || hlon>lnmax ||
            hdr->lat<ltmin || hdr->lat>ltmax ) selflg=0;
        if (selflg && stn->rflag &&
            hypot(hlon-stn->dmin[0],hdr->lat-stn->dmin[1])>stn->radius) {
          selflg = 0;
        }
      }

      /* Determine size of the data portion of this report */
      if (hdr->flag) {
	fnum = (hdr->nlev-1) * (pfi->lvnum+1) + pfi->ivnum;
      } else {
	fnum =  hdr->nlev * (pfi->lvnum+1);
      }

      /* calc size of floating point data section in the FILE not the machine */
      sizd=fnum*sizeof(float);
      if(pfi->cray_ieee) {
	sizf= fnum*4;
      } else {
	sizf=fnum*sizeof(float);
      }

      /* Read the data portion of this report, byteswap it if needed,
	 and set exact missing data values if specified.*/
      if (selflg) {
        if (scuca) {                                  /* from the cache */
	  gagcst (sizd, (char *)pfi->rbuf);
        } else {                                       /* from the file */
          if (pfi->seqflg) {
            ch = (char *)(pfi->rbuf);
            nsiz = rdw - sizhdrf;
            if (nsiz>0) {
              rc = garstn(nsiz,ch,fpos);
              if (rc) goto err;
              ch += nsiz;
            }
            rtot = rdw;
            nsiz = sizf + sizhdrf;
            while (rtot<=nsiz) {
              fpos = fpos + rdw + 8;
              rc = gasstn(fpos);
              if (rc) goto err;
              if (rtot==nsiz) break;
              rc = garstn(4,(char *)(&rdw),fpos);  /*mf changed 4 to sizeof(int) */
              if (rc) goto err;
              if (pfi->bswap) gabswp((float *)(&rdw),1);
	      if(pfi->cray_ieee) {
		idum=be_int2int((unsigned char*)(&rdw));
		rdw=idum;
	      }
              rtot +=rdw;
              if (rtot>nsiz) break;
              rc = garstn(rdw,ch,fpos);

              if (rc) goto err;
	      idum=rdw*2;
              ch += idum;
            }
            if (rtot>nsiz) {
              gaprnt (0,"Low Level I/O Error:  Sequential read error\n");
              gaprnt (0,"  Record size exceeds report size\n");
              sprintf (pout,"  Data file name = %s \n",pfi->name);
              gaprnt (0,pout);
              goto err;
            }

	  /* normal read -- NON sequential */
          } else {
            rc = garstn (sizf, (char *)pfi->rbuf, fpos+sizhdrf);
            fpos = fpos + sizf + sizhdrf;
            if (rc) goto err;
          }
          if (pfi->bswap) gabswp(pfi->rbuf,fnum);
	  /* convert to float on cray */
	  if(pfi->cray_ieee) {
	    rc=cvflt((unsigned char*)pfi->rbuf,fnum);
	  }
          if (SETMISS) {
            for (i=0; i<fnum; i++) {
              if ((*(pfi->rbuf+i) >= pfi->ulow) && (*(pfi->rbuf+i) <= pfi->uhi))
                 *(pfi->rbuf+i) = pfi->undef;
            }
          }
        }

        /* Check the data portion for any matches.  */
        rc = gaglvs (tim,hdr,stn);
        if (rc) goto err;

        /* Cache this report if appropriate */
        if (scok) gacstn((char *)hdr,(char *)pfi->rbuf,sizd,sizhdrd);

      /* Skip the data portion of this report.*/
      } else {
        if (scuca) {
          gaprnt (0,"Logic Error 8 in gaio\n");
          goto err;
        }
        if (pfi->seqflg) {
          rtot = rdw;
          sizf += sizhdrf;
          while (rtot<=sizf) {
            fpos = fpos + rdw + 8;
            rc = gasstn(fpos);
            if (rc) goto err;
            if (rtot==sizf) break;
            rc = garstn(4,(char *)(&rdw),fpos);    /*mf changed 4 to sizeof(int) */
            if (rc) goto err;
            if (pfi->bswap) gabswp((float *)(&rdw),1);
	    if(pfi->cray_ieee) {
	      idum=be_int2int((unsigned char*)(&rdw));
	      rdw=idum;
	    }
            rtot +=rdw;
          }
          if (rtot>sizf) {
            gaprnt (0,"Low Level I/O Error:  Sequential read error\n");
            gaprnt (0,"  Record size exceeds report size\n");
            sprintf (pout,"  Data file name = %s \n",pfi->name);
            gaprnt (0,pout);
            goto err;
          }
        } else {
          fpos = fpos + sizf + sizhdrf;
          rc = gasstn(fpos);
          if (rc) goto err;
        }
      }  /* END OF if (scuca) -- use the cache or not */
    }    /* END OF  while (1) */

    if (scok) {
      hdr->nlev = 0;
      gacstn((char *)hdr, NULL, 0,sizhdrd);
    }
  }
  if (scok) {
    scflg = 1;
    scstn = *stn;
    scseq = stn->pfi->fseq;
  } else scflg = 0;
  if (scuca) scflg = 1;
  return (0);

err:
  for (i=0; i<BLKNUM; i++) {
    if (stn->blks[i] != NULL) free (stn->blks[i]);
  }
  return (1);
}

/* Select appropriate variable and levels from report, and chain
   them off the stn block.  */

int gaglvs (int tim, struct rpthdr *hdr, struct gastn *stn) {
struct garpt *rpt;
float *vals,*pval;
int i,k,voff,mlev;

  vals = pfi->rbuf;
  voff = pvr->offset;
  if (pvr->levels==0) {
    if (hdr->flag) {
      pval = vals+voff;
      rpt = gaarpt (stn);
      if (rpt==NULL) return(1);
      rpt->lat = hdr->lat;
      rpt->lon = hdr->lon;
      rpt->lev = -9.99e33;
      rpt->tim = tim + hdr->t;
      rpt->val = *pval;
      for (k=0; k<8; k++) *(rpt->stid+k) = *(hdr->id+k);
      stn->rnum++;
    }
  } else {
    if (hdr->flag) vals = vals + pfi->ivnum;
    mlev = hdr->nlev;
    if (hdr->flag) mlev--;
    for (i=0; i<mlev; i++) {
      pval = vals+(i*(pfi->lvnum+1));
      if (stn->dmax[2]==stn->dmin[2]) {
        if (fabs(*pval-stn->dmin[2])>0.01) continue;
      } else {
        if (*pval<stn->dmax[2] || *pval>stn->dmin[2]) continue;
      }
      rpt = gaarpt (stn);
      if (rpt==NULL) return(1);
      rpt->lat = hdr->lat;
      rpt->lon = hdr->lon;
      rpt->lev = *pval;
      rpt->tim = tim + hdr->t;
      rpt->val = *(pval+voff+1);
      for (k=0; k<8; k++) *(rpt->stid+k) = *(hdr->id+k);
      stn->rnum++;
    }
  }
  return (0);
}

/* Allocate a rpt structure, return pointer to allocated buffer.
   On the first request, stn->rpt should be set to NULL. */

struct garpt *gaarpt (struct gastn *stn) {
struct garpt *rpt;
int i;

  /* First time through, define the static variables. */

  if (stn->rpt == NULL) {
    stn->prev = &(stn->rpt);
    for (i=0; i<BLKNUM; i++) {
      stn->blks[i] = NULL;
    }
    stn->rptcnt = RPTNUM;    /* Force new block allocation */
    stn->blkcnt = -1;
  }

  stn->rptcnt++;
  rpt = stn->crpt;

  if (stn->rptcnt>=RPTNUM) {
    stn->blkcnt++;
    if (stn->blkcnt==BLKNUM) {
      printf ("Out of memory blocks to allocate \n");
      return(NULL);
    }
    rpt = (struct garpt *)malloc(sizeof(struct garpt)*(RPTNUM+2));
    if (rpt==NULL) {
      printf ("Couldn't allocate memory for stn block \n");
      return(NULL);
    }
    stn->blks[stn->blkcnt] = rpt;
    stn->rptcnt = 0;
  } else rpt++;

  *(stn->prev) = rpt;
  stn->prev = &(rpt->rpt);
  rpt->rpt = NULL;
  stn->crpt = rpt;
  return(rpt);
}

void gacstn (char *hdr, char *rdat, int siz, int sizhdr) {
int i;
  if (scpnt+sizhdr*2+siz+10 > SCNUM) {
    scok = 0;
  } else {
    for (i=0; i<sizhdr; i++) *(scbuf+scpnt+i) = *(hdr+i);
    scpnt += sizhdr;
    if (siz>0) {
      for (i=0; i<siz; i++) *(scbuf+scpnt+i) = *(rdat+i);
      scpnt += siz;
    }
  }
}

/* Return info from the station data cache */

void gagcst (int siz, char *ch) {
int i;
  for (i=0; i<siz; i++) *(ch+i) = *(scbuf+scpnt+i);
  scpnt += siz;
}

/* Seek to specified location in a station data file */

int gasstn (off_t fpos) {
int rc;

  rc = fseeko(pfi->infile, fpos, 0);
  if (rc!=0) {
    gaprnt (0,"Low Level I/O Error:  Seek error on data file \n");
    sprintf (pout,"  Data file name = %s \n",pfi->name);
    gaprnt (0,pout);
    sprintf (pout,"%d  rc=%ld pos=%ld pfi->fhdr =%ld \n",__LINE__,rc,fpos,pfi->fhdr);
    gaprnt (0,pout);
    sprintf (pout,"  Error occurred when seeking to byte %li \n",fpos);
    gaprnt (0,pout);
    return (1);
  }
  return (0);
}

/* Read specified amount of data from a station data file */

int garstn (int siz, char *val, off_t fpos) {

int rc;

  rc = fread (val, siz, 1, pfi->infile);
  if (rc<1) {
    gaprnt (0,"Low Level I/O Error:  Read error on data file \n");
    sprintf (pout,"  Data file name = %s \n",pfi->name);
    gaprnt (0,pout);
    sprintf (pout,"  Error reading %i bytes at location %li \n",
            siz, fpos);
    gaprnt (0,pout);
    return (1);
  }
  return (0);
}

/*  Obtain user requested grid from defined variable */

int gagdef (void) {
int id, jd, i, flag;
int ys,zs,ts,siz,pos;
int d[4],d1min,d1max,d2min,d2max,xt,yt;
float *v;

  /* If a dimension is a fixed dimension in the defined
     variable, it must be a fixed dimension in the output
     grid.  */

  id = pgr->idim;
  jd = pgr->jdim;
  if (jd>-1) {
    if (pfi->dnum[jd]==1) {
      jd = -1;
      pgr->jdim = -1;
      pgr->jsiz = -1;
    }
  }
  if (id>-1) {
    if (pfi->dnum[id]==1) {
      id = jd;
      pgr->idim = pgr->jdim;
      pgr->isiz = pgr->jsiz;
      pgr->igrab = pgr->jgrab;
      pgr->iabgr = pgr->jabgr;
      pgr->ivals = pgr->jvals;
      pgr->iavals = pgr->javals;
      pgr->ilinr = pgr->jlinr;
      jd = -1;
      pgr->jdim = -1;
      pgr->jsiz = 1;
    }
  }

  /* Set up constants for array subscripting */

  ys = pfi->dnum[0];
  zs = ys * pfi->dnum[1];
  ts = zs * pfi->dnum[2];

  /* Set up dimension ranges */

  for (i=0; i<4; i++) d[i] = pgr->dimmin[i] - pfi->dimoff[i] - 1;
  for (i=0; i<4; i++) if (pfi->dnum[i]==1) d[i] = 0;
  if (id>-1) {
    d1min = d[id];
    d1max = pgr->dimmax[id] - pfi->dimoff[id] - 1;
  }
  if (jd>-1) {
    d2min = d[jd];
    d2max = pgr->dimmax[jd] - pfi->dimoff[jd] - 1;
  }

  /* Get storage for output grid */

  pgr->isiz = 1;
  pgr->jsiz = 1;
  if (id>-1) pgr->isiz = 1 + d1max - d1min;
  if (jd>-1) pgr->jsiz = 1 + d2max - d2min;
  siz = pgr->isiz * pgr->jsiz;
  if (siz>1) {
    pgr->grid = (float *)malloc(sizeof(float)*siz);
    if (pgr->grid==NULL) {
      gaprnt (0,"Memory Allocation Error: Grid Request\n");
      return (2);
    }
  } else {
    pgr->grid = &(pgr->rmin);
  }

  /* Normalize time coordinate if not varying */
  /* This does not handle leap years properly!!!!  Gotta fix this
     someday */

  if (pfi->climo && id!=3 && jd!=3) clicyc(d+3);

  /* Check for entirely undefined grid */

  flag = 0;
  for (i=0; i<4; i++) {
    if (i!=id && i!=jd && (d[i]<0 || d[i]>=pfi->dnum[i])) flag = 1;
  }
  if (flag) {
    for (i=0; i<siz; i++) *(pgr->grid+i) = pfi->undef;
    return (0);
  }

  /* Move appropriate grid values */

  if (id==-1 && jd==-1) {
    pos = d[0] + d[1]*ys + d[2]*zs + d[3]*ts;
    pgr->rmin = *(pfi->rbuf+pos);
    return (0);
  }

  v = pgr->grid;

  if (jd==-1) {
    for (xt=d1min; xt<=d1max; xt++) {
      d[id] = xt;
      if (id==3 && pfi->climo) clicyc(d+3);
      if (d[id]<0 || d[id]>=pfi->dnum[id]) *v = pfi->undef;
      else {
        pos = d[0] + d[1]*ys + d[2]*zs + d[3]*ts;
        *v = *(pfi->rbuf+pos);
      }
      v++;
    }
    return (0);
  }

  for (yt=d2min; yt<=d2max; yt++) {
    d[jd] = yt;
    if (jd==3 && pfi->climo) clicyc(d+3);
    for (d[id]=d1min; d[id]<=d1max; d[id]++) {
      if (d[jd]<0 || d[jd]>=pfi->dnum[jd] ||
          d[id]<0 || d[id]>=pfi->dnum[id]) *v = pfi->undef;
      else {
        pos = d[0] + d[1]*ys + d[2]*zs + d[3]*ts;
        *v = *(pfi->rbuf+pos);
      }
      v++;
    }
  }
  return(0);
}

void clicyc (int *ti) {
struct dt dtim,otim;
float tt;

  if (pfi->climo>0) {
    while (*ti>pfi->cysiz-1) *ti = *ti - pfi->cysiz;
    while (*ti<0) *ti = *ti + pfi->cysiz;
  }
}

/* Fill in grid for predefined variable */

int gagpre (void) {
float (*conv)(float *, float);
int d[4],id,jd,i,j,dim;
float *gr,*vals,t;

  id = pgr->idim;
  jd = pgr->jdim;
  for (i=0; i<4; i++) d[i] = pgr->dimmin[i];

  dim = pvr->offset;
  conv = pfi->gr2ab[dim];
  vals = pfi->grvals[dim];

  gr = pgr->grid;

  if (id>-1 && jd>-1) {
    for (d[jd]=pgr->dimmin[jd]; d[jd]<=pgr->dimmax[jd]; d[jd]++) {
      for (d[id]=pgr->dimmin[id]; d[id]<=pgr->dimmax[id]; d[id]++) {
        t = (float)(d[dim]);
        *gr = conv(vals, t);
        gr++;
      }
    }
  } else if (id>-1) {
    for (d[id]=pgr->dimmin[id]; d[id]<=pgr->dimmax[id]; d[id]++) {
      t = (float)(d[dim]);
      *gr = conv(vals, t);
      gr++;
    }
  } else {
    t = (float)(d[dim]);
    *gr = conv(vals, t);
  }
  return (0);
}


/* Read index data, in this case GRIB type data.
   Currently assumes no pole point, and only one record
   per grid.  */

int gairow (int x, int y, int z, int t, int offset, int len, float *gr) {
int irec,ioff,bstrt,bend,blen,cstrt,cend,clen,rc;
 int ival,i,yy,bpos,boff,siz,gtyp,xsiz,ysiz;
 off_t fpos;
float dsf,bsf,ref;

  /* Figure out position and length of the I/O */

  gtyp = *(pindx->hipnt+3);
  irec = (t-1)*pfi->trecs + pvr->recoff + z - 1;
  if (pfi->ppflag) {xsiz = pfi->ppisiz; ysiz = pfi->ppjsiz;}
  else  {xsiz = pfi->dnum[0]; ysiz = pfi->dnum[1];}
  if (gtyp==29) {
    xsiz = 145;
    irec = irec*6;
    if (y<37) y--;
    else {irec+=3; y-=37; }
    yy = y;
  } else {
    irec = irec*3;
    if (pfi->yrflg) yy = ysiz - y;
    else yy = y-1;
  }
  if (pfi->ppflag) ioff = offset;
  else ioff = yy*xsiz + x - 1;
  boff = ioff;
  blen = *(pindx->intpnt + irec + 2);
  if (blen<0) {
    for (i=0; i<len; i++) *(gr+i) = pfi->undef;
    return (0);
  }
  bpos = *(pindx->intpnt + irec + 1);
  dsf = *(pindx->fltpnt+irec);
  bsf = *(pindx->fltpnt+irec+1);
  ref = *(pindx->fltpnt+irec+2);
  if (bpos>-900 && bpos!=bpsav) {
    bpsav = bpos;
    siz = 2+(xsiz*ysiz)/8;
    if (siz>bssav) {
      if (bcflag) {
        free(bcache);
        free(bpcach);
      }
      bcache = (char *)malloc(siz);
      bpcach = (int *)malloc(sizeof(int)*(xsiz*ysiz+1));
      if (bcache==NULL||bpcach==NULL) {
        gaprnt(0,"Memory Allocation Error During GRIB I/O\n");
        return (1);
      }
      bssav = siz;
      bcflag = 1;
    }
    rc = fseeko(pfi->infile, bpos, 0);
    rc = fread(bcache,1,siz,pfi->infile);
    if (rc!=siz) {
      gaprnt(0,"GRIB I/O Error: Bit Map I/O\n");
      return(1);
    }
    boff=1;
    for (i=0; i<xsiz*ysiz; i++) {
      if (gagbb(bcache,i,1)) {
        *(bpcach+i) = boff;
        boff++;
      } else {
        *(bpcach+i) = -1*boff;
      }
    }
    *(bpcach+xsiz*ysiz) = boff;  /* Provide an ending offset */
  }
  if (bpos>-900) {
    boff = *(bpcach+ioff);
    if (boff<0) boff = -1*boff;
    boff--;
    bstrt = blen * boff;
    boff = *(bpcach+ioff+len);
    if (boff<0) boff = -1*boff;
    boff--;
    bend = blen * boff - 1;
  } else {
    bstrt = blen * boff;
    bend = bstrt + blen*len;
  }
  cstrt = bstrt/8;
  cend = bend/8;
  clen = cend-cstrt+2;
  fpos = *(pindx->intpnt+irec);
  rc = gaird(fpos,cstrt,clen,xsiz,ysiz,blen);
  if (rc) return(rc);
  bstrt = bstrt - cstrt*8;
  for (i=0; i<len; i++) {
    if (bpos>-900 && *(bpcach+ioff+i)<0) *(gr+i) = pfi->undef;
    else {
      ival = gagbb(pfi->pbuf,bstrt,blen);
      *(gr+i) = ( ref + (float)ival * bsf )/dsf;
      bstrt += blen;
    }
  }
  return (0);
}

int gaird (off_t fpos, int cstrt, int clen, int xsiz, int ysiz, int blen) {
int rc,siz,i;
  if (pfi->ppflag && pgr->idim==0 && pgr->jdim==1) {
    if (!cflag) {
      cflag = 1;
      siz = 5 + xsiz*ysiz*blen/8;  /* qqq  Warning:  siz calc does not */
                                   /* qqq  take into account bms!!! */
      cache = (char *)malloc(siz);
      if (cache==NULL) {
        gaprnt(0,"GRIB Memory Allocation Error\n");
        return (1);
      }
      rc = fseeko(pfi->infile, fpos, 0);
      rc = fread(cache,sizeof(char),siz,pfi->infile);
      if (rc==0) {
        sprintf (pout,"GRIB I/O Error reading %i bytes at %ld\n",siz,fpos);  /* xxx */
        gaprnt (0,pout);
        gaprnt (0,"  File name is: ");
        if (pfi->tempname) gaprnt(0,pfi->tempname);
        else gaprnt(0,pfi->name);
        gaprnt (0,"\n");
        return (1);
      }
    }
    if (cache==NULL) return(1);
    for (i=0; i<clen; i++) {
      *(pfi->pbuf+i) = *(cache+cstrt+i);
    }
  } else {
    rc = fseeko(pfi->infile, fpos+cstrt, 0);
    rc = fread (pfi->pbuf, sizeof(char), clen, pfi->infile);
    if (rc==0) {
      sprintf (pout,"GRIB I/O Error reading %i bytes at %ld\n",clen,fpos+cstrt);
      gaprnt (0,pout);
      gaprnt (0,"  File name is: ");
      if (pfi->tempname) gaprnt(0,pfi->tempname);
      else gaprnt(0,pfi->name);
      gaprnt (0,"\n");
      return(1);
    }
  }
  return(0);
}

/* Routine to open appropriate file when using file templates */
/* Warning -- changes time value to time with respect to this file */

int gaopfn(int t, int *oflg, struct gafile *pfi) {
int i,rc;
struct dt dtim, dtimi;
char *fn;

  *oflg = 0;
  if (t<1 || t>pfi->dnum[3]) return(-99999);
  i = *(pfi->fnums+t-1);

  /* The current file is not the one we need; close it */
  if (i != pfi->fnumc) {   
    /* close SDF file */
    if (pfi->ncflg) {
#if USESDF == 1
      if (pfi->ncflg==1) {
	if (pfi->ncid != -999) ncclose(pfi->ncid);  /* no error checking */
      }
#if USEHDF == 1
      else if (pfi->ncflg==2) {
	if (pfi->sdid != -999) SDend(pfi->sdid);     /* no error checking */
      }
#endif
#endif
    } 
    /* close BUFR file*/
    else if (pfi->bufrflg) {
      if (pfi->bufrdset) {
	gabufr_close(pfi->bufrdset);  /* free memory */
	pfi->bufrdset=NULL;           /* reset the pointer */
      }
    } 
    /* close non-SDF, non-BUFR file */
    else {
      if (pfi->infile!=NULL) fclose(pfi->infile);
    }

    /* find the filename that goes with current time */
    if (pfi->tempname!=NULL) free(pfi->tempname);
    gr2t(pfi->grvals[3], (float)t, &dtim);
    gr2t(pfi->grvals[3], 1.0, &dtimi);
    fn = gafndt(pfi->name, &dtim, &dtimi, pfi->abvals[3], pfi->pchsub1, t);
    if (fn==NULL) return (-99999);

    /* Open the data file */
    rc = 0;
    pfi->tempname = fn;
    pfi->fnumc = i;
    /* open netcdf */
    if (pfi->ncflg==1) {  
      rc = gaopnc (pfi,1,0);
      if (rc) pfi->ncid = -999;
    }
    /* open hdfsds */
    else if (pfi->ncflg==2) { 
      rc = gaophdf (pfi,1,0);
      if (rc) pfi->sdid = -999;
    }
    /* open all others except BUFR */
    else if (!pfi->bufrflg) {        
      pfi->infile = fopen (fn, "rb");
      if (pfi->infile == NULL) rc = 1;
    } 

    /* Error checking on file open */
    if (rc) {
      if (pfi->errflg && timerr!=t) {
        gaprnt(1,"Warning: Open error, fn = ");
        gaprnt(1,fn);
        gaprnt(1,"\n");
        timerr = t;
      }
      pfi->fnumc = 0;
      return(-88888);
    }
    *oflg = 1;
  }
  t = 1 + t - pfi->fnumc;
  return (t);
}

/* Read in a row of data from a pre-projected grid data set.
   This involves doing interpolation to the lat-lon
   grid */

int gaprow (int x, int y, int z, int t, int tt, int len, float *gr) {
float p[4],dx,dy,g1,g2;
float vals[9],wts[9],sum,wt;
int ioffs[9],cnt,ig0,goflg;
int rc,i,j,ig,ioff,ix,iy,ncig,ncjg;
off_t pos,pos0;

  /* Handle generalized arbitrary points + weights */

  if (pfi->ppflag==8) {
    cnt = (int)(pfi->ppvals[0]+0.1);
    pos0 = (tt-1)*(pfi->tsiz) + pvr->offset + (z-1)*(pfi->gsiz);
    ig0 = (y-1) * pfi->dnum[0] + x - 1;
    for (i=0; i<len; i++) {
      ig = ig0 + i;
      goflg = 0;
      for (j=0; j<cnt; j++) {
        ioffs[j] = *(pfi->ppi[j]+ig);
        if (ioffs[j] >= 0 && ioffs[j] <= pfi->gsiz) goflg = 1;
        wts[j] = *(pfi->ppf[j]+ig);
      }
      if (!goflg) *gr = pgr->undef;
      else {
        goflg = 1;
        j = 0;
        sum = 0.0; wt = 0.0;
        while (j<cnt) {
          if (ioffs[j] > 0) {
            pos = pos0 + ioffs[j] - 1;
            if (pfi->idxflg) {
	      rc = gairow(x,y,z,t,ioffs[j],1,vals+j);
	    }
            else if (pfi->ncflg==1) {
	      /* The y-argument to ganrow/gahrow is hardcoded to 1 for PDEF file option */
	      rc = ganrow(ioffs[j],1,z,tt,1,vals+j); /* PDEF FILE for netcdf */
	    }
            else if (pfi->ncflg==2) {
	      rc = gahrow(ioffs[j],1,z,tt,1,vals+j); /* PDEF FILE for hdfsds */
	    }
            else 
	      rc = garead(pos,1,vals+j);
            if (rc) return(rc);
            if ( *(vals+j)==pgr->undef) {
              goflg = 0;
              break;
            }
            sum = sum + *(vals+j) * *(wts+j);
            wt = wt + *(wts+j);
          }
          j++;
        }
        if (goflg && wt!=0.0) *gr = sum/wt;
        else *gr = pgr->undef;
      }
      gr++;
    }
  } else {
    for (i=0; i<len; i++) {
      ig = (y-1) * pfi->dnum[0] + x + i - 1;
      ioff = *(pfi->ppi[0]+ig);
      if (ioff<0) *gr = pgr->undef;
      else {
        dx = *(pfi->ppf[0]+ig);
        dy = *(pfi->ppf[1]+ig);
        pos = (tt-1)*(pfi->tsiz) + pvr->offset + (z-1)*(pfi->gsiz) + ioff;

	/* Get the first two pre-projected grid values */
        if (pfi->idxflg) {
	  rc = gairow(x,y,z,t,ioff,2,p);
	}
        else if (pfi->ncflg==1) {
	  ncig = (int)(1 + ioff%pfi->ppisiz);
          ncjg = (int)(1 + ioff/pfi->ppisiz);
	  rc = ganrow(ncig,ncjg,z,tt,2,p); /* PDEF BILIN for netcdf */
	}
        else if (pfi->ncflg==2) {
	  gaprnt(0,"PDEF not fully implemented for dtype hdfsds\n");
	  rc = 1; 
	}
        else {
	  rc = garead(pos,2,p);
	}
        if (rc) return(rc);

	/* Get the second two pre-projected grid values */
        if (pfi->idxflg) {
	  rc = gairow(x,y,z,t,ioff+pfi->ppisiz,2,p+2);
	}
        else if (pfi->ncflg==1) {
          ncjg++;
	  rc = ganrow(ncig,ncjg,z,tt,2,p+2); /* PDEF BILIN for netcdf */
	}
        else if (pfi->ncflg==2) {
	  gaprnt(0,"PDEF not fully implemented for dtype hdfsds\n");
	  rc = 1; 
	}
        else {
	  rc = garead(pos+pfi->ppisiz,2,p+2);
	}
        if (rc) return(rc);

	/* Do the bilinear interpolation, as long as we have no undefs */
        if (p[0]==pgr->undef || p[1]==pgr->undef ||
            p[2]==pgr->undef || p[3]==pgr->undef) *gr = pgr->undef;
        else {
          g1 = p[0] + (p[1]-p[0])*dx;
          g2 = p[2] + (p[3]-p[2])*dx;
          *gr = g1 + (g2-g1)*dy;
        }
      }
      gr++;
    }
  }
  return(0);
}

/* Convert header */

int cvhdr (unsigned char *rec, struct rpthdr *hdr) {
int rc;
  memcpy(hdr->id,&rec[0],8);
  hdr->lat=ieee2flt(&rec[8]);
  hdr->lon=ieee2flt(&rec[12]);
  hdr->t=ieee2flt(&rec[16]);
  hdr->nlev=be_int2int(&rec[20]);
  hdr->flag=be_int2int(&rec[24]);
  return (0);
}


int cvflt(unsigned char *rec, int fnum) {
float *val;
int i;
  val=(float *)malloc(sizeof(float)*fnum);
  for(i=0;i<fnum;i++) {
    *(val+i)=ieee2flt(&rec[i*4]);
  }
  memcpy(rec,val,fnum*sizeof(float));
  free(val);
  return (0);
}

 
/* Open a netcdf file */
int gaopnc (struct gafile *pfil, int tflag, int eflag) {
#if  USESDF == 1
  struct gavar *pvarl;
  int i,rc;
  
#if USEHDF == 1
  if (tflag) {
    i = ncopen (pfil->tempname, NC_NOWRITE); 
  } 
  else {
    i = ncopen (pfil->name, NC_NOWRITE); 
  }
  if (i == -1) {
    if (eflag) {
      gaprnt(0,"NetCDF Error (ncopen): Unable to open file\n");
    }
    return(1);
  }
#else
  if (tflag) {
    rc = nc_open(pfil->tempname, NC_NOWRITE, &i);
  } 
  else {
    rc = nc_open(pfil->name, NC_NOWRITE, &i);
  }
  if (rc != NC_NOERR) {
    if (eflag) {
      gaprnt(0,"NetCDF Error (nc_open): Unable to open file\n");
    }
    return (1);
  }
#endif
 
  pfil->ncid = i;     
  pvarl = pfil->pvar1;
  for (i=0; i<pfil->vnum; i++) {
    if (pvarl->ncvid != -888) pvarl->ncvid = -999;
    pvarl++;
  }
  return (0);
#else
  return(1);
#endif
}


/* Read a row varying in the X direction from a netcdf grid */
int ganrow (int x, int y, int z, int t, int len, float *gr) {
#if USESDF == 1
  int     vid,error,rc,dim,i,j,yy,zz;
  size_t  start[16],count[16];
  nc_type data_dtype,attr_dtype;
  short   *sval,sundef,sscale,sadd;
  long    *lval,lundef,lscale,ladd;
  double  dscale,dadd;
  float   val,ulow,uhi;
  int oldncopts ;         /* to save and restore setting for automatic error handling */
  
  /* Turn off automatic error handling. */
  ncopts = NC_VERBOSE ;
  oldncopts = ncopts ;
  ncopts = 0;

  /* Get the varid if we haven't already done that for this file */
  if (pvr->ncvid == -888) {
    ncopts = oldncopts ;
    return(1);  /* already tried and failed */
  }
  if (pvr->ncvid == -999) {
    error=0;
#if USEHDF == 1
    if (pvr->longnm) {
      vid = ncvarid(pfi->ncid, pvr->longnm);
    }
    else {
      vid = ncvarid(pfi->ncid, pvr->abbrv);
    }
    if (vid == -1) error=1;
#else    
    if (pvr->longnm) {
      rc = nc_inq_varid(pfi->ncid, pvr->longnm, &vid);
    }
    else {
      rc = nc_inq_varid(pfi->ncid, pvr->abbrv, &vid);
    }
    if (rc != NC_NOERR) error=1;
#endif
    if (error) {
      pvr->ncvid = -888;  /* set flag so we won't try this variable ever again */
      if (pvr->longnm) {
	sprintf(pout,"Error: Variable %s not in netcdf file\n",pvr->longnm);
      }
      else {
	sprintf(pout,"Error: Variable %s not in netcdf file\n",pvr->abbrv);
      }
      gaprnt (0,pout);
      ncopts = oldncopts ;
      return (1);
    }
    /* No errors, so we can set the varid in the gavar structure */
    pvr->ncvid = vid;
    
    /* If undef attribute name is given, get the undef value */
    if (pfi->undefattrflg) {
#if USEHDF == 1
      /* get the undef attribute type */
      if (ncattinq(pfi->ncid, pvr->ncvid, pfi->undefattr, &attr_dtype, NULL) == -1) {
	sprintf(pout,"Warning: Could not retrieve data type for \"%s\"  -- using %g instead\n",
		pfi->undefattr,pfi->undef);
	gaprnt(1,pout);
	pvr->undef = pfi->undef;
      }
      /* get the undef attribute value */
      switch(attr_dtype)
      {
      case (NC_SHORT):
	if (ncattget(pfi->ncid, pvr->ncvid, pfi->undefattr, &sundef) == -1) {
	  sprintf(pout,"Warning: Could not retrieve \"%s\" -- using %g instead\n",
		  pfi->undefattr,pfi->undef);
	  gaprnt(1,pout);
	  pvr->undef = pfi->undef;
	}
	else {
	  pvr->undef = (float)sundef;  /* Cast short to float */
	}
	break;
      case (NC_LONG):
	if (ncattget(pfi->ncid, pvr->ncvid, pfi->undefattr, &lundef) == -1) {
	  sprintf(pout,"Warning: Could not retrieve \"%s\" -- using %g instead\n",
		  pfi->undefattr,pfi->undef);
	  gaprnt(1,pout);
	  pvr->undef = pfi->undef;
	}
	else {
	  pvr->undef = (float)lundef;  /* Cast long to float */
	}
	break;
      case (NC_FLOAT):
	if (ncattget(pfi->ncid, pvr->ncvid, pfi->undefattr, &val) == -1) {
	  sprintf(pout,"Warning: Could not retrieve \"%s\" -- using %g instead\n",
		  pfi->undefattr,pfi->undef);
	  gaprnt(1,pout);
	  pvr->undef = pfi->undef;
	}
	else {
	  pvr->undef = val;             /* Value is already a float */
	}
	break; 
      default:
	sprintf(pout,"Warning: data type for \"%s\" not handled -- using %g instead\n", 
		pfi->undefattr,pfi->undef);
	gaprnt(1,pout);
	pvr->undef = pfi->undef;
      };
#else
      if (nc_get_att_float(pfi->ncid, pvr->ncvid, pfi->undefattr, &val) != NC_NOERR) {
	sprintf(pout,"Warning: Could not retrieve \"%s\" -- using %g instead\n",
		pfi->undefattr,pfi->undef);
	gaprnt(1,pout);
	pvr->undef = pfi->undef;
      }
      else {
	pvr->undef = val;
      }
#endif
    }
    /* If no undef attribute name is given, copy the file-wide undef */
    else {
      pvr->undef = pfi->undef;
    }

    /* If data are packed, get the scale factor and offset attribute values */
    if (pfi->packflg) {
      /* initialize values */
      pvr->scale=1.0;
      pvr->add=0.0;

      /* get scale factor */
#if USEHDF == 1
      /* get the scale factor attribute type */
      if (ncattinq(pfi->ncid, pvr->ncvid, pfi->scattr, &attr_dtype, NULL) == -1) {
	gaprnt(1,"Warning: Could not retrieve scale factor data type -- setting scale factor to 1.0\n");
	pvr->scale = 1.0;
      }
      else {
	/* get the scale factor attribute value */
	switch(attr_dtype)
        {
	case (NC_SHORT):
	  if (ncattget(pfi->ncid, pvr->ncvid, pfi->scattr, &sscale) == -1) {
	    gaprnt(1,"Warning: Could not retrieve scale factor -- setting to 1.0\n");
	    pvr->scale = 1.0;
	  }
	  else {
	    pvr->scale = (float)sscale;  /* Cast short to float */
	  }
	  break;
	case (NC_LONG):
	  if (ncattget(pfi->ncid, pvr->ncvid, pfi->scattr, &lscale) == -1) {
	    gaprnt(1,"Warning: Could not retrieve scale factor -- setting to 1.0\n");
	    pvr->scale = 1.0;
	  }
	  else {
	    pvr->scale = (float)lscale;  /* Cast long to float */
	  }
	  break;
	case (NC_FLOAT):
	  if (ncattget(pfi->ncid, pvr->ncvid, pfi->scattr, &val) == -1) {
	    gaprnt(1,"Warning: Could not retrieve scale factor -- setting to 1.0\n");
	    pvr->scale = 1.0;
	  }
	  else {
	    pvr->scale = val;            /* Value is already a float */
	  }
	  break; 
	case (NC_DOUBLE):
	  if (ncattget(pfi->ncid, pvr->ncvid, pfi->scattr, &dscale) == -1) {
	    gaprnt(1,"Warning: Could not retrieve scale factor -- setting to 1.0\n");
	    pvr->scale = 1.0;
	  }
	  else {
	    pvr->scale = (float)dscale;  /* Cast double to float */
	  }
	  break;
	default:
	  gaprnt(1,"Warning: scale factor data type not handled -- setting scale factor to 1.0\n");
	  pvr->scale = 1.0;
	};
      }
#else
      /* get the scale factor attribute value */
      if (nc_get_att_float(pfi->ncid, pvr->ncvid, pfi->scattr, &val) != NC_NOERR) {
	gaprnt(1,"Warning: Could not retrieve scale factor -- setting to 1.0\n");
	pvr->scale = 1.0;
      }
      else {
	pvr->scale = val;
      }
#endif

      /* get add offset if required */
      if (pfi->packflg == 2) {
#if USEHDF == 1
	/* get the add offset attribute type */
	if (ncattinq(pfi->ncid, pvr->ncvid, pfi->ofattr, &attr_dtype, NULL) == -1) {
	  gaprnt(1,"Warning: Could not retrieve add offset data type -- setting add offset to 0.0\n");
	  pvr->add = 0.0;
	}
	else {
	  /* get the add offset attribute value */
	  switch(attr_dtype)
          {
	  case (NC_SHORT):
	    if (ncattget(pfi->ncid, pvr->ncvid, pfi->ofattr, &sadd) == -1) {
	      gaprnt(1,"Warning: Could not retrieve add offset -- setting to 0.0\n");
	      pvr->add = 0.0;
	    }
	    else {
	      pvr->add = (float)sadd;  /* Cast short to float */
	    }
	    break;
	  case (NC_LONG):
	    if (ncattget(pfi->ncid, pvr->ncvid, pfi->ofattr, &ladd) == -1) {
	      gaprnt(1,"Warning: Could not retrieve add offset -- setting to 0.0\n");
	      pvr->add = 0.0;
	    }
	    else {
	      pvr->add = (float)ladd;  /* Cast long to float */
	    }
	    break;
	  case (NC_FLOAT):
	    if (ncattget(pfi->ncid, pvr->ncvid, pfi->ofattr, &val) == -1) {
	      gaprnt(1,"Warning: Could not retrieve add offset -- setting to 0.0\n");
	      pvr->add = 0.0;
	    }
	    else {
	      pvr->add = val;           /* Value is already a float */
	    }
	    break; 
	  case (NC_DOUBLE):
	    if (ncattget(pfi->ncid, pvr->ncvid, pfi->ofattr, &dadd) == -1) {
	      gaprnt(1,"Warning: Could not retrieve add offset -- setting to 0.0\n");
	      pvr->add = 0.0;
	    }
	    else {
	      pvr->add = (float)dadd;  /* Cast double to float */
	    }
	    break;
	  default:
	    gaprnt(1,"Warning: add offset data type not handled -- setting add offset to 0.0\n");
	    pvr->add = 0.0;
	  };
	}
#else
	/* get the add offset attribute value */
	if (nc_get_att_float(pfi->ncid, pvr->ncvid, pfi->ofattr, &val) != NC_NOERR) {
	  gaprnt(1,"Warning: Could not retrieve add offset -- setting to 0.0\n");
	  pvr->add = 0.0;
	}
	else {
	  pvr->add = val;
	}
#endif
      } /* matches if (pfi->packflg == 2) {  */
    }  /* matches if (pfi->packflg) {  */
  }

  /* Change the Y indexes if yrev flag is set */
  if (pfi->yrflg) {
    yy = pfi->dnum[1] - y;
  }
  else {
    yy = y-1;
  }
  /* Change the Z indexes if zrev flag is set */
  if (pfi->zrflg) {
    if (pvr->levels==0) {
      zz=0;
    }
    else {
      zz = pvr->levels-z;
    }
  } 
  else {
    zz = z-1;
  }

  /* Set up the start and count array.  The units values
     provided for each variable indicate the mapping of the 
     netcdf variable shape into the grads dimensions */
  for (i=0; i<16; i++) {
    start[i] = -999;
    count[i] = -999;
    if (pvr->units[i] == -100) { start[i] = x-1; count[i] = len; }
    if (pvr->units[i] == -101) { start[i] = yy;  count[i] = 1; }
    if (pvr->units[i] == -102) { start[i] = zz;  count[i] = 1; }
    if (pvr->units[i] == -103) { start[i] = t-1; count[i] = 1; }
    if (pvr->units[i] >=0) { start[i] = pvr->units[i];  count[i] = 1; }
  }

  /* Now we are ready to do the I/O  */
#if USEHDF == 1

  /* Get the data type */
  if (ncvarinq (pfi->ncid, pvr->ncvid, NULL, &data_dtype, NULL, NULL, NULL) == -1) {
    gaprnt(0,"NetCDF Error (ncvarinq): Unable to retrieve variable data type\n");
    ncopts = oldncopts ;
    return (1);
  }
  switch(data_dtype)
  {
  case (NC_SHORT):
    sval = (short *)malloc(len * sizeof(NC_SHORT));
    if (sval==NULL) {
      gaprnt(0,"Unable to allocate memory for data type NC_SHORT\n");
      ncopts = oldncopts ;
      return(1);
    }
    if (ncvarget (pfi->ncid, pvr->ncvid, (long *)start, (long *)count, (void*)sval) == -1) {
      gaprnt(0,"NetCDF Read Error (ncvarget) for data type NC_SHORT\n");
      ncopts = oldncopts ;
      return (1);
    }
    for (i=0; i<len; i++) gr[i] = (float)sval[i];  /* Cast short to float */
    free(sval);
    break;

  case (NC_LONG):
    lval = (long *)malloc(len * sizeof(NC_LONG));
    if (lval==NULL) {
      gaprnt(0,"Unable to allocate memory for data type NC_LONG\n");
      ncopts = oldncopts ;
      return(1);
    }
    if (ncvarget (pfi->ncid, pvr->ncvid, (long *)start, (long *)count, (void*)lval) == -1) {
      gaprnt(0,"NetCDF Read Error (ncvarget) for data type NC_LONG\n");
      ncopts = oldncopts ;
      return (1);
    }
    for (i=0; i<len; i++) gr[i] = (float)lval[i];  /* Cast long to float */
    free(lval);
    break;

  case (NC_FLOAT):
    if (ncvarget (pfi->ncid, pvr->ncvid, (long *)start, (long *)count, gr) == -1) {
      gaprnt(0,"NetCDF Read Error (ncvarget) for data type NC_FLOAT\n");
      ncopts = oldncopts ;
      return (1);
    }
    break; 

  default:
    sprintf(pout,"NetCDF Error: Data type %d not handled\n", data_dtype);
    gaprnt(0,pout);
    ncopts = oldncopts ;
    return(1);
  };
#else
  rc = nc_get_vara_float(pfi->ncid, pvr->ncvid, start, count, gr);
  if (rc != NC_NOERR) {
    sprintf(pout,"NetCDF Error (nc_get_vara_float): %s\n",nc_strerror(rc));
    gaprnt(0,pout);
    ncopts = oldncopts ;
    return (1);
  }
#endif

  /* Set missing data values to gafile undef and then unpack if necessary */
  if (SETMISS) {
    /* use the gavar undef to set the fuzzy test limits */
    /* If gavar undef equals zero, change it to 1/EPSILON */
    if (pvr->undef == 0) {   
      ulow = 1e-5;
    } 
    else {
      ulow = fabs(pvr->undef/EPSILON);   
    }
    uhi  = pvr->undef + ulow;
    ulow = pvr->undef - ulow;
    /* set the gagrid undef equal to the gafile undef */
    pgr->undef = pfi->undef;           

    /* Do the fuzzy test for undef values before unpacking */
    for (i=0;i<len;i++) {

      /* set the missing data to the gafile undef */
      if (*(gr+i) >= ulow && *(gr+i) <= uhi) {
	*(gr+i) = pfi->undef;   
      }
      else {
	/* Data is not missing, so unpack with scale and offset if necessary */
	if (pfi->packflg) {
	    *(gr+i) = *(gr+i)*pvr->scale + pvr->add; 
	}
      }
    }
  }


  ncopts = oldncopts ;
  return(0);
#else 
  gaprnt(0,"Reading NetCDF files is not supported in this build\n");
  return(1);  
#endif
}

/* Open an HDF-SDS file */
int gaophdf (struct gafile *pfil, int tflag, int eflag) {
#if USESDF == 1
#if USEHDF == 1
struct gavar *pvarl;
int i,rc;
int32 sd_id;

  sd_id = -999;
  if (tflag) {
    sd_id = SDstart(pfil->tempname, DFACC_READ);
  } 
  else {
    sd_id = SDstart(pfil->name, DFACC_READ);
  }
  if (sd_id==FAIL) {
    if (eflag) {
      gaprnt(0,"HDF-SDS Error: Unable to open file\n");
    }
    return (1);
  }
  pfil->sdid = sd_id;     
  pvarl = pfil->pvar1;
  for (i=0; i<pfil->vnum; i++) {
    if (pvarl->sdvid != -888) pvarl->sdvid = -999;
    pvarl++;
  }
  return (0);
#endif
#endif
  gaprnt(0,"Opening HDF-SDS files is not supported in this build\n");
  return(1);
}


/* Read a row varying in the X direction from an HDF-SDS grid */
int gahrow (int x, int y, int z, int t, int len, float *gr) {
#if USESDF == 1
#if USEHDF == 1
int rc,i,j,dim,size,yy,zz;
int32  start[16],count[16];
int32  sd_id, v_id, sds_id, attr_index;
int32  data_dtype, n_atts, rank, dim_sizes[MAX_VAR_DIMS];
int32  attr_dtype, attr_count;
float  val,ulow,uhi;

int8    *bval;
uint8   *ubval;
int16   *sval;
uint16  *usval;
int32   *ival;
uint32  *uival;
char    *cattr_val;
float32 *fattr_val;
float64 *dattr_val;

  /* Get the vid if we haven't already done that for this file */
  if (pvr->sdvid == -888) return(1);  /* already tried and failed */

  sd_id = pfi->sdid;   /* this might be risky, casting int to int32 */
  if (pvr->sdvid == -999) {

    /* Get the variable index number from the variable name */
    if (pvr->longnm) {
      v_id = SDnametoindex(sd_id, pvr->longnm);
    }
    else {
      v_id = SDnametoindex(sd_id, pvr->abbrv);
    }
    if (v_id==FAIL) {
      pvr->sdvid = -888;
      if (pvr->longnm) {
	sprintf(pout,"Error: Variable %s not in HDF-SDS file\n",pvr->longnm);
      }
      else {
	sprintf(pout,"Error: Variable %s not in HDF-SDS file\n",pvr->abbrv);
      }
      gaprnt(0,pout);
      return (1);
    }
    pvr->sdvid = v_id; 

    /* If undef attribute name is used, get the undef value */
    if (pfi->undefattrflg) {
      /* Select the variable (get sds_id) */
      v_id = pvr->sdvid;
      sds_id = SDselect(sd_id,v_id);
      if (sds_id==FAIL) {
	if (pvr->longnm) {
	  sprintf(pout,"Error: SDselect failed for %s \n",pvr->longnm);
	}
	else {
	  sprintf(pout,"Error: SDselect failed for %s \n",pvr->abbrv);
	}
	gaprnt(0,pout);
	return (1);
      }
      /* Retrieve the HDF undef attribute value */
      if (hdfattr(sds_id, pfi->undefattr, &val) != 0) {
	sprintf(pout,"Warning: Could not retrieve undef attribute \"%s\" -- using %g instead\n",
		pfi->undefattr,pfi->undef);
	gaprnt(1,pout);
	pvr->undef = pfi->undef;
      }
      else {
	pvr->undef = val;
      }
    }
    /* If undef attribute name is not given, copy the file-wide undef */
    else {
      pvr->undef = pfi->undef;
    }
 

    /* If data are packed, get the scale factor and offset attribute values */
    if (pfi->packflg) {
      /* initialize values */
      pvr->scale=1.0;
      pvr->add=0.0;

      /* Select the variable (get sds_id) */
      v_id = pvr->sdvid;
      sds_id = SDselect(sd_id,v_id);
      if (sds_id==FAIL) {
	if (pvr->longnm) {
	  sprintf(pout,"Error: SDselect failed for %s \n",pvr->longnm);
	}
	else {
	  sprintf(pout,"Error: SDselect failed for %s \n",pvr->abbrv);
	}
	gaprnt(0,pout);
	return (1);
      }
      /* Retrieve the scale factor attribute value */
      if (hdfattr(sds_id, pfi->scattr, &pvr->scale) != 0) {
	sprintf(pout,"Warning: Could not retrieve \"%s\" -- setting to 1.0\n",pfi->scattr);
        gaprnt(1,pout);
	pvr->scale = 1.0;
      }
      /* Retrieve the add offset attribute value if required */
      if (pfi->packflg == 2) {
	if (hdfattr(sds_id, pfi->ofattr, &pvr->add) != 0) {
	  sprintf(pout,"Warning: Could not retrieve \"%s\" -- setting to 0.0\n",pfi->ofattr);
	  gaprnt(1,pout);
	  pvr->add = 0.0;
	}
      }
    }
  }

  /* Select the variable (get sds_id) */
  v_id = pvr->sdvid;
  sds_id = SDselect(sd_id,v_id);

  if (sds_id==FAIL) {
    if (pvr->longnm) {
      sprintf (pout,"Error: SDselect failed for %s \n",pvr->longnm);
    }
    else {
      sprintf (pout,"Error: SDselect failed for %s \n",pvr->abbrv);
    }
    gaprnt(0,pout);
    return (1);
  }

  /* Change the Y indexes if yrev flag is set */
  if (pfi->yrflg) {
    yy = pfi->dnum[1] - y;
  }
  else {
    yy = y-1;
  }

  /* Change the Z indexes if zrev flag is set */
  if (pfi->zrflg) {
    if (pvr->levels==0) {
      zz=0;
    }
    else {
      zz = pvr->levels-z;
    }
  } 
  else {
    zz = z-1;
  }

  /* Set up the start and count array.  The units records in the
     descriptor file for each variable indicate the mapping of the 
     hdf-sds variable shape into the grads dimensions */
  for (i=0; i<16; i++) {
    start[i] = -999;
    count[i] = -999;
    if (pvr->units[i] == -100) { start[i] = x-1; count[i] = len;}
    if (pvr->units[i] == -101) { start[i] = yy;  count[i] = 1; }
    if (pvr->units[i] == -102) { start[i] = zz;  count[i] = 1; }
    if (pvr->units[i] == -103) { start[i] = t-1; count[i] = 1; }
    if (pvr->units[i] >= 0) { start[i] = pvr->units[i];   count[i] = 1; }
  }

  /* Get the data type */
  if (pvr->longnm) {
    rc = SDgetinfo(sds_id, pvr->longnm, &rank, dim_sizes, &data_dtype, &n_atts);
  }
  else {
    rc = SDgetinfo(sds_id, pvr->abbrv,  &rank, dim_sizes, &data_dtype, &n_atts);
  }

  /* Data types that are handled are 8-bit unsigned ints (uint8), shorts (int16), 
     ints (int32) and float. shorts and ints are converted to float. 
     Unpacking done after I/O is finished  */
  switch (data_dtype)
  {
    case (DFNT_INT8):   /* definition value 20 */
      bval = (int8 *)malloc(len * sizeof (DFNT_INT8));
      if (bval==NULL) {
	gaprnt(0,"HDF-SDS Error: unable to allocate memory for dtype INT8\n");
	return(1);
      }
      if (SDreaddata(sds_id, start, NULL, count, (VOIDP *)bval) != 0) {
	gaprnt(0,"HDF-SDS Read Error for dtype INT8\n");
	return(1);
      } 
      else {
	for (i=0; i<len; i++) gr[i] = (float)bval[i];  /* Cast int8 to float */
      }
      free(bval);
      break;

    case (DFNT_UINT8):   /* definition value 21 */
      ubval = (uint8 *)malloc(len * sizeof (DFNT_UINT8));
      if (ubval==NULL) {
	gaprnt(0,"HDF-SDS Error: unable to allocate memory for dtype UINT8\n");
	return(1);
      }
      if (SDreaddata(sds_id, start, NULL, count, (VOIDP *)ubval) != 0) {
	gaprnt(0,"HDF-SDS Read Error for dtype UINT8\n");
	return(1);
      } 
      else {
	for (i=0; i<len; i++) gr[i] = (float)ubval[i];  /* Cast uint8 to float */
      }
      free(ubval);
      break;

    case (DFNT_INT16):    /* definition value 22 */
      sval = (int16 *)malloc(len * sizeof(DFNT_INT16));
      if (sval==NULL) {
	gaprnt(0,"HDF-SDS Error: unable to allocate memory for dtype INT16\n");
	return(1);
      }
      if (SDreaddata(sds_id, start, NULL, count, (VOIDP *)sval) != 0) {
	gaprnt(0,"HDF-SDS Read Error for dtype INT16\n");
	return(1);
      }
      else {
	for (i=0; i<len; i++) gr[i] = (float)sval[i];  /* Cast int16 to float */
      }
      free(sval);
      break;

    case (DFNT_UINT16):   /* definition value 23 */
      usval = (uint16 *)malloc(len * sizeof (DFNT_UINT16));
      if (usval==NULL) {
	gaprnt(0,"HDF-SDS Error: unable to allocate memory for dtype UINT16\n");
	return(1);
      }
      if (SDreaddata(sds_id, start, NULL, count, (VOIDP *)usval) != 0) {
	gaprnt(0,"HDF-SDS Read Error for dtype UINT16\n");
	return(1);
      } 
      else {
	for (i=0; i<len; i++) gr[i] = (float)usval[i];  /* Cast uint16 to float */
      }
      free(usval);
      break;

    case (DFNT_INT32):    /* definition value 24 */
      ival = (int32 *)malloc(len * sizeof (DFNT_INT32));
      if (ival==NULL) {
	gaprnt(0,"HDF-SDS Error: unable to allocate memory for dtype INT32\n");
	return(1);
      }
      if (SDreaddata(sds_id, start, NULL, count, (VOIDP *)ival) != 0) {
	gaprnt(0,"HDF-SDS Read Error for dtype INT32\n");
	return(1);
      } 
      else {
	for (i=0; i<len; i++) gr[i] = (float)ival[i];  /* Cast int32 to float */
      }
      free(ival);
      break;

    case (DFNT_UINT32):   /* definition value 25 */
      uival = (uint32 *)malloc(len * sizeof (DFNT_UINT32));
      if (uival==NULL) {
	gaprnt(0,"HDF-SDS Error: unable to allocate memory for dtype UINT32\n");
	return(1);
      }
      if (SDreaddata(sds_id, start, NULL, count, (VOIDP *)uival) != 0) {
	gaprnt(0,"HDF-SDS Read Error for dtype UINT32\n");
	return(1);
      } 
      else {
	for (i=0; i<len; i++) gr[i] = (float)uival[i];  /* Cast uint32 to float */
      }
      free(uival);
      break;

    case (DFNT_FLOAT32):  /* definition value  5 */
      if (SDreaddata(sds_id, start, NULL, count, (VOIDP *)gr) != 0) {
	gaprnt(0,"HDF-SDS Read Error for dtype FLOAT32\n");
	return(1);
      } 
      break;

    default:
      sprintf(pout,"HDF-SDS Error: Data type %d not handled\n", data_dtype);
      gaprnt(0,pout);
      return(1);
  };

  /* Set missing data values to exact value if specified */
  if (SETMISS) {
    /* Use the gavar undef to set the fuzzy test limits */
    /* If gavar undef equals zero, change it to 1/EPSILON */
    if (pvr->undef == 0) {   
      ulow = 1e-5;
    } 
    else {
      ulow = fabs(pvr->undef/EPSILON);   
    }
    uhi  = pvr->undef + ulow;
    ulow = pvr->undef - ulow;
    /* set the gagrid undef equal to the gafile undef */
    pgr->undef = pfi->undef;           

    /* Do the fuzzy test for undef values before unpacking */
    for (i=0;i<len;i++) {
      /* set the missing data to the gafile undef */
      if (*(gr+i) >= ulow && *(gr+i) <= uhi) {
	*(gr+i) = pfi->undef;   
      }
      else {
	/* Data is not missing, so unpack with scale and offset if necessary */
	if (pfi->packflg) {
	    *(gr+i) = *(gr+i)*pvr->scale + pvr->add; 
	}
      }
    }
  }

  return (0);

#endif
#endif
  gaprnt(0,"Reading HDF-SDS files is not supported in this build\n");
  return(1);
}


/* Retrieves a non-character HDF-SDS Attribute. 
   Attribute values that are short, int, or double are converted into floats. */
int hdfattr(int sds_id, char *attr_name, float *value) {
#if USESDF == 1
#if USEHDF == 1
int32   rc, attr_index, attr_dtype, attr_count;
uint8   *battr_val;
int16   *sattr_val;
uint16  *usattr_val;
int32   *iattr_val;
uint32  *uiattr_val;
float32 *fattr_val;
float64 *dattr_val;

  /* Get the attribute index number from its name */
  attr_index = SDfindattr(sds_id, attr_name);
  if (attr_index == -1) {
    sprintf(pout,"Warning: HDF attribute named \"%s\" does not exist\n",attr_name);
    gaprnt(1,pout);
    return(1);
  } 

  /* Get info about the attribute, make sure there's only one value */
  if (SDattrinfo(sds_id, attr_index, attr_name, &attr_dtype, &attr_count) == -1) {
    gaprnt(1,"Warning: SDattrinfo failed\n");
    return(1);
  } 
  else {
    if (attr_count != 1) {
      sprintf(pout,"Warning: HDF attribute named \"%s\" has more than one value\n",attr_name);
      gaprnt(1,pout);
      return(1);
    }

    /* Read the attribute value */
    switch (attr_dtype)
      {
      case (DFNT_UINT8):    /* definition value 21 */
	battr_val = malloc (attr_count * sizeof (DFNT_UINT8));
	if (SDreadattr(sds_id, attr_index, battr_val) == -1) {
	  gaprnt(1,"Warning: SDreadattr failed for attribute type UINT8\n");
	  return(1);
	}
	else {
	  *value = *battr_val;
	}
	free(battr_val);
	break;
	
      case (DFNT_INT16):    /* definition value 22 */
	sattr_val = malloc (attr_count * sizeof (DFNT_INT16));
	if (SDreadattr(sds_id, attr_index, sattr_val) == -1) {
	  gaprnt(1,"Warning: SDreadattr failed for attribute type INT16\n");
	  return(1);
	}
	else {
	  *value = *sattr_val;
	}
	free(sattr_val);
	break;
	
      case (DFNT_UINT16):   /* definition value 23 */
	usattr_val = malloc (attr_count * sizeof (DFNT_UINT16));
	if (SDreadattr(sds_id, attr_index, usattr_val) == -1)  {
	  gaprnt(1,"Warning: SDreadattr failed for attribute type UINT16\n");
	  return(1);
	}
	else {
	  *value = *usattr_val;
	}
	free(usattr_val);
	break;
	
      case (DFNT_INT32):    /* definition value 24 */
	iattr_val = malloc (attr_count * sizeof (DFNT_INT32));
	if (SDreadattr(sds_id, attr_index, iattr_val) == -1) {
	  gaprnt(1,"Warning: SDreadattr failed for attribute type INT32\n");
	  return(1);
	}
	else {
	  *value = *iattr_val;
	}
	free(iattr_val);
	break;
	
      case (DFNT_UINT32):   /* definition value 25 */
	uiattr_val = malloc (attr_count * sizeof (DFNT_UINT32));
	if (SDreadattr(sds_id, attr_index, uiattr_val) == -1) {
	  gaprnt(1,"Warning: SDreadattr failed for attribute type UINT32\n");
	  return(1);
	}
	else {
	  *value = *uiattr_val;
	}
	free(uiattr_val);
	break;
	
      case (DFNT_FLOAT32):  /* definition value  5 */
	fattr_val = malloc (attr_count * sizeof (DFNT_FLOAT32));
	if (SDreadattr(sds_id, attr_index, fattr_val) == -1) {
	  gaprnt(1,"Warning: SDreadattr failed for attribute type FLOAT32\n");
	  return(1);
	}
	else {
	  *value = *fattr_val;
	}
	free(fattr_val);
	break;
	
      case (DFNT_FLOAT64):  /* definition value  6 */
	dattr_val = malloc (attr_count * sizeof (DFNT_FLOAT64));
	if (SDreadattr(sds_id, attr_index, dattr_val) == -1) {
	  gaprnt(1,"Warning: SDreadattr failed for attribute type FLOAT64\n");
	  return(1);
	}
	else {
	  *value = *dattr_val;
	}
	free(dattr_val);
	break;
	
      default:
	sprintf(pout,"Warning: HDF Attribute \"%s\" is not a numeric data type (%d)\n", 
		attr_name, attr_dtype);
	gaprnt(1,pout);
	return(1);
      };
  }
  return(0);

#endif
#endif
  gaprnt(0,"Reading HDF-SDS files is not supported in this build\n");
  return(1);
}


/* Subroutine to print out NetCDF attributes */
int ncpattrs(int ncid, char *varnam, char *abbrv, int hdrflg, int fnum, char* ftit) {
#if USESDF == 1
  int attr_index, attr_count, rc, i, truncflag, varid, n_atts;
  char attr_name[MAX_NC_NAME];
  nc_type attr_dtype;
  char   *battr_val;
  char   *cattr_val;
  short  *sattr_val;
  long   *iattr_val;
  float  *fattr_val;
  double *dattr_val;
  int error=0;

/* Get the variable id and number of attributes */
#if USEHDF == 1
  if (cmpwrd("global",abbrv)) {
    varid = NC_GLOBAL;
    rc = ncinquire(ncid, NULL, NULL, &n_atts, NULL);
    if (rc == -1) error=1;
  }
  else {
    varid = ncvarid(ncid, varnam);
    if (varid == -1) error=1;
    if (!error) {
      rc = ncvarinq(ncid, varid, NULL, NULL, NULL, NULL, &n_atts);
      if (rc == -1) error=1;
    }
  }
#else
  if (cmpwrd("global",abbrv)) {
    varid = NC_GLOBAL;
    rc = nc_inq_natts(ncid,&n_atts);
    if (rc != NC_NOERR) error=1;
  }
  else {
    rc = nc_inq_varid(ncid, varnam, &varid);
    if (rc != NC_NOERR) error=1;
    if (!error) {
      rc = nc_inq_varnatts(ncid, varid, &n_atts);
      if (rc != NC_NOERR) error=1;
    }
  }
#endif

  /* Print out the header */
  if (!error) {
    if (hdrflg) {
      if (n_atts > 0) {
	sprintf(pout,"Native Attributes for File %i : %s \n",fnum,ftit);
	gaprnt(2,pout);
      }
    }
  }
  else {
    return(0);  /* zero attributes printed */
  }
  
  /* Loop through list of attributes, print the name of each one */ 
  for (attr_index = 0 ; attr_index < n_atts ; attr_index++) {

    /* Get current attribute's name */
#if USEHDF == 1
    if (ncattname(ncid, varid, attr_index, attr_name) == -1) {
      sprintf(pout,"ncattname failed for variable %s, attribute number %d\n", abbrv, attr_index);
      gaprnt(2,pout);
    }
#else
    if (nc_inq_attname(ncid, varid, attr_index, attr_name) == -1) {
      sprintf(pout,"nc_inq_attname failed for variable %s, attribute number %d\n", abbrv, attr_index);
      gaprnt(2,pout);
    }
#endif
    else {
      /* Get current attribute's data type and length */
#if USEHDF == 1
      if (ncattinq(ncid, varid, attr_name, &attr_dtype, &attr_count) == -1){
	sprintf(pout,"ncattinq failed for variable %s, attribute number %d\n", abbrv, attr_index);
	gaprnt(2,pout);
      }
#else
      if (nc_inq_att(ncid, varid, attr_name, &attr_dtype, (size_t*)&attr_count) == -1) {
	sprintf(pout,"nc_inq_att failed for variable %s, attribute number %d\n", abbrv, attr_index);
	gaprnt(2,pout);
      }
#endif
      else {
        if (attr_count>0) {
	  /* Retrieve and print out the attribute */
	  switch (attr_dtype) 
	    {
	    case (NC_BYTE):
	      battr_val = malloc ((attr_count+1) * sizeof (NC_BYTE));
#if USEHDF == 1
	      if (ncattget(ncid, varid, attr_name, battr_val) == -1) {
		gaprnt(2,"ncattget failed for type NC_BYTE\n"); 
	      }
#else
	      if (nc_get_att_schar(ncid, varid, attr_name, (signed char*)battr_val) == -1) {
		gaprnt(2,"nc_get_att_schar failed for type NC_BYTE\n"); 
	      }
#endif
	      else {
		gaprnt(2,abbrv); 
		gaprnt(2," Byte "); 
		gaprnt(2,attr_name); 
		gaprnt(2," ");
		for (i=0; i<attr_count; i++) {
		  sprintf(pout,"%d ", (int)(battr_val[i])); 
		  gaprnt(2,pout);
		}
		gaprnt(2,"\n");
	      }
	      free(battr_val);
	      break;
	    case (NC_CHAR):
	      cattr_val = malloc ((attr_count+1) * sizeof (NC_CHAR));
#if USEHDF == 1
	      if (ncattget(ncid, varid, attr_name, cattr_val) == -1) {
		gaprnt(2,"ncattget failed for type NC_CHAR\n"); 
	      }
#else
	      if (nc_get_att_text(ncid, varid, attr_name, cattr_val) == -1) {
		gaprnt(2,"nc_get_att_text failed for type NC_CHAR\n"); 
	      }
#endif
	      else {
		cattr_val[attr_count]='\0';
		gaprnt(2,abbrv); 
		gaprnt(2," String ");
		gaprnt(2,attr_name); 
		gaprnt(2," ");
		prntwrap(abbrv, attr_name, cattr_val);
	      }
	      free(cattr_val);
	      break;
	    case (NC_SHORT):
	      sattr_val = malloc (attr_count * sizeof (NC_SHORT));
#if USEHDF == 1
	      if (ncattget(ncid, varid, attr_name, sattr_val) == -1) {
		gaprnt(2,"ncattget failed for type NC_SHORT\n"); 
	      }
#else
	      if (nc_get_att_short(ncid, varid, attr_name, sattr_val) == -1) {
		gaprnt(2,"nc_get_att_short failed for type NC_SHORT\n"); 
	      }
#endif
	      else {
		gaprnt(2,abbrv); 
		gaprnt(2," Int16 "); 
		gaprnt(2,attr_name); 
		gaprnt(2," ");
		for (i=0; i<attr_count; i++) {
		  sprintf(pout,"%d ", (int)(sattr_val[i])); 
		  gaprnt(2,pout);
		}
		gaprnt(2,"\n");
	      }
	      free(sattr_val);
	      break;
	    case (NC_LONG):
	      iattr_val = malloc (attr_count * sizeof (NC_LONG));
#if USEHDF == 1
	      if (ncattget(ncid, varid, attr_name, iattr_val) == -1) {
		gaprnt(2,"ncattget failed for type NC_LONG\n"); 
	      }
#else
	      if (nc_get_att_long(ncid, varid, attr_name, iattr_val) == -1) {
		gaprnt(2,"nc_get_att_long failed for type NC_LONG\n"); 
	      }
#endif
	      else {
		gaprnt(2,abbrv); 
		gaprnt(2," Int32 "); 
		gaprnt(2,attr_name); 
		gaprnt(2," ");
		for (i=0; i<attr_count; i++) {
		  sprintf(pout,"%d ", iattr_val[i]); 
		  gaprnt(2,pout);
		}
		gaprnt(2,"\n");
	      }
	      free(iattr_val);
	      break;
	    case (NC_FLOAT):
	      fattr_val = malloc (attr_count * sizeof (NC_FLOAT));
#if USEHDF == 1
	      if (ncattget(ncid, varid, attr_name, fattr_val) == -1) {
		gaprnt(2,"ncattget failed for type NC_FLOAT\n"); 
	      }
#else
	      if (nc_get_att_float(ncid, varid, attr_name, fattr_val) == -1) {
		gaprnt(2,"nc_get_att_float failed for type NC_FLOAT\n"); 
	      }
#endif
	      else {
		gaprnt(2,abbrv); 
		gaprnt(2," Float32 "); 
		gaprnt(2,attr_name); 
		gaprnt(2," ");
		for (i=0; i<attr_count; i++) {
		  sprintf(pout,"%g ", fattr_val[i]);
		  gaprnt(2,pout);
		}
		gaprnt(2,"\n");
	      }
	      free(fattr_val);
	      break;
	    case (NC_DOUBLE): 
	      dattr_val = malloc (attr_count * sizeof (NC_DOUBLE));
#if USEHDF == 1
	      if (ncattget(ncid, varid, attr_name, dattr_val) == -1) {
		gaprnt(2,"ncattget failed for type NC_DOUBLE\n"); 
	      }
#else
	      if (nc_get_att_double(ncid, varid, attr_name, dattr_val) == -1) {
		gaprnt(2,"nc_get_att_double failed for type NC_FLOAT\n"); 
	      }
#endif
	      else {
		gaprnt(2,abbrv); 
		gaprnt(2," Float64 "); 
		gaprnt(2,attr_name); 
		gaprnt(2," ");
		for (i=0; i<attr_count; i++) {
		  sprintf(pout,"%g ", dattr_val[i]); 
		  gaprnt(2,pout);
		}
		gaprnt(2,"\n");
	      }
	      free(dattr_val);
	      break;
	    default:
	      sprintf(pout,"Failed to retrieve attribute %d of type %d \n", attr_index, attr_dtype);
	      gaprnt(2,pout);
	    };
	} /* end of if statement for attr_count >0 */
      } /* end of if-else statement for ncattinq */
    } /* end of if-else statement for ncattname */
  } /* end of for loop on attr_index */
  return(n_atts);
#endif
  gaprnt(0,"Reading NetCDF attributes is not supported in this build\n");
  return(1);
}


/* Subroutine to print out HDF attributes */
int hdfpattrs(int sdid, char *varname, char *abbrv, int hdrflg, int fnum, char* ftit) {
#if USESDF == 1
#if USEHDF == 1
  int     attr_index, rc, i;
  char    *pos, *line;
  char    attr_name[MAX_NC_NAME];
  int32   attr_dtype, attr_count;
  char8   *cattr_val;
  uchar8  *ucattr_val;
  int8    *icattr_val;
  uint8   *uicattr_val;
  int16   *sattr_val;
  uint16  *usattr_val;
  int32   *iattr_val;
  uint32  *uiattr_val;
  float32 *fattr_val;
  float64 *dattr_val;
  int error=0;
  char name[MAX_NC_NAME];
  int32 sds_id, n_atts, n_dsets,  varid, rank, type, dim_sizes[4];


  /* Get the dataset id and number of attributes */
  if (cmpwrd("global",abbrv)) {
    sds_id = sdid;
    rc = SDfileinfo(sdid, &n_dsets, &n_atts);
    if (rc == -1) error=1;
  }
  else {
    sds_id = SDnametoindex(sdid, varname);
    if (sds_id == -1) error=1;
    if (!error) {
      sds_id = SDselect(sdid,sds_id);
      rc = SDgetinfo(sds_id, name, &rank, dim_sizes, &type, &n_atts);
      if (rc == -1) error=1;
    }
  }
  /* Print out the header */
  if (!error) {
    if (hdrflg) {
      if (n_atts > 0) {
	sprintf(pout,"Native Attributes for File %i : %s \n",fnum,ftit);
	gaprnt(2,pout);
      }
    }
  }
  else {
    return(0);  /* zero attributes printed */
  }

  /* Loop through list of attributes, print the name of each one */ 
  for (attr_index = 0 ; attr_index < n_atts ; attr_index++) {
	 
    /* Get info about the current attribute and then print out Name, Type, and Value */
    if (SDattrinfo(sds_id, attr_index, attr_name, &attr_dtype, &attr_count) == -1) {
      sprintf(pout,"SDattrinfo failed for variable %s, attribute number %d\n", abbrv, attr_index);
      gaprnt(2,pout);
    }
    else {
      switch (attr_dtype) 
	{
	case (DFNT_CHAR8):    /* definition value 4 */
	  cattr_val = malloc ((attr_count+1) * sizeof (char8));
	  if (SDreadattr(sds_id, attr_index, cattr_val) == -1) {
	    gaprnt(2,"SDreadattr failed for type CHAR8\n"); 
	  }
	  else {
	    cattr_val[attr_count]='\0';
	    gaprnt(2,abbrv); 
	    gaprnt(2," String ");
	    gaprnt(2,attr_name); 
	    gaprnt(2," ");
	    prntwrap(abbrv, attr_name, cattr_val);
	  }
	  free(cattr_val);
	  break;
	case (DFNT_UCHAR8):   /* definition value 3 */
	  ucattr_val = malloc ((attr_count+1) * sizeof (uchar8));
	  if (SDreadattr(sds_id, attr_index, ucattr_val) == -1) { 
	    gaprnt(2,"SDreadattr failed for type UCHAR8\n"); 
	  }
	  else {
	    ucattr_val[attr_count]='\0';
	    gaprnt(2,abbrv); 
	    gaprnt(2," String ");
	    gaprnt(2,attr_name); 
	    gaprnt(2," ");
	    prntwrap(abbrv, attr_name, (char*)ucattr_val);
	  }
	  free(ucattr_val);
	  break;
	case (DFNT_INT8):     /* definition value 20 */
	  icattr_val = malloc ((attr_count+1) * sizeof (int8));
	  if (SDreadattr(sds_id, attr_index, icattr_val) == -1) {
	    gaprnt(2,"SDreadattr failed for type INT8\n"); 
	  }
	  else {
	    gaprnt(2,abbrv); 
	    gaprnt(2," Byte "); 
	    gaprnt(2,attr_name); 
	    gaprnt(2," ");
	    for (i=0; i<attr_count; i++) {
	      sprintf(pout,"%d ", (int)(icattr_val[i])); 
	      gaprnt(2,pout);
	    }
	    gaprnt(2,"\n");
	  }
	  free(cattr_val);
	  break;
	case (DFNT_UINT8):    /* definition value 21 */
	  uicattr_val = malloc ((attr_count) * sizeof (uint8));
	  if (SDreadattr(sds_id, attr_index, cattr_val) == -1) {
	    gaprnt(2,"SDreadattr failed for type UINT8\n"); 
	  }
	  else {
	    gaprnt(2,abbrv); 
	    gaprnt(2," Byte "); 
	    gaprnt(2,attr_name); 
	    gaprnt(2," ");
	    for (i=0; i<attr_count; i++) {
	      sprintf(pout,"%u ", (unsigned int)(uicattr_val[i])); 
	      gaprnt(2,pout);
	    }
	    gaprnt(2,"\n");
	  }
	  free(uicattr_val);
	  break;
	case (DFNT_INT16):    /* definition value 22 */
	  sattr_val = malloc (attr_count * sizeof (int16));
	  if (SDreadattr(sds_id, attr_index, sattr_val) == -1) {
	    gaprnt(2,"SDreadattr failed for type INT16\n"); 
	  }
	  else {
	    gaprnt(2,abbrv); 
	    gaprnt(2," Int16 "); 
	    gaprnt(2,attr_name); 
	    gaprnt(2," ");
	    for (i=0; i<attr_count; i++) {
	      sprintf(pout,"%d ", (int)(sattr_val[i])); 
	      gaprnt(2,pout);
	    }
	    gaprnt(2,"\n");
	  }
	  free(sattr_val);
	  break;
	case (DFNT_UINT16):   /* definition value 23 */
	  usattr_val = malloc (attr_count * sizeof (uint16));
	  if (SDreadattr(sds_id, attr_index, usattr_val) == -1) { 
	    gaprnt(2,"SDreadattr failed for type UINT16\n"); 
	  }
	  else {
	    gaprnt(2,abbrv); 
	    gaprnt(2," UInt16 "); 
	    gaprnt(2,attr_name); 
	    gaprnt(2," ");
	    for (i=0; i<attr_count; i++) {
	      sprintf(pout,"%u ", (unsigned int)(usattr_val[i])); 
	      gaprnt(2,pout);
	    }
	    gaprnt(2,"\n");
	  }
	  free(usattr_val);
	  break;
	case (DFNT_INT32):    /* definition value 24 */
	  iattr_val = malloc (attr_count * sizeof (int32));
	  if (SDreadattr(sds_id, attr_index, iattr_val) == -1) {
	    gaprnt(2,"SDreadattr failed for type INT32\n"); 
	  }
	  else {
	    gaprnt(2,abbrv); 
	    gaprnt(2," Int32 "); 
	    gaprnt(2,attr_name); 
	    gaprnt(2," ");
	    for (i=0; i<attr_count; i++) {
	      sprintf(pout,"%d ", iattr_val[i]); 
	      gaprnt(2,pout);
	    }
	    gaprnt(2,"\n");
	  }
	  free(iattr_val);
	  break;
	case (DFNT_UINT32):   /* definition value 25 */
	  uiattr_val = malloc (attr_count * sizeof (uint32));
	  if (SDreadattr(sds_id, attr_index, uiattr_val) == -1) { 
	    gaprnt(2,"SDreadattr failed for type UINT32\n"); 
	  }
	  else {
	    gaprnt(2,abbrv); 
	    gaprnt(2," UInt32 "); 
	    gaprnt(2,attr_name); 
	    gaprnt(2," ");
	    for (i=0; i<attr_count; i++) {
	      sprintf(pout,"%u ", uiattr_val[i]); 
	      gaprnt(2,pout);
	    }
	    gaprnt(2,"\n");
	  }
	  free(iattr_val);
	  break;
	case (DFNT_FLOAT32):  /* definition value  5 */
	  fattr_val = malloc (attr_count * sizeof (float32));
	  if (SDreadattr(sds_id, attr_index, fattr_val) == -1) {
	    gaprnt(2,"SDreadattr failed for type FLOAT32\n"); 
	  }
	  else {
	    gaprnt(2,abbrv); 
	    gaprnt(2," Float32 "); 
	    gaprnt(2,attr_name); 
	    gaprnt(2," ");
	    for (i=0; i<attr_count; i++) {
	      sprintf(pout,"%g ", fattr_val[i]);
	      gaprnt(2,pout);
	    }
	    gaprnt(2,"\n");
	  }
	  free(fattr_val);
	  break;
	case (DFNT_FLOAT64):  /* definition value  6 */
	  dattr_val = malloc (attr_count * sizeof (float64));
	  if (SDreadattr(sds_id, attr_index, dattr_val) == -1) {
	    gaprnt(2,"SDreadattr failed for type FLOAT64\n"); 
	  }
	  else {
	    gaprnt(2,abbrv); 
	    gaprnt(2," Float64 "); 
	    gaprnt(2,attr_name); 
	    gaprnt(2," ");
	    for (i=0; i<attr_count; i++) {
	      sprintf(pout,"%g ", dattr_val[i]); 
	      gaprnt(2,pout);
	    }
	    gaprnt(2,"\n");
	  }
	  free(dattr_val);
	  break;
	default:
	  sprintf(pout,"Failed to retrieve attribute %d of type %d \n", attr_index, attr_dtype);
	  gaprnt(2,pout);
	};
    }  /* end of if-else statment following call to SDattrinfo */
  } 
  return(n_atts);
#endif
#endif
  gaprnt(0,"Reading HDF-SDS attributes is not supported in this build\n");
  return(1);
}

/* routine to print out a string attribute that may have carriage returns in it */
void prntwrap(char *vname, char *aname, char *str ) {
  char *pos, *line;
  pos = line = str;        
  while (*pos != '\0') { 
    if (*pos == '\n') {
      *pos = '\0';         /* swap null for carriage return */
      gaprnt(2,line);
      sprintf(pout," \n%s String %s ",vname,aname); /* add varname, attr_type, and attr_name after carriage return */
      gaprnt(2,pout);
      *pos = '\n';         /* put the carriage return back in */
      line = pos+1;
    }
    pos++;
  }
  if (line < pos) {    /* Print string that has no carriage returns in it */
    gaprnt(2,line); 
  }
  gaprnt(2,"\n");
}
